#!/usr/bin/env python
# Copyright (c) 2019 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

from __future__ import absolute_import, division, print_function

import BasicCli
import CliMatcher
import CliCommand
import ShowCommand
import CliPlugin.VrfCli as VrfCli
import CliPlugin.VrfLeakCli as VrfLeakCli
from CliPlugin.IpGenAddrMatcher import IpGenAddrOrPrefixExprFactory
from CliPlugin.IpAddrMatcher import PREFIX_OVERLAP_AUTOZERO
import CliPlugin.TechSupportCli as TechSupportCli

matcherLeak = CliMatcher.KeywordMatcher( 'leak', helpdesc='Show VRF leaking state' )

matcherPrefix = IpGenAddrOrPrefixExprFactory( ipOverlap=PREFIX_OVERLAP_AUTOZERO,
                                              ip6Overlap=PREFIX_OVERLAP_AUTOZERO,
                                              allowAddr=False )

#--------------------------------------------------------------------------------
# show vrf leak flapping [ source VRF_KW SRC_VRF ] [ destination VRF_KW DEST_VRF ]
#                        [ VRF_KW VRF ] [ prefix PREFIX ]
#--------------------------------------------------------------------------------
class VrfLeakFlappingCmd( ShowCommand.ShowCliCommandClass ):
   syntax = '''show vrf leak flapping [ source VRF_KW SRC_VRF ]
               [ destination VRF_KW DEST_VRF ] [ VRF_KW VRF ] [ prefix PREFIX ]'''
   data = {
      'vrf': VrfCli.vrfKwForShow,
      'leak': matcherLeak,
      'flapping': 'Display flapping prefixes',
      'source': 'Display flapping prefixes sourced from a VRF',
      'VRF_KW': CliCommand.Node(
         matcher=CliMatcher.KeywordMatcher( 'vrf', 'Display flapping prefixes '
                                                   'associated with a VRF' ),
         noResult=True ),
      'SRC_VRF': VrfCli.vrfMatcher,
      'destination': 'Display flapping prefixes destined to a VRF',
      'DEST_VRF': VrfCli.vrfMatcher,
      'VRF': VrfCli.vrfMatcher,
      'prefix': 'Display flapping routes for a prefix',
      'PREFIX': matcherPrefix,
   }

   handler = VrfLeakCli.showVrfLeakFlapping

BasicCli.addShowCommandClass( VrfLeakFlappingCmd )

#--------------------------------------------------------------------------------
# show vrf leak id
#--------------------------------------------------------------------------------
class VrfLeakIdCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show vrf leak id'
   data = {
      'vrf': VrfCli.vrfKwForShow,
      'leak': matcherLeak,
      'id': 'Show VRF ID mappings',
   }

   handler = VrfLeakCli.showLeakVrfId

BasicCli.addShowCommandClass( VrfLeakIdCmd )

#--------------------------------------------------------------------------------
# show tech-support vrfleak overview
#--------------------------------------------------------------------------------

showTechVrfLeakKwMatcher = CliMatcher.KeywordMatcher(
   'vrfleak', 'Show internal agent state for VrfLeak' )

class VrfLeakOverview( ShowCommand.ShowCliCommandClass ):
   syntax = 'show tech-support vrfleak overview'
   data = {
      'tech-support': TechSupportCli.techSupportKwMatcher,
      'vrfleak': showTechVrfLeakKwMatcher,
      'overview': 'Overview of internal agent state',
   }

   handler = VrfLeakCli.showTechVrfLeakOverview

BasicCli.addShowCommandClass( VrfLeakOverview )

#--------------------------------------------------------------------------------
# show vrf leak route
#
# The optional filter syntax is a little complicated here.  we want to
# allow matching on prefixes or VRFs:
#
#   these two filters are mutually exclusive:
#   - prefix: match all routes contained within a prefix (including itself)
#   - exact-prefix: match routes with exactly this prefix, but not ones it contains
#
#   For these three VRF filters we allow just 'vrf', just 'source', just
#   'destination', or both 'source' and 'destination'
#   - vrf: match a given VRF name either as source or destination
#   - source vrf: match a given VRF name as source
#   - destination vrf: match a given VRF name destination
#
# prefix and vrf filters can be combined to give logical AND semantics
#--------------------------------------------------------------------------------
class VrfLeakRoute( ShowCommand.ShowCliCommandClass ):
   syntax = """show tech-support vrfleak route
                  [ ( prefix | exact-prefix ) PREFIX ]
                  [ ( [ source VRF_KW SRC_VRF ]
                      [ destination VRF_KW DEST_VRF ] ) |
                    ( VRF_KW VRF ) ]
            """
   data = {
      'tech-support': TechSupportCli.techSupportKwMatcher,
      'vrfleak': showTechVrfLeakKwMatcher,
      'route': 'Show VrfLeak agent candidate routes',
      'prefix': 'Display routes within prefix',
      'PREFIX': matcherPrefix,
      'exact-prefix': 'Display routes with a specific prefix',
      'source': 'Display routes sourced from a VRF',
      'destination': 'Display routes destined to a VRF',
      'VRF_KW': CliCommand.Node(
         matcher=CliMatcher.KeywordMatcher(
            'vrf', 'Display routes associated with a VRF' ),
         noResult=True ),
      'VRF': VrfCli.vrfMatcher,
      'SRC_VRF': VrfCli.vrfMatcher,
      'DEST_VRF': VrfCli.vrfMatcher,
   }

   handler = VrfLeakCli.showTechVrfLeakRoute

BasicCli.addShowCommandClass( VrfLeakRoute )

