# Copyright (c) 2020 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

'''This should be the one VRF CLI plugin to rule them all.
'''

from __future__ import absolute_import, division, print_function

import os
import subprocess
from functools import partial
from itertools import chain

import CliCommand
import CliMatcher
import CliParser
import CmdExtension
import LazyMount
import ManagedSubprocess
import Tac
from Arnet.NsCliLib import CliNsCmdExtension
from Arnet.NsLib import DEFAULT_NS
from CliModel import GeneratorDict, Model, UncheckedModel
from IpLibConsts import ( # pylint: disable-msg=unused-import
      ALL_VRF_NAME,
      DEFAULT_VRF,
      DEFAULT_VRF_OLD, # Unused.
      VRFNAMES_RESERVED as RESERVED_VRF_NAMES, # Also unused.
)

allVrfConfig = None

def getVrfNames( mode=None ):
   '''Returns an unsorted iterable of the names of the configured VRFs.
   '''
   return allVrfConfig.vrf

def getAllVrfNames( mode=None ):
   '''Returns an unsorted iterable of the names of the configured VRFs,
   including the default VRF.
   '''
   return chain( getVrfNames( mode ), ( DEFAULT_VRF, ) )

def getAllPlusReservedVrfNames( mode=None ):
   '''Returns an unsorted iterable of the names of the configured VRFs,
   including the default VRF and the reserved word 'all'.
   '''
   return chain( getAllVrfNames( mode ), ( ALL_VRF_NAME ) )

def vrfExists( vrf, collection=None ):
   '''Function is pretty straightforward.
   The `collection` parameter can be either the ip/vrf/config mountpoint,
   from which we automatically look at the `vrf` attribute,
   or any other collection type.'''
   if not vrf:
      raise ValueError( 'Invalid VRF name: %r' % vrf )

   collection = collection or allVrfConfig
   try:
      # pylint: disable=unsupported-membership-test
      return vrf == DEFAULT_VRF or vrf in getattr( collection, 'vrf', collection )
   except IndexError: # len( vrf ) > 100
      return False

def getVrfNameFromIntfConfig( ipConfig, intf ):
   ipIntfConfig = ipConfig.ipIntfConfig.get( intf )
   if ipIntfConfig is None:
      return DEFAULT_VRF
   return ipIntfConfig.vrf

vrfKwForShow = CliMatcher.KeywordMatcher( 'vrf',
      helpdesc='Display VRF state' )
vrfMatcher = CliMatcher.DynamicNameMatcher( getVrfNames,
      helpdesc='VRF name' )
vrfMatcherAll = CliMatcher.KeywordMatcher( ALL_VRF_NAME,
      helpdesc='All Virtual Routing and Forwarding instances' )
vrfMatcherDefault = CliMatcher.KeywordMatcher( DEFAULT_VRF,
      helpdesc='Default Virtual Routing and Forwarding instance' )

class VrfNameExprFactory( CliCommand.CliExpressionFactory ):
   '''Generates a `VRF | DEFAULT_VRF | all` expression, where:
   'VRF' is a dynamic list of VRFs, which defaults to all VRFS;
   'DEFAULT_VRF' is the default VRF; and
   'all' is a kewyord for all VRFs.
   The latter two are added based on the `defaultVrf` and `allVrf` flags.
   '''
   def __init__( self, vrfFunc=getVrfNames, maxMatches=0,
                 inclAllVrf=False, inclDefaultVrf=False ):
      '''Create a dictionary of desired matchers:
      dynamic (from f'n), default VRF, and all VRFs.
      '''
      assert callable( vrfFunc ), (
            'vrfFunc must be a function which takes `mode` as a parameter' )
      CliCommand.CliExpressionFactory.__init__( self )
      self.maxMatches = maxMatches
      self.matchers_ = {
         'DYNAMIC' : vrfMatcher,
      }
      if inclAllVrf:
         self.matchers_[ 'ALL' ] = vrfMatcherAll
      if inclDefaultVrf:
         self.matchers_[ 'DEFAULT' ] = vrfMatcherDefault

   def generate( self, name ):
      '''For each matcher, unique-fy its key and wrap it in a `Node` with `alias`.
      '''
      Node = partial( CliCommand.Node, alias=name, maxMatches=self.maxMatches )
      class VrfExpression( CliCommand.CliExpression ):
         data = { name + '_' + key: Node( matcher=matcher )
                  for key, matcher in self.matchers_.items() }
         expression = ' | '.join( data )

      return VrfExpression

class VrfExprFactory( CliCommand.CliExpressionFactory ):
   '''Generates a `vrf VRF` expression.
   '''
   def __init__( self, keyword='vrf', helpdesc='Configure VRF',
                 guard=None, hidden=False, sharedMatchObj=None, maxMatches=0,
                 vrfFunc=getVrfNames, inclAllVrf=False, inclDefaultVrf=False ):
      CliCommand.CliExpressionFactory.__init__( self )
      self.vrfNode_ = CliCommand.guardedKeyword( keyword,
                                                 helpdesc=helpdesc,
                                                 guard=guard,
                                                 hidden=hidden,
                                                 sharedMatchObj=sharedMatchObj,
                                                 maxMatches=maxMatches,
                                                 noResult=True )
      self.vrfExprFactory_ = VrfNameExprFactory( vrfFunc=vrfFunc,
                                                 inclAllVrf=inclAllVrf,
                                                 inclDefaultVrf=inclDefaultVrf,
                                                 maxMatches=maxMatches )

   def generate( self, name ):
      kwName = name + '_VRF_KW'
      exprFactoryName = name + '_VRF'

      class VrfExpression( CliCommand.CliExpression ):
         expression = kwName + ' ' + exprFactoryName
         data = {
            kwName: self.vrfNode_,
            exprFactoryName: self.vrfExprFactory_
         }

         @staticmethod
         def adapter( mode, args, argsList ):
            vrf = args.pop( exprFactoryName, None )
            if vrf:
               args[ name ] = vrf

      return VrfExpression

class _VrfCliModel( Model ):
   pass

class _VrfCliUncheckedModel( UncheckedModel ):
   pass

def generateVrfCliModel( cliModel, desc, uncheckedModel=False, revision=None ):
   '''
   This is used for generating Capi Models that uses the VrfExecCmdDec.
   All Cli functions that are decorated by VrfExecCmdDec will not be returning
   the models to the correct parent function because the decorator can run
   the same Cli function on multiple Vrfs. Therefore, rather than creating a
   top level model for every single Cli command that will be the exactly the same
   as each other, this function will automatically generate it.

   The name of the model is 'Vrf' + model name and contains a generator dictionary
   attribute which will expect the render function to iterate through and call
   render on.
   '''
   vrfModels = GeneratorDict( keyType=str, valueType=cliModel,
                              help=desc )

   def render( self ):
      for _, model in self.vrfs:
         model.render()

   if not revision:
      revision = cliModel.__revision__

   className = 'Vrf%s' % cliModel.__name__
   classBody = { '__revision__': revision,
                 '__streamable__': cliModel.__streamable__,
                 'vrfs': vrfModels,
                 'render': render }

   baseModel = _VrfCliUncheckedModel if uncheckedModel else _VrfCliModel

   return type( className, ( baseModel, ), classBody )

vrfMap = None

class VrfExecCmdDec( object ):
   '''
   Decorator class that can be used to process
   Cli commands of the type "show <command> vrf [VRFNAME|default|all]"
   or "show <command>" in a VRF routing-context mode

   This is to be used in conjunction with the Vrf Cli Parser rules defined
   above. The parsed VRF name token is returned in the keyword argument called
   'vrfName' and the decorated function returned by this class assumes the
   'vrfName' argument is present

   Example usage:
   to extend the 'show ip bgp' command with 'vrf [VRFNAME|default|all]'
   add the 'VrfExecCmdDec' decorator to the doShowXXX function and pass
   a @getVrfsFunc argument to the decorator. @getVrfsFunc is a function pointer
   for a function that returns the list of configured VRF names

   @VrfExecCmdDec( getVrfsFunc )
   def doShowIpBgp( mode, addr=None, longerPrefixes=None, detail=None,
                    vrfName=None ):
   '''
   def __init__( self, getVrfsFunc=None, cliModel=None ):
      '''The function to be decorated is not passed into the constructor'''
      self.getVrfsFunc = getVrfsFunc
      self.modelType = cliModel
      assert ( cliModel is None or
               issubclass( cliModel, ( _VrfCliModel, _VrfCliUncheckedModel ) ) )

   def __call__( self, func ):
      '''return a decoroated function that wraps @func
         arguments to the wrapped function are passed as is'''
      def vrfExecCmdFunc( *args, **kwargs ):
         '''
         first argument in the args list must be of type 'CliParser.Mode'
         and one of the keyword arguments must be named 'vrfName'
         '''
         if kwargs.keys() == [ 'args' ]:
            kwargRef = kwargs[ 'args' ]
            vrfNameKw = 'VRF'
            if vrfNameKw not in kwargRef:
               kwargRef[ vrfNameKw ] = None
         else:
            kwargRef = kwargs
            vrfNameKw = 'vrfName'

         if not isinstance( args[ 0 ], CliParser.Mode ) or \
            vrfNameKw not in kwargRef:
            # call func as is since we can't do anything better
            func( *args, **kwargs )
         v = kwargRef.get( vrfNameKw )
         mode = args[ 0 ]
         # actual list of VRFs that the command needs to be execd over
         vrfList = []
         # determine the VRF in the current routing context
         vrfName = vrfMap.lookupCliModeVrf( mode, v )
         if not vrfName:
            vrfList.append( DEFAULT_VRF )
         elif vrfName == ALL_VRF_NAME:
            vrfList.append( DEFAULT_VRF )
            for v in sorted( self.getVrfsFunc() ):
               if vrfName != DEFAULT_VRF:
                  vrfList.append( v )
         else:
            vrfList.append( vrfName )

         def execFunc():
            for v in vrfList:
               kwargRef[ vrfNameKw ] = v
               model = func( *args, **kwargs )
               if not model:
                  continue
               yield v, model

         length = len( vrfList )
         if self.modelType:
            model = self.modelType()
            model.vrfs = execFunc()
            return model
         else:
            for index, vrf in enumerate( vrfList ):
               kwargRef[ vrfNameKw ] = vrf
               func( *args, **kwargs )
               if index < ( length - 1 ):
                  # separate output between 2 vrfs by an empty line
                  print()

         return None

      return vrfExecCmdFunc

class CliVrfMapper( object ):
   def __init__( self, allVrfStatusLocal, uid, gid ):
      self.uid = uid
      self.gid = gid
      self.allVrfStatusLocal = allVrfStatusLocal

   def setCliVrf( self, session, vrfName=DEFAULT_VRF, nsName=DEFAULT_NS ):
      session.sessionDataIs( 'vrf', ( vrfName, nsName ) )

   def getCliSessVrf( self, session ):
      vinf = session.sessionData( 'vrf', defaultValue=( DEFAULT_VRF, DEFAULT_NS ) )
      return vinf[ 0 ]

   def isDefaultVrf( self, vrfName ):
      return vrfName == DEFAULT_VRF

   def getVrfNamespace( self, vrfName ):
      assert vrfName != ''
      if vrfName == DEFAULT_VRF:
         return DEFAULT_NS
      vrfstat = self.allVrfStatusLocal.vrf.get( vrfName )
      if ( not vrfstat ) or ( vrfstat.state != 'active' ):
         raise EnvironmentError( 0, "Invalid input - VRF '%s' does not exist"
                                 % vrfName )
      return vrfstat.networkNamespace

   def lookupCliSessVrf( self, session, vrfName ):
      assert vrfName != ''
      if vrfName is None:
         vrfName = self.getCliSessVrf( session )
      assert vrfName != None
      return vrfName

   def lookupCliModeVrf( self, mode, vrfName ):
      return self.lookupCliSessVrf( mode.session, vrfName )

class CliVrfCmdExtension( CliNsCmdExtension ):
   def __init__( self, cliVrfMap ):
      CliNsCmdExtension.__init__( self )
      self.cliVrfMap = cliVrfMap

   def _genVrfCmdLine( self, vrfName, cliSession, cmdLine,
                       vrfMount=False, useSudo=True, results=None ):
      assert vrfName != ''
      assert isinstance( cmdLine, list )
      vrfName = self.cliVrfMap.lookupCliSessVrf( cliSession, vrfName )
      assert vrfName != None and vrfName != ''
      if vrfName == DEFAULT_VRF:
         sudoAdvice = False
         if cmdLine[ 0 ] == 'sudo' and useSudo is False:
            cmdLine.pop( 0 )
            sudoAdvice = True
         if results is not None:
            assert isinstance( results, dict )
            results[ 'Sudo' ] = sudoAdvice
         return True

      # verify the namespace exists. Caller handles exception
      nsName = self.cliVrfMap.getVrfNamespace( vrfName )

      self._genNsCmdLine( nsName, cliSession, cmdLine, useSudo=useSudo,
                          results=results, vrfMount=vrfMount, vrfName=vrfName )

      if results is not None:
         assert isinstance( results, dict )
         results[ 'vrf' ] = vrfName
      return True

   @CmdExtension.setDefaultArgs
   def extendCmd( self, cmdLine, session, **kwargs ):
      vrfName = kwargs[ 'vrfName' ] if 'vrfName' in kwargs else None
      useSudo = kwargs[ 'useSudo' ] if 'useSudo' in kwargs else True
      results = kwargs[ 'results' ] if 'results' in kwargs else None
      return self._genVrfCmdLine( vrfName, session, cmdLine,
                                  useSudo=useSudo,
                                  results=results )

   def _filterKwargs( self, filterArgList, kwargs ):
      for arg in filterArgList:
         if arg in kwargs:
            del kwargs[ arg ]

   @CmdExtension.setDefaultArgs
   def runCmd( self, execCmd, session, **kwargs ):
      vrfName = kwargs[ 'vrfName' ] if 'vrfName' in kwargs else None
      assert vrfName != ''
      asRoot = kwargs[ 'asRoot' ] if 'asRoot' in kwargs else None
      vinfo = {}
      self._genVrfCmdLine( vrfName, session, execCmd,
                           useSudo=False, results=vinfo )
      if not asRoot:
         # Override asRoot only if it is not already True
         asRoot = vinfo[ 'Sudo' ]

      # Remove vrfName, asRoot and useSudo (sudo cannot be used in Tac.run()
      self._filterKwargs( [ 'vrfName', 'asRoot', 'useSudo' ], kwargs )
      return Tac.run( execCmd, asRoot=asRoot, **kwargs )

   @CmdExtension.setDefaultArgs
   def subprocessPopen( self, execCmd, session, **kwargs ):
      self.extendCmd( execCmd, session, **kwargs )
      self._filterKwargs( [ 'vrfName', 'asRoot', 'useSudo' ], kwargs )
      return subprocess.Popen( args=execCmd, **kwargs )

   @CmdExtension.setDefaultArgs
   def managedSubprocessPopen( self, execCmd, session, **kwargs ):
      self.extendCmd( execCmd, session, **kwargs )
      self._filterKwargs( [ 'vrfName', 'asRoot', 'useSudo' ], kwargs )
      return ManagedSubprocess.Popen( execCmd, **kwargs )

   @CmdExtension.setDefaultArgs
   def system( self, execCmd, session, **kwargs ):
      cmdList = [ execCmd ]
      self.extendCmd( cmdList, session, **kwargs )
      cmdStr = kwargs[ 'env' ] if 'env' in kwargs else ''
      cmdStr += ' '
      cmdStr += ' '.join( cmdList )
      return os.system( cmdStr )

vrfCmdExt = None

def Plugin( em ):
   global allVrfConfig
   allVrfConfig = LazyMount.mount( em, 'ip/vrf/config', 'Ip::AllVrfConfig', 'wi' )
