#!/usr/bin/env python
# Copyright (c) 2019 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

from __future__ import absolute_import, division, print_function

import CliCommand
import CliMatcher
import CliPlugin.VlanCli as VlanCli
import CliPlugin.VrfCli as VrfCli
import CliPlugin.VmTracerCli as VmTracerCli
from CliPlugin.IntfCli import Intf

from ReversibleSecretCli import reversibleAuthPasswordExpression

matcherSourceInterface = CliMatcher.KeywordMatcher( 'source-interface',
      helpdesc=( 'Interface providing the IP source address of '
                 'VmTracer session packets' ) )
matcherUrl = CliMatcher.KeywordMatcher( 'url',
      helpdesc='URL of VMware vCenter instance to connect to' )
matcherUsername = CliMatcher.KeywordMatcher( 'username',
      helpdesc='Login username for vCenter instance' )
matcherVrf = CliMatcher.KeywordMatcher( 'vrf',
      helpdesc='VCenter/vShield VRF' )
matcherVxlan = CliMatcher.KeywordMatcher( 'vxlan',
      helpdesc='VXLAN' )

#--------------------------------------------------------------------------------
# [ no | default ] allowed-vlan ( all | none | VLANSET | ( VLANSETOP VLANSET ) )
#--------------------------------------------------------------------------------
class AllowedVlanCmd( CliCommand.CliCommandClass ):
   syntax = ( 'allowed-vlan ( all | none | VLANSET | ( VLANSETOP VLANSET ) )' )
   noOrDefaultSyntax = 'allowed-vlan ...'
   data = {
      'allowed-vlan' : 'Set allowed VLANs for the session',
      'all' : 'All VLANs',
      'none' : 'No VLANs',
      'VLANSETOP' : CliMatcher.EnumMatcher( {
                        'add': 'Add VLANs to the current list',
                        'remove': 'Remove VLANs from the current list',
                        'except': 'All VLANs except the following'
                    } ),
      'VLANSET' : VlanCli.vlanIdRangeMatcher,
   }

   handler = VmTracerCli.VCenterConfigMode.setTrunkAllowedVlan
   noHandler = VmTracerCli.VCenterConfigMode.noTrunkAllowedVlan
   defaultHandler = VmTracerCli.VCenterConfigMode.defaultTrunkAllowedVlan

VmTracerCli.VCenterConfigMode.addCommandClass( AllowedVlanCmd )

#--------------------------------------------------------------------------------
# [ no | default ] autovlan disable
#--------------------------------------------------------------------------------
class AutovlanDisableCmd( CliCommand.CliCommandClass ):
   syntax = 'autovlan disable'
   noOrDefaultSyntax = 'autovlan disable ...'
   data = {
      'autovlan' : 'Parameter for auto provisioning VLANs',
      'disable' : 'Disable auto provisioning VLANs',
   }

   handler = VmTracerCli.VCenterConfigMode.disableAutoVlan
   noOrDefaultHandler = VmTracerCli.VCenterConfigMode.enableAutoVlan

VmTracerCli.VCenterConfigMode.addCommandClass( AutovlanDisableCmd )

#--------------------------------------------------------------------------------
# [ no | default ] password PASSWORD
#--------------------------------------------------------------------------------
class PasswordCmd( CliCommand.CliCommandClass ):
   syntax = 'password PASSWORD'
   noOrDefaultSyntax = 'password ...'
   data = {
      'password' : 'Login password for vCenter instance',
      'PASSWORD' : reversibleAuthPasswordExpression( 'PASSWORD',
                   unencryptedName='TEXT', encryptedName='PWD' ),
   }

   handler = VmTracerCli.VCenterConfigMode.setPassword
   noOrDefaultHandler = VmTracerCli.VCenterConfigMode.noPassword

VmTracerCli.VCenterConfigMode.addCommandClass( PasswordCmd )

#--------------------------------------------------------------------------------
# source-interface INTF
# ( no | default ) source-interface ...
#--------------------------------------------------------------------------------
class SourceInterfaceIntfCmd( CliCommand.CliCommandClass ):
   syntax = 'source-interface INTF'
   noOrDefaultSyntax = 'source-interface ...'
   data = {
      'source-interface' : matcherSourceInterface,
      'INTF' : Intf.matcherWithIpSupport,
   }

   handler = VmTracerCli.VCenterConfigMode.setSourceIntf
   noOrDefaultHandler = VmTracerCli.VCenterConfigMode.noSourceIntf

VmTracerCli.VCenterConfigMode.addCommandClass( SourceInterfaceIntfCmd )

#--------------------------------------------------------------------------------
# url URL
# ( no | default ) url ...
#--------------------------------------------------------------------------------
class UrlUrlCmd( CliCommand.CliCommandClass ):
   syntax = 'url URL'
   noOrDefaultSyntax = 'url ...'
   data = {
      'url' : matcherUrl,
      'URL' : CliMatcher.PatternMatcher( pattern='.+',
              helpdesc='URL of vCenter', helpname='URL' ),
   }

   handler = VmTracerCli.VCenterConfigMode.setUrl
   noOrDefaultHandler = VmTracerCli.VCenterConfigMode.noUrl

VmTracerCli.VCenterConfigMode.addCommandClass( UrlUrlCmd )

#--------------------------------------------------------------------------------
# username USERNAME
# ( no | default ) username ...
#--------------------------------------------------------------------------------
class UsernameUsernameCmd( CliCommand.CliCommandClass ):
   syntax = 'username USERNAME'
   noOrDefaultSyntax = 'username ...'
   data = {
      'username' : matcherUsername,
      'USERNAME' : CliMatcher.PatternMatcher( pattern='.+',
                   helpdesc='Username to login to vCenter', helpname='username' ),
   }

   handler = VmTracerCli.VCenterConfigMode.setUsername
   noOrDefaultHandler = VmTracerCli.VCenterConfigMode.noUsername

VmTracerCli.VCenterConfigMode.addCommandClass( UsernameUsernameCmd )

#--------------------------------------------------------------------------------
# vrf VRFNAME
# ( no | default ) vrf ...
#--------------------------------------------------------------------------------
class VrfVrfnameCmd( CliCommand.CliCommandClass ):
   syntax = 'vrf VRFNAME'
   noOrDefaultSyntax = 'vrf ...'
   data = {
      'vrf' : matcherVrf,
      'VRFNAME' : VrfCli.vrfMatcher,
   }

   handler = VmTracerCli.VCenterConfigMode.setVrf
   noOrDefaultHandler = VmTracerCli.VCenterConfigMode.noVrf

VmTracerCli.VCenterConfigMode.addCommandClass( VrfVrfnameCmd )

#--------------------------------------------------------------------------------
# [ no | default ] vxlan
#--------------------------------------------------------------------------------
class VxlanCmd( CliCommand.CliCommandClass ):
   syntax = 'vxlan'
   noOrDefaultSyntax = 'vxlan ...'
   data = {
      'vxlan' : matcherVxlan,
   }

   handler = VmTracerCli.enterVShieldMode
   noOrDefaultHandler = VmTracerCli.delVxlan

VmTracerCli.VCenterConfigMode.addCommandClass( VxlanCmd )
