# Copyright (c) 2012 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

from __future__ import absolute_import, division, print_function

import Arnet
from CliMode.Intf import IntfMode
from CliModel import (
   Bool,
   Dict,
   Enum,
   GeneratorDict,
   Int,
   List,
   Model,
   Str,
   Submodel,
)
import CliPlugin.IntfCli as IntfCli
import IntfModels
from TableOutput import (
   Format,
   Headings,
   TableFormatter,
   createTable,
)
from Vlan import (
   vlanSetToCanonicalString,
   vlanSetToCanonicalStringGen,
)
from six.moves import map

class Vlans( Model ):
   class Status( Model ):
      class Interface( Model ):
         annotation = Str( optional=True, help="Annotation explaining any "
                           "configuration specific to this interface, i.e. "
                           "that the interface is in a port-channel" )
         _annotationSymbol = Str( default='', # Eventually change to an Enum
                                  help="The symbol corresponding to the annotation "
                                  "used in ASCII representations" )
         privatePromoted = Bool( default=False, help="True if this is a "
                                 "private VLAN promoted port" )

      name = Str( help='VLAN name' )
      dynamic = Bool( help='Indicates whether or not this VLAN is dynamic' )
      status = Enum( values=( 'active', 'suspended', 'inactive' ),
                     help='VLAN administrative status' )
      interfaces = Dict( valueType=Interface,
                         help='List of interfaces that are a member of this VLAN, '
                         'keyed by interface name' )

   vlans = GeneratorDict( keyType=int, valueType=Status,
                 help='Configured VLANs, keyed by VLAN Id' )
   sourceDetail = Str( help='A description of the source of this'
         ' configuration.  When using MLAG, this field may indicate that the'
         ' primary peer provided a config that overrides the secondary\'s'
         ' local config.', optional=True )

   def render( self ):
      if self.sourceDetail:
         print( self.sourceDetail )

      briefFormat = '%-5s %-32s %-9s'
      prefix = ' ' * len( briefFormat % ( 1, '', '' ) )
      notes = { "dynVlan": False, "promotedPort": False }
      legends = set()

      #--------------------------------------------------------------------
      # Add one or two '*'s to intf.shortname to indicate if intf is a Lag
      # member or Mirror destination respectively, in show vlan configured.
      #--------------------------------------------------------------------
      def annotateIntf( intfName, intf ):
         # pylint: disable=protected-access
         annotatedIntf = IntfMode.getShortname( intfName )
         if intf._annotationSymbol:
            annotatedIntf += intf._annotationSymbol
            legends.add( intf._annotationSymbol +
                         ' interface specific configuration overridden by ' +
                         intf.annotation + '.' )
         if intf.privatePromoted:
            annotatedIntf += '+'
            notes[ "promotedPort" ] = True

         return annotatedIntf

      headerPrinted = False

      for vid, v in self.vlans:
         if not headerPrinted:
            print ( 'VLAN  Name                             Status    Ports\n'
                    '----- -------------------------------- --------- '
                    '-------------------------------' )
            headerPrinted = True

         vid = str( vid )
         if v.dynamic:
            # If the VLAN is in dynamic set and not in static set
            # (configured from Cli), suffix it with '*' to indicate it
            # as a dynamic VLAN.
            vid += '*'
            notes[ "dynVlan" ] = True
         print( briefFormat % ( vid, v.name, v.status ), end=' ' )
         if not v.interfaces:
            print()
         else:
            annotatedIntfs = [ annotateIntf( intfName, v.interfaces[ intfName ] )
                               for intfName in Arnet.sortIntf( v.interfaces ) ]
            portsLine = annotatedIntfs[ 0 ]
            # Print the list of ports that are members of this VLAN.  If the list is
            # too long to fit on one line, wrap it onto subsequent lines.
            firstLine = True
            for intf in annotatedIntfs[ 1: ]:
               if len( portsLine ) + len( ', ' ) + len( intf ) > 31:
                  if not firstLine:
                     print( prefix, end=' ' )
                  print( portsLine )
                  firstLine = False
                  portsLine = ''
               else:
                  portsLine += ', '
               portsLine += intf
            if not firstLine:
               print( prefix, end=' ' )
            print( portsLine )

      # If we did not show any vlans, return right away
      if not headerPrinted:
         return

      for l in legends:
         print( l )

      print()
      if notes[ "dynVlan" ]:
         print( '* indicates a Dynamic VLAN' )
      if notes[ "promotedPort" ]:
         print( '+ indicates a private VLAN promoted port' )
      # if not brief:
      #    # BUG191 This table removed until we have something useful to display.
      #    print 'VLAN Type  SAID       MTU   Parent RingNo BridgeNo Stp  BrdgMode' \
      #         ' Trans1 Trans2'
      #    print '---- ----- ---------- ----- ------ ------ -------- ---- --------' \
      #         ' ------ ------'
      #    for v in vlans:
      #      # BUG191 This information shouldn't all be hard-coded.
      #      print detailFormat % ( v.id, 'enet', 100000 + v.id, 1500,
      #                             '-', '-', '-', '-', '-', 0, 0 )
      #    print
      #    pass


class VlanTrunkGroups( Model ):
   class GroupNames( Model ):
      names = List( valueType=str, help="Trunk group names" )

   trunkGroups = Dict( keyType=int, valueType=GroupNames,
                       help='Trunk groups associated with each VLAN.' )
   sourceDetail = Vlans.sourceDetail


   def render( self ):
      if self.sourceDetail:
         print( self.sourceDetail )

      tgFormat = '%-4d    '
      maxLen = 70
      prefix = ' ' * len( tgFormat % ( 1, ) )

      print( 'VLAN     Trunk Groups\n' '----     ' + '-' * maxLen )
      for v in sorted( self.trunkGroups ):
         print( tgFormat % ( v, ), end=' ' )

         groups = self.trunkGroups[ v ].names
         if not groups:
            print()
         else:
            firstLine = True
            curLine = groups[ 0 ]
            for trunkGroup in groups[ 1: ]:
               if len( curLine ) + len( ', ' ) + len( trunkGroup ) > maxLen:
                  if not firstLine:
                     print( prefix, end=' ' )
                  print( curLine )
                  firstLine = False
                  curLine = ''
               else:
                  curLine += ', '
               curLine += trunkGroup
            if not firstLine:
               print( prefix, end=' ' )
            print( curLine )
      print()


def getDirection( ingressDefined, egressDefined ):
   ''' This method returns a string representing the direction in which a vlan
       mapping is defined based on the given boolean variables ingressDefined and 
       egressDefined. '''
   if ingressDefined and egressDefined:
      direction = "In/Out"
   elif ingressDefined:
      direction = "In"
   elif egressDefined:
      direction = "Out"
   else:
      direction = "-"

   return direction

class MappedVlan( Model ):
   # Value type for the vlan mappings collection
   vlanId = Int( help='VLAN ID that packets are mapped to (the outer VLAN in a '
                      'double mapping).' )
   innerVlanId = Int( help='Inner VLAN ID that packets are mapped to.',
                      optional=True )
   dot1qTunnel = Bool( help='Whether the tag is dot1q tunneled.',
                       optional=True )
   active = Bool( help='Whether this mapping is currently active.' )
   configured = Bool( help='Whether this mapping is currently configured.' )

class InnerTagMappings( Model ):
   # Value type for the double tagged collections
   mappings = Dict( keyType=int, valueType=MappedVlan,
                    help='Inner tag VLAN mappings' )

class InterfaceVlanMappings( Model ):
   # collection of vlan mappings defined for ingress packets on an interface
   ingressVlanMappings = Dict( keyType=int, valueType=MappedVlan,
                               help='VLAN mappings configured for ingress '
                                    'packets on an interface.' )
   # collection of vlan mappings defined for egress packets on an interface
   egressVlanMappings = Dict( keyType=int, valueType=MappedVlan,
                              help='VLAN mappings configured for egress packets '
                                   'on an interface.' )
   ingressDoubleTaggedMappings = Dict( keyType=int, valueType=InnerTagMappings,
                                       help='Double tagged VLAN mappings '
                                            'configured for ingress packets on '
                                            'an interface',
                                       optional=True )
   egressDoubleTaggedMappings = Dict( keyType=int, valueType=InnerTagMappings,
                                      help='Double tagged VLAN mappings '
                                           'configured for egress packets on an '
                                           'interface',
                                      optional=True )
   ingressVlanMappingRequired = Bool( default=False,
                                      help='Whether vlan mapping is required for '
                                           'ingress packets' )

   def render( self ):
      def renderVlanMap( ingressVlanId, ingressInnerVlanId, ingressMappedVlan,
                         egressVlanId, egressInnnerVlanId, egressMappedVlan ):
         assert ingressMappedVlan or egressMappedVlan
         activeDirection = \
               getDirection( ingressMappedVlan and ingressMappedVlan.active,
                             egressMappedVlan and egressMappedVlan.active )
         configDirection = \
               getDirection( ingressMappedVlan and ingressMappedVlan.configured,
                             egressMappedVlan and egressMappedVlan.configured )
         dot1qTunnelDirection = \
               getDirection( ingressMappedVlan and ingressMappedVlan.dot1qTunnel,
                             egressMappedVlan and egressMappedVlan.dot1qTunnel )
         status = 'Inactive' if activeDirection == '-' else 'Active'

         if ingressMappedVlan:
            assert not ingressMappedVlan.innerVlanId
            params = ( ingressVlanId, ingressInnerVlanId or "-",
                       ingressMappedVlan.vlanId )
         else:
            if egressInnnerVlanId:
               assert egressMappedVlan.dot1qTunnel
               assert not egressMappedVlan.innerVlanId
               assert egressMappedVlan.vlanId == egressInnnerVlanId
            else:
               assert not egressMappedVlan.dot1qTunnel
            params = ( egressMappedVlan.vlanId,
                       egressMappedVlan.innerVlanId or "-",
                       egressVlanId )

         fmtArgs = params + ( status, configDirection, activeDirection,
                              dot1qTunnelDirection )
         print( fmt % fmtArgs )

      def handleIngressVlanMap( egressSkip, ingressVlanId, ingressInnerVlanId,
                                ingressMappedVlan ):
         def egressVlanMap( ingressVlanId, ingressInnerVlanId, ingressMappedVlan ):
            egressVlanId = ingressMappedVlan.vlanId
            egressInnerVlanId = \
                  ingressVlanId if ingressMappedVlan.dot1qTunnel else None

            if ingressMappedVlan.dot1qTunnel:
               innerTagMappings = \
                     self.egressDoubleTaggedMappings.get( egressVlanId )
               egressMappedVlan = \
                     innerTagMappings.mappings.get( egressInnerVlanId ) if \
                        innerTagMappings and innerTagMappings.mappings else None
            else:
               egressMappedVlan = self.egressVlanMappings.get( egressVlanId )

            if egressMappedVlan and \
                  egressMappedVlan.vlanId == ingressVlanId and \
                  ( ( egressMappedVlan.dot1qTunnel and
                      ingressMappedVlan.dot1qTunnel ) or
                    ( egressMappedVlan.innerVlanId == ingressInnerVlanId ) ):
               return ( egressVlanId, egressInnerVlanId, egressMappedVlan )

            return ( None, None, None )

         # BEGIN handleIngressVlanMap
         assert ingressVlanId and ingressMappedVlan
         egressVlanId, egressInnnerVlanId, egressMappedVlan = \
               egressVlanMap( ingressVlanId, ingressInnerVlanId, ingressMappedVlan )
         assert bool( egressVlanId ) == bool( egressMappedVlan )
         renderVlanMap( ingressVlanId, ingressInnerVlanId, ingressMappedVlan,
                        egressVlanId, egressInnnerVlanId, egressMappedVlan )

         if egressMappedVlan:
            egressSkip.add( ( egressVlanId, egressInnnerVlanId ) )

      def handleEgressVlanMap( egressSkip, egressVlanId, egressInnnerVlanId,
                               egressMappedVlan ):
         assert egressVlanId and egressMappedVlan
         if ( egressVlanId, egressInnnerVlanId ) not in egressSkip:
            renderVlanMap( None, None, None,
                           egressVlanId, egressInnnerVlanId, egressMappedVlan )

      # BEGIN render
      # print header
      fmt = "%-13.11s %-13.11s %-11.9s %-11.9s %-13.11s %-13.11s %-13.11s"
      print( fmt % ( "", "", "", "", "Direction", "Direction", "" ) )
      print( fmt % ( "Outer Tag", "Inner Tag", "VLAN ID", "Status", "Configured",
                     "Active", "Dot1qTunnel" ) )
      print( fmt %
             ( '-' * 11, '-' * 11, '-' * 9, '-' * 9, '-' * 11, '-' * 11, '-' * 11 ) )
      egressSkip = set()
      for ingressVlanId, ingressMappedVlan in \
            sorted( self.ingressVlanMappings.items() ):
         handleIngressVlanMap( egressSkip, ingressVlanId, None, ingressMappedVlan )

      if self.ingressDoubleTaggedMappings:
         for ingressVlanId, innerTagMappings in \
               sorted( self.ingressDoubleTaggedMappings.items() ):
            for ingressInnerVlanId, ingressMappedVlan in \
                  sorted( innerTagMappings.mappings.items() ):
               handleIngressVlanMap( egressSkip, ingressVlanId, ingressInnerVlanId,
                                     ingressMappedVlan )

      for egressVlanId, egressMappedVlan in \
            sorted( self.egressVlanMappings.items() ):
         handleEgressVlanMap( egressSkip, egressVlanId, None, egressMappedVlan )

      if self.egressDoubleTaggedMappings:
         for egressVlanId, innerTagMappings in \
               sorted( self.egressDoubleTaggedMappings.items() ):
            for egressInnnerVlanId, egressMappedVlan in \
                  sorted( innerTagMappings.mappings.items() ):
               handleEgressVlanMap( egressSkip, egressVlanId, egressInnnerVlanId,
                                    egressMappedVlan )

class VlanMappings( Model ):
   ''' Model to communicate the currently configured vlan mappings '''
   intfVlanMappings = Dict( valueType=InterfaceVlanMappings,
                            help='Collection of VLAN mappings, keyed by interface.' )

   def render( self ):
      for intf in Arnet.sortIntf( self.intfVlanMappings ):
         intfVlanMappings = self.intfVlanMappings[ intf ]
         if ( intfVlanMappings.ingressVlanMappings or
              intfVlanMappings.ingressDoubleTaggedMappings or
              intfVlanMappings.egressVlanMappings or
              intfVlanMappings.egressDoubleTaggedMappings or
              intfVlanMappings.ingressVlanMappingRequired ):
            print( "%-25s" % ( '-' * 14 ) )
            if intfVlanMappings.ingressVlanMappingRequired:
               print( "%-57s" %
                      ( str( intf ) + " (Ingress VLAN Mapping Required)" ) )
            else:
               print( "%-25s" % str( intf ) )
            intfVlanMappings.render()
            print( "" )

class FlexEncapTaggedValue( Model ):
   """
   Model for a tagged flexible encapsulation value. This will later be extended
   to support ranges.
   """
   tag = Int( help='VLAN tag', optional=True )

class FlexEncapValue( Model ):
   """
   Model for flexible encapsulation value
   eg.
      valueType dot1q: 100,200
      valueType client: client
   """
   valueType = Enum( values=( 'dot1q', 'client' ), help='Type of encapsulation' )
   # Only applicable for tagged valueTypes (eg. dot1q)
   tags = List( valueType=FlexEncapTaggedValue, help='VLAN tags', optional=True )

class FlexEncapTaggedSpec( Model ):
   """
   Model for flexible encapsulation tagged encapsulation
   eg.
      outer 100
      outer 100 inner 200
   """
   outer = Submodel( valueType=FlexEncapValue, help='Outer VLAN tags' )
   inner = Submodel( valueType=FlexEncapValue, help='Inner VLAN tags',
                     optional=True )

class FlexEncapSpec( Model ):
   """
   Model for flexible encapsulation specification
   eg.
      specType tagged: outer 100 inner 200
      specType client: client
   """
   specType = Enum( values=( 'tagged', 'client' ), help='Type of encapsulation' )
   # only applicable for tagged specTypes
   tagged = Submodel( valueType=FlexEncapTaggedSpec, help='Tagged encapsulation',
                      optional=True )

class FlexEncapEntry( Model ):
   """
   Model for flexible encapsulation entry
   eg. client dot1q outer 100 inner 200 network client
   """
   status = Enum( values=( 'active', 'inactive' ), help='VLAN encapsulation status' )
   client = Submodel( valueType=FlexEncapSpec, help='Client side encapsulation' )
   network = Submodel( valueType=FlexEncapSpec, help='Network side encapsulation',
                       optional=True )

class FlexibleEncapsulations( Model ):
   """
   Top level model for flexible encapsulation (ie. encapsulation vlan submode)
   eg. interface et1.1
         encapsulation vlan
            client dot1q outer 100 inner 200 network client
   """
   entries = List( valueType=FlexEncapEntry, help='VLAN encapsulation entries' )

class EncapsulationVlans( Model ):
   """ Top level model for show interface encapsulation vlan """
   intfEncapsulations = Dict( valueType=FlexibleEncapsulations,
                              help='Interface VLAN encapsulation status' )

   def renderFlexEncapTaggedValue( self, taggedValue ):
      """
      Helper method to render FlexEncapTaggedValue
      """
      return str( taggedValue.tag )

   def renderFlexEncapValue( self, value ):
      """
      Helper method to render FlexEncapValue
      eg.
      valueType: dot1q
         100,200,300
      valueType: client
         client
      """
      if value.valueType == 'dot1q':
         return ','.join( map( self.renderFlexEncapTaggedValue, value.tags ) )
      else:
         assert value.valueType == 'client', \
               'Unexpected valueType {}'.format( value.valueType )
         return 'client'

   def renderFlexEncapSpec( self, spec ):
      """
      Helper method to render FlexEncapSpec
      eg.
      specType: tagged
         dot1q outer 100 inner 200
      specType: client
         client
      """
      if spec.specType == 'tagged':
         # We only support dot1q at the moment, so we will unconditionally
         # prefix the spec with dot1q
         outerValue = self.renderFlexEncapValue( spec.tagged.outer )
         ret = 'dot1q outer {}'.format( outerValue )
         if spec.tagged.inner:
            innerValue = self.renderFlexEncapValue( spec.tagged.inner )
            ret += ' inner {}'.format( innerValue )
         return ret
      elif spec.specType == 'client':
         return 'client'
      else:
         assert False, 'Unexpected specType {}'.format( spec.specType )
         return ''

   def renderFlexEncap( self ):
      """
      Helper method to render flexible encapsulations (Vlan encapsulations)
      eg.
      Interface         Status       Client Encapsulation       Network Encapsulation
      ----------------- ------------ -------------------------- ---------------------
      Ethernet1.1       up           dot1q outer 100            client
      """

      fmt = Format( justify="left" )
      fmt.noPadLeftIs( True )
      headings = (
         'Interface',
         'Status',
         'Client Encapsulation',
         'Network Encapsulation',
      )
      table = createTable( headings )
      table.formatColumns( fmt, fmt, fmt, fmt )
      printTable = False
      for intf in Arnet.sortIntf( self.intfEncapsulations ):
         flexEncap = self.intfEncapsulations[ intf ]
         printIntf = True
         for entry in flexEncap.entries:
            # There can be multiple client entries for one particular interface
            if printIntf:
               intfStr = intf
               printIntf = False
            else:
               intfStr = ''
            # Client encapsulation is mandatory
            client = self.renderFlexEncapSpec( entry.client )
            # Network encapsulation is optional
            if entry.network:
               network = self.renderFlexEncapSpec( entry.network )
            else:
               network = ''
            table.newRow( intfStr, entry.status, client, network )
            printTable = True
      if printTable:
         return table.output()
      else:
         return ''

   def render( self ):
      print( self.renderFlexEncap() )

class EncapsulationDot1QVlans( Model ):
   ''' Model to communicate the currently configured encapsulation dot1q VLAN '''
   class InterfaceEncapsulations( Model ):
      vlanId = Str( help='The dot1q encapsulation VLAN Id.' )
      egressActive = Bool( help = "Whether the encapsulation is active on egress" )
      ingressActive = Bool( help = "Whether the encapsulation is active on ingress" )

   intfEncapsulations = Dict( valueType=InterfaceEncapsulations,
                              help='The currently configured encapsulation vlan, '
                                   'keyed by interface name.' )

   def render( self ):
      """
      Interface         Encapsulation Vlan    Status     Direction
      ---------------   -------------------   --------   ----------
      Ethernet1         100                   Active     In/Out
      """
      printHeader = True
      fmt = "%-17.16s %-21.20s %-10.9s %-14.11s"
      header = ( "Interface", "Encapsulation Vlan", "Status", "Direction" )
      for intf in Arnet.sortIntf( self.intfEncapsulations ):
         encap = self.intfEncapsulations[ intf ]
         active = encap.ingressActive or encap.egressActive
         status = "Active" if active else "Inactive"
         # only display interfaces that have something interesting going on
         if active:
            if printHeader:
               print( fmt % header )
               print( fmt % ( '-' * 15, '-' * 19, '-' * 8, '-' * 10 ) )
               printHeader = False
            direction = getDirection( encap.ingressActive, encap.egressActive )
            print( fmt % ( intf, encap.vlanId, status, direction ) )

class VlanSummary( Model ):
   sourceDetail = Vlans.sourceDetail
   totalVlans = Int( help="Number of existing VLANs" )

   def render( self ):
      if self.sourceDetail:
         print( self.sourceDetail )
      print( 'Number of existing VLANs           : %d' % self.totalVlans )
      print()

class VlanCounters( Model ):
   class VlanStats( Model ):
      inOctets = Int( help="Input octets" )
      inUcastPkts = Int( help="Input unicast packets" )
      inMcastPkts = Int( help="Input multicast packets" )
      inBcastPkts = Int( help="Input broadcast packets" )

      outOctets = Int( help="Output octets" )
      outUcastPkts = Int( help="Output unicast packets" )
      outMcastPkts = Int( help="Output multicast packets" )
      outBcastPkts = Int( help="Output broadcast packets" )

   _displayIngressCounters = Bool(
         help="True if VLAN supports input counters", default=True )
   _displayEgressCounters = Bool(
         help="True if VLAN supports output counters", default=True )
   _displayBumCounters = Bool(
         help="True if VLAN supportes BUM counters", default=True )
   _inactiveVlans = List( valueType=int, help="List of inactive VLANs" )
   vlanCountersInfo = Dict( keyType=int, valueType=VlanStats,
                            help="A mapping between VLAN ID and counter data" )

   def printRow( self, intf, totalOctets, ucastPkts, mcastPkts=None,
         bcastPkts=None ):
      if not mcastPkts and not bcastPkts:
         counterDisplayFormat = "%-10s %18s %15s"
         print( counterDisplayFormat % ( intf, totalOctets, ucastPkts ) )
      else:
         counterDisplayFormat = "%-10s %18s %15s %15s %15s"
         print( counterDisplayFormat %
                ( intf, totalOctets, ucastPkts, mcastPkts, bcastPkts ) )

   def printHeader( self, direction='in' ):
      if direction == 'in':
         if self._displayBumCounters:
            self.printRow( "Vlan", "InOctets", "InUcastPkts",
                  "InBcastPkts", "InMcastPkts" )
         else:
            self.printRow( "Vlan", "InOctets", "InPkts" )

      elif direction == 'out':
         if self._displayBumCounters:
            self.printRow( "Vlan", "OutOctets", "OutUcastPkts",
                  "OutBcastPkts", "OutMcastPkts" )
         else:
            self.printRow( "Vlan", "OutOctets", "OutPkts" )

   def printIncoming( self, vlanId ):
      vlanName = "Vlan" + str( vlanId )
      if self._displayBumCounters:
         self.printRow( vlanName,
                        self.vlanCountersInfo[ vlanId ].inOctets,
                        self.vlanCountersInfo[ vlanId ].inUcastPkts,
                        self.vlanCountersInfo[ vlanId ].inMcastPkts,
                        self.vlanCountersInfo[ vlanId ].inBcastPkts )
      else:
         self.printRow( vlanName,
                        self.vlanCountersInfo[ vlanId ].inOctets,
                        self.vlanCountersInfo[ vlanId ].inUcastPkts )

   def printOutgoing( self, vlanId ):
      vlanName = "Vlan" + str( vlanId )
      if self._displayBumCounters:
         self.printRow( vlanName,
                        self.vlanCountersInfo[ vlanId ].outOctets,
                        self.vlanCountersInfo[ vlanId ].outUcastPkts,
                        self.vlanCountersInfo[ vlanId ].outMcastPkts,
                        self.vlanCountersInfo[ vlanId ].outBcastPkts )
      else:
         self.printRow( vlanName,
                        self.vlanCountersInfo[ vlanId ].outOctets,
                        self.vlanCountersInfo[ vlanId ].outUcastPkts )

   def vlanCountersEnabled( self ):
      return ( self._displayIngressCounters or
               self._displayEgressCounters or
               self._displayBumCounters )

   def render( self ):
      if not self.vlanCountersEnabled():
         print( "Hardware VLAN Counters: Disabled" )
         return

      if not self.vlanCountersInfo:
         return

      for vlanId in self._inactiveVlans:
         print( "Vlan%d is not active." % vlanId )

      if self._displayIngressCounters:
         self.printHeader( 'in' )
         for vlanId in sorted( self.vlanCountersInfo.keys() ):
            self.printIncoming( vlanId )
         print()

      if self._displayEgressCounters:
         self.printHeader( 'out' )
         for vlanId in sorted( self.vlanCountersInfo.keys() ):
            self.printOutgoing( vlanId )
         print()

class VlanIds( Model ):
   vlanIds = List( valueType=int, help="A list of the VLAN Ids" )

   def getVlanStr( self ):
      if not self.vlanIds:
         return "NONE"

      vlanStr = vlanSetToCanonicalString( self.vlanIds )
      if vlanStr == '1-4094':
         return "ALL"

      return vlanStr

class VlanIntfs( Model ):
   interfaces = List( valueType=IntfModels.Interface,
                      help="A list of the VLAN interfaces" )

   def getIntfsStr( self ):
      if not self.interfaces:
         return "NONE"
      return ", ".join( sorted( self.interfaces ) )

class VlanDynamic( Model ):
   dynamicVlans = Dict( valueType=VlanIds, help="A Mapping between the Dynamic VLAN "
                                                "source and the VLANS" )

   def render( self ):
      fmt = "%-20s      %s"
      print( fmt % ( "Dynamic VLAN source", "VLANS" ) )
      for agent in sorted( self.dynamicVlans ):
         print( fmt % ( agent, self.dynamicVlans[ agent ].getVlanStr() ) )

class VlanInterfaceDynamic( Model ):
   agents = Dict( valueType=VlanIntfs,
                                 help="A Mapping between the Dynamic VLAN "
                                 " interface source and the VLANS" )

   def render( self ):
      table = TableFormatter()
      headings = ( 'Dynamic source', 'VLAN interfaces' )
      Headings( headings ).doApplyHeaders( table )

      sourceFmt = Format( justify="left", minWidth=16, maxWidth=25 )
      intfFmt = Format( justify="left", minWidth=16, maxWidth=60, wrap=True )

      table.formatColumns( sourceFmt, intfFmt )

      for agent in sorted( self.agents ):
         table.newRow( agent, self.agents[ agent ].getIntfsStr() )

      print( table.output() )

class VlanInternalAllocationPolicy( Model ):
   policy = Enum( values=( "ascending", "descending" ),
                  help="Internal VLAN allocation policy" )
   startVlanId = Int( help="Starting VLAN Id" )
   endVlanId = Int( help="Ending VLAN Id" )

   def render( self ):
      print( "Internal VLAN Allocation Policy: %s" % self.policy )
      print( "Internal VLAN Allocation Range: %d-%d" %
             ( self.startVlanId, self.endVlanId ) )

class VlanInternalUsage( Model ):
   internalVlans = Dict( keyType=int, valueType=str,
                         help="Mapping between the VLAN Id and the VLAN Name" )

   def render( self ):
      for vlanId in sorted( self.internalVlans ):
         print( "%d  %s" % ( vlanId, self.internalVlans[ vlanId ] ) )

class VlanInternalUsageReserved( Model ):
   internalVlans = Dict( keyType=int, valueType=str,
                         help="Mapping between the VLAN Id and the corresponding "
                         "Interface Name" )

   def render( self ):
      for vlanId in sorted( self.internalVlans ):
         intfName, tag = self.internalVlans[ vlanId ].split( ':' )
         print( "%d %s 802.1Q VLAN %s" % ( vlanId, intfName, tag ) )

class TrunkVlans( VlanIds ):
   allVlans = Bool( help="When set it means all vlans 1-4094 in which case the "
                    " vlanIds will be empty", optional=True )

class InterfaceTrunk( Model ):
   portMode = Enum( values=( "access", "trunk", "dot1qTunnel", "tap", "tool",
                             "tapTool", "routed" ),
                    help="Port mode of the trunk", optional=True )
   phoneTrunk = Bool( help="Mode is 'trunk phone'", optional=True )
   linkStatus = Enum( values=( "nonTrunking", "down", "trunking" ),
                      help="Trunk link status for the interface", optional=True )
   nativeVlan = Int( help="Native vlan of the trunk", optional=True )
   allowedVlans = Submodel( valueType=TrunkVlans,
                            help="Vlans allowed on the trunk. No value means no "
                            "information is available and array with one element of "
                            "0 means no vlans are allowed",
                            optional=True )
   activeVlans = Submodel( valueType=TrunkVlans,
                           help="Vlans allowed and active in management domain",
                           optional=True )
   forwardingVlans = Submodel( valueType=TrunkVlans,
                               help="Vlans in spannning tree forwarding state",
                               optional=True )


class TrunkInterfaces( Model ):
   _noIntfs = Bool( help="Attribute to tell if there are any interfaces or not" )
   trunks = Dict( keyType=IntfModels.Interface, valueType=InterfaceTrunk,
                  help="Mapping from an interface name to its' trunk information" )

   def render( self ):

      # We need this private attribute to figure out whether there were no
      # interfaces found from the show command or actually there are no active
      # trunk ports. The original CLI checks for list of interfaces and returns
      # siliently if none found or print this message below if no active trunks
      # are found. We need to differentiate between the two cases and hence this
      # attribute
      if self._noIntfs:
         return

      if not self.trunks:
         print( 'There are no active trunk ports' )
         return

      # print summary with status, mode and native vlan
      fmt = '%-15s %-15s %-15s %s'
      sortedIntfs = Arnet.sortIntf( self.trunks )
      print( fmt % ( 'Port', 'Mode', 'Status', 'Native vlan' ) )
      for intf in sortedIntfs:
         trunk = self.trunks[ intf ]
         native = ( trunk.nativeVlan if trunk.nativeVlan else "-" if
                    trunk.portMode == "tool" else "None" )
         portname = IntfCli.Intf.getShortname( intf )
         linkStatus = "non-trunking" if trunk.linkStatus == "nonTrunking" else\
             trunk.linkStatus if trunk.linkStatus else "-"
         if trunk.portMode=="trunk" and trunk.phoneTrunk:
            portMode = "trunk phone"
         else:
            portMode = trunk.portMode if trunk.portMode else "-"
         print( fmt % ( portname, portMode, linkStatus, native ) )
      print( '' )

      # Allowed ports information
      fmt = '%-15s %-s'
      strLimit = 50
      print( fmt % ( 'Port', 'Vlans allowed' ) )
      for intf in sortedIntfs:
         trunk = self.trunks[ intf ]
         allowedVlans = trunk.allowedVlans

         # Vlan 0 means no vlans are allowed
         if not allowedVlans:
            allowed = [ 'None' ]
         elif allowedVlans.allVlans:
            allowed = [ 'All' ]
         elif ( not allowedVlans.vlanIds or not allowedVlans.vlanIds[ 0 ] ):
            allowed = [ 'None' ]
         else:
            allowed = vlanSetToCanonicalStringGen( allowedVlans.vlanIds, strLimit )

         portname = IntfCli.Intf.getShortname( intf )
         for s in allowed:
            print( fmt % ( portname, s ) )
            portname = ''
      print()

      # Active ports information
      print( fmt % ( 'Port', 'Vlans allowed and active in management domain' ) )
      for intf in sortedIntfs:
         trunk = self.trunks[ intf ]
         activeVlans = trunk.activeVlans

         if not activeVlans:
            active = [ 'None' ]
         elif activeVlans.allVlans:
            active = [ 'All' ]
         elif ( not activeVlans.vlanIds or not activeVlans.vlanIds[ 0 ] ):
            active = [ 'None' ]
         else:
            active = vlanSetToCanonicalStringGen( activeVlans.vlanIds, strLimit )

         portname = IntfCli.Intf.getShortname( intf )
         if trunk.linkStatus == 'down':
            print( fmt % ( portname, 'None' ) )
            continue
         for s in active:
            print( fmt % ( portname, s ) )
            portname = ''
      print()

      # Spanning tree information
      print( fmt % ( 'Port', 'Vlans in spanning tree forwarding state' ) )
      for intf in sortedIntfs:
         trunk = self.trunks[ intf ]
         tagged = trunk.forwardingVlans

         if not tagged:
            fwdStateVlans = [ 'None' ]
         elif tagged.allVlans:
            fwdStateVlans = [ 'All' ]
         elif ( not tagged.vlanIds or not tagged.vlanIds[ 0 ] ):
            fwdStateVlans = [ 'None' ]
         else:
            fwdStateVlans = vlanSetToCanonicalStringGen( tagged.vlanIds, strLimit )

         portname = IntfCli.Intf.getShortname( intf )
         if trunk.linkStatus == 'down':
            print( fmt % ( portname, 'None' ) )
            continue
         for s in fwdStateVlans:
            print( fmt % ( portname, s ) )
            portname = ''

class EtreeRoleModel( Model ):
   etreeRoleLeaf = Bool( help='True if this VLAN has the E-Tree leaf role' )

   def displayString( self ):
      return 'Leaf' if self.etreeRoleLeaf else 'Root'

class VlanEtreeRolesModel( Model ):
   vlans = Dict( keyType=int, valueType=EtreeRoleModel,
         help="A VLAN's E-Tree role, keyed by VLAN ID" )

   def render( self ):
      tableHeadings = [ ( 'VLAN', 'l' ), ( 'E-Tree Role', 'l' ) ]
      table = createTable( tableHeadings )
      for vlan, role in sorted( self.vlans.items() ):
         table.newRow( vlan, role.displayString() )
      print( table.output() )
