# Copyright (c) 2014 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.
import IntfCli, Tac
from CliModel import Float, Submodel, Model, Int, Str
from CliMode.Intf import IntfMode
from IntfModel import InterfaceStatus
from IntfModel import InterfaceCountersBase
from IntfModel import InterfaceCountersRateBase
from Ark import timestampToStr
from HumanReadable import scaleValueSi, formatValueSigFigs

#-------------------------------------------------------------------------------
# VlanIntf specific Capi Models
#-------------------------------------------------------------------------------
class VlanInterfaceCounters( InterfaceCountersBase ):
   lastClear = Float( help='Last clear of interface counters', optional=True )
   counterRefreshTime = Float( help="UTC of last counter refresh", optional=True )
   linkStatusChanges = Int( help='Number of link status changes since last clear',
                            optional=True )

   inOctets = Int( help="Input octets", optional=True )
   inUcastOctets = Int( help="Input unicast octets", optional=True )
   inMulticastOctets = Int( help="Input multicast octets", optional=True )
   inPkts = Int( help="Input packets", optional=True )
   inUcastPkts = Int( help="Input unicast packets", optional=True )
   inMulticastPkts = Int( help="Input multicast packets", optional=True )

   outOctets = Int( help="Output octets", optional=True )
   outUcastOctets = Int( help="Output unicast octets", optional=True )
   outMulticastOctets = Int( help="Output multicast octets", optional=True )
   outPkts = Int( help="Output packets", optional=True )
   outUcastPkts = Int( help="Output unicast packets", optional=True )
   outMulticastPkts = Int( help="Output multicast packets", optional=True )

   def renderLastClearCounter( self ):
      if self.lastClear is not None:
         lastClear = timestampToStr( self.lastClear  - Tac.utcNow() + Tac.now() )
      else:
         lastClear = 'never'
      print '  Last clearing of "show interface" counters %s' % lastClear

   def renderLinkStatusChanges( self ):
      print '  %d link status changes since last clear' % self.linkStatusChanges

   def renderCounters( self ):
      self._renderInputStatistics()
      self._renderOutputStatistics()

   def constructCntDict( self ):
      # order in this dictionary is reflected in CLI
      return {
               'in' :
                  [ 
                     ( "InOctets",      self.inOctets ),
                     ( "InUcastOctets", self.inUcastOctets ),
                     ( "InMcastOctets", self.inMulticastOctets ),
                     ( "InPkts",        self.inPkts ),
                     ( "InUcastPkts",   self.inUcastPkts ),
                     ( "InMcastPkts",   self.inMulticastPkts ),
                  ],
               'out':
                  [
                     ( "OutOctets",      self.outOctets ),
                     ( "OutUcastOctets", self.outUcastOctets ),
                     ( "OutMcastOctets", self.outMulticastOctets ),
                     ( "OutPkts",        self.outPkts ),
                     ( "OutUcastPkts",   self.outUcastPkts ),
                     ( "OutMcastPkts",   self.outMulticastPkts ),
                  ]
            }

   def printRow( self, cList, header=False ):
      def checkValue( value ):
         return value if value is not None else 'n/a'
      
      counterDisplayFormat = "%-10s" % \
            ( "Port" if header else IntfMode.getShortname( self._name ) )

      for ( cntName, val ) in cList:
         if val is not None:
            printVal = checkValue( cntName ) if header else checkValue( val )
            counterDisplayFormat += " %18s" % printVal
      print counterDisplayFormat

   def renderHeader( self, direction='in' ):
      self.printRow( self.constructCntDict().get( direction ), header=True ) 

   def renderIncoming( self ):
      self.printRow( self.constructCntDict().get( 'in' ) ) 

   def renderOutgoing( self ):
      self.printRow( self.constructCntDict().get( 'out' ) ) 

   def _renderInputStatistics( self ):
      if self.inPkts is not None and self.inOctets is not None:
         print '     %d packet, %d bytes input' % ( self.inPkts, self.inOctets )
      elif self.inOctets is not None:
         print '     %d bytes input' % ( self.inOctets )
      elif self.inPkts is not None:
         print '     %d packets input' % ( self.inPkts )

      if self.inUcastOctets is not None and self.inMulticastOctets is not None: 
         print '     Received %d unicast, %d multicast bytes' % (
               self.inUcastOctets, self.inMulticastOctets )
      elif self.inUcastOctets is not None:
         print '     Received %d unicast bytes' % ( self.inUcastOctets )
      elif self.inMulticastOctets is not None:
         print '     Received %d multicast bytes' % ( self.inMulticastOctets )

      if self.inUcastPkts is not None and self.inMulticastPkts is not None:
         print '     Received %d unicast, %d multicast packets' % (
               self.inUcastPkts, self.inMulticastPkts )
      elif self.inUcastPkts is not None:
         print '     Received %d unicast packets' % ( self.inUcastPkts )
      elif self.inMulticastPkts is not None:
         print '     Received %d multicast packets' % ( self.inMulticastPkts )

   def _renderOutputStatistics( self ):
      if self.outPkts is not None and self.outOctets is not None:
         print '     %d packets, %d bytes output' % ( self.outPkts, self.outOctets )
      elif self.outOctets is not None:
         print '     %d bytes output' % ( self.outOctets )
      elif self.outPkts is not None:
         print '     %d packets output' % ( self.outPkts )

      if self.outUcastOctets is not None and self.outMulticastOctets is not None: 
         print '     Sent %d unicast, %d multicast bytes' % (
               self.outUcastOctets, self.outMulticastOctets )
      elif self.outUcastOctets is not None:
         print '     Sent %d unicast bytes' % ( self.outUcastOctets )
      elif self.outMulticastOctets is not None:
         print '     Sent %d multicast bytes' % ( self.outMulticastOctets )

      if self.outUcastPkts is not None and self.outMulticastPkts is not None:
         print '     Sent %d unicast, %d multicast packets' % (
               self.outUcastPkts, self.outMulticastPkts )
      elif self.outUcastPkts is not None:
         print '     Sent %d unicast packets' % ( self.outUcastPkts )
      elif self.outMulticastPkts is not None:
         print '     Sent %d multicast packets' % ( self.outMulticastPkts )

# Vlan Interface rate counters
class VlanInterfaceCountersRate( InterfaceCountersRateBase ):
   description = Str( help="Port description" )
   interval = Int( help="Interval in seconds" )

   inBpsRate = Float( help="Input bps rate", optional=False )
   outBpsRate = Float( help="Output bps rate", optional=False )

   def renderHeader( self ):
      if self.inBpsRate is None and self.outBpsRate is None: 
         return

      header = '%-9s %-11.11s %5s' % ( 'Port', 'Name',
                                               'Intvl' ) 
      if self.inBpsRate is not None:
         header += '  %8s' % 'In Mbps'

      if self.outBpsRate is not None:
         header += '  %8s' % 'Out Mbps'

      print header

   def renderRates( self ):
      if self.inBpsRate is None and self.outBpsRate is None: 
         return

      minutes = self.interval / 60
      seconds = self.interval % 60
      formatStr = '%-9s %-11.11s %2d:%02d' % ( IntfMode.getShortname( self._name ),
                          self.description, minutes, seconds )

      if self.inBpsRate is not None:
         inMbpsRate = self.inBpsRate / 1000.0 / 1000.0
         formatStr += '  %8.1f' % inMbpsRate

      if self.outBpsRate is not None:
         outMbpsRate = self.outBpsRate / 1000.0 / 1000.0
         formatStr += '  %8.1f' % outMbpsRate

      print formatStr 

class VlanInterfaceStatistics( Model ):
   updateInterval = Float( help='Stats update interval string' )
   inBitsRate = Float( help='input bits rate', optional=False )
   outBitsRate = Float( help='output bits rate', optional=False )
   
   def printRate( self, direction, bps, bandwidth ):
      ( scaledBps, bpsPrefix ) = scaleValueSi( bps )

      if bpsPrefix:
         formattedBps = formatValueSigFigs( scaledBps, 3 )
      else:
         formattedBps = '%.0f' % scaledBps

      fmt = '  %s %s rate %s %sbps'
      intervalStr = IntfCli.getLoadIntervalPrintableString( self.updateInterval )
      print fmt % ( intervalStr, direction, formattedBps, bpsPrefix )
   
   def renderStatistics( self, bandwidth ):
      if self.inBitsRate is not None:
         self.printRate( 'input', self.inBitsRate, bandwidth )
      if self.outBitsRate is not None:
         self.printRate( 'output', self.outBitsRate, bandwidth )
   
class VlanInterfaceStatus( InterfaceStatus ):  
   interfaceCounters = Submodel( valueType=VlanInterfaceCounters, optional=True,
                                 help='Interface Counters and related information')
   interfaceStatistics = Submodel( valueType=VlanInterfaceStatistics, optional=True, 
                                   help='Interface statistics, such as rates' )
   
   def renderLinkTypeSpecific( self ):
      raise NotImplementedError()
         
   def render( self ):
      self.renderHeader()
      self.renderHardware()
      self.renderInterfaceAddress()
      self.renderMtuMruAndBw()
      self.renderUptime()
      self.renderMaintEnterTime()
      
      if self.interfaceCounters:
         self.interfaceCounters.renderLinkStatusChanges()
         self.interfaceCounters.renderLastClearCounter()
      if self.interfaceStatistics:
         self.interfaceStatistics.renderStatistics( self.bandwidth )
      if self.interfaceCounters:
         self.interfaceCounters.renderCounters()
