# Copyright (c) 2018 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

# pkgdeps: library SecureBoot

import os

import Cell
import CliModel
import CliPlugin.ReloadImageEpochCheckCli
import CliPlugin.Reload2GbImageCheckCli
import LazyMount
import ProductAttributes
import ReloadPolicy

abootSbStatus = None

#--------------------------------------------------------------------------------
# show version compatibility SWI
#--------------------------------------------------------------------------------
class VersionCompatibility( CliModel.Model ):
   errorList = CliModel.List( valueType=str,
                              help='List of compatibility blocking errors' )
   warningList = CliModel.List( valueType=str,
                                help='List of compatibility warnings' )
   image = CliModel.Str( help='Software image URL' )

   def render( self ):
      if self.errorList:
         print ( "Image is incompatible with the current configuration due to"
                 " the following errors:" )
         for error in self.errorList:
            print " ", error
      if self.warningList:
         print ( "Image might be incompatible with the current configuration due to"
                 " the following warnings:" )
         for warning in self.warningList:
            print " ", warning

class CommandFailedException( Exception ):
   pass

class CompatibilityChecker( object ):
   '''Contains all logic needed to verify SWI version compatibility. Create an
   instance of this class and call run() in order to run the compatibility checks.
   '''

   def __init__( self ):
      self.mode_ = None
      self.tmpSwiFile_ = None

   def _getLocalSwiFile( self, swiUrl ):
      try:
         swiName = os.path.basename( swiUrl.url )
         swiUrl.open()
         return swiUrl.localFilename(), swiName
      except ValueError as e:
         raise CommandFailedException( 'Error: %s' % repr( e ) )
      except IOError as e:
         raise CommandFailedException( 'Error: %s' % repr( e ) )

   def run( self, mode, swiUrl ):
      self.mode_ = mode
      retVal = True
      result = None
      category = [ 'General' ]

      try:
         localSwi, swiName = self._getLocalSwiFile( swiUrl )

         # Check for ASU support. If ASU is supported, run the ASU checks as well.
         # For now, support ASU2.
         if ProductAttributes.productAttributes().bootAttributes.asuHitlessSupported:
            category += [ 'ASU' ]

         # Run ReloadPolicy checks from the image.
         result = ReloadPolicy.doCheck( localSwi, category=category,
                                        mode=self.mode_,
                                        abootSbStatus=abootSbStatus )

         # Run existing general compatibility checks.
         epochComp, epochErr = (
               CliPlugin.ReloadImageEpochCheckCli.checkImageEpochAllowed(
               self.mode_, localSwi, swiName ) )
         if not epochComp:
            result.addError( epochErr )
         twoGbComp, twoGbErr = (
               CliPlugin.Reload2GbImageCheckCli.checkImage2GbCompatible(
               self.mode_, localSwi, swiName ) )
         if not twoGbComp:
            result.addError( twoGbErr )

      except CommandFailedException as e:
         retVal = False
         self.mode_.addError( str( e ) )
      return retVal, result

def showVersionCompatibility( mode, args ):
   '''Download the Swi at swiUrl to the switch. Run version compatibility checks on
   the Swi and print out errors and warnings.
   '''
   swiUrl = args.get( 'SWI' )
   compatibilityChecker = CompatibilityChecker()
   success, result = compatibilityChecker.run( mode, swiUrl )
   errors = []
   warnings = []

   if success:
      errors = result.errors
      warnings = result.warnings
      if not errors and not warnings:
         print 'Image is compatible with the current configuration.'
   else:
      print 'Version compatibility check failed.'

   return VersionCompatibility( errorList=errors,
                                warningList=warnings,
                                image=swiUrl.url )

def Plugin( entityManager ):
   global abootSbStatus
   abootSbStatus = LazyMount.mount( entityManager, Cell.path( "aboot/sb/status" ),
                                    "Aboot::Secureboot::Status", "r" )
