# Copyright (c) 2006-2010 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import os
import time

import BasicCli
import CliMatcher
import EosVersion
import ShowCommand
import Tac
import TacSigint
import Tracing
import LazyMount
import OnieVersionLib

from .VersionModel import Version
from .VersionModel import Licenses

__defaultTraceHandle__ = Tracing.Handle( 'VersionCli' )
t0 = Tracing.trace0

hwEpochStatus_ = None
onieStatus_ = None
hwEntMib_ = None
sysMacAddrConfig_ = None

versionKwMatcher = CliMatcher.KeywordMatcher(
   'version',
   helpdesc='Software and hardware versions' )

#-----------------------------------------------------------------------------------
# show version [ detail ]
#-----------------------------------------------------------------------------------
# We do not count meminfo[ 'Cached' ] as free memory because this number
# includes unreclaimable space from tmpfs/ramfs filesystems
def _getMemUsage():
   meminfo = file( '/proc/meminfo' ).read().strip().split( '\n' )
   meminfo = ( m.split() for m in meminfo )
   meminfo = dict( ( m[ 0 ][ :-1 ], int( m[ 1 ] ) ) for m in meminfo )
   if not meminfo.has_key( "MemAvailable" ):
      return ( meminfo[ "MemTotal" ], meminfo[ "MemFree" ] )
   else:
      return ( meminfo[ "MemTotal" ], meminfo[ "MemAvailable" ] )

def _getPackages():
   resultPackage = {}
   packageInfo = Tac.run( [ 'rpm', '-qa', '--qf', '%-20{N} %-15{V} %{R}\n' ],
                          stdout=Tac.CAPTURE )
   packages = packageInfo.strip().split( '\n' )
   packages.sort()
   packages = ( package.split() for package in packages )
   for package in packages:
      if not package:
         continue
      p, v, r = package
      resultPackage[ p ] = Version.Details.Package( version=v, release=r )

   return resultPackage

def _getNewComponent( name, version ):
   return Version.Details.Component( name=name, version=version )

def _getChassisComponents( entityMibRoot ):
   components = []
   cardMibs = [ x.card for x in entityMibRoot.cardSlot.values( ) if x.card ]
   for cm in sorted( cardMibs,
     cmp=lambda x,y: cmp( x.tag, y.tag ) or int( x.label ) - int( y.label ) ):
      prefix = "%s%s-" % ( cm.tag, cm.label )
      if cm.firmwareRev:
         name = prefix + "Aboot"
         components.append( _getNewComponent( name, cm.firmwareRev ) )
      for chip in sorted( cm.chip.values(), key=lambda x: x.tag ):
         if chip.firmwareRev:
            name = prefix + chip.tag
            components.append( _getNewComponent( name, chip.firmwareRev ) )

   return components

def _getFixedSystemComponents( entityMibRoot ):
   components = []
   components.append( _getNewComponent( "Aboot", entityMibRoot.firmwareRev ) )
   for chip in sorted( entityMibRoot.chip.values(), key=lambda x: x.tag ):
      if chip.firmwareRev:
         components.append( _getNewComponent( chip.tag, chip.firmwareRev ) )

   return components

def _getComponents( mode ):
   entityMibRoot = mode.entityManager.lookup( 'hardware/entmib' ).root
   if entityMibRoot is None or entityMibRoot.initStatus != "ok":
      # System is not yet initialized
      return ( [], "unknown" )

   if entityMibRoot.tacType.fullTypeName == "EntityMib::Chassis":
      return ( _getChassisComponents( entityMibRoot ), "chassis" )
   elif entityMibRoot.tacType.fullTypeName == "EntityMib::FixedSystem":
      return ( _getFixedSystemComponents( entityMibRoot ), "fixedSystem" )

   return ( [], "unknown" )

def _getSystemEpoch():
   if hwEpochStatus_ and hwEpochStatus_.systemHwEpoch:
      return hwEpochStatus_.systemHwEpoch
   return "unknown"

def _getOnieVersion():
   if onieStatus_ and onieStatus_.version:
      return onieStatus_.version
   else:
      return "unknown"

def _getDetails( mode, versionInfo ):
   details = Version.Details()
   details.packages = _getPackages()
   ( details.components, details.switchType ) = _getComponents( mode )
   deviations = versionInfo.deviations()
   if deviations:
      details.deviations = deviations
   details.systemEpoch = _getSystemEpoch()
   if OnieVersionLib.getOniePlatform():
      details.onieVersion = _getOnieVersion()

   return details


def opt( var, default ):
   return var if var is not None else default

def showVersion( mode, args ):
   detail = 'detail' in args
   result = Version()

   # Some of these values are optional because they might not be initialized.
   # Usually this happens only on standby supervisors.
   vi = EosVersion.VersionInfo( mode.sysdbRoot )
   result.mfgName = opt( vi.mfgName(), "Arista" ).replace( "Arista Networks",
                                                           "Arista" )
   result.modelName = opt( vi.modelNameExtended(), "Switch (model not available)" )
   result.hardwareRevision = opt( vi.hardwareRev(), "Not available" )
   result.serialNumber = opt( vi.serialNum(), "Not available" )
   result.version = opt( vi.version(), "Not available" )
   result.architecture = opt( vi.architecture(), "(unknown)" )
   result.internalVersion = opt( vi.internalVersion(), "(unknown)" )
   result.internalBuildId = opt( vi.internalBuildId(), "(unknown)" )
   result.systemMacAddress = hwEntMib_.systemMacAddr 
   result.hwMacAddress = hwEntMib_.hwMacAddr 
   result.configMacAddress = sysMacAddrConfig_.macAddr
   curTime = time.time()
   if vi.isCEos():
      boot = os.stat( '/proc/1/cmdline' )
      result.bootupTimestamp = boot.st_ctime
      result.uptime = round( curTime - result.bootupTimestamp )
      result.cEosToolsVersion = opt( vi.cEosToolsVersion(), '(unknown)' )
      result.kernelVersion = opt( os.uname()[ 2 ], '(unknown)' )
   else:
      uptime = file( "/proc/uptime" ).read()
      result.uptime = float( uptime.split()[ 0 ] )
      result.bootupTimestamp = round( curTime - result.uptime )
   result.isIntlVersion = vi.isIntlVersion()
   ( result.memTotal, result.memFree ) = _getMemUsage()

   if detail:
      result.details = _getDetails( mode, vi )

   return result

class ShowVersionDetailCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show version [ detail ]'
   data = {
            'version': versionKwMatcher,
            'detail': 'Show additional version information',
          }
   cliModel = Version
   handler = showVersion

BasicCli.addShowCommandClass( ShowVersionDetailCmd )

#-----------------------------------------------------------------------------------
# show version license
#-----------------------------------------------------------------------------------
licenseDir = '/usr/share/licenses'
docDir = '/usr/share/doc'
def showLicense( mode, args ):
   licenses = {}
   def _walk( arg, dirname, fnames ):
      t0( "Processing", dirname, "with contents", fnames )
      for fn in fnames:
         TacSigint.check()
         # All files in licenseDir, no matter what, or
         # certain files from docDir.
         if dirname.startswith( licenseDir ) or \
               fn.startswith( "COPY" ) or fn.startswith( "LICEN" ) or \
               fn.startswith( "NOTICE" ):
            fp = dirname + "/" + fn
            if os.path.isfile( fp ):
               try:
                  licenses[ fp ] = open( fp ).read()
               except IOError:
                  mode.addError( "Failed to display file %s" % fp )

   os.path.walk( licenseDir, _walk, None )
   os.path.walk( docDir, _walk, None )

   if len( licenses ) == 0:
      mode.addError( "No software licenses found." )

   return Licenses( licenses=licenses )

class ShowVersionLicenseCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show version license'
   data = {
            'version': versionKwMatcher,
            'license': 'Show software license information',
          }
   cliModel = Licenses
   handler = showLicense

BasicCli.addShowCommandClass( ShowVersionLicenseCmd )

def Plugin( entityManager ):
   global hwEpochStatus_
   global onieStatus_
   global hwEntMib_
   global sysMacAddrConfig_
   hwEpochStatus_ = LazyMount.mount( entityManager,
                                     'hwEpoch/status', 'HwEpoch::Status', 'r' )
   onieStatus_ = LazyMount.mount( entityManager, 'onie/status', 'Onie::Status', 'r' )
   hwEntMib_ = LazyMount.mount( entityManager,  'hardware/entmib', 
                                          "EntityMib::Status", 'r' )
   sysMacAddrConfig_ = LazyMount.mount( entityManager, 'sys/macAddress/config',
                                        "Fru::MacAddressConfig", 'r' )
