# Copyright (c) 2011 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

"""This module implements the CLI for Uplink Failure Detection.
Specifically, it provides the following commands:

In Basic Cli Mode:
-  show link tracking group [GROUP-LIST]
   - legacy: show link state group [GROUP-LIST]
-  show link tracking group detail [GROUP-LIST]
   - legacy: show link state group detail [GROUP-LIST]

In Configuration Mode:
-  link tracking group GROUP
   - legacy: link state track GROUP
-  no link tracking group [GROUP-LIST]
   - legacy: no link state track [GROUP-LIST]

In Link State Config Mode
-  links minimum MIN-LINKS
-  NO/DEFAULT links minimum [TRAILING-GARBAGE]
-  recovery delay SECONDS
-  NO/DEFAULT recovery delay [TRAILING-GARBAGE]

In Interface Configuration Mode:
-  link tracking group GROUP UPSTREAM/DOWNSTREAM
   - legacy: link state group GROUP UPSTREAM/DOWNSTREAM
-  no link tracking group [GROUP-LIST]
   - legacy: no link state group [GROUP-LIST]
"""

from Ark import timestampToStr
import BasicCli
import CliCommand
import CliMatcher
from CliMode.UplinkFailureDetection import LinkStateMode
import CliParser
import CliPlugin.IntfCli as IntfCli
import ConfigMount
import LazyMount
import ShowCommand
import TableOutput

config = None
status = None

### CLI token definitions.
matcherGroup = CliMatcher.DynamicKeywordMatcher(
   lambda mode: config.group,
   value=lambda mode, match: config.group[ match ],
   emptyTokenCompletion=[ CliParser.Completion( 'WORD',
                                                'Link state group name',
                                                literal=False ) ] )

matcherGroupName = CliMatcher.DynamicNameMatcher( lambda mode: config.group,
                                                  helpdesc='Link state group name' )
# -----------------------------
# The "config-link-state" mode.
# -----------------------------
class LinkStateConfigMode( LinkStateMode, BasicCli.ConfigModeBase ):

   # ---------------------------------------
   # Attributes required of every Mode class
   # ---------------------------------------
   name = "Link state configuration"
   modeParseTree = CliParser.ModeParseTree()

   #----------------------------------------------------------------------------
   # Constructs a new LinkStateConfigMode instance for a particular Link State
   # object.
   #
   # Note that the group attribute must be set up before calling the superclass
   # constructor, since the Modelet instances created by the superclass
   # constructor need to use it.
   #----------------------------------------------------------------------------
   def __init__( self, parent, session, param ):
      self.config = config
      # Save the group name.  Need to save the group name instead of the
      # configuration entity itself to support auto-session triggering on the
      # mode entry.
      self.groupName_ = param
      LinkStateMode.__init__( self, param )
      BasicCli.ConfigModeBase.__init__( self, parent, session )

   @property
   def groupConfig( self ):
      return self.config.group[ self.groupName_ ]

   def setMinLinks( self, args ):
      self.groupConfig.minLinks = args[ 'MIN_LINKS' ]

   def resetMinLinks( self, args ):
      self.groupConfig.minLinks = self.groupConfig.minLinksDefault

   def setRecoveryDelay( self, args ):
      self.groupConfig.recoveryDelay = args[ 'DELAY' ]

   def resetRecoveryDelay( self, args ):
      self.groupConfig.recoveryDelay = self.groupConfig.recoveryDelayDefault

class LinkStateConfigCleaner( IntfCli.IntfDependentBase ):
   """This class is responsible for removing per-interface Link State config
   when the interface is deleted."""
   def setDefault( self ):
      for groupConfig in config.group:
         del config.group[ groupConfig ].intfsToMonitor[ self.intf_.name ]
         del config.group[ groupConfig ].intfsToDisable[ self.intf_.name ]

### CLI rule definitons.
#--------------------------------------------------------------------------------
# Basic Cli Mode
# show link ( state | tracking ) group [ detail ] [ { GROUP } ]
#--------------------------------------------------------------------------------
class LinkStateGroupDetailCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show link ( state | tracking ) group [ detail ] [ { GROUP } ]'
   data = {
      'link' : 'Show link state tracking information',
      'state' : CliCommand.Node(
         matcher=CliMatcher.KeywordMatcher( 'state',
            helpdesc='Show link state tracking' ),
         deprecatedByCmd='show link tracking group' ),
      'tracking' : 'Show link state tracking groups',
      'group' : 'Show which link state group an interface belongs to',
      'detail' : 'Show detailed link state tracking information',
      'GROUP' : matcherGroupName,
   }

   @staticmethod
   def handler( mode, args ):
      groupNameListOpt = args.get( 'GROUP' )
      if groupNameListOpt:
         groupList = [ group for group in groupNameListOpt if group in status.group ]
      else:
         groupList = status.group

      if 'detail' not in args:
         t = TableOutput.createTable( ( "Link State Group", "Status" ) )
         for group in groupList:
            currentStatus = status.group[ group ].groupLinkStatus
            t.newRow( group, currentStatus )
         f1 = TableOutput.Format( justify = "left" )
         f2 = TableOutput.Format( justify = "right" )
         t.formatColumns( f1, f2 )
         print t.output()
      else:
         first = True
         for group in groupList:
            groupConfig = config.group.get( group )
            if not groupConfig:
               continue
            groupStatus = status.group[ group ]
            groupLinkStatus = groupStatus.groupLinkStatus
            numTimesDisabled = groupStatus.numTimesDisabled
            prevDisabledTime = groupStatus.prevDisabledTime
            intfsToMonitor = groupConfig.intfsToMonitor
            intfsToDisable = groupConfig.intfsToDisable
            if first:
               first = False
            else:
               print
            print "Link State Group:", group, "Status:", groupLinkStatus
            print "Upstream Interfaces :", " ".join( intfsToMonitor.keys() )
            print "Downstream Interfaces :", " ".join( intfsToDisable.keys() )
            print "Number of times disabled :", numTimesDisabled
            print "Last disabled", timestampToStr( prevDisabledTime )

BasicCli.addShowCommandClass( LinkStateGroupDetailCmd )

#--------------------------------------------------------------------------------
# Configuration Mode
# link ( ( state track ) | ( tracking group ) ) GROUP
#--------------------------------------------------------------------------------
class LinkStateTrackGroupCmd( CliCommand.CliCommandClass ):
   syntax = 'link ( ( state track ) | ( tracking group ) ) GROUP'
   noOrDefaultSyntax = 'link ( ( state track ) | ( tracking group ) ) [ { GROUPS } ]'
   data = {
      'link' : 'Configure link state tracking information',
      'state' : CliCommand.Node(
         matcher=CliMatcher.KeywordMatcher( 'state',
            helpdesc='Configure link state tracking' ),
         deprecatedByCmd='link tracking group' ),
      'track' : 'Configure link state tracking groups',
      'tracking' : 'Configure link state tracking groups',
      'group' : 'Configure which link state group an interface belongs to',
      'GROUP' : matcherGroupName,
      'GROUPS' : matcherGroup
   }

   @staticmethod
   def handler( mode, args ):
      group = args.get( 'GROUP' )
      config.newGroup( group )
      childMode = mode.childMode( LinkStateConfigMode, param=group )
      mode.session_.gotoChildMode( childMode )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      groups = args.get( 'GROUPS' )
      if groups:
         for group in groups:
            del config.group[ group.name ]
      else:
         config.group.clear()

BasicCli.GlobalConfigMode.addCommandClass( LinkStateTrackGroupCmd )

#--------------------------------------------------------------------------------
# Link State Config Mode
# [ no | default ] links minimum MIN_LINKS
#--------------------------------------------------------------------------------
class LinksMinimumMinlinksvalueCmd( CliCommand.CliCommandClass ):
   syntax = 'links minimum MIN_LINKS'
   noOrDefaultSyntax = 'links minimum ...'
   data = {
      'links' : 'Set value of minimum links for this link state group',
      'minimum' : 'Set value of minimum links for this link state group',
      'MIN_LINKS' : CliMatcher.IntegerMatcher( 1, 100000,
         helpdesc='Minimum links value' ),
   }

   handler = LinkStateConfigMode.setMinLinks
   noOrDefaultHandler = LinkStateConfigMode.resetMinLinks

LinkStateConfigMode.addCommandClass( LinksMinimumMinlinksvalueCmd )

#--------------------------------------------------------------------------------
# [ no | default ] recovery delay DELAY
#--------------------------------------------------------------------------------
class RecoveryDelayDelaysecCmd( CliCommand.CliCommandClass ):
   syntax = 'recovery delay DELAY'
   noOrDefaultSyntax = 'recovery delay ...'
   data = {
      'recovery' : 'Bring up downstream links in this link state group',
      'delay' : 'Set delay time',
      'DELAY' : CliMatcher.IntegerMatcher( 0, 300,
         helpdesc='Recovery delay time in seconds' ),
   }

   handler = LinkStateConfigMode.setRecoveryDelay
   noOrDefaultHandler = LinkStateConfigMode.resetRecoveryDelay

LinkStateConfigMode.addCommandClass( RecoveryDelayDelaysecCmd )

#--------------------------------------------------------------------------------
# Interface Config Mode
# [ no | default ] link ( tracking | state ) group GROUP ( downstream | upstream )
#--------------------------------------------------------------------------------
class LinkStateGroupGroupDownstreamCmd( CliCommand.CliCommandClass ):
   syntax = 'link ( tracking | state ) group GROUP ( downstream | upstream )'
   noOrDefaultSyntax = 'link ( tracking | state ) group [ { GROUPS } ] ...'
   data = {
      'link' : 'Configure link state tracking information',
      'tracking' : 'Configure link state tracking groups',
      'state' : CliCommand.Node(
         matcher=CliMatcher.KeywordMatcher( 'state',
            helpdesc='Configure link state tracking' ),
         deprecatedByCmd='link tracking group' ),
      'group' : 'Configure which link state group an interface belongs to',
      'GROUP' : matcherGroupName,
      'GROUPS' : matcherGroup,
      'downstream' : 'Set as downstream intf',
      'upstream' : 'Set as upstream intf'
   }

   @staticmethod
   def handler( mode, args ):
      group = args.get( 'GROUP' ) 
      if 'upstream' in args:
         presentAsDownstream = False
         for existingGroup in config.group.itervalues():
            if mode.intf.name in existingGroup.intfsToDisable:
               presentAsDownstream = True
               msg = "Interface already present in group " + \
                   existingGroup.name + " as downstream link"
               mode.addWarning( msg )
         if presentAsDownstream:
            mode.addWarning( "Cannot configure interface as requested" )
         else:
            if group not in config.group:
               config.group.newMember( group )
            config.group[ group ].intfsToMonitor[ mode.intf.name ] = True
      else:
         present = False
         for existingGroup in config.group.itervalues():
            if mode.intf.name in existingGroup.intfsToDisable:
               present = True
               msg = "Interface already present in group " + \
                   existingGroup.name + " as downstream interface"
               mode.addWarning( msg )
            if mode.intf.name in existingGroup.intfsToMonitor:
               present = True
               msg = "Interface already present in group " + \
                   existingGroup.name + " as upstream interface"
               mode.addWarning( msg )
         if present:
            mode.addWarning( "Cannot configure interface as requested" )
         else:
            if group not in config.group:
               config.group.newMember( group )
            config.group[ group ].intfsToDisable[ mode.intf.name ] = True

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      groups = args.get( 'GROUPS' )
      if groups:
         for group in groups:
            del config.group[ group.name ].intfsToMonitor[ mode.intf.name ]
            del config.group[ group.name ].intfsToDisable[ mode.intf.name ]
      else:
         for group in config.group.itervalues():
            del group.intfsToMonitor[ mode.intf.name ]
            del group.intfsToDisable[ mode.intf.name ]

IntfCli.IntfConfigMode.addCommandClass( LinkStateGroupGroupDownstreamCmd )

def Plugin( entityManager ):
   global config, status
   config = ConfigMount.mount(
      entityManager,
      "interface/errdisable/uplinkFailureDetection/config",
      "UplinkFailureDetection::Config", "w" )
   status = LazyMount.mount(
      entityManager,
      "interface/errdisable/uplinkFailureDetection/status",
      "UplinkFailureDetection::Status", "r" )
   IntfCli.Intf.registerDependentClass( LinkStateConfigCleaner )
