#!/usr/bin/env python
# Copyright (c) 2019 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

from __future__ import absolute_import, division, print_function

# pylint: disable=ungrouped-imports

import CliCommand
import CliMatcher
import CliPlugin.TunnelRibCli as TunnelRibCli
import BasicCliModes

#--------------------------------------------------------------------------------
# [ no | default ] tunnel-ribs
#--------------------------------------------------------------------------------
class TunnelRibsCmd( CliCommand.CliCommandClass ):
   syntax = 'tunnel-ribs'
   noOrDefaultSyntax = syntax
   data = {
      'tunnel-ribs': 'Tunnel RIBs configuration',
   }

   handler = TunnelRibCli.gotoTunnelRibsMode
   noOrDefaultHandler = TunnelRibCli.deleteTunnelRibs

BasicCliModes.GlobalConfigMode.addCommandClass( TunnelRibsCmd )

#--------------------------------------------------------------------------------
# [ no | default ] tunnel-rib ( system-tunnel-rib | TUNNEL_NAME )
#--------------------------------------------------------------------------------
class TunnelRibCmd( CliCommand.CliCommandClass ):
   syntax = 'tunnel-rib ( system-tunnel-rib | TUNNEL_NAME )'
   noOrDefaultSyntax = syntax
   data = {
      'tunnel-rib': 'Configure a tunnel RIB',
      'system-tunnel-rib': 'The system tunnel RIB',
      'TUNNEL_NAME': TunnelRibCli.TunnelRibNameMatcher(
         helpdesc='Name of the tunnel RIB' ),
   }
   
   handler = TunnelRibCli.gotoTunnelRibMode
   noOrDefaultHandler = TunnelRibCli.deleteTunnelRib

TunnelRibCli.TunnelRibsMode.addCommandClass( TunnelRibCmd )

matcherBgp = CliMatcher.KeywordMatcher( 'bgp',
      helpdesc='BGP tunnel' )
matcherCostValue = CliMatcher.IntegerMatcher( 0, 255,
      helpdesc='The IGP-cost value' )
matcherIgpCost = CliMatcher.KeywordMatcher( 'igp-cost',
      helpdesc='Configure the preference to influence IGP cost '
               'of next-hops resolving over tunnels' )
matcherIgpPreference = CliMatcher.KeywordMatcher( 'preference',
      helpdesc='Configure the preference attribute of igp cost' )
matcherIsis = CliMatcher.KeywordMatcher( 'isis',
      helpdesc='IS-IS tunnel' )
matcherLabeledUnicast = CliMatcher.KeywordMatcher( 'labeled-unicast',
      helpdesc='BGP labeled unicast tunnel' )
matcherLdp = CliMatcher.KeywordMatcher( 'ldp',
      helpdesc='LDP tunnel' )
matcherNexthopGroup = CliMatcher.KeywordMatcher( 'nexthop-group',
      helpdesc='Nexthop group tunnel' )
matcherPreference = CliMatcher.KeywordMatcher( 'preference',
      helpdesc='Configure the preference to influence tunnel rib election' )
matcherPreferenceValue = CliMatcher.IntegerMatcher( 0, 255,
      helpdesc='The preference value' )
matcherSegmentRouting = CliMatcher.KeywordMatcher( 'segment-routing',
      helpdesc='IS-IS segment routing tunnel' )
matcherSourceProtocol = CliMatcher.KeywordMatcher( 'source-protocol',
      helpdesc='Configure the tunnel source' )
matcherStatic = CliMatcher.KeywordMatcher( 'static',
      helpdesc='Static tunnel' )
matcherRsvpLer = CliMatcher.KeywordMatcher( 'rsvp-ler',
      helpdesc='RSVP LER tunnel' )

class SourceProtocolCmd( CliCommand.CliCommandClass ):
   _prefArgs = ( '[ ( preference PREFERENCE [ igp-cost igp-preference COST ] ) '
                 '| ( igp-cost igp-preference COST [ preference PREFERENCE ] ) ]' )
   syntax = ( 'source-protocol ' 
              '( ldp '
              '| rsvp-ler '
              '| static '
              '| ( bgp labeled-unicast ) '
              '| ( isis segment-routing ) '
              '| nexthop-group ) '
              '%s' % _prefArgs )
   defaultSyntax = syntax + ' ...'
   data = {
      'source-protocol': matcherSourceProtocol,
      'ldp': matcherLdp,
      'rsvp-ler': matcherRsvpLer,
      'static': matcherStatic,
      'bgp': matcherBgp,
      'labeled-unicast': matcherLabeledUnicast,
      'isis': matcherIsis,
      'segment-routing': matcherSegmentRouting,
      'nexthop-group': matcherNexthopGroup,
      'preference': matcherPreference,
      'PREFERENCE': matcherPreferenceValue,
      'igp-cost': matcherIgpCost,
      'igp-preference': matcherIgpPreference,
      'COST': matcherCostValue,
   }

#--------------------------------------------------------------------------------
# [ no | default ] source-protocol 
#                             ( ldp 
#                             | rsvp-ler 
#                             | static 
#                             | ( bgp labeled-unicast ) 
#                             | ( isis segment-routing ) 
#                             | nexthop-group ) 
#                                [ ( preference PREFERENCE
#                                    [ igp-cost igp-preference COST ] ) 
#                                | ( igp-cost igp-preference COST
#                                    [ preference PREFERENCE ] ) ]
#--------------------------------------------------------------------------------
class SourceProtocolCmdForCustomTunnelRibMode( SourceProtocolCmd ):
   noSyntax = SourceProtocolCmd.defaultSyntax

   # This inherited class needs data dict to be separately defined because of 
   # BUG380538. The fix for BUG380538 should move this dict to the parent class
   handler = TunnelRibCli.CustomTunnelRibMode.handleEntry
   noOrDefaultHandler = TunnelRibCli.CustomTunnelRibMode.deleteEntry

#--------------------------------------------------------------------------------
# [ default ] source-protocol 
#                             ( ldp 
#                             | rsvp-ler 
#                             | static 
#                             | ( bgp labeled-unicast ) 
#                             | ( isis segment-routing ) 
#                             | nexthop-group ) 
#                                [ ( preference PREFERENCE
#                                    [ igp-cost igp-preference COST ] ) 
#                                | ( igp-cost igp-preference COST
#                                    [ preference PREFERENCE ] ) ]
#--------------------------------------------------------------------------------
class SourceProtocolCmdForSystemTunnelRibMode( SourceProtocolCmd ):
   # noSyntax is not present for this command as we cannot remove
   # source-protocols from contributing to the system-tunnel-rib

   # This inherited class needs data dict to be separately defined because of 
   # BUG380538. The fix for BUG380538 should move this dict to the parent class
   data = {
      'source-protocol': matcherSourceProtocol,
      'ldp': matcherLdp,
      'rsvp-ler': matcherRsvpLer,
      'static': matcherStatic,
      'bgp': matcherBgp,
      'labeled-unicast': matcherLabeledUnicast,
      'isis': matcherIsis,
      'segment-routing': matcherSegmentRouting,
      'nexthop-group': matcherNexthopGroup,
      'preference': matcherPreference,
      'PREFERENCE': matcherPreferenceValue,
      'igp-cost': matcherIgpCost,
      'igp-preference': matcherIgpPreference,
      'COST': matcherCostValue,
   }
   handler = TunnelRibCli.SystemTunnelRibMode.handleEntry
   defaultHandler = handler

TunnelRibCli.CustomTunnelRibMode.addCommandClass( 
           SourceProtocolCmdForCustomTunnelRibMode )
TunnelRibCli.SystemTunnelRibMode.addCommandClass( 
           SourceProtocolCmdForSystemTunnelRibMode )
