# Copyright (c) 2018 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

from ArnetModel import IpGenericPrefix
from Arnet import IpGenPrefix
from CliModel import Dict
from CliModel import Enum
from CliModel import List
from CliModel import Int
from CliModel import Str
from CliModel import Model
from TableOutput import createTable, Format
from TunnelModels import tableTypeAttrs
import Tac

class TunnelRibEntry( Model ):
   __revision__ = 2
   tunnelType = Enum( values=tableTypeAttrs, help="Tunnel type" )
   tunnelIndexes = List( valueType=long, help="List of tunnel indexes ending at "
                                              "this tunnel endpoint" )
   metric = Int( help="IGP metric of tunnel endpoint entry" )
   metric2 = Int( help="IGP metric 2 of tunnel endpoint entry" )
   pref = Int( help="Underlying IGP route's preference" )
   pref2 = Int( help="Preference 2 of tunnel endpoint entry" )
   tunnelPreference = Int( help="Tunnel preference used for tunnel RIB election" )

   def degrade( self, dictRepr, revision ):
      if revision == 1:
         if dictRepr[ 'tunnelType' ] == "Nexthop Group":
            dictRepr[ 'tunnelType' ] = "Nexthop-Group"
      return dictRepr

   def renderTunnelRibEntry( self, table, prefix ):
      table.newRow( prefix, self.tunnelType,
            ', '.join( str( index ) for index in self.tunnelIndexes ),
            self.tunnelPreference, self.pref, self.metric )

   def renderColoredTunnelRibEntry( self, table, prefix, color ):
      table.newRow( prefix, color, self.tunnelType,
            ', '.join( str( index ) for index in self.tunnelIndexes ),
            self.tunnelPreference, self.pref, self.metric )

class TunnelRibSummary( Model ):
   name = Str( help='Tunnel RIB name' )
   tunnels = Dict( keyType=str, valueType=int,
                   help='Tunnel count keyed by tunnel type' )

   def render( self ):
      print "Tunnel RIB: %s" % self.name
      headings = ( 'Tunnel Type', 'Tunnel Count' )
      table = createTable( ( ( h, 'l' ) for h in headings ) )
      tunnelTypeFormat = Format( justify='left' )
      countFormat = Format( justify='right' )
      table.formatColumns( tunnelTypeFormat, countFormat )
      totalCount = 0
      for tunnelType, count in sorted( self.tunnels.iteritems() ):
         table.newRow( tunnelType, count )
         totalCount = totalCount + count
      table.newRow()
      table.newRow( 'Total tunnels', totalCount )
      print table.output()

class TunnelRibAllSummary( Model ):
   tunnels = Dict( keyType=str, valueType=int,
                   help='Tunnel count keyed by tunnel RIB name' )

   def render( self ):
      headings = ( 'Tunnel RIB', 'Tunnel Count' )
      table = createTable( ( ( h, 'l' ) for h in headings ) )
      tunnelRibFormat = Format( justify='left' )
      tunnelRibFormat.noPadLeftIs( True )
      countFormat = Format( justify='right' )
      table.formatColumns( tunnelRibFormat, countFormat )
      for tunnelRibName, count in sorted( self.tunnels.iteritems() ):
         table.newRow( tunnelRibName, count )
      print table.output()

class TunnelRib( Model ):
   __revision__ = 2
   # _isBrief = Bool( help="Only show prefix to tunnel index mappings" )
   name = Str( help='Tunnel RIB name' )
   entries = Dict( keyType=IpGenericPrefix, valueType=TunnelRibEntry,
                   help="Tunnel entries keyed by prefix" )

   def render( self ):
      print "Tunnel RIB: %s" % self.name
      headings = ( "Endpoint", "Tunnel Type", "Index(es)",
                   "Tunnel Preference", "IGP Preference", "IGP Metric" )
      table = createTable( ( ( h, 'l' ) for h in headings ) )

      def _orderByPrefix( entriesItem ):
         pfx = IpGenPrefix( entriesItem[ 0 ] )
         # Need address-family first as sortKey order puts v6 prefixes first
         return ( pfx.af, pfx.sortKey )

      # pylint: disable-msg=E1101
      for prefix, tunnelRibEntry in sorted( self.entries.iteritems(),
            key=_orderByPrefix ):
         tunnelRibEntry.renderTunnelRibEntry( table, prefix )

      print table.output()

class TunnelRibColors( Model ):
   colors = Dict( keyType=int, valueType=TunnelRibEntry,
                   help="Colored tunnel entries keyed by color" )

class ColoredTunnelRib( Model ):
   name = Str( help='System colored tunnel RIB name' )
   endpoints = Dict( keyType=IpGenericPrefix, valueType=TunnelRibColors,
                     help="Colored tunnel entries keyed by endpoint" )

   def addColoredRibEntry( self, prefix, color, ribEntry ):
      prefixColors = TunnelRibColors()
      if prefix in self.endpoints:
         prefixColors = self.endpoints[ prefix ]
      prefixColors.colors[ color ] = ribEntry
      self.endpoints[ prefix ] = prefixColors

   def render( self ):
      print "Tunnel RIB: %s" % self.name
      headings = ( "Endpoint", "Color", "Tunnel Type", "Index(es)",
                   "Tunnel Preference", "IGP Preference", "IGP Metric" )
      table = createTable( ( ( h, 'l' ) for h in headings ), tableWidth=120 )

      def _orderByPrefix( entriesItem ):
         pfx = IpGenPrefix( entriesItem[ 0 ] )
         # Need address-family first as sortKey order puts v6 prefixes first
         return ( pfx.af, pfx.sortKey )

      # pylint: disable-msg=E1101
      for prefix, colors in sorted( self.endpoints.iteritems(), key=_orderByPrefix ):
         for color, tunnelRibEntry in sorted( colors.colors.iteritems() ):
            tunnelRibEntry.renderColoredTunnelRibEntry( table, prefix, color )

      print table.output()

