# Copyright (c) 2018 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import re

import Arnet
import BasicCli
import CliMatcher
import CliMode.TunnelRib
import CliParser
import CliParserCommon
import ConfigMount
import Tac
from TypeFuture import TacLazyType

cliConfig = None
systemTunnelRibName = Tac.Type( "Tunnel::TunnelTable::TunnelRibNameIdMap"
                              ).systemTunnelRibName
systemTunnelRibId = Tac.Type( "Tunnel::TunnelTable::TunnelRibId"
                            ).systemTunnelRibId
systemColoredTunnelRibName = Tac.Type( "Tunnel::TunnelTable::TunnelRibNameIdMap"
                                     ).systemColoredTunnelRibName
systemColoredTunnelRibId = Tac.Type( "Tunnel::TunnelTable::TunnelRibId"
                                   ).systemColoredTunnelRibId
systemTunnelingLdpTunnelRibName = Tac.Type( "Tunnel::TunnelTable::TunnelRibNameIdMap"
                                     ).systemTunnelingLdpTunnelRibName
systemTunnelingLdpTunnelRibId = Tac.Type( "Tunnel::TunnelTable::TunnelRibId"
                                     ).systemTunnelingLdpTunnelRibId
systemIgpShortcutTunnelRibName = Tac.Type( "Tunnel::TunnelTable::TunnelRibNameIdMap"
                                     ).systemIgpShortcutTunnelRibName
systemIgpShortcutTunnelRibId = Tac.Type( "Tunnel::TunnelTable::TunnelRibId"
                                     ).systemIgpShortcutTunnelRibId
ProtoPrefAction = TacLazyType( "Tunnel::TunnelTable::ProtoPrefAction" )
ProtoPrefMapping = TacLazyType( "Tunnel::TunnelTable::ProtoPrefMapping" )

# -----------------------------------------------------------------------------------
#      modes
# -----------------------------------------------------------------------------------

class TunnelRibsMode( CliMode.TunnelRib.TunnelRibsBaseMode,
                      BasicCli.ConfigModeBase ):
   modeParseTree = CliParser.ModeParseTree()

   def __init__( self, parent, session ):
      CliMode.TunnelRib.TunnelRibsBaseMode.__init__( self )
      BasicCli.ConfigModeBase.__init__( self, parent, session )

class TunnelRibMode( CliMode.TunnelRib.TunnelRibBaseMode, BasicCli.ConfigModeBase ):

   def __init__( self, parent, session, tunnelRibName ):
      CliMode.TunnelRib.TunnelRibBaseMode.__init__( self, tunnelRibName )
      BasicCli.ConfigModeBase.__init__( self, parent, session )
      self.config = cliConfig.config.get( self.tunnelRibName )
      if not self.config:
         self.config = cliConfig.config.newMember( self.tunnelRibName )

   def getTunnelTableId( self, args ):
      tokenList = [ 'ldp', 'static', 'labeled-unicast', 
                    'segment-routing', 'nexthop-group', 'rsvp-ler' ]
      sourceProtocolLastToken = ''
      for token in tokenList:
         if token in args:
            sourceProtocolLastToken = token
            break
      tunnelTableId = CliMode.TunnelRib.tunnelTableIdFromCliToken( 
            sourceProtocolLastToken )
      return tunnelTableId

   def handleEntry( self, args ):
      tunnelTableId = self.getTunnelTableId( args )

      preference = args.get( "PREFERENCE" )
      prefAction = ( ProtoPrefAction.prefDefault
                     if preference is None else ProtoPrefAction.prefStatic )
      prefMapping = ProtoPrefMapping( prefAction, preference or 0 )

      cost = args.get( "COST" )
      igpCostAction = ( ProtoPrefAction.prefDefault
                        if cost is None else ProtoPrefAction.prefStatic )
      igpCostPrefMapping = ProtoPrefMapping( igpCostAction, cost or 0 )

      entry = self.config.entry.get( tunnelTableId )
      if entry:
         entry.prefMapping = prefMapping
         entry.igpCostPrefMapping = igpCostPrefMapping
      else:
         entry = self.config.entry.newMember( tunnelTableId, prefMapping,
                                              igpCostPrefMapping )

class CustomTunnelRibMode( TunnelRibMode, 
                           BasicCli.ConfigModeBase ):
   modeParseTree = CliParser.ModeParseTree()

   def __init__( self, parent, session, tunnelRibName ):
      TunnelRibMode.__init__( self, parent, session, tunnelRibName )

   def deleteEntry( self, args ):
      tunnelTableId = self.getTunnelTableId( args ) 
      del self.config.entry[ tunnelTableId ]

class SystemTunnelRibMode( TunnelRibMode, 
                           BasicCli.ConfigModeBase ):
   modeParseTree = CliParser.ModeParseTree()

   def __init__( self, parent, session, tunnelRibName ):
      TunnelRibMode.__init__( self, parent, session, tunnelRibName )

# -----------------------------------------------------------------------------------
#      config commands
# -----------------------------------------------------------------------------------
def getCustomTunnelRibNames( mode ):
   ctrNames = cliConfig.config.keys()
   ctrNames.remove( systemTunnelRibName )
   ctrNames.remove( systemColoredTunnelRibName )
   ctrNames.remove( systemTunnelingLdpTunnelRibName )
   ctrNames.remove( systemIgpShortcutTunnelRibName )
   return ctrNames


class TunnelRibNameMatcher( CliMatcher.Matcher ):
   '''Type of matcher that matches custom tunnel rib names that are not ip address or
   ipv6 address or brief keywords. This is to avoid confusion between
   'show tunnel-rib [tep] brief', 'show tunnel-rib [trib-name] brief' and
   'show tunnel-rib brief'. With this restriction custom tunnel ribs cannot be named
   as brief or any other case combination of "brief" say "BRIEF", "bRIEF" etc or a
   valid ip or ipv6 address.
   '''
   def __init__( self, helpdesc='Name of the tunnel RIB',
                 excludeSystemSpecialTunnelRibs=True,
                 **kargs ):
      super( TunnelRibNameMatcher, self ).__init__( helpdesc=helpdesc, **kargs )
      self.completion_ = CliParser.Completion( "TUNNEL_NAME", helpdesc, False )
      self.expectedPatternRe_ = re.compile( r"(?:^([a-zA-Z0-9:\._-]{1,100})$)" )
      self.briefRe_ = re.compile( r"(?:^(brief)$)", re.IGNORECASE )
      self.coloredRe_ = re.compile( r"(?:^(colored)$)", re.IGNORECASE )
      self.summaryRe_ = re.compile( r"(?:^(summary)$)", re.IGNORECASE )
      self.excludeSystemSpecialTunnelRibs = excludeSystemSpecialTunnelRibs

   def match( self, mode, context, token ):
      # First check that the token contains valid characters and expected length
      if not self.expectedPatternRe_.match( token ):
         return CliParserCommon.noMatch

      # see if the token matches to brief keyword
      if self.briefRe_.match( token ):
         # matches brief token
         return CliParserCommon.noMatch

      if self.coloredRe_.match( token ):
         # matches colored token
         return CliParserCommon.noMatch

      if self.summaryRe_.match( token ):
         # matches summary token
         return CliParserCommon.noMatch

      excludeTunnelRibs = [ systemColoredTunnelRibName,
                            systemTunnelingLdpTunnelRibName,
                            systemIgpShortcutTunnelRibName ]

      if self.excludeSystemSpecialTunnelRibs and token in excludeTunnelRibs:
         return CliParserCommon.noMatch

      # see if the token matches to ipv4 address or ipv6 address regex
      ipv4Match = Arnet.IpAddrCompiledRe.match( token )
      ipv6Match = Arnet.Ip6AddrCompiledRe.match( token )
      if ( ipv4Match is None or ipv4Match.group( 0 ) != token ) and \
         ( ipv6Match is None or ipv6Match.group( 0 ) != token ):
         # It doesn't match with either ipv4 address or ipv6 address regex. So it's a
         # valid tunnel rib name token
         return CliParserCommon.MatchResult( token, token )

      # token matches ipv4 or ipv6 address regex but IpAddrCompiledRe,
      # Ip6AddrCompiledRe are bit lenient in matching the token. So, try to get a
      # IpGenAddr from the token
      try:
         _ = Arnet.IpGenAddr( token )
      except ( IndexError, ValueError ):
         return CliParserCommon.MatchResult( token, token )
      return CliParserCommon.noMatch

   def completions( self, mode, context, token ):
      if token == '':
         return [ self.completion_ ]
      # completions are case-sensitive
      return [ CliParser.Completion( k, k ) for k in getCustomTunnelRibNames( mode )
               if k.startswith( token ) ]

#----------------------------------------------
# [no] tunnel-ribs
#----------------------------------------------
def gotoTunnelRibsMode( mode, args ):
   childMode = mode.childMode( TunnelRibsMode )
   mode.session_.gotoChildMode( childMode )

def deleteTunnelRibs( mode, args ):
   systemTunnelRibs = [ systemTunnelRibName, systemColoredTunnelRibName,
                        systemTunnelingLdpTunnelRibName,
                        systemIgpShortcutTunnelRibName ]
   for tunnelRibName in cliConfig.config.keys():
      if tunnelRibName not in systemTunnelRibs:
         del cliConfig.config[ tunnelRibName ]

#----------------------------------------------
# [no] tunnel-rib <name>
#----------------------------------------------
def gotoTunnelRibMode( mode, args ):
   tunnelRibName = args.get( "TUNNEL_NAME" )
   if systemTunnelRibName in args:
      childMode = mode.childMode( SystemTunnelRibMode, 
                                  tunnelRibName=systemTunnelRibName )
      mode.session_.gotoChildMode( childMode )
   elif systemColoredTunnelRibName in args or \
        tunnelRibName == systemColoredTunnelRibName :
      mode.addError( "%s is system-defined, can not be modified" %
                     systemColoredTunnelRibName )
   elif systemTunnelingLdpTunnelRibName in args or \
        tunnelRibName == systemTunnelingLdpTunnelRibName :
      mode.addError( "%s is system-defined, can not be modified" %
                     systemTunnelingLdpTunnelRibName )
   elif systemIgpShortcutTunnelRibName in args or \
        tunnelRibName == systemIgpShortcutTunnelRibName :
      mode.addError( "%s is system-defined, can not be modified" %
                     systemIgpShortcutTunnelRibName )
   else:
      childMode = mode.childMode( CustomTunnelRibMode, tunnelRibName=tunnelRibName )
      mode.session_.gotoChildMode( childMode )

def deleteTunnelRib( mode, args ):
   tunnelRibName = args.get( "TUNNEL_NAME" )
   if systemTunnelRibName in args:
      mode.addError( "%s is system-defined, can not be deleted" %
                     systemTunnelRibName )
   elif systemColoredTunnelRibName in args or \
        tunnelRibName == systemColoredTunnelRibName :
      mode.addError( "%s is system-defined, can not be deleted" %
                     systemColoredTunnelRibName )
   elif systemTunnelingLdpTunnelRibName in args or \
        tunnelRibName == systemTunnelingLdpTunnelRibName :
      mode.addError( "%s is system-defined, can not be deleted" %
                     systemTunnelingLdpTunnelRibName )
   elif systemIgpShortcutTunnelRibName in args or \
        tunnelRibName == systemIgpShortcutTunnelRibName :
      mode.addError( "%s is system-defined, can not be deleted" %
                     systemIgpShortcutTunnelRibName )
   elif tunnelRibName in cliConfig.config:
      del cliConfig.config[ tunnelRibName ]

def Plugin( entityManager ):
   global cliConfig
   cliConfig = ConfigMount.mount( entityManager, "tunnel/tunnelRibs/config",
                                  "Tunnel::TunnelTable::TunnelRibConfigDir", "wi" )
