# Copyright (c) 2015 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

from IntfModel import InterfaceStatus
from ArnetModel import IpGenericAddress
from CliModel import Enum
from CliModel import Int
from CliModel import Bool
from CliModel import Model, Dict, List, Str
from TableOutput import createTable, Format
from TunnelIntfUtil import tunnelDefaultKey
from TunnelModels import IpVia
from operator import attrgetter
import Tac

tunnelMode = [ s for s in Tac.Type( 'Arnet::TunnelIntfMode' ).attributes ]

# Diagnostics information mapping
diagnosticInfo = dict(
   configSourceInvalid = 'Tunnel source or source interface address is not' \
                          ' configured or invalid',
   configDestinationInvalid = 'Tunnel destination address is not configured' \
                              ' or invalid',
   configAddrFamilyMismatch = 'Source and destination address families don\'t' \
                               ' match',
   configModeNotSupported = 'Configured tunnel mode is not supported by this' \
                             ' platform',
   configAddrFamilyNotSupported = 'Configured address family is not supported' \
                                   ' by this platform',
   configDuplicateDevice = 'Duplicate kernel device already exists',
   configModeMtuNotSupported = 'Configured tunnel mode and MTU combination' \
                               ' is not supported on this platform',
   statusSourceInterfaceDown = 'Tunnel source address does not belong to an' \
                               ' interface in the up state',
   statusNexthopNotResolved = 'Waiting for tunnel destination nexthop to be' \
                              ' resolved',
   statusSecurityAssociationPending = 'Waiting for successful completion of' \
                                      ' security association negotiation with' \
                                      ' remote peer',
   statusHwProgrammingEncapFailure = 'Hardware encap programming failure',
   statusHwProgrammingDecapFailure = 'Hardware decap programming failure',
   statusHwProgrammingFailure = 'Hardware programming failure',
   statusHwProgrammingPending = 'Hardware programming pending',
   statusPlatformDeviceNotReady = 'Kernel device not ready',
   configShutdown = 'Admin disabled', # This is considered a status error
   maxDiagnostic = 'Invalid configuration',
   )

class TunnelIntfStatus( InterfaceStatus ):
   tunSrcAddr = IpGenericAddress( help="Address for source" )
   tunDstAddr = IpGenericAddress( help="Address for destination" )
   tunMode = Enum( help="Tunnel mode", values=tunnelMode )
   tunHwFwd = Bool( help="Hardware forwarding" )
   tunMtu = Int( help="Tunnel transport MTU" ) #For interop compatibility
   tunDiagnosticsInfo = List( help="Diagnostic messages", valueType=str )
   # Everything else kept primarily for CAPI backwards-compatibility and for
   # the config-sanity model to inherit from
   tunTtl = Int( help="Tunnel TTL" )
   tunTos = Int( help="Tunnel TOS" )
   tunPathMtuDiscovery = Bool( help="Tunnel path MTU discovery" )
   tunOKey = Int( help="Input key" )
   tunIKey = Int( help="Output key" )
   tunUnderlayVrf = Str( help="Tunnel underlay VRF" )

   def tunAddrStr( self, addr ):
      return str( addr ) if ( type( addr ) == Tac.Type( "Arnet::IpGenAddr" )
                              and not addr.isUnspecified ) else "UNKNOWN"
   
   def tunModeStr( self, tunMode ):
      if tunMode == 'tunnelIntfModeGre':
         return "GRE/IP"
      elif tunMode == 'tunnelIntfModeIpip':
         return "IPIP/IP"
      elif tunMode == 'tunnelIntfModeIpsec':
         return "IPSec/IP"
      else: # This is the old behavior, hopefully benign
         return ""

   def renderDiagnosticsInfo( self, diagnostics, label, indent ):
      if diagnostics:
         print label
         for value in diagnostics:
            print indent + "%s" % diagnosticInfo[ value ]
   
   def renderSrcDst( self ):
      print "  Tunnel source %s, destination %s" % (
            self.tunAddrStr( self.tunSrcAddr ),
            self.tunAddrStr( self.tunDstAddr ) )

   def renderMode( self ):
      print "  Tunnel protocol/transport %s" % self.tunModeStr( self.tunMode ) 
   
   def renderMtu( self ):
      #Keeping this for interop compatibility, until we understand it more
      print ( "  Tunnel transport MTU %s bytes" % str( self.tunMtu ) ) + \
         ( " (default)" if not self.l3MtuConfigured else "" )

   def renderHardwareForwarding( self ):
      print "  Hardware forwarding %s" % ( "enabled" if self.tunHwFwd
         else "not supported" )

   def render( self ):
      self.renderHeader()
      self.renderHardware() 
      self.renderInterfaceAddress()
      self.renderSrcDst()
      self.renderMode()
      self.renderHardwareForwarding()
      self.renderMtu()

      # Print diagnostics information. Also print "Configuration errors" if 
      # any exist to direct the user to config-sanity.
      diagnostics = [ v for v in self.tunDiagnosticsInfo
            if v.startswith('status') or v == 'configShutdown' ]
      self.renderDiagnosticsInfo( diagnostics, "  Line protocol status details",
            "   " )
      if [ v for v in self.tunDiagnosticsInfo if v.startswith('config') and
            v != 'configShutdown' ]:
         print "   Configuration error"
      
      self.renderUptime()  

class TunnelIntfConfigSanity( TunnelIntfStatus ):
   
   def renderHeader( self ):
      # pylint: disable-msg=E1101
      print "%s configuration:" % self.name.stringValue
   
   def renderSrcDst( self ):
      print "Source: %s" % self.tunAddrStr( self.tunSrcAddr )
      print "Destination: %s" % self.tunAddrStr( self.tunDstAddr )

   def renderMode( self ):
      print "Protocol/transport: %s" % self.tunModeStr( self.tunMode )
      if self.tunIKey != tunnelDefaultKey or self.tunOKey != tunnelDefaultKey:
         print "Okey: %s" % self.tunOKey
         print "Ikey: %s" % self.tunIKey
      elif self.tunMode !='tunnelIntfModeIpip' and \
           self.tunIKey == 0 and self.tunOKey == 0:
         print "Key: disabled"
         print "Sequencing: disabled"
      if self.tunMode == 'tunnelIntfModeGre':
         print "Packet checksumming: disabled"
   
   def renderTtl( self ):
      print "TTL: %s" % str( self.tunTtl )
   
   def renderTos( self ):
      print "TOS: %s" % str( self.tunTos )
   
   def renderMtu( self ):
      #Keeping this for interop compatibility, until we understand it more
      print ( "Transport MTU: %s bytes" % str( self.tunMtu ) ) + \
         ( " (default)" if not self.l3MtuConfigured else "" )

   def renderPathMtuDiscovery( self ):
      if self.tunPathMtuDiscovery:
         print "Path MTU discovery: enabled"
      else:
         print "Path MTU discovery: disabled"

   def renderUnderlayVrf( self ):
      print "Underlay VRF: \"%s\"" % self.tunUnderlayVrf

   def render( self ):
      self.renderHeader()
      self.renderSrcDst()
      self.renderMode()
      self.renderTtl()
      self.renderTos()
      self.renderPathMtuDiscovery()
      self.renderMtu()
      self.renderUnderlayVrf()
      diagnostics = [ v for v in self.tunDiagnosticsInfo
            if v.startswith('config') and v != 'configShutdown' ]
      self.renderDiagnosticsInfo( diagnostics, 'Configuration errors:', '  ')

class L3IntfTunnelTableEntry( Model ):
   name = Str( help="Tunnel name" )
   index = Int( help="Tunnel index" )
   srcAddr = IpGenericAddress(
         help="Location in the IP network where the tunnel starts" )
   dstAddr = IpGenericAddress(
         help="Location in the IP network where the tunnel terminates" )
   vias = List( valueType=IpVia, help="List of tunnel vias" )

   def renderL3IntfTunnelTableEntry( self, table ):
      def getNhAndIntfStrs( via ):
         return str( via.nexthop ), via.interface.stringValue
      nhStr = intfStr = '-'
      vias = []
      if self.vias:
         vias = sorted( self.vias, key=attrgetter( 'nexthop', 'interface' ) )
         nhStr, intfStr = getNhAndIntfStrs( vias[ 0 ] )
      table.newRow( self.name, self.index,
                    str( self.srcAddr ), str( self.dstAddr ),
                    nhStr, intfStr )
      for via in vias[ 1 : ]:
         nhStr, intfStr = getNhAndIntfStrs( via )
         table.newRow( '-', '-', '-', '-', nhStr, intfStr )

class L3IntfTunnelTable( Model ):
   def renderL3IntfTable( self, l3IntfTunnelTableEntries ):
      headings = ( "Name", "Index", "Source", "Destination",
                   "Nexthop", "Interface" )
      fl = Format( justify='left' )
      fln = Format( justify='left', maxWidth=11, minWidth=11 )
      fli = Format( justify='left', maxWidth=7, minWidth=7 )
      fln.padLimitIs ( True )
      fli.padLimitIs ( True )
      table = createTable( headings, tableWidth=200 )
      table.formatColumns( fln, fli, fl, fl, fl, fl )
      for key in sorted( l3IntfTunnelTableEntries.keys() ):
         l3IntfTunnelTableEntries[ key ].renderL3IntfTunnelTableEntry( table )

      print table.output()

class GreIntfTunnelTable( L3IntfTunnelTable ):
   '''
   Currently there are no expected CLI/CAPI output differences for any of the
   L3 tunnel encap types. This may change over time. Currently, this
   class simply initializes the L3IntfTunnelTable base class with the
   GRE specific parameters.
   '''
   greTunnels = Dict( keyType=int, valueType=L3IntfTunnelTableEntry,
                   help="Collection of GRE tunnels" )

   def render( self ):
      self.renderL3IntfTable( self.greTunnels )
