#!/usr/bin/env python
# Copyright (c) 2009-2010, 2011 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

#-------------------------------------------------------------------------------
# This module implements the Tunnel interface type.  In particular, it provides
# the TunnelIntf class.
#-------------------------------------------------------------------------------
import math
import re

import Ark
import BasicCli
import Cell
import CliCommand
import CliExtensions
import CliMatcher
import CliParser
import ConfigMount
import Ethernet
import IntfCli
import LazyMount
import SharedMem
import ShowCommand
import Smash
import Tac
import Tracing
import TunnelIntfUtil
import VirtualIntfRule
from Arnet.NsLib import DEFAULT_NS
from CliPlugin import TechSupportCli
from CliPlugin.EthIntfCli import EthPhyIntf, EthPhyAutoIntfType
from CliPlugin.LagIntfCli import LagAutoIntfType
from CliPlugin.MirroringModelets import (
      matcherDefaultInterface,
      matcherGroupName,
      nodeTap,
)
from CliPlugin.TapAggIntfCli import addOrRemoveTapGroups, addOrRemoveTapRawIntf
from CliPlugin.VrfCli import DEFAULT_VRF, VrfExprFactory, vrfExists
from Intf.IntfRange import IntfRangeMatcher
from IntfModel import ErrorCounters, InterfaceCounters, InterfaceCountersRate
from IntfRangePlugin.TunnelIntfRange import TunnelAutoIntfType
from IpGenAddrMatcher import IpGenAddrMatcher
from IraIpIntfCli import canSetVrfHook
from Toggles.MirroringToggleLib import toggleTapAggGreTunnelTerminationEnabled
from Toggles.TunnelIntfToggleLib import toggleTunnelIntfUnderlayVrfEnabled
from TunnelCli import (
      TunnelTableIdentifier,
      getTunnelIdFromIndex,
      getTunnelIndexFromId,
      readMountTunnelTable,
)
from TunnelIntfModel import L3IntfTunnelTableEntry, GreIntfTunnelTable
from TunnelIntfModel import TunnelIntfStatus, TunnelIntfConfigSanity
from TunnelModels import IpVia

em = None

tunModeEnum = Tac.Type( 'Arnet::TunnelIntfMode' )
afEnum = Tac.Type( 'Arnet::AddressFamily' )

# Maps Eos tunnel (mode, address-family) to Linux tunnel (mode, device)
kernelTunnelAttributesMap = {
   ( tunModeEnum.tunnelIntfModeGre, afEnum.ipv4 )  : ( 'gre', 'gre0' ),
   ( tunModeEnum.tunnelIntfModeGre, afEnum.ipv6 )  : ( 'ip6gre', 'ip6gre0' ),
   ( tunModeEnum.tunnelIntfModeIpip, afEnum.ipv4 ) : ( 'ipip', 'tunl0' ),
   ( tunModeEnum.tunnelIntfModeIpip, afEnum.ipv6 ) : ( 'ipip6', 'ip6tnl0' ),
   ( tunModeEnum.tunnelIntfModeIpsec, afEnum.ipv4 ) : ( 'vti', 'ip_vti0' ) }

ip4ConfigDir = None
ip6ConfigDir = None
tunIntfConfigDir = None
tunIntfStatusDir = None
allIntfNsConfigDir = None
hwCapabilities = None
greIntfTunnelTable = None
hw6Capabilities = None
tunnelCountersDir = None
tunnelCountersCheckpointDir = None
aleCliConfig = None
tunnelIntfCliConfig = None
bridgingHwCapabilities = None

U32_MAX_VALUE =  0xFFFFFFFF

# Add trace filter class to the Cli agent for tunnel interface commands
__defaultTraceHandle__ = Tracing.Handle( 'TunnelIntfCli' )
t0 = Tracing.trace0

# Returns the Linux device name for an Eos interface name
def intfNameToDeviceName( intfName ):
   return Tac.Value('Arnet::IntfId', intfName ).shortName

# Returns true if tunnel interface can handle a 'vrf' change
def intfCanSetVrfHookForTunnel( intfName, oldVrf, newVrf, vrfDelete ):
   t0( 'CanSetVrfHook callback for interface', intfName, 'vrf', newVrf )
   cfg = tunIntfConfigDir.intfConfig.get( intfName )
   if cfg == None:
      return ( True, None )
   ( alias, errMsg ) = findTunnelConfig( cfg.mode, cfg.srcAddr, cfg.srcIntf,
                                         cfg.dstAddr, cfg, cfg.tunnelNs, 
                                         cfg.key )
   if alias:
      return ( True, errMsg )

   return ( True, None )

# Indicates if the supplied IpGenAddr can be used for a tunnel endpoint
def isValidTunnelEndpointAddress( addr ):
   if addr.isLoopback:
      return ( False, 'IP address must not be loopback' )
   if addr.isMulticast:
      return ( False, 'IP address must not be multicast' )
   if addr.isLinkLocal:
      return ( False, 'IP address must not be link-local' )
   if addr.isUnspecified:
      return ( False, 'IP address must not be zero' )
   if addr.isReserved:
      return ( False, 'IP address must not be reserved' )
   return ( True, '' )

def getHardwareSupportedModes():
   encapModes = []
   if hwCapabilities and hwCapabilities.staticInterfaceTunnelSupport.keys():
      encapModes = hwCapabilities.staticInterfaceTunnelSupport.keys()
   if hw6Capabilities and hw6Capabilities.staticInterfaceTunnelSupport.keys():
      encapModes += hw6Capabilities.staticInterfaceTunnelSupport.keys()
   return encapModes

def getDefaultEncapMode():
   encapModes = getHardwareSupportedModes()
   if encapModes and tunModeEnum.tunnelIntfModeGre not in encapModes:
      return encapModes[0]

   return tunModeEnum.tunnelIntfModeGre

def checkTunnelMode( tunMode ):
   if not tunMode or tunMode == tunModeEnum.tunnelIntfModeUnspecified:
      tunMode = getDefaultEncapMode()
   return tunMode

# Indicates if a mode and endpoint address are compatible.
def isValidTunnelModeContext( cfg, tunMode=None, tunAddr=None, tunKey=None ):
   assert cfg and ( tunMode or tunAddr or tunKey )
   if not tunAddr:
      if not cfg.srcAddr.isUnspecified:
         tunAddr = cfg.srcAddr
      elif not cfg.dstAddr.isUnspecified:
         tunAddr = cfg.dstAddr
   if not tunMode:
      tunMode = cfg.mode
   tunMode = checkTunnelMode( tunMode )

   # check for trying to use mode supported only over IPv4
   if tunMode == tunModeEnum.tunnelIntfModeIpsec:
      if tunAddr and ( not tunAddr.isUnspecified ) and \
         ( tunAddr.af != afEnum.ipv4 ):
         return ( False, 'IPSec over' + tunAddr.af + ' is unsupported' )

   if tunMode == tunModeEnum.tunnelIntfModeIpip: 
      if tunKey or cfg.key:
         return ( False, "IPIP does not support keying" )

   return ( True, '' )

# Returns the interface namespace name or 'None' if default namespace.
def getIntfNetnsName( ifName ):
   nscfg = allIntfNsConfigDir.intfNamespaceConfig.get( ifName )
   ns = nscfg.netNsName if nscfg else DEFAULT_NS
   t0( ifName, 'namespace is', ns )
   return ns

# Returns true if srcAddr/srcIntf matches srcAddr/srcIntf in cfg, else False.
# When comparing source address with source interface, returns False.
def srcCmp( cfg, srcAddr, srcIntf, dstAddr ):
   if ( srcAddr and not srcAddr.isUnspecified and cfg.srcAddr and
        not cfg.srcAddr.isUnspecified ):
      return ( srcAddr == cfg.srcAddr )
   if ( srcIntf and cfg.srcIntf ):
      return ( srcIntf == cfg.srcIntf )
   return False

# Returns the tunnel config that matches the supplied keys
def findTunnelConfig( mode, srcAddr, srcIntf, dstAddr, cfgToSkip,
                      tunnelNs=DEFAULT_NS, key=TunnelIntfUtil.tunnelDefaultKey,
                      ipsecProfile='' ):
   if not ( mode and ( srcAddr or srcIntf ) and dstAddr ):
      return ( None, '' )
   if ( not srcIntf and srcAddr.isUnspecified ) or dstAddr.isUnspecified:
      return ( None, '' )
   for cfg in tunIntfConfigDir.intfConfig.itervalues():
      if cfgToSkip and ( cfg == cfgToSkip ):
         continue
      cfgMode = checkTunnelMode( cfg.mode )
      cmpMode = checkTunnelMode( mode )
      srcMatch = srcCmp( cfg, srcAddr, srcIntf, dstAddr )

      #Disallow config if:
      # 1. While configuring Ipsec tunnel, there exists a GRE tunnel with the
      #    same src/dst/tunnelNs
      # 2. While configuring a GRE tunnel, there exists an Ipsec tunnel with the
      #    same src/dst/tunnelNs
      # 3. While configuring an Ipsec tunnel, there exists an Ipsec tunnel with the
      #    same src/dst/tunnelNs
      # Note: This only applies to Ipsec tunnels. Multiple GRE tunnels with the same
      #       same src/dst/tunnelNs are allowed, if the key is different.
      if cmpMode == tunModeEnum.tunnelIntfModeIpsec and \
         ( cfgMode == tunModeEnum.tunnelIntfModeGre or \
         cfgMode == tunModeEnum.tunnelIntfModeIpsec ) and \
         dstAddr == cfg.dstAddr and srcMatch and \
         cfg.tunnelNs == tunnelNs:
         errString = "Conflicts with tunnel %s with same source and destination." \
                     % ( cfg.intfId ) 
         return ( cfg, errString )

      if cmpMode == tunModeEnum.tunnelIntfModeGre and \
         cfgMode == tunModeEnum.tunnelIntfModeIpsec and \
         dstAddr == cfg.dstAddr and srcMatch and \
         cfg.tunnelNs == tunnelNs:
         errString = "Conflicts with tunnel %s with same source and destination." \
                     % ( cfg.intfId ) 
         return ( cfg, errString )

      #If for 2 tunnels, the src, dest tunnelNs match; the ipsec profiles
      #have to match too.
      if ipsecProfile != '' and \
         dstAddr == cfg.dstAddr and srcMatch and \
         cfg.tunnelNs == tunnelNs and cfg.ipsecProfile != '' and \
         cfg.ipsecProfile != ipsecProfile:
         errString = "Ipsec profile mismatch between %s and %s" \
                     % ( cfg.intfId, cfgToSkip.intfId ) 
         return ( cfg, errString ) 
      # If for 2 tunnels, all 4 of ( src, dest, key, mode ) match
      # but they use different tunnelNs then it's considred unique.
      # BUT if they use same tunnelNs and even if they are in different
      # nsName (VRFs), then this is a DUPLICATE config because the kernel
      # needs to use the same tunnelNs for packet I/O in/out of the box
      if cfgMode != cmpMode or cfg.dstAddr != dstAddr or ( not srcMatch ) \
         or cfg.key != key or cfg.tunnelNs != tunnelNs:
         continue
      return ( cfg, 'Conflicts with ' + cfg.intfId )

   return ( None, '' )

# Returns tunnel source address. In case the source is an interface, returns 
# IPV4 or IPv6 address (matching the address family of the destination IP) 
# of the interface
def findTunnelSourceAddr( srcAddr, srcIntf, dstAddr ):
   if srcAddr and not srcAddr.isUnspecified:
      return srcAddr
   elif srcIntf:
      if dstAddr.af == afEnum.ipv4:
         ip4Config = ip4ConfigDir.ipIntfConfig.get( srcIntf )
         if not ip4Config or not ip4Config.addrWithMask:
            return None
         srcAddr = Tac.Value( 'Arnet::IpGenAddr', ip4Config.addrWithMask.address )
         return None if srcAddr.isUnspecified else srcAddr
      elif dstAddr.af == afEnum.ipv6:
         ip6Config = ip6ConfigDir.intf.get( srcIntf )
         if not ip6Config or not ip6Config.addr :
            return None
         # take the first configured Ipv6 address
         addr = next( iter( ip6Config.addr ) )
         srcAddr = addr.address.stringValue
         srcAddr = Tac.Value( 'Arnet::IpGenAddr', srcAddr )
         return srcAddr
   return None

#-------------------------------------------------------------------------------
# Clear counters hook to clear tunnel counters
#-------------------------------------------------------------------------------
def clearTunnelIntfCountersHook( mode, intfs, sessionOnly, allIntfs ):
   if hwCapabilities.staticTunIntfPlatformCounterSupported:
      # Platform handles counters. TunnelIntf agent doesn't do anything.
      return
   request = tunnelIntfCliConfig.clearCountersRequest

   # "clear counters session" shouldn't trigger copy from current to snapshot table.
   if sessionOnly:
      return

   if allIntfs:
      del request[ 'all' ]
      request[ 'all' ] = True
      return

   # Clear interface list
   for x in intfs:
      del request[ x.name ]
      request[ x.name ] = True
      
IntfCli.registerClearCountersHook( clearTunnelIntfCountersHook )


#-------------------------------------------------------------------------------
# A subclass for tunnel interfaces
#-------------------------------------------------------------------------------
class TunnelIntf( IntfCli.VirtualIntf ):

   #----------------------------------------------------------------------------
   # Creates a new TunnelIntf instance of the specified name.
   #----------------------------------------------------------------------------
   def __init__( self, name, cliMode ):
      m = re.match( r'Tunnel(\d+)$', name )
      self.tunnelId = int( m.group( 1 ) )
      IntfCli.VirtualIntf.__init__( self, name, cliMode )
      self.intfConfigDir = tunIntfConfigDir
      self.intfStatuses = tunIntfStatusDir.intfStatus
      self.intfStatus = None

   #----------------------------------------------------------------------------
   # The rule for matching Tunnel interface names. When this pattern matches,
   # it returns an instance of the TunnelIntf class.
   #
   # This rule gets added to the Intf.rule when this class is registered with
   # the Intf class by calling Intf.addPhysicalIntfType, below.
   #----------------------------------------------------------------------------
   matcher = VirtualIntfRule.VirtualIntfMatcher(
      'Tunnel',
      None, None,
      value=lambda cliMode, intf: TunnelIntf( intf, cliMode ),
      helpdesc='Tunnel interface',
      rangeFunc=lambda mode: ( 0, hwCapabilities.maxTunnelIntfNum - 1
                              if hwCapabilities
                              else TunnelIntfUtil.maxTunnelIntfNum ) )

   #----------------------------------------------------------------------------
   # Creates the Interface::TunnelIntfConfig object for this interface if it
   # does not already exist.
   #----------------------------------------------------------------------------
   def createPhysical( self, startupConfig=False ):
      self.intfConfigDir.intfConfig.newMember( self.name )

   #----------------------------------------------------------------------------
   # Determines if the Interface::TunnelIntfStatus object for this interface
   # exists.
   #----------------------------------------------------------------------------
   def lookupPhysical( self ):
      self.intfStatus = self.intfStatuses.get( self.name, None )
      return ( self.intfStatus is not None )

   #----------------------------------------------------------------------------
   # Destroys the Interface::TunnelIntfConfig object for this interface if it
   # already exists.
   #----------------------------------------------------------------------------
   def destroyPhysical( self ):
      del self.intfConfigDir.intfConfig[ self.name ]

   #----------------------------------------------------------------------------
   # Returns the TunnelIntfConfig object for this interface.
   #----------------------------------------------------------------------------
   def config( self ):
      return self.intfConfigDir.intfConfig.get( self.name )

   #----------------------------------------------------------------------------
   # Returns the TunnelIntfStatus object for this interface.
   #----------------------------------------------------------------------------
   def status( self ):
      if not self.intfStatus and not self.lookupPhysical():
         return None
      return self.intfStatus

   #----------------------------------------------------------------------------
   # Returns the TunnelIntfCounterDir object for this interface.
   #----------------------------------------------------------------------------
   def getIntfCounterDir( self ):
      s = self.status()
      assert s
      sParent = s.parent
      if not sParent:
         return None
      # Read s.counter to trigger the externally implemented accessor which may
      # populate s.parent.counterDir as a side effect.
      _ignore = s.counter
      return tunnelCountersDir
   
   def ingressCountersSupported( self ):
      return self.status().ingressCounterSupported
   
   def egressCountersSupported( self ):
      return self.status().egressCounterSupported

   def countersSupported( self ):
      return self.ingressCountersSupported() or self.egressCountersSupported()
   
   def bumCountersSupported( self ):
      return bridgingHwCapabilities.tunnelIntfBumCountersSupported
   
   def umCountersSupported( self ):
      return bridgingHwCapabilities.tunnelIntfUmCountersSupported

   def countersRateSupported( self ):
      return self.countersSupported()

   def getUpdateTime( self, checkpoint ):
      currentTime = self.counter().statistics.lastUpdate

      # when counters have been cleared, return zero until they have been updated
      if checkpoint and checkpoint.statistics.lastUpdate == currentTime:
         return 0.0

      # convert 'uptime' to UTC
      return Ark.switchTimeToUtc( currentTime )

   def updateInterfaceCountersModel( self, checkpoint=None, notFoundString=None,
                                     zeroOut=False ):
      intfCountersModel = InterfaceCounters( _name=self.status().intfId )
      intfCountersModel._ingressCounters = self.ingressCountersSupported()
      intfCountersModel._egressCounters = self.egressCountersSupported()
      intfCountersModel._umCounters = self.umCountersSupported()
      # _bumCounters will be True only on software forwarded platforms.
      intfCountersModel._bumCounters = self.bumCountersSupported()
      intfCountersModel._tunnelIntfCounters = True

      def stat( attr, supported=True ):
         if not supported or self.counter() is None:
            return 0
         currentValue = getattr( self.counter().statistics, attr, None )
         if currentValue is None:
            return notFoundString
         checkpointValue = 0
         if checkpoint:
            checkpointValue = getattr( checkpoint.statistics, attr, None )
         return currentValue - checkpointValue

      if not zeroOut:
         intfCountersModel.inOctets = stat( 'inOctets' )
         intfCountersModel.inUcastPkts = stat( 'inUcastPkts' )
         intfCountersModel.inMulticastPkts = stat( 'inMulticastPkts',
                                                   ( self.bumCountersSupported() or 
                                                     self.umCountersSupported()) )
         intfCountersModel.inBroadcastPkts = stat( 'inBroadcastPkts',
                                                   self.bumCountersSupported() )
         intfCountersModel.inDiscards = stat( 'inDiscards' )
         intfCountersModel.outDiscards = stat( 'outDiscards' )
         intfCountersModel.outOctets = stat( 'outOctets' )
         intfCountersModel.outUcastPkts = stat( 'outUcastPkts' )
         intfCountersModel.outMulticastPkts = stat( 'outMulticastPkts',
                                                    ( self.bumCountersSupported() or 
                                                      self.umCountersSupported()) )
         intfCountersModel.outBroadcastPkts = stat( 'outBroadcastPkts',
                                                    self.bumCountersSupported() )
         # NOTE: do NOT use 'stat()' since last update time should never be a delta
         intfCountersModel.lastUpdateTimestamp = self.getUpdateTime( checkpoint )
      else:
         intfCountersModel.inOctets = 0
         intfCountersModel.inUcastPkts = 0
         intfCountersModel.inMulticastPkts = 0
         intfCountersModel.outOctets = 0
         intfCountersModel.outUcastPkts = 0
         intfCountersModel.outMulticastPkts = 0
         intfCountersModel.outBroadcastPkts = 0
         intfCountersModel.inDiscards = 0
         intfCountersModel.outDiscards = 0
         intfCountersModel.lastUpdateTimestamp = 0

      return intfCountersModel

   def rateValue( self, attr ):
      # Calculate the rate counter value by exponentially decay the
      # previously saved value and subtract it from the current rate
      # value.
      if self.counter() is None or not self.countersRateSupported():
         return 0.0
      counter = self.counter()
      ckpt = self.getLatestCounterCheckpoint()
      loadInterval = IntfCli.getActualLoadIntervalValue(
         self.config().loadInterval )

      currentValue = getattr( counter.rates, attr )
      if ckpt:
         ckptValue = getattr( ckpt.rates, attr, 0 )
         ckptStatsUpdateTime = getattr( ckpt.rates, "statsUpdateTime", 0 )
      else:
         ckptValue = 0
         ckptStatsUpdateTime = 0

      # If loadInterval = 0, return rate is the rate at the most recent counter
      # without being decayed.
      if loadInterval == 0:
         return currentValue

      # If an interface is error disabled, its counters would never be updated. For 
      # an error disabled interface, If clear command is issued, the last time when 
      # counters are updated will be less than the last time counter are cleared. 
      # So Addin a guard for this case!
      if counter.rates.statsUpdateTime < ckptStatsUpdateTime:
         return 0.0

      expFactor = - ( ( counter.rates.statsUpdateTime - ckptStatsUpdateTime ) /
                      float( loadInterval ) )
      decayedCkptValue = ckptValue * math.exp( expFactor )
      return max( 0.0, currentValue - decayedCkptValue )

   def updateInterfaceCountersRateModel( self ):
      cfg = self.config()
      intfCountersRateModel = InterfaceCountersRate( _name=self.name )
      intfCountersRateModel.description = cfg.description
      interval = IntfCli.getActualLoadIntervalValue( cfg.loadInterval )
      intfCountersRateModel.interval = int( round( interval ) )
      intfCountersRateModel.inBpsRate = self.rateValue( "inBitsRate" )
      intfCountersRateModel.inPpsRate = self.rateValue( "inPktsRate" )
      intfCountersRateModel.outBpsRate = self.rateValue( "outBitsRate" )
      intfCountersRateModel.outPpsRate = self.rateValue( "outPktsRate" )

      return intfCountersRateModel
   
   def getCounterCheckpoint( self, className=None, sessionOnly=False ):
      if not sessionOnly:
         # Platform agent manages the checkpoint for hardware forwarded platforms.
         # TunnelIntf agent manages the checkpoint for software forwarded platforms.
         x = tunnelCountersCheckpointDir
         if x is None:
            return None
         y = x.counterSnapshot.get( self.name )
         return y
      else:
         return IntfCli.Intf.getCounterCheckpoint( self, 
                                          className='Interface::IntfCounter',
                                          sessionOnly=sessionOnly )
      
   def clearCounters( self, sessionOnly=False ):
      if not sessionOnly:
         # We have a hook to take care of global clear
         return
      else:
         IntfCli.Intf.clearCounters( self, sessionOnly=sessionOnly )

   #----------------------------------------------------------------------------
   # Outputs information about this interface in an interface type-specific
   # manner.
   #----------------------------------------------------------------------------
   def showPhysical( self, cliMode, intfStatusModel ):
      pass
      
   def bandwidth( self ):
      return 0
   
   def hardware( self ):
      return 'tunnel'
   
   #----------------------------------------------------------------------------
   # Returns the MAC address of the Tunnel interface. Overrides the addr()
   # function of the base class.
   #----------------------------------------------------------------------------
   def addr( self ):
      tunMacAddr = self.status().macAddr
      if tunMacAddr is None:
         t0( self.name, 'has no MAC address' )
         return None
      result = Ethernet.convertMacAddrCanonicalToDisplay( tunMacAddr )
      t0( self.name, 'canonical MAC address is', result )
      return result

   #----------------------------------------------------------------------------
   # Returns a address string to be used in the "show interface" output
   #----------------------------------------------------------------------------
   def addrStr( self ):
      if not self.addr():
         return None
      # Tunnels do not have burned-in addresses (bia), so leave bia out
      return 'address is ' + self.addr()

   #----------------------------------------------------------------------------
   # Returns an unsorted list of TunnelIntf objects representing all the
   # existing Interface::TunnelIntfStatus objects.
   #----------------------------------------------------------------------------
   @staticmethod
   def getAllPhysical( cliMode ):
      intfs = []
      iDir = tunIntfStatusDir.intfStatus
      for name in iDir:
         intf = TunnelIntf( name, cliMode )
         if intf.lookupPhysical():
            intfs.append( intf )
      return intfs

   # Everytime you add a new tunnel config value
   # please please update this function. Else it is a
   # _LOT_ of debugging.
   def setDefault( self ):
      IntfCli.VirtualIntf.setDefault( self )
      cfg = self.config()
      cfg.tos = TunnelIntfUtil.tunnelDefaultTos
      cfg.ttl = TunnelIntfUtil.tunnelDefaultTtl
      cfg.mode = tunModeEnum.tunnelIntfModeUnspecified
      cfg.srcIntf = ''
      cfg.srcAddr = Tac.Value( 'Arnet::IpGenAddr', '0.0.0.0' )
      cfg.dstAddr = cfg.srcAddr
      cfg.pathMtuDiscovery = False
      cfg.mtu = TunnelIntfUtil.tunnelDefaultMtu
      cfg.key = TunnelIntfUtil.tunnelDefaultKey
      cfg.maxMss = TunnelIntfUtil.tunnelDefaultMaxMss
      cfg.ipsec = False
      cfg.ipsecProfile = ""
      cfg.tapGroup.clear()
      cfg.tapRawIntf.clear()
      cfg.tapEncapGrePreserve = False

   def getIntfStatusModel( self ):
      tunIntfStatus = TunnelIntfStatus( name=self.status().intfId )
      tunIntfStatus.tunSrcAddr = self.status().srcAddr
      tunIntfStatus.tunDstAddr = self.status().dstAddr
      tunIntfStatus.tunMode = self.status().mode
      tunIntfStatus.tunTtl = self.status().ttl
      tunIntfStatus.tunTos = self.status().tos
      tunIntfStatus.tunMtu = self.status().mtu
      tunIntfStatus.tunPathMtuDiscovery = self.status().pathMtuDiscovery
      tunIntfStatus.tunOKey = self.status().okey
      tunIntfStatus.tunIKey = self.status().ikey
      tunIntfStatus.tunHwFwd = self.status().hardwareForwarded
      tunIntfStatus.tunUnderlayVrf = self.status().underlayVrfName
      tunIntfStatus.tunDiagnosticsInfo = self.status().diagnosticsInfo.keys()
      return tunIntfStatus

   def getIntfConfigSanityModel( self ):
      tunIntfConfigSanity = TunnelIntfConfigSanity( name=self.status().intfId )
      tunIntfConfigSanity.tunSrcAddr = self.status().srcAddr
      tunIntfConfigSanity.tunDstAddr = self.status().dstAddr
      tunIntfConfigSanity.tunMode = self.status().mode
      tunIntfConfigSanity.tunTtl = self.status().ttl
      tunIntfConfigSanity.tunTos = self.status().tos
      tunIntfConfigSanity.tunMtu = self.status().mtu
      tunIntfConfigSanity.tunPathMtuDiscovery = self.status().pathMtuDiscovery
      tunIntfConfigSanity.tunOKey = self.status().okey
      tunIntfConfigSanity.tunIKey = self.status().ikey
      tunIntfConfigSanity.tunHwFwd = self.status().hardwareForwarded
      tunIntfConfigSanity.tunUnderlayVrf = self.status().underlayVrfName
      tunIntfConfigSanity.tunDiagnosticsInfo = (
            self.status().diagnosticsInfo.keys() )
      return tunIntfConfigSanity

   def getCountersErrorsModel( self ):
      if not self.countersErrorsSupported():
         return None
      tunCE = ErrorCounters()
      counter = self.counter()
      ckptInErrors = 0
      ckptOutErrors = 0
      ckpt = self.getLatestCounterCheckpoint()
      if ckpt:
         ckptInErrors = getattr( ckpt.statistics, 'inErrors', 0 )
         ckptOutErrors = getattr( ckpt.statistics, 'outErrors', 0 )
      tunCE.fcsErrors = 0
      tunCE.inErrors = counter.statistics.inErrors - ckptInErrors
      tunCE.outErrors = counter.statistics.outErrors - ckptOutErrors
      # fields that are not related to tunnel intfs. 
      tunCE.alignmentErrors = 0
      tunCE.frameTooShorts = 0
      tunCE.frameTooLongs = 0
      tunCE.symbolErrors = 0
      return tunCE

   def countersErrorsSupported( self ):
      return bridgingHwCapabilities.tunnelIntfErrorCountersSupported

#-------------------------------------------------------------------------------
# Add a group of tunnel-specific CLI commands to the "config-if" mode
#-------------------------------------------------------------------------------
class TunnelIntfConfigModelet ( CliParser.Modelet ):
   modeletParseTree = CliParser.ModeletParseTree()

   def __init__( self, cliMode ):
      CliParser.Modelet.__init__( self )

   @staticmethod
   def shouldAddModeletRule( mode ):
      return isinstance( mode.intf, TunnelIntf )

#-------------------------------------------------------------------------------
# Have the Cli Agent mount all needed state from sysdb
#-------------------------------------------------------------------------------
def Plugin( entityManager ):
   global em
   em = entityManager
   
   global tunIntfConfigDir
   tunIntfConfigDir = ConfigMount.mount(
         entityManager, 'interface/config/tunnel/intf',
         'Interface::TunnelIntfConfigDir', 'w' )

   global ip4ConfigDir
   ip4ConfigDir = LazyMount.mount(
         entityManager, "ip/config", "Ip::Config", "r" )

   global ip6ConfigDir
   ip6ConfigDir = LazyMount.mount(
         entityManager, "ip6/config", "Ip6::Config", "r" )

   global tunIntfStatusDir
   tunIntfStatusDir = LazyMount.mount(
         entityManager, 'interface/status/tunnel/intf',
         'Interface::TunnelIntfStatusDir', 'r' )

   global allIntfNsConfigDir
   allIntfNsConfigDir = LazyMount.mount(
         entityManager, Cell.path( 'interface/nsconfig' ),
         'Interface::AllIntfNamespaceConfigDir', 'r' )

   global hwCapabilities
   hwCapabilities = LazyMount.mount(
         entityManager, 'routing/hardware/status',                                   
         'Routing::Hardware::Status', 'r' )
   
   global greIntfTunnelTable
   greIntfTunnelTable = readMountTunnelTable(
      TunnelTableIdentifier.greTunnelInterfaceTunnelTable, entityManager )

   global hw6Capabilities
   hw6Capabilities = LazyMount.mount(
         entityManager, 'routing6/hardware/status',
         'Routing6::Hardware::Status', 'r' )

   sMountGroup = SharedMem.entityManager( sysdbEm=em )
   sMountInfo = Smash.mountInfo( 'reader' )
   global tunnelCountersDir
   tunnelCountersDir = sMountGroup.doMount(
      'interface/counter/tunnel/fastpath/current',
      'Smash::Interface::AllIntfCounterDir', sMountInfo )

   global tunnelCountersCheckpointDir
   tunnelCountersCheckpointDir = sMountGroup.doMount(
      'interface/counter/tunnel/fastpath/snapshot',
      'Smash::Interface::AllIntfCounterSnapshotDir', sMountInfo )

   global aleCliConfig
   aleCliConfig = LazyMount.mount( entityManager, 
         'hardware/ale/cliconfig',
         'Ale::HwCliConfig', 'r' )

   global tunnelIntfCliConfig
   tunnelIntfCliConfig = LazyMount.mount( entityManager,
         'tunnelintf/cliconfig',
         'TunnelIntf::TunnelIntfCliConfig', 'w' )
   
   global bridgingHwCapabilities
   bridgingHwCapabilities = LazyMount.mount(
         entityManager, 'bridging/hwcapabilities',
         'Bridging::HwCapabilities', 'r' )
   
   canSetVrfHook.addExtension( intfCanSetVrfHookForTunnel )

#-------------------------------------------------------------------------------
# Register the TunnelIntf class as a type of physical interface.
#-------------------------------------------------------------------------------
IntfCli.Intf.addPhysicalIntfType( TunnelIntf, TunnelAutoIntfType,
                                  withIpSupport=True )

#-------------------------------------------------------------------------------
# Associate the TunnelIntfConfigModelet with the "config-if" mode.
#-------------------------------------------------------------------------------
IntfCli.IntfConfigMode.addModelet( TunnelIntfConfigModelet )

#-------------------------------------------------------------------------------
# Cli keyword handler functions
#-------------------------------------------------------------------------------
def setSrcAddr( cliMode, args ):
   ipGenAddr = args[ 'IPGENADDR' ]
   ( valid, errString ) = isValidTunnelEndpointAddress( ipGenAddr )
   if not valid:
      cliMode.addError( errString )
      return
   tun = cliMode.intf
   cfg = tun.config()
   if not cfg.dstAddr.isUnspecified:
      if ipGenAddr.af != cfg.dstAddr.af:
         cliMode.addError( 'Must be same IP version as destination' )
         return
   ( valid, errString ) = isValidTunnelModeContext( cfg, tunAddr=ipGenAddr )
   if not valid:
      cliMode.addError( errString )
      return
   cfg.srcIntf = ''
   cfg.srcAddr = ipGenAddr
   t0( tun.name, 'source =', ipGenAddr )
      
def unsetSrc( cliMode, args ):
   tun = cliMode.intf
   cfg = tun.config()
   cfg.srcIntf = ''
   cfg.srcAddr = Tac.Value( 'Arnet::IpGenAddr', '0.0.0.0' )
   t0( tun.name, 'source unset' )

def setSrcIntf( cliMode, args ):
   srcIntf = args[ 'INTF' ]
   tun = cliMode.intf
   cfg = tun.config()
   cfg.srcAddr = Tac.Value( 'Arnet::IpGenAddr', '0.0.0.0' )
   cfg.srcIntf = srcIntf.name
   t0( tun.name, 'source intf =', srcIntf.name )
   return

def setDst( cliMode, args ):
   ipGenAddr = args[ 'IPGENADDR' ]
   ( valid, errString ) = isValidTunnelEndpointAddress( ipGenAddr )
   if not valid:
      cliMode.addError( errString )
      return
   tun = cliMode.intf
   cfg = tun.config()
   if not cfg.srcAddr.isUnspecified:
      if ipGenAddr.af != cfg.srcAddr.af:
         cliMode.addError( 'Must be same IP version as source' )
         return
   ( valid, errString ) = isValidTunnelModeContext( cfg, tunAddr=ipGenAddr )
   if not valid:
      cliMode.addError( errString )
      return
   cfg.dstAddr = ipGenAddr
   t0( tun.name, 'destination =', ipGenAddr )

def unsetDst( cliMode, args ):
   tun = cliMode.intf
   cfg = tun.config()
   cfg.dstAddr = Tac.Value( 'Arnet::IpGenAddr', '0.0.0.0' )
   t0( tun.name, 'destination unset' )

def setMode( cliMode, args ):

   keyToTunModeMap = {
      'gre': tunModeEnum.tunnelIntfModeGre,
      'ipip': tunModeEnum.tunnelIntfModeIpip,
      'ipsec': tunModeEnum.tunnelIntfModeIpsec
   }

   tunnelType = keyToTunModeMap[ args[ 'TUNNEL_MODE' ] ]
   tun = cliMode.intf
   cfg = tun.config()
   if tunnelType == tunModeEnum.tunnelIntfModeIpip:
      cliMode.addWarning( "IP in IP tunnel encapsulation is unsupported. "
         "Please use GRE encapsulation instead." )
   ( valid, errString ) = isValidTunnelModeContext( cfg, tunMode=tunnelType )
   if not valid:
      cliMode.addError( errString )
      return
   cfg.mode = tunnelType
   t0( tun.name, 'mode set to', tunnelType )

def unsetMode( cliMode, args ):
   tun = cliMode.intf
   cfg = tun.config()
   newTunMode = tunModeEnum.tunnelIntfModeUnspecified
   ( valid, errString ) = isValidTunnelModeContext( cfg, tunMode=newTunMode )
   if not valid:
      cliMode.addError( errString )
      return
   cfg.mode = newTunMode
   t0( tun.name, 'mode set to default' )

def setTtl( cliMode, args ):
   tunnelTTL = args[ 'TUNNEL_TTL' ]
   tun = cliMode.intf
   cfg = tun.config()
   if tunnelTTL and not cfg.pathMtuDiscovery:
      cliMode.addError( 'Path MTU Discovery must be enabled first' )
      return
   cfg.ttl = tunnelTTL
   t0( tun.name, 'ttl set to', tunnelTTL )

def unsetTtl( cliMode, args ):
   tun = cliMode.intf
   cfg = tun.config()
   cfg.ttl = TunnelIntfUtil.tunnelDefaultTtl
   t0( tun.name, 'ttl unset' )

def setTos( cliMode, args ):
   tunnelTos = args[ 'TUNNEL_TOS' ]
   tun = cliMode.intf
   cfg = tun.config()
   cfg.tos = tunnelTos
   t0( tun.name, 'tos set to', tunnelTos )

def unsetTos( cliMode, args ):
   tun = cliMode.intf
   cfg = tun.config()
   cfg.tos = TunnelIntfUtil.tunnelDefaultTos
   t0( tun.name, 'tos unset' )

def setPathMtuDiscovery( cliMode, args ):
   tun = cliMode.intf
   cfg = tun.config()
   cfg.pathMtuDiscovery = True
   t0( tun.name, 'path MTU discovery enabled' )

def unsetPathMtuDiscovery( cliMode, args ):
   tun = cliMode.intf
   cfg = tun.config()
   if cfg.ttl:
      cliMode.addError( 'Incompatible with non-zero TTL' )
      return
   cfg.pathMtuDiscovery = False
   t0( tun.name, 'path MTU discovery disabled' )

#-----------------------------------------------------------------
#ipsecHook that allows the TunnelIntf agent to check if a profile is 
#present in ipsec/ike/config.
#-----------------------------------------------------------------
ipsecHook = CliExtensions.CliHook()

def setIpsec( cliMode, args ):
   profile = args[ 'PROFILE' ]
   tun = cliMode.intf
   cfg = tun.config()
   if not ipsecHook.extensions():
      return
   for hook in ipsecHook.extensions():
      [ val, message ] = hook( profile )
      if not val:
         cliMode.addError( message )
         return

   if cfg.mode == tunModeEnum.tunnelIntfModeGre or \
      tunModeEnum.tunnelIntfModeUnspecified:
      cliMode.addWarning( "IPSec adds an overhead of up to 82 bytes. "
         "Example: A GRE tunnel with an MTU=1476 should be changed to 1394 "
         "when using IPSec." )

   if cfg.ipsecProfile != profile:
      cfg.ipsec = True
      cfg.ipsecProfile = profile

def noSetIpsec( cliMode, args ):
   tun = cliMode.intf
   cfg = tun.config()
   cfg.ipsec = False
   cfg.ipsecProfile = ""
 
def setKey( cliMode, args ):
   tunnelKey = args[ 'TUNNEL_KEY' ]
   tun = cliMode.intf
   cfg = tun.config()
   ( valid, errString ) = isValidTunnelModeContext( cfg, tunKey = tunnelKey )
   if not valid:
      cliMode.addError( errString )
      return
   cfg.key = tunnelKey
   t0( "%s key set to %u" % ( tun.name, tunnelKey ) )
   cliMode.addWarning( "Configuring key adds an overhead of 4 bytes. "
         "Example: A GRE tunnel with an MTU=1476 should be changed to 1472 "
         "when key is configured on it." )

def unsetKey( cliMode, args ):
   tun = cliMode.intf
   cfg = tun.config()
   cfg.key = TunnelIntfUtil.tunnelDefaultKey
   t0( "%s key unset " % ( tun.name ) )

def setMss( cliMode, args ):
   tunnelMss = args[ 'TUNNEL_MSS' ]
   tun = cliMode.intf
   cfg = tun.config()
   cfg.maxMss = tunnelMss
   t0( "%s MSS set to %u" % ( tun.name, cfg.maxMss ) )

def unsetMss( cliMode, args ):
   tun = cliMode.intf
   cfg = tun.config()
   cfg.maxMss = TunnelIntfUtil.tunnelDefaultMaxMss
   t0( "%s MSS set to %d" % ( tun.name, cfg.maxMss ) )

def setUnderlayVrf( cliMode, args ):
   underlayVrfName = args.get( 'VRF', DEFAULT_VRF )
   if not vrfExists( underlayVrfName ):
      cliMode.addError( "VRF %s doesn't exist" % underlayVrfName )
      return
   tun = cliMode.intf
   cfg = tun.config()
   cfg.underlayVrfName = underlayVrfName
   t0( "%s vrf set to %s" % ( tun.name, underlayVrfName ) )

def staticTunIntfUnderlayVrfSupported( mode, token ):
   if hwCapabilities and hwCapabilities.staticTunIntfUnderlayVrfSupported:
      return None
   return CliParser.guardNotThisPlatform

modelet = TunnelIntfConfigModelet

#-------------------------------------------------------------------------
# The "show gre tunnel static [ TUNNEL-INDEX ]" command
#-------------------------------------------------------------------------
def getL3IntfTunnelEntryModel( tunnelId, tunnelTable ):
   tunnelEntry = tunnelTable.entry.get( tunnelId, None )
   if not tunnelEntry:
      return None
   viaList = list()
   for via in tunnelEntry.via.itervalues():
      viaModel = IpVia( nexthop=via.nexthop, interface=via.intfId, type='ip' )
      viaList.append( viaModel )

   tunnelIndex = int( getTunnelIndexFromId( tunnelId ) )
   return L3IntfTunnelTableEntry( name="Tunnel%d" % tunnelIndex,
                                  index=tunnelIndex,
                                  srcAddr=tunnelEntry.tunnelInfo.src,
                                  dstAddr=tunnelEntry.tunnelInfo.dst,
                                  vias=viaList )

class ShowGreTunnelStaticCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show gre tunnel static [ TUNNEL-INDEX ]'
   data = {
      'gre': 'Show GRE information',
      'tunnel': 'GRE tunnel information',
      'static': 'Static GRE tunnel information',
      'TUNNEL-INDEX': CliMatcher.IntegerMatcher( TunnelIntfUtil.minTunnelIntfNum,
                                                 TunnelIntfUtil.maxTunnelIntfNum,
                                                 helpdesc='Tunnel Index' )
   }
   cliModel = GreIntfTunnelTable

   @staticmethod
   def handler( mode, args ):
      if 'TUNNEL-INDEX' in args:
         tunnelIds = [ getTunnelIdFromIndex( 'staticInterfaceTunnel',
                                             args[ 'TUNNEL-INDEX' ] ) ]
      else:
         tunnelIds = greIntfTunnelTable.entry

      greTunnels = {}
      for tunnelId in tunnelIds:
         tunnelEntryModel = getL3IntfTunnelEntryModel( tunnelId, greIntfTunnelTable )
         if tunnelEntryModel:
            greTunnels[ tunnelEntryModel.index ] = tunnelEntryModel
      return GreIntfTunnelTable( greTunnels=greTunnels )

BasicCli.addShowCommandClass( ShowGreTunnelStaticCmd )

#-------------------------------------------------------------------------
# register show tech-support extended gre
#-------------------------------------------------------------------------

import AgentDirectory

def _showTechGreCmds():
   cmds = [ 'bash sudo iptables --list-rules -t mangle',
            'bash sudo iptables --list-rules -t raw' ]

   maxTunnelNum = hwCapabilities.maxTunnelIntfNum - 1 \
                     if hwCapabilities else TunnelIntfUtil.maxTunnelIntfNum
   confifSanityCmd = 'show interface Tunnel0-%s config-sanity' % maxTunnelNum
   cmds += [ confifSanityCmd ]

   # Check if we aren't vEOS(Sfa)
   if not AgentDirectory.agent( em.sysname(), 'Sfa' ):
      cmds += [ 'show ip hardware fib routes' ]

   # Check if we are Sand platform
   if AgentDirectory.agent( em.sysname(), 'Sand' ):
      cmds += [ 'show platform fap fec all',
                'show platform fap eedb ip-tunnel gre interface',
                'platform fap diag getreg EPNI_IPV4_TOS',
                'platform fap diag getreg EPNI_IPV4_TTL',
                'platform fap diag getreg EPNI_IPV4_SIP' ]

   return cmds

TechSupportCli.registerShowTechSupportCmdCallback( '2018-07-06 06:42:29',
                                                   _showTechGreCmds,
                                                   extended='gre' )

matcherSource = CliMatcher.KeywordMatcher( 'source',
      helpdesc='Source of tunnel packets' )
matcherTunnel = CliMatcher.KeywordMatcher( 'tunnel',
      helpdesc='Protocol-over-protocol tunneling' )
supportedEncapModes = { 'gre': 'Generic route encapsulation protocol',
                        'ipsec': 'IPSec-over-IP encapsulation' }
hiddenEncapModes = { 'ipip': 'IP-over-IP encapsulation' }
matcherEncapMode = CliMatcher.EnumMatcher( supportedEncapModes, hiddenEncapModes )

#--------------------------------------------------------------------------------
# tunnel destination IPGENADDR
#--------------------------------------------------------------------------------
class TunnelDestinationIpGenAddrCmd( CliCommand.CliCommandClass ):
   syntax = 'tunnel destination IPGENADDR'
   noOrDefaultSyntax = 'tunnel destination ...'
   data = {
      'tunnel': matcherTunnel,
      'destination': CliMatcher.KeywordMatcher( 'destination',
         helpdesc='Destination of tunnel' ),
      'IPGENADDR': IpGenAddrMatcher( helpdesc='IP address' ),
   }
   handler = setDst
   noOrDefaultHandler = unsetDst

modelet.addCommandClass( TunnelDestinationIpGenAddrCmd )

#--------------------------------------------------------------------------------
# tunnel ipsec profile PROFILE
#--------------------------------------------------------------------------------
class TunnelIpsecProfileCmd( CliCommand.CliCommandClass ):
   syntax = 'tunnel ipsec profile PROFILE'
   noOrDefaultSyntax = 'tunnel ipsec profile ...'
   data = {
      'tunnel': matcherTunnel,
      'ipsec': CliMatcher.KeywordMatcher( 'ipsec',
         helpdesc='Secure tunnel with IPSec' ),
      'profile': CliMatcher.KeywordMatcher( 'profile',
         helpdesc='IPSec profile Name' ),
      'PROFILE': CliMatcher.PatternMatcher( pattern='[a-zA-Z0-9_-]+',
         helpdesc='Name of the profile', helpname='WORD' )
   }
   handler = setIpsec
   noOrDefaultHandler = noSetIpsec

modelet.addCommandClass( TunnelIpsecProfileCmd )

#--------------------------------------------------------------------------------
# tunnel key TUNNELKEY
#--------------------------------------------------------------------------------
class TunnelKeyCmd( CliCommand.CliCommandClass ):
   syntax = 'tunnel key TUNNEL_KEY'
   noOrDefaultSyntax = 'tunnel key ...'
   data = {
      'tunnel': matcherTunnel,
      'key': CliMatcher.KeywordMatcher( 'key', helpdesc='Set tunnel key' ),
      'TUNNEL_KEY': CliMatcher.IntegerMatcher( 1, 2**32-1, helpdesc='Key value' )
   }
   handler = setKey
   noOrDefaultHandler = unsetKey

modelet.addCommandClass( TunnelKeyCmd )

#--------------------------------------------------------------------------------
# tunnel mode ( gre | ipip | ipsec )
#--------------------------------------------------------------------------------

class TunnelModeCmd( CliCommand.CliCommandClass ):
   syntax = 'tunnel mode TUNNEL_MODE'
   noOrDefaultSyntax = 'tunnel mode ...'
   data = {
      'tunnel': matcherTunnel,
      'mode': 'Tunnel encapsulation method',
      'TUNNEL_MODE': matcherEncapMode
   }
   handler = setMode
   noOrDefaultHandler = unsetMode

modelet.addCommandClass( TunnelModeCmd )

#--------------------------------------------------------------------------------
# tunnel mss ceiling TUNNELMSS
#--------------------------------------------------------------------------------
class TunnelMssCeilingCmd( CliCommand.CliCommandClass ):
   syntax = 'tunnel mss ceiling TUNNEL_MSS'
   noOrDefaultSyntax = 'tunnel mss ceiling ...'
   # hidden as generalized "tcp mss ceiling" is preferred
   hidden = True
   data = {
      'tunnel': matcherTunnel,
      'mss': CliMatcher.KeywordMatcher( 'mss',
         helpdesc='Maximum segment in TCP SYN' ),
      'ceiling': CliMatcher.KeywordMatcher( 'ceiling',
         helpdesc='Set maximum limit' ),
      'TUNNEL_MSS': CliMatcher.IntegerMatcher( 64, 16384, helpdesc='MSS' )
   }
   handler = setMss
   noOrDefaultHandler = unsetMss

modelet.addCommandClass( TunnelMssCeilingCmd )

#--------------------------------------------------------------------------------
# tunnel path-mtu-discovery
#--------------------------------------------------------------------------------
class TunnelPathMtuDiscoveryCmd( CliCommand.CliCommandClass ):
   syntax = 'tunnel path-mtu-discovery'
   noOrDefaultSyntax = syntax
   data = {
      'tunnel': matcherTunnel,
      'path-mtu-discovery': 'Enable Path MTU discovery on tunnel'
   }
   handler = setPathMtuDiscovery
   noOrDefaultHandler = unsetPathMtuDiscovery

modelet.addCommandClass( TunnelPathMtuDiscoveryCmd )

#--------------------------------------------------------------------------------
# tunnel source IPGENADDR
#--------------------------------------------------------------------------------
class TunnelSourceIpGenAddrCmd( CliCommand.CliCommandClass ):
   syntax = 'tunnel source IPGENADDR'
   noOrDefaultSyntax = 'tunnel source ...'
   data = {
      'tunnel': 'Protocol-over-protocol tunneling',
      'source': matcherSource,
      'IPGENADDR': IpGenAddrMatcher( helpdesc='IP address' )
   }
   handler = setSrcAddr
   noOrDefaultHandler = unsetSrc

modelet.addCommandClass( TunnelSourceIpGenAddrCmd )

#--------------------------------------------------------------------------------
# tunnel source interface INTF
#--------------------------------------------------------------------------------
class TunnelSourceInterfaceCmd( CliCommand.CliCommandClass ):
   syntax = 'tunnel source interface INTF'
   data = {
      'tunnel': matcherTunnel,
      'source': matcherSource,
      'interface': 'Source interface of tunnel packets',
      'INTF': IntfCli.Intf.matcherWithIpSupport
   }
   handler = setSrcIntf

modelet.addCommandClass( TunnelSourceInterfaceCmd )

#--------------------------------------------------------------------------------
# tunnel tos TUNNEL_TOS
#--------------------------------------------------------------------------------
class TunnelTosCmd( CliCommand.CliCommandClass ):
   syntax = 'tunnel tos TUNNEL_TOS'
   noOrDefaultSyntax = 'tunnel tos ...'
   data = {
      'tunnel': matcherTunnel,
      'tos': CliMatcher.KeywordMatcher( 'tos', helpdesc='Set IP type of service' ),
      'TUNNEL_TOS': CliMatcher.IntegerMatcher( 0, 255, helpdesc='ToS' )
   }
   handler = setTos
   noOrDefaultHandler = unsetTos

modelet.addCommandClass( TunnelTosCmd )

#--------------------------------------------------------------------------------
# tunnel ttl TUNNEL_TTL
#--------------------------------------------------------------------------------
class TunnelTtlCmd( CliCommand.CliCommandClass ):
   syntax = 'tunnel ttl TUNNEL_TTL'
   noOrDefaultSyntax = 'tunnel ttl ...'
   data = {
      'tunnel': matcherTunnel,
      'ttl': CliMatcher.KeywordMatcher( 'ttl', helpdesc='Set time to live' ),
      'TUNNEL_TTL': CliMatcher.IntegerMatcher( 1, 255, helpdesc='TTL' )
   }
   handler = setTtl
   noOrDefaultHandler = unsetTtl

modelet.addCommandClass( TunnelTtlCmd )

#--------------------------------------------------------------------------------
# tunnel underlay vrf VRF
#--------------------------------------------------------------------------------
class TunnelUnderlayVrfCmd( CliCommand.CliCommandClass ):
   syntax = 'tunnel underlay VRF'
   noOrDefaultSyntax = 'tunnel underlay ...'
   hidden = not toggleTunnelIntfUnderlayVrfEnabled()
   data = {
      'tunnel': matcherTunnel,
      'underlay': CliCommand.guardedKeyword( 'underlay',
                             helpdesc='Tunnel underlay',
                             guard=staticTunIntfUnderlayVrfSupported ),
      'VRF': VrfExprFactory( helpdesc='Set tunnel underlay VRF' ),
   }
   handler = setUnderlayVrf
   noOrDefaultHandler = handler

modelet.addCommandClass( TunnelUnderlayVrfCmd )

## Commands for TapAgg GRE tunnel ##
def handleTunnelTapRawIntf( mode, args ):
   t0( "handleTunnelTapRawIntf" )
   tun = mode.intf
   cfg = tun.config()

   addOrRemoveTapRawIntf( mode, args, cfg )

def handleTunnelTapGroups( mode, args ):
   t0( "handleTunnelTapGroups" )
   tun = mode.intf
   cfg = tun.config()

   addOrRemoveTapGroups( mode, args, cfg )

def handleTapEncapGrePreserve( mode, args ):
   t0( "handleTapEncapGrePreserve" )
   tun = mode.intf
   cfg = tun.config()

   tapEncapGrePreserve = not CliCommand.isNoOrDefaultCmd( args )
   cfg.tapEncapGrePreserve = tapEncapGrePreserve

#--------------------------------------------------------------------------------
# [ no | default ] tap DEFAULT_INTF interface
#                  ( ETH_INTF | PHY_AUTO_INTF | LAG_INTF )
#--------------------------------------------------------------------------------
class TunnelTapDefaultInterfaceCmd( CliCommand.CliCommandClass ):
   syntax = ( 'tap DEFAULT_INTF interface '
              '( ETH_INTF | PHY_AUTO_INTF | LAG_INTF )' )
   noOrDefaultSyntax = ( 'tap DEFAULT_INTF interface '
                         '[ ETH_INTF | PHY_AUTO_INTF | LAG_INTF ]' )
   data = {
      'tap' : nodeTap,
      'DEFAULT_INTF' : matcherDefaultInterface,
      'interface' : 'Set interfaces for the tap port',
      'ETH_INTF' : EthPhyIntf.ethMatcher,
      'PHY_AUTO_INTF' : IntfRangeMatcher( noSingletons=True,
         explicitIntfTypes=( EthPhyAutoIntfType, ) ),
      'LAG_INTF' : IntfRangeMatcher( noSingletons=False,
         explicitIntfTypes=( LagAutoIntfType, ) ),
   }

   handler = handleTunnelTapRawIntf
   noOrDefaultHandler = handler

#--------------------------------------------------------------------------------
# [ no | default ] tap default { group GROUP }
#--------------------------------------------------------------------------------
class TunnelTapDefaultCmd( CliCommand.CliCommandClass ):
   syntax = 'tap default { group GROUP_NAME }'
   noOrDefaultSyntax = ( 'tap default '
                         ' ( group | { group GROUP_NAME } )' )
   data = {
      'tap' : nodeTap,
      'default' : 'Configure default tap group',
      'group': 'Set tap group for the interface',
      'GROUP_NAME' : matcherGroupName,
   }

   handler = handleTunnelTapGroups
   noOrDefaultHandler = handler

#--------------------------------------------------------------------------------
# [ no | default ] tap encapsulation gre preserve
#--------------------------------------------------------------------------------
class TunnelTapEncapGrePreserveCmd( CliCommand.CliCommandClass ):
   syntax = 'tap encapsulation gre preserve'
   noOrDefaultSyntax = syntax
   data = {
      'tap' : nodeTap,
      'encapsulation' : CliCommand.Node(
         matcher=CliMatcher.KeywordMatcher( 'encapsulation',
         helpdesc='Configure encapsulation parameters' ) ),
      'gre' : 'Configure GRE parameters',
      'preserve' : 'Preserve encapsulation header',
   }

   handler = handleTapEncapGrePreserve
   noOrDefaultHandler = handler

if toggleTapAggGreTunnelTerminationEnabled():
   modelet.addCommandClass( TunnelTapDefaultInterfaceCmd )
   modelet.addCommandClass( TunnelTapDefaultCmd )
   modelet.addCommandClass( TunnelTapEncapGrePreserveCmd )
