#!/usr/bin/env python
# Copyright (c) 2019 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

from __future__ import absolute_import, division, print_function

# pylint: disable=ungrouped-imports

import CliCommand
import CliMatcher
import BasicCliModes
import CliPlugin.DecapGroupCli as DecapGroupCli
from CliToken.Ip import ipMatcherForConfig
from CliPlugin.DecapGroupCli import (
      decapGroupSupportedGuard,
      decapTunnelTypeUdpSupportedGuard,
      globalUdpDestPortHandler,
      globalUdpDestPortDelHandler,
      decapTunnelGlobalUdpPortSupportedGuard,
)

matcherDecapGroup = CliCommand.guardedKeyword( 'decap-group', 
      helpdesc='Configure a decap group',
      guard=decapGroupSupportedGuard )

matcherUdp = CliCommand.guardedKeyword( 'udp',
      helpdesc='Configure tunnel type UDP global parameters',
      guard=decapTunnelTypeUdpSupportedGuard )

matcherPort = CliCommand.guardedKeyword( 'port',
      helpdesc='Configure port for the tunnel',
      guard=decapTunnelGlobalUdpPortSupportedGuard )

#--------------------------------------------------------------------------------
# [ no | default ] ip decap-group DGNAME
#--------------------------------------------------------------------------------
class IpDecapGroupDgnameCmd( CliCommand.CliCommandClass ):
   syntax = 'ip _decap-group DGNAME'
   noOrDefaultSyntax = syntax
   data = {
      'ip': ipMatcherForConfig,
      '_decap-group': matcherDecapGroup,
      'DGNAME': CliMatcher.PatternMatcher( 
                           pattern='((?!prefix-length)\\w)([a-zA-Z0-9_-]+)', 
                           helpdesc='Name of the decap group', 
                           helpname='WORD' ),
   }
   handler = DecapGroupCli.gotoDecapGroupMode
   noOrDefaultHandler = DecapGroupCli.deleteDg

BasicCliModes.GlobalConfigMode.addCommandClass( IpDecapGroupDgnameCmd )

#---------------------------------------------------------------------------------
#  [no | default] ip decap-group type udp destination port <port> payload <type>
#---------------------------------------------------------------------------------
class GlobalDecapGroupUdpDestPortCmd( CliCommand.CliCommandClass ):
   syntax = 'ip decap-group type udp destination port DEST_PORT payload PAYLOAD_TYPE'
   noOrDefaultSyntax = 'ip decap-group type udp destination port DEST_PORT ...'

   data = {
      'ip': ipMatcherForConfig,
      'decap-group': matcherDecapGroup,
      'type' : 'Configure tunnel type',
      'udp' : matcherUdp,
      'destination' : 'Configure destination global parameters',
      'port' : matcherPort,
      'DEST_PORT': CliMatcher.IntegerMatcher( 1024, 65535, 
                                              helpdesc='Destination port value' ),
      'payload': 'Configure payload for the tunnel',
      'PAYLOAD_TYPE': CliMatcher.EnumMatcher( {
                        'ip': 'IP payload type',
                        'mpls': 'MPLS payload type' } )
   }

   handler = globalUdpDestPortHandler
   noOrDefaultHandler = globalUdpDestPortDelHandler

BasicCliModes.GlobalConfigMode.addCommandClass( GlobalDecapGroupUdpDestPortCmd )
