#!/usr/bin/env python
# Copyright (c) 2020 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

from __future__ import absolute_import, division, print_function

import CliCommand
import CliMatcher
import CliPlugin.DecapGroupCli as DecapGroupCli
from CliPlugin.IntfCli import Intf
from CliPlugin.IpGenAddrMatcher import IpGenAddrMatcher
from CliPlugin.DecapGroupCli import (
      DecapGroupMode,
      tunnelTypeGuard,
      decapPayloadTypeGuard,
)
from Intf.IntfRange import IntfRangeMatcher

matcherTunnel = CliMatcher.KeywordMatcher( 'tunnel',
      helpdesc='Configure tunnel settings' )

matcherDecapInterface = CliCommand.guardedKeyword( 'decap-interface',
      helpdesc='Configure the decap interface',
      guard=DecapGroupCli.decapGroupIntfGuard )

matcherIntfAll = CliMatcher.KeywordMatcher( 'all',
      helpdesc='Use all interfaces for decap group (except ones in exclude option)' )

matcherAddrFamily = CliMatcher.KeywordMatcher( 'address-family',
      helpdesc='Configure address family' )

matcherIpv4 = CliMatcher.KeywordMatcher( 'ipv4',
      helpdesc='Configure IPv4 address family' )

matcherIpv6 = CliMatcher.KeywordMatcher( 'ipv6',
      helpdesc='Configure IPv6 address family' )

matcherAddress = CliMatcher.KeywordMatcher( 'address',
      helpdesc='Configure address on decap interface' )

matcherAll = CliMatcher.KeywordMatcher( 'all',
      helpdesc='Configure all IP addresses' )

matcherPrimary = CliMatcher.KeywordMatcher( 'primary',
      helpdesc='Configure primary IP address' )

matcherExclude = CliMatcher.KeywordMatcher( 'exclude',
      helpdesc='Exclude specific interfaces from decap group' )

matcherUdp = CliCommand.guardedKeyword( 'udp',
      helpdesc='Configure tunnel type UDP parameters',
      guard=DecapGroupCli.decapTunnelTypeUdpSupportedGuard )

matcherDestination = CliCommand.guardedKeyword( 'destination',
      helpdesc='Configure destination parameters',
      guard=DecapGroupCli.decapTunnelUdpPortSupportedGuard )

matcherTunnelType = CliCommand.Node(
      matcher=CliMatcher.EnumMatcher( {
         'ipip' : 'Set the tunnel type as IpIp',
         'gre' : 'Set the tunnel type as GRE',
         'udp' : 'Set the tunnel type as UDP',
      } ), guard=tunnelTypeGuard )

matcherPayloadType = CliCommand.Node(
      matcher=CliMatcher.EnumMatcher( {
         'mpls' : 'Set the payload type as mpls',
      } ), guard=decapPayloadTypeGuard )

def tunnelDecapIntfHandler( mode, args ):
   intfOption = args.get( 'INTF', 'all' )
   addressOption = args.get( 'primary' ) or args.get( 'all' )
   excludeIntfOption = args.get( 'INTF1' )

   if 'ipv6' in args:
      mode.handleTunnelDecapIntfV6( intfOption=intfOption,
                                    excludeIntfOption=excludeIntfOption )
   else:
      mode.handleTunnelDecapIntf( intfOption=intfOption,
                                  addressOption=addressOption,
                                  excludeIntfOption=excludeIntfOption )

def delTunnelDecapIntfHandler( mode, args ):
   intfOption = args.get( 'INTF', 'all' )
   if 'ipv6' in args:
      mode.delTunnelDecapIntfV6( intfOption=intfOption )
   else:
      mode.delTunnelDecapIntf( intfOption=intfOption )

def tunnelDecapIpHandler( mode, args ):
   decapIps = args.get( 'DECAP_IP' )
   if DecapGroupCli.maxIterForDecapIp() == 1 and len( decapIps ) > 1:
      mode.addError( 'Multiple decap IPs not supported' )
      return

   mode.handleTunnelDecapIp( decapIps=decapIps )

def delTunnelDecapIpHandler( mode, args ):
   decapIps = args.get( 'DECAP_IP' )
   mode.delTunnelDecapIp( decapIps=decapIps )

def tunnelTypeHandler( mode, args ):
   tunnelType = args.get( 'TUNNEL_TYPE' )
   mode.handleTunnelType( tunnelType )

def tunnelDestinationPortHandler( mode, args ):
   payloadType = args.get( 'PAYLOAD_TYPE' )
   mode.handleTunnelDestinationPort( payloadType=payloadType )

def delTunnelDestinationPortHandler( mode, args ):
   mode.delTunnelDestinationPort()

#--------------------------------------------------------------------------------
# [ no | default ] tunnel decap-interface all [ exclude { INTF1 } ]
#--------------------------------------------------------------------------------
class TunnelDecapAllIntfCmd( CliCommand.CliCommandClass ):
   syntax = 'tunnel decap-interface all-intf [ exclude { INTF1 } ]'
   noOrDefaultSyntax = 'tunnel decap-interface all-intf'
   
   data = {
      'tunnel' : matcherTunnel,
      'decap-interface' : matcherDecapInterface,
      'all-intf' : matcherIntfAll,
      'exclude' : matcherExclude,
      'INTF1' : IntfRangeMatcher(),
   }

   handler = tunnelDecapIntfHandler
   noOrDefaultHandler = delTunnelDecapIntfHandler

DecapGroupMode.addCommandClass( TunnelDecapAllIntfCmd )

#--------------------------------------------------------------------------------
# [ no | default ] tunnel decap-interface INTF
#--------------------------------------------------------------------------------
class TunnelDecapIntfCmd( CliCommand.CliCommandClass ):
   syntax = 'tunnel decap-interface INTF'
   noOrDefaultSyntax = syntax
   
   data = {
      'tunnel' : matcherTunnel,
      'decap-interface' : matcherDecapInterface,
      'INTF' : Intf.matcher,
   }

   handler = tunnelDecapIntfHandler
   noOrDefaultHandler = delTunnelDecapIntfHandler

DecapGroupMode.addCommandClass( TunnelDecapIntfCmd )

#--------------------------------------------------------------------------------
# [ no | default ] tunnel decap-interface all
# address-family ipv4 address ( primary | all ) [ exclude { INTF1 } ]
#--------------------------------------------------------------------------------
class TunnelDecapAllIntfIpv4Cmd( CliCommand.CliCommandClass ):
   syntax = 'tunnel decap-interface all-intf ' \
            'address-family ipv4 address ( primary | all ) [ exclude { INTF1 } ]'
   noOrDefaultSyntax = 'tunnel decap-interface all-intf address-family ipv4 ...'
   
   data = {
      'tunnel' : matcherTunnel,
      'decap-interface' : matcherDecapInterface,
      'all-intf' : matcherIntfAll,
      'address-family' : matcherAddrFamily,
      'ipv4' : matcherIpv4,
      'address' : matcherAddress,
      'primary' : matcherPrimary,
      'all' : matcherAll,
      'exclude' : matcherExclude,
      'INTF1' : IntfRangeMatcher(),
   }

   handler = tunnelDecapIntfHandler
   noOrDefaultHandler = delTunnelDecapIntfHandler

DecapGroupMode.addCommandClass( TunnelDecapAllIntfIpv4Cmd )

#--------------------------------------------------------------------------------
# [ no | default ] tunnel decap-interface INTF
# address-family ipv4 address ( primary | all )
#--------------------------------------------------------------------------------
class TunnelDecapIntfIpv4Cmd( CliCommand.CliCommandClass ):
   syntax = 'tunnel decap-interface INTF ' \
            'address-family ipv4 address ( primary | all )'
   noOrDefaultSyntax = 'tunnel decap-interface INTF address-family ipv4 ...'
   
   data = {
      'tunnel' : matcherTunnel,
      'decap-interface' : matcherDecapInterface,
      'INTF' : Intf.matcher,
      'address-family' : matcherAddrFamily,
      'ipv4' : matcherIpv4,
      'address' : matcherAddress,
      'primary' : matcherPrimary,
      'all' : matcherAll,
   }

   handler = tunnelDecapIntfHandler
   noOrDefaultHandler = delTunnelDecapIntfHandler

DecapGroupMode.addCommandClass( TunnelDecapIntfIpv4Cmd )

#--------------------------------------------------------------------------------
# [ no | default ] tunnel decap-interface all address-family ipv6 address all 
# [ exclude { INTF1 } ]
#--------------------------------------------------------------------------------
class TunnelDecapAllIntfIpv6Cmd( CliCommand.CliCommandClass ):
   syntax = 'tunnel decap-interface all-intf address-family ipv6 address all ' \
            '[ exclude { INTF1 } ]'
   noOrDefaultSyntax = 'tunnel decap-interface all-intf address-family ipv6 ...'
   
   data = {
      'tunnel' : matcherTunnel,
      'decap-interface' : matcherDecapInterface,
      'all-intf' : matcherIntfAll,
      'address-family' : matcherAddrFamily,
      'ipv6' : matcherIpv6,
      'address' : matcherAddress,
      'all' : matcherAll,
      'exclude' : matcherExclude,
      'INTF1' : IntfRangeMatcher(),
   }

   handler = tunnelDecapIntfHandler
   noOrDefaultHandler = delTunnelDecapIntfHandler

DecapGroupMode.addCommandClass( TunnelDecapAllIntfIpv6Cmd )

#--------------------------------------------------------------------------------
# [ no | default ] tunnel decap-interface INTF address-family ipv6 address all
#--------------------------------------------------------------------------------
class TunnelDecapIntfIpv6Cmd( CliCommand.CliCommandClass ):
   syntax = 'tunnel decap-interface INTF address-family ipv6 address all'
   noOrDefaultSyntax = 'tunnel decap-interface INTF address-family ipv6 ...'
   
   data = {
      'tunnel' : matcherTunnel,
      'decap-interface' : matcherDecapInterface,
      'INTF' : Intf.matcher,
      'address-family' : matcherAddrFamily,
      'ipv6' : matcherIpv6,
      'address' : matcherAddress,
      'all' : matcherAll,
   }

   handler = tunnelDecapIntfHandler
   noOrDefaultHandler = delTunnelDecapIntfHandler

DecapGroupMode.addCommandClass( TunnelDecapIntfIpv6Cmd )

#--------------------------------------------------------------------------------
# [ no | default ] tunnel decap-ip { DECAP_IP }
#--------------------------------------------------------------------------------
class TunnelDecapIpCmd( CliCommand.CliCommandClass ):
   syntax = 'tunnel decap-ip { DECAP_IP }'
   noOrDefaultSyntax = syntax

   data = {
      'tunnel' : matcherTunnel,
      'decap-ip' : CliCommand.guardedKeyword( 'decap-ip',
                     helpdesc='Configure the decap IP address',
                     guard=DecapGroupCli.decapGroupIpGuard ),
      'DECAP_IP' : CliCommand.Node(
                     matcher=IpGenAddrMatcher( 'Decap IP address' ) )
   }

   handler = tunnelDecapIpHandler
   noOrDefaultHandler = delTunnelDecapIpHandler

DecapGroupMode.addCommandClass( TunnelDecapIpCmd )

#--------------------------------------------------------------------------------
# tunnel type TUNNEL_TYPE
#--------------------------------------------------------------------------------
class TunnelDecapTypeCmd( CliCommand.CliCommandClass ):
   syntax = 'tunnel type TUNNEL_TYPE'

   data = {
      'tunnel' : matcherTunnel,
      'type' : 'Configure tunnel type',
      'TUNNEL_TYPE' : matcherTunnelType,
   }

   handler = tunnelTypeHandler

DecapGroupMode.addCommandClass( TunnelDecapTypeCmd )

#--------------------------------------------------------------------------------
# tunnel udp destination port 6635 payload PAYLOAD_TYPE
#--------------------------------------------------------------------------------
class TunnelUdpDestinationPortCmd( CliCommand.CliCommandClass ):
   syntax = 'tunnel udp destination port 6635 payload PAYLOAD_TYPE'
   noOrDefaultSyntax = 'tunnel udp destination port ...'

   data = {
      'tunnel' : matcherTunnel,
      'udp' : matcherUdp,
      'destination' : matcherDestination,
      'port' : 'Configure port for the tunnel',
      '6635' : 'Destination port',
      'payload' : 'Configure payload for the tunnel',
      'PAYLOAD_TYPE' : matcherPayloadType,
   }

   handler = tunnelDestinationPortHandler
   noOrDefaultHandler = delTunnelDestinationPortHandler

DecapGroupMode.addCommandClass( TunnelUdpDestinationPortCmd )
