# Copyright (c) 2020 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

'''
Common helper methods for Tunnel package CLI
'''

from __future__ import absolute_import, division, print_function

import Tac
from TypeFuture import TacLazyType

_tacDynamicTunnelIntfId = TacLazyType( 'Arnet::DynamicTunnelIntfId' )

staticTunnelTypeStrs = [ 'staticTunnel', 'staticV4Tunnel', 'staticV6Tunnel' ]
srTunnelTypeStrs = [ 'srTunnel', 'srV4Tunnel', 'srV6Tunnel' ]

# The 44th bit is used to differentiate between tunnel endpoints of the same
# type but different address families.
TunnelIdConstants = Tac.Type( "Tunnel::TunnelTable::TunnelIdConstants" )
tunnelIndexMask = TunnelIdConstants.tunnelIndexMask
tunnelIdAfBitMask = TunnelIdConstants.tunnelAfMask

def getNhAndIntfStrs( via ):
   if via.type == 'tunnel':
      nhStr = via.tunnelId.renderStr()
      intfStr = '-'
   else:
      nhStr = str( via.nexthop )
      intfStr = via.interface.stringValue
   return nhStr, intfStr

def isDyTunIntfId( intfId ):
   '''Return True if a given intfId from Via Model is a dynamic Interface'''
   return _tacDynamicTunnelIntfId.isDynamicTunnelIntfId( intfId )

def getDyTunTidFromIntfId( intfId ):
   '''Create a tacc DynamicTunnelIntfId from an intfId in Via Model'''
   assert isDyTunIntfId( intfId )
   return _tacDynamicTunnelIntfId.tunnelId( intfId )

def getTunnelIdFromIndex( tunnelType, index, af=None ):
   """ Given a tunnelType and a (unique) index number, returns a
   Tunnel::TunnelTable::TunnelId.

   INPUTS
   - tunnelType (str): an enum name as defined in TunnelBasicTypes.tac, or one
                       of the more specific strings { 'staticV4Tunnel',
                       'staticV6Tunnel', ... } defined in this file.
   - index (int/long): an integer represented by at most 43 bits. The 43-bit
                       limit comes from the implementation of TunnelId. The 44th
                       bit of index is used to denote whether the tunnel is
                       IPv4 or IPv6.

   OUTPUTS
   - A Tunnel::TunnelTable::TunnelId instance. This is the unique identifier of
     all tunnels.
   """
   # TunnelType enum is defined in TunnelBasicTypes.tac
   tunnelTypeEnumVal = tunnelType
   if tunnelType in staticTunnelTypeStrs:
      tunnelTypeEnumVal = "staticTunnel"
      # FIXME BUG200023: shift this logic into TunnelId.convertToTunnelValue.
      # Use the 44th bit set aside for the index in order to differentiate
      # between static IPv6 and static IPv4 tunnels.
      if tunnelType == 'staticV6Tunnel' or af == 'ipv6' :
         index = index | tunnelIdAfBitMask
   elif tunnelType in srTunnelTypeStrs:
      tunnelTypeEnumVal = "srTunnel"
      # FIXME BUG200023: shift this logic into TunnelId.convertToTunnelValue.
      # Use the 44th bit set aside for the index in order to differentiate between
      # ISIS-SR IPv6 and ISIS-SR IPv4 tunnels. Two different instances of LfibGenSm,
      # one each for ISIS-SR IPv4 and IPv6, write tunnels into the same tunnel table.
      # Hence this need to offset the tunnelIndex.
      if tunnelType == 'srV6Tunnel' or af == 'ipv6' :
         index = index | tunnelIdAfBitMask
   return Tac.Value( "Tunnel::TunnelTable::TunnelId" ) \
            .convertToTunnelValue( tunnelTypeEnumVal, index )
