#Copyright (c) 2015 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import AgentDirectory
import Arnet
import BasicCli
import CliCommon
import CliCommand
import CliMatcher
import CliPlugin.IpAddrMatcher as IpAddrMatcher
import CliPlugin.IpGenAddrMatcher as IpGenAddrMatcher
from CliPlugin import (
   TechSupportCli,
   TunnelFibModel,
   TunnelModels,
)
from CliPlugin.TunnelCliLib import (
   getDyTunTidFromIntfId,
   getNhAndIntfStrs,
   isDyTunIntfId,
   srTunnelTypeStrs,
   staticTunnelTypeStrs,
   tunnelIndexMask,
   tunnelIdAfBitMask,
)
from CliPlugin.TunnelFibModel import (
   getTunnelFibViaModel,
)
from CliPlugin.TunnelModels import (
   getTunnelViaModelFromTunnelIntf,
)
import ShowCommand
import SharedMem
import Smash
import SmashLazyMount
import Tac
import Tracing
import LazyMount
import TunnelAgent
import TunnelRibModel
import AgentCommandRequest

from TunnelRibCli import (
   systemTunnelRibName,
   systemTunnelRibId,
   systemColoredTunnelRibName,
   systemColoredTunnelRibId,
   systemTunnelingLdpTunnelRibName,
   systemIgpShortcutTunnelRibName,
   TunnelRibNameMatcher,
)

from TypeFuture import TacLazyType

# All imports below are for compatibility only. DO NOT place new imports below here.
# pylint: disable=unused-import,ungrouped-imports
from CliPlugin.TunnelCliLib import getTunnelIdFromIndex

FecIdIntfId = TacLazyType( 'Arnet::FecIdIntfId' )
FecId = TacLazyType( 'Smash::Fib::FecId' )
TunnelTableIdentifier = TacLazyType( "Tunnel::TunnelTable::TunnelTableIdentifier" )
TunnelTableMounter = TacLazyType( "Tunnel::TunnelTable::TunnelTableMounter" )
readerInfo = SmashLazyMount.mountInfo( 'reader' )
keyshadowInfo = SmashLazyMount.mountInfo( 'keyshadow' )

em = None
sysname = None
ldpTunnelTable = None
staticTunnelTable = None
srTunnelTable = None
tilfaTunnelTable = None
nexthopGroupTunnelTable = None
rsvpFrrTunnelTable = None
rsvpLerTunnelTable = None
bgpLuTunnelTable = None
srTePolicyTunnelTable = None
systemTunnelRib = None
systemColoredTunnelRib = None
tunnelRibNameIdMap = None
tunnelFib = None
forwardingStatus = None
forwarding6Status = None
programmingStatus = None
nhgStatus = None
tunnelSmashEm = None
tunnelRibWatcher = None

__defaultTraceHandle__ = Tracing.Handle( 'TunnelCli' )
t0 = Tracing.trace0

#-------------------------------------------------------------------------
# Common CLI tokens for Tunnel package
#-------------------------------------------------------------------------
tokenColored = CliMatcher.KeywordMatcher(
   "colored", helpdesc="Colored tunnel RIBs" )
tokenTunnelMatcher = CliMatcher.KeywordMatcher(
   'tunnel', helpdesc="Show tunnel information" )

tunnelIndexMin = 0
tunnelIndexMax = 0x00000FFFFFFFFFFF
tunnelIndexMatcher = CliMatcher.IntegerMatcher( tunnelIndexMin, tunnelIndexMax,
                                                helpdesc="Tunnel Index" )

tunnelIgpPreferenceMatcher = CliMatcher.KeywordMatcher(
   'preference', helpdesc="IGP preference for tunnel" )
tunnelIgpPreferenceRangeMatcher = CliMatcher.IntegerMatcher(
   0, 255, helpdesc="IGP preference for tunnel" )

tunnelIgpMetricMatcher = CliMatcher.KeywordMatcher(
   'metric', helpdesc="IGP metric for tunnel" )
tunnelIgpMetricRangeMatcher = CliMatcher.IntegerMatcher( 0, 4294967295,
                                                         helpdesc='IGP metric' )

#-------------------------------------------------------------------------
# Common helper methods for Tunnel package CLI
#-------------------------------------------------------------------------
def isAfBitSetInTunnelId( tunnelId ):
   # Mask out the tunnel type and tunnel index bit
   if ( tunnelId & tunnelIdAfBitMask ) >> 43:
      return True
   return False

TunnelViaStatus = Tac.Type( 'Tunnel::Hardware::TunnelViaStatus' )

def getEndpointFromTunnelId( tunnelId ):
   tunnelType = Tac.Value( "Tunnel::TunnelTable::TunnelId",
                           tunnelId ).tunnelType()
   tunnelTable = None
   if tunnelType == 'bgpLuTunnel':
      tunnelTable = bgpLuTunnelTable
   elif tunnelType == 'ldpTunnel':
      tunnelTable = ldpTunnelTable
   elif tunnelType in staticTunnelTypeStrs:
      tunnelTable = staticTunnelTable
   elif tunnelType in srTunnelTypeStrs:
      tunnelTable = srTunnelTable
   elif tunnelType == 'nexthopGroupTunnel':
      tunnelTable = nexthopGroupTunnelTable
   elif tunnelType == 'rsvpFrrTunnel':
      tunnelTable = rsvpFrrTunnelTable
   elif tunnelType == 'rsvpLerTunnel':
      tunnelTable = rsvpLerTunnelTable

   if tunnelTable and tunnelTable.entry:
      tunnelEntry = tunnelTable.entry.get( tunnelId )
      if tunnelEntry:
         return tunnelEntry.tep
   return None

def getColoredEndpointFromTunnelId( tunnelId ):
   tunnelType = Tac.Value( "Tunnel::TunnelTable::TunnelId",
                           tunnelId ).tunnelType()
   tunnelTable = None
   if tunnelType == 'srTePolicyTunnel':
      tunnelTable = srTePolicyTunnelTable

   if tunnelTable and tunnelTable.entry:
      tunnelEntry = tunnelTable.entry.get( tunnelId )
      if tunnelEntry:
         return tunnelEntry.tep, tunnelEntry.color
   return None, None

def getTunnelIndexFromId( tunnelId ):
   return long( Tac.Value( "Tunnel::TunnelTable::TunnelId", tunnelId ).tunnelIndex() 
                & tunnelIndexMask )

def getTunnelTypeFromTunnelId( tunnelId ):
   # The 44th bit is used to differentiate between tunnel endpoints of the same
   # type but different address families. Currently only SR
   tunnelType = Tac.Value( "Tunnel::TunnelTable::TunnelId",
                           tunnelId ).tunnelType()
   # If this is a SR or static tunnel, return tunnel type corresponding to AF
   if tunnelType == 'srTunnel':
      if isAfBitSetInTunnelId( tunnelId ):
         return 'srV6Tunnel'
      else:
         return 'srV4Tunnel'
   elif tunnelType == 'staticTunnel':
      if isAfBitSetInTunnelId( tunnelId ):
         return 'staticV6Tunnel'
      else:
         return 'staticV4Tunnel'
   else:
      return tunnelType

def getPrimaryIntfFromTunnel( tunnelTable, tunnelId ):
   tunnelIntf = None
   tunnelTableEntry = tunnelTable.entry.get( tunnelId )
   if tunnelTableEntry:
      tunnelIntf = tunnelTableEntry.via.intfId
   return tunnelIntf

def getMplsViaModel( mplsVia ):
   labels = []

   for mplsStackIndex in reversed ( range( mplsVia.labels.stackSize ) ):
      labels.append( str( mplsVia.labels.labelStack( mplsStackIndex ) ) )
   intf = mplsVia.intfId
   backupTunnelInfo = None
   if isDyTunIntfId( mplsVia.intfId ):
      # SR tunnels point to TI-LFA tunnels
      tunnelId = Tac.Value( "Tunnel::TunnelTable::TunnelId",
                            getDyTunTidFromIntfId( mplsVia.intfId ) )
      tunnelIndex = tunnelId.tunnelIndex()
      tunnelType = tunnelId.typeCliStr()
      backupTunnelInfo = TunnelModels.TunnelInfo( tunnelIndex=tunnelIndex,
                                                  tunnelType=tunnelType,
                                                  tunnelVias=None )
      intf = getPrimaryIntfFromTunnel(
         tilfaTunnelTable, tunnelId ) or intf

   return TunnelModels.MplsVia( nexthop=mplsVia.nexthop, interface=intf,
                                type='ip', labels=labels,
                                backupTunnelInfo=backupTunnelInfo )

def getMplsViaModelFromTunnelVia( tunnelVia ):
   labelStackEncap = tunnelFib.labelStackEncap.get( tunnelVia.encapId )
   if labelStackEncap is not None:
      labelOp = labelStackEncap.labelStack
   else:
      labelOp = Tac.Value( 'Arnet::MplsLabelOperation' )
   labels = []
   for mplsStackIndex in reversed ( range( labelOp.stackSize ) ):
      labels.append( str( labelOp.labelStack( mplsStackIndex ) ) )
   vias = []
   if FecIdIntfId.isFecIdIntfId( tunnelVia.intfId ):
      fecId = FecId( FecIdIntfId.intfIdToFecId( tunnelVia.intfId ) )
      if fecId.adjType() == 'usedByTunnelV4Adj':
         fecId = FecId( FecId.fecIdToNewAdjType( 'fibV4Adj', fecId ) )
      elif fecId.adjType() == 'usedByTunnelV6Adj':
         fecId = FecId( FecId.fecIdToNewAdjType( 'fibV6Adj', fecId ) )

      fec = None
      if fecId.adjType() == 'fibV4Adj':
         fec = forwardingStatus.fec.get( fecId )
      elif fecId.adjType() == 'fibV6Adj':
         fec = forwarding6Status.fec.get( fecId )
      if fec is None:
         return vias
      for fecVia in sorted( fec.via.values() ):
         addr = Tac.Value( "Arnet::IpGenAddr", str( fecVia.hop ) )
         vias.append( TunnelModels.MplsVia( nexthop=addr,
                                            interface=fecVia.intfId,
                                            type='ip',
                                            labels=labels ) )
   else:
      vias.append( TunnelModels.MplsVia( nexthop=tunnelVia.nexthop,
                                         interface=tunnelVia.intfId,
                                         type='ip',
                                         labels=labels ) )
   return vias

# create a TunnelId Model object for a tunnelId tacc object.
def getTunnelIdModel( tunnelId ):
   tacTid = Tac.Value( "Tunnel::TunnelTable::TunnelId", tunnelId )
   # If tunnelId type is 'staticTunnel' or 'srTunnel', we need to make a
   # distinction between IPv4 and IPv6 tunnels. We do this by looking at the
   # 44th bit of the tunnel ID.
   TunnelType = Tac.Type( "Tunnel::TunnelTable::TunnelType" )
   tunnelType = tacTid.tunnelType()
   if tunnelType == TunnelType.staticTunnel:
      if not isAfBitSetInTunnelId( tunnelId ): # IPv4
         tunnelType = 'staticV4Tunnel'
      else: # IPv6
         tunnelType = 'staticV6Tunnel'
   elif tunnelType == TunnelType.srTunnel:
      if not isAfBitSetInTunnelId( tunnelId ): # IPv4
         tunnelType = 'srV4Tunnel'
      else: # IPv6
         tunnelType = 'srV6Tunnel'
   return TunnelModels.TunnelId(
         index=tacTid.tunnelIndex(),
         type=TunnelModels.tunnelTypeStrDict[ tunnelType ] )

# create a TunnelVia object from via dictionary.
def getTunnelViaModel( via ):
   if 'tunnelId' not in via:
      # this can happen with EosImage/test/CliModelRevisionTest.py 
      # The test choose all possible values of type ['ip', 'tunnel'] 
      # to generate a dictionary of via for MplsVia with type as 'tunnel'.
      # MplsVia does not have tunnelId argument, so to avoid keyError
      # we raise CliModelNotDegradable to make CliModelRevisionTest.py
      # happy.
      raise CliCommon.CliModelNotDegradable( "Invalid via dictionary" )
   tunnelId = TunnelModels.TunnelId(
         index=via[ 'tunnelId' ][ 'index' ],
         type=via[ 'tunnelId' ][ 'type' ] )
   return TunnelModels.TunnelVia( tunnelId=tunnelId, type='tunnel' )

# create a IpVia object from via dictionary.
def getIpViaModel( via ):
   return TunnelModels.IpVia( nexthop=via[ 'nexthop' ], 
                              interface=via[ 'interface' ],
                              type='ip' )

# create a TunnelVia or IpVia Model object from via dict.
def getViaModelFromViaDict( via ):
   if via[ 'type' ] == 'tunnel':
      return getTunnelViaModel( via )
   else:
      return getIpViaModel( via )

# create a list of TunnelVias and/or IpVias from generic Via Model object.
def getViaModels( via, bgpLuPushTunnelTable=None, flattenLuPush=True,
                  flatteningCall=False, useMplsVias=False ):
   viaModels = []
   if not isDyTunIntfId( via.intfId ):
      if useMplsVias:
         labels = []
         for mplsStackIndex in reversed ( range( via.labels.stackSize ) ):
            labels.append( str( via.labels.labelStack( mplsStackIndex ) ) )
         viaModels.append( TunnelModels.MplsVia( nexthop=via.nexthop,
                                                 interface=via.intfId,
                                                 labels=labels,
                                                 type='ip' ) )
      else:
         viaModels.append( TunnelModels.IpVia( nexthop=via.nexthop,
                                               interface=via.intfId, type='ip' ) )
   else:
      tid = getDyTunTidFromIntfId( via.intfId )
      tacTid = Tac.Value( "Tunnel::TunnelTable::TunnelId", tid )
      TunnelType = Tac.Type( "Tunnel::TunnelTable::TunnelType" )
      if tacTid.tunnelType() != TunnelType.bgpLuPushTunnel or not flattenLuPush:
         if useMplsVias:
            viaModels.append( TunnelModels.MplsTunnelVia( 
                                 tunnelId=getTunnelIdModel( tid ), type='tunnel' ) )
         else:
            viaModels.append( TunnelModels.TunnelVia( 
                                 tunnelId=getTunnelIdModel( tid ), type='tunnel' ) )
      else:
         flattenedVias = getFlattenedViaModels( via, bgpLuPushTunnelTable,
                                                tacTid, TunnelType )
         viaModels.extend( flattenedVias )
   return viaModels

# recursively flatten out any BGP LU Push Via Models
def getFlattenedViaModels( via, bgpLuPushTunnelTable, tacTid, TunnelType ):
   viaModels = []
   labels = []
   entry = bgpLuPushTunnelTable.entry.get( tacTid )
   if entry:
      for i in reversed( range( entry.labels.stackSize ) ):
         labels.append( str( entry.labels.labelStack( i ) ) )
      for nextVia in entry.via.itervalues():
         if not isDyTunIntfId( nextVia.intfId ):
            viaModels.append( TunnelModels.MplsVia(
                                 nexthop=nextVia.nexthop, labels=labels,
                                 interface=nextVia.intfId, type='ip' ) )
         else:
            tid = getDyTunTidFromIntfId( nextVia.intfId )
            tacTid = Tac.Value( "Tunnel::TunnelTable::TunnelId", tid )
            if tacTid.tunnelType() != TunnelType.bgpLuPushTunnel:
               viaModels.append( TunnelModels.MplsTunnelVia( labels=labels,
                                    tunnelId=getTunnelIdModel( tid ),
                                    type='tunnel' ) )
            else:
               flattenedVias = getFlattenedViaModels( nextVia, 
                                                      bgpLuPushTunnelTable,
                                                      tacTid, TunnelType )
               fViasWithLabels = appendEntryLabelsToViaModels( flattenedVias,
                                                               labels )
               viaModels.extend( fViasWithLabels )
   return viaModels

def appendEntryLabelsToViaModels( vias, labels ):
   viaModels = []
   for via in vias:
      if hasattr( via, 'labels' ):
         baseLabels = []
         for l in via.labels:
            baseLabels.append( l )
         via.labels = baseLabels + labels
      viaModels.append( via )
   return viaModels

def getNhAndIntfAndLabelStrs( via ):
   nhStr, intfStr = getNhAndIntfStrs( via )
   if hasattr( via, 'labels' ) and via.labels:
      labelsStr = '[ ' + ' '.join( via.labels ) + '  ]'
   else:
      labelsStr = '-'
   return nhStr, intfStr, labelsStr

def getTunnelViaStatusFromTunnelId( tunnelId ):
   if tunnelId in programmingStatus.tunnelStatus.keys():
      tunnelStatus = programmingStatus.tunnelStatus[ tunnelId ]
      return TunnelFibModel.tunnelViaStatusTacToCapi[ tunnelStatus.tunnelViaStatus ]
   else:
      return 'notProgrammed'

def getFullTunnelTableMountInfo( tunnelTableId ):
   return TunnelTableMounter.getMountInfo( tunnelTableId )

def getTunnelTableMountInfo( tunnelTableId ):
   return getFullTunnelTableMountInfo( tunnelTableId ).tableInfo

def mountTunnelTable( entityManager, tableInfo, mountInfo, lazy=True ):
   if lazy:
      return SmashLazyMount.mount(
         entityManager, tableInfo.mountPath, tableInfo.tableType, mountInfo )
   else:
      shmemEm = SharedMem.entityManager( sysdbEm=entityManager )
      return shmemEm.doMount( tableInfo.mountPath, tableInfo.tableType, mountInfo )

def readMountTunnelTable( tunnelTableId, entityManager, lazy=True ):
   """NOTE: Make sure to set lazy=False if passing handle to a C++ function."""
   tableInfo = getTunnelTableMountInfo( tunnelTableId )

   return mountTunnelTable(
      entityManager, tableInfo, Smash.mountInfo( "reader" ), lazy=lazy )

def keyshadowMountTunnelTable( tunnelTableId, entityManager, lazy=True ):
   """NOTE: Make sure to set lazy=False if passing handle to a C++ function."""
   tableInfo = getTunnelTableMountInfo( tunnelTableId )

   return mountTunnelTable(
      entityManager, tableInfo, Smash.mountInfo( "keyshadow" ), lazy=lazy )

def writeMountTunnelTable( tunnelTableId, entityManager, lazy=True ):
   """NOTE: Make sure to set lazy=False if passing handle to a C++ function."""
   tableMountInfo = getFullTunnelTableMountInfo( tunnelTableId )
   tableInfo = tableMountInfo.tableInfo
   writerInfo = tableMountInfo.writerMountInfo

   return mountTunnelTable( entityManager, tableInfo, writerInfo, lazy=lazy )

#-------------------------------------------------------------------------
# The "show tunnel rib [<prefix>] [brief]" command.
#-------------------------------------------------------------------------
def getTunnelRibEntry( tunnelRib, key ):
   if tunnelRib and key:
      tunnelRibEntry = tunnelRib.entry.get( key )
      if not tunnelRibEntry:
         return None
      if not tunnelRibEntry.tunnelId:
         # Somehow we have ended up in a transient state where there exists a
         # Tunnel RIB entry with no Tunnel IDs. We should just ignore this
         # Tunnel RIB entry and not add it to the CLI/CAPI output.
         return None
      return tunnelRibEntry
   return None

def getTunnelType( tunnelRib, key ):
   tunnelRibEntry = getTunnelRibEntry( tunnelRib, key )
   if not tunnelRibEntry:
      return None

   # only care about the tunnelType for first tunnelId
   tunnelIds = tunnelRibEntry.tunnelId
   tunnelType = getTunnelTypeFromTunnelId( tunnelIds[ 0 ] )
   return TunnelModels.tunnelTypeStrDict[ tunnelType ]

def getTunnelRibEntryModel( tunnelRib, key ):
   tunnelRibEntry = getTunnelRibEntry( tunnelRib, key )
   if not tunnelRibEntry:
      return None

   # get tunnel indexes
   tunnelIds = tunnelRibEntry.tunnelId.values()
   tunnelIndexes = \
      [ getTunnelIndexFromId( tunnelId ) for tunnelId in tunnelIds ]
   # only care about the tunnelType for first tunnelId
   tunnelType = getTunnelTypeFromTunnelId( tunnelIds[ 0 ] )
   tunnelIndexes.sort()

   return TunnelRibModel.TunnelRibEntry(
      tunnelType=TunnelModels.tunnelTypeStrDict[ tunnelType ],
      tunnelIndexes=tunnelIndexes,
      metric=tunnelRibEntry.metric, metric2=tunnelRibEntry.metric2,
      pref=tunnelRibEntry.pref, pref2=tunnelRibEntry.pref2,
      tunnelPreference=tunnelRibEntry.entryPref )

def getColoredTunnelRibModel( trName, tunnelRib, prefix, color ):
   coloredRibModel = TunnelRibModel.ColoredTunnelRib( name=trName )
   tep = Arnet.IpGenPrefix( str( prefix ) ) if prefix is not None else None
   if tep and color is not None:
      key = Tac.Value( "Tunnel::TunnelTable::ColoredTunnelEndpoint",
                       tep, color, True )
      tunnelRibEntryModel = getTunnelRibEntryModel( tunnelRib, key )
      if tunnelRibEntryModel:
         coloredRibModel.addColoredRibEntry( key.tep, key.color,
                                             tunnelRibEntryModel )
      return coloredRibModel
   if tunnelRib:
      for coloredTep in tunnelRib.entry:
         tunnelRibEntryModel = getTunnelRibEntryModel( tunnelRib, coloredTep )
         if not tunnelRibEntryModel:
            continue
         if tep and tep != coloredTep.tep:
            continue
         if color is not None and color != coloredTep.color:
            continue
         coloredRibModel.addColoredRibEntry( coloredTep.tep, coloredTep.color,
                                             tunnelRibEntryModel )
   return coloredRibModel

def getTunnelRibAllSummaryModel( tunnelRibs ):
   tepCountByTunnelRibName = {}
   for tunnelName, tunnelRib in tunnelRibs.iteritems():
      if tunnelRib:
         tepCountByTunnelRibName[ tunnelName ] = len( tunnelRib.entry )
   return TunnelRibModel.TunnelRibAllSummary( tunnels=tepCountByTunnelRibName )

def getTunnelRibSummaryModel( trName, tunnelRib ):
   tepCountByTunnelType = {}
   if tunnelRib:
      for tep in tunnelRib.entry:
         tunnelType = getTunnelType( tunnelRib, tep )
         tepCountByTunnelType[ tunnelType ] = \
               tepCountByTunnelType.get( tunnelType, 0 ) + 1
   return TunnelRibModel.TunnelRibSummary( name=trName,
                                           tunnels=tepCountByTunnelType )

def getTunnelRibModel( trName, tunnelRib, prefix=None, color=None ):
   entries = {}
   if tunnelRib:
      if tunnelRib.isColored():
         return getColoredTunnelRibModel( trName, tunnelRib, prefix, color )
      if prefix is None:
         for tep in tunnelRib.entry:
            tunnelRibEntryModel = getTunnelRibEntryModel( tunnelRib, tep )
            if tunnelRibEntryModel:
               entries[ tep ] = tunnelRibEntryModel
      else:
         if not isinstance( prefix, str ):
            prefix = str( prefix )
         prefix = Arnet.IpGenPrefix( prefix )
         tunnelRibEntryModel = getTunnelRibEntryModel( tunnelRib, prefix )
         if tunnelRibEntryModel:
            entries[ prefix ] = tunnelRibEntryModel
   return TunnelRibModel.TunnelRib( name=trName, entries=entries )

class TunnelRibWatcher( Tac.Notifiee ) :
   '''
   TunnelRibWatcher reacts to the nameToId attribute of TunnelRibNameIdMap,
   mounts, and caches all tunnel ribs including the system tunnel rib. This
   allows for more efficient mounting( and unmounting ) of tunnel rib
   entities required for show commands.

   Reactor:
   TunnelRibNameIdMap:: | mountedTunnelRibs | action
      nameToId          |                   |
   --------------------------------------------------------------------------------
   --------------------------------------------------------------------------------
          no            |        no         |  nothing to do
   --------------------------------------------------------------------------------
          no            |        yes        |  tunnel has been
                        |                   |  deleted, remove from
                        |                   |  watcher
   --------------------------------------------------------------------------------
         yes            |       yes         | already mounted so
                        |                   | nothing to do
   --------------------------------------------------------------------------------
         yes            |       no          | tunnel not mounted so
                        |                   | mount it

   show command:
   TunnelRibNameIdMap:: | mountedTunnelRibs | action
      nameToId          |                   |
   --------------------------------------------------------------------------------
   --------------------------------------------------------------------------------
          no            |        no         |  nothing to show
   --------------------------------------------------------------------------------
          no            |       yes         |  Do nothing as watcher will be called
                        |                   |  to delete the entry
   --------------------------------------------------------------------------------
         yes            |       yes         | tunnel in Watcher
                        |                   | return the model
   --------------------------------------------------------------------------------
         yes            |       no          | tunnel not mounted so
                        |                   | mount and return model


   '''
   notifierTypeName = "Tunnel::TunnelTable::TunnelRibNameIdMap"
   def __init__( self ):
      Tac.Notifiee.__init__( self, tunnelRibNameIdMap )
      # mountedTunnelRibs is dictionary of mounted TunnelRibs indexed by id 
      self.mountedTunnelRibs = {}
      self.tunnelRibBaseSmashPath = 'tunnel/tunnelRibs/status/'
      for tribName, tribId in tunnelRibNameIdMap.nameToId.iteritems():
         tunnelRibSmashPath = self.tunnelRibBaseSmashPath + str( tribId )
         if tribName == systemColoredTunnelRibName:
            tunnelRibType = 'Tunnel::TunnelTable::ColoredTunnelRib'
         else:
            tunnelRibType = 'Tunnel::TunnelTable::TunnelRib'
         tunnelRib = tunnelSmashEm.doMount( tunnelRibSmashPath,
                                            tunnelRibType,
                                            readerInfo )
         tunnelRib.id = tribId
         self.mountedTunnelRibs[ tribName ] = tunnelRib

   # Get Id of from TunnelRibNameIdMap using name as the key
   def getTunnelId( self, name ):
      return tunnelRibNameIdMap.nameToId.get( name )

   def getAllTunnelRibs( self ):
      ''' cache and return all tunnel ribs '''
      for tunnelName, tunnelId in tunnelRibNameIdMap.nameToId.iteritems():
         if tunnelName not in tunnelRibWatcher.mountedTunnelRibs:
            tunnelRibWatcher.cacheTunnelRib( tunnelName, tunnelId )
      return self.mountedTunnelRibs

   def getMountedTunnelRib( self, name ):
      tunnelId = getTunnelId( name )
      if tunnelId is None:
         return None
      if name not in tunnelRibWatcher.mountedTunnelRibs:
         tunnelRibWatcher.cacheTunnelRib( name, tunnelId )
      return self.mountedTunnelRibs[ name ]

   def cacheTunnelRib( self, name, tunnelId ):
      '''
         For a new key in tunnelRibNameIdMap, mount and add the respective tunnel
         rib to the mountedTunnelRibs class variable
      '''
      t0( "Creating/mounting tunnel rib" )
      if tunnelId is None:
         t0( "Tried to add Tunnel %s that is not in the TunnelRibNameIdMap" % name )
         return
      if name in self.mountedTunnelRibs:
         t0( "TunnelRib %s with Id %s is already mounted" % (name, tunnelId ) )
         return
      tunnelRibSmashPath = self.tunnelRibBaseSmashPath + str( tunnelId )
      if name == systemColoredTunnelRibName:
         tunnelRibType = 'Tunnel::TunnelTable::ColoredTunnelRib'
      else:
         tunnelRibType = 'Tunnel::TunnelTable::TunnelRib'
      tunnelRib = tunnelSmashEm.doMount( tunnelRibSmashPath,
                                         tunnelRibType,
                                         readerInfo )
      if not tunnelRib:
         t0( "Mounting %s unsuccessful" % name )
         return
      tunnelRib.id = tunnelId
      t0( "Successfully mounted ", name, " from ", tunnelRibSmashPath )
      self.mountedTunnelRibs[ name ] = tunnelRib

   def removeTunnelRibFromCache( self, name ):
      '''
        Deletes cached Tunnel Rib when we detect that an entry has been deleted
        from tunnelRibNameIdMap. We delete when we detect that a key is in
        our dictionary of cached Tunnel Ribs but not in the tunnelRibNameIdMap.
      '''
      # At this point we know tunnelId is None
      if not name in self.mountedTunnelRibs:
         t0( "Tunnel %s is not mounted" % name )
         return
      if name is tunnelRibNameIdMap.systemTunnelRibName:
         t0( "Can't delete system tunnel rib" )
         return
      if name is tunnelRibNameIdMap.systemColoredTunnelRibName:
         t0( "Can't delete system colored tunnel rib" )
         return
      tunnelId = self.mountedTunnelRibs[ name ].name
      del self.mountedTunnelRibs[ name ]
      tunnelRibSmashPath = self.tunnelRibBaseSmashPath + str ( tunnelId )
      t0( "Deleting %s from %s" % ( name, tunnelRibSmashPath ) )
      tunnelSmashEm.doUnmount( tunnelRibSmashPath )
      return

   @Tac.handler( 'nameToId' )
   def handletunRibNametunnelIdMap( self, trName ):
      t0( "Updating tunnel config for tunnel rib %s" % trName )
      tunnelId = self.getTunnelId( trName )
      if tunnelId is None and trName in self.mountedTunnelRibs:
         # Delete cached tunnelRibs for absent tunnel rib config
         self.removeTunnelRibFromCache( trName )
         return
      # Tunnel is being added to TunnelRibNameIdMap
      if trName in self.mountedTunnelRibs:
         t0( "Tunnel %s already mounted" % trName )
         return
      # trName is not in Mounted TunnelRibs so we add 
      self.cacheTunnelRib( trName, tunnelId )

def getTunnelRibWatcher():
   global tunnelRibWatcher
   if not tunnelRibWatcher:
      t0( "Instantiating TunnelRibWatcher..." )
      tunnelRibWatcher = TunnelRibWatcher()
   return tunnelRibWatcher

def showTunnelRib( mode, args ):
   tunnelRibs = [ 'TUNNEL_RIB_NAME', systemColoredTunnelRibName,
                  systemTunnelRibName, systemTunnelingLdpTunnelRibName,
                  systemIgpShortcutTunnelRibName ]
   for name in tunnelRibs:
      trName = args.get(name)
      if trName:
         break

   if trName is None:
      trName = systemTunnelRibName

   prefix = args.get( 'PREFIX' )
   _tunnelRibWatcher = getTunnelRibWatcher()
   color = args.get( 'COLOR' )
   mountedTunnelRib = _tunnelRibWatcher.getMountedTunnelRib( trName )
   return getTunnelRibModel( trName, mountedTunnelRib, prefix, color )

def getTunnelId( trName ):
   return tunnelRibNameIdMap.nameToId.get( trName )

def showTunnelRibAllSummary( mode, args ):
   _tunnelRibWatcher = getTunnelRibWatcher()
   tunnelRibs = _tunnelRibWatcher.getAllTunnelRibs()
   return getTunnelRibAllSummaryModel( tunnelRibs )

def showTunnelRibSummary( mode, args ):
   tunnelRibs = [ 'TUNNEL_RIB_NAME', systemTunnelRibName,
                  systemTunnelingLdpTunnelRibName, 
                  systemIgpShortcutTunnelRibName ]
   for name in tunnelRibs:
      trName = args.get(name)
      if trName:
         break

   if trName is None:
      trName = systemTunnelRibName

   _tunnelRibWatcher = getTunnelRibWatcher()
   mountedTunnelRib = _tunnelRibWatcher.getMountedTunnelRib( trName )
   return getTunnelRibSummaryModel( trName, mountedTunnelRib )

matcherSystemTunnelRib = CliMatcher.KeywordMatcher(
   'system-tunnel-rib', helpdesc='The system tunnel RIB' )

matcherSystemTunnelingLdpTunnelRib = CliMatcher.KeywordMatcher(
   'system-tunneling-ldp-tunnel-rib',
   helpdesc='The system tunneling LDP tunnel RIB' )

matcherSystemIgpShortcutTunnelRib = CliMatcher.KeywordMatcher(
   'system-igp-shortcut-tunnel-rib', helpdesc='The system IGP shortcut tunnel RIB' )

matcherSystemColoredTunnelRib = CliMatcher.KeywordMatcher(
   'system-colored-tunnel-rib', helpdesc='The system colored tunnel RIB' )
matcherColor = CliMatcher.KeywordMatcher( 'color', helpdesc='Match with color' )
matcherColorValue = CliMatcher.IntegerMatcher( 0, 2 ** 32 - 1,
                                               helpdesc='Match this color value' )
matcherRib = CliMatcher.KeywordMatcher(
      'rib', helpdesc='Tunnel routing information' )

class ShowColoredTunnelRibCmd( ShowCommand.ShowCliCommandClass ):
   syntax = ( 'show tunnel rib colored system-colored-tunnel-rib '
              '[ ( PREFIX [ color COLOR ] ) | ( color COLOR ) ] brief' )
   data = {
      'tunnel': tokenTunnelMatcher,
      'rib': matcherRib,
      'colored': tokenColored,
      'system-colored-tunnel-rib':  matcherSystemColoredTunnelRib,
      'PREFIX': IpGenAddrMatcher.IpGenAddrOrPrefixExprFactory(
                   ipOverlap=IpAddrMatcher.PREFIX_OVERLAP_AUTOZERO,
                   ip6Overlap=IpAddrMatcher.PREFIX_OVERLAP_AUTOZERO ),
      'color': matcherColor,
      'COLOR': matcherColorValue,
      'brief': 'Only show Tunnel IDs',
   }

   cliModel = TunnelRibModel.ColoredTunnelRib
   handler = showTunnelRib

class ShowTunnelRibCmd( ShowCommand.ShowCliCommandClass ):
   syntax = '''show tunnel rib
               [ system-tunneling-ldp-tunnel-rib
               | system-igp-shortcut-tunnel-rib
               | system-tunnel-rib
               | TUNNEL_RIB_NAME ]
               [ PREFIX ] brief'''
   data = {
      'tunnel': tokenTunnelMatcher,
      'rib': matcherRib,
      'system-tunneling-ldp-tunnel-rib': matcherSystemTunnelingLdpTunnelRib,
      'system-igp-shortcut-tunnel-rib': matcherSystemIgpShortcutTunnelRib,
      'system-tunnel-rib': matcherSystemTunnelRib,
      'TUNNEL_RIB_NAME': TunnelRibNameMatcher(
                   helpdesc='Name of the tunnel RIB' ),
      'PREFIX': IpGenAddrMatcher.IpGenAddrOrPrefixExprFactory(
                   ipOverlap=IpAddrMatcher.PREFIX_OVERLAP_AUTOZERO,
                   ip6Overlap=IpAddrMatcher.PREFIX_OVERLAP_AUTOZERO ),
      'brief': 'Only show Tunnel IDs',
   }
   cliModel = TunnelRibModel.TunnelRib
   handler = showTunnelRib

class ShowNamedTunnelRibSummaryCmd( ShowCommand.ShowCliCommandClass ):
   syntax = '''show tunnel rib
               ( system-tunneling-ldp-tunnel-rib
               | system-igp-shortcut-tunnel-rib
               | system-tunnel-rib
               | TUNNEL_RIB_NAME ) summary'''
   data = {
      'tunnel': tokenTunnelMatcher,
      'rib': matcherRib,
      'system-tunneling-ldp-tunnel-rib': matcherSystemTunnelingLdpTunnelRib,
      'system-igp-shortcut-tunnel-rib': matcherSystemIgpShortcutTunnelRib,
      'system-tunnel-rib': matcherSystemTunnelRib,
      'TUNNEL_RIB_NAME': TunnelRibNameMatcher(
                  helpdesc='Name of the tunnel RIB' ),
      'summary': 'Tunnel RIB summary'
   }
   cliModel = TunnelRibModel.TunnelRibSummary
   handler = showTunnelRibSummary 

class ShowTunnelRibsSummaryCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show tunnel rib summary'
   data = {
      'tunnel': tokenTunnelMatcher,
      'rib': matcherRib,
      'summary': 'All tunnel RIBs summary'
   }
   cliModel = TunnelRibModel.TunnelRibAllSummary
   handler = showTunnelRibAllSummary 

BasicCli.addShowCommandClass( ShowColoredTunnelRibCmd )
BasicCli.addShowCommandClass( ShowTunnelRibCmd )
BasicCli.addShowCommandClass( ShowNamedTunnelRibSummaryCmd )
BasicCli.addShowCommandClass( ShowTunnelRibsSummaryCmd )

#------------------------------------------------------------------------
# Auxillary function to extract vias from tunnel entries containing
# optimized fecIds
#------------------------------------------------------------------------
def getViaFromOptimizedFec( intfId ):
   vias = []
   fecId = FecId( FecIdIntfId.intfIdToFecId( intfId ) )
   if fecId.adjType() == 'usedByTunnelV4Adj':
      fecId = FecId( FecId.fecIdToNewAdjType( 'fibV4Adj', fecId ) )
   elif fecId.adjType() == 'usedByTunnelV6Adj':
      fecId = FecId( FecId.fecIdToNewAdjType( 'fibV6Adj', fecId ) )

   fec = None
   if fecId.adjType() == 'fibV4Adj':
      fec = forwardingStatus.fec.get( fecId )
   elif fecId.adjType() == 'fibV6Adj':
      fec = forwarding6Status.fec.get( fecId )
   if fec is None:
      return vias
   for fecVia in sorted( fec.via.values() ):
      addr = Tac.Value( "Arnet::IpGenAddr", str( fecVia.hop ) )
      interface = fecVia.intfId
      vias.append( ( addr, interface ) )
   return vias

#-------------------------------------------------------------------------
# The "show tunnel fib" command.
#-------------------------------------------------------------------------
def isTunnelTypeHidden( tunnelType ):
   return tunnelType == 'bgpLuPushTunnel'

def getTunnelFibEntryModel( tunnelId, encapFilter, cachedNHG, debug=False ):
   tunnelFibEntry = tunnelFib.entry.get( tunnelId )
   if tunnelFibEntry is None:
      return None

   def skipVia( tacTunVia ):
      return encapFilter is not None and encapFilter != tacTunVia.encapId.encapType

   viaList = list()
   for tunVia in tunnelFibEntry.tunnelVia.itervalues():
      if skipVia( tunVia ):
         continue
      if FecIdIntfId.isFecIdIntfId( tunVia.intfId ):
         igpVias = getViaFromOptimizedFec( tunVia.intfId )
         for ( nh, intf ) in igpVias:
            createdTunVia = Tac.newInstance( 'Tunnel::TunnelTable::TunnelVia',
                                             nh, intf, tunVia.encapId )
            viaModel = getTunnelFibViaModel( createdTunVia, cachedNHG )
            if viaModel:
               viaList.append( viaModel )
      else:
         resolvingTunnel = None
         if isDyTunIntfId( tunVia.intfId ):
            resolvingTunnel = getTunnelViaModelFromTunnelIntf( tunVia.intfId )

         viaModel = getTunnelFibViaModel( tunVia, cachedNHG,
                                          resolvingTunnel=resolvingTunnel )
         if viaModel:
            viaList.append( viaModel )
   if not viaList:
      return None
   
   # Get the backupTunnelVia
   backupViaList = list()
   for backupTunnelVia in tunnelFibEntry.backupTunnelVia.itervalues():
      if skipVia( backupTunnelVia ):
         continue
      viaModel = getTunnelFibViaModel( backupTunnelVia, cachedNHG, debug,
                                       backup=True )
      if viaModel:
         backupViaList.append( viaModel )

   tacTunnelId = Tac.Value( "Tunnel::TunnelTable::TunnelId", tunnelId )
   tunnelViaStatus = getTunnelViaStatusFromTunnelId( tunnelId )

   entryModel = TunnelFibModel.TunnelFibEntry(
         tunnelType=tacTunnelId.typeCliStr(), tunnelIndex=tacTunnelId.tunnelIndex(),
         endpoint=getEndpointFromTunnelId( tunnelId ), vias=viaList,
         backupVias=backupViaList,
         tunnelViaStatus=tunnelViaStatus )
   if debug:
      entryModel.tunnelId = tunnelId
      entryModel.tunnelFecId = FecId.tunnelIdToFecId( tacTunnelId )
      entryModel.interface = tacTunnelId.intfCliStr()
      entryModel.seqNo = tunnelFibEntry.seqNo

   return entryModel

def getTunnelFibModel( typeFilter=None, encapFilter=None, indexFilter=None,
                       debug=False ):
   t0( 'getTunnelFibModel: typeFilter', typeFilter, 'encapFilter', encapFilter,
       'indexFilter', indexFilter, 'debug', debug )

   def skipTunnel( tacTunnelId ):
      # Match on tunnel type. If no type filter, skip any hidden tunnel types.
      if typeFilter is not None:
         if tacTunnelId.tunnelType() != typeFilter:
            return True
      elif not debug and isTunnelTypeHidden( tacTunnelId.tunnelType() ):
         return True

      # Match on tunnel index (integer, possibly zero )
      if indexFilter is not None:
         if long( tacTunnelId.tunnelIndex() ) != long( indexFilter ):
            return True
      return False

   cachedNHG = {}
   for key, entry in nhgStatus.nexthopGroupEntry.iteritems():
      cachedNHG[ entry.nhgId ] = key.nhgName()
   fibModel = TunnelFibModel.TunnelFib()
   for tid in tunnelFib.entry:
      tacTunnelId = Tac.Value( 'Tunnel::TunnelTable::TunnelId', tid )
      if skipTunnel( tacTunnelId ):
         continue
      entryModel = getTunnelFibEntryModel( tid, encapFilter, cachedNHG, debug )
      if entryModel is None:
         continue
      key = tacTunnelId.typeCliStr()
      # pylint: disable-msg=E1101
      if not key in fibModel.categories.keys():
         fibModel.categories[ key ] = TunnelFibModel.TunnelFibCategory()
      subclass = fibModel.categories[ key ]
      subclass.entries[ long( tacTunnelId.tunnelIndex() ) ] = entryModel

   t0( 'getTunnelFibModel: finished' )
   return fibModel

parseTable = [
      ( frozenset( [ 'bgp', 'labeled-unicast' ] ), 'bgpLuTunnel', 'mplsEncap' ),
      ( frozenset( [ 'bgp', 'labeled-unicast', 'push' ] ),
         'bgpLuPushTunnel', 'mplsEncap' ),
      ( frozenset( [ 'static', 'mpls' ] ), 'staticTunnel', 'mplsEncap' ),
      ( frozenset( [ 'static', 'interface', 'gre' ] ),
         'staticInterfaceTunnel', 'greEncap' ),
      ( frozenset( [ 'static', 'interface', 'ipsec' ] ),
         'staticInterfaceTunnel', 'ipSecEncap' ),
      ( frozenset( [ 'static', 'interface', 'gre-over-ipsec' ] ),
         'staticInterfaceTunnel', 'ipSecGreEncap' ),
      ( frozenset( [ 'ldp' ] ), 'ldpTunnel', 'mplsEncap' ),
      ( frozenset( [ 'rsvp' ] ), 'rsvpFrrTunnel', 'mplsEncap' ),
      ( frozenset( [ 'rsvp-ler' ] ), 'rsvpLerTunnel', 'mplsEncap' ),
      ( frozenset( [ 'ti-lfa' ] ), 'tiLfaTunnel', 'mplsEncap' ),
      ( frozenset( [ 'isis', 'segment-routing' ] ), 'srTunnel', 'mplsEncap' ),
      ( frozenset( [ 'traffic-engineering', 'segment-routing', 'policy' ] ),
         'srTeSegmentListTunnel', 'mplsEncap' ),
      ( frozenset( [ 'nexthop-group' ] ), 'nexthopGroupTunnel', None ),
      ( frozenset(), None, None ), # Guard
      ]

class ShowTunnelFibCmd( ShowCommand.ShowCliCommandClass ):
   syntax = ( 'show tunnel fib [ debug ] ['
         '( bgp labeled-unicast [ push ] [ TUNNEL_INDEX ] ) |'
         '( ldp [ TUNNEL_INDEX ] ) |'
         '( static ( ( mpls [ TUNNEL_INDEX ] ) |'
                    '( interface gre [ TUNNEL_INDEX ] ) |'
                    '( interface ipsec [ TUNNEL_INDEX ] ) |'
                    '( interface gre-over-ipsec [ TUNNEL_INDEX ] ) ) ) |'
         '( isis segment-routing [ TUNNEL_INDEX ] ) |'
         '( traffic-engineering segment-routing policy [TUNNEL_INDEX] ) |'
         '( rsvp [ TUNNEL_INDEX ] ) |'
         '( rsvp-ler [ TUNNEL_INDEX ] ) |'
         '( ti-lfa [ TUNNEL_INDEX ] ) |'
         '( nexthop-group [ TUNNEL_INDEX ] )'
      ']' )
   data = {
         'bgp' : 'Border Gateway Protocol',
         'debug' : 'Show debugging information',
         'fib' : 'Forwarding Information Base',
         'gre' : 'Generic Routing Encapsulation',
         'ipsec' : 'IP Security',
         'gre-over-ipsec' : 'GRE over IPsec',
         'interface' : 'Tunnel interface',
         'isis' : 'Intermediate System to Intermediate System',
         'labeled-unicast' : 'Labeled Unicast',
         'ldp' : 'Label Distribution Protocol',
         'mpls' : 'MultiProtocol Label-Switching',
         'nexthop-group' : 'Nexthop Group',
         'policy' : 'Policy-based',
         'push' : CliCommand.Node(
            matcher=CliMatcher.KeywordMatcher( 'push',
               helpdesc='Pushed to the Tunnel FIB, bypassing the Tunnel RIB' ),
            hidden=True),
         'rsvp' : 'Resource Reservation Protocol Fast Reroute',
         'rsvp-ler' : 'Resource Reservation Protocol LER',
         'segment-routing' : 'Segment Routing',
         'static' : 'Static Configuration',
         'ti-lfa' : 'TI-LFA',
         'traffic-engineering' : 'Traffic Engineering',
         'tunnel' : tokenTunnelMatcher,
         'TUNNEL_INDEX' : tunnelIndexMatcher
         }
   cliModel = TunnelFibModel.TunnelFib

   @staticmethod
   def handler( mode, args ):
      t0( 'ShowTunnelFibCmd handler args:', args.values() )
      tunnelIndex = args.get( 'TUNNEL_INDEX' )
      debug = 'debug' in args
      for key in [ 'show', 'tunnel', 'fib', 'debug', 'TUNNEL_INDEX' ]:
         args.pop( key, None )
      argSet = set( args )
      for tokenSet, tunnelType, encapType in parseTable:
         if tokenSet == argSet:
            return getTunnelFibModel( typeFilter=tunnelType, encapFilter=encapType,
                                      indexFilter=tunnelIndex, debug=debug )
      raise Tac.InternalException( 'Parse table inconsistency' )

BasicCli.addShowCommandClass( ShowTunnelFibCmd )

#------------------------------------------------------------------------------------
# show tech-support tunnel graceful-restart
#------------------------------------------------------------------------------------
class ShowTechTunnelGr( ShowCommand.ShowCliCommandClass ):
   syntax = ( 'show tech-support tunnel graceful-restart' )
   data = {
      'tech-support' : TechSupportCli.techSupportKwMatcher,
      'tunnel' : "Show aggregated status details for Tunnel",
      'graceful-restart' : "Graceful restart information for Tunnel agent",
   }
   privileged = True

   @staticmethod
   def handler( mode, args ):
      AgentCommandRequest.runSocketCommand( em, TunnelAgent.name,
                                            'debug', 'DUMP_GR_STATE' )

BasicCli.addShowCommandClass( ShowTechTunnelGr )


#------------------------------------------------------------------------------------
# show <ldp | segment-routing> tunnel [ <tunnel-index> | <tep> ]
#------------------------------------------------------------------------------------
def addrMatchTep( endpoint=None , tunnel=None ):
   if ( not tunnel or not endpoint ):
      return False
   if endpoint == tunnel.tep.stringValue:
      return True
   return False

def getTunnelEntryAndIdFromIndexAndAf( tunnelIndex=None, tunnelTable=None,
                                  tunnelModel=None ):
   tunnelTableModel = tunnelModel( entries={} )
   # the tunnel with corresponding tunnelIndex can belong to
   # v4 tunnel or v6 tunnel but not both
   tunnelIdV4 = tunnelTableModel.getTunnelIdFromIndex( index=tunnelIndex,
                                                       addrFamily='ipv4' )
   tunnelV4 = tunnelTable.entry.get( tunnelIdV4 )
   tunnelIdV6 = tunnelTableModel.getTunnelIdFromIndex( index=tunnelIndex,
                                                       addrFamily='ipv6' )
   tunnelV6 = tunnelTable.entry.get( tunnelIdV6 )
   tunnel = tunnelV4 if tunnelV4 else tunnelV6
   tunnelId = tunnelIdV4 if tunnelV4 else tunnelIdV6

   return ( tunnel, tunnelId )

def getTunnelTableCommon( tunnelIndex=None, endpoint=None , 
                          tunnelTable=None, tunnelModel=None ):

   entries = {}
   if tunnelIndex is None:
      for tunnelId, tunnel in tunnelTable.entry.items():
         if( addrMatchTep( endpoint=endpoint, tunnel=tunnel ) or
             ( endpoint is None ) ):
            entryModel = getTunnelTableEntryWithMplsViasModel( tunnelId=tunnelId,
                                                tunnelTable=tunnelTable )
            if entryModel:
               entries[ getTunnelIndexFromId( tunnelId ) ] = entryModel
      tunnelTableModel = tunnelModel( entries=entries )
   else:
      tunnelTableModel = tunnelModel( entries=entries )
      tunnel, tunnelId = getTunnelEntryAndIdFromIndexAndAf( tunnelIndex, tunnelTable,
                                                       tunnelModel )
      if tunnel:
         if( addrMatchTep( endpoint=endpoint, tunnel=tunnel )
             or ( endpoint is None ) ):
            entryModel = getTunnelTableEntryWithMplsViasModel( tunnelId=tunnelId,
                                                               tunnelTable=\
                                                               tunnelTable )
            if entryModel:
               entries[ long( tunnelIndex ) ] = entryModel
               tunnelTableModel.entries = entries
   return tunnelTableModel

def getFecFromIntfId( intfId ):
   fecId = FecId( FecIdIntfId.intfIdToFecId( intfId ) )
   if fecId.adjType() == 'usedByTunnelV4Adj':
      fecId = FecId( FecId.fecIdToNewAdjType( 'fibV4Adj', fecId ) )
   elif fecId.adjType() == 'usedByTunnelV6Adj':
      fecId = FecId( FecId.fecIdToNewAdjType( 'fibV6Adj', fecId ) )

   fec = None
   if fecId.adjType() == 'fibV4Adj':
      fec = forwardingStatus.fec.get( fecId )
   elif fecId.adjType() == 'fibV6Adj':
      fec = forwarding6Status.fec.get( fecId )

   return fec

def getTunnelTableEntryWithMplsViasModel( tunnelId , tunnelTable=None ):
   vias = []
   tunnelTableEntry = tunnelTable.entry.get( tunnelId )
   if tunnelTableEntry:
      for via in tunnelTableEntry.via.itervalues():
         if FecIdIntfId.isFecIdIntfId( via.intfId ):
            fec = getFecFromIntfId( via.intfId )
            if fec is None:
               continue

            labels = []
            for mplsStackIndex in reversed ( range( via.labels.stackSize ) ):
               labels.append( str( via.labels.labelStack( mplsStackIndex ) ) )

            for fecVia in sorted( fec.via.values() ):
               addr = Tac.Value( "Arnet::IpGenAddr", str( fecVia.hop ) )
               intf = fecVia.intfId
               vias.append( TunnelModels.MplsVia(
                  nexthop=addr, interface=intf, type='ip',
                  labels=labels ) )
         else:
            viaModel = getMplsViaModel( via )
            if viaModel:
               vias.append( viaModel )
      # Make via rendering order deterministic
      vias.sort( cmp=TunnelModels.mplsViaCmp )
      return TunnelModels.TunnelTableEntryWithMplsVias(
         endpoint=tunnelTableEntry.tep,
         vias=vias )
   return None

#--------------------------------------------------------------------------
# register show tech-support for tunnel infrastructure
#--------------------------------------------------------------------------
def _showTechTunnelInfraCmds():
   if AgentDirectory.agentIsRunning( sysname, TunnelAgent.name ):
      cmds = [ 'show tunnel rib brief',
               'show tunnel rib colored system-colored-tunnel-rib brief',
               'show tunnel rib system-tunneling-ldp-tunnel-rib brief',
               'show tunnel rib system-igp-shortcut-tunnel-rib brief',
               'show tunnel fib debug' ]
   else:
      cmds = []
   return cmds

TechSupportCli.registerShowTechSupportCmdCallback(
      '2020-07-28 14:04:13',
      cmdCallback=_showTechTunnelInfraCmds )

def _showTechSummaryTunnel():
   return [ 'show tunnel rib summary' ] if AgentDirectory.agentIsRunning(
                                             sysname, TunnelAgent.name ) else []

TechSupportCli.registerShowTechSupportCmdCallback(
      '2020-06-12 11:31:01',
      summaryCmdCallback=_showTechSummaryTunnel )

#--------------------------------------------------------------------------
# register show tech-support extended evpn
#--------------------------------------------------------------------------
def _showTechEvpnCmds():
   return _showTechTunnelInfraCmds()

TechSupportCli.registerShowTechSupportCmdCallback(
      '2017-11-03 12:06:10',
      cmdCallback=_showTechEvpnCmds, extended='evpn' )

#------------------------------------------------------------------------
# Plugin
#------------------------------------------------------------------------
def Plugin( entityManager ):
   global sysname
   global nexthopGroupTunnelTable
   global systemTunnelRib
   global systemColoredTunnelRib
   global tunnelRibNameIdMap
   global tunnelFib
   global ldpTunnelTable
   global staticTunnelTable
   global srTunnelTable
   global rsvpFrrTunnelTable
   global rsvpLerTunnelTable
   global bgpLuTunnelTable
   global srTePolicyTunnelTable
   global forwardingStatus
   global forwarding6Status
   global programmingStatus
   global nhgStatus
   global tunnelSmashEm
   global tilfaTunnelTable
   global tunnelRibWatcher
   global em

   em = entityManager
   sysname = entityManager.sysname()
   systemTunnelRib = SmashLazyMount.mount(
      entityManager, 'tunnel/tunnelRibs/status/0', 'Tunnel::TunnelTable::TunnelRib',
      readerInfo )
   systemTunnelRib.id = systemTunnelRibId
   tunnelRibNameIdMap = LazyMount.mount( entityManager,
                                         'tunnel/tunnelRibs/tunnelRibNameIdMap',
                                         'Tunnel::TunnelTable::TunnelRibNameIdMap',
                                         'r' )
   systemColoredTunnelRib = SmashLazyMount.mount(
      entityManager, 'tunnel/tunnelRibs/status/1',
      'Tunnel::TunnelTable::ColoredTunnelRib',
      readerInfo )
   systemColoredTunnelRib.id = systemColoredTunnelRibId
   tunnelRibWatcher = None
   tunnelFib = SmashLazyMount.mount(
      entityManager, 'tunnel/tunnelFib', 'Tunnel::TunnelFib::TunnelFib',
      readerInfo )
   ldpTunnelTable = readMountTunnelTable(
      TunnelTableIdentifier.ldpTunnelTable, entityManager )
   staticTunnelTable = readMountTunnelTable(
      TunnelTableIdentifier.staticTunnelTable, entityManager )
   srTunnelTable = readMountTunnelTable(
      TunnelTableIdentifier.srTunnelTable, entityManager )
   tilfaTunnelTable = readMountTunnelTable(
      TunnelTableIdentifier.tiLfaTunnelTable, entityManager )
   nexthopGroupTunnelTable = readMountTunnelTable( 
      TunnelTableIdentifier.nexthopGroupTunnelTable, entityManager )
   bgpLuTunnelTable = readMountTunnelTable( 
      TunnelTableIdentifier.bgpLuTunnelTable, entityManager )
   srTePolicyTunnelTable = readMountTunnelTable(
      TunnelTableIdentifier.srTePolicyTunnelTable, entityManager )
   rsvpFrrTunnelTable = readMountTunnelTable(
      TunnelTableIdentifier.rsvpFrrTunnelTable, entityManager )
   rsvpLerTunnelTable = readMountTunnelTable( 
      TunnelTableIdentifier.rsvpLerTunnelTable, entityManager )
   forwardingStatus = SmashLazyMount.mount( entityManager,
                                            "forwarding/status",
                                            "Smash::Fib::ForwardingStatus",
                                            readerInfo )
   forwarding6Status = SmashLazyMount.mount( entityManager,
                                             "forwarding6/status",
                                             "Smash::Fib6::ForwardingStatus",
                                             readerInfo )
   programmingStatus = SmashLazyMount.mount( entityManager,
                                       "routing/hardware/tunnelProgramming/status",
                                       "Tunnel::Hardware::TunnelProgrammingStatus",
                                             readerInfo )
   nhgStatus = SmashLazyMount.mount( entityManager,
                                     "routing/nexthopgroup/entrystatus",
                                     "NexthopGroup::EntryStatus",
                                     readerInfo )
   tunnelSmashEm = SharedMem.entityManager( sysdbEm=entityManager )
