#!/usr/bin/env python
# Copyright (c) 2018 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import sys
import BasicCli
import Cell
import CliCommand
import CliMatcher
import CliParser
import LazyMount
import ShowCommand
import TacSigint
import Tac
import Tracing
from TrafficPolicyCliModel import ( TrafficPolicyModel, TrafficPolicySummaryModel,
                                    ProtocolNeighborsModel, ProtocolNeighbor,
                                    VrfRules, NeighborRule,
                                    TrafficPolicyErrorsModel,
                                    TrafficPolicyCountersModel )
from TrafficPolicyCliLib import protectedTrafficPolicyNamesRegex
from PolicyMapModelImpl import waitForPmapCounters

t0 = Tracing.trace0

config = None
vrfConfig = None
allVrfConfig = None
statusDir = None
l3IntfStatus = None
cpuCounter = None
intfCounter = None
aegisIntfConfig = None
intfParamStatus = None

def interfaceGuard( mode, token ):
   if statusDir and statusDir.get( 'sandAegis' ):
      return None
   return CliParser.guardNotThisPlatform

def cpuGuard( mode, token ):
   if statusDir and statusDir.get( 'sand' ):
      return None
   return CliParser.guardNotThisPlatform

def showTrafficPolicy( mode, policyName, cpu, interface,
                       summary, errors, detail, counters ):
   t0( "showTrafficPolicy", policyName, cpu, interface, summary, errors )
   if cpu:
      status = statusDir.get( 'sand' )
      if status is not None:
         LazyMount.force( config )
         LazyMount.force( vrfConfig )
         LazyMount.force( l3IntfStatus )
         mostRecentCounter = countersSnapshot( mode, "cpu" )
         byteCountersSupported = False
         helper = Tac.newInstance( 'TrafficPolicyCli::ShowTrafficPolicyCpu',
                                   config, status, mostRecentCounter, policyName,
                                   summary, errors, detail, counters,
                                   byteCountersSupported, vrfConfig,
                                   allVrfConfig, l3IntfStatus )
      else:
         t0( "Feature not present to report TrafficPolicy cpu status" )
   elif interface:
      status = statusDir.get( 'sandAegis' )
      if status is not None:
         LazyMount.force( aegisIntfConfig )
         LazyMount.force( intfParamStatus )
         mostRecentCounter = countersSnapshot( mode, "interface" )
         byteCountersSupported = True
         helper = Tac.newInstance( 'TrafficPolicyCli::ShowTrafficPolicyInterface',
                                   config, status, mostRecentCounter, policyName,
                                   summary, errors, detail, counters,
                                   byteCountersSupported, aegisIntfConfig,
                                   intfParamStatus )
      else:
         t0( "Feature not present to report TrafficPolicy interface status" )
   else:
      assert False, "invalid command"

   if status:
      fd = sys.stdout.fileno()
      fmt = mode.session_.outputFormat()
      revision = mode.session_.requestedModelRevision()

      if not summary:
         handleCounters( config, status )

      helper.render( fd, fmt, revision )

   if summary:
      return TrafficPolicySummaryModel
   elif errors:
      return TrafficPolicyErrorsModel
   elif counters:
      return TrafficPolicyCountersModel
   else:
      return TrafficPolicyModel

def handleCounters( policyConfig, status ):
   for policy, pStatus in status.status.iteritems():
      waitForPmapCounters( policy, policyConfig.pmapType,
                           [ pStatus ] )

# Returns the latest counter snapshot by looking at global snapshot and
# session snapshot timestamps.
def countersSnapshot( mode, counterType ):
   # get global snapshot
   counter = None
   if counterType == "cpu":
      counter = cpuCounter
      sessionSnapshot = mode.session.sessionData( 'cpuTrafficPolicySessionCounter',
                                                  None )
   elif counterType == "interface":
      counter = intfCounter
      sessionSnapshot = mode.session.sessionData( 'intfTrafficPolicySessionCounter',
                                                  None )
   assert counter is not None
   LazyMount.force( counter )
   globalSnapshot = counter

   # compare session and global snapshots, return newest
   return globalSnapshot if sessionSnapshot is None or \
      globalSnapshot.timestamp > sessionSnapshot.timestamp else sessionSnapshot

# NOTE: DynamicNameMatcher's low priority comes handy when we have
# both policyNameMatcher and summaryMatcher take a hit on keyword
# summary. For eg: show traffic-policy summary. At this point, higher
# priority summary keyword wins and we display summary for all the
# policies. In the event we do have a policy-name called _SUMMARY_ (
# aka summary ), then the user has to type out 'show traffic-policy
# summary summary' to view its summary in isolation or 'show
# traffic-policy summary detail'. The base command for 'show
# traffic-policy summary' for the policy called _SUMMARY_ will not
# take effect.
trafficPolicyMatcher = CliMatcher.KeywordMatcher(
   'traffic-policy', helpdesc='Show traffic-policy rules' )
summaryMatcher = CliMatcher.KeywordMatcher(
   'summary', helpdesc='Traffic policy summary' )
detailMatcher = CliMatcher.KeywordMatcher(
   'detail', helpdesc='Traffic policy detail' )
errorsMatcher = CliMatcher.KeywordMatcher(
   'errors', helpdesc='Traffic policy errors' )
countersMatcher = CliMatcher.KeywordMatcher(
   'counters', helpdesc='Traffic policy counters' )

policyNameMatcher = CliMatcher.DynamicNameMatcher(
   lambda mode: sorted( config.pmapType.pmap ),
   "Traffic policy name", pattern=protectedTrafficPolicyNamesRegex(),
   priority=CliParser.PRIO_LOW )
interfaceMatcher = CliCommand.guardedKeyword( 'interface',
                                              helpdesc="Policy applied on interface",
                                              guard=interfaceGuard )
cpuMatcher = CliCommand.guardedKeyword( 'cpu',
                                        helpdesc="Policy applied on VRF(s)",
                                        guard=cpuGuard )

def showTrafficPolicyHandler( mode, args ):
   """Digest the args dict and pass it down to the helper to render"""
   policyName = args.get( 'POLICY_NAME', '' )
   cpu = 'cpu' in args
   interface = 'interface' in args
   summary = 'summary' in args
   detail = 'detail' in args
   errors = 'errors' in args
   counters = 'counters' in args
   t0( "showTrafficPolicyHandler args", args )
   return showTrafficPolicy( mode, policyName, cpu, interface,
                             summary, errors, detail, counters )
# show traffic-policy [ <name> ] ( cpu | interface )

# NOTE: There are 4 different classes when we could have had just
# one ShowTrafficPolicy. This is because the CLI model ( see cliModel attribute )
# returned for each of these variants is different.
# ------------------------------------------------------------------------
# ------------------------------------------------------------------------
class ShowTrafficPolicy( ShowCommand.ShowCliCommandClass ):
   syntax = 'show traffic-policy [ POLICY_NAME ] ( cpu | interface ) [ detail ]'
   data = {
      'traffic-policy': trafficPolicyMatcher,
      'POLICY_NAME' : policyNameMatcher,
      'cpu' : cpuMatcher,
      'interface' : interfaceMatcher,
      'detail' : detailMatcher,
   }
   cliModel = TrafficPolicyModel
   handler = showTrafficPolicyHandler

BasicCli.addShowCommandClass( ShowTrafficPolicy )
#------------------------------------------------------------------------
# show traffic-policy [ <name> ] ( cpu | interface ) summary
#------------------------------------------------------------------------
class ShowTrafficPolicySummary( ShowCommand.ShowCliCommandClass ):
   syntax = ( 'show traffic-policy [ POLICY_NAME ] '
              '( cpu | interface ) summary' )
   data = {
      'traffic-policy': trafficPolicyMatcher,
      'POLICY_NAME' : policyNameMatcher,
      'summary': summaryMatcher,
      'cpu' : cpuMatcher,
      'interface' : interfaceMatcher,
   }
   cliModel = TrafficPolicySummaryModel
   handler = showTrafficPolicyHandler

BasicCli.addShowCommandClass( ShowTrafficPolicySummary )
#------------------------------------------------------------------------
# show traffic-policy [ <name> ] ( cpu | interface ) errors
#------------------------------------------------------------------------
class ShowTrafficPolicyErrors( ShowCommand.ShowCliCommandClass ):
   syntax = ( 'show traffic-policy [ POLICY_NAME ] '
              '( cpu | interface ) errors' )
   data = {
      'traffic-policy': trafficPolicyMatcher,
      'POLICY_NAME' : policyNameMatcher,
      'errors': errorsMatcher,
      'cpu' : cpuMatcher,
      'interface' : interfaceMatcher,
   }
   cliModel = TrafficPolicyErrorsModel
   handler = showTrafficPolicyHandler

BasicCli.addShowCommandClass( ShowTrafficPolicyErrors )

#------------------------------------------------------------------------
# show traffic-policy protocol neighbors bgp
#------------------------------------------------------------------------
class ProtocolNeighborBgp( ShowCommand.ShowCliCommandClass ):
   syntax = '''show traffic-policy protocol neighbors bgp'''
   data = {
      'traffic-policy': 'Show traffic-policy rules',
      'protocol': 'protocol',
      'neighbors': 'neighbors',
      'bgp': 'BGP',
   }
   cliModel = ProtocolNeighborsModel

   @staticmethod
   def handler( mode, args ):
      def getRules( structuredFilterColl ):
         rules = []
         for sf in structuredFilterColl.itervalues():
            assert len( sf.proto ) <= 1

            if not sf.proto:
               continue
            protoField = sf.proto.values()[ 0 ]
            assert len( protoField.port ) <= 1
            if protoField.port:
               portField = protoField.port.values().pop()
               assert len( portField.sport ) <= 1
               assert len( portField.dport ) <= 1
            for source in sf.source:
               rule = NeighborRule()
               rule.sourcePrefix = source
               if protoField.port:
                  if portField.sport.keys():
                     rule.sourcePort = portField.sport.keys()[ 0 ].rangeStart
                  if portField.dport.keys():
                     rule.destinationPort = portField.dport.keys()[ 0 ].rangeStart
               rules.append( rule )
         return rules

      def getVrfRules( trafficPolicyClassMapVrfStatus ):
         vrfRules = {}
         for vrfName, vrfStatus in trafficPolicyClassMapVrfStatus.iteritems():
            rules = getRules( vrfStatus.structuredFilter )
            vrfRules[ vrfName ] = VrfRules( rules=rules )
         return vrfRules

      status = statusDir.get( 'sand' )

      if status is None:
         t0( "Feature not present to report TrafficPolicy cpu status" )
         return ProtocolNeighborsModel()
      assert status is not None

      policies = {}
      for policyName, policyStatus in status.status.iteritems():
         vrfRules = {}
         for classStatus in status.trafficPolicyClassMapStatus.itervalues():
            className = classStatus.className
            if className not in policyStatus.cmap:
               continue
            classVrfRules = \
                  getVrfRules( classStatus.trafficPolicyClassMapVrfStatus )
            for vrf, rules in classVrfRules.iteritems():
               if vrf in vrfRules:
                  vrfRules[ vrf ].rules.extend( rules.rules )
               else:
                  vrfRules[ vrf ] = rules

         assert policyName not in policies
         policies[ policyName ] = ProtocolNeighbor( vrfs=vrfRules )
      return ProtocolNeighborsModel( policies=policies )

BasicCli.addShowCommandClass( ProtocolNeighborBgp )

# ------------------------------------------------------------------------
# show traffic-policy [ <name> ] ( cpu | interface ) counters
# ------------------------------------------------------------------------
class ShowTrafficPolicyCounters( ShowCommand.ShowCliCommandClass ):
   syntax = 'show traffic-policy [ POLICY_NAME ] ( cpu | interface ) counters'
   data = {
      'traffic-policy' : trafficPolicyMatcher,
      'POLICY_NAME' : policyNameMatcher,
      'cpu' : cpuMatcher,
      'interface' : interfaceMatcher,
      'counters' : countersMatcher,
   }
   cliModel = TrafficPolicyCountersModel
   handler = showTrafficPolicyHandler

BasicCli.addShowCommandClass( ShowTrafficPolicyCounters )

def Plugin( entityManager ):
   global config
   global vrfConfig
   global statusDir
   global l3IntfStatus
   global allVrfConfig
   global cpuCounter
   global intfCounter
   global aegisIntfConfig
   global intfParamStatus
   # Cell specific mount path which holds status entities per linecard
   # in a Tac::Dir. At the moment, we can simply take one instance
   # under the Tac::Dir which belongs to Sand platform. On other
   # platforms ( like Strata ), we need a unified view of the status
   # reporting model or per slice view ( TBD ).
   statusDirPath = 'cell/%d/trafficPolicies/status' % Cell.cellId()
   mountGroup = entityManager.mountGroup()
   statusDir = mountGroup.mount( statusDirPath, 'Tac::Dir', 'ri' )
   mountGroup.close( callback=None )

   # Mount the traffic policy config as writable as the show command will
   # write a timestamp to the config when requesting counter values.
   config = LazyMount.mount( entityManager, "trafficPolicies/input/cli",
                             "TrafficPolicy::TrafficPolicyConfig", "w" )
   vrfConfig = LazyMount.mount( entityManager, "trafficPolicies/cpu/vrf",
                                "PolicyMap::VrfConfig", "r" )
   allVrfConfig = LazyMount.mount( entityManager, 'ip/vrf/config',
                                   'Ip::AllVrfConfig', 'r' )
   l3IntfStatus = LazyMount.mount( entityManager, "l3/intf/status",
                                   "L3::Intf::StatusDir", "r" )
   aegisIntfConfig = LazyMount.mount( entityManager,
                                      "trafficPolicies/intf/input/aegis",
                                      "PolicyMap::IntfConfig", "r" )
   intfParamStatus = LazyMount.mount( entityManager,
                                      "trafficPolicies/param/status/interface",
                                      "TrafficPolicy::TrafficPolicyIntfParamStatus",
                                      "r" )
   cpuCounter = LazyMount.mount( entityManager, "trafficPolicies/counter/cpu",
                                 "TrafficPolicy::Counter", "r" )
   intfCounter = LazyMount.mount( entityManager, "trafficPolicies/counter/interface",
                                  "TrafficPolicy::Counter", "r" )
