#!/usr/bin/env python
# Copyright (c) 2018 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

"""Definition of the CAPI model. All the rendering logic is deferred
and takes place in TrafficPolicyShowCliHelper(tac/tin)
"""

import TableOutput
from ArnetModel import IpGenericPrefix
from CliModel import Bool, Model, DeferredModel, Dict, Int, List, Str, Enum, Submodel
from CliCommon import CliModelNotDegradable
from IntfModels import Interface
import Toggles.InbandTelemetryCommonToggleLib
import Toggles.PostcardTelemetryCommonToggleLib

class TcpFlag( DeferredModel ):
   name = Str( help="TCP flag name" )
   value = Bool( help="TCP flag bit value" )

class NumericalRange( DeferredModel ):
   low = Int( help="Minimum value" )
   high = Int( help="Maximum value" )

class IcmpType( DeferredModel ):
   icmpTypes = Submodel( valueType=NumericalRange, help="ICMP type", optional=True )
   icmpCodes = List( valueType=NumericalRange, help="ICMP code", optional=True )

class Port( DeferredModel ):
   srcPorts = List( valueType=NumericalRange, help="Source port", optional=True )
   destPorts = List( valueType=NumericalRange, help="Destination port",
                     optional=True )
   srcL4PortFieldSets = List( valueType=str, help="Source l4-port field-sets",
                              optional=True )
   destL4PortFieldSets = List( valueType=str, help="Destination l4-port field-sets",
                               optional=True )

class Protocol( DeferredModel ):
   __revision__ = 2
   protocolRange = Submodel( valueType=NumericalRange, help="Protocol",
                             optional=True )
   ports = List( valueType=Port, help="List of L4 ports that can be ORed together",
                 optional=True )
   icmps = List( valueType=IcmpType, help="ICMP info", optional=True )
   tcpFlags = List( valueType=TcpFlag, help="TCP flags", optional=True )

class Fragment( DeferredModel ):
   matchAllFragments = Bool( help="Match all fragments", optional=True )
   fragmentOffsets = List( valueType=NumericalRange, help="Fragment offset",
                           optional=True )

class Matches( DeferredModel ):
   destPrefixes = List( valueType=IpGenericPrefix,
                        help="Destination prefix", optional=True )
   srcPrefixes = List( valueType=IpGenericPrefix,
                       help="Source prefix", optional=True )
   srcPrefixSets = List( valueType=str, help="Source prefix sets", optional=True )
   destPrefixSets = List( valueType=str, help="Destination prefix sets",
                          optional=True )
   protocols = List( valueType=Protocol, help="Protocol",
                     optional=True )
   ipLengths = List( valueType=NumericalRange, help="IP length",
                     optional=True )
   fragment = Submodel( valueType=Fragment, help="Fragment", optional=True )
   dscps = List( valueType=NumericalRange, help="DSCP", optional=True )
   ttls = List( valueType=NumericalRange, help="TTL", optional=True )
   neighborProtocol = Str( help="Neighbor protocol", optional=True )
   matchL4Protocol = Str( help="Protocol BGP", optional=True )
   matchIpOptions = Bool( help="Match on any IP options", optional=True )

class PoliceRate( DeferredModel ):
   rate = Int( help="Lower police rate" )
   unit = Str( help="Unit of police rate" )

class CountData( DeferredModel ):
   packetCount = Int( help="Number of packets matched", optional=True )
   byteCount = Int( help="Number of bytes matched", optional=True )
   undefined = Bool( help="Named counter has not been defined", optional=True )

class Actions( DeferredModel ):
   drop = Bool( help="Drop action", optional=True )
   police = Submodel( valueType=PoliceRate, help="Police action", optional=True )
   count = Int( help="Number of packets hitting the rule", optional=True )
   countNamed = Dict( keyType=str, valueType=CountData,
                      help="A mapping of counter name to its count data",
                      optional=True )
   setDscp = Int( help="Set header DSCP value", optional=True )
   setTc = Int( help="Set forwarding traffic class", optional=True )
   log = Bool( help="Log action", optional=True )
   if Toggles.InbandTelemetryCommonToggleLib.\
      toggleFeatureInbandTelemetrySamplePolicyEnabled() or \
      Toggles.PostcardTelemetryCommonToggleLib.\
      toggleFeaturePostcardTelemetryEnabled():
      sample = Bool( help="Sample action", optional=True )
      sampleAll = Bool( help="Sample all action", optional=True )

class RuleSummary( DeferredModel ):
   ruleString = Str( help="Match rule name" )
   matchOption = Str( help="Match on packet type" )

class Rule( RuleSummary ):
   matches = Submodel( valueType=Matches, help="Matches for the rule",
                       optional=True )
   actions = Submodel( valueType=Actions, help="Actions for the rule",
                       optional=True )


class TrafficPolicyErrors( DeferredModel ):
   __revision__ = 2
   ipv4FailedVrfs = List( valueType=str, help="Failed VRFs for IPv4 traffic",
                          optional=True )
   ipv6FailedVrfs = List( valueType=str, help="Failed VRFs for IPv6 traffic",
                          optional=True )
   ipv4FailedIntfs = List( valueType=Interface,
                           help="Failed interfaces for IPv4 traffic", optional=True )
   ipv6FailedIntfs = List( valueType=Interface,
                           help="Failed interfaces for IPv6 traffic", optional=True )
   ipv4InstallationErrors = List( valueType=str,
                                  help="List of ipv4 installation errors",
                                  optional=True )
   ipv6InstallationErrors = List( valueType=str,
                                  help="List of ipv6 installation errors",
                                  optional=True )

   def degrade( self, dictRepr, revision ):
      if revision < 2:
         # The next revision of the model has no information for the rules as we
         # display errors encountered on the configured interface or vrf. So we can't
         # show any useful data in the dictRepr for rules.
         raise CliModelNotDegradable( "No resonable values for rules available" )

class InterfaceInfo( DeferredModel ):
   packetCount = Int( help="Number of packets matched", optional=True )
   byteCount = Int( help="Number of bytes matched", optional=True )

class TotalInfo( DeferredModel ):
   packetTotal = Int( help="Total packet count", optional=True )
   byteTotal = Int( help="Total byte count", optional=True )

class CounterInfo( DeferredModel ):
   interfaces = Dict( valueType=InterfaceInfo, help="Count data per interface",
                      optional=True )
   totals = Submodel( valueType=TotalInfo, help="Total hit counts", optional=True )

class TrafficPolicyCounters( DeferredModel ):
   namedCounters = Dict( keyType=str, valueType=CounterInfo,
                         help="Mapping of named counter to count data",
                         optional=True )
   unnamedCounters = Dict( keyType=str, valueType=CounterInfo,
                           help="Mapping of rule name to count data", optional=True )

class TrafficPolicyCountersModel( DeferredModel ):
   ingressCounterGranularity = Enum(
      values=( "per-policy", "per-interface" ),
      help="Enabled counter granularity for ingress policies", optional=True )
   trafficPolicies = Dict( valueType=TrafficPolicyCounters,
                           help="Traffic policy indexed by policy name",
                           optional=True )

# Common state shared between summary and detailed view of policy.
class TrafficPolicyCommon( DeferredModel ):
   configuredVrfs = List( valueType=str, help="List of configured VRFs",
                          optional=True )
   ipv4AppliedVrfs = List( valueType=str, help="Applied VRFs for IPv4 traffic",
                           optional=True )
   ipv6AppliedVrfs = List( valueType=str, help="Applied VRFs for IPv6 traffic",
                           optional=True )
   configuredIntfs = List( valueType=Interface,
                           help="List of configured interfaces",
                           optional=True )
   ipv4AppliedIntfs = List( valueType=Interface,
                            help="Applied interfaces for IPv4 traffic",
                            optional=True )
   ipv6AppliedIntfs = List( valueType=Interface,
                            help="Applied interfaces for IPv6 traffic",
                            optional=True )

# Rules are ordered based on priority enforced in CLI configuration
class TrafficPolicySummary( TrafficPolicyCommon ):
   rules = List( valueType=RuleSummary, help="Summary of match rules configured" )

class TrafficPolicy( TrafficPolicyCommon ):
   namedCounters = List( valueType=str,
                         help="Defined named counters for a traffic-policy",
                         optional=True )
   rules = List( valueType=Rule, help="Detailed information of match rules" )

class TrafficPolicySummaryModel( DeferredModel ):
   trafficPolicies = Dict( valueType=TrafficPolicySummary,
                           help="Traffic policy indexed by policy name",
                           optional=True )

class TrafficPolicyModel( DeferredModel ):
   __revision__ = 2
   trafficPolicies = Dict( valueType=TrafficPolicy,
                           help="Traffic policy indexed by policy name",
                           optional=True )

class TrafficPolicyErrorsModel( DeferredModel ):
   __revision__ = 2
   trafficPolicies = Dict( valueType=TrafficPolicyErrors,
                           help="Traffic policy indexed by policy name",
                           optional=True )

# Models for 'show traffic-policy protocol neighbors bgp' commands
# NOTE: Model is used here (not DeferredModel) as this command is implemented as a
# standard Python CAPI command (not CliPrint). This was a deliberate choice to take
# advantage of TableOutput to implement dynamic column widths. CliPrint is best for
# high scale commands. As the output from this command is O( number of BGP peers ),
# it should be limited to the low tens of thousands at the most. Therefore, a Python
# implementation should be fast enough.
class NeighborRule( Model ):
   sourcePrefix = IpGenericPrefix( help="Source prefix" )
   sourcePort = Int( help="Source port", optional=True )
   destinationPort = Int( help="Destination port", optional=True )

class VrfRules( Model ):
   rules = List( valueType=NeighborRule, help="Neighbor rules for IPV4/6" )

class ProtocolNeighbor( Model ):
   vrfs = Dict( valueType=VrfRules, help="Neighbors per VRF" )

class ProtocolNeighborsModel( Model ):
   # Keyed on policy name
   policies = Dict( valueType=ProtocolNeighbor, help="Neighbors per policy" )

   def createTable( self ):
      headers = [ "VRF", "Source IP", "Source Port", "Destination Port" ]
      headingEntries = [ header for header in headers ]
      # NOTE: 81 is the maximum length of a row in the table assuming the longest
      # possible IPV6 address.
      table = TableOutput.createTable( headingEntries, indent=0 )
      # Center justify and wrap all columns
      columnFormat = TableOutput.Format( justify='center', wrap=True )
      table.formatColumns( *( [ columnFormat ] * len( headers ) ) )
      return table

   def render( self ):
      for policyName in sorted( self.policies.keys() ):
         print 'Traffic Policy %s' % policyName

         table = self.createTable()

         protocolNeighbor = self.policies.get( policyName )
         for vrfName in sorted( protocolNeighbor.vrfs ):
            vrfRules = protocolNeighbor.vrfs[ vrfName ]
            for rule in sorted( vrfRules.rules, key=lambda key: key.sourcePrefix ):
               sourcePort = 'any'
               if rule.sourcePort:
                  sourcePort = rule.sourcePort
               destinationPort = 'any'
               if rule.destinationPort:
                  destinationPort = rule.destinationPort
               table.newRow( vrfName, rule.sourcePrefix, sourcePort,
                             destinationPort )
         print table.output()
