#!/usr/bin/env python
# Copyright (c) 2018 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

from AclLib import tcpServiceByName, udpServiceByName
from AclCliLib import genericIpProtocols, genericIp6Protocols
import BasicCli
import Cell
import CliCommand
import CliMatcher
import ConfigMount
import LazyMount
import Tac
import Tracing
import CliToken.Clear
from ClassificationLib import ( extraIpv4Protocols, extraIpv6Protocols,
                                getKeywordMap )
import CliParser
import ShowCommand
from CliMode.Classification import ( FieldSetL4PortConfigMode )
from CliMode.TrafficPolicy import ( ActionsConfigMode, MatchRuleIpv4ConfigMode,
                                    MatchRuleIpv6ConfigMode,
                                    MatchRuleDefaultConfigMode,
                                    TrafficPoliciesConfigMode,
                                    TrafficPolicyConfigMode, FEATURE,
                                    FieldSetIpPrefixConfigMode,
                                    FieldSetIpv6PrefixConfigMode )
from TrafficPolicyLib import structuredFilterToCmds
from TrafficPolicyCliLib import ( TrafficPolicyContext,
                                  neighborsConfigConflictMsg,
                                  matchL4ProtocolConflictMsg )
from TrafficPolicyCliLib import ( MatchRuleContext,
                                  TrafficPolicyMatchRuleAction,
                                  PolicyConfigCmdBase, ActionType,
                                  protectedTrafficPolicyNamesRegex,
                                  ProtocolTcpFlagsOnlyCmd )
from ClassificationCliLib import ( generateMultiRangeMatcher,
                                   generateFieldSetExpression,
                                   portKwMatcher,
                                   tcpUdpProtoExpr,
                                   PrefixIpv4Cmd,
                                   PrefixIpv6Cmd,
                                   ProtocolIcmpV4ConfigCmd,
                                   ProtocolIcmpV6ConfigCmd,
                                   ProtocolIcmpV4TypeCodeConfigCmd,
                                   ProtocolIcmpV6TypeCodeConfigCmd,
                                   IpLengthConfigCmd,
                                   MatchAllFragmentConfigCmd,
                                   FragmentOffsetConfigCmd,
                                   IpOptionsConfigCmd,
                                   NumericalRangeConfigCmdBase,
                                   CommitAbortModelet,
                                   PrefixFieldSetCmdBase,
                                   FieldSetIpPrefixConfigCmds,
                                   FieldSetIpv6PrefixConfigCmds,
                                   FieldSetIpPrefixExceptConfigCmds,
                                   FieldSetIpv6PrefixExceptConfigCmds,
                                   FieldSetIpPrefixBaseConfigCmd,
                                   FieldSetL4PortBaseConfigCmd,
                                   ProtocolMixin,
                                   ProtocolFieldSetBaseCmd,
                                   generateTcpFlagExpression,
                                   protectedFieldSetNamesRegex )

t0 = Tracing.trace0

policiesCliConfig = None
policiesIntfParamConfig = None
policiesStatusRequestDir = None
policiesStatus = None
qosHwStatus = None
qosAclHwStatus = None
entityManager = None
cpuCounter = None
intfCounter = None
fieldSetConfig = None
tacMatchOption = Tac.Type( 'PolicyMap::ClassMapMatchOption' )
matchIpAccessGroup = tacMatchOption.matchIpAccessGroup
matchIpv6AccessGroup = tacMatchOption.matchIpv6AccessGroup
tacNeighborProtocol = Tac.Type( 'Classification::NeighborProtocol' )
protocolNone = tacNeighborProtocol.protocolNone
protocolBgp = tacNeighborProtocol.protocolBgp
tacMatchL4Protocol = Tac.Type( 'Classification::MatchL4Protocol' )
matchL4None = tacMatchL4Protocol.matchL4None
matchL4Bgp = tacMatchL4Protocol.matchL4Bgp
tacFieldConflict = Tac.Type( 'Classification::FieldConflict' )
conflictNeighborProtocol = tacFieldConflict.conflictNeighborProtocol
conflictMatchL4Protocol = tacFieldConflict.conflictMatchL4Protocol
ReservedClassMapNames = Tac.Type( 'TrafficPolicy::ReservedClassMapNames' )
ActionType = Tac.Type( "PolicyMap::ActionType" )
RateUnit = Tac.Type( "PolicyMap::RateUnit" )

def getMatchRules( mode ):
   currCfg = mode.trafficPolicy
   if currCfg:
      return currCfg.rawClassMap.keys()
   return []

def getTrafficPolicyCommands( trafficPolicyContext, matchRuleContext=None ):
   """
   Takes in a trafficPolicyContext and optional matchRuleContext and returns
   a list of command dictionaries. Each command dictionary contains all commands used
   to create a match rule in the traffic-policy.
   If a matchRuleContext is given, we  use the matchRuleContext to generate
   the command dictionary rather then using the corresponding match rule in the
   trafficPolicyContext. We place the command dictionary in the list of commands
   according to its seqnum.
   """
   assert trafficPolicyContext is not None
   reservedNames = trafficPolicyContext.reservedClassMapNames()
   filterCmds = []
   trafficPolicy = trafficPolicyContext.currentPolicy()
   if not trafficPolicy:
      return filterCmds

   def matchRuleToCmds( prio, ruleName, sfilter, actions, matchType ):
      cmds = structuredFilterToCmds( sfilter, actions, matchType )
      cmds[ "ruleHeader" ] = "match %s %s" % ( ruleName, matchType )
      cmds[ "ruleName" ] = ruleName
      cmds[ "prio" ] = prio
      return cmds

   def addMatchRule( ruleName ):
      matchRule = trafficPolicy.rawClassMap[ ruleName ]
      classAction = trafficPolicy.classAction.get( ruleName )
      match = matchRule.match.values()[ 0 ]
      matchOption = matchRule.match.keys()[ 0 ]
      matchType = "ipv4" if matchOption == matchIpAccessGroup else "ipv6"
      cmds = matchRuleToCmds( prio, ruleName, match.structuredFilter,
                              classAction.policyAction, matchType )
      filterCmds.append( cmds )

   def addMatchRuleContextRule( matchRuleContext ):
      # Add given rule to its appriopriate position in the command list
      # based on its seqnum, removing the rule if it is already present
      overrideCmd = matchRuleToCmds( matchRuleContext.seqnum,
                                     matchRuleContext.ruleName,
                                     matchRuleContext.filter,
                                     matchRuleContext.matchRuleAction.actions(),
                                     matchRuleContext.matchType )
      # Remove previous entry, if any (happens when already commited rule is edited)
      # And add current entry being edited
      for i, cmd in enumerate( filterCmds ):
         if cmd[ "ruleName" ] == matchRuleContext.ruleName:
            filterCmds.pop( i )
            break

      index = len( filterCmds )
      if matchRuleContext.seqnum:
         for i, cmd in enumerate( filterCmds ):
            if cmd[ "prio" ] > matchRuleContext.seqnum:
               index = i
               break
      filterCmds.insert( index, overrideCmd )

   for prio, ruleName in trafficPolicy.classPrio.iteritems():
      if ruleName in reservedNames:
         # These rules should be displayed at the end
         continue
      addMatchRule( ruleName )

   if matchRuleContext and matchRuleContext.ruleName not in reservedNames:
      # do not add reserved rules here, add them once all unreserved rules have
      # been handled
      addMatchRuleContextRule( matchRuleContext )

   for prio, ruleName in trafficPolicy.classPrio.iteritems():
      if ruleName not in reservedNames:
         continue
      elif matchRuleContext and matchRuleContext.ruleName == ruleName:
         addMatchRuleContextRule( matchRuleContext )
      else:
         addMatchRule( ruleName )

   return filterCmds

def printCmd( cmd, spacing=3 ):
   if cmd:
      print "%s%s" % ( spacing * " ", cmd )

def printMatchRule( cmds ):
   printCmd( cmds.get( "ruleHeader" ) )
   printCmd( cmds.get( "source" ), spacing=6 )
   printCmd( cmds.get( "srcPrefixSet" ), spacing=6 )
   printCmd( cmds.get( "destination" ), spacing=6 )
   printCmd( cmds.get( "dstPrefixSet" ), spacing=6 )
   printCmd( cmds.get( "protoNeighborsBgp" ), spacing=6 )
   printCmd( cmds.get( "protoBgp" ), spacing=6 )
   protoFields = cmds.get( "protocol", [] )
   for p in protoFields:
      printCmd( p, spacing=6 )
   printCmd( cmds.get( "ttl" ), spacing=6 )
   printCmd( cmds.get( "dscp" ), spacing=6 )
   printCmd( cmds.get( "sport" ), spacing=6 )
   printCmd( cmds.get( "dport" ), spacing=6 )
   printCmd( cmds.get( "length" ), spacing=6 )
   printCmd( cmds.get( "fragment" ), spacing=6 )
   printCmd( cmds.get( "matchIpOptions" ), spacing=6 )
   actions = cmds.get( "actions" )
   if actions:
      printCmd( "!", spacing=6 )
      printCmd( "actions", spacing=6 )
      for action in actions:
         printCmd( action, spacing=9 )

class ShowPendingCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show [ pending ]'
   data = { 'pending': 'show pending traffic-policy' }

   @staticmethod
   def handler( mode, args ):
      trafficPolicyContext = getattr( mode, "trafficPolicyContext", None )
      matchRuleContext = getattr( mode, "context", None )
      cmdOutput = getTrafficPolicyCommands( trafficPolicyContext, matchRuleContext )

      # Print traffic-policy name along with all pending/committed match rules
      printCmd( "%s %s" % ( mode.feature,
                            trafficPolicyContext.pmapName_ ),
                spacing=0 )
      # Print named counters before rules
      if trafficPolicyContext.currentPolicy().namedCounter:
         counterCmd = "counter %s" % (
            " ".join( sorted( trafficPolicyContext.currentPolicy().namedCounter ) ) )
         printCmd( counterCmd, spacing=3 )
      for matchRule in cmdOutput[ : -1 ]:
         printMatchRule( matchRule )
         printCmd( "!", spacing=3 )
      if cmdOutput:
         printMatchRule( cmdOutput[ -1 ] )

matcherCounters = CliMatcher.KeywordMatcher( 'counters',
      helpdesc='Clear traffic-policy counters in all sessions' )
matcherTrafficPolicy = CliMatcher.KeywordMatcher( 'traffic-policy',
      helpdesc='Clear traffic-policy' )

def getTrafficPolicyNames():
   if not policiesCliConfig:
      return []
   return policiesCliConfig.pmapType.pmap.iterkeys()

trafficPolicyNameMatcher = CliMatcher.DynamicNameMatcher(
      lambda mode: getTrafficPolicyNames(), "Traffic Policy Name",
      pattern=protectedTrafficPolicyNamesRegex() )

pMapConstants = Tac.Value( 'Classification::Constants' )
ttlRangeMatcher = generateMultiRangeMatcher( 'ttl', pMapConstants.maxTtl )
dscpRangeMatcher = generateMultiRangeMatcher( 'dscp', pMapConstants.maxDscp )

def nameAndNumberExpression( name, rangeMatcher, nameMatcher,
                             nameToValueMaps ):
   """nameToValueMaps is a list of maps that have the format of
   name : ( value, description ).

   This function generates an expression that allows:
   1. A numerical range
   2. A list of names

   And sets the set of numeric values in the specified argument.
   """
   rangeName = name + "_RANGE"
   nameName = name + "_NAME"

   class NameAndNumberExpression( CliCommand.CliExpression ):
      expression = "%s | { %s }" % ( rangeName, nameName )
      data = {
         rangeName : rangeMatcher,
         nameName : nameMatcher
         }

      @staticmethod
      def adapter( mode, args, argsList ):
         names = args.pop( nameName, None )
         if names:
            valueSet = set()
            for n in names:
               for nmap in nameToValueMaps:
                  entry = nmap.get( n, None )
                  if entry:
                     valueSet.add( entry[ 0 ] )
         else:
            valueSet = args.pop( rangeName, None )
         if valueSet is not None:
            args[ name ] = valueSet
   return NameAndNumberExpression

# port number/range
@Tac.memoize
def _portNames():
   """Accepts one udp or tcp port keyword"""
   return getKeywordMap( tcpServiceByName, udpServiceByName )

portRangeMatcher = generateMultiRangeMatcher( 'port', pMapConstants.maxL4Port )
portNameMatcher = CliMatcher.DynamicKeywordMatcher( lambda mode: _portNames() )

def portExpression( name ):
   return nameAndNumberExpression( name, portRangeMatcher, portNameMatcher,
                                   ( tcpServiceByName, udpServiceByName ) )

# protocol number/range
protoRangeMatcherV4 = generateMultiRangeMatcher( 'protocol ',
                                                 minVal=pMapConstants.minProtoV4,
                                                 maxVal=pMapConstants.maxProto )

protoRangeMatcherV6 = generateMultiRangeMatcher( 'protocol ',
                                                 minVal=pMapConstants.minProtoV6,
                                                 maxVal=pMapConstants.maxProto )

def generateIpProtoMatcher( name, genericProtocols, extraProtocols,
                            protoRangeMatcher ):
   nameMap = getKeywordMap( genericProtocols, extraProtocols )
   protoNameMatcher = CliMatcher.DynamicKeywordMatcher( lambda mode:
                                                        nameMap )
   return nameAndNumberExpression( name, protoRangeMatcher, protoNameMatcher,
                                   ( genericProtocols, extraProtocols ) )

ipv4ProtoMatcher = generateIpProtoMatcher( 'PROTOCOL', genericIpProtocols,
                                           extraIpv4Protocols, protoRangeMatcherV4 )
ipv6ProtoMatcher = generateIpProtoMatcher( 'PROTOCOL', genericIp6Protocols,
                                           extraIpv6Protocols, protoRangeMatcherV6 )

#--------------------------------------
# The "traffic-policies" mode command
#--------------------------------------

class TrafficPoliciesConfigCmd( CliCommand.CliCommandClass ):
   syntax = 'traffic-policies'
   noOrDefaultSyntax = syntax
   data = {
      'traffic-policies' : 'Configure traffic policies'
   }

   _noHandlers = []

   @staticmethod
   def _registerNoHandler( handler ):
      TrafficPoliciesConfigCmd._noHandlers.append( handler )

   @staticmethod
   def handler( mode, args ):
      childMode = mode.childMode( TrafficPoliciesConfigMode )
      mode.session_.gotoChildMode( childMode )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      # pylint: disable=protected-access
      for handler in TrafficPoliciesConfigCmd._noHandlers:
         handler( mode, args )
      for name in policiesCliConfig.pmapType.pmap:
         TrafficPolicyConfigCmd._removePolicy( mode, name )
      for name in fieldSetConfig.fieldSetIpPrefix:
         FieldSetIpPrefixConfigCmd._removeFieldSet( mode, name, "ipv4" )
      for name in fieldSetConfig.fieldSetIpv6Prefix:
         FieldSetIpv6PrefixConfigCmd._removeFieldSet( mode, name, "ipv6" )
      for name in fieldSetConfig.fieldSetL4Port:
         FieldSetL4PortConfigCmd._removeFieldSet( mode, name )
#---------------------------------------------------------------
# The "[no|default] traffic-policy <traffic-policy-name>" mode command
#---------------------------------------------------------------
class TrafficPolicyConfigCmd( PolicyConfigCmdBase ):
   syntax = 'traffic-policy POLICY_NAME'
   noOrDefaultSyntax = 'traffic-policy POLICY_NAME'
   data = {
      'traffic-policy' : 'Configure traffic policy',
      'POLICY_NAME' : trafficPolicyNameMatcher
   }

   @staticmethod
   def _feature():
      return FEATURE

   @staticmethod
   def _context( name ):
      return TrafficPolicyContext( policiesCliConfig, policiesStatusRequestDir,
                                   policiesStatus, name )

#--------------------------------------------------------------------------------
# The "match RULE_NAME <ipv4 | ipv6>" mode command
#--------------------------------------------------------------------------------
matchRuleName = CliMatcher.DynamicNameMatcher( getMatchRules, "Match Rule Name" )

class MatchRuleConfigCmdBase( CliCommand.CliCommandClass ):
   @staticmethod
   def _feature():
      raise NotImplementedError

   @staticmethod
   def _context( trafficPolicyContext, ruleName, matchOption ):
      raise NotImplementedError

   @staticmethod
   def _matchRulePresent( currentPolicy, ruleName, matchOption=None ):
      """ Checks if match rule is present and the match type has not changed """
      if matchOption:
         return ruleName in currentPolicy.classAction and \
                currentPolicy.rawClassMap[ ruleName ].match.get( matchOption )
      else:
         return ruleName in currentPolicy.rawClassMap.keys()

   @staticmethod
   def _matchRuleNameReserved( ruleName ):
      return ruleName == ReservedClassMapNames.classV4Default or \
             ruleName == ReservedClassMapNames.classV6Default

   @classmethod
   def _createMatchRuleContext( cls, mode, ruleName, matchOption, seqno=None,
                                beforeRule=None, afterRule=None ):
      """
      This routine creates (or fetches) the appropriate match rule context
      (MatchRuleContext) for the given CLI command. If we are creating a new rule, a
      new match rule context is needed. In other cases, we are editing an existing
      rule, in which case we also create a new context but copy in the existing rule.
      In yet other cases we are _moving_ an existing rule to another sequence number.
      This is identical to editing an existing rule. Finally, we may be attempting to
      move a rule that doesn't exist for the provided match option (e.g. ipv6 when we
      meant ipv4). In this case, we return the previous match rule context, and issue
      a CLI warning.

      @return context - This routine returns a match rule context, or None if one
      could not be created.
      """
      currentPolicy = mode.trafficPolicyContext.currentPolicy()
      matchRulePresent = cls._matchRulePresent( currentPolicy, ruleName,
                                                  matchOption )
      otherMatchOption = ( matchIpv6AccessGroup if matchOption is
                           matchIpAccessGroup
                           else matchIpAccessGroup )
      otherMatchRulePresent = cls._matchRulePresent( currentPolicy, ruleName,
                                                     otherMatchOption )

      # Here we check if we are attempting to edit or move an existing rule but
      # with the opposite match option. In this case, we issue a warning and
      # return to the previous mode.
      if otherMatchRulePresent:
         otherMatchType = ( 'ipv4' if otherMatchOption is matchIpAccessGroup else
                            'ipv6' )
         # We can never get in a scenario where both rules are present, even
         # transiently
         assert not matchRulePresent
         if seqno and ( beforeRule or afterRule ):
            if beforeRule:
               cmd = 'match %s %s before %s' % ( ruleName, otherMatchType,
                                                 beforeRule )
            else:
               assert afterRule
               cmd = 'match %s %s after %s' % ( ruleName, otherMatchType,
                                                afterRule )
            warning = "Trying to move match rule with incorrect match type. " \
                      "Did you mean '%s'?" % ( cmd )
         else:
            cmd = 'match %s %s' % ( ruleName, otherMatchType )
            warning = "Trying to edit match rule with incorrect match type. " \
                      "Did you mean '%s'?" % ( cmd )

         mode.addWarning( warning )
         return None

      matchRuleContext = cls._context( mode.trafficPolicyContext, ruleName,
                                       matchOption )

      if matchRulePresent:
         matchRuleContext.copyEditMatchRule( ruleName, seqno )
      else:
         matchRuleContext.newEditMatchRule( ruleName, seqno )

      if seqno and ( beforeRule or afterRule ):
         # We are placing a rule via before / after
         mode.trafficPolicyContext.shouldResequence = True
         if matchRulePresent:
            mode.trafficPolicyContext.moving = True

      return matchRuleContext

   @classmethod
   def handler( cls, mode, args ):
      ruleName = args[ 'RULE_NAME' ]
      beforeRule = afterRule = None
      if 'before' in args:
         beforeRule = args[ 'REFERENCE_RULE' ]
      elif 'after' in args:
         afterRule = args[ 'REFERENCE_RULE' ]
      if 'ipv4' in args:
         matchOption = matchIpAccessGroup
      else:
         matchOption = matchIpv6AccessGroup

      seqno = None
      locationRule = beforeRule or afterRule

      # The default rules are always the last two rules, with the IPV4 rule first,
      # followed by the IPV6 rule. Rules can be moved _before_ the IPV4 rule but not
      # before the IPV6 or after either rule.
      if cls._matchRuleNameReserved( locationRule ):
         errorMsg = 'User-defined rules cannot be moved such that they follow ' \
                    'a built-in rule.'
         if afterRule or beforeRule \
            and locationRule == ReservedClassMapNames.classV6Default:
            mode.addError( errorMsg )
            return

      if cls._matchRuleNameReserved( ruleName ) and locationRule:
         errorMsg = 'The built-in rules cannot be moved.'
         mode.addError( errorMsg )
         return

      # Ignore 'before / after' if the rule is the same as the current rule
      if locationRule != ruleName:
         currentPolicy = mode.trafficPolicyContext.currentPolicy()
         if cls._matchRulePresent( currentPolicy, locationRule ):
            for lseqno, rule in currentPolicy.classPrio.iteritems():
               if rule == locationRule:
                  seqno = lseqno
                  break
         elif cls._matchRulePresent( currentPolicy, ruleName ):
            for lseqno, rule in currentPolicy.classPrio.iteritems():
               if rule == ruleName:
                  seqno = lseqno
                  break
      # If we have before / after, simply insert at the seqno of the before /
      # after rule - / + 1, then resequence. This is slow in general, but simple,
      # and could be optimized easily.
      if seqno:
         if beforeRule:
            seqno -= 1
         elif afterRule:
            seqno += 1

      context = cls._createMatchRuleContext( mode, ruleName, matchOption,
                                             seqno=seqno, beforeRule=beforeRule,
                                             afterRule=afterRule )

      if not context and not mode.trafficPolicyContext.matchRuleContext:
         # We are trying to edit a match rule with the wrong match option,
         # so simply return
         return
      elif not context:
         context = mode.trafficPolicyContext.matchRuleContext

      assert context
      mode.trafficPolicyContext.matchRuleContext = context

      childMode = mode.childMode( context.childMode( ruleName, matchOption ),
                                  trafficPolicyContext=mode.trafficPolicyContext,
                                  matchRuleContext=context,
                                  feature=cls._feature() )

      mode.session_.gotoChildMode( childMode )

   @classmethod
   def noOrDefaultHandler( cls, mode, args ):
      ruleName = args[ 'RULE_NAME' ]

      # First, check if the ruleName is eligible
      if cls._matchRuleNameReserved( ruleName ):
         errorMsg = "Only the actions of the built-in rules can be changed. " \
                    "Rule %s cannot be removed." % ruleName
         mode.addError( errorMsg )
         return

      matchRule = mode.trafficPolicy.rawClassMap.get( ruleName )
      if 'ipv4' in args:
         matchOption = matchIpAccessGroup
      elif 'ipv6' in args:
         matchOption = matchIpv6AccessGroup
      else:
         matchOption = None

      # Only remove the rule if the match option matches the named rule
      if ( matchRule and
           ( matchOption is None or matchRule.match.get( matchOption, None ) ) ):
         matchRuleAction = TrafficPolicyMatchRuleAction( mode.trafficPolicyContext,
                                                         ruleName,
                                                         matchOption,
                                                         None )
         mode.trafficPolicyContext.delRuleCommon( 0, matchRuleAction )

class MatchRuleConfigCmd( MatchRuleConfigCmdBase ):
   syntax = ( 'match RULE_NAME ipv4 | ipv6 '
              '[ before | after REFERENCE_RULE ]' )
   noOrDefaultSyntax = 'match RULE_NAME [ ipv4 | ipv6 ]'
   data = {
      'match' : 'Configure match rule',
      'RULE_NAME' : matchRuleName,
      'ipv4' : 'IPv4 match rule',
      'ipv6' : 'IPv6 match rule',
      'before' : 'Add this rule immediately before another rule',
      'after' : 'Add this rule immediately after another rule',
      'REFERENCE_RULE' : matchRuleName,
   }

   @staticmethod
   def _feature():
      return FEATURE

   @staticmethod
   def _context( trafficPolicyContext, ruleName, matchOption ):
      return MatchRuleContext( trafficPolicyContext, ruleName, matchOption )

#------------------------------------------------------------------------------------
# Base Action Command Class
#------------------------------------------------------------------------------------
class ActionCmdBase( CliCommand.CliCommandClass ):
   _actionType = None

   @classmethod
   def checkErrors( cls, mode, args ):
      errorStr = mode.context.matchRuleAction.actionCombinationError()
      if errorStr:
         mode.addError( errorStr )
      return errorStr is not None

   @classmethod
   def handler( cls, mode, args ):
      mode.context.setAction( cls._actionType, no=False )
      hasError = cls.checkErrors( mode, args )
      if hasError:
         # revert change
         mode.context.setAction( cls._actionType, no=True )

   @classmethod
   def noOrDefaultHandler( cls, mode, args ):
      mode.context.setAction( cls._actionType, no=True )
      hasError = cls.checkErrors( mode, args )
      if hasError:
         # revert change
         mode.context.setAction( cls._actionType, no=False )

#------------------------------------------------------------------------------------
# The "drop" action
#------------------------------------------------------------------------------------
class DropActionCmd( ActionCmdBase ):
   _actionType = ActionType.deny

   syntax = 'drop'
   noOrDefaultSyntax = syntax
   data = {
      'drop' : 'Drop action',
   }

#------------------------------------------------------------------------------------
# The "counter { NAMES }" command for policy counter definitions
#------------------------------------------------------------------------------------
def getNamedCounters( mode ):
   currCfg = mode.trafficPolicy
   if currCfg:
      return currCfg.namedCounter.keys()
   return []

availNamedCounters = CliMatcher.DynamicNameMatcher( getNamedCounters,
                                                    "Counter name" )

class NamedCounterCmd( CliCommand.CliCommandClass ):
   syntax = '''counter { NAMES }'''
   noOrDefaultSyntax = '''counter [ { NAMES } ]'''
   data = {
      'counter' : 'counter',
      'NAMES' : availNamedCounters
   }

   @classmethod
   def handler( cls, mode, args ):
      context = mode.getContext()
      counterNames = args[ 'NAMES' ]
      context.updateNamedCounters( counterNames, add=True )

   @classmethod
   def noOrDefaultHandler( cls, mode, args ):
      context = mode.getContext()
      counterNames = args.get( 'NAMES', [] )
      context.updateNamedCounters( counterNames, add=False )

#------------------------------------------------------------------------------------
# The "count [ NAME ]" action
#------------------------------------------------------------------------------------
class CountActionCmd( ActionCmdBase ):
   _actionType = ActionType.count

   syntax = '''count [ NAME ]'''
   noOrDefaultSyntax = syntax
   data = {
      'count' : 'Configure count action',
      'NAME' : availNamedCounters
   }

   @classmethod
   def handler( cls, mode, args ):
      # set the counter name if present
      namedCounter = args.get( "NAME", "" )
      mode.context.setAction( cls._actionType, namedCounter, no=False )

   @classmethod
   def noOrDefaultHandler( cls, mode, args ):
      namedCounter = args.get( "NAME", "" )
      mode.context.setAction( cls._actionType, namedCounter, no=True )


#------------------------------------------------------------------------------------
# The "log" action
#------------------------------------------------------------------------------------
class LogActionCmd( ActionCmdBase ):
   _actionType = ActionType.log

   syntax = '''log'''
   noOrDefaultSyntax = '''log'''
   data = {
      'log' : 'Configure log action',
   }

#------------------------------------------------------------------------------------
# The "set dscp DSCP_VALUE" action
#------------------------------------------------------------------------------------
class SetDscpActionCmd( ActionCmdBase ):
   _actionType = ActionType.setDscp

   syntax = '''set dscp DSCP_VALUE'''
   noOrDefaultSyntax = '''set dscp ...'''
   data = {
      'set' : 'Set packet header or property',
      'dscp' : 'Set header DSCP',
      'DSCP_VALUE' : CliMatcher.IntegerMatcher( 0,
                                                63,
                                                helpdesc='specify dscp value' ),
   }

   @classmethod
   def handler( cls, mode, args ):
      dscpValue = args.get( 'DSCP_VALUE' )
      mode.context.setAction( cls._actionType, dscpValue, no=False )

#------------------------------------------------------------------------------------
# The "set traffic class TRAFFIC_CLASS" action
#------------------------------------------------------------------------------------
def tcRangeFunc( mode ):
   return ( 0, qosHwStatus.numTcSupported - 1 )

class SetTcActionCmd( ActionCmdBase ):
   _actionType = ActionType.setTc

   syntax = '''set traffic class TRAFFIC_CLASS'''
   noOrDefaultSyntax = '''set traffic class ...'''
   data = {
      'set' : 'Set packet header or property',
      'traffic' : 'Set traffic management property',
      'class' : 'Set forwarding traffic class',
      'TRAFFIC_CLASS' : CliMatcher.DynamicIntegerMatcher(
         tcRangeFunc, helpdesc='specify traffic class' ),
   }

   @classmethod
   def handler( cls, mode, args ):
      tc = args.get( 'TRAFFIC_CLASS' )
      mode.context.setAction( cls._actionType, tc, no=False )

#------------------------------------------------------------------------------------
# The "police rate <rate-value> [<rate-unit>]" action
#------------------------------------------------------------------------------------
rateUnitTokens = {
   RateUnit.bps : 'Rate in bps',
   RateUnit.kbps : 'Rate in kbps (default unit)',
   RateUnit.mbps : 'Rate in mbps',
   RateUnit.gbps : 'Rate in gbps',
}

class PoliceActionCmd( ActionCmdBase ):
   _actionType = ActionType.police

   syntax = '''police rate <rate-value> [<rate-unit>]'''
   noOrDefaultSyntax = '''police ...'''
   data = {
      'police' : 'Configure policer parameters',
      'rate' : 'Set lower rate',
      '<rate-value>' : CliMatcher.IntegerMatcher( 1, # FIXME
                                                  1000000000, # FIXME
                                                  helpdesc='specify lower rate' ),
      '<rate-unit>' : CliMatcher.EnumMatcher( rateUnitTokens ),
   }

   @staticmethod
   def _policerOutOfRangeCheck( mode, policeRate ):
      outOfRange = False
      if mode.session_.startupConfig():
         # Do not check this during startup
         return outOfRange
      if policeRate.rateInKbps() < qosAclHwStatus.policeByteModeMinRateKbps or \
         policeRate.rateInKbps() > qosAclHwStatus.policeByteModeMaxRateKbps:
         mode.addError( 'Invalid lower rate value' )
         mode.addError( ' Lower rate range <%d-%d> kbps' % (
            qosAclHwStatus.policeByteModeMinRateKbps,
            qosAclHwStatus.policeByteModeMaxRateKbps ) )
         outOfRange = True
      return outOfRange

   @classmethod
   def handler( cls, mode, args ):
      rateValue = args.get( '<rate-value>' )
      rateUnit = args.get( '<rate-unit>', 'kbps' )
      rate = Tac.newInstance( "PolicyMap::PoliceRate", rateValue, rateUnit )
      if PoliceActionCmd._policerOutOfRangeCheck( mode, rate ):
         # Do nothing for out-of-range police rate configuration
         return
      mode.context.setAction( cls._actionType, rate, no=False )

#------------------------------------------------------------------------------------
# The "actions" sub-prompt mode command
#------------------------------------------------------------------------------------
class ActionsConfigCmdBase( CliCommand.CliCommandClass ):
   syntax = 'actions'
   data = {
      'actions' : 'Configure actions',
   }

   @staticmethod
   def _feature():
      raise NotImplementedError

   @classmethod
   def handler( cls, mode, args ):
      childMode = mode.childMode( mode.context.actionMode(),
                                  trafficPolicyContext=mode.trafficPolicyContext,
                                  matchRuleContext=mode.context,
                                  feature=cls._feature() )
      mode.session_.gotoChildMode( childMode )

class ActionsConfigCmd( ActionsConfigCmdBase ):
   @staticmethod
   def _feature():
      return FEATURE

#--------------------------------------------------
# The "ttl <ttl-list>" command for traffic-policy
#--------------------------------------------------

class TtlConfigCmd( NumericalRangeConfigCmdBase ):
   _attrName = 'ttl'
   _rangeType = 'Classification::TtlRange'
   _argListName = 'TTL'

   syntax = 'ttl TTL'
   noOrDefaultSyntax = 'ttl [ TTL ]'
   data = {
      'ttl' : 'Configure ttl match criteria',
      'TTL' : ttlRangeMatcher
   }

#--------------------------------------------------
# The "dscp <dscp-list>" command for traffic-policy
#--------------------------------------------------

class DscpConfigCmd( NumericalRangeConfigCmdBase ):
   _attrName = 'dscp'
   _rangeType = 'Classification::DscpRange'
   _argListName = 'DSCP'

   syntax = 'dscp DSCP'
   noOrDefaultSyntax = 'dscp [ DSCP ]'
   data = {
      'dscp' : 'dscp',
      'DSCP' : dscpRangeMatcher
   }

#------------------------------------------------------------------------------------
# The "protocol neighbors bgp" command for traffic-policy
#------------------------------------------------------------------------------------
class ProtoNeighborsConfigCmd( CliCommand.CliCommandClass ):
   syntax = 'protocol neighbors bgp'
   noOrDefaultSyntax = syntax
   data = {
      'protocol' : 'Protocol',
      'neighbors' : 'Neighbors',
      'bgp' : 'BGP'
   }

   @staticmethod
   def handler( mode, args ):
      mode.context.updateFilterAttr( 'neighborProtocol', attrValue=protocolBgp )
      # If structuredFilter is no longer valid, revert the change
      if not mode.context.filter.isValidConfig( conflictNeighborProtocol ):
         mode.context.updateFilterAttr( 'neighborProtocol', attrValue=protocolNone )
         mode.addError( neighborsConfigConflictMsg )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      mode.context.updateFilterAttr( 'neighborProtocol', attrValue=protocolNone )

#------------------------------------------------------------------------------------
# The "protocol bgp" command for traffic-policy
#------------------------------------------------------------------------------------

class ProtoMatchConfigCmd( CliCommand.CliCommandClass ):
   syntax = 'protocol bgp'
   noOrDefaultSyntax = syntax
   data = {
      'protocol' : 'Protocol',
      'bgp' : 'BGP'
   }

   @staticmethod
   def handler( mode, args ):
      mode.context.updateFilterAttr( 'matchL4Protocol', attrValue=matchL4Bgp )
      # If structuredFilter is no longer valid, revert the change
      if not mode.context.filter.isValidConfig( conflictMatchL4Protocol ):
         mode.context.updateFilterAttr( 'matchL4Protocol', attrValue=matchL4None )
         mode.addError( matchL4ProtocolConflictMsg )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      mode.context.updateFilterAttr( 'matchL4Protocol', attrValue=matchL4None )

# The protocol command for traffic-policy:
#  "protocol <proto-list> | <tcp-udp> [ (source|destination) port <port-list> ]"
#  where that list is similar to acl "permit":
#  ahp      Authentication Header Protocol
#  gre      Generic Routing Encapsulation
#  icmp     Internet Control Message Protocol
#  igmp     Internet Group Management Protocol (IGMP)
#  ospf     OSPF routing protocol
#  pim      Protocol Independent Multicast (PIM)
#  rsvp     Resource Reservation Protocol (RSVP)
#  tcp      Transmission Control Protocol
#  udp      User Datagram Protocol
#  vlan     Vlan
#  vrrp     Virtual Router Redundancy Protocol
#  <1-255>  IP protocol number for ipv4 ( 0 illegal for v4 )
#  <0-255>  IP protocol number for ipv6
#
#------------------------------------------------------------------------------------
def getL4PortFieldSetNames( mode ):
   if fieldSetConfig is None:
      return []
   return fieldSetConfig.fieldSetL4Port

l4PortFieldSetNameMatcher = CliMatcher.DynamicNameMatcher(
   getL4PortFieldSetNames,
   "Layer-4 port field-set name",
   pattern=protectedFieldSetNamesRegex( 'port' ),
   priority=CliParser.PRIO_LOW )
#------------------------------------------------------------------------------------
# The "protocol (tcp [ flags [ not ] TCP_FLAGS ] | udp) source port field-set
# FIELD_SET destination port field-set FIELD_SET" command for traffic-policy
#------------------------------------------------------------------------------------

l4PortSrcFieldSetExpr = generateFieldSetExpression( l4PortFieldSetNameMatcher,
                                                   'SRC_FIELD_SET_NAME',
                                                    allowMultiple=True )
l4PortDstFieldSetExpr = generateFieldSetExpression( l4PortFieldSetNameMatcher,
                                                   'DST_FIELD_SET_NAME',
                                                    allowMultiple=True )
class ProtocolFieldSetCmd( ProtocolFieldSetBaseCmd ):
   _sportFieldSetAttr = 'sportFieldSet'
   _dportFieldSetAttr = 'dportFieldSet'
   data = {
      'FLAGS_EXPR' : generateTcpFlagExpression( tcpFlagsSupported=True ),
      'SRC_FIELD_SET_NAME' : l4PortSrcFieldSetExpr,
      'DST_FIELD_SET_NAME' : l4PortDstFieldSetExpr
   }
   data.update( ProtocolFieldSetBaseCmd._baseData )

#------------------------------------------------------------------------------------
# The "protocol PROTOCOL | (tcp | udp) source port SPORT
#      destination port DPORT" command for traffic-policy
#------------------------------------------------------------------------------------
class ProtocolBase( ProtocolMixin ):
   syntax = ( 'protocol PROTOCOL | ( ( TCP_UDP | FLAGS_EXPR ) '
              '( source port1 SPORT ) |'
              '( destination port2 DPORT ) | '
              '( source port1 SPORT destination port2 DPORT ) )' )
   noOrDefaultSyntax = ( 'protocol [ PROTOCOL | ( ( TCP_UDP | FLAGS_EXPR )'
                         '( source port1 [ SPORT ] ) | '
                         '( destination port2 [ DPORT ] ) | '
                         '( source port1 SPORT destination port2 DPORT ) ) ]' )

   _baseData = {
      'protocol' : 'Protocol',
      'TCP_UDP' : tcpUdpProtoExpr,
      'port1' : portKwMatcher,
      'port2' : portKwMatcher,
      'source' : 'Source',
      'SPORT' : portExpression( 'SPORT' ),
      'destination' : 'Destination',
      'DPORT' : portExpression( 'DPORT' )
   }

   @classmethod
   def handler( cls, mode, args ):
      proto = ( args.get( cls._protoArgsListName ) or
                args.get( cls._tcpUdpArgsListName ) or
                args.get( cls._tcpFlagArgsListName ) )
      assert proto, args
      source = args.get( 'source' )
      destination = args.get( 'destination' )
      flags = args.get( 'flags', False )
      cls._updateProtoAndPort( mode, args, proto,
                               source, destination, flags=flags, add=True )
      cls._maybeHandleErrors( mode, args, proto, source, destination )

   @classmethod
   def noOrDefaultHandler( cls, mode, args ):
      proto = ( args.get( cls._protoArgsListName ) or
                args.get( cls._tcpUdpArgsListName ) or
                args.get( cls._tcpFlagArgsListName ) )
      source = args.get( 'source' )
      destination = args.get( 'destination' )
      flags = args.get( 'flags', False )

      if not proto and not source and not destination:
         # 'no protocol' removes all protocols
         proto = True

      cls._updateProtoAndPort( mode, args, proto,
                               source, destination, flags=flags, add=False )

class ProtocolIpv4ListConfigCmd( ProtocolBase ):
   data = {
      'PROTOCOL' : ipv4ProtoMatcher,
      'FLAGS_EXPR' : generateTcpFlagExpression( tcpFlagsSupported=True ),
   }
   data.update( ProtocolBase._baseData )

class ProtocolIpv6ListConfigCmd( ProtocolBase ):
   data = {
      'PROTOCOL' : ipv6ProtoMatcher,
      'FLAGS_EXPR' : generateTcpFlagExpression( tcpFlagsSupported=True ),
   }
   data.update( ProtocolBase._baseData )

#--------------------------------------------------------------------------------
# clear traffic-policy counters
#--------------------------------------------------------------------------------
class ClearTrafficPolicyCountersCmd( CliCommand.CliCommandClass ):
   syntax = 'clear traffic-policy counters'
   data = {
      'clear' : CliToken.Clear.clearKwNode,
      'traffic-policy' : matcherTrafficPolicy,
      'counters' : matcherCounters,
   }

   @staticmethod
   def handler( mode, args ):
      for key, status in policiesStatus.iteritems():
         counter = None
         if key == 'sand':
            counter = cpuCounter
         elif key == 'sandAegis':
            counter = intfCounter
         assert counter is not None
         assert status is not None
         LazyMount.force( counter )
         cliHelper = Tac.newInstance( 'TrafficPolicyCli::TrafficPolicyCounter',
                                      status, counter )
         cliHelper.clearCounter()

BasicCli.EnableMode.addCommandClass( ClearTrafficPolicyCountersCmd )

#--------------------------------------------------------------------------------
# clear traffic-policy counters session
#--------------------------------------------------------------------------------
class ClearTrafficPolicyCountersSessionCmd( CliCommand.CliCommandClass ):
   syntax = 'clear traffic-policy counters session'
   data = {
      'clear' : CliToken.Clear.clearKwNode,
      'traffic-policy' : matcherTrafficPolicy,
      'counters' : matcherCounters,
      'session' : 'Clear traffic-policy counters in current session',
   }

   @staticmethod
   def handler( mode, args ):
      for key, status in policiesStatus.iteritems():
         sessionCounter = None
         if key == 'sand':
            sessionCounter = mode.session.sessionData(
               'cpuTrafficPolicySessionCounter', None )
            if sessionCounter is None:
               sessionCounter = Tac.newInstance( 'TrafficPolicy::Counter',
                                                 'cpuSessionCounter' )
               mode.session.sessionDataIs( 'cpuTrafficPolicySessionCounter',
                                           sessionCounter )
         elif key == 'sandAegis':
            sessionCounter = mode.session.sessionData(
               'intfTrafficPolicySessionCounter', None )
            if sessionCounter is None:
               sessionCounter = Tac.newInstance( 'TrafficPolicy::Counter',
                                                 'intfSessionCounter' )
               mode.session.sessionDataIs( 'intfTrafficPolicySessionCounter',
                                           sessionCounter )

         assert status is not None
         assert sessionCounter is not None

         cliHelper = Tac.newInstance( 'TrafficPolicyCli::TrafficPolicyCounter',
                                      status, sessionCounter )
         cliHelper.clearCounter()

BasicCli.EnableMode.addCommandClass( ClearTrafficPolicyCountersSessionCmd )

# --------------------------------------------------------------------------
# The "field-set (ipv4 | ipv6) prefix FIELD_SET_NAME" command
# --------------------------------------------------------------------------
def getIpPrefixFieldSetNames( mode ):
   return fieldSetConfig.fieldSetIpPrefix.keys() if fieldSetConfig else []

ipPrefixFieldSetNameMatcher = CliMatcher.DynamicNameMatcher(
   getIpPrefixFieldSetNames,
   "IP prefix field-set name",
   pattern=protectedFieldSetNamesRegex( 'prefix' ),
   priority=CliParser.PRIO_LOW )

class FieldSetIpPrefixConfigCmd( FieldSetIpPrefixBaseConfigCmd ):
   syntax = 'field-set ipv4 prefix FIELD_SET_NAME'
   noOrDefaultSyntax = syntax
   _feature = "tp"
   data = {
      'FIELD_SET_NAME' : ipPrefixFieldSetNameMatcher,
   }
   data.update( FieldSetIpPrefixBaseConfigCmd._baseData )

   @classmethod
   def _getContextKwargs( cls, fieldSetIpPrefixName, setType, mode=None ):
      return {
         'fieldSetIpPrefixName' : fieldSetIpPrefixName,
         'fieldSetConfig' : fieldSetConfig,
         'setType' : setType,
         'childMode' : FieldSetIpPrefixConfigMode
      }

def getIpv6PrefixFieldSetNames( mode ):
   return fieldSetConfig.fieldSetIpv6Prefix.keys() if fieldSetConfig else []

ipv6PrefixFieldSetNameMatcher = CliMatcher.DynamicNameMatcher(
   getIpv6PrefixFieldSetNames,
   "IPv6 prefix field-set name",
   pattern=protectedFieldSetNamesRegex( 'prefix' ),
   priority=CliParser.PRIO_LOW )

class FieldSetIpv6PrefixConfigCmd( FieldSetIpPrefixBaseConfigCmd ):
   syntax = 'field-set ipv6 prefix FIELD_SET_NAME'
   noOrDefaultSyntax = syntax
   _feature = "tp"
   data = {
      'FIELD_SET_NAME' : ipv6PrefixFieldSetNameMatcher,
   }
   data.update( FieldSetIpPrefixBaseConfigCmd._baseData )

   @classmethod
   def _getContextKwargs( cls, fieldSetIpPrefixName, setType, mode=None ):
      return {
         'fieldSetIpPrefixName' : fieldSetIpPrefixName,
         'fieldSetConfig' : fieldSetConfig,
         'setType' : setType,
         'childMode' : FieldSetIpv6PrefixConfigMode
      }

# Allow multiple prefixes to be defined in prefix field-set
ipPrefixFieldSetExpr = generateFieldSetExpression( ipPrefixFieldSetNameMatcher,
                                                   'FIELD_SET',
                                                   allowMultiple=True )

# --------------------------------------------------------------------------
# The "( source|destination ) prefix field-set { FIELD_SET }" command
# --------------------------------------------------------------------------
class PrefixFieldSetIpv4Cmd( PrefixFieldSetCmdBase ):
   data = {
      "FIELD_SET" : ipPrefixFieldSetExpr
   }
   data.update( PrefixFieldSetCmdBase._baseData )

ipv6PrefixFieldSetExpr = generateFieldSetExpression( ipv6PrefixFieldSetNameMatcher,
                                                     'FIELD_SET',
                                                     allowMultiple=True )

class PrefixFieldSetIpv6Cmd( PrefixFieldSetCmdBase ):
   data = {
      "FIELD_SET" : ipv6PrefixFieldSetExpr
   }
   data.update( PrefixFieldSetCmdBase._baseData )

# --------------------------------------------------------------------------
# The "field-set l4-port PORT_SET_NAME" command
# --------------------------------------------------------------------------
class FieldSetL4PortConfigCmd( FieldSetL4PortBaseConfigCmd ):
   syntax = 'field-set l4-port FIELD_SET_NAME'
   noOrDefaultSyntax = syntax
   _feature = "tp"
   data = {
      'FIELD_SET_NAME' : l4PortFieldSetNameMatcher,
   }
   data.update( FieldSetL4PortBaseConfigCmd._baseData )

   @classmethod
   def _getContextKwargs( cls, fieldSetL4PortName, mode=None ):
      return {
         'fieldSetL4PortName' : fieldSetL4PortName,
         'fieldSetConfig' : fieldSetConfig,
         'childMode' : FieldSetL4PortConfigMode
      }

BasicCli.GlobalConfigMode.addCommandClass( TrafficPoliciesConfigCmd )
TrafficPoliciesConfigMode.addCommandClass( TrafficPolicyConfigCmd )
TrafficPolicyConfigMode.addCommandClass( MatchRuleConfigCmd )
TrafficPolicyConfigMode.addCommandClass( NamedCounterCmd )

# Most modes support 'commit', and 'abort'
for _mode in [ TrafficPolicyConfigMode,
               MatchRuleIpv4ConfigMode,
               MatchRuleIpv6ConfigMode,
               MatchRuleDefaultConfigMode,
               ActionsConfigMode,
               FieldSetIpPrefixConfigMode,
               FieldSetIpv6PrefixConfigMode ]:
   _mode.addModelet( CommitAbortModelet )

# AF-agnostic rules
for _mode in [ MatchRuleIpv4ConfigMode,
               MatchRuleIpv6ConfigMode ]:
   for _cmd in [ ActionsConfigCmd,
                 ProtoMatchConfigCmd,
                 ProtoNeighborsConfigCmd,
                 DscpConfigCmd,
                 TtlConfigCmd,
                 IpLengthConfigCmd,
                 ProtocolFieldSetCmd,
                 ProtocolTcpFlagsOnlyCmd ]:
      _mode.addCommandClass( _cmd )

# AF-specific rules
MatchRuleIpv4ConfigMode.addCommandClass( ProtocolIcmpV4ConfigCmd )
MatchRuleIpv6ConfigMode.addCommandClass( ProtocolIcmpV6ConfigCmd )
MatchRuleIpv4ConfigMode.addCommandClass( ProtocolIcmpV4TypeCodeConfigCmd )
MatchRuleIpv6ConfigMode.addCommandClass( ProtocolIcmpV6TypeCodeConfigCmd )
MatchRuleIpv4ConfigMode.addCommandClass( PrefixIpv4Cmd )
MatchRuleIpv4ConfigMode.addCommandClass( PrefixFieldSetIpv4Cmd )
MatchRuleIpv4ConfigMode.addCommandClass( IpOptionsConfigCmd ) # options are only v4
MatchRuleIpv4ConfigMode.addCommandClass( MatchAllFragmentConfigCmd )
MatchRuleIpv4ConfigMode.addCommandClass( FragmentOffsetConfigCmd )
MatchRuleIpv6ConfigMode.addCommandClass( PrefixIpv6Cmd )
MatchRuleIpv6ConfigMode.addCommandClass( PrefixFieldSetIpv6Cmd )
MatchRuleIpv4ConfigMode.addCommandClass( ProtocolIpv4ListConfigCmd )
MatchRuleIpv6ConfigMode.addCommandClass( ProtocolIpv6ListConfigCmd )

MatchRuleDefaultConfigMode.addCommandClass( ActionsConfigCmd )

ActionsConfigMode.addCommandClass( DropActionCmd )
ActionsConfigMode.addCommandClass( PoliceActionCmd )
ActionsConfigMode.addCommandClass( CountActionCmd )
ActionsConfigMode.addCommandClass( LogActionCmd )
ActionsConfigMode.addCommandClass( SetDscpActionCmd )
ActionsConfigMode.addCommandClass( SetTcActionCmd )

TrafficPoliciesConfigMode.addCommandClass( FieldSetIpPrefixConfigCmd )
TrafficPoliciesConfigMode.addCommandClass( FieldSetIpv6PrefixConfigCmd )
TrafficPoliciesConfigMode.addCommandClass( FieldSetL4PortConfigCmd )

# Add field-set commands to appropriate modes
FieldSetIpPrefixConfigMode.addCommandClass( FieldSetIpPrefixConfigCmds )
FieldSetIpv6PrefixConfigMode.addCommandClass( FieldSetIpv6PrefixConfigCmds )
FieldSetIpPrefixConfigMode.addCommandClass( FieldSetIpPrefixExceptConfigCmds )
FieldSetIpv6PrefixConfigMode.addCommandClass( FieldSetIpv6PrefixExceptConfigCmds )

for m in [ TrafficPolicyConfigMode, MatchRuleIpv4ConfigMode,
           MatchRuleIpv6ConfigMode, MatchRuleDefaultConfigMode, ActionsConfigMode ]:
   m.addShowCommandClass( ShowPendingCmd )

# --------------------------------------------------------------------------
# The "update interface default action drop" command
# --------------------------------------------------------------------------
class UpdateDefaultActionConfigCmd( CliCommand.CliCommandClass ):
   syntax = "update interface default action drop"
   noOrDefaultSyntax = "update interface default action ..."
   data = {
      'update' : 'Traffic policy update parameters',
      'interface' : 'Set behavior for interface traffic policy',
      'default' : 'Set default behavior during update',
      'action' : 'Set default action during update',
      'drop' : 'Set default action during update to drop'
   }

   @staticmethod
   def handler( mode, args ):
      policiesIntfParamConfig.actionDuringUpdate = "drop"

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      policiesIntfParamConfig.actionDuringUpdate = "permit"

TrafficPoliciesConfigMode.addCommandClass( UpdateDefaultActionConfigCmd )

# --------------------------------------------------------------------------
# The "counter interface per-interface ingress" command
# --------------------------------------------------------------------------
class CounterIntfPerInterfaceIngressConfigCmd( CliCommand.CliCommandClass ):
   syntax = "counter interface per-interface ingress"
   noOrDefaultSyntax = syntax
   data = {
      'counter' : 'Traffic policy counter parameters',
      'interface' : 'Set behavior for interface traffic policy',
      'per-interface' : 'Set count to be per-interace',
      'ingress' : 'Set parameter for ingress traffic policies',
   }

   @staticmethod
   def handler( mode, args ):
      policiesIntfParamConfig.ingressCounterGranularity = "counterPerInterface"

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      policiesIntfParamConfig.ingressCounterGranularity = "counterPerPolicy"

TrafficPoliciesConfigMode.addCommandClass( CounterIntfPerInterfaceIngressConfigCmd )

def noTrafficPolicies( mode, args ):
   policiesIntfParamConfig.actionDuringUpdate = "permit"
   policiesIntfParamConfig.ingressCounterGranularity = "counterPerPolicy"

# pylint: disable=protected-access
TrafficPoliciesConfigCmd._registerNoHandler( noTrafficPolicies )
# pylint: enable=protected-access

def Plugin( em ):
   global policiesCliConfig, policiesIntfParamConfig
   global policiesStatusRequestDir, policiesStatus
   global qosHwStatus
   global qosAclHwStatus
   global entityManager
   global cpuCounter
   global intfCounter
   global fieldSetConfig

   policiesRootNode = 'trafficPolicies'
   policiesCellRootNode = 'cell/%d/trafficPolicies' % Cell.cellId()
   statusNode = 'status'
   policiesCliConfigNode = 'input/cli'
   policiesCliConfigPath = policiesRootNode + '/' + policiesCliConfigNode
   policiesCliConfigType = 'TrafficPolicy::TrafficPolicyConfig'
   policiesIntfParamConfigPath = policiesRootNode + '/param/config/interface'
   policiesIntfParamConfigType = 'TrafficPolicy::TrafficPolicyIntfParamConfig'
   policiesStatusPath = policiesCellRootNode + '/' + statusNode
   policiesStatusType = 'Tac::Dir'
   statusRequestDirNode = 'statusRequest'
   policiesStatusRequestDirPath = policiesRootNode + '/' + statusRequestDirNode
   policiesStatusRequestDirType = 'PolicyMap::PolicyMapStatusRequestDir'
   policiesCounterNode = 'counter'
   policiesCounterPath = policiesRootNode + '/' + policiesCounterNode
   policiesCounterType = 'TrafficPolicy::Counter'
   entityManager = em

   mountGroup = entityManager.mountGroup()
   policiesStatus = mountGroup.mount( policiesStatusPath, policiesStatusType, 'ri' )
   mountGroup.close( callback=None, blocking=False )

   policiesCliConfig = ConfigMount.mount( entityManager, policiesCliConfigPath,
                                          policiesCliConfigType, 'wi' )
   policiesIntfParamConfig = ConfigMount.mount(
      entityManager, policiesIntfParamConfigPath, policiesIntfParamConfigType, 'wi' )
   policiesStatusRequestDir = LazyMount.mount( entityManager,
                                               policiesStatusRequestDirPath,
                                               policiesStatusRequestDirType,
                                               'w' )
   qosHwStatus = LazyMount.mount( entityManager, "qos/hardware/status/global",
                                  "Qos::HwStatus", "r" )
   qosAclHwStatus = LazyMount.mount( entityManager,
                                     "qos/hardware/acl/status/global",
                                     "Qos::AclHwStatus", "r" )
   cpuCounter = LazyMount.mount( entityManager, policiesCounterPath + '/cpu',
                                 policiesCounterType, "wr" )
   intfCounter = LazyMount.mount( entityManager, policiesCounterPath + '/interface',
                                  policiesCounterType, "wr" )
   fieldSetConfig = ConfigMount.mount( entityManager, 'trafficPolicies/fieldset/cli',
                                       'Classification::FieldSetConfig', 'w' )
