# Copyright (c) 2008, 2009, 2010, 2011 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

#
# This module implements 'show spanning-tree topology status' commands.
#

import BasicCli
from VlanCli import vlanIdMatcher
from CliModel import UnknownEntityError
from .TopoModel import TopoStatus
import ShowCommand

topoStatus = None
topoConfig = None
vlanConfig = None

def getTopoStatus( status, vlans, detail ):
   ports = {}
   timeDelta = 0
   if detail:
      import Tac
      timeDelta = Tac.utcNow() - Tac.now()  # Adjustment between monotonic and UTC
   for topoPortStatusName in status.topoPortStatus:
      topoPortStatus = status.topoPortStatus.get( topoPortStatusName )
      if not topoPortStatus:
         continue
      port = TopoStatus.Topology.PortStatus( state=topoPortStatus.state )
      if detail:
         ci = topoPortStatus.changeInfo
         port.numChanges = ci.changeCount
         port.lastChange = ci.lastChangeTime + timeDelta
      ports[ topoPortStatus.intfId ] = port
   return ports

def topoToVidMap():
   topoToVids = {}
   activeVlans = set( k for k, v in vlanConfig.vlanConfig.iteritems()
                      if v.adminState == 'active' )
   vlansWithTopology = set()
   activeTopos = set( topoStatus.topoStatus.keys() )
   for vlanId, topo in topoConfig.vidToTopoMap.iteritems():
      if topo.name not in activeTopos:
         continue
      vlansWithTopology.add( vlanId )
      vids = topoToVids.setdefault( topo.name, set() )
      vids.add( vlanId )
   return ( topoToVids, activeVlans - vlansWithTopology )

def showSpanTopo( mode, args ):
   vlanId = args.get( 'VLANID' )
   result = TopoStatus()
   ( topoToVids, noTopoVlans ) = topoToVidMap()
   if vlanId and str( vlanId.id ) in noTopoVlans:
      raise UnknownEntityError( "Vlan %d not mapped to any topology " % vlanId.id )
   if noTopoVlans:
      result.unmappedVlans = list( noTopoVlans )
   for topoName, status in sorted( topoStatus.topoStatus.iteritems() ):
      if status is None:
         continue      
      mappedVlans = topoToVids.get( status.name )
      if not mappedVlans:
         continue
      vlans = None
      if vlanId:
         if vlanId.id in mappedVlans:
            vlans = [ vlanId.id ]
      else:
         vlans = list( mappedVlans )
      if vlans is not None:
         topology = TopoStatus.Topology( vlans=vlans )
         topology.interfaces = getTopoStatus( status, vlans, 'detail' in args )
         result.topologies[ topoName ] = topology
   return result

#--------------------------------------------------------------------------------
# show spanning-tree topology [ vlan [ VLANID ] ] status [ detail ]
#--------------------------------------------------------------------------------
class SpanningTreeTopologyVlanStatusCmd( ShowCommand.ShowCliCommandClass ):
   syntax = ( 'show spanning-tree topology '
               '[ vlan [ VLANID ] ] '
               'status [ detail ] ' )
   data = {
      'spanning-tree': 'Details on STP',
      'topology': 'Details of the STP topology',
      'vlan': 'Report on a VLAN',
      'VLANID': vlanIdMatcher,
      'status': 'Status information',
      'detail': 'With more debugging details',
   }
   handler = showSpanTopo
   cliModel = TopoStatus

BasicCli.addShowCommandClass( SpanningTreeTopologyVlanStatusCmd )

#-------------------------------------------------------------------------------
# Have the Cli Agent mount all needed state from sysdb
#-------------------------------------------------------------------------------
def Plugin( entityManager ):
   global topoStatus, topoConfig, vlanConfig
   import LazyMount
   topoStatus = LazyMount.mount( entityManager,
                                 "bridging/topology/status",
                                 "Bridging::Topology::Status", "r" )
   topoConfig = LazyMount.mount( entityManager,
                                 "bridging/topology/config",
                                 "Bridging::Topology::Config", "r" )
   vlanConfig = LazyMount.mount( entityManager,
                                 "bridging/config",
                                 "Bridging::Config", "r" )
