# Copyright (c) 2007, 2008, 2009, 2010 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

from __future__ import absolute_import, division, print_function

import json
import re
import sys

import ArPyUtils
import BasicCli
import BasicCliModes
import BasicCliSession
import CliApi
import CliCommand
import CliCommon
from CliMode.TechSupport import TechSupportPolicyMode
import CliModel
import CliMatcher
import CliParser
import CliPlugin.ConfigMgmtMode as ConfigMgmtMode
import ConfigMount
import ShowCommand
import Tac
import TacSigint

cliConfig = None

#------------------------------------------------------------------------------------
# The "show tech-support" command
#------------------------------------------------------------------------------------
cmdCallbacks = []
# show tech-support summary commands
summaryCmdCallbacks = []
# Extra "show tech <this and that keyword> [debugging ...]" commands can be regist-
# ered. Keep the <this and that> keywords in this list (for dynamic keyword matcher)
extendedOptions = [] # list(path); path=list(token)

# Dictionary of Callback lists for each of the options under 'extended'
extendedOptCallbacks = {}

class ShowTechModel( CliModel.DeferredModel ):
   __public__ = False
   showTech = CliModel.Str( help="Dummy show tech-support model" )

def registerShowTechSupportCmdCallback( timeStamp, cmdCallback=None, extended=None,
                                        summaryTimeStamp=None,
                                        summaryCmdCallback=None ):
   """
   API used by other CliPlugins to register commands to be called when
   'show tech-support' is called.

   'timesStamp' is a string in the format returned by time.strftime(
   '%Y-%m-%d %H:%M:%S ), for example '2010-06-11 23:46:19'.
   Commands are run in the ascending order of the time stamps so that
   we get the effect of adding commands at the end as we add them in
   new releases over time.  When you add a new command, please
   generate a new time stamp to use using the following python code:

   import time
   print time.strftime( '%Y-%m-%d %H:%M:%S' )

   It is important that we try to keep the order of the list consistent from
   release to release, so please make sure to generate a real time stamp based
   on the current time when you add a call to this function.  Using a time stamp
   with granularity down to seconds should remove the need for any tie breakers.

   The format chosen here explicitly avoids any locale specific formatting
   so that our code is not sensitive to the setting of the LANG environment
   variable.  We validate that the string matches the desired format with
   a regexp, then sort the strings as strings.

   'cmdCallback' is a function that returns a list of commands to run
   on the given system. It is run when 'show tech-support' is called,
   not when it is registered.

   To register new commands with 'show tech-support', do this in your
   CliPlugin:

   import CliPlugin.TechSupportCli

   def _showTechCmds():
      # This function could return commands unconditionally, as shown
      # here, or conditionally (e.g., Sand/CliPlugin/SandCli.py).
      cmds = [ 'show chickens',
               'show donkeys',
               ]
      return cmds
   timeStamp = '2010-06-11 23:46:19'

   CliPlugin.TechSupportCli.registerShowTechSupportCmdCallback( timeStamp,
                                                                _showTechCmds )

   summaryTimeStamp and summaryCmdCallback work similar to timeStamp and
   cmdCallback, but for "show tech-support summary" command. If None is passed for
   summaryTimeStamp, then it assumes the same value as timeStamp
   """
   assert re.match( r'\d\d\d\d-\d\d-\d\d \d\d:\d\d:\d\d', timeStamp )
   if extended is not None:
      assert cmdCallback
   assert cmdCallback or summaryCmdCallback

   if extended is None:
      if cmdCallback:
         cmdCallbacks.append( ( timeStamp, cmdCallback ) )
      if summaryCmdCallback:
         if summaryTimeStamp is None:
            summaryTimeStamp = timeStamp
         summaryCmdCallbacks.append( ( summaryTimeStamp, summaryCmdCallback ) )
   else:
      tokens = extended.split( " " )
      if tokens not in extendedOptions:
         extendedOptions.append( tokens )
      if extended not in extendedOptCallbacks:
         extendedOptCallbacks[ extended ] = []
      extendedOptCallbacks[ extended ].append( ( timeStamp, cmdCallback ) )

def showTechSupportExec( mode, inputCmd, inCmdCallbacks, benchmark=None,
                         outputFormat=None, showAll=True ):
   def sortKey( timestampAndCallback ):
      return timestampAndCallback[ 0 ]

   benchmarkData = {}
   cmds = []
   for ( _, cmd ) in sorted( inCmdCallbacks, key=sortKey ):
      try:
         cmds.extend( cmd() )
      except Exception, e:      # pylint: disable-msg=broad-except
         # The callback to tell us what commands to run raised an exception.
         callbackName = 'plugin for %s.%s' % ( cmd.__module__, cmd.__name__ )
         logFileNum = mode.session.writeInternalError( cmd=callbackName )
         cmds.append( '! %s raised an exception: %r' % ( callbackName, e ) )
         if logFileNum is not None:
            cmds.append( 'show error %d' % logFileNum )

   # add user configurable commands
   # Don't do this for 'show tech-support summary'. It is supposed to have it's own
   # include/exclude commands
   if inputCmd != 'show tech-support summary':
      cmds.extend( cliConfig.includedShowTechCmds.keys() )

   capiExecutor = None
   firstCmd = True
   jsonFmt = mode.session.outputFormat_ == 'json' or outputFormat == 'json'
   if jsonFmt:
      session = BasicCliSession.Session( BasicCliModes.EnableMode,
                                    mode.session.entityManager,
                                    privLevel=CliCommon.MAX_PRIV_LVL,
                                    disableAaa=True,
                                    disableAutoMore=True,
                                    isEapiClient=True,
                                    shouldPrint=False,
                                    disableGuards=not mode.session.guardsEnabled(),
                                    interactive=False,
                                    cli=mode.session.cli,
                                    aaaUser=mode.session.aaaUser() )
      capiExecutor = CliApi.CapiExecutor( mode.session.cli, session,
                                          stateless=False )
      print( '{\n"%s": {' % inputCmd )
      sys.stdout.flush()

   for cmd in cmds:
      # Excluded commands are stored in lowercase to eliminate misses from users
      # capitalizing commands.
      cmdLowercase = cmd.lower()
      if not showAll and cmdLowercase in cliConfig.excludedShowTechCmds:
         continue

      if jsonFmt:
         if cmd.startswith( 'bash' ):
            # BASH commands aren't don't work with JSON
            continue

         if cmd in CliCommon.skippedJsonCmds:
            # commands are buggy, skip them
            continue

         if not showAll and cmdLowercase in cliConfig.excludedShowTechJsonCmds:
            continue

         if not firstCmd:
            print( ',' )
         print( '"%s": ' % cmd )
         sys.stdout.flush()
      else:
         print( '\n------------- %s -------------\n' % cmd )
         sys.stdout.flush()
      startTime = Tac.now()
      try:
         if jsonFmt:
            print( '{ "json":' )
            sys.stdout.flush()
            result = capiExecutor.executeCommands( [ CliApi.CliCommand( cmd ) ],
                                                   textOutput=False,
                                                   autoComplete=True,
                                                   streamFd=sys.stdout.fileno() )
            sys.stdout.flush()
            if result[ 0 ].status == CliApi.CapiStatus.NOT_CAPI_READY:
               with ArPyUtils.FileHandleInterceptor( [ sys.stdout.fileno() ] ) \
                      as capturedStdout:
                  capiExecutor.executeCommands( [ CliApi.CliCommand( cmd ) ],
                                                textOutput=True,
                                                autoComplete=True,
                                                streamFd=sys.stdout.fileno() )
               output = capturedStdout.contents().decode( 'utf-8', 'replace' )
               print( ', "text": { "output": %s }' % json.dumps( output ) )
            print( '}' )
            sys.stdout.flush()
         else:
            mode.session_.runCmd( cmd, aaa=False )
         firstCmd = False
      except CliParser.GuardError, e:
         if not jsonFmt:
            print( "(unavailable: %s)" % e.guardCode )
      except CliParser.AlreadyHandledError, e:
         mode.session_.handleAlreadyHandledError( e )
      except KeyboardInterrupt:
         raise
      except: # pylint: disable=bare-except
         # Catch all errors, so that one command failure doesn't cause
         # output from others to be skipped.
         #
         # NOTE NOTE NOTE!!
         #
         # If you change what is printed when a command fails, PLEASE
         # ALSO UPDATE src/Eos/ptest/ShowTechCli.py, which looks for
         # this string in the output of "show tech-support" on our
         # various platforms.
         if not jsonFmt:
            print( "(unavailable)" )
      TacSigint.check()

      if benchmark is not None:
         duration = Tac.now() - startTime
         benchmarkData[ cmd ] = duration

   if benchmark is not None:
      if jsonFmt:
         # TODO: this needs to be converted to print json as well
         # TODO: add test for benchmark and JSON
         pass
      else:
         print()
         print( '-' * 40 )
         print( 'Benchmark for top %d commands:' % benchmark )
         print( '-' * 40 )
         for cmd in sorted( benchmarkData,
                            key=lambda c: -benchmarkData[ c ] )[ : benchmark ]:
            print( '%r took %.2f seconds' % ( cmd, benchmarkData[ cmd ] ) )

   if jsonFmt:
      print( '}\n}\n' )
      sys.stdout.flush()
   else:
      # Clear any errors, they pertain to the last run command only anyway; we don't
      # want ugly syslogs about show tech failures when last cmd was 'Not supported'
      # and the message itself was already printed anyway (since !json mode).
      mode.session_.clearMessages()

   return CliModel.noValidationModel( ShowTechModel )

#-------------------------------------------------------
# tech-support config commands
#
# From config mode
#    management tech-support
#       [ no | default ] policy show tech-support
#          [ no | default ] exclude command CMD
#-------------------------------------------------------
class TechSupportMgmtMode( ConfigMgmtMode.ConfigMgmtMode ):
   name = 'Tech-Support configuration'
   modeParseTree = CliParser.ModeParseTree()

   def __init__( self, parent, session ):
      ConfigMgmtMode.ConfigMgmtMode.__init__( self, parent, session,
                                             'tech-support' )

class ManagementShowTechConfigCmd( CliCommand.CliCommandClass ):
   syntax = 'management tech-support'
   data = {
            'management': ConfigMgmtMode.managementKwMatcher,
            'tech-support': 'Configure tech-support policy',
          }

   @staticmethod
   def handler( mode, args ):
      childMode = mode.childMode( TechSupportMgmtMode )
      mode.session_.gotoChildMode( childMode )

BasicCliModes.GlobalConfigMode.addCommandClass( ManagementShowTechConfigCmd )

#-------------------------------------------------------
# policy show tech-support
#-------------------------------------------------------
class ShowTechPolicyMode( TechSupportPolicyMode, BasicCliModes.ConfigModeBase ):
   name = 'show tech-support configuration'
   modeParseTree = CliParser.ModeParseTree()

   def __init__( self, parent, session ):
      TechSupportPolicyMode.__init__( self, 'show tech-support' )
      BasicCliModes.ConfigModeBase.__init__( self, parent, session )

class ShowTechPolicyCmd( CliCommand.CliCommandClass ):
   syntax = 'policy show tech-support'
   noOrDefaultSyntax = 'policy show tech-support'
   data = {
            'policy': 'Configure tech-support policy',
            'show': 'Configure a show command policy',
            'tech-support': 'Configure show tech-support policy'
          }

   @staticmethod
   def handler( mode, args ):
      childMode = mode.childMode( ShowTechPolicyMode )
      mode.session_.gotoChildMode( childMode )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      cliConfig.excludedShowTechCmds.clear()
      cliConfig.excludedShowTechJsonCmds.clear()
      cliConfig.includedShowTechCmds.clear()

TechSupportMgmtMode.addCommandClass( ShowTechPolicyCmd )

#-------------------------------------------------------
# [ no | default ] exclude command [ json ] CMD
# "no|default exclude command" will clear all currently excluded commands
#-------------------------------------------------------
# Add/remove a command from 'show tech-support' or 'show tech-support | json'.
# If excludeCmd is None and noOrDefault=True then clear all excluded commands.
def techSupportExclude( mode, excludeCmd, excludeJson=None, noOrDefault=False ):
   if not excludeCmd:
      if noOrDefault:
         if excludeJson:
            cliConfig.excludedShowTechJsonCmds.clear()
         else:
            cliConfig.excludedShowTechCmds.clear()
      return

   # Convert to lowercase
   excludeCmd = excludeCmd.lower()

   if noOrDefault:
      if excludeJson and excludeCmd in cliConfig.excludedShowTechJsonCmds:
         del cliConfig.excludedShowTechJsonCmds[ excludeCmd ]
      elif not excludeJson and excludeCmd in cliConfig.excludedShowTechCmds:
         del cliConfig.excludedShowTechCmds[ excludeCmd ]
      else:
         mode.addWarning( 'No matched command' )
   else:
      matched = False
      for ( _, cmdCallback ) in cmdCallbacks:
         cmds = []
         try:
            cmds = cmdCallback()
         except Exception: # pylint: disable-msg=broad-except
            # If the callback raised an exception, treat is an empty list.
            pass
         matched = any( cmd.lower() == excludeCmd for cmd in cmds )
         if matched:
            break

      # Add a warning if there is no matching command
      if not matched:
         mode.addWarning( 'No matched command' )

      excludeConfig = ( cliConfig.excludedShowTechJsonCmds if excludeJson else
                        cliConfig.excludedShowTechCmds )
      excludeConfig[ excludeCmd ] = True

class ExlcudeCommand( CliCommand.CliCommandClass ):
   syntax = 'exclude command [ json ] CMD'
   noOrDefaultSyntax = 'exclude command [ json ] [ CMD ]'
   data = {
            'exclude': 'Exclude command from "show tech-support"',
            'command': 'Exclude command from "show tech-support"',
            'json': 'Exclude command from "show tech-support | json"',
            'CMD': CliMatcher.StringMatcher( helpname='CMD',
                                               helpdesc='Command to exclude' )
          }

   @staticmethod
   def handler( mode, args ):
      excludeJson = args.get( 'json' )
      excludeCmd = args[ 'CMD' ]
      techSupportExclude( mode, excludeCmd=excludeCmd, excludeJson=excludeJson )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      excludeJson = args.get( 'json' )
      excludeCmd = args.get( 'CMD' )
      techSupportExclude( mode, excludeCmd=excludeCmd, excludeJson=excludeJson,
            noOrDefault=True )

ShowTechPolicyMode.addCommandClass( ExlcudeCommand )

#-------------------------------------------------------
# [ no | default ] include command CMD
# "no|default include command" will clear all currently included commands
#-------------------------------------------------------
class IncludeCommand( CliCommand.CliCommandClass ):
   syntax = 'include command CMD'
   noOrDefaultSyntax = 'include command [ CMD ]'
   data = {
            'include': 'Include command in "show tech-support"',
            'command': 'Include command in "show tech-support"',
            'CMD': CliMatcher.StringMatcher( helpname='CMD',
                                               helpdesc='Command to include' )
          }

   @staticmethod
   def handler( mode, args ):
      includedCmd = args[ 'CMD' ]
      cliConfig.includedShowTechCmds[ includedCmd ] = True

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      if 'CMD' not in args:
         cliConfig.includedShowTechCmds.clear()
         return
      includedCmd = args[ 'CMD' ]
      del cliConfig.includedShowTechCmds[ includedCmd ]

ShowTechPolicyMode.addCommandClass( IncludeCommand )

#------------------------------------------------------------------------------------
# Show Commands
#------------------------------------------------------------------------------------
techSupportKwMatcher = CliMatcher.KeywordMatcher( 'tech-support',
                        helpdesc='Show aggregated status and configuration details' )
extendedKwMatcher = CliMatcher.KeywordMatcher( 'extended',
                                    helpdesc='Show tech-support extended command' )
summaryKwMatcher = CliMatcher.KeywordMatcher(
    'summary', helpdesc='Show summarized status and configuration details' )
extendedValueMatcher = CliMatcher.KeywordListsMatcher( extendedOptions,
                                     helpdesc="Extended Show tech-support for %s" )
#-----------------------------------------------------------------------------------
# show tech-support [ all | ( extended EXTENDED ) ]
#                   [ benchmark BENCHMARK ] [ json | text ]
#-----------------------------------------------------------------------------------

class ShowTechSupport( ShowCommand.ShowCliCommandClass ):
   syntax = ( 'show tech-support [ summary | all | ( extended EXTENDED ) ] '
                                '[ benchmark BENCHMARK ] [ json | text ]' )
   data = {
            'tech-support': techSupportKwMatcher,
            'summary': summaryKwMatcher,
            'all': 'Show all aggregated status and configuration details',
            'extended': extendedKwMatcher,
            'EXTENDED': extendedValueMatcher,
            'benchmark': CliCommand.Node(
                                 matcher=CliMatcher.KeywordMatcher( 'benchmark',
                                             helpdesc='Show command benchmark' ),
                                 hidden=True ),
            'BENCHMARK': CliCommand.Node(
                                 matcher=CliMatcher.IntegerMatcher( 1, 9999,
                                            helpdesc='Show top N slowest commands' ),
                                 hidden=True ),
            'json': CliCommand.Node(
                        matcher=CliMatcher.KeywordMatcher( 'json',
                                                         helpdesc='output in json' ),
                        hidden=True,
                        alias='OUTPUT_FORMAT' ),
            'text': CliCommand.Node(
                        matcher=CliMatcher.KeywordMatcher( 'text',
                                                         helpdesc='output in text' ),
                        hidden=True,
                        alias='OUTPUT_FORMAT' ),
          }
   cliModel = ShowTechModel
   privileged = True

   @staticmethod
   def handler( mode, args ):
      inCmdCallbacks = cmdCallbacks
      showAll = True
      extended = args.get( 'EXTENDED' )
      if 'all' in args:
         inputCmd = 'show tech-support all'
      elif extended is not None:
         extended = ' '.join( extended )
         inputCmd = 'show tech-support extended %s' % extended
         inCmdCallbacks = extendedOptCallbacks[ extended ]
      elif 'summary' in args:
         inputCmd = 'show tech-support summary'
         inCmdCallbacks = summaryCmdCallbacks
      else:
         inputCmd = 'show tech-support'
         showAll = False

      return showTechSupportExec( mode, inputCmd=inputCmd,
                                  inCmdCallbacks=inCmdCallbacks,
                                  benchmark=args.get( 'BENCHMARK' ),
                                  outputFormat=args.get( 'OUTPUT_FORMAT' ),
                                  showAll=showAll )

BasicCli.addShowCommandClass( ShowTechSupport )

#-----------------------------------------------------------------------------------
# show tech-support [ extended EXTENDED ] debugging
# Debugging help for what callbacks exist and what they return
#-----------------------------------------------------------------------------------
class ShowTechSupportDebugging( ShowCommand.ShowCliCommandClass ):
   syntax = 'show tech-support [ summary | ( extended EXTENDED ) ] debugging'
   data = {
            'tech-support': techSupportKwMatcher,
            'summary': summaryKwMatcher,
            'extended': extendedKwMatcher,
            'EXTENDED': extendedValueMatcher,
            'debugging': 'Show debugging for tech-support callbacks',
          }
   privileged = True
   hidden = True

   @staticmethod
   def handler( mode, args ):
      extended = args.get( 'EXTENDED' )
      if 'summary' in args:
         callbacks = summaryCmdCallbacks
      elif extended is None:
         callbacks = cmdCallbacks
      else:
         extended = ' '.join( extended )
         callbacks = extendedOptCallbacks[ extended ]
      for ( ts, cmd ) in sorted( callbacks, key=lambda t: t[ 0 ] ):
         try:
            val = cmd()
         except Exception, e:   # pylint: disable-msg=broad-except
            val = e
         print( '%s %s.%s: %r' % ( ts, cmd.__module__, cmd.__name__, val ) )

BasicCli.addShowCommandClass( ShowTechSupportDebugging )

def Plugin( entityManager ):
   global cliConfig
   cliConfig = ConfigMount.mount( entityManager, 'cli/config',
                                  'Cli::Config', 'w' )
