#!/usr/bin/env python
# Copyright (c) 2017 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import BasicCli
import CliCommand
import CliMatcher
import CliParser
from CliMode.Te import GlobalTeMode
import CliPlugin.IntfCli as IntfCli
import CliPlugin.IpAddrMatcher as IpAddrMatcher
import CliPlugin.Ip6AddrMatcher as Ip6AddrMatcher
import CliToken.Router
from MultiRangeRule import MultiRangeMatcher
import ConfigMount
import Tac
from TypeFuture import TacLazyType

# pkgdeps: rpmwith %{_libdir}/libTe.so*

# When just traffic-engineering bandwidth is configured,
# it would imply 75% of the link bw is available as maximum
# reservable bandwidth
IMPLIED_MAX_RESERVABLE_BW = 75

teConfig = None

def getSrlgIdToNameMap():
   config = teConfiguration()
   srlgIdToNameMap = {}
   for srlgNameKey, srlgMapEntry in config.srlgMap.iteritems():
      srlgIdToNameMap[ srlgMapEntry.srlgId ] = srlgNameKey
   return srlgIdToNameMap

def adminGroupRange():
   return ( 0, 31 )

def rangeToValue( rangeList ):
   bitmask = 0
   if rangeList:
      for i in rangeList:
         bitmask += 1 << i
   return bitmask

bandwidth = TacLazyType( 'TrafficEngineering::Bandwidth' )
bwUnit = TacLazyType( 'TrafficEngineering::BwUnitType' )
srlgGid = TacLazyType( 'TrafficEngineering::SrlgId' )
metric = TacLazyType( 'TrafficEngineering::Metric' )

#-------------------------------------------------------------------------------
# Adds TE-specific CLI commands to the 'config-if' mode for routed ports.
# Enable TE commands under config-if only on routed and loopback ports
#-------------------------------------------------------------------------------
def teSupportedOnIntfConfigMode( intfConfigMode ):
   # Don't configure TE on ineligible interfaces
   if intfConfigMode.intf.routingSupported() and \
         not intfConfigMode.intf.name.startswith( 'Management' ):
      return True
   return False

class RoutingProtocolTeIntfConfigModelet( CliParser.Modelet ):

   modeletParseTree = CliParser.ModeletParseTree()

   def __init__( self, intfConfigMode ):
      CliParser.Modelet.__init__( self )
      self.intf = TeIntf( intfConfigMode.intf, intfConfigMode.sysdbRoot )
      self.mode = intfConfigMode

   @staticmethod
   def shouldAddModeletRule( mode ):
      return teSupportedOnIntfConfigMode( mode )

#-------------------------------------------------------------------------------
# Associate the RoutingProtocolIntfConfigModelet with the 'config-if' mode.
#-------------------------------------------------------------------------------
IntfCli.IntfConfigMode.addModelet( RoutingProtocolTeIntfConfigModelet )

def teConfiguration():
   return teConfig

class TeIntf( IntfCli.IntfDependentBase ):

   def setDefault( self ):
      config = teConfiguration()
      del config.intfConfig[ self.intf_.name ]

modelet = RoutingProtocolTeIntfConfigModelet

#-------------------------------------------------------------------------------
# IntfConfig for Traffic-engineering configuration, is created when one of its
# attributes is configured. It is deleted when all the attributes are at their
# defaults What this means is that after the last 'no traffic engineering...'
# command is run on the interface, we delete the intfConfig object
#------------------------------------------------------------------------------
def _getIntfConfig( intfName ):
   return teConfiguration().intfConfig.get( intfName, None )

def _getOrCreateIntfConfig( config, intfName ):
   intfConfig = config.intfConfig.get( intfName, None )
   if intfConfig is None:
      intfConfig = config.intfConfig.newMember( intfName )
   return intfConfig

def _deleteIntfConfigIfAllAttributeHaveDefaults( config, intfName ):
   '''Delete the intfConfig collection element if all the attributes of intfConfig
   have default values. This needs to be called by command handlers for 'no' form
   of the Traffic-engineering interface config commands.
   '''
   intfConfig = config.intfConfig.get( intfName, None )
   if intfConfig is None:
      return
   if ( ( intfConfig.enabled == intfConfig.enabledDefault ) and
        ( intfConfig.maxReservableBwUnit == \
              intfConfig.maxReservableBwUnitDefault ) and
        ( intfConfig.maxReservableBw == intfConfig.maxReservableBwDefault ) and
        ( intfConfig.adminGroup == intfConfig.adminGroupDefault ) and
        ( intfConfig.metric == intfConfig.metricDefault ) and
        ( len( intfConfig.srlgIdList ) == 0 ) and
        ( len( intfConfig.srlgNameList ) == 0 ) ):

      del config.intfConfig[ intfName ]

#-------------------------------------------------------------------------
# Helper Class. This is used as a base class for all dependent classes of
# 'config-te' mode. When the traffic engineering is unconfigured the
# dependents can cleanup their configuration.
#-------------------------------------------------------------------------
class TeDependentBase( object ):
   def __init__( self, mode ):
      pass

   def setDefault( self ):
      pass

#----------------------------------------
# Routine to register dependent class
#----------------------------------------
class TeModeDependents( object ):
   dependentClasses_ = []

   @classmethod
   def registerDependentClass( cls, derived, priority=20 ):
      d = cls.dependentClasses_ + [ ( priority, derived ) ]
      d = sorted( d, key=lambda x: x[ 0 ] )
      cls.dependentClasses_ = d

#------------------------------------------------------------------------------
# router traffic engineering config mode. This mode is created when the user
# enters 'router traffic engineering'
#------------------------------------------------------------------------------
class RouterGlobalTeMode( GlobalTeMode, BasicCli.ConfigModeBase ):

   name = 'Traffic-engineering global configuration'
   modeParseTree = CliParser.ModeParseTree()

   def __init__( self, parent, session ):
      GlobalTeMode.__init__( self, None )
      BasicCli.ConfigModeBase.__init__( self, parent, session )

class RouterGlobalTeModelet( CliParser.Modelet ):
   modeletParseTree = CliParser.ModeletParseTree()

   def __init__( self, mode ):
      CliParser.Modelet.__init__( self )
      self.mode = mode

RouterGlobalTeMode.addModelet( RouterGlobalTeModelet )

#--------------------------------------------------------------------------------
# [ no | default ] router traffic-engineering
#--------------------------------------------------------------------------------
class RouterTrafficEngineeringCmd( CliCommand.CliCommandClass ):
   syntax = 'router traffic-engineering'
   noOrDefaultSyntax = syntax
   data = {
      'router' : CliToken.Router.routerMatcherForConfig,
      'traffic-engineering' : 'Traffic-engineering global config',
   }

   @staticmethod
   def handler( mode, args ):
      teConfig.enabled = True
      childMode = mode.childMode( RouterGlobalTeMode )
      mode.session_.gotoChildMode( childMode )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      config = teConfiguration()
      config.routerId = config.routerIdDefault
      config.routerIdV6 = config.routerIdV6Default
      config.srlgMap.clear()
      teConfig.enabled = False

      for ( _priority, cls ) in TeModeDependents.dependentClasses_:
         cls( mode ).setDefault()

BasicCli.GlobalConfigMode.addCommandClass( RouterTrafficEngineeringCmd )

#--------------------------------------------------------------------------------
# [ no | default ] srlg NAME GROUP_ID
#--------------------------------------------------------------------------------
# SRLG name must contain atleast one letter
matcherSrlgName = CliMatcher.PatternMatcher( '.{0,100}[A-Za-z]+.{0,100}',
                                             helpname='WORD', helpdesc='SRLG Name' )
matcherSrlgGroupId = CliMatcher.IntegerMatcher( srlgGid.srlgIdMin, srlgGid.srlgIdMax,
                                                helpdesc='SRLG Group ID' )

class SrlgSrlgnameGidCmd( CliCommand.CliCommandClass ):
   syntax = 'srlg NAME GROUP_ID'
   noOrDefaultSyntax = syntax
   data = {
      'srlg' : 'Shared Risk Link Group mapping for traffic engineering',
      'NAME' : matcherSrlgName,
      'GROUP_ID' : matcherSrlgGroupId,
   }

   @staticmethod
   def handler( mode, args ):
      gid = args[ 'GROUP_ID' ]
      config = teConfiguration()
      # If a previous mapping with the same srlgId exist, delete that. Multiple
      # srlgNames mapping to the same srlgId is not allowed.
      for srlgNameKey, srlgMapEntry in config.srlgMap.iteritems():
         if srlgMapEntry.srlgId == gid:
            del config.srlgMap[ srlgNameKey ] 
            break
      srlgMapEntry = teConfig.srlgMap.newMember( args[ 'NAME' ], srlgGid( gid ) )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      del teConfiguration().srlgMap[ args[ 'NAME' ] ]

RouterGlobalTeModelet.addCommandClass( SrlgSrlgnameGidCmd )

#--------------------------------------------------------------------------------
# [ no | default ] router-id ipv4 ROUTER_ID
#--------------------------------------------------------------------------------
matcherRouterId = CliMatcher.KeywordMatcher( 'router-id',
      helpdesc='Configure the router ID for traffic engineering' )

class RouterIdIpv4RouteridCmd( CliCommand.CliCommandClass ):
   syntax = 'router-id ipv4 ROUTER_ID'
   noOrDefaultSyntax = 'router-id ipv4 ...'
   data = {
      'router-id' : matcherRouterId,
      'ipv4' : 'TE router ID in IPv4 address format',
      'ROUTER_ID' : IpAddrMatcher.IpAddrMatcher(
         helpdesc='Traffic engineering router ID in IP address format' ),
   }

   @staticmethod
   def handler( mode, args ):
      teConfiguration().routerId = args[ 'ROUTER_ID' ]

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      teConfiguration().routerId = teConfiguration().routerIdDefault

RouterGlobalTeModelet.addCommandClass( RouterIdIpv4RouteridCmd )

#--------------------------------------------------------------------------------
# router-id ipv6 ROUTER_ID
#--------------------------------------------------------------------------------
class RouterIdIpv6Routeridv6Cmd( CliCommand.CliCommandClass ):
   syntax = 'router-id ipv6 ROUTER_ID'
   noOrDefaultSyntax = 'router-id ipv6 ...'
   data = {
      'router-id' : matcherRouterId,
      'ipv6' : 'TE router ID in IPv6 address format',
      'ROUTER_ID' : Ip6AddrMatcher.Ip6AddrMatcher(
         helpdesc='Traffic engineering router ID in IPv6 address format' ),
   }

   @staticmethod
   def handler( mode, args ):
      teConfiguration().routerIdV6 = args[ 'ROUTER_ID' ]

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      teConfiguration().routerIdV6 = teConfiguration().routerIdV6Default

RouterGlobalTeModelet.addCommandClass( RouterIdIpv6Routeridv6Cmd )

#--------------------------------------------------------------------------------
# [ no | default ] traffic-engineering
#--------------------------------------------------------------------------------
matcherTrafficEngineering = CliMatcher.KeywordMatcher( 'traffic-engineering',
      helpdesc='Configure traffic-engineering' )

class TrafficEngineeringCmd( CliCommand.CliCommandClass ):
   syntax = 'traffic-engineering'
   noOrDefaultSyntax = syntax
   data = {
      'traffic-engineering' : matcherTrafficEngineering,
   }

   @staticmethod
   def handler( mode, args ):
      intfConfig = _getOrCreateIntfConfig( teConfiguration(), mode.intf.name )
      intfConfig.enabled = True
      _deleteIntfConfigIfAllAttributeHaveDefaults( teConfiguration(),
                                                   mode.intf.name )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      intfConfig = _getIntfConfig( mode.intf.name )
      if intfConfig is None:
         return
      intfConfig.enabled = intfConfig.enabledDefault
      _deleteIntfConfigIfAllAttributeHaveDefaults( teConfiguration(),
                                                   mode.intf.name )

modelet.addCommandClass( TrafficEngineeringCmd )

#--------------------------------------------------------------------------------
# [ no | default ] traffic-engineering bandwidth [ BW UNIT ]
#--------------------------------------------------------------------------------

class TrafficEngineeringBandwidthCmd( CliCommand.CliCommandClass ):
   syntax = 'traffic-engineering bandwidth [ BW UNIT ]'
   noOrDefaultSyntax = syntax
   data = {
      'traffic-engineering' : matcherTrafficEngineering,
      'bandwidth' : 'Maximum reservable bandwidth for traffic-engineering',
      'BW' : CliMatcher.IntegerMatcher( bandwidth.bandwidthMin,
         bandwidth.bandwidthMax,
         helpdesc='Maximum reservable bandwidth' ),
      'UNIT' : CliMatcher.EnumMatcher( {
         'gbps' : 'Maximum reservable bandwidth in Gbps',
         'mbps' : 'Maximum reservable bandwidth in Mbps',
         'percent' : 'Maximum reservable bandwidth as percentage of link bandwidth',
      } ),
   }

   @staticmethod
   def handler( mode, args ):
      intfConfig = _getOrCreateIntfConfig( teConfiguration(), mode.intf.name )
      if 'BW' in args:
         intfConfig.maxReservableBw = args[ 'BW' ]
         if args[ 'UNIT' ] == 'percent':
            intfConfig.maxReservableBwUnit = bwUnit.percent
         elif args[ 'UNIT' ] == 'mbps':
            intfConfig.maxReservableBwUnit = bwUnit.mbps
         elif args[ 'UNIT' ] == 'gbps':
            intfConfig.maxReservableBwUnit = bwUnit.gbps
         else:
            assert False, 'Unknown unit'
      else:
         # Just configuring 'traffic-engineering bandwidth' means bw = 75% of link bw
         intfConfig.maxReservableBw = IMPLIED_MAX_RESERVABLE_BW
         intfConfig.maxReservableBwUnit = bwUnit.percent
      _deleteIntfConfigIfAllAttributeHaveDefaults( teConfiguration(),
                                                   mode.intf.name )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      intfConfig = _getIntfConfig( mode.intf.name )
      if intfConfig is None:
         return

      intfConfig.maxReservableBwUnit = intfConfig.maxReservableBwUnitDefault
      intfConfig.maxReservableBw = intfConfig.maxReservableBwDefault

      _deleteIntfConfigIfAllAttributeHaveDefaults( teConfiguration(),
                                                   mode.intf.name )

modelet.addCommandClass( TrafficEngineeringBandwidthCmd )

#------------------------------------------------------------------------------------
# [ no | default ] traffic-engineering administrative-group ( GROUP | GROUP_LIST )
#------------------------------------------------------------------------------------
class IntfTrafficEngineeringAdminGroupCmd( CliCommand.CliCommandClass ):
   syntax = 'traffic-engineering administrative-group ( GROUP_HEX | GROUP_LIST )'
   noOrDefaultSyntax = 'traffic-engineering administrative-group ...'
   data = {
      'traffic-engineering' : matcherTrafficEngineering,
      'administrative-group' : 'Administrative group for traffic-engineering',
      'GROUP_HEX' : CliMatcher.PatternMatcher( pattern='0x[0-9a-fA-F]{1,8}',
         helpdesc='Administrative Group value in hexadecimal',
         helpname='<0x0-0xFFFFFFFF>' ),
      'GROUP_LIST' : MultiRangeMatcher( adminGroupRange, False,
                                        'Admininistrative Group Ids' ),
   }

   @staticmethod
   def handler( mode, args ):
      intfConfig = _getOrCreateIntfConfig( teConfiguration(), mode.intf.name )
      if 'GROUP_HEX' in args:
         intfConfig.adminGroup = int( args[ 'GROUP_HEX' ], 16 )
         intfConfig.adminGroupListRepr = False
      else:
         adminGroupList = args.get( 'GROUP_LIST' )
         adminGroupValue = rangeToValue( adminGroupList.values() )
         intfConfig.adminGroup = adminGroupValue
         intfConfig.adminGroupListRepr = True
      _deleteIntfConfigIfAllAttributeHaveDefaults( teConfiguration(),
                                                   mode.intf.name )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      intfConfig = _getIntfConfig( mode.intf.name )
      if intfConfig is None:
         return

      intfConfig.adminGroup = intfConfig.adminGroupDefault
      intfConfig.adminGroupListRepr = False
      _deleteIntfConfigIfAllAttributeHaveDefaults( teConfiguration(),
                                                   mode.intf.name )

modelet.addCommandClass( IntfTrafficEngineeringAdminGroupCmd )

#--------------------------------------------------------------------------------
# [ no | default ] traffic-engineering srlg ( NAME | GROUP_ID )
#--------------------------------------------------------------------------------
class TrafficEngineeringSrlgSrlgnameCmd( CliCommand.CliCommandClass ):
   syntax = 'traffic-engineering srlg ( NAME | GROUP_ID )'
   noOrDefaultSyntax = syntax
   data = {
      'traffic-engineering' : matcherTrafficEngineering,
      'srlg' : 'SRLG for traffic-engineering',
      'NAME' : matcherSrlgName,
      'GROUP_ID' : matcherSrlgGroupId,
   }

   @staticmethod
   def handler( mode, args ):
      intfConfig = _getOrCreateIntfConfig( teConfiguration(), mode.intf.name )
      if 'NAME' in args: # pylint: disable-msg=simplifiable-if-statement
         intfConfig.srlgNameList[ args[ 'NAME' ] ] = True
      else:
         intfConfig.srlgIdList[ srlgGid( args[ 'GROUP_ID' ] ) ] = True

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      intfConfig = _getIntfConfig( mode.intf.name )
      if intfConfig is None:
         return

      if 'NAME' in args:
         del intfConfig.srlgNameList[ args[ 'NAME' ] ]
      else:
         del intfConfig.srlgIdList[ args[ 'GROUP_ID' ] ]
      _deleteIntfConfigIfAllAttributeHaveDefaults( teConfiguration(),
                                                   mode.intf.name )

modelet.addCommandClass( TrafficEngineeringSrlgSrlgnameCmd )

#--------------------------------------------------------------------------------
# [ no | default ] traffic-engineering metric METRIC
#--------------------------------------------------------------------------------
class IntfTrafficEngineeringMetricCmd( CliCommand.CliCommandClass ):
   syntax = 'traffic-engineering metric METRIC'
   noOrDefaultSyntax = 'traffic-engineering metric ...'
   data = {
      'traffic-engineering' : matcherTrafficEngineering,
      'metric' : 'Metric for traffic-engineering',
      'METRIC' : CliMatcher.IntegerMatcher( metric.metricMin + 1, metric.metricMax,
         helpdesc='Value of the route metric' ),
   }

   @staticmethod
   def handler( mode, args ):
      intfConfig = _getOrCreateIntfConfig( teConfiguration(), mode.intf.name )
      intfConfig.metric = args[ 'METRIC' ]
      _deleteIntfConfigIfAllAttributeHaveDefaults( teConfiguration(),
                                                   mode.intf.name )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      intfConfig = _getIntfConfig( mode.intf.name )
      if intfConfig is None:
         return

      intfConfig.metric = intfConfig.metricDefault
      _deleteIntfConfigIfAllAttributeHaveDefaults( teConfiguration(),
                                                   mode.intf.name )

modelet.addCommandClass( IntfTrafficEngineeringMetricCmd )

def Plugin( entityManager ):
   global teConfig

   teConfig = ConfigMount.mount( entityManager, 'te/config',
                                 'TrafficEngineering::Config', 'w' )
   IntfCli.Intf.registerDependentClass( TeIntf, priority=20 )
