#!/usr/bin/env python
# Copyright (c) 2013 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

from eunuchs.in_h import IPPROTO_IP

import Cell
import CliParser
import CliCommand
import CliMatcher
import ConfigMount
import CliPlugin.AclCli as AclCli
import CliPlugin.AclCliRules as AclCliRules
from CliPlugin.EthIntfCli import EthPhyAutoIntfType
import CliPlugin.IntfCli as IntfCli
from CliPlugin.LagIntfCli import LagAutoIntfType
from CliPlugin.MirroringPolicyMapClassModeTapAgg import (
      getIntfOpAndList,
      guardPmapDot1qRemove,
      guardPmapRemoveDot1qRange,
      PolicyMapClassModeTapAgg,
)
import CliPlugin.PolicyMapCliLib as PolicyMapCliLib
import CliPlugin.PolicyMapModel as PolicyMapModel
from CliPlugin.PolicyMapModelImpl import (
      PolicyMapModelContainer,
      clearPolicyMapCounters,
      clearPolicyMapCheckpoint,
      registerCmapStatusFilter
)
import CliPlugin.TapAggPmapCliLib as TapAggPmapCliLib
from CliPlugin.TapAggIntfCli import (
      qinqIdentityTaggingGuard,
      matcherTapAggGroupName,
      guardMacAcls
)
import Intf.IntfRange
import LazyMount
import MultiRangeRule
import PolicyMap
import Tac
import Tracing

t0 = Tracing.trace0

aclConfig = None
tapAggHwStatus = None
tapAggHwConfig = None
tapAggIntfConfig = None
tapAggPmapConfig = None
tapAggStatusRequestDir = None
tapAggPmapStatus = None
aclStatus = None
policyOpChkr = None
tapAggPmapCheckpoint = None
rawAclCmds = None

UniqueId = Tac.Type( 'Ark::UniqueId' )
VlanIdTuple = Tac.Type( 'Bridging::VlanIdTuple' )

def _tapAggPmapSessionCheckpoint( mode ):
   tapAggPmapSessionCheckpoint = mode.session.sessionData(
                                          'TapAggPmapCli.sessionCheckpoint', None )
   if tapAggPmapSessionCheckpoint is None:
      tapAggPmapSessionCheckpoint = Tac.newInstance( 'TapAgg::PmapCheckpointStatus',
                                                    'tapagg-pmapcounter-checkpoint' )
      mode.session.sessionDataIs( 'TapAggPmapCli.sessionCheckpoint',
                                  tapAggPmapSessionCheckpoint )
   return tapAggPmapSessionCheckpoint

def guardPMapTapAgg( mode, token ):
   if not tapAggHwStatus.modeSupported:
      return CliParser.guardNotThisPlatform
   elif not tapAggHwStatus.pmapSupported:
      return CliParser.guardNotThisPlatform
   else:
      return None

def guardIp6Acls( mode, token ):
   if not aclStatus.dpIp6AclSupported:
      return CliParser.guardNotThisPlatform
   elif not tapAggHwStatus.pmapIpv6Supported:
      return CliParser.guardNotThisPlatform
   else:
      return None

#------------------------------------------------------------------------
#                         POLICY MAP CLI
#------------------------------------------------------------------------
class RemoveBytesExpr( CliCommand.CliExpression ):
   expression = 'remove dot1q outer REMOVE_VLANS'
   data = {
      'remove' : CliCommand.guardedKeyword( 'remove', helpdesc='Remove a header',
         guard=guardPmapDot1qRemove ),
      'dot1q' : 'Remove dot1q tag',
      'outer' : 'Remove outer tag',
      'REMOVE_VLANS' : CliCommand.Node(
         matcher=MultiRangeRule.MultiRangeMatcher(
            rangeFn=lambda: ( 1, tapAggHwStatus.dot1qRemoveMaxIndex ),
            noSingletons=False,
            helpdesc='Specify indices of vlan tags to be removed',
            value=lambda mode, grList: str( grList ) ),
         guard=guardPmapRemoveDot1qRange )
   }

   @staticmethod
   def adapter( mode, args, argsList ):
      if not args.pop( 'remove', None ):
         return
      args.pop( 'dot1q' )
      args.pop( 'outer' )
      args[ 'removeBytesActType' ] = (
         TapAggPmapCliLib.actStripHdrBytes,
         TapAggPmapCliLib.stripHdrAction( TapAggPmapCliLib.stripHdrDot1q,
                                          args.pop( 'REMOVE_VLANS' ) ) )

class AggGroupExpr( CliCommand.CliExpression ):
   expression = ( 'aggregation-group ( AGG_SINGLE_GROUP | '
                                      '{ AGG_GROUP_KW AGG_GROUPS } )' )
   data = {
      'aggregation-group' : 'list of groups to aggregate flow',
      'AGG_SINGLE_GROUP' : matcherTapAggGroupName,
      'AGG_GROUP_KW' : CliMatcher.KeywordMatcher( 'group',
         helpdesc='Set tap group for the interface' ),
      'AGG_GROUPS' : matcherTapAggGroupName,
   }

   @staticmethod
   def adapter( mode, args, argsList ):
      if not args.pop( 'aggregation-group', None ):
         return

      if args.pop( 'AGG_GROUP_KW', None ):
         args[ 'AGG_GROUP' ] = args.pop( 'AGG_GROUPS' )
      else:
         args[ 'AGG_GROUP' ] = [ args.pop( 'AGG_SINGLE_GROUP' ) ]

class AggIdTagExpr( CliCommand.CliExpression ):
   expression= 'id-tag AGG_PORT_ID [ AGG_INNER_PORT_KW AGG_INNER_PORT_ID ]'
   data = {
      'id-tag' : CliMatcher.KeywordMatcher( 'id-tag', helpdesc='port identity' ),
      'AGG_PORT_ID' : CliMatcher.IntegerMatcher( 1, 4094,
         helpdesc='Tap port id tag' ),
      'AGG_INNER_PORT_KW' : CliCommand.guardedKeyword( 'inner',
         helpdesc='inner port identity',
         guard=qinqIdentityTaggingGuard ),
      'AGG_INNER_PORT_ID' : CliMatcher.IntegerMatcher( 1, 4094,
         helpdesc='Inner tap port id tag' )
   }

   @staticmethod
   def adapter( mode, args, argsList ):
      if not args.pop( 'id-tag', None ):
         return
      args.pop( 'AGG_INNER_PORT_KW', None )
      args[ 'AGG_ID_TAG' ] = VlanIdTuple( args.pop( 'AGG_PORT_ID' ),
            args.pop( 'AGG_INNER_PORT_ID', 0 ) )

class AggIntfExpr( CliCommand.CliExpression ):
   expression= 'AGG_INTF_KW { AGG_ET_INTF | AGG_LAG_INTF }'
   data = {
      'AGG_INTF_KW' : CliMatcher.KeywordMatcher( 'interface',
         helpdesc='list of interfaces to aggregate flow' ),
      'AGG_ET_INTF' : CliCommand.singleNode(
         matcher=Intf.IntfRange.IntfRangeMatcher( noSingletons=False,
            explicitIntfTypes=( EthPhyAutoIntfType, ) ) ),
      'AGG_LAG_INTF' : CliCommand.singleNode(
         matcher=Intf.IntfRange.IntfRangeMatcher( noSingletons=False,
            explicitIntfTypes=( LagAutoIntfType, ) ) ),
   }

   @staticmethod
   def adapter( mode, args, argsList ):
      if not args.pop( 'AGG_INTF_KW', None ):
         return
      etIntfs = args.pop( 'AGG_ET_INTF', None )
      lagIntfs = args.pop( 'AGG_LAG_INTF', None )
      args[ 'AGG_INTF' ] = []
      if etIntfs:
         args[ 'AGG_INTF' ] += list( etIntfs.intfNames() )
      if lagIntfs:
         args[ 'AGG_INTF' ] += list( lagIntfs.intfNames() )

class PmapActTypeExpr( CliCommand.CliExpression ):
   expression = ( 'set ( AGG_GROUP | '
                      '( [ AGG_GROUP ] ID_TAG ) | '
                      '( [ AGG_GROUP ] AGG_INTF [ ID_TAG ] ) )' )
   data = {
      'set' : PolicyMapCliLib.matcherActionSet,
      'AGG_GROUP' : AggGroupExpr,
      'ID_TAG' : AggIdTagExpr,
      'AGG_INTF': AggIntfExpr
   }

   @staticmethod
   def adapter( mode, args, argsList ):
      if not args.pop( 'set', None ):
         return
      group = args.pop( 'AGG_GROUP', None )
      intf = args.pop( 'AGG_INTF', None )
      idTag = args.pop( 'AGG_ID_TAG', None )
      actions = []
      data = []
      if group or intf:
         aggGroupData = {}
         if group:
            aggGroupData[ 'group' ] = ( 'set', group )
         if intf:
            aggGroupData[ 'intf' ] = ( 'set', intf )
         actions.append( TapAggPmapCliLib.actSetAggregationGroup )
         data.append( aggGroupData )
      if idTag:
         actions.append( TapAggPmapCliLib.actSetIdentityTag )
         data.append( idTag )

      args[ 'rawRuleActType' ] = ( tuple( actions ), tuple( data ) )

#-------------------------------------------------------------------------------
# Raw match statements in policy map mode
#-------------------------------------------------------------------------------
class TapAggCfgCmdBinder( PolicyMapCliLib.PolicyCfgCmdBinder ):

   def __init__( self, status ):
      PolicyMapCliLib.PolicyCfgCmdBinder.__init__( self, status )
      cls = self.__class__

      ########################################################
      #                       IP filters
      ########################################################
      class IpAclCommand( CliCommand.CliCommandClass ):
         syntax = ( '[ SEQ_NUM ] ACTION [ VLAN_EXPR ] ip IP_FILTER [ log ] '
                                                '[ RAW_RULE ] [ REMOVE_BYTES ]' )
         noOrDefaultSyntax = ( 'ACTION [ VLAN_EXPR ] ip IP_FILTER [ log ] '
                                                '[ RAW_RULE ] [ REMOVE_BYTES ]' )
         data = {
            'SEQ_NUM': AclCli.seqnumMatcher,
            'ACTION': self.actionMatcher,
            'VLAN_EXPR': self.vlanExpression,
            'ip' : PolicyMapCliLib.matcherIpAccessGroupType,
            'IP_FILTER': self.ipFilterExpression,
            'log': AclCliRules.logMatcher,
            'RAW_RULE' : PmapActTypeExpr,
            'REMOVE_BYTES' : RemoveBytesExpr,
         }

         @staticmethod
         def adapter( mode, args, argsList ):
            cls.adaptRuleCommonArgs( mode, args )
            args[ 'mirrorSession' ] = None
            args[ 'rawRuleActType' ] = args.get( 'rawRuleActType' )
            args[ 'removeBytesActType' ] = args.get( 'removeBytesActType' )
            args[ 'protocol' ] = args.get( 'protocol', IPPROTO_IP )
            args.pop( 'ip' )
            cls._handleIpArgs( mode, args ) # pylint: disable-msg=protected-access

         @staticmethod
         def handler( mode, args ):
            if CliCommand.isNoOrDefaultCmd( args ):
               cls.adaptNoOrDefaultRuleCommonArgs( mode, args )

            cls.handleIpRule( mode, **args )

         noOrDefaultHandler = handler

      PolicyMapModeTapAgg.addCommandClass( IpAclCommand )

      ########################################################
      #                    IPv6 filters
      ########################################################
      class Ip6AclCommand( CliCommand.CliCommandClass ):
         syntax = ( '[ SEQ_NUM ] ACTION [ VLAN_EXPR ] ipv6 IP_FILTER [ log ] '
                                                   '[ RAW_RULE ] [ REMOVE_BYTES ]' )
         noOrDefaultSyntax = ( 'ACTION [ VLAN_EXPR ] ipv6 IP_FILTER [ log ] '
                                                   '[ RAW_RULE ] [ REMOVE_BYTES ]' )
         data = {
            'SEQ_NUM': AclCli.seqnumMatcher,
            'ACTION': self.actionMatcher,
            'VLAN_EXPR': self.vlanExpression,
            'ipv6' : CliCommand.guardedKeyword( 'ipv6',
               helpdesc='Specify Ipv6 Access-groups',
               guard=guardIp6Acls ),
            'IP_FILTER': self.ip6FilterExpression,
            'log': AclCliRules.logMatcher,
            'RAW_RULE' : PmapActTypeExpr,
            'REMOVE_BYTES' : RemoveBytesExpr,
         }

         @staticmethod
         def adapter( mode, args, argsList ):
            cls.adaptRuleCommonArgs( mode, args )
            args[ 'mirrorSession' ] = None
            args[ 'rawRuleActType' ] = args.get( 'rawRuleActType' )
            args[ 'removeBytesActType' ] = args.get( 'removeBytesActType' )
            args[ 'protocol' ] = args.get( 'protocol', IPPROTO_IP )
            args.pop( 'ipv6' )
            self._handleIp6Args( mode, args ) # pylint: disable-msg=protected-access

         @staticmethod
         def handler( mode, args ):
            if CliCommand.isNoOrDefaultCmd( args ):
               cls.adaptNoOrDefaultRuleCommonArgs( mode, args )

            cls.handleIp6Rule( mode, **args )

         noOrDefaultHandler = handler

      PolicyMapModeTapAgg.addCommandClass( Ip6AclCommand )

      ########################################################
      #                    Mac filters
      ########################################################
      class MacAclCommand( CliCommand.CliCommandClass ):
         syntax = ( '[ SEQ_NUM ] ACTION [ VLAN_EXPR ] mac MAC_FILTER [ log ] '
                                                   '[ RAW_RULE ] [ REMOVE_BYTES ]' )
         noOrDefaultSyntax = ( 'ACTION [ VLAN_EXPR ] mac MAC_FILTER [ log ] '
                                                   '[ RAW_RULE ] [ REMOVE_BYTES ]' )
         data = {
            'SEQ_NUM': AclCli.seqnumMatcher,
            'ACTION': self.actionMatcher,
            'VLAN_EXPR': self.vlanExpression,
            'mac' : CliCommand.guardedKeyword( 'mac',
               helpdesc='Specify Mac Access-groups',
               guard=guardMacAcls ),
            'MAC_FILTER': self.macFilterExpression,
            'log': AclCliRules.logMatcher,
            'RAW_RULE' : PmapActTypeExpr,
            'REMOVE_BYTES' : RemoveBytesExpr,
         }
         @staticmethod
         def adapter( mode, args, argsList ):
            cls.adaptRuleCommonArgs( mode, args )
            args[ 'mirrorSession' ] = None
            args[ 'rawRuleActType' ] = args.get( 'rawRuleActType' )
            args[ 'removeBytesActType' ] = args.get( 'removeBytesActType' )
            args.pop( 'mac' )

         @staticmethod
         def handler( mode, args ):
            if CliCommand.isNoOrDefaultCmd( args ):
               cls.adaptNoOrDefaultRuleCommonArgs( mode, args )
            cls.handleMacRule( mode, **args )

         noOrDefaultHandler = handler

      PolicyMapModeTapAgg.addCommandClass( MacAclCommand )

      #-----------------------------------------------------------------
      #              no sequence-number
      #-----------------------------------------------------------------
      class NoSequenceCmd( CliCommand.CliCommandClass ):
         noOrDefaultSyntax = 'SEQUENCES'
         data = {
            'SEQUENCES': AclCliRules.seqRangeMatcher
         }
         noOrDefaultHandler = cls.handleNoSeq

      PolicyMapModeTapAgg.addCommandClass( NoSequenceCmd )

      #-----------------------------------------------------------------
      #              resequence
      #-----------------------------------------------------------------
      class ResequenceCmd( CliCommand.CliCommandClass ):
         syntax = 'resequence [ START [ INC ] ]'
         data = {
            'resequence' : PolicyMapCliLib.matcherResequence,
            'START' : self.startSeqMatcher,
            'INC' : self.incSeqMatcher,
         }

         @staticmethod
         def handler( mode, args ):
            start = args.get( 'START', self.defaultSeqStart )
            inc = args.get( 'INC', self.defaultSeqInc )
            cls.handleResequence( mode, start, inc )

      PolicyMapModeTapAgg.addCommandClass( ResequenceCmd )

def getPMapNameRuleTapAgg( mode ):
   return sorted( tapAggPmapConfig.pmapType.pmap.keys() )

def getCMapNameRuleTapAgg( mode ):
   return sorted( tapAggPmapConfig.cmapType.cmap.keys() )

#-------------------------------------------------------------------------------
# [no] service-policy type tapagg input <name>
#-------------------------------------------------------------------------------
def handleServicePolicy( mode, args ):
   no = CliCommand.isNoOrDefaultCmd( args )
   pmapName = args.get( 'PMAP_NAME' )

   if no is False:
      intfName = mode.intf.name
      if not intfName in tapAggHwConfig.tapPort:
         mode.addWarning(
            "Service policy will not be deployed until the interface"
            " is in tap mode and state is up." )

   PolicyMapCliLib.handleServicePolicy( mode, no, pmapName, tapAggPmapConfig,
                                        tapAggStatusRequestDir, tapAggPmapStatus,
                                        intfConfig=tapAggIntfConfig )

class PolicyMapModeTapAgg( PolicyMapCliLib.PolicyMapMode ):
   name = 'Policy Map TapAgg Configuration'
   modeParseTree = CliParser.ModeParseTree()

   def __init__( self, parent, session, context ):
      PolicyMapCliLib.PolicyMapMode.__init__( self, parent, session, 
                                               context, 'tapagg' )

   def switchToPmapClassMode( self, context ):
      childMode = self.childMode( PolicyMapClassModeTapAgg, context=context )
      self.session_.gotoChildMode( childMode )

   def defaultActionTypeAndData( self ):
      return ( tuple(), tuple() )

   def actionDataToClassAction( self, cmapName, actType, actData ):
      actions = tapAggPmapConfig.tapAggActions
      if actType == TapAggPmapCliLib.actSetAggregationGroup:
         groupData = actData.get( 'group' )
         intfData = actData.get( 'intf' )
         tapAggAct = actions.aggGroup.newMember( cmapName, UniqueId() )
         pmapGroup = tapAggPmapConfig.group.newMember( self.pmapName + '_' + \
                                                          cmapName )
         pmapGroup.rawAclMatch = True
         if groupData:
            groupOp, groupList = groupData
            assert groupOp == 'set'
            for group in groupList:
               pmapGroup.groupName[ group ] = True
               tapAggAct.aggGroup[ group ] = True
         if intfData:
            intfOp, intfList = getIntfOpAndList( intfData )
            assert intfOp == 'set'
            for intf in intfList:
               tapAggAct.aggIntf[ intf ] = True
      elif actType == TapAggPmapCliLib.actSetIdentityTag:
         tapAggAct = actions.idTag.newMember( cmapName, UniqueId(), actData )
      elif actType == TapAggPmapCliLib.actStripHdrBytes:
         tapAggAct = actions.stripHdrBytes.newMember( cmapName, UniqueId() )
         tapAggAct.stripHdrBytes = actData
      else:
         tapAggAct = None
      return tapAggAct

   def actionArgsToActionData( self, rawRuleActType=None, **kwargs ):
      actType, actData = self.defaultActionTypeAndData()

      if rawRuleActType:
         rawActType, rawActData = rawRuleActType
         if isinstance( rawActType, str ):
            actType += ( rawActType, )
            actData += ( rawActData, )
         else:
            actType += rawActType
            actData += rawActData

      if kwargs[ 'removeBytesActType' ]:
         removeBytesType, removeBytesData = kwargs.pop( 'removeBytesActType' )
         actType += ( removeBytesType, )
         actData += ( removeBytesData, )

      return ( actType, actData )

   @staticmethod
   def showPolicyMap( mode, args ):
      pmapName = args.get( 'PMAPNAME' )
      summary = 'summary' in args
      pmapAllModel = PolicyMapModel.PolicyMapAllModel()
      pmapAllModel.mapType = 'mapTapAgg'
      emapType = PolicyMap.mapTypeToEnum( 'tapagg' )
      intfConfig = tapAggIntfConfig

      # PolicyMapModelContainer requires both cli input config and merged config.
      # In case of tapAgg, merged config is the same as cli input config since
      # there is no configMergeSm.
      pmapsContainer = PolicyMapModelContainer( tapAggPmapConfig, intfConfig,
                                                intfConfig, tapAggPmapConfig,
                                                tapAggPmapStatus, aclConfig,
                                                None, None, emapType, None,
                                                pmapAllModel )
      checkpoints = ( tapAggPmapCheckpoint, _tapAggPmapSessionCheckpoint( mode ) )
      if pmapName is None:
         pmapsContainer.populateAll( mode, summary, checkpoints )
      else:
         pmapsContainer.populatePolicyMap( mode, pmapName, summary, checkpoints )
      return pmapAllModel


def gotoPolicyMapMode( mode, args ):
   pmapName = args.get( 'PMAPNAME' )
   mapType = 'tapagg'
   context = None
   t0( 'gotoPolicyMapMode of policy', pmapName )
   emapType = PolicyMap.mapTypeToEnum( mapType )
   if context is None:
      context = TapAggPmapCliLib.TapAggPolicyMapContext( tapAggPmapConfig,
                                                   tapAggStatusRequestDir,
                                                   tapAggPmapStatus, pmapName,
                                                   emapType )

   if pmapName in tapAggPmapConfig.pmapType.pmap:
      context.copyEditPmap()
   else:
      context.newEditPmap()

   mode.pmapContext = context
   childMode = mode.childMode( PolicyMapModeTapAgg, context=context )
   mode.session_.gotoChildMode( childMode )

def clearPolicyMapCheckpointStatus( mode, pmapName ):
   clearPolicyMapCheckpoint( pmapName, _tapAggPmapSessionCheckpoint( mode ) )
   clearPolicyMapCheckpoint( pmapName, tapAggPmapCheckpoint )

def deletePolicyMap( mode, args ):
   pmapName = args.get( 'PMAPNAME' )
   mapType = 'tapagg'
   lastMode = mode.session_.modeOfLastPrompt()
   if isinstance( lastMode, PolicyMapModeTapAgg ):
      if pmapName == lastMode.pmapName:
         lastMode.pmapContext = None
   if pmapName not in tapAggPmapConfig.pmapType.pmap:
      return

   for className in \
          tapAggPmapConfig.pmapType.pmap[ pmapName ].currCfg.classAction.keys():
      groupKey = pmapName + '_' + className
      if groupKey in tapAggPmapConfig.group.keys():
         del tapAggPmapConfig.group[ groupKey ]

   clearPolicyMapCheckpointStatus( mode, pmapName )

   emapType = PolicyMap.mapTypeToEnum( mapType )
   dummyContext = TapAggPmapCliLib.TapAggPolicyMapContext( tapAggPmapConfig, 
                                                tapAggStatusRequestDir,
                                                tapAggPmapStatus, pmapName, 
                                                emapType )
   dummyContext.delPmap( mode, pmapName )

#-------------------------------------------------------------------------------
# [ no | default ] [ SEQ_NUM ] class CLASS
#-------------------------------------------------------------------------------
class SetClassCmd( CliCommand.CliCommandClass ):
   syntax = '[ SEQ_NUM ] class CLASS'
   noOrDefaultSyntax = syntax
   data = {
      'SEQ_NUM': PolicyMapCliLib.matcherSeqNum,
      'class': PolicyMapCliLib.matcherClass,
      'CLASS': CliMatcher.DynamicNameMatcher( getCMapNameRuleTapAgg,
         helpdesc='Class Map Name' )
   }

   @staticmethod
   def handler( mode, args ):
      mode.setClass( False, args.get( 'SEQ_NUM' ), args[ 'CLASS' ] )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      mode.setClass( True, args.get( 'SEQ_NUM' ), args[ 'CLASS' ] )

PolicyMapModeTapAgg.addCommandClass( SetClassCmd )

#-------------------------------------------------------------------------------
# Support for default interface command.
#-------------------------------------------------------------------------------
class TapAggPmapIntfJanitor( IntfCli.IntfDependentBase ):
   def setDefault( self ):
      t0( "TapAggPmapIntfJanitor: interface", self.intf_.name, "going away" )
      if self.intf_.name in tapAggIntfConfig.intf:
         del tapAggIntfConfig.intf[ self.intf_.name ]

def clearPolicyMapCountersAll( mode, ckptStatus ):
   for policyName in tapAggPmapConfig.pmapType.pmap:
      clearOnePolicyMapCounters( mode, policyName, ckptStatus )

def clearOnePolicyMapCounters( mode, policyName, ckptStatus ):
   pmapConfig = PolicyMapCliLib.getPmapSubConfig( tapAggPmapConfig, policyName )
   pmapStatuses = [ status.status.get( policyName ) for status in
                    tapAggPmapStatus.itervalues() ]
   pmapStatuses = filter( None, pmapStatuses )
   pmapStatus = pmapStatuses[ 0 ]

   if not pmapConfig or not pmapStatus:
      return

   clearPolicyMapCounters( mode, policyName, tapAggPmapConfig, tapAggPmapConfig,
                           tapAggPmapStatus, ckptStatus )

def clearTapAggPMapCounters( mode, args ):
   policyName = args.get( 'PMAPNAME' )
   session = args.get( 'session' )

   if session:
      ckptStatus = _tapAggPmapSessionCheckpoint( mode )
      logAction = False
   else:
      ckptStatus = tapAggPmapCheckpoint
      logAction = True

   if policyName:
      if logAction:
         Intf.Log.logClearCounters( "tapagg policy", "policy map %s" % policyName )
      clearOnePolicyMapCounters( mode, policyName, ckptStatus )
   else:
      if logAction:
         Intf.Log.logClearCounters( "tapagg policy", "policy map all" )
      clearPolicyMapCountersAll( mode, ckptStatus )

def Plugin( entityManager ):
   global aclConfig, tapAggHwStatus, tapAggHwConfig
   global tapAggPmapConfig, tapAggStatusRequestDir, tapAggPmapStatus, aclStatus
   global policyOpChkr, tapAggPmapCheckpoint, tapAggIntfConfig
   global rawAclCmds

   #
   # Acl change status callback that returns the impact of an Acl change on
   # routing policy.
   def isPolicyHwStatusOk( mode, aclName, aclType, aclUpdateType ):
      rc = True
      err = {}
      err[ 'error' ] = ''
      if not mode.session_.startupConfig() and AclCli.allHwStatusEnabled():
         rc, result = policyOpChkr.verify( 'acl', aclName )
         if not rc:
            err[ 'error' ] = ( result.error if result else 
                               'unknown TagAgg policy status' )
         else:
            if result and result.warning:
               err[ 'warning' ] = "Warning: ACL %s %s" % ( aclName, result.warning )
      return rc, err

   tapAggCellRootNode = 'cell/%d/tapagg' % Cell.cellId()
   tapAggStatusPath = tapAggCellRootNode + '/pmapstatus'
   tapAggPmapStatus = LazyMount.mount( entityManager, tapAggStatusPath, 
                                       "Tac::Dir", 'ri' )

   aclConfig = ConfigMount.mount( entityManager, "acl/config/cli",
                                  "Acl::Input::Config", "w" )
   tapAggHwStatus = LazyMount.mount( entityManager, 'tapagg/hwstatus',
                                     'TapAgg::HwStatus', 'r' )
   tapAggHwConfig = LazyMount.mount( entityManager, 'tapagg/hwconfig',
                                     'TapAgg::HwConfig', 'r' )
   tapAggStatusRequestDir = LazyMount.mount( entityManager,
                                             'tapagg/pmapStatusRequestDir',
                                             'PolicyMap::PolicyMapStatusRequestDir',
                                             'w')
   tapAggIntfConfig = ConfigMount.mount( entityManager, 'tapagg/intfconfig',
                                         'PolicyMap::IntfConfig', 'w' )
   tapAggPmapConfig = ConfigMount.mount( entityManager, 'tapagg/pmapconfig',
                                         'TapAgg::PmapConfig', 'w' )

   aclStatus = LazyMount.mount( entityManager, "acl/status/all",
                                "Acl::Status", "r" )
   rawAclCmds = TapAggCfgCmdBinder( aclStatus )

   tapAggPmapCheckpoint = LazyMount.mount( entityManager,
                                           "tapagg/pmapcheckpoint",
                                           "TapAgg::PmapCheckpointStatus", "w" )

   # Intall a callback to verify impact of acl changes on policy
   policyOpChkr = PolicyMapCliLib.PolicyOpChkr( tapAggStatusRequestDir,
                                                tapAggPmapStatus )
   AclCli.registerAclConfigValidCallback( isPolicyHwStatusOk )
   IntfCli.Intf.registerDependentClass( TapAggPmapIntfJanitor )
   registerCmapStatusFilter( PolicyMap.tapaggMapType, ( lambda x: x ) )
