#!/usr/bin/env python
# Copyright (c) 2013 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

from CliModel import Dict
from CliModel import List
from CliModel import Model
from CliModel import Submodel
from CliModel import Str
from CliModel import Int
from CliModel import Enum
from CliModel import Bool
from IntfModels import Interface
from CliMode.Intf import IntfMode
import Arnet
import IntfModel
from TableOutput import TableFormatter, Headings, Format

switchportModes = ( 'access', 'trunk', 'dot1qTunnel',
                    'tap', 'tool', 'tap-tool', 'routed' )
intfLinkStatus = IntfModel.InterfaceStatus.interfaceStatus.values.\
    difference( [ 'connected' ] ).\
    union( [ 'tap', 'tool', 'non-tap', 'non-tool' ] )

def mapPropertyListKey( mpl ):
   '''We sort by policy name, group name, then interface.
   '''
   return ( mpl.policyName, mpl.groupName,
            Arnet.intfNameKey( mpl.mappedIntfName.stringValue ) )

class AclsAppliedPerType( Model ):
   aclsApplied = List( valueType=str, help="List of ACL names applied on a port" )

class TapAggProperties( Model ):
   class TapProperty( Model ):
      configuredMode = Enum( values=switchportModes,
                             help="The configured switchport mode of an interface" )
      linkStatus = Enum( values=intfLinkStatus,
                         help="The link status of an interface" )
      portIdentity = Int( help="The port identity of a tap port" )
      innerPortIdentity = Int( help="The inner port identity of a tap port" )
      allowedVlans = Str( help="The set of allowed vlans on a tap port "
                          "in the range 1-4094" )
      nativeVlan = Int( help="The native vlan associated with a tap port" )
      truncationSize = Int( help="The size to which packet is to be truncated,"
                            " in bytes" )

      class Details( Model ):
         # Now there could be multiple default groups. keeping default group for
         # backward compatibility reasons
         defaultGroup = Str( help="The default group associated with a tap port" )
         #list for group and interfaces added
         groups = List( valueType=str,
                        help="List of default groups for the tap port",
                        optional=True)
         toolPorts = List( valueType=Interface,
                        help="List of tool ports for the tap port",
                        optional=True)
         aclsAppliedPerType = Dict( valueType=AclsAppliedPerType,
                                     help="The list of ACLs applied on a port keyed "
                                           "by ACL type" )
         vxlanStrip = Bool( help="Indicates whether VXLAN strip is enabled on a "
                                "tap port" )
         mplsPop = Bool( help="Indicates whether MPLS pop is enabled on a tap port" )

      details = Submodel( valueType=Details, help="Detailed tap properties",
                          optional=True )
         
   class ToolProperty( Model ):
      configuredMode = Enum( values=switchportModes,
                             help="The configured switchport mode of an interface" )
      linkStatus = Enum( values=intfLinkStatus,
                         help="The link status of an interface" )
      identificationTag = Enum( values=( 'Off', 'dot1q', 'qinq' ),
                                help="The identity tag requirement for packets "
                               "recieved on a tool port" )
      allowedVlans = Str( help="The set of allowed vlans on a tool port "
                          "in the range 1-4094" )
      timestampMode = Enum( values=( 'NA', 'None', 'before-fcs', 'replace-fcs',
                            'header' ),
                            help="The timestamping mode on a packet recieved "
                            "on a tool port" )

      class Details( Model ):
         #list for groups added
         groups = List( valueType=str, help="List of groups of the tool port",
                        optional=True)
         removedVlans = Str( help="The set of 802.1Q tags to remove on a tool port",
                             optional=True)
         truncationSize = Int( help="The size to which packet is to be truncated,"
                               " in bytes" )
         aclsAppliedPerType = Dict( valueType=AclsAppliedPerType, 
                                    help="The dictionary of ACLs applied on a port"
                                         "keyed by ACL type",
                                    optional=True )
      details = Submodel( valueType=Details, help="Detailed tool properties",
                          optional=True )

   tapProperties = Dict( keyType=Interface, valueType=TapProperty, 
                         help="Dictionary of tap port "
                         "properties keyed by interface name" )
   toolProperties = Dict( keyType=Interface, valueType=ToolProperty, 
                          help="Dictionary of tool port "
                          "properties keyed by interface name" )
   _detail = Bool( help="Flag for details on tap-agg group interfaces" )

   def render( self ):
      tf = TableFormatter()
      width = ( tf.width_ - 25 )/2
      portFormat = Format( justify ="left", wrap=True,
                           maxWidth = 10, minWidth = 1 )
      configModeFormat = Format( justify ="left", wrap=True,
                                 maxWidth = 12, minWidth = 1 )
      statusFormat = Format( justify ="left", wrap=True,
                             maxWidth = 12, minWidth = 1 )
      aVlansFormat = Format( justify ="left", wrap=True,
                             maxWidth = width, minWidth = 1 )
      if self.tapProperties:
         portIdFormat = Format( justify ="left", wrap=True,
                                maxWidth = 9, minWidth = 1 )
         nVlansFormat = Format( justify ="left", wrap=True,
                                 maxWidth = 7, minWidth = 1 )
         truncSizeFormat = Format( justify ="left", wrap=True,
                                             maxWidth = 11, minWidth = 1 )
         heading = ( 'Port',  'Configured Mode', 'Status', 'Port Identity',
                     'Allowed Vlans', 'Native Vlan', 'Truncation Size' )
         header = Headings( heading )
         header.doApplyHeaders( tf )
         tf.formatColumns( portFormat, configModeFormat, statusFormat, portIdFormat,
                           aVlansFormat, nVlansFormat, truncSizeFormat )
         tapKeys = Arnet.sortIntf( self.tapProperties )
         for tapKey in tapKeys:
            tapProperty = self.tapProperties[ tapKey ]
            portIdentity = tapProperty.portIdentity
            if tapProperty.innerPortIdentity:
               portIdentity = '%d,%d' % ( tapProperty.portIdentity,
                                          tapProperty.innerPortIdentity )
            tf.newRow( IntfMode.getShortname( tapKey ), 
                       tapProperty.configuredMode, 
                       tapProperty.linkStatus, portIdentity,
                       tapProperty.allowedVlans, tapProperty.nativeVlan,
                       tapProperty.truncationSize )
         print tf.output()
         if self._detail:
            print '\n\n'
            t = TableFormatter()
            width = t.width_ - 25
            width = width/2
            #max width of columns are restricted so that table doesn't split into two
            f1 = Format( justify ="left", wrap=True, maxWidth = 10, minWidth = 1)
            f2 = Format( justify ="left", wrap=True, maxWidth = 15, minWidth = 1)
            f3 = Format( justify ="left", wrap=True, maxWidth = width, minWidth = 1)
            headings = ( "Port", "ACLs Applied", "Groups", "Tool Ports",
                         "VXLAN Strip", "MPLS Pop" )
            th = Headings( headings )
            th.doApplyHeaders(t)

            for tapKey in tapKeys:
               tapPort = IntfMode.getShortname( tapKey )
               tapProperty = self.tapProperties[ tapKey ]
               aclsApplied = []
               for aclType in sorted( tapProperty.details.aclsAppliedPerType ):
                  aclsApplied.extend( tapProperty.details. \
                                         aclsAppliedPerType[ aclType ].aclsApplied )
               if aclsApplied:
                  aclsAppliedStr = ', '.join( aclsApplied )
               else:
                  aclsAppliedStr = '---'
               tapGroups = sorted( list(tapProperty.details.groups) )
               if tapGroups:
                  tapGroupsStr = ', '.join( tapGroups )
               else:
                  tapGroupsStr = '---'
               tapIntfs = [ IntfMode.getShortname( intf ) for \
                              intf in sorted( tapProperty.details.toolPorts ) ]
               if not tapIntfs:
                  tapIntfs.append( "None" )
               tapIntfsStr = tapIntfs[ 0 ]
               for intf in tapIntfs[ 1: ]:
                  tapIntfsStr += ', '
                  tapIntfsStr += intf
               vxlanStrip = tapProperty.details.vxlanStrip
               mplsPop = tapProperty.details.mplsPop
               t.newRow( tapPort, aclsAppliedStr, tapGroupsStr, tapIntfsStr,
                         vxlanStrip, mplsPop )
            t.formatColumns( f1, f2, f3, f3 )
            print t.output()

      elif self.toolProperties:
         idTagFormat = Format( justify ="left", wrap=True,
                               maxWidth = 9, minWidth = 1 )
         timeStampFormat = Format( justify ="left", wrap=True,
                                   maxWidth = 11, minWidth = 1 )
         heading = ( 'Port', 'Configured Mode', 'Status', 'Id Tag', 'Allowed Vlans',
                     'Timestamp Mode' )
         header = Headings( heading )
         header.doApplyHeaders( tf )
         tf.formatColumns( portFormat, configModeFormat, statusFormat,
                           idTagFormat, aVlansFormat, timeStampFormat )
         toolKeys = Arnet.sortIntf( self.toolProperties )
         for toolKey in toolKeys:
            toolProperty = self.toolProperties[ toolKey ]
            tf.newRow( IntfMode.getShortname( toolKey ),
                       toolProperty.configuredMode, 
                       toolProperty.linkStatus, 
                       toolProperty.identificationTag,
                       toolProperty.allowedVlans,
                       toolProperty.timestampMode )

         print tf.output()
         if self._detail:
            print '\n\n' 
            t = TableFormatter()
            width = t.width_ - 55
            #max width of columns are restricted so that table doesn't split into two
            f1 = Format( justify ="left", wrap=True, maxWidth = 10, minWidth = 1)
            f2 = Format( justify ="left", wrap=True, maxWidth = 15, minWidth = 1)
            f3 = Format( justify ="left", wrap=True, maxWidth = width, minWidth = 1)
            headings = ( 'Port', 'Truncation Size', 'Remove Outer 802.1Q',
                         'ACLs Applied', 'Groups' )
            th = Headings( headings )
            th.doApplyHeaders(t)

            for toolKey in toolKeys:
               toolPort = IntfMode.getShortname( toolKey )
               toolProperty = self.toolProperties[ toolKey ]
               aclsApplied = []
               for aclType in sorted( toolProperty.details.aclsAppliedPerType ):
                  aclsApplied.extend( toolProperty.details. \
                                         aclsAppliedPerType[ aclType ].aclsApplied )
               if aclsApplied:
                  aclsAppliedStr = ', '.join( aclsApplied )
               else:
                  aclsAppliedStr = '---'
               toolGroups = sorted( list(toolProperty.details.groups) )
               if toolGroups:
                  toolGroupsStr = ', '.join( toolGroups )
               else:
                  toolGroupsStr = '---'
               t.newRow( toolPort, toolProperty.details.truncationSize,
                     toolProperty.details.removedVlans,
                     aclsAppliedStr, toolGroupsStr )
            t.formatColumns( f1, f2, f2, f2, f3 )
            print t.output()
   
class MapProperty( Model ):
   mappedIntfName = Interface( help="Tap/Tool Port Name" )
   groupName = Str( help="Group Name" )
   policyName = Str( help="PolicyMap/ClassMap Name" )

class MapPropertyList( Model ):
   mapPropertyList = List( valueType=MapProperty,
                           help='List of tap/tool map properties' )

def renderMapping( headings, mapping ):
   t = TableFormatter()
   f1 = Format( justify="left", wrap=True, maxWidth=15, minWidth=1 )
   f2 = Format( justify="left", wrap=True, maxWidth=25, minWidth=1 )
   th = Headings( headings )
   th.doApplyHeaders(t)
   for port in Arnet.sortIntf( mapping ):
      mapPropertyList = sorted( mapping[ port ].mapPropertyList,
                                key=mapPropertyListKey )
      for mapProperty in mapPropertyList:
         policyName = mapProperty.policyName
         t.newRow( port, mapProperty.mappedIntfName.stringValue,
                   mapProperty.groupName, policyName )
         
   t.formatColumns( f1, f1, f1, f2 )
   print t.output()

class TapToToolMap( Model ):
   tapPortsMap = Dict( keyType=Interface, valueType=MapPropertyList,
                       help="A mapping from tap to tool ports" )
   def render( self ):
      headings = ( 'Tap Port', 'Tool Ports', 'Groups', 'PolicyMap/ClassMap' )
      renderMapping( headings, self.tapPortsMap )

class ToolToTapMap( Model ):
   toolPortsMap = Dict( keyType=Interface, valueType=MapPropertyList,
                        help="A mapping from tool to tap ports" )
   def render( self ):
      headings = ( 'Tool Port', 'Tap Ports', 'Via Groups', 'Via PolicyMap/ClassMap' )
      renderMapping( headings, self.toolPortsMap )

class TapAggGroups( Model ):

   class TapAggGroup( Model ):
      """ Represents a TapAggGroup """

      class PortStatus( Model ):
         statuses = List( valueType=str, help="List statuses of the interface. "
                                              "Active, Interface status not found, "
                                              "Link down, Link unknown, Lag member, "
                                              "Mirror destination" )
         policies = List( valueType=str, help="List policies by which"
                                              "the port is added to the group" )

      tapPortStatuses = Dict( keyType=Interface, valueType=PortStatus,
                              help="Mapping from interface name to a list of their"
                                   "statuses and configured policies" )
      toolPortStatuses = Dict( keyType=Interface, valueType=PortStatus,
                               help="Mapping from interface name to a list of their"
                                    "statuses" )
   _detail = Bool( help="Flag for details on tap-agg group interfaces" )
   groups = Dict( valueType=TapAggGroup, help="Tap-agg groups keyed by group name" )

   def _getActiveIntfs( self, intfStatusMap ):
      activePorts = []
      for intf in intfStatusMap:
         intfStatus = intfStatusMap.get( intf )
         if not intfStatus:
            continue
         for status in intfStatus.statuses:
            if status == "Active":
               activePorts.append( intf )
 
      if not activePorts:
         activePorts.append( "None" )
 
      return activePorts

   def _getInActiveIntfs( self, intfStatusMap ):
      inactivePorts = {}
      for intf in intfStatusMap:
         intfStatus = intfStatusMap.get( intf )
         if not intfStatus:
            continue
         for status in intfStatusMap[ intf ].statuses:
            if status == "Active":
               continue
 
            if status not in inactivePorts:
               inactivePorts[ status ] = []
 
            inactivePorts[ status ].append( intf )
 
      return inactivePorts

   def formatPolicy( self, intf, intfStatusMap ):
      # return a string including the shortname of the port
      # and, if there is, the policy that the port applies
      portInfo = IntfMode.getShortname( intf )
      intfStatus = intfStatusMap.get( intf )
      if not intfStatus:
         return portInfo
      if not intfStatus.policies:
         return portInfo
      if not intfStatus.policies[ 1: ] and intfStatus.policies[ 0 ] == '*':
         # if a tap port has only default setting and no configured policies
         return portInfo

      if intfStatus.policies[ 0 ] == '*':
         # if a tap port has both default setting and configured policies
         policyList = intfStatus.policies[ 1: ]
         policyList.sort()
         portInfo += ', ' + portInfo + ' (' + policyList[ 0 ]
      else:
         policyList = intfStatus.policies[ 0: ]
         policyList.sort()
         portInfo += ' (' + policyList[ 0 ]
      for policy in policyList[ 1: ]:
         portInfo += ', ' + policy
      portInfo += ')'
      return portInfo
 
   def formatOutput( self, inputList, portStatuses, fieldWidth, prefixFields,
                     ports=True ):
      # Print the list of ports. If the list is
      # too long to fit on one line, wrap it onto subsequent lines.
      if ports:
         inputList = Arnet.sortIntf( inputList )
      else:
         if not inputList:
            inputList = ['---']
         inputList.sort()
      prefixLine = ' ' * prefixFields
      firstLine = True
      inputLine = self.formatPolicy( inputList[ 0 ], portStatuses )
      outputLine = ''
      for intf in inputList[ 1: ]:
         inputLine += ', ' + self.formatPolicy( intf, portStatuses )
         while len( inputLine ) > fieldWidth:
            outputLine = inputLine[ :fieldWidth ]
            breakpoint = outputLine.rfind( ', ' ) + 2
            outputLine = inputLine[ :breakpoint ]
            inputLine = inputLine[ breakpoint: ]
            if not firstLine and prefixFields:
               print prefixLine,
            print outputLine
            firstLine = False
      if not firstLine and prefixFields:
         print prefixLine,
      print inputLine
 
   def renderDetailGroup( self, groupName, group ):
      print
      print 'Group %s' % groupName
      print '---------------------------------------------------------------'
      self.renderPortStatus( group.tapPortStatuses, 'Source' )
      print
      self.renderPortStatus( group.toolPortStatuses, 'Destination' )
 
   def renderPortStatus( self, portStatuses, portType ):
      print
      print '%s Port:' % portType
      print '  Active:',
      self.formatOutput( self._getActiveIntfs( portStatuses ),
                         portStatuses, fieldWidth=70, prefixFields=9 )
      print
      self.printInactivePorts( self._getInActiveIntfs( portStatuses ),
                               portStatuses )
 
   def printInactivePorts( self, inactivePorts, portStatuses ):
      """ print inactive ports if present """
      if not inactivePorts:
         return
 
      prefixFields = 29
      detailedFormat = "%-5s%-25s%-s"
      briefFormat = "%-5s%-24s"
      print '  Configured, but inactive ports:'
      print detailedFormat % ( '', 'Reason inactive', 'Ports' )
      print detailedFormat % ( '', '--------------------', '----------' )
      for status in inactivePorts:
         print briefFormat % ( '', status ),
         self.formatOutput( inactivePorts[ status ], portStatuses,
                            fieldWidth=80-prefixFields,
                            prefixFields=prefixFields )
 
   def renderDetail( self ):
      groupKeys = sorted( self.groups.keys() )
      for groupName in groupKeys:
         group = self.groups.get( groupName )
         if not group:
            continue
         self.renderDetailGroup( groupName, group )
 
   def renderNoDetail( self ):
      t = TableFormatter()
      width = t.width_ - 15
      width = width / 2 - 1
      # Max width of columns are restricted so that table does not split into two
      f1 = Format( justify="left", wrap=True, maxWidth=15, minWidth=10 )
      # Remove padding to avoid the last column to be wrapped.
      f1.noPadLeftIs( True )
      f1.noTrailingSpaceIs( True )
      # If f2's columns does not exceed 20, fix the max to the min value to
      # allow columns to be wrapped.
      if width < 20:
         width = 20
      f2 = Format( justify="left", wrap=True, maxWidth=width, minWidth=20 )
      f2.noPadLeftIs( True )
      f2.noTrailingSpaceIs( True )
      headings = ( ("Group Name","l"), ("Tool Members","l"), ("Tap Members","l") )
      th = Headings( headings)
      th.doApplyHeaders(t)
      groupKeys = sorted( self.groups.keys() )
      for groupName in groupKeys:
         group = self.groups.get( groupName )
         if not group:
            continue
         tools = Arnet.sortIntf( self._getActiveIntfs( group.toolPortStatuses ) )
         taps = Arnet.sortIntf( self._getActiveIntfs( group.tapPortStatuses ) )
         tools = [ IntfMode.getShortname( intf ) for intf in tools ]
         toolsList = tools[ 0 ]
         for intf in tools[ 1: ]:
            toolsList += ', '
            toolsList += intf
         tapsList = self.formatPolicy( taps[ 0 ], group.tapPortStatuses )
         for intf in taps[ 1: ]:
            tapsList += ', '
            tapsList += self.formatPolicy( intf, group.tapPortStatuses )
         t.newRow( groupName, toolsList, tapsList )
      t.formatColumns( f1, f2, f2 )
      print t.output()

   def render( self ):
      if not self.groups:
         return
      if self._detail:
         self.renderDetail()
      else:
         self.renderNoDetail()


class CounterValue( Model ):
   """Wraps the value of a Hardware::Sand::CounterValue for TapAgg CLI."""
   packetCount = Int( help="The number of packets that have been counted" )
   byteCount = Int( help="The number of bytes that have been counted" )


class IntfCounterModel( Model ):
   """A mapping of generic interfaces to counter values.

   Rendering presents the interfaces as TAP ports only by hard-coded help text."""
   interfaces = Dict(
      keyType=Interface,
      valueType=CounterValue,
      help="A mapping from interfaces to counter values" )

   def render( self ):
      if not self.interfaces:
         print "No tap ports supporting default-forwarded counters are configured."
         return

      t = TableFormatter()
      colFormat = Format( justify="left", wrap=True, minWidth=1, padding=1 )
      colFormat.noPadLeftIs( True )
      th = Headings( ( ( "TAP Port", "l" ), ( "Packets", "l" ),
                       ( "Bytes", "l" ) ) )
      th.doApplyHeaders( t )
      for intf, counterValue in sorted( self.interfaces.iteritems() ):
         t.newRow( intf, counterValue.packetCount, counterValue.byteCount )
      t.formatColumns( colFormat, colFormat, colFormat )
      print t.output()
