#!/usr/bin/env python
# Copyright (c) 2012 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import Tac, CliParser, BasicCli, LazyMount, ConfigMount
import IntfCli, VlanCli, EthIntfCli
import MultiRangeRule
import Tracing
import TapAggModels
import CliExtensions
from CliMode.TapAgg import TapAggMode
from CliCommon import InvalidInputError, IncompleteCommandError
from FilteredDictView import FilteredDictView
import CliPlugin.ConfigConvert
import Intf.IntfRange
import CliCommand
import CliMatcher

__defaultTraceHandle__ = Tracing.Handle( 'TapAggIntfCli' )
t0 = Tracing.trace0

tapAggCliConfig = None
tapAggHwConfig = None
lagConfig = None
bridgingConfig = None
bridgingSwitchIntfConfigDir = None
ethPhyIntfConfigDir = None
aclIntfConfig = None
tapAggStatus = None
tapAggHwStatus = None
tapAggIntfConfig = None
tapAggPmapConfig = None
tapAggPmapStatus = None
intfStatusAll = None
hwSlice = None

# BUG162413: upper and lower limit of TapAggGlobalTruncationSize is 169. If egress
# truncation isn't supported just use ingress sizes
def globalTruncationRange( mode ):
   if tapAggHwStatus.egressTruncationSupported:
      return ( 169, 169 )
   else:
      return ( 100, BrInSicType.tapTruncationSizeMax )

FcsModeType = Tac.Type( 'TapAgg::TapAggFcsMode' )
sicType = Tac.Type( 'Bridging::Input::SwitchIntfConfig' )
TapAggModeType = Tac.Type( "TapAgg::TapAggMode" )
TimestampHeaderFormatType = Tac.Type( 'TapAgg::TapAggTimestampHeaderFormat' )
ToolIdentityMode = Tac.Type( 'Bridging::Input::ToolIdentityMode' )
VlanIdTuple = Tac.Type( 'Bridging::VlanIdTuple' )

def tapaggGuard( mode, token ):
   if ( len( tapAggHwStatus.modeSupported ) == 0  or 
        ( token == 'exclusive' and 
          'tapAggModeExclusive' not in tapAggHwStatus.modeSupported ) or 
        ( token == 'mixed' and 
          'tapAggModeMixed' not in tapAggHwStatus.modeSupported ) ):
      return CliParser.guardNotThisPlatform
   else:
      return None

def dot1qRemoveGuard( mode, token ):
   if not tapAggHwStatus.dot1qRemoveSupported:
      return CliParser.guardNotThisPlatform
   else:
      return None

def ingressTruncationGuard( mode, token ):
   if not tapAggHwStatus.ingressTruncationSupported:
      return CliParser.guardNotThisPlatform
   else:
      return None

def egressTruncationGuard( mode, token ):
   if not tapAggHwStatus.egressTruncationSupported:
      return CliParser.guardNotThisPlatform
   else:
      return None

def truncationSizePerIngressPortGuard( mode, token ):
   if not tapAggHwStatus.truncationSizePerIngressPortSupported:
      return CliParser.guardNotThisPlatform
   else:
      return None

def truncationNoSizePerIngressPortGuard( mode, token ):
   if not tapAggHwStatus.globalIngressTruncationSizeSupported:
      return CliParser.guardNotThisPlatform
   else:
      return None

def truncationSizePerEgressPortGuard( mode, token ):
   if not tapAggHwStatus.truncationSizePerEgressPortSupported:
      return CliParser.guardNotThisPlatform
   else:
      return None

def vxlanStripGuard( mode, token ):
   if not tapAggHwStatus.vxlanStripSupported:
      return CliParser.guardNotThisPlatform
   else:
      return None

def truncationNoSizePerEgressPortGuard( mode, token ):
   if not tapAggHwStatus.globalEgressTruncationSizeSupported:
      return CliParser.guardNotThisPlatform
   else:
      return None

def truncationGlobalSizeGuard( mode, token ):
   if truncationNoSizePerIngressPortGuard( mode, token ) is None or \
      truncationNoSizePerEgressPortGuard( mode, token ) is None:
      return None
   else:
      return CliParser.guardNotThisPlatform

def mplsPopGuard( mode, token ):
   if not tapAggHwStatus.mplsPopSupported:
      return CliParser.guardNotThisPlatform
   else:
      return None

def mplsPopToolGuard( mode, token ):
   if not tapAggHwStatus.mplsPopToolSupported:
      return CliParser.guardNotThisPlatform
   else:
      return None

def guardMacAcls( mode, token ):
   if tapAggHwStatus.pmapMacSupported:
      return None
   return CliParser.guardNotThisPlatform

def macAclMatchIpConfigurableGuard( mode, token ):
   if tapAggHwStatus.macAclMatchIpConfigurable:
      return None
   return CliParser.guardNotThisPlatform

def brVnTagStripGuard( mode, token ):
   if not tapAggHwStatus.brVnTagStripSupported:
      return CliParser.guardNotThisPlatform
   else:
      return None

def lldpReceiveConfigurableGuard( mode, token ):
   if tapAggHwStatus.lldpReceiveConfigurable:
      return None
   return CliParser.guardNotThisPlatform

def tcamProfileGuard( mode, token ):
   if tapAggHwStatus.tcamProfileSupported:
      return None
   return CliParser.guardNotThisPlatform

def timestampReplaceSmacGuard( mode, token ):
   if not tapAggHwStatus.timestampReplaceSmacSupported:
      return CliParser.guardNotThisPlatform
   else:
      return None

def fcsErrorModeGuard( mode, token ):
   if not tapAggHwStatus.fcsErrorModeSupported:
      return CliParser.guardNotThisPlatform
   else:
      return None

def fcsErrorModePassthroughGuard( mode, token ):
   if not tapAggHwStatus.fcsErrorModePassthroughSupported \
         or not tapAggHwStatus.fcsErrorModeSupported:
      return CliParser.guardNotThisPlatform
   else:
      return None

def fcsModeGuard( mode, token ):
   if not tapAggHwStatus.fcsModeSupported:
      return CliParser.guardNotThisPlatform
   else:
      return None

def qinqIdentityTaggingGuard( mode, token ):
   if tapAggHwStatus.qinqIdentityTaggingSupported:
      return None
   return CliParser.guardNotThisPlatform

def tapToolPortGuard( mode, token ):
   if not tapAggHwStatus.tapToolPortSupported:
      return CliParser.guardNotThisPlatform
   else:
      return tapaggGuard( mode, token )

def counterAccessGuard( mode, token ):
   if tapPortCountersGetHook.extensions() and \
         tapAggHwStatus.defaultForwardingCountersSupported and \
         tapAggHwStatus.trafficSteeringCountersSupported:
      return None
   else:
      return CliParser.guardNotThisPlatform

class TapAggModelet( CliParser.Modelet ):
   modeletParseTree = CliParser.ModeletParseTree()
   def __init__( self, mode ):
      CliParser.Modelet.__init__( self )
      
   @staticmethod
   def shouldAddModeletRule( mode ):
      # This is a bit hacky, but there's no better way to recognize physical Ethernet
      # interfaces than to look at their name.
      return ( ( mode.intf.name.startswith( 'Ethernet' ) or
                 mode.intf.name.startswith( 'Port-Channel' ) ) and
               not mode.intf.isSubIntf() )

IntfCli.IntfConfigMode.addModelet( TapAggModelet )

#----------------------------------------------------------------------------
# Use CLI hook to retrieve profiles (from AleTcam, presumably)
#----------------------------------------------------------------------------
tcamProfileGetHook = CliExtensions.CliHook()
def tapAggProfiles( mode ):
   profiles = {}
   for hook in tcamProfileGetHook.extensions():
      profiles.update( hook( mode ).items() )
   return profiles

#----------------------------------------------------------------------------
# Helper functions
#----------------------------------------------------------------------------

def isValidSource( groupName, srcIntf ):
   # Source is invalid if it is present as a destination in any other
   # tap-agg group or if it is present as a destination in the current group
   tpc = tapAggCliConfig
   for ( name, _tapAggGroup ) in tpc.group.items():
      if srcIntf in tpc.group[ name ].toolInterfaces:
         return( False, 'Duplicate' )
   #Source is invalid if its part of any Lag 
   if srcIntf in lagConfig.phyIntf and \
          lagConfig.phyIntf[ srcIntf ].lag:
      return( False, 'Interface mode conflict, %s is part of lag %s' % \
                 ( srcIntf, lagConfig.phyIntf[ srcIntf ].lag.name ) )

   return( True, '' )

def intfUp( intf ):
   return intfStatusAll[ intf ].operStatus == 'intfOperUp'
 
# ------------------------------------------------------------------------
# '[no|default] switchport mode [tap|tool|tap-tool]'
# ------------------------------------------------------------------------
switchportModeCmdHook = CliExtensions.CliHook()
def setSwitchportTapToolMode( mode, args ):
   sic = VlanCli.getSwitchIntfConfigEvenIfDisabled( mode )
   if not sic:
      return
   modeMapping = { 'tap': 'tap',
                   'tool': 'tool',
                   'tap-tool': 'tapTool' }
   chosenMode = None
   for availableMode in modeMapping:
      if availableMode in args:
         chosenMode = availableMode
         break
   assert chosenMode
   for hook in switchportModeCmdHook.extensions():
      if not hook( mode, mode.intf.name ):
         return
   sic.switchportMode = modeMapping[ chosenMode ]

# ------------------------------------------------------------------------
# ' tap aggregation'
# '[no|default] mode mixed|exclusive' 
# ------------------------------------------------------------------------

#-------------------------------------------------------------------------------
# tap-agg config mode. A new instance of this mode is created when the
# user enters "tap aggregation>" in config mode.
#-------------------------------------------------------------------------------
class TapAggConfigMode( TapAggMode, BasicCli.ConfigModeBase ):
   #----------------------------------------------------------------------------
   # Attributes required of every Mode class.
   #----------------------------------------------------------------------------

   name = 'Tap-agg group configuration'
   modeParseTree = CliParser.ModeParseTree()

   #----------------------------------------------------------------------------
   # Constructs a new TapAggMode instance.
   #----------------------------------------------------------------------------
   def __init__( self, parent, session ):
      self.modeKey = 'tap-agg'
      assert isinstance( parent, BasicCli.GlobalConfigMode ) 
      TapAggMode.__init__( self, None )
      BasicCli.ConfigModeBase.__init__( self, parent, session )

def setTapAggNoErrdisableIntf( mode, args ):
   tapAggCliConfig.noerrdisableIntf[ args[ 'ETHINTF' ].name ] = True

def noTapAggNoErrdisableIntf( mode, args ):
   del tapAggCliConfig.noerrdisableIntf[ args[ 'ETHINTF' ].name ]

def setMode( mode, args ):
   tapAggModeMapping = {
      'exclusive' : TapAggModeType.tapAggModeExclusive,
      'mixed' : TapAggModeType.tapAggModeMixed,
   }
   tapAggMode = TapAggModeType.tapAggModeNone
   for opt in tapAggModeMapping.iterkeys():
      if opt in args:
         tapAggMode = tapAggModeMapping[ opt ]
         break

   cardList = args.get( 'CARDNUMBERS' )
   isTapAggLc = args[ 'isTapAggLc' ] if 'isTapAggLc' in args else True

   tcamProfile = ''
   for opt in [ 'TAPAGGPROFILES', 'USERPROFILE' ]:
      if opt in args:
         tcamProfile = args[ opt ]
         break

   # modeChanged flag indicates whether to bump the mode version
   modeChanged = False

   if tapAggCliConfig.mode != tapAggMode:
      tapAggCliConfig.mode = tapAggMode
      modeChanged = True
      tapAggCliConfig.tcamProfile = ""
      tapAggCliConfig.sliceProfileConfig.clear()

   if tapAggMode == TapAggModeType.tapAggModeMixed:
      if cardList:
         for card in cardList:
            if isTapAggLc:
               if card not in tapAggCliConfig.sliceProfileConfig or \
                  tapAggCliConfig.sliceProfileConfig[ card ] != tcamProfile:
                  modeChanged = True
                  tapAggCliConfig.sliceProfileConfig[ card ] = tcamProfile
            elif card in tapAggCliConfig.sliceProfileConfig:
               modeChanged = True
               del tapAggCliConfig.sliceProfileConfig[ card ]
      # set mode to None if all linecard are removed out of TapAgg
      if len( tapAggCliConfig.sliceProfileConfig ) == 0:
         tapAggCliConfig.mode = TapAggModeType.tapAggModeNone
   elif tapAggMode == TapAggModeType.tapAggModeExclusive:
      if tapAggCliConfig.tcamProfile != tcamProfile:
         modeChanged = True
         tapAggCliConfig.tcamProfile = tcamProfile

   # Set mirroringRunnability after mode is updated
   tapAggCliConfig.mirroringRunnability = \
      ( tapAggCliConfig.mode != TapAggModeType.tapAggModeNone )

   # Now bump the mode version
   if modeChanged:
      tapAggCliConfig.modeVersion = tapAggCliConfig.modeVersion + 1

   restartWarn = ''
   if tapAggHwStatus.modeChangeRestartsForwardingAgent:
      restartWarn = 'will restart the forwarding agent(s) and '

   # warn if the selected profile doesn't exist
   if tcamProfile and tcamProfile not in tapAggProfiles( mode ):
      mode.addWarning( 'Configured TCAM profile ' + tcamProfile +
                       ' does not exist. ' \
                       'TAP aggregation will be enabled with the selected profile' \
                       ' once available.' )

   mode.addWarning( 'Changing modes ' + restartWarn +
         'may affect available functionality. '\
         'Unsupported configuration elements will be ignored.' )

def noMode( mode, args ):
   tapAggModeMapping = {
      'exclusive' : TapAggModeType.tapAggModeExclusive,
      'mixed' : TapAggModeType.tapAggModeMixed,
   }
   tapAggMode = None
   tapAggModeName = ''
   for opt in tapAggModeMapping.iterkeys():
      if opt in args:
         tapAggMode = tapAggModeMapping[ opt ]
         tapAggModeName = opt
         break

   cardList = args.get( 'CARDNUMBERS' )
   if tapAggMode and tapAggMode != tapAggCliConfig.mode:
      mode.addWarning( 'Mode ' + tapAggMode +' is not current active mode. '\
                          'Command is ignored.' )
      return
   if tapAggMode == TapAggModeType.tapAggModeMixed and cardList:
      args[ 'isTapAggLc' ] = False
   else:
      if tapAggMode:
         del args[ tapAggModeName ]
      if cardList:
         del args[ 'CARDNUMBERS' ]
   setMode( mode, args )

def gotoTapAggregationMode( mode, args ):
   childMode = mode.childMode( TapAggConfigMode )
   mode.session_.gotoChildMode( childMode )

def noTapAggregationConfig( mode, args ):
   noMode( mode, args )
   for intf in tapAggCliConfig.noerrdisableIntf.keys():
      del tapAggCliConfig.noerrdisableIntf[ intf ]
   tapAggCliConfig.trafficSteeringMacAclMatchIp = False
   tapAggCliConfig.vnTagStrip = False
   tapAggCliConfig.brTagStrip = False
   tapAggCliConfig.timestampReplaceSmac = False
   tapAggCliConfig.fcsErrorMode = FcsModeType.tapAggFcsErrorDefault
   tapAggCliConfig.lldpReceive = False

def vnTagStripConfig( mode, args ):
   tapAggCliConfig.vnTagStrip = True

def noVnTagStripConfig( mode, args ):
   tapAggCliConfig.vnTagStrip = False

def brTagStripConfig( mode, args ):
   tapAggCliConfig.brTagStrip = True

def noBrTagStripConfig( mode, args ):
   tapAggCliConfig.brTagStrip = False

def lldpReceiveConfig( mode, no ):
   tapAggCliConfig.lldpReceive = not no

def setTrafficSteeringMacAclMatchIp( mode, args ):
   tapAggCliConfig.trafficSteeringMacAclMatchIp = True

def noTrafficSteeringMacAclMatchIp( mode, args ):
   tapAggCliConfig.trafficSteeringMacAclMatchIp = False

def setGlobalTruncationSize( mode, truncationSize ):
   tapAggCliConfig.globalTruncationSize = truncationSize

def setDefaultGlobalTruncationSize( mode, args ):
   setGlobalTruncationSize( mode, 0 ) # 0 means platform default

def getGlobalTruncationSize():
   configuredSize = tapAggCliConfig.globalTruncationSize
   if configuredSize != 0:
      return configuredSize
   else:
      return tapAggHwStatus.globalTruncationSizeDefault

def timestampReplaceSmacConfig( mode, args ):
   tapAggCliConfig.timestampReplaceSmac = True
   if tapAggCliConfig.timestampHeaderFormat != \
         TimestampHeaderFormatType.tapAggTimestampDefault:
      mode.addWarning(
         'More than one timestamp format configured. '
         'Replace source-mac takes precedence.' )

def setDefaultTimestampReplaceSmac( mode, args ):
   tapAggCliConfig.timestampReplaceSmac = False

def setTimestampHeaderFormat( mode, args ):
   formatMapping = {
      '48-bit' : TimestampHeaderFormatType.tapAggTimestamp48bit,
      '64-bit' : TimestampHeaderFormatType.tapAggTimestamp64bit,
   }
   headerFormat = None
   for formatOption in formatMapping.iterkeys():
      if formatOption in args:
         headerFormat = args[ formatOption ]
         break
   assert headerFormat in formatMapping
   tapAggCliConfig.timestampHeaderFormat = formatMapping[ headerFormat ]
   if tapAggCliConfig.timestampReplaceSmac: # replace-smac configured
      mode.addWarning(
         'More than one timestamp format configured. '
         'Replace source-mac takes precedence.' )

def setDefaultTimestampHeaderFormat( mode, args ):
   tapAggCliConfig.timestampHeaderFormat = \
      TimestampHeaderFormatType.tapAggTimestampDefault

def setFcsErrorMode( mode, args ):
   modeMapping = {
      'correct' : FcsModeType.tapAggFcsErrorCorrect,
      'discard' : FcsModeType.tapAggFcsErrorDiscard,
      'pass-through' : FcsModeType.tapAggFcsErrorPassThrough,
   }
   fcsErrorMode = None
   for modeOption in modeMapping.iterkeys():
      if modeOption in args:
         fcsErrorMode = args[ modeOption ]
         break
   assert fcsErrorMode in modeMapping
   tapAggCliConfig.fcsErrorMode = modeMapping[ fcsErrorMode ]
   if tapAggCliConfig.fcsMode != FcsModeType.tapAggFcsErrorDefault:
      mode.addWarning( "'mac fcs' has already been configured and takes "
                       "precedence over 'mac fcs-error'." )

def noFcsErrorMode( mode, args ):
   tapAggCliConfig.fcsErrorMode = FcsModeType.tapAggFcsErrorDefault

def setFcsMode( mode, args ):
   modeMapping = {
      'append' : FcsModeType.tapAggFcsAppend,
   }
   fcsMode = args[ 'ACTION' ]
   assert fcsMode in modeMapping
   tapAggCliConfig.fcsMode = modeMapping[ fcsMode ]

   if tapAggCliConfig.fcsErrorMode != FcsModeType.tapAggFcsErrorDefault:
      mode.addWarning( "'mac fcs-error' has already been configured. 'mac fcs' "
                       "takes precedence." )

def noFcsMode( mode, args ):
   tapAggCliConfig.fcsMode = FcsModeType.tapAggFcsErrorDefault

# Note that tap-agg group names are silently truncated to 32 characters.
# This is consistent with VLAN names.

def getGroupList( mode ):
   result = set( tapAggStatus.group )
   for group in tapAggPmapConfig.group.values():
      result |= set( group.groupName )

   return result

matcherTapAggGroupName = CliMatcher.DynamicNameMatcher( getGroupList,
      'Group name', pattern=r'[A-Za-z0-9_:{}\[\]-]+' )

def filterInternal( token ):
   return token != sicType.tapGroupDefault

matcherTapAggGroupNameDynamic = CliMatcher.DynamicKeywordMatcher(
      lambda mode: FilteredDictView( tapAggStatus.group,
                                     filterInternal ),
      emptyTokenCompletion=[ CliParser.Completion( 'WORD',
                        'TapAgg group name', literal=False ) ] )

# ------------------------------------------------------------------------
# switchport tap default group <comma-saparated-group-list>
# [no|default] switchport tap default ( group |  ( group <name> group <name> ) )
# "no trailing garbage as it is a list but list is optional"
# ------------------------------------------------------------------------
def addOrRemoveTapGroups( mode, args, cfg ):
   groupList = []
   groupOp = 'set'
   if 'GROUP_NAME' in args:
      groupList = args[ 'GROUP_NAME' ]
      groupOp = 'remove' if CliCommand.isNoOrDefaultCmd( args ) else 'add'

   for group in groupList:
      if groupOp == 'add' or ( groupOp == 'set' and group ):
         cfg.tapGroup[ group ] = True
      elif groupOp == 'remove':
         del cfg.tapGroup[ group ]
   if groupOp == 'set':
      for group in cfg.tapGroup.keys():
         if group not in groupList:
            del cfg.tapGroup[ group ]

def handleTapGroups( mode, args ):
   t0( "handleTapGroups" )
   sic = VlanCli.getSwitchIntfConfigEvenIfDisabled( mode )
   if not sic:
      return
   addOrRemoveTapGroups( mode, args, sic )

# ------------------------------------------------------------------------
# switchport tap default interface add|remove <interface-list>
# [no|default] switchport tap default interface <trailing-garbage>
# ------------------------------------------------------------------------
def addOrRemoveTapRawIntf( mode, args, cfg ):
   intfOp = 'remove' if CliCommand.isNoOrDefaultCmd( args ) else 'add'
   configuredIntfs = None
   acceptedIntfTypes = { 'ETH_INTF', 'PHY_AUTO_INTF', 'LAG_INTF' }
   for intfType in acceptedIntfTypes:
      if intfType in args:
         configuredIntfs = args[ intfType ]
         break

   intfList = []
   if ( configuredIntfs ):
      if type( configuredIntfs ) is EthIntfCli.EthPhyIntf:
         # It's a singleton object, not a list of intfs
         intfList = [ configuredIntfs.name ]
      elif type( configuredIntfs ) is list:
         intfList = configuredIntfs
      else:
         assert type( configuredIntfs ) is Intf.IntfRange.IntfList
         intfList += list( configuredIntfs.intfNames() )

   if configuredIntfs is None:
      for intf in cfg.tapRawIntf:
         if intf not in intfList:
            del cfg.tapRawIntf[ intf ]
   else:
      for intf in intfList:
         if intf == mode.intf.name:
            mode.addError( "Cannot configure the port as its own default interface" )
            continue
         if intfOp == 'add':
            cfg.tapRawIntf[ intf ] = True
         elif intfOp == 'remove':
            del cfg.tapRawIntf[ intf ]

def handleTapRawIntf( mode, args ):
   t0( "handleTapRawIntf" )
   sic = VlanCli.getSwitchIntfConfigEvenIfDisabled( mode )
   if not sic:
      return

   addOrRemoveTapRawIntf( mode, args, sic )

# ------------------------------------------------------------------------
# switchport tap native vlan <vlan>
# [no|default] switchport tap native vlan <trailing-garbage>
# ------------------------------------------------------------------------

defaultTapNativeVlan = 1
defaultTapIdentificationVlan = 1

def handleTapNativeVlan( mode, args ):
   t0( "handleTapNativeVlan" )
   sic = VlanCli.getSwitchIntfConfigEvenIfDisabled( mode )
   if not sic:
      return
   if CliCommand.isNoOrDefaultCmd( args ):
      sic.tapNativeVlan = defaultTapNativeVlan
   else:
      sic.tapNativeVlan = args[ 'VLAN_ID' ].id

# ------------------------------------------------------------------------
# switchport tap identity <1-4094> [ inner <1-4094> ]
# [ no|default ] switchport tap identity <trailing garbage>
# ------------------------------------------------------------------------

def handleTapIdentificationVlan( mode, args ):
   t0( "handleTapIdentificationVlan" )
   sic = VlanCli.getSwitchIntfConfigEvenIfDisabled( mode )
   if not sic:
      return
   portId = args.get( 'PORT_ID', 0 )
   innerPortId = args.get( 'INNER_PORT_ID', 0 )
   sic.tapIdentificationVlan = VlanIdTuple( portId, innerPortId )

# ------------------------------------------------------------------------
# switchport tap encapsulation vxlan strip
# [ no|default ] switchport tap encapsulation vxlan strip
# ------------------------------------------------------------------------

def handleTapVxlanStrip( mode, args ):
   sic = VlanCli.getSwitchIntfConfigEvenIfDisabled( mode )
   if not sic:
      return
   tapVxlanStrip = not CliCommand.isNoOrDefaultCmd( args )
   if tapVxlanStrip and sic.tapMplsPop:
      mode.addError( 'Error: Must disable TAP Port MPLS Pop' )
      return
   sic.tapVxlanStrip = tapVxlanStrip

# ------------------------------------------------------------------------
# switchport tap mpls pop all
# [ no|default ] switchport tap mpls pop [ all ]
# ------------------------------------------------------------------------

def handleTapMplsPop( mode, args ):
   sic = VlanCli.getSwitchIntfConfigEvenIfDisabled( mode )
   if not sic:
      return
   tapMplsPop = not CliCommand.isNoOrDefaultCmd( args )
   if tapMplsPop and sic.tapVxlanStrip:
      mode.addError( 'Error: Must disable TAP Port VXLAN Strip' )
      return
   sic.tapMplsPop = tapMplsPop

# ------------------------------------------------------------------------
# switchport tool mpls pop all
# [ no|default ] switchport tool mpls pop [ all ]
# ------------------------------------------------------------------------

def handleToolMplsPop( mode, args ):
   sic = VlanCli.getSwitchIntfConfigEvenIfDisabled( mode )
   if not sic:
      return
   toolMplsPop = not CliCommand.isNoOrDefaultCmd( args )
   sic.toolMplsPop = toolMplsPop

# ------------------------------------------------------------------------
# switchport tool group set <group-list> #old format
# switchport tool group <name> group <name> ...  #new format
# [no|default] switchport tool group <name> group <name>
# ------------------------------------------------------------------------

def handleInvalidGroupOpAndList( groupOp, groupList ):
   invalidGroupNames = [ 'set', 'group', 'add', 'remove' ]
   if groupList == None:
      raise IncompleteCommandError()
   if groupOp == 'add' or groupOp == 'set':
      for groupName in invalidGroupNames:
         if groupName in groupList:
            errorMsg = 'Cannot name group using reserved keyword \'%s\'' \
                       % groupName
            raise InvalidInputError( ': ' + errorMsg )

def handleToolGroups( mode, args ):
   t0( "handleToolGroups" )
   groupList = []
   groupOp = 'set'
   if 'GROUP_NAME' in args:
      groupList = args[ 'GROUP_NAME' ]
      if CliCommand.isNoOrDefaultCmd( args ):
         groupOp = 'remove'
      elif 'set' not in args:
         groupOp = 'add'

   sic = VlanCli.getSwitchIntfConfigEvenIfDisabled( mode )
   if not sic:
      return
   handleInvalidGroupOpAndList( groupOp, groupList )
   for group in groupList:
      if groupOp == 'add' or ( groupOp == 'set' and group ):
         sic.toolGroup[ group ] = True
      elif groupOp == 'remove':
         del sic.toolGroup[ group ]
   if groupOp == 'set':
      for group in sic.toolGroup.keys():
         if group not in groupList:
            del sic.toolGroup[ group ]

# ------------------------------------------------------------------------
# switchport tool identity <dot1q|qinq>
# [no|default] switchport tool identity <trailing garbage>
# ------------------------------------------------------------------------

def handleIdentityTagging( mode, args ):
   t0( "handleIdentityTagging" )
   sic = VlanCli.getSwitchIntfConfigEvenIfDisabled( mode )
   if not sic:
      return
   identityModeMapping = { 'dot1q' : ToolIdentityMode.dot1q,
                           'qinq'  : ToolIdentityMode.qinq }
   chosenIdentityMode = None
   if CliCommand.isNoOrDefaultCmd( args ):
      chosenIdentityMode = ToolIdentityMode.none
   else:
      for identityModeOption in identityModeMapping:
         if identityModeOption in args:
            chosenIdentityMode = identityModeMapping[ identityModeOption ]
            break

   sic.toolIdentityTagging = chosenIdentityMode

# ------------------------------------------------------------------------
# switchport tool dot1q remove outer <range>
# [no|default] switchport tool dot1q remove outer <trailing garbage>
# ------------------------------------------------------------------------

def handleDot1qRemoveVlan( mode, args ):
   tag = str( args.get( 'DOT_1Q_TAG', '' ) )
   if tag:
      # args[ 'DOT_1Q_TAG' ] only accepts the following input: '1' and '1-2'
      # with '1' indicates outer VLAN id, and '2' inner VLAN id.
      if ( mode.session.guardsEnabled() ) and \
         not tapAggHwStatus.dot1qRemoveInnerOnlySupported and '1' not in tag:
         mode.addError( "Removing only inner VLAN is not supported "
                        "on this hardware platform" )
         return
   sic = VlanCli.getSwitchIntfConfigEvenIfDisabled( mode )
   sic.toolDot1qRemoveVlans = str( tag )

matcherDot1QTag = MultiRangeRule.MultiRangeMatcher(
   lambda: ( 1, tapAggHwStatus.dot1qRemoveMaxIndex ),
   False,
   helpdesc='Specify indices of vlan tags to be removed'
)

# ------------------------------------------------------------------------
# switchport tap truncation <size>
# switchport tap truncation
# [no|default] switchport tap truncation <trailing garbage>
# ------------------------------------------------------------------------

def handleTapTruncation( mode, args ):
   sic = VlanCli.getSwitchIntfConfigEvenIfDisabled( mode )
   if not sic:
      return
   tapTruncationSize = 0
   if not CliCommand.isNoOrDefaultCmd( args ):
      tapTruncationSize = args.get( 'SIZE', 1 )
   sic.tapTruncationSize = tapTruncationSize

BrInSicType = Tac.Type( "Bridging::Input::SwitchIntfConfig" )
matcherTapTruncationSize = CliMatcher.IntegerMatcher(
   100, BrInSicType.tapTruncationSizeMax,
   helpdesc='ingress packet truncation size in bytes' )

# ------------------------------------------------------------------------
# switchport tool truncation <size>
# switchport tool truncation
# [no|default] switchport tool truncation <trailing garbage>
# ------------------------------------------------------------------------

def handleToolTruncation( mode, args ):
   sic = VlanCli.getSwitchIntfConfigEvenIfDisabled( mode )
   if not sic:
      return
   toolTruncationSize = 0
   if not CliCommand.isNoOrDefaultCmd( args ):
      toolTruncationSize = args.get( 'SIZE', 1 )
   sic.toolTruncationSize = toolTruncationSize

matcherToolTruncationSize = CliMatcher.IntegerMatcher(
   160, 160,
   helpdesc='egress packet truncation size in bytes' )

#------------------------------------------------------------------------
# 'show interface [ <intfName> ] tap [ detail ]'
# 'show interface [ <intfName> ] tool [ detail ]'
# 'show tap aggregation group [ <group-name> ... ] [ detail ]'
# 'show tap aggregation group detail [ <group-name> ]'
# 'show tap aggregation counters default-forwarding'
# ------------------------------------------------------------------------

def doShowTapAggMode( mode, args ):
   intf = args.get( 'INTF' )
   mod = args.get( 'MOD' )
   detail = 'detail' in args
   tapAggModeShow = args.get( 'tap' ) or args[ 'tool' ]

   def getAllowedVlanStr( allowedVlans ):
      if allowedVlans == '1-4095':
         allowedVlans = 'All'
      elif allowedVlans == '':
         allowedVlans = 'None'
      return allowedVlans

   def getRemovedVlanStr( removedVlans ):
      return removedVlans if removedVlans else None  

   model = TapAggModels.TapAggProperties( _detail=bool( detail ) )

   sicdir = bridgingSwitchIntfConfigDir
   ethIntfs = IntfCli.Intf.getAll( mode, intf, mod, EthIntfCli.EthIntf )
   if ethIntfs is None:
      return None

   for intfName in ethIntfs:
      switchIntfConfig = sicdir.switchIntfConfig.get( intfName.name )
      if switchIntfConfig is None:
         continue

      switchportMode = switchIntfConfig.switchportMode
      configuredMode = switchportMode
 
      if not switchIntfConfig.enabled:
         switchportMode = 'routed'
         configuredMode = 'routed'

      if intf is None and switchportMode != tapAggModeShow:
         if not( switchportMode == 'tapTool' and \
               ( tapAggModeShow == 'tap' or tapAggModeShow == 'tool' ) ):
            continue


      linkStatus = 'non-%s' % tapAggModeShow
      intfStatus = intfName.linkStatus()
      if intfStatus and intfStatus not in [ 'connected', 'up' ]:
         linkStatus = intfStatus
      elif configuredMode == tapAggModeShow or configuredMode == 'tapTool':
         linkStatus = tapAggModeShow

      if( configuredMode == 'tapTool' and \
            ( tapAggModeShow == 'tap' or tapAggModeShow == 'tool' ) ):
         configuredMode = 'tap-tool'

      if tapAggModeShow == 'tap':
         nativeVlan = switchIntfConfig.tapNativeVlan
         idVlan = switchIntfConfig.tapIdentificationVlan

         truncationSize = switchIntfConfig.tapTruncationSize
         if truncationSize != 0:
            if not tapAggHwStatus.truncationSizePerIngressPortSupported:
               truncationSize = getGlobalTruncationSize()
            elif truncationSize == sicType.tapTruncationUseGlobalSize:
               # If per-port size is supported but configured to 1, this means to use
               # the global truncation size. We need this for backward compatibility
               # reasons.
               truncationSize = getGlobalTruncationSize()

         allowedVlans = getAllowedVlanStr( switchIntfConfig.tapAllowedVlans )
         tapProperty = TapAggModels.TapAggProperties.TapProperty(
            configuredMode=configuredMode,
            linkStatus=linkStatus,
            portIdentity=idVlan.outerVid,
            innerPortIdentity=idVlan.innerVid,
            allowedVlans=allowedVlans,
            nativeVlan=nativeVlan,
            truncationSize=truncationSize
            )
         if detail:
            tapGroup = []
            intfList = []
            defaultGroup = '---'
            if switchIntfConfig.tapGroup.keys():
               tapGroup = switchIntfConfig.tapGroup.keys()
            if switchIntfConfig.tapRawIntf.keys():
               intfList = switchIntfConfig.tapRawIntf.keys()
            if len( tapGroup ) != 0:
               defaultGroup = tapGroup[ 0 ]
            vxlanStrip = switchIntfConfig.tapVxlanStrip
            mplsPop = switchIntfConfig.tapMplsPop
            tapProperty.details = TapAggModels.TapAggProperties.TapProperty. \
                Details( defaultGroup=defaultGroup, groups=tapGroup,
                         toolPorts=intfList,
                         aclsAppliedPerType=getAppliedAcls( 'in', intfName.name ),
                         vxlanStrip=vxlanStrip,
                         mplsPop=mplsPop
                         )
         model.tapProperties[ intfName.name ] = tapProperty

      elif tapAggModeShow == 'tool':
         idTag = 'Off'
         if switchIntfConfig.toolIdentityTagging != ToolIdentityMode.none:
            idTag = switchIntfConfig.toolIdentityTagging
         allowedVlans = getAllowedVlanStr( switchIntfConfig.toolAllowedVlans )
         if intfName.name.startswith( 'Port-Channel' ):
            timestampMode = 'NA'
         else:
            ethPhyIntfConfig = ethPhyIntfConfigDir.get( intfName.name )
            timestamp = Tac.Type( 'Interface::EthTimestampMode' )
            timestampMode = 'None'
            if ethPhyIntfConfig.timestampMode == timestamp.timestampModeBeforeFcs:
               timestampMode = 'before-fcs'
            elif ethPhyIntfConfig.timestampMode == \
                   timestamp.timestampModeReplaceFcs:
               timestampMode = 'replace-fcs'
            elif ethPhyIntfConfig.timestampMode == timestamp.timestampModeHeader:
               timestampMode = 'header'
            
         toolProperty = TapAggModels.TapAggProperties.ToolProperty(
            configuredMode=configuredMode,
            linkStatus=linkStatus,
            identificationTag=idTag,
            allowedVlans=allowedVlans,
            timestampMode=timestampMode )
         if detail:
            removedVlans = getRemovedVlanStr( switchIntfConfig.toolDot1qRemoveVlans )
 
            truncationSize = switchIntfConfig.toolTruncationSize
            if truncationSize != 0 and \
               not tapAggHwStatus.truncationSizePerEgressPortSupported:
               truncationSize = getGlobalTruncationSize()
            toolGroup = []
            if switchIntfConfig.toolGroup.keys():
               toolGroup = switchIntfConfig.toolGroup.keys()
            mplsPop = switchIntfConfig.toolMplsPop
            toolProperty.details = TapAggModels.TapAggProperties.ToolProperty. \
                Details( truncationSize=truncationSize,
                         aclsAppliedPerType=getAppliedAcls( 'out', intfName.name ),
                         groups=toolGroup,removedVlans=removedVlans )
 
         model.toolProperties[ intfName.name ] = toolProperty

   return model

def doShowTapAggMap( mode, args ):
   configured = 'configured' in args
   sicdir = bridgingSwitchIntfConfigDir
   ethIntfs = IntfCli.Intf.getAll( mode, None, intfType=EthIntfCli.EthIntf )

   if 'TAPPORT' in args:
      mapType = 'tap'
      model = TapAggModels.TapToToolMap() 
   elif 'TOOLPORT' in args:
      mapType = 'tool'
      model = TapAggModels.ToolToTapMap()
 
   tapGroupsDict = { }
   toolGroupsDict = { }
   rawIntfDict = { }

   if ethIntfs is None:
      return None

   for intfName in ethIntfs:
      switchIntfConfig = sicdir.switchIntfConfig.get( intfName.name )
      if switchIntfConfig is None:
         continue
 
      if configured:
         rawIntfDict[ intfName.name ] = switchIntfConfig.tapRawIntf.keys()
         for tapGroup in switchIntfConfig.tapGroup:
            if tapGroup in tapGroupsDict:
               tapGroupsDict[ tapGroup ].append( intfName.name )
            else:
               tapGroupsDict[ tapGroup ] = [ intfName.name ]
         for toolGroup in switchIntfConfig.toolGroup:
            if toolGroup in toolGroupsDict:
               toolGroupsDict[ toolGroup ].append( intfName.name )
            else:
               toolGroupsDict[ toolGroup ] = [ intfName.name ]
      else:
         for intf in switchIntfConfig.tapRawIntf:
            if intf in tapAggHwConfig.toolPort and \
                   intfName.name in tapAggHwConfig.tapPort and \
                   intfUp( intf ) and intfUp( intfName.name ):
               if intfName.name in rawIntfDict:
                  rawIntfDict[ intfName.name ].append( intf )
               else:
                  rawIntfDict[ intfName.name ] = [ intf ]
         for tapaggGroup in tapAggStatus.group:
            if intfName.name in tapAggStatus.group[ tapaggGroup ].tapPort and \
                   intfName.name in tapAggHwConfig.tapPort and \
                   intfUp( intfName.name ):
               if tapaggGroup in tapGroupsDict:
                  tapGroupsDict[ tapaggGroup ].append( intfName.name )
               else:
                  tapGroupsDict[ tapaggGroup ] = [ intfName.name ]
            if intfName.name in tapAggStatus.group[ tapaggGroup ].toolPort and \
                   intfName.name in tapAggHwConfig.toolPort and \
                   intfUp( intfName.name ):
               if tapaggGroup in toolGroupsDict:
                  toolGroupsDict[ tapaggGroup ].append( intfName.name )
               else:
                  toolGroupsDict[ tapaggGroup ] = [ intfName.name ]

   for intfName in ethIntfs:
      switchIntfConfig = sicdir.switchIntfConfig.get( intfName.name )
      if switchIntfConfig is None or \
             ( not configured and not intfUp( intfName.name ) ):
         continue
      tapProperty = []
      toolProperty = []

      # tap tool group config
 
      if mapType == 'tap':
         for tapGroup in switchIntfConfig.tapGroup:
            if tapGroup not in tapGroupsDict or tapGroup not in toolGroupsDict:
               continue
            for toolPort in toolGroupsDict[ tapGroup ]:
               tapIntfProperty = TapAggModels.MapProperty(
                  mappedIntfName=toolPort,
                  groupName=tapGroup,
                  policyName='' )
               tapProperty.append( tapIntfProperty )
      elif mapType == 'tool':
         for toolGroup in switchIntfConfig.toolGroup:
            if toolGroup not in tapGroupsDict or toolGroup not in toolGroupsDict:
               continue
            for tapPort in tapGroupsDict[ toolGroup ]:
               toolIntfProperty = TapAggModels.MapProperty(
                  mappedIntfName=tapPort,
                  groupName=toolGroup,
                  policyName='' )
               toolProperty.append( toolIntfProperty )

      # raw intf config
      if intfName.name in rawIntfDict:
         for intf in rawIntfDict[ intfName.name ]:
            rawIntfProperty = TapAggModels.MapProperty( 
               mappedIntfName=intf,
               groupName='',
               policyName='' )
            tapProperty.append( rawIntfProperty )
         
      for key in rawIntfDict:
         if intfName.name in rawIntfDict[ key ]:
            toolIntfProperty = TapAggModels.MapProperty(
               mappedIntfName=key,
               groupName="",
               policyName='' )
            toolProperty.append( toolIntfProperty )
               
      # policy config

      if intfName.name in tapAggIntfConfig.intf:
         pmap = tapAggIntfConfig.intf[ intfName.name ]
         if pmap in tapAggPmapConfig.pmapType.pmap:
            currCfg = tapAggPmapConfig.pmapType.pmap[ pmap ].currCfg
            for classMap in currCfg.classAction:
               if classMap not in currCfg.rawClassMap:
                  policyName = "%s/%s" % ( pmap, classMap )
               else:
                  policyName = "%s/*" % ( pmap )
               if 'setAggregationGroup' in \
                      currCfg.classAction[ classMap ].policyAction:
                  aggGroups = currCfg.classAction[ classMap ].\
                      policyAction[ 'setAggregationGroup' ].aggGroup
                  aggIntfs = currCfg.classAction[ classMap ].\
                      policyAction[ 'setAggregationGroup' ].aggIntf
                  for aggGroup in aggGroups:
                     if aggGroup not in toolGroupsDict:
                        continue
                     for toolPort in toolGroupsDict[ aggGroup ]:
                        tapIntfProperty = TapAggModels.MapProperty(
                           mappedIntfName=toolPort,
                           groupName=aggGroup,
                           policyName=policyName )
                        tapProperty.append( tapIntfProperty )
                  for aggIntf in aggIntfs:
                     if not configured and not intfUp( aggIntf ):
                        continue
                     tapIntfProperty = TapAggModels.MapProperty(
                        mappedIntfName=aggIntf,
                        groupName='',
                        policyName=policyName )
                     tapProperty.append( tapIntfProperty )

      for intf in tapAggIntfConfig.intf:
         pmap = tapAggIntfConfig.intf[ intf ]
         if pmap in tapAggPmapConfig.pmapType.pmap:
            currCfg = tapAggPmapConfig.pmapType.pmap[ pmap ].currCfg
            for classMap in currCfg.classAction:
               if classMap not in currCfg.rawClassMap:
                  policyName = "%s/%s" % ( pmap, classMap )
               else:
                  policyName = "%s/*" % ( pmap )
               if 'setAggregationGroup' in \
                      currCfg.classAction[ classMap ].policyAction:
                  aggGroups = currCfg.classAction[ classMap ].\
                      policyAction[ 'setAggregationGroup' ].aggGroup
                  aggIntfs = currCfg.classAction[ classMap ].\
                      policyAction[ 'setAggregationGroup' ].aggIntf
                  if intfName.name in aggIntfs:
                     toolIntfProperty = TapAggModels.MapProperty(
                        mappedIntfName=intf,
                        groupName='',
                        policyName=policyName )
                     toolProperty.append( toolIntfProperty )
                  for aggGroup in aggGroups:
                     if aggGroup not in toolGroupsDict:
                        continue
                     if intfName.name in toolGroupsDict[ aggGroup ]:
                        toolIntfProperty = TapAggModels.MapProperty(
                           mappedIntfName=intf,
                           groupName=aggGroup,
                           policyName=policyName )
                        toolProperty.append( toolIntfProperty )

      if mapType == 'tap':
         model.tapPortsMap[ intfName.name ] = \
             TapAggModels.MapPropertyList( 
            mapPropertyList=tapProperty )
      elif mapType == 'tool':
         model.toolPortsMap[ intfName.name ] = \
             TapAggModels.MapPropertyList( 
            mapPropertyList=toolProperty )

   return model

def getAppliedAcls( direction, interfaceName ):
   aclDic = {}
   for aclType in aclIntfConfig.config:
      aclIntfTypeConfig = aclIntfConfig.config[ aclType ]
      aclIntfs = aclIntfTypeConfig.intf[ direction ]
      for aclIntf in aclIntfs.intf:
         acl = aclIntfs.intf.get( aclIntf )
         if not acl:
            continue
         aclDic.setdefault( aclIntf, {} )
         aclDic[ aclIntf ].setdefault( aclType, [] ).append( acl )

   aclsAppliedPerType = {}
   if interfaceName in aclDic:
      for aclType in aclDic[ interfaceName ]:
         aclsApplied = aclDic[ interfaceName ][ aclType ]
         appliedAcls = TapAggModels.AclsAppliedPerType( aclsApplied=aclsApplied )
         aclsAppliedPerType[ aclType ] = appliedAcls
   return aclsAppliedPerType

def getPortStatuses( portInfo, policy=None ):
   model = TapAggModels.TapAggGroups.TapAggGroup.PortStatus()
   if not policy:
      model.policies = [ '*' ]
      if portInfo.active:
         model.statuses = [ 'Active' ]
      else:
         model.statuses = [ s for s in portInfo.reason.split( ',' ) if s ]
   else:
      model.policies = [ policy ]
      if not portInfo:
         model.statuses = [ 'Link unknown' ]
      elif portInfo == 'linkUp':
         model.statuses = [ 'Active' ]
      elif portInfo == 'linkDown':
         model.statuses = [ 'Link down' ]
   return model

def getPortStatusMap( port ):
   return { port: getPortStatuses( portInfo ) for ( port, portInfo ) in
           port.iteritems() }

def updateTapPortByPolicy( tapPortMap, groupName ):
   for policyName, policyInfo in tapAggPmapConfig.pmapType.pmap.iteritems():
      # Skip policies without a configured class action
      if not policyInfo.currCfg:
         continue
      for className, classInfo in policyInfo.currCfg.classAction.iteritems():
         if 'setAggregationGroup' in classInfo.policyAction and \
            groupName in classInfo.policyAction[ 'setAggregationGroup' ].aggGroup:
            for port, portPolicy in tapAggIntfConfig.intf.iteritems():
               if port in tapAggHwConfig.tapPort and portPolicy == policyName:
                  policyClassTag = policyName + '_' + className
                  if port in tapPortMap:
                     tapPortMap[ port ].policies.append( policyClassTag )
                  else:
                     portInfo = intfStatusAll[ port ].linkStatus
                     tapPortMap[ port ] = getPortStatuses( portInfo,
                                                           policyClassTag )

def createTapAggGroupModel( group ):
   model = TapAggModels.TapAggGroups.TapAggGroup()
   model.tapPortStatuses = getPortStatusMap( group.tapPort )
   updateTapPortByPolicy( model.tapPortStatuses, group.name )
   model.toolPortStatuses = getPortStatusMap( group.toolPort )
   return model

def doShowTapAggGroups( mode, args ):
   groups = args.get( 'GROUPNAME' )
   detail = 'detail' in args
   model = TapAggModels.TapAggGroups( _detail=detail )

   if not tapAggCliConfig.enabled:
      mode.addError( "Tap-aggregation is not enabled" )
      return model
   if groups:
      for group in groups:
         if group not in tapAggStatus.group:
            mode.addError( 'Group %s not created' % group )
   elif not tapAggStatus.group:
      return model
   
   for groupName, group in tapAggStatus.group.iteritems():
      if groups and groupName not in groups:
         continue

      if groupName == sicType.tapGroupDefault:
         continue

      model.groups[ groupName ] = createTapAggGroupModel( group ) 
   return model


#----------------------------------------------------------------------------
# Use CLI hook to aggregate counter values of TAP ports (initially the hook's
# only extension is implemented by SandDanz)
#----------------------------------------------------------------------------
tapPortCountersGetHook = CliExtensions.CliHook()
def doShowAllTapPortCounters( mode, args ):
   """For every tap port, print the corresponding default-forwarded counter stats."""
   # aggregate all counters of all extensions in the model
   aggregatedTapPortCounters = TapAggModels.IntfCounterModel()
   for hook in tapPortCountersGetHook.extensions():
      aggregatedTapPortCounters.interfaces.update( hook( mode ).interfaces )
   return aggregatedTapPortCounters

tapShowKw = CliCommand.guardedKeyword( 'tap',
                                       'Show interface TAP information',
                                       tapaggGuard )
toolShowKw = CliCommand.guardedKeyword( 'tool',
                                        'Show interface TOOL information',
                                        tapaggGuard )
detailShowKw = CliMatcher.KeywordMatcher( 'detail',
                                          helpdesc='More comprehensive output' )

class ShowIntfTapTool( IntfCli.ShowIntfCommand ):
   syntax = 'show interfaces tap | tool [ detail ]'
   data = dict( tap=tapShowKw,
                tool=toolShowKw,
                detail=detailShowKw )
   handler = doShowTapAggMode
   moduleAtEnd = True
   cliModel = TapAggModels.TapAggProperties

BasicCli.addShowCommandClass( ShowIntfTapTool )

def convertConfigToolGroupsFormat( mode ):
   tapAggCliConfig.toolGroupNewFormat = True

# Register convertConfigToolGroups via "config convert new-syntax" 
CliPlugin.ConfigConvert.registerConfigConvertCallback(
                           convertConfigToolGroupsFormat )

def Plugin( entityManager ):
   global tapAggCliConfig, lagConfig, bridgingConfig, bridgingSwitchIntfConfigDir
   global hwSlice, ethPhyIntfConfigDir, aclIntfConfig, tapAggStatus, tapAggHwStatus
   global tapAggHwConfig, tapAggIntfConfig, tapAggPmapConfig, intfStatusAll

   tapAggCliConfig = ConfigMount.mount( entityManager, 'tapagg/cliconfig',
                                      'TapAgg::CliConfig', 'w' )
   lagConfig = LazyMount.mount( entityManager, "lag/config",
                                "Lag::Config", "r" )
   bridgingConfig = LazyMount.mount( entityManager, "bridging/config",
                                     "Bridging::Config", "r" )
   bridgingSwitchIntfConfigDir = LazyMount.mount( entityManager, 
                                               "bridging/switchIntfConfig",
                                               "Bridging::SwitchIntfConfigDir", "r" )
   hwSlice = LazyMount.mount( entityManager, 'hardware/slice', 'Tac::Dir', 'r' )
   ethPhyIntfConfigDir = LazyMount.mount( entityManager,
                                          "interface/config/eth/phy/all",
                                          "Interface::AllEthPhyIntfConfigDir", "r" )
   aclIntfConfig = LazyMount.mount( entityManager, "acl/intf/config/cli",
                                    "Acl::IntfConfig", "r" )
   tapAggStatus = LazyMount.mount( entityManager, 'tapagg/status',
                                   'TapAgg::Status', 'r' )
   tapAggHwStatus = LazyMount.mount( entityManager, 'tapagg/hwstatus',
                                     'TapAgg::HwStatus', 'r' )
   tapAggHwConfig = LazyMount.mount( entityManager, 'tapagg/hwconfig',
                                     'TapAgg::HwConfig', 'r' )
   tapAggIntfConfig = LazyMount.mount( entityManager, 'tapagg/intfconfig',
                                       'PolicyMap::IntfConfig', 'r' )
   tapAggPmapConfig = LazyMount.mount( entityManager, 'tapagg/pmapconfig',
                                       'TapAgg::PmapConfig', 'r' )
   intfStatusAll = LazyMount.mount( entityManager, 
                                    "interface/status/all",
                                    "Interface::AllIntfStatusDir", "r" )
