# Copyright (c) 2012 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import CliExtensions
import BasicCli, IntfCli, ConfigMount
from BridgingCli import switchIntfConfigIfEnabled
from VlanCli import SwitchportModelet, switchportEligible, vlanIdMatcher
import CliCommand

# Module globals set up by the Plugin function below
taggedPduIntfConfigDir = None

#-------------------------------------------------------------------------------
# The "show interfaces [<intf>] l2-protocol encapsulation" command, in "enable" mode.
#
# The full syntax of this command is:
#
#   show interfaces [<intf>] l2-protocol encapsulation
#   show interfaces l2-protocol encapsulation module <mod>
#   show interfaces module <mod> l2-protocol encapsulation
#-------------------------------------------------------------------------------
def doShowL2Protocol( mode, args ):
   intf = args.get( 'INTF' )
   mod = args.get( 'MOD' )
   intfs = IntfCli.Intf.getAll( mode, intf, mod )
   if intfs is None:
      return

   if intf is None:
      def intfIsL2ProtocolEncap( i ):
         switchIntfConfig = switchIntfConfigIfEnabled( mode, i )
         return switchIntfConfig
      intfs = filter( intfIsL2ProtocolEncap, intfs )

   fmt = '%-20.20s %-25.25s'
   print fmt % ( 'Interface', 'L2 Protocol Encapsulation' )

   for i in intfs:
      # Show tagged pdu info irrespective of the port mode
      tpc = taggedPduIntfConfigDir.taggedPduIntfConfig.get( i.name )
      if tpc and tpc.acceptTaggedPdu:
         print fmt % ( i.shortname, "Enabled (VLAN %d)" % tpc.allowedTag )
      else:
         print fmt % ( i.shortname, "Disabled" )

class ShowIntfL2Protocol( IntfCli.ShowIntfCommand ):
   syntax = 'show interfaces l2-protocol encapsulation'
   data = { 'l2-protocol' : 'interface l2-protocol information',
            'encapsulation' : 'interface l2-protocol encapsulation information' }
   handler = doShowL2Protocol
   moduleAtEnd = True

BasicCli.addShowCommandClass( ShowIntfL2Protocol )

#-------------------------------------------------------------------------------
# -  the "[no] l2-protocol encapsulation dot1q vlan <vlan>" command
#-------------------------------------------------------------------------------
def getTaggedPduIntfConfigEvenIfDisabled( mode ):
   assert switchportEligible( mode )
   tpc = taggedPduIntfConfigDir.taggedPduIntfConfig.newMember( mode.intf.name )
   return tpc

taggedPduConfigHook = CliExtensions.CliHook()

def setTaggedPduConfig( mode, args ):
   vlanId = args[ 'VLANID' ]
   tpc = getTaggedPduIntfConfigEvenIfDisabled( mode )
   tpc.acceptTaggedPdu = True
   tpc.allowedTag = vlanId.id
   for hook in taggedPduConfigHook.extensions():
      hook( mode, vlanId )

noTaggedPduConfigHook = CliExtensions.CliHook()

def noTaggedPduConfig( mode, args ):
   for hook in noTaggedPduConfigHook.extensions():
      hook( mode )
   del taggedPduIntfConfigDir.taggedPduIntfConfig[ mode.intf.name ]

class SetIntfL2Protocol( CliCommand.CliCommandClass ):
   syntax = 'l2-protocol encapsulation dot1q vlan VLANID'
   noOrDefaultSyntax = 'l2-protocol encapsulation dot1q vlan ...'
   data = {
          'l2-protocol' : 'Set L2 protocol characteristics of the interface',
          'encapsulation' : 'Encapsulation of the PDUs',
          'dot1q' : '802.1q encapsulation of the PDUs',
          'vlan' :  'VLAN tag',
          'VLANID': vlanIdMatcher,
   }
   handler = setTaggedPduConfig
   noOrDefaultHandler = noTaggedPduConfig

SwitchportModelet.addCommandClass( SetIntfL2Protocol )

# Cli extensions for clearing tagged pdu configuration when an
# interface is deleted
taggedPduIntfDeletionHook = CliExtensions.CliHook()

class TaggedPduIntfConfigCleaner( IntfCli.IntfDependentBase ):
   """This class is responsible for removing entries from the 
   taggedPduIntfConfig collection when the interfaces associated with those
   taggedPduIntfConfigs are removed."""

   def setDefault( self ):
      self.removeTaggedPduIntfConfigs()

   def removeTaggedPduIntfConfigs( self ):
      for hook in taggedPduIntfDeletionHook.extensions():
         hook( self.intf_ )
      # this delete is safe even if self.intfName_ is not present in
      # the collection because it is a tacc collection, i.e. no KeyError will
      # be generated
      del taggedPduIntfConfigDir.taggedPduIntfConfig[ self.intf_.name ]


#-------------------------------------------------------------------------------
# Have the Cli Agent mount all needed state from sysdb
#-------------------------------------------------------------------------------
def Plugin( entityManager ):
   global taggedPduIntfConfigDir

   taggedPduIntfConfigDir = ConfigMount.mount( entityManager, 
         "l2protocol/taggedPduIntfConfigDir",
         "L2Protocol::TaggedPduIntfConfigDir", "w" )
   IntfCli.Intf.registerDependentClass( TaggedPduIntfConfigCleaner,
         priority=10 )
