# Copyright (c) 2008-2011, 2013 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

#-------------------------------------------------------------------------------
# This module implements TACACS+ configuration.
#
# In enable mode:
#
#     show tacacs
#     clear aaa counters tacacs
#
# In config mode:
#
#     [no] tacacs-server key [0] <key-text>
#     [no] tacacs-server timeout <1-1000>
#     [no] tacacs-server policy unknown-mandatory-attribute ignore
#     [no] tacacs-server host <ip-addr-or-hostname> [single-connection]
#           [port <1-65535>] [timeout <1-1000>] [key [0 | 7] <key-text>]
#     [no] tacacs-server qos dscp <0-63>
#     [no] tacacs-server username max-length <1-255>
#
# Child mode of config mode:
#
#     aaa group server tacacs+ <server-group-name>
#        [no] server <ip-addr-or-hostname> [port <1-65535>]
#
#
# In global IP config mode:
#     [no] ip tacacs source-interace <interface-name>
#-------------------------------------------------------------------------------
import AaaCli
import AaaCliLib
import Ark
import BasicCli
import ConfigMount
import CliCommand
import CliMatcher
import CliToken.Clear
import CliToken.Ip
import DscpCliLib
import HostnameCli
import IntfCli
import Intf.Log
import LazyMount
import ReversibleSecretCli
import ShowCommand
import Tac
import Tacacs
import TacacsGroup
import TacacsModel
import Tracing
from CliPlugin.VrfCli import DEFAULT_VRF, DEFAULT_VRF_OLD, VrfExprFactory

t0 = Tracing.trace0

# Need to define this here because I can't access these .tac definitions
# from Python due to lack of constAttr support
_timeoutDefault = 5
_timeoutMin = 1
_timeoutMax = 1000

tacacsConfig = None
tacacsCounterConfig = None
tacacsStatus = None
aaaConfig = None
dscpConfig = None

def status():
   status_ = tacacsStatus
   assert status_ is not None
   return status_

def tacacsHost( mode, hostname, port, vrf, create=False ):
   hosts = tacacsConfig.host
   assert vrf and vrf != ''
   spec = Tac.Value( "Aaa::HostSpec", hostname=hostname, port=port,
         acctPort=0, vrf=vrf )
   if spec in hosts:
      host = hosts[ spec ]
   elif create:
      Tracing.trace0( "Creating host:", hostname, ":", port )
      host = hosts.newMember( spec )
      if host is None:
         Tracing.trace0( "Unable to create Host:", hostname, ":", port )
      else:
         host.index = AaaCliLib.getHostIndex( hosts )
   else:
      host = None
   if host is not None:
      assert host.hostname == hostname
      assert host.port == port
   return host

# passTheBuck can be used as a CliParser rule value function that returns its
# args as its value, which provides a mechanism for a subrule to get its info
# up to its parent rule's value function.
def passTheBuck( mode, **args ):
   return args

#-------------------------------------------------------------------------------
# "show tacacs" in enable mode
#-------------------------------------------------------------------------------
def showTacacsHost( host, counters ):
   assert host.vrf != ''
   if not counters:
      counters = Tac.Value( "Tacacs::Counters" )
   ret1 = TacacsModel.TacacsStats()
   ret1.serverInfo = TacacsModel.ServerInfo( hostname=host.hostname,
                                             authport=host.port )
   if host.vrf != DEFAULT_VRF:
      ret1.serverInfo.vrf = host.vrf
   ret1.connectionOpens = counters.connOpens
   ret1.connectionCloses = counters.connCloses
   ret1.connectionDisconnects = counters.connDisconnects
   ret1.connectionFailures = counters.connFails
   ret1.connectionTimeouts = counters.connTimeouts
   ret1.messagesSent = counters.messagesSent
   ret1.messagesReceived = counters.messagesReceived
   ret1.receiveErrors = counters.receiveErrors
   ret1.receiveTimeouts = counters.receiveTimeouts
   ret1.sendTimeouts = counters.sendTimeouts
   ret1.dnsErrors = counters.dnsErrors

   # the following is really bad - should remove
   vrfString = " (vrf %s)" % host.vrf if host.vrf != DEFAULT_VRF else ""
   if vrfString:
      ret1.vrf = vrfString
   return ret1

def showTacacs( mode ):
   ret = TacacsModel.ShowTacacs()
   for h in sorted( tacacsConfig.host.values(), key=lambda host: host.index ):
      ret.tacacsServers.append(
         showTacacsHost( h, tacacsStatus.counter.get( h.spec ) ) )
   ret.srcIntf = dict( tacacsConfig.srcIntfName )
   for k in sorted( aaaConfig.hostgroup.keys() ):
      g = aaaConfig.hostgroup[ k ]
      if g.groupType == 'tacacs':
         serverGroupDisplay = AaaCliLib.getCliDisplayFromGroup( g.groupType )
         serverGroupName = g.name
         ret.groups[ serverGroupName ] = TacacsModel.ServerGroup()
         ret.groups[ serverGroupName ].serverGroup = serverGroupDisplay
         for m in g.member.itervalues():
            ret2 = TacacsModel.ServerInfo()
            ret2.hostname = m.spec.hostname
            ret2.authport = m.spec.port
            if m.spec.vrf != DEFAULT_VRF :
               ret2.vrf = m.spec.vrf
            ret.groups[ serverGroupName ].members.append( ret2 )
   ret.lastCounterClearTime = Ark.switchTimeToUtc( tacacsStatus.lastClearTime )
   return ret

class ShowTacacsCommand( ShowCommand.ShowCliCommandClass ):
   syntax = "show tacacs"
   data = { "tacacs" : 'TACACS+ server attributes' }
   cliModel = TacacsModel.ShowTacacs

   @staticmethod
   def handler( mode, args ):
      return showTacacs( mode )

BasicCli.addShowCommandClass( ShowTacacsCommand )

#-------------------------------------------------------------------------------
# "clear aaa counters tacacs" in enable mode
#-------------------------------------------------------------------------------
def clearCounters( mode ):
   Intf.Log.logClearCounters( "tacacs" )
   tacacsCounterConfig.clearCounterRequestTime = Tac.now()
   try:
      Tac.waitFor( lambda: tacacsStatus.lastClearTime >=
                   tacacsCounterConfig.clearCounterRequestTime,
                   description='Tacacs clear counter request to complete',
                   warnAfter=None, sleep=True, maxDelay=0.5, timeout=5 )
   except Tac.Timeout:
      mode.addWarning(
         "TACACS counters may not have been reset yet" )

class ClearTacacsCounterCommand( CliCommand.CliCommandClass ):
   syntax = "clear aaa counters tacacs"
   data = {
      'clear' : CliToken.Clear.clearKwNode,
      'aaa' : AaaCli.aaaAfterClearMatcher,
      'counters' : AaaCli.aaaCounterMatcher,
      'tacacs' : "Clear TACACS counters"
      }
   @staticmethod
   def handler( mode, args ):
      clearCounters( mode )

BasicCli.EnableMode.addCommandClass( ClearTacacsCounterCommand )

#-------------------------------------------------------------------------------
# config mode commands
#-------------------------------------------------------------------------------
configMode = BasicCli.GlobalConfigMode

tacacsServerKwMatcher = CliMatcher.KeywordMatcher(
   'tacacs-server',
   helpdesc='Modify TACACS+ parameters' )

keyExpression = ReversibleSecretCli.reversibleSecretCliExpression( '<KEY>' )

#-------------------------------------------------------------------------------
# "[no] tacacs-server key <KEY>" in config mode
#-------------------------------------------------------------------------------
class TacacsServerKeyCommand( CliCommand.CliCommandClass ):
   syntax = 'tacacs-server key <KEY>'
   noOrDefaultSyntax = 'tacacs-server key ...'
   data = { 'tacacs-server' : tacacsServerKwMatcher,
            'key' : 'Set TACACS+ secret key',
            '<KEY>' : keyExpression }

   @staticmethod
   def handler( mode, args ):
      key = args[ '<KEY>' ]
      t0( 'setKey', str( key ) )
      tacacsConfig.key = key

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      t0( 'noKey' )
      tacacsConfig.key = ""

configMode.addCommandClass( TacacsServerKeyCommand )

#-------------------------------------------------------------------------------
# "[no] tacacs-server timeout <1-1000>" in config mode
#-------------------------------------------------------------------------------
class TacacsServerTimeoutCommand( CliCommand.CliCommandClass ):
   syntax = "tacacs-server timeout <TIMEOUT>"
   noOrDefaultSyntax = "tacacs-server timeout ..."
   data = {
      'tacacs-server' : tacacsServerKwMatcher,
      'timeout' : 'Time to wait for a TACACS+ server to respond',
      '<TIMEOUT>' : CliMatcher.IntegerMatcher(
         _timeoutMin, _timeoutMax,
         helpdesc='Number of seconds' )
   }
   @staticmethod
   def handler( mode, args ):
      timeout = args[ '<TIMEOUT>' ]
      t0( 'setTimeout', str( timeout ) )
      tacacsConfig.timeout = timeout

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      t0( 'noTimeout' )
      tacacsConfig.timeout = tacacsConfig.defaultTimeout

configMode.addCommandClass( TacacsServerTimeoutCommand )

#-------------------------------------------------------------------------------
# "[no] tacacs-server policy unknown-mandatory-attribute ignore" in config mode
#-------------------------------------------------------------------------------
class TacacsServerUnknownAttrCommand( CliCommand.CliCommandClass ):
   syntax = "tacacs-server policy unknown-mandatory-attribute ignore"
   noOrDefaultSyntax = syntax
   data = {
      'tacacs-server' : tacacsServerKwMatcher,
      'policy' : 'Set TACACS+ policy',
      'unknown-mandatory-attribute' : 'Unknown mandatory attributes',
      'ignore' : 'Ignore unknown mandatory attributes'
   }
   @staticmethod
   def handler( mode, args ):
      t0( 'setIgnoreUnknownMandatoryAttr' )
      tacacsConfig.ignoreUnknownMandatoryAttr = True

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      t0( 'noIgnoreUnknownMandatoryAttr' )
      tacacsConfig.ignoreUnknownMandatoryAttr = False

configMode.addCommandClass( TacacsServerUnknownAttrCommand )

#-------------------------------------------------------------------------------
# "[no] tacacs-server qos dscp <0-63>" in config mode
#-------------------------------------------------------------------------------
def updateDscpRules():
   dscpValue = tacacsConfig.dscpValue

   if not dscpValue:
      del dscpConfig.protoConfig[ 'tacacs' ]
      return

   protoConfig = dscpConfig.newProtoConfig( 'tacacs' )
   ruleColl = protoConfig.rule
   ruleColl.clear()

   for spec in tacacsConfig.host:
      # Traffic connecting to external tacacs server auth.
      DscpCliLib.addDscpRule( ruleColl, spec.hostname,
                              spec.port, False, spec.vrf,
                              'tcp', dscpValue )
      DscpCliLib.addDscpRule( ruleColl, spec.hostname,
                              spec.port, False, spec.vrf,
                              'tcp', dscpValue, v6=True )

      # Traffic connecting to internal tacacs server acct.
      DscpCliLib.addDscpRule( ruleColl, spec.hostname,
                              spec.acctPort, False, spec.vrf,
                              'tcp', dscpValue )
      DscpCliLib.addDscpRule( ruleColl, spec.hostname,
                              spec.acctPort, False, spec.vrf,
                              'tcp', dscpValue, v6=True )

def setDscp( mode, args ):
   tacacsConfig.dscpValue = args[ 'DSCP' ]
   updateDscpRules()

def noDscp( mode, args ):
   tacacsConfig.dscpValue = tacacsConfig.dscpValueDefault
   updateDscpRules()

DscpCliLib.addQosDscpCommandClass( configMode, setDscp, noDscp,
                                   tokenProto=tacacsServerKwMatcher )

#-------------------------------------------------------------------------------
# "[no] tacacs-server username max-length <1-255>" in config mode
#-------------------------------------------------------------------------------
class TacacsServerUsernameMaxLenCommand( CliCommand.CliCommandClass ):
   syntax = "tacacs-server username max-length <LENGTH>"
   noOrDefaultSyntax = "tacacs-server username max-length ..."
   data = {
      'tacacs-server' : tacacsServerKwMatcher,
      'username' : 'Configure username parameters',
      'max-length' : 'Maximum username length',
      '<LENGTH>' : CliMatcher.IntegerMatcher( 1, 255,
                                              helpdesc='Specify username length' )
   }
   @staticmethod
   def handler( mode, args ):
      maxLength = args[ '<LENGTH>' ]
      tacacsConfig.maxUsernameLength = maxLength

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      tacacsConfig.maxUsernameLength = tacacsConfig.maxUsernameLengthDefault

configMode.addCommandClass( TacacsServerUsernameMaxLenCommand )

#-------------------------------------------------------------------------------
# "[no] tacacs-server host <ip-addr-or-hostname> [single-connection]
#     [port <1-65535>] [timeout <1-1000>] [key [0 | 7] <key-text>]"
# in config mode
#-------------------------------------------------------------------------------
def singularNode( keyword, helpdesc ):
   return CliCommand.Node( CliMatcher.KeywordMatcher( keyword,
                                                      helpdesc=helpdesc ),
                           maxMatches=1 )

def _getArg( args, name, defaultValue ):
   # turn list into single-elements for args in a set
   arg = args.get( name, defaultValue )
   if arg and isinstance( arg, list ):
      assert len( arg ) == 1
      return arg[ 0 ]
   return arg

class TacacsServerHostCommand( CliCommand.CliCommandClass ):
   syntax = 'tacacs-server host <HOSTNAME> ' \
            '[ { single-connection | ( VRF ) | ( port <PORT> ) | ' \
            '( timeout <TIMEOUT> ) } ] [ key <KEY> ]'
   noOrDefaultSyntax = 'tacacs-server host ' \
                       '[ <HOSTNAME> [ single-connection ] [ VRF ] ' \
                       '[ port <PORT> ] ] ...'
   data = { 'tacacs-server' : tacacsServerKwMatcher,
            'host' : 'TACACS+ server configuration',
            '<HOSTNAME>' : HostnameCli.IpAddrOrHostnameMatcher(
               helpname='WORD',
               helpdesc='Hostname or IP address of TACACS+ server',
               ipv6=True ),
            'single-connection' :
            singularNode( 'single-connection',
                          'Use one TCP connection for multiple sessions' ),
            'VRF' : VrfExprFactory( helpdesc='VRF for this TACACS+ server',
                                    maxMatches=1 ),
            'port' :
            singularNode( 'port',
                          'TCP port of TACACS+ server ( default is % s )' % \
                          TacacsGroup.defaultPort ),
            '<PORT>' : CliMatcher.IntegerMatcher( 0, 65535,
                                             helpdesc="Number of the port to use" ),
            'timeout' :
            singularNode( 'timeout',
                          'Time to wait for this TACACS+ server to respond ' \
                          '( overrides default)' ),
            '<TIMEOUT>' : CliMatcher.IntegerMatcher(
               _timeoutMin, _timeoutMax,
               helpdesc='Timeout value in seconds to '
                        'wait for the server\'s response' ),
            'key' : 'Encryption key for this TACACS+ server (overrides default)',
            '<KEY>' : keyExpression
            }
   @staticmethod
   def handler( mode, args ):
      hostname = args[ '<HOSTNAME>' ]
      singleConn = _getArg( args, 'single-connection', False )
      vrf = _getArg( args, 'VRF', DEFAULT_VRF )
      port = _getArg( args, '<PORT>', TacacsGroup.defaultPort )
      timeout = _getArg( args, '<TIMEOUT>', None )
      key = args.get( '<KEY>' )
      t0( 'setHost hostname:', hostname, "port:", port, "timeout:", timeout,
          "key:", str( key ), 'vrf:', vrf )
      HostnameCli.resolveHostname( mode, hostname, doWarn=True )
      if timeout is None:
         timeoutVal = 1
      else:
         timeoutVal = int( timeout )
      assert vrf != ''
      host = tacacsHost( mode, hostname, port, vrf, create=True )
      host.useKey = ( key is not None )
      host.key = key or ""
      host.useTimeout = ( timeout is not None )
      host.timeout = timeoutVal
      if singleConn:
         host.connType = 'singleConnection'
      else:
         host.connType = 'multiConnection'
      if mode.session_.interactive_:
         vrfString = " in vrf %s" % vrf if vrf != DEFAULT_VRF else ""
         mode.addMessage( "TACACS+ host %s with port %s created%s" %
                          ( hostname, port, vrfString ) )
      updateDscpRules()

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      hostname = args.get( '<HOSTNAME>' )
      hosts = tacacsConfig.host
      if hostname:
         vrf = args.get( 'VRF', DEFAULT_VRF )
         port = args.get( '<PORT>', TacacsGroup.defaultPort )
         t0( 'noHost hostname:', hostname, "vrf: ", vrf, "port:", port )
         spec = Tac.Value( "Aaa::HostSpec", hostname=hostname, port=port,
                           acctPort=0, vrf=vrf )
         if spec in hosts:
            del hosts[ spec ]
         else:
            if mode.session_.interactive_:
               warningMessage = "TACACS+ host %s with port %s not found" \
                                % ( hostname, port )
               mode.addWarning( warningMessage )
      else:
         # Delete all hosts since no hostname was specified
         hosts.clear()
      updateDscpRules()

configMode.addCommandClass( TacacsServerHostCommand )

#-------------------------------------------------------------------------------
# "[no] ip tacacs source-interface <interface-name>"
# global command
#-------------------------------------------------------------------------------
class TacacsSourceIntfCommand( CliCommand.CliCommandClass ):
   syntax = "ip tacacs [ VRF ] source-interface INTF"
   noOrDefaultSyntax = "ip tacacs [ VRF ] source-interface ..."
   data = {
      "ip" : CliToken.Ip.ipMatcherForConfig,
      "tacacs" : "TACACS+ configuration",
      "VRF" : VrfExprFactory( helpdesc='Specify VRF' ),
      "source-interface" : 'Interface providing the IP source ' \
      'address of TACACS+ packets',
      "INTF" : IntfCli.Intf.matcherWithIpSupport
      }

   @staticmethod
   def handler( mode, args ):
      intf = args[ "INTF" ]
      vrf = args.get( "VRF" )
      if not vrf or vrf == DEFAULT_VRF_OLD:
         vrf = DEFAULT_VRF
      assert vrf
      tacacsConfig.srcIntfName[ vrf ] = intf.name

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      vrf = args.get( "VRF" )
      if not vrf or vrf == DEFAULT_VRF_OLD:
         vrf = DEFAULT_VRF
      assert vrf
      del tacacsConfig.srcIntfName[ vrf ]

configMode.addCommandClass( TacacsSourceIntfCommand )

#-------------------------------------------------------------------------------
# Register "show tech-support" commands
#-------------------------------------------------------------------------------
import CliPlugin.TechSupportCli

CliPlugin.TechSupportCli.registerShowTechSupportCmdCallback(
   '2010-08-27 04:34:25',
   lambda: [ 'show tacacs' ],
   summaryCmdCallback=lambda: [ 'show tacacs' ] )

#-------------------------------------------------------------------------------
# Have the Cli Agent mount all needed state from sysdb
#-------------------------------------------------------------------------------
def Plugin( entityManager ):
   import Cell
   global aaaConfig, tacacsConfig, tacacsCounterConfig, tacacsStatus, dscpConfig
   aaaConfig = LazyMount.mount( entityManager, "security/aaa/config",
                                "Aaa::Config", "r" )
   tacacsConfig = ConfigMount.mount( entityManager, "security/aaa/tacacs/config",
                                     "Tacacs::Config", "w" )
   tacacsCounterConfig = LazyMount.mount( entityManager,
                                          "security/aaa/tacacs/counterConfig",
                                          "AaaPlugin::CounterConfig", "w" )
   tacacsStatus = LazyMount.mount( entityManager,
                                   Cell.path( "security/aaa/tacacs/status" ),
                                   "Tacacs::Status", "r" )
   dscpConfig = ConfigMount.mount( entityManager,  "mgmt/dscp/config",
                                   "Mgmt::Dscp::Config", "w" )

