#
# Copyright (c) 2016 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

from datetime import datetime
from CliModel import Submodel
from CliModel import Model, Enum
from CliModel import Str, Int
from CliModel import Dict, Float

class SystemdUnitState( Model ):
   # from systemd source. systemd/src/basic/unit-name.c
   unitActiveState = Enum( values=[
                           'undefined',
                           # mount unit state
                           'dead',   'mounting',   'mounting-done', 'mounted',
                           'remounting', 'unmounting', 'mounting-sigterm',
                           'mounting-sigkill', 'remounting-sigterm',
                           'remounting-sigkill', 'unmounting-sigterm',
                           'unmounting-sigkill', 'failed',

                           # service unit state
                           'dead', 'start-pre', 'start', 'start-post',
                           'running', 'exited', 'reload', 'stop', 'stop-sigabrt',
                           'stop-sigterm', 'stop-sigkill', 'stop-post',
                           'final-sigterm', 'final-sigkill',
                                  # 'failed' => already in mount state
                           'auto-restart',

                           # socket unit state
                                  # 'dead' => present in service states
                                  # 'start-pre', 
                           'start-chown', 'listening',
                                  # 'start-post', 'stop-post', 'running'
                           'stop-pre-sigterm', 'stop-pre-sigkill',
                                  #'final-sigterm', 'final-sigkill' 
                           ],
                           help="Active state of a systemd unit" )

class SystemdUnitStatus( Model ):
   unitState = SystemdUnitState.unitActiveState
   unitDesc = Str( help='Brief description of unit' )


class SystemdStatusBrief( Model ):

   unitsStatus = Dict( keyType=str, valueType=SystemdUnitStatus,
                       help='Mapping of a systemd unit to its status' )

   def render( self ):
      for k in self.unitsStatus:
         v = self.unitsStatus[ k ]
         print '{0:<34.34s} {1:<10s} {2:s}'.format( k,
                                                   v.unitState, v.unitDesc ) 

class SystemdServiceData( Model ):
   svcDesc = Str( help='Systemd service short description' )
   svcActvState = SystemdUnitState.unitActiveState
   timestamp = Float( help="Timestamp of when unit reached current state",
                      optional=True )
   svcPid = Int( help='Process id. Not shown for reloaded SYS-V service' )
   svcState = Str( help='Running/Exited status of service with status code' )
   cgrpName = Str( help='Name of with CGroup for system service unit' )
   cgrpTasks = Dict( keyType=int, valueType=str,
                     help='Mapping of a cgroup process id to its command line.' )

def timeToStr( timestamp ):
   s = None
   if timestamp is not None:
      s = datetime.fromtimestamp( timestamp ).strftime( '%Y-%m-%d %H:%M:%S' )
   return '' if timestamp is None else 'since ' + s

class SystemdServicesDetail( Model ):
   systemdServicesInfo = Dict( keyType=str, valueType=SystemdServiceData,
                          help='Mapping of a systemd service unit to detailed'
                               'description of its state.' )
                               

   def _renderCgroup( self, cgrp ):
      keys = sorted( cgrp.keys() )
      for pid in keys:
         print '{0:>10d} {1:s}'.format( pid, cgrp[ pid ] )

   def render( self ):
      keys = sorted( self.systemdServicesInfo.keys() )
      for e in keys:
         v = self.systemdServicesInfo[ e ]
         print "    Service:", e
         print "Description:", v.svcDesc
         print "     Active:", 
         print "{0:<15s} {1:s}".format( v.svcActvState, 
                                              timeToStr( v.timestamp) )
         if v.svcPid:
            print "        PID:", v.svcPid
         print "     Status:", v.svcState
         if len( v.cgrpName ) > 0:
            print "     CGroup:", v.cgrpName
            self._renderCgroup( v.cgrpTasks )
         print

class SystemdSocketInfo( Model ):
   sockDesc = Str( help='Description of socket' )
   sockState = SystemdUnitState.unitActiveState
   sockType = Str( help='Type of socket ie., stream/datagram etc' )
   timestamp = Float( help="Timestamp of when unit reached current state",
                      optional=True )

class SystemdSocketsDetail( Model ):
   systemdSocketsInfo = Dict( keyType=str, valueType=SystemdSocketInfo,
                              help='Mapping of a systemd socket to detailed'
                                   'description of its state' )

   def render( self ):
      keys = sorted( self.systemdSocketsInfo.keys() )
      for s in keys:
         v = self.systemdSocketsInfo[ s ]
         print "     Socket:", s
         print "Description:", v.sockDesc
         print "     Active:",
         print "{0:<15s} {1:s}".format( v.sockState, 
                                              timeToStr( v.timestamp ) )
         print "     Listen:", v.sockType
         print 

class SystemdMountpointInfo( Model ):
   mountDesc = Str( help='Mount point description' )
   mountState = SystemdUnitState.unitActiveState
   mountPath = Str( help='Where is the mountpoint' )
   fsType    = Str( help='Filesystem type' )
   mountTime = Float( help="Timestamp of when unit reached current state",
                      optional=True )

class SystemdMountpointsDetail( Model ):
   systemdMountpointsInfo = Dict( keyType=str, valueType=SystemdMountpointInfo,
                                  help='Mapping of a systemd mount to detailed'
                                       'description of its state' )

   def render( self ):
      keys = sorted( self.systemdMountpointsInfo.keys() )
      for m in keys:
         v = self.systemdMountpointsInfo[ m ]
         print "       Name:", m
         print "Description:", v.mountDesc
         print "     Active:", 
         print "{0:<15s} {1:s}".format( v.mountState,
                                              timeToStr( v.mountTime ) )
         print "      Where:", v.mountPath
         print "       What:", v.fsType
         print

class SystemdAllUnitsDetail( Model ):
   systemdServices = Submodel( valueType=SystemdServicesDetail,
                           help='Systemd managed services information' )
   systemdSockets = Submodel( valueType=SystemdSocketsDetail,
                           help='Systemd managed sockets information' )
   systemdMounts = Submodel( valueType=SystemdMountpointsDetail,
                           help='Systemd aware mount points' )
   def render( self ):
      self.systemdServices.render()
      self.systemdSockets.render()
      self.systemdMounts.render()

