#
# Copyright (c) 2016 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import re
import time

import BasicCli
import CliMatcher
import CliPlugin.TechSupportCli
import ShowCommand
import SystemdModels
import Tac

systemKwMatcher = CliMatcher.KeywordMatcher( 'system',
      helpdesc='Show system status' )
systemdKwMatcher = CliMatcher.KeywordMatcher( 'systemd',
      helpdesc='Show status of system units managed by systemd' )
detailKwMatcher = CliMatcher.KeywordMatcher( 'detail',
      helpdesc='Show detailed information of systemd unit' )
mountpointsKwMatcher = CliMatcher.KeywordMatcher( 'mountpoints',
      helpdesc='status of systemd mount units' )
servicesKwMatcher = CliMatcher.KeywordMatcher( 'services',
      helpdesc='status of systemd service units' )
socketsKwMatcher = CliMatcher.KeywordMatcher( 'sockets',
      helpdesc='status of systemd socket units' )

warnAboutDaemonReload = "Unit file changed on disk"

def skipBlankWarningLines( idx, lines ):
   while idx < len ( lines ) and any( [ len ( lines[ idx ] ) == 0,
         warnAboutDaemonReload in lines [ idx ] ] ):
      idx += 1
   return idx

def parseActiveLine( line ):
   state = re.findall( r"\(\w+\)", line )
   runState = 'failed'
   # If a unit is in failed state, its status is printed by systemd as:
   # Active: failed (Result: exit-code) since Tue 2016-06-28
   # For such unit, above re.findall() would return []
   if len( state ):
      runState = state[ 0 ][ 1: ][ :-1 ]

   timestamp = None
   if 'since' in line:
      tstamp = line[ line.index("since") + 10: ][ :19 ]
      time_tuple = time.strptime(tstamp, "%Y-%m-%d %H:%M:%S")
      timestamp = time.mktime( time_tuple )
   # return state, timestamp
   return runState, timestamp
   
def showSystemdStatusBrief( mode, args ):
   model = SystemdModels.SystemdStatusBrief()
   if 'services' in args:
      units = [ 'service' ]
   elif 'sockets' in args:
      units = [ 'socket' ]
   elif 'mountpoints' in args:
      units = [ 'mount' ]
   else:
      units = [ 'service', 'socket', 'mount' ]

   ents = {}
   for u in units:
      cmd = [ '/usr/bin/systemctl', '--full', '--no-legend', '--no-pager',
              '--type=%s' % u ]
      output = Tac.run( cmd, asRoot=True, stdout=Tac.CAPTURE, 
                        stderr=Tac.DISCARD )
      for l in output.splitlines():
         parts = l.split( None, 4 )
         assert len( parts ) > 4
         status = SystemdModels.SystemdUnitStatus()
         status.unitState = parts[ 3 ]
         status.unitDesc = parts[ 4 ]
         ents[ parts[ 0 ] ] = status
   model.unitsStatus = ents
   return model

def showSystemdServicesDetail( mode, args=None ):
   # Read all lines with <pid> <cmdline>
   def parseCgroupTasks( idx, lines ):
      cgrpTasks = {}

      while idx < len( lines ) and len( lines[ idx ] ) > 0:
         # Skip special char, blank(s). Extract just pid and commandline
         pidAndCmdline = re.search( r"(\d+) (.*)", lines[ idx ] )
         assert pidAndCmdline is not None
         assert pidAndCmdline.lastindex == 2
         cgrpTasks[ int( pidAndCmdline.group( 1 ) ) ] = pidAndCmdline.group( 2 )
         idx += 1
      idx += 1 # Skip blank line after Cgroup info
  
      return idx, cgrpTasks

   def skipDocumentationLines( idx, lines ):
      if "Docs:" in lines[ idx ]:
         idx += 1
         while ( idx < len( lines) and len( lines[ idx ] ) and
               lines[ idx ][ 9 ] != ':' ):
            idx += 1
      return idx

   def findCGroup( idx, lines ):
      while idx < len( lines ) and len( lines[ idx ] ):
         if lines[ idx ].startswith( "   CGroup:" ):
            break
         idx += 1
      return idx

   def parseServiceInfo( idx, lines ):
      svcInfo = SystemdModels.SystemdServiceData()
      nameDesc = lines[ idx ].split( None, 2 )  # Service name, description
      svcInfo.svcDesc = nameDesc[ 1 ]           # Service description
      if len( nameDesc ) == 3:  # If description is included, show it
         svcInfo.svcDesc = nameDesc[ 2 ][ 2: ]  # Service description
      idx += 2                                  # Skip "Loaded:" line as well
      svcInfo.svcActvState, svcInfo.timestamp = parseActiveLine( lines[ idx ] )
      idx += 1
      idx = skipDocumentationLines( idx, lines )
      pidState = None
      if idx < len( lines ) and "Process:" in lines[ idx ]:
         pidState = lines[ idx ].split( ':', 1 )[ 1 ]   # Pid, state info
         while idx < len( lines ) and "Process:" in lines[ idx ]:
            idx += 1    # Skip extra Process lines
         if idx < len( lines ) and "Main PID:" in lines[ idx ]:
            idx += 1    # Skip this line as main PID is present in Cgroup info
      elif idx < len( lines ) and "Main PID:" in lines[ idx ]:
         pidState = lines[ idx ].split( ':', 1 )[ 1 ]   # Pid info
         idx += 1

      if pidState:
         try:
            s = pidState.split( None, 1 )
         except ValueError:
            print "Error in parsing ", pidState
         else:
            svcInfo.svcPid = int( s[ 0 ] )
            svcInfo.svcState = s[ 1 ]

      # Could be native fc14 service
      else:
         svcInfo.svcPid = 0
         svcInfo.svcState = svcInfo.svcActvState

      # Find CGroup block
      idx = findCGroup( idx, lines )
      if idx < len( lines ) and 'CGroup:' in lines[ idx ]:
         svcInfo.cgrpName = lines[ idx ].split( ':', 1 )[ 1 ]
         idx, svcInfo.cgrpTasks = parseCgroupTasks( idx+1, lines )
      else:
         svcInfo.cgrpName = ''
         svcInfo.cgrpTasks = {}

      return idx, nameDesc[ 1 ], svcInfo
         

   servicesDetail = {}
   cmd = [ '/usr/bin/systemctl', 'status', '*.service' ]
   output = Tac.run( cmd, asRoot=True, stdout=Tac.CAPTURE,
                     stderr=Tac.DISCARD, ignoreReturnCode=True )
   l = output.splitlines()
   idx = 0
   while idx < len( l ):
      idx, svcName, svcInfo = parseServiceInfo( idx, l )
      servicesDetail[ svcName ] = svcInfo
      idx = skipBlankWarningLines( idx, l )
   
   model = SystemdModels.SystemdServicesDetail()
   model.systemdServicesInfo = servicesDetail
   return model

def showSystemdSocketsDetail( mode, args=None ):
   def skipSockDocumentationLines( idx, lines ):
      if "Docs:" in lines[ idx ]:
         while idx < len( lines ) and len( lines[ idx ] )\
               and " Listen:" not in lines[ idx ]:
            idx += 1
      return idx
 
   def skipToNextSocketdUnit( idx, lines ):
      while idx < len( lines ):
         if len( lines[ idx ] ) and lines[ idx ][ 0 ] == '*':
            break
         idx += 1
      return idx

   def parseSocketInfo( idx, lines ):
      sockInfo = SystemdModels.SystemdSocketInfo()
      nameDesc = lines[ idx ].split( None , 2 )  # sock name, socket description
      sockInfo.sockDesc = nameDesc[ 1 ]         # socket description
      if len( nameDesc ) == 3:  # If description is included, show it
         sockInfo.sockDesc = nameDesc[ 2 ][ 2: ]   # socket description
      idx += 2                                  # Skip "Loaded:" line as well
      sockInfo.sockState, sockInfo.timestamp = parseActiveLine( lines [ idx ] )
      idx += 1
      idx = skipSockDocumentationLines( idx, lines )
      sockInfo.sockType = lines[ idx ].split( ':', 1 )[ 1 ][ 1: ]
      # Skip blank line(s) after Listen: line
      idx = skipToNextSocketdUnit( idx, lines )

      return idx, nameDesc[ 1 ], sockInfo
         
   socketsDetail = {}
   cmd = [ '/usr/bin/systemctl', 'status', '*.socket' ]
   output = Tac.run( cmd, asRoot=True, stdout=Tac.CAPTURE,
                     stderr=Tac.DISCARD, ignoreReturnCode=True )
   l = output.splitlines()
   idx = 0
   while idx < len( l ):
      idx, sockName, sockInfo = parseSocketInfo( idx, l )
      socketsDetail[ sockName ] = sockInfo
      idx = skipBlankWarningLines( idx, l )
   
   model = SystemdModels.SystemdSocketsDetail()
   model.systemdSocketsInfo = socketsDetail
   return model

def showSystemdMountsDetail( mode, args=None ):
   def skipMountDocumentationLines( idx, lines ):
      if idx < len( lines ) and "Docs:" in lines[ idx ]:
         while idx < len( lines ) and len( lines[ idx ] ):
            idx += 1
      idx += 1
      return idx
 
   def parseMountpointInfo( idx, lines ):
      fsMountInfo = SystemdModels.SystemdMountpointInfo()
      nameDesc = lines[ idx ].split( None, 2 )  # mount name, description
      fsMountInfo.mountDesc = nameDesc[ 1 ]     # mount description
      if len( nameDesc ) == 3:  # If description is included, show it
         fsMountInfo.mountDesc = nameDesc[ 2 ][ 2: ] # mount description
      idx += 2                                  # Skip "Loaded:" line as well
      fsMountInfo.mountState, fsMountInfo.mountTime = parseActiveLine(
                                                                lines[ idx ] )
      idx += 1
      fsMountInfo.mountPath = lines[ idx ].split( ':', 1 )[ 1 ][ 1: ]
      idx += 1
      fsMountInfo.fsType = lines[ idx ].split( ':', 1 )[ 1 ][ 1: ]
      idx += 1
      idx = skipMountDocumentationLines( idx, lines )

      return idx, nameDesc[ 1 ], fsMountInfo
    
   mountpointsDetail = {}
   cmd = [ '/usr/bin/systemctl', 'status', '*.mount' ]
   output = Tac.run( cmd, asRoot=True, stdout=Tac.CAPTURE,
                     stderr=Tac.DISCARD, ignoreReturnCode=True )
   l = output.splitlines()
   idx = 0
   while idx < len( l ):
      idx, mountName, mountInfo = parseMountpointInfo( idx, l )
      mountpointsDetail[ mountName ] = mountInfo
      idx = skipBlankWarningLines( idx, l )
      
   model = SystemdModels.SystemdMountpointsDetail()
   model.systemdMountpointsInfo = mountpointsDetail
   return model

def showSystemdAllUnitsDetail( mode, args ):
   model = SystemdModels.SystemdAllUnitsDetail()
   model.systemdServices = showSystemdServicesDetail( mode )
   model.systemdSockets = showSystemdSocketsDetail( mode )
   model.systemdMounts = showSystemdMountsDetail( mode )
   return model

#-----------------------------------------------------------------------------------
# show system systemd [ services | sockets | mountpoints ]
#-----------------------------------------------------------------------------------
class ShowSystemd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show system systemd [ services | sockets | mountpoints ]'
   data = {
            'system': systemKwMatcher,
            'systemd': systemdKwMatcher,
            'services': servicesKwMatcher,
            'sockets': socketsKwMatcher,
            'mountpoints': mountpointsKwMatcher,
          }
   cliModel = SystemdModels.SystemdStatusBrief
   handler = showSystemdStatusBrief

BasicCli.addShowCommandClass( ShowSystemd )

#-----------------------------------------------------------------------------------
# show system systemd services detail
#-----------------------------------------------------------------------------------
class ShowSystemdServicesDetail( ShowCommand.ShowCliCommandClass ):
   syntax = 'show system systemd services detail'
   data = {
            'system': systemKwMatcher,
            'systemd': systemdKwMatcher,
            'services': servicesKwMatcher,
            'detail': detailKwMatcher,
          }
   cliModel = CliPlugin.SystemdModels.SystemdServicesDetail
   handler = showSystemdServicesDetail

BasicCli.addShowCommandClass( ShowSystemdServicesDetail )

#-----------------------------------------------------------------------------------
# show system systemd sockets detail
#-----------------------------------------------------------------------------------
class ShowSystemdSocketsDetail( ShowCommand.ShowCliCommandClass ):
   syntax = 'show system systemd sockets detail'
   data = {
            'system': systemKwMatcher,
            'systemd': systemdKwMatcher,
            'sockets': socketsKwMatcher,
            'detail': detailKwMatcher,
          }
   cliModel = SystemdModels.SystemdSocketsDetail
   handler = showSystemdSocketsDetail

BasicCli.addShowCommandClass( ShowSystemdSocketsDetail )

#-----------------------------------------------------------------------------------
# show system systemd mountpoints detail
#-----------------------------------------------------------------------------------
class ShowSystemdMountpointsDetail( ShowCommand.ShowCliCommandClass ):
   syntax = 'show system systemd mountpoints detail'
   data = {
            'system': systemKwMatcher,
            'systemd': systemdKwMatcher,
            'mountpoints': mountpointsKwMatcher,
            'detail': detailKwMatcher,
          }
   cliModel = SystemdModels.SystemdMountpointsDetail
   handler = showSystemdMountsDetail

BasicCli.addShowCommandClass( ShowSystemdMountpointsDetail )

#-----------------------------------------------------------------------------------
# show system systemd detail
#-----------------------------------------------------------------------------------
class ShowSystemdDetail( ShowCommand.ShowCliCommandClass ):
   syntax = 'show system systemd detail'
   data = {
            'system': systemKwMatcher,
            'systemd': systemdKwMatcher,
            'detail': detailKwMatcher,
          }
   cliModel = SystemdModels.SystemdAllUnitsDetail
   handler = showSystemdAllUnitsDetail

BasicCli.addShowCommandClass( ShowSystemdDetail )

#------------------------------------------------------------------
# Add "show system systemd services detail" and 
#     "show system systemd sockets detail" to "show tech-support".
#------------------------------------------------------------------
# Timestamps are made up to maintain historical order within show tech-support
CliPlugin.TechSupportCli.registerShowTechSupportCmdCallback( 
                           '2016-01-20 00:03:30',
                           lambda : [ 'show system systemd services detail' ],
                           summaryCmdCallback=lambda :
                              [ "show system systemd services" ] )

CliPlugin.TechSupportCli.registerShowTechSupportCmdCallback(
                           '2016-01-21 16:49:21',
                           lambda : [ 'show system systemd sockets detail' ] )


