# Copyright (c) 2013 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

from CliModel import Dict
from CliModel import Int
from CliModel import List
from CliModel import Model
from CliModel import Str
from CliModel import Enum
from CliModel import Bool
from ArnetModel import Ip4Address
from ArnetModel import Ip6Address
from Ark import ReversibleDict
import SshCertLib

MAX_NAME_LEN = 29

class DomainName( Model ):
   domainName = Str( help="The domain name of the switch" )
   
   def render( self ):
      if self.domainName:
         print self.domainName

class SysMgrNameServerInfo( Model ):
   v4NameServers = List( valueType=str, 
                         help="List of IPV4 name servers" )
   v6NameServers = List( valueType=str,
                         help="List of IPV6 name servers" )

   def render( self ):
      for addr in self.v4NameServers:
         print addr
      for addr in self.v6NameServers:
         print addr

class Hostname( Model ):
   hostname = Str( help="The current hostname of the switch" )
   fqdn = Str( help="The fully qualified domain name of the switch" )

   def render( self ):
      if not self.hostname:
         print "No hostname set"
      else:
         print "Hostname:", self.hostname
         print "FQDN:    ", self.fqdn

class SshHostKey( Model ):
   hostKey = Str( help="The public host key of the switch" )

   def render( self ):
      if self.hostKey:
         print self.hostKey
      
class LoginBanner( Model ):
   loginBanner = Str( help="Login banner for the switch" )

   def render( self ):
      print self.loginBanner

class Motd( Model ):
   motd = Str( help="Message of the day for the switch" )

   def render( self ):
      print self.motd

class BootConfig( Model ):
   softwareImage = Str( help="Location of software image" )
   personalityOnReboot = Enum( help='The personality on reboot',
                               values=( 'cpu', 'bmc' ),
                               optional=True )
   consoleSpeed = Int( help="Baud rate of console", optional=True )
   abootPassword = Str( help="Encrypted aboot password" )
   memTestIterations = Int( help="Iterations of Memtest to perform at boot time" )
   securebootSupported = Bool( help="Secure boot is supported" )
   securebootEnabled = Bool( help="Secure boot is enabled" )
   spiFlashWriteProtected = Bool( help="SPI flash is physically protected against "
                                       "writes" )
   tpmPassword = Bool( help="Aboot password is stored in the TPM" )
   aristaCertEnabled = Bool( help="SWIs signed by Arista Networks are authorized "
                                  "to boot" )
   certsLoaded = Bool( help="Certificates were successfully loaded" )
   aristaCert = Str( help="Arista certificate" )
   userCert = Str( help="User certficate" )

   def render( self ):
      def _boolState( value ):
         return 'enabled' if value else 'disabled'

      def _printAttr( description, attr ):
         attr = attr if attr and attr != 0 else "(not set)"
         print "%s: %s" % ( description, attr )

      _printAttr( "Software image", self.softwareImage )

      if self.personalityOnReboot:
         cpuOrBmc = self.personalityOnReboot.upper()
         print "Personality on reboot (CPU or BMC): %s" % cpuOrBmc

      _printAttr( "Console speed", self.consoleSpeed )
      _printAttr( "Aboot password (encrypted)", self.abootPassword )
      _printAttr( "Memory test iterations", self.memTestIterations )

      # If we don't support secureboot, don't bother to print anything about it
      if not self.securebootSupported:
         return

      print
      _printAttr( "Secure boot", _boolState( self.securebootEnabled ) )
      _printAttr( "SPI flash protection", _boolState( self.spiFlashWriteProtected ) )
      _printAttr( "Password storage", 'tpm' if self.tpmPassword else 'boot-config' )
      _printAttr( "Arista certificate", _boolState( self.aristaCertEnabled ) )

      # Don't print anything if the certificate is empty
      if not self.certsLoaded or not self.userCert:
         return

      print "Extra certificate:"
      print self.userCert

class SysMgrHostInfo( Model ):
   class IpAddresses( Model ):
      ipv4Addresses = List( valueType=Ip4Address, help="List of IPV4 addresses" )
      ipv6Addresses = List( valueType=Ip6Address, help="List of IPV6 addresses" )

   hosts = Dict( keyType=str, valueType=IpAddresses,
                 help="A mapping between host names and their IP addresses" )
   domainName = Str( help="The domain name that the listed hosts are "
                     "associated with",
                     optional=True )
   domainList = List( valueType=str,
                      help="List of additional domains to lookup domains with" )
   nameServers = List( valueType=str,
                       help="List of domain name servers used to resolve names" )

   def render( self ):
      if self.domainName:
         print '\nDefault domain is: %s' % self.domainName
      else:
         print '\nDefault domain is not configured'
      if self.domainList:
         print 'Additional domains are: ' + ', '.join( self.domainList )
      print 'Name/address lookup uses domain service'

      print 'Name servers are: ' + ', '.join( self.nameServers )

      print '\nStatic Mappings:\n'

      print '%-*s  %s     %s\n' % ( MAX_NAME_LEN, 'Hostname', 'IP', 'Addresses' )

      for host, addresses in sorted( self.hosts.items() ):
         self._printAddresses( addresses.ipv4Addresses, host, 'IPV4' )
         self._printAddresses( addresses.ipv6Addresses, host, 'IPV6' )
   
   def _printAddresses( self, addresses, host, ipType ):
      firstLineFlag = True
      for addr in addresses:
         if ipType == 'IPV6':
            addr = addr.stringValue
         if firstLineFlag:
            print ( self._formatHostMapping( addr, ipType, host ) )
            firstLineFlag = False
         else:
            print ( self._formatHostMapping( addr ) )

   def _formatHostMapping( self, address, ipType='', name='' ):
      return '%-*s %4s    %s' % ( MAX_NAME_LEN, name, ipType, address )

class ManagementTelnet( Model ):
   serverState = Enum( values=( "enabled", "disabled" ),
                  help="State of the telnet session either enabled or disabled" ) 
   vrfName = Str( help="Configured VRF for telnet session", default="default" )
   maxTelnetSessions = Int( help="Maximum telnet sessions", default=20 )
   maxTelnetSessionsPerHost = Int( help="Maximum telnet sessions from one host",
                                   default=20 )
   globalDefault = Bool( help="VRF service state is dependent on default \
                         telnet state", optional=True )

   def render( self ):
      vrf = ( 'Default VRF' if self.vrfName == 'default' else 
            'VRF %s' % self.vrfName )
      if not self.serverState:
         return
      print 'Telnet status for', vrf, 'is', self.serverState,
      print '( globalDefault )' if self.globalDefault else ''
      print 'Telnet session limit is', self.maxTelnetSessions
      print 'Telnet session limit per host is', self.maxTelnetSessionsPerHost
      print

class TrustedCaKeys( Model ):
   keys = List( valueType=str, help="List of trusted CA public keys" )

   def render( self ):
      for key in self.keys:
         print( key )

class HostCertificates( Model ):
   certificates = Dict( keyType=str, valueType=str,
                        help="A Mapping between configured host certificate"
                             " filenames and the raw certificate" )
   def render( self ):
      for certFile in self.certificates:
         try:
            print SshCertLib.validateHostCert( certFile )
         except SshCertLib.SshHostCertError:
            pass

class RevokedKeys( Model ):
   keys = List( valueType=str, help="List of revoked public keys" )

   def render( self ):
      for key in self.keys:
         print( key )

CapiStringToCliString = ReversibleDict( {
      'default' : 'default',
      'qsfpExpanded' : 'qsfp_expanded',
      'qsfpDense' : 'qsfp_dense',
      'qsfpSparse' : 'qsfp_sparse',
      'notAvailable' : 'notAvailable',
      'unknown' : 'Unknown' } )

class PortNumbering( Model ):
   numberingScheme = Enum( values=CapiStringToCliString.keys(), 
                           help='Port numbering scheme' )

   def render( self ):
      print "\nPort numbering scheme: %r\n" % (
                              CapiStringToCliString[ self.numberingScheme ] )
