# Copyright (c) 2005-2011 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import sys, os
import BasicCli, CliParser, IntfCli, TechSupportCli
import AgentDirectory
from VlanCli import getSwitchInputConfig, Vlan, vlanIdMatcher, vlanNameMatcher, \
   vlanInUseErrStr
import LazyMount, Ethernet
import Tac
import TacSigint
import ConfigMount
import PhysicalIntfRule
import EthIntfLib
import SharedMem
import Smash
import ShowCommand
import CliCommand
import CliMatcher
import CliToken.Mac
from EthIntfCli import EthPhyAutoIntfType, EthPhyIntf
from Intf.IntfRange import subintfSupportedGuard
from IntfModel import InterfaceCounters, InterfaceCountersRate
from IntfCli import subintfMtuHook, RoutedIntfModelet, subintfMtuGuardHook
from TypeFuture import TacLazyType
import Toggles.EbraToggleLib

subIntfConfigDir = None
subIntfStatusDir = None
subIntfPwStatus = None
bridgingHwEnabled = None
bridgingInputConfig = None
bridgingConfig = None
bridgingHwCapabilities = None
allEthIntfConfigDir = None
allIntfStatusDir = None
intfCapability = None
allVrfConfig = None
subIntfCountersDir = None
subIntfCountersCheckpointDir = None
aleCliConfig = None

Dot1qEncapMode = TacLazyType( "Interface::SubIntfDot1qEncapConfigMode" )
FlexEncapRange = TacLazyType( 'Ebra::FlexEncapRange' )
FlexEncapField = TacLazyType( 'Ebra::FlexEncapField' )
FlexEncapSpec = TacLazyType( 'Ebra::FlexEncapSpec' )
FlexEncapEntry = TacLazyType( 'Ebra::FlexEncapEntry' )
VlanEncapReason = TacLazyType( "Interface::SubIntfConfigDir::VlanEncapReason" )

def getPlatformName( sysname ):
   if AgentDirectory.agent( sysname, 'Sand' ) or \
      os.environ.get( 'SIMULATION_SAND' ):
      return 'Sand'
   if AgentDirectory.agent( sysname, 'StrataCentral' ) or \
      os.environ.get( 'SIMULATION_STRATA' ):
      return 'Strata'
   return None

def theSubintfSupportedGuard( mode, token ):
   if ( not mode.session_.isInteractive() or 
        intfCapability.l3SubintfSupported or intfCapability.l2SubintfSupported or
        not mode.session_.guardsEnabled() ):
      return None
   return CliParser.guardNotThisPlatform

def theQinQL3SubintfSupportedGuard( mode, token ):
   if ( not mode.session_.isInteractive() or 
        intfCapability.qinql3SubintfSupported or
        not mode.session_.guardsEnabled() ):
      return None
   return CliParser.guardNotThisPlatform

def theL2SubintfSupportedGuard( mode, token ):
   if ( not mode.session_.isInteractive() or 
        intfCapability.l2SubintfSupported or
        not mode.session_.guardsEnabled() ):
      return None
   return CliParser.guardNotThisPlatform

def theL2SubintfMacLearningDisableSupported( mode, token ):
   if ( not mode.session_.isInteractive() or
        intfCapability.l2SubintfMacLearningDisableSupported or
        not mode.session_.guardsEnabled() ):
      return None
   return CliParser.guardNotThisPlatform

def copyParentMtuToSubintf( intfName, mtu ):
   for ifName in subIntfConfigDir.intfConfig:
      if ifName.split( '.' )[0] == intfName:
         if not subIntfConfigDir.intfConfig[ ifName ].l3MtuConfigured:
            # subinterface doesn't have MTU config
            subIntfConfigDir.intfConfig[ ifName ].mtu = mtu

def subintfMtuGuard ( mode, token ):
   if ( not mode.session_.isInteractive() or 
        intfCapability.subIntfMtuSupported or
        not mode.session_.guardsEnabled() ):
      return None
   return CliParser.guardNotThisPlatform

subintfSupportedGuard.append( theSubintfSupportedGuard )
subintfMtuHook.append( copyParentMtuToSubintf )
subintfMtuGuardHook.append( subintfMtuGuard )

def ethernetSubintfParentCfg( parentName ):
   if not parentName.startswith( 'Et' ):
      return None
   sliceName = EthIntfLib.sliceName( parentName )
   # In case sliceName is empty, we just use allEthIntfConfigDir.
   if sliceName in allEthIntfConfigDir:
      ethPhyDir = allEthIntfConfigDir[ sliceName ]
      parentConf = ethPhyDir.intfConfig.get( parentName ) if ethPhyDir else None
   else:
      parentConf = allEthIntfConfigDir.get( parentName )
   return parentConf

matcherLearning = CliMatcher.KeywordMatcher( 'learning',
      helpdesc='Learning keyword' )

# Currently because SubIntf is in Ebra package, we have to add this hook
# to get parentConfig for ethernet or port-channel sub interfaces. This
# is not a clean design, and needs to be revisited. Instead, the SubIntf
# should really be defined in Intf package. And in appropriate packages,
# we can createSubIntf corresponding sub classes of SubIntf class, to override
# the _parentConfig method, which will be a much cleaner design.
# BUG111733 created to track this.
subintfParentCfgHook = [ ethernetSubintfParentCfg ]

class SubIntf( IntfCli.VirtualIntf ):
   def __init__( self, name, mode ):
      IntfCli.VirtualIntf.__init__( self, name, mode )
      self.subIntfStatuses = subIntfStatusDir.intfStatus
      self.intfStatus = None
      self.parentIntfConfig = None
      self.parentIntfStatus = None
      self.subIntfConfig = None

   #----------------------------------------------------------------------------
   # Returns the SubIntfConfig object for this interface.
   # SubIntfConfig is cached to prevent an update from the EntityLog thread
   # in the middle of processing a "show interfaces" command from causing an
   # inconsistent snapshot to be displayed.
   # The caching is done only if not in a config session, since the above
   # scenario cannot happen when in a config session.
   #----------------------------------------------------------------------------
   def config( self ):
      config = subIntfConfigDir.intfConfig.get( self.name )
      if not self.mode_.session_.inConfigSession():
         self.subIntfConfig = config
      return config

   @Tac.memoize
   def status( self ):
      return self.subIntfStatuses.get( self.name )

   # Performance of this function is not optimal, because we need to
   # search for parent name in config collections every time.
   def _parentConfig( self ):
      parentName = self.name.split( '.' )[ 0 ]
      for hook in subintfParentCfgHook:
         parentConf = hook( parentName )
         if parentConf:
            return parentConf
      return None

   def parentStatus( self ):
      if not self.parentIntfStatus:
         parentName = self.name.split( '.' )[ 0 ]
         self.parentIntfStatus = allIntfStatusDir.intfStatus.get( parentName )

      return self.parentIntfStatus

   def inherit( self ):
      parentConf = self._parentConfig()
      if self.config().l3MtuConfigured:
         # If subinterface has MTU config then don't inherit
         # parent interface's MTU
         return
      if parentConf:
         self.config().mtu = parentConf.mtu
      else:
         RoutedIntfModelet.setDefaultMtu( self )

   def createPhysical( self, startupConfig=False ):
      subIntfConfigDir.intfConfig.newMember( self.name )
      self.inherit()
      self.config().encapType = 'dot1q'
      if self.name not in bridgingInputConfig.switchIntfConfig:
         sic = bridgingInputConfig.newSwitchIntfConfig( self.name, 'dot1qTunnel' )
         # By default, no bridging on this interface. But if a vlan is configured on
         # this interface, then it makes this a l2 subintf, where only bridging is
         # allowed.
         sic.enabled = False

   #----------------------------------------------------------------------------
   # Determines whether the Interface::SubIntfStatus object for this interface
   # exists.
   #----------------------------------------------------------------------------
   def lookupPhysical( self ):
      self.intfStatus = self.subIntfStatuses.get( self.name, None )
      return self.intfStatus is not None

   #----------------------------------------------------------------------------
   # Outputs information about this interface in an interface type-specific
   # manner.
   #----------------------------------------------------------------------------
   def showPhysical( self, mode, intfStatusModel ):
      pass

   #----------------------------------------------------------------------------
   # Destroys the Interface::SubIntfConfig object for this interface if it already
   # exists.
   #----------------------------------------------------------------------------
   def destroyPhysical( self ):
      del subIntfConfigDir.intfConfig[ self.name ]
      del bridgingInputConfig.switchIntfConfig[ self.name ]

   def setDefault( self ):
      IntfCli.VirtualIntf.setDefault( self )
      dot1qEncapModeNone = Dot1qEncapMode.subIntfDot1qEncapConfigNone
      reason = subIntfConfigDir.deleteVlanEncap(
            self.name, FlexEncapEntry(), False, dot1qEncapModeNone )
      if reason != VlanEncapReason.success:
         self.mode_.addError( "Unexpected reason: {}".format( reason ) )
         return
      sic = getSwitchInputConfig( self.name )
      if sic:
         # Clear vlan assigned to L2 SubIntf
         sic.l2SubIntfVlan = Tac.Value( 'Bridging::VlanNameOrId', 'none', '', 0 )
         sic.enabled = False # Disable bridging
      self.setMacAddr( None )

   @staticmethod
   def getAllPhysical( mode ):
      intfs = []
      for name in subIntfStatusDir.intfStatus.keys():
         intf = SubIntf( name, mode )
         if intf.lookupPhysical():
            intfs.append( intf )
      return intfs

   #----------------------------------------------------------------------------
   # Encap Vlan for a subinterface
   #----------------------------------------------------------------------------
   def setVlanEncap( self, mode, configMode, vlanId=None, innerVlanId=None ):
      ModeConflictStr = ( 'encapsulation dot1q vlan command cannot be used in '
                          'conjunction with encapsulation submode' )
      if not vlanId:
         reason = subIntfConfigDir.deleteVlanEncap(
               self.name, FlexEncapEntry(), False, configMode )
         if reason == VlanEncapReason.modeConflict:
            mode.addWarning( ModeConflictStr )
         elif reason != VlanEncapReason.success:
            # sanity check
            mode.addError( "Unexpected reason: {}".format( reason ) )
      else:
         # Create an equivalent FlexEncapEntry
         outer = Tac.nonConst( FlexEncapField.createDot1qTagged() )
         outerTag = int( vlanId.id )
         outer.insertTaggedRange( FlexEncapRange( outerTag, outerTag ) )
         innerTag = int( innerVlanId.id ) if innerVlanId else None
         if innerTag:
            inner = Tac.nonConst( FlexEncapField.createDot1qTagged() )
            inner.insertTaggedRange( FlexEncapRange( innerTag, innerTag ) )
         else:
            inner = FlexEncapField.createAnyOrNone()
         client = FlexEncapSpec.createTagged( outer, inner )
         network = FlexEncapSpec.createAnyOrNone()
         status = subIntfConfigDir.setVlanEncap(
               self.name, FlexEncapEntry( client, network ), configMode, True )
         if status.reason == VlanEncapReason.modeConflict:
            mode.addError( ModeConflictStr )
         elif status.reason == VlanEncapReason.siblingConflict:
            errStr = 'VLAN %d' % outerTag
            if innerTag:
               errStr += ' inner VLAN %d' % innerTag
            errStr += ' already found in sibling %s' % status.siblingIntfId
            mode.addError( errStr )
         elif status.reason != VlanEncapReason.success:
            # sanity check
            mode.addError( "Unexpected reason: {}".format( status.reason ) )

   def switchportEligible( self ):
      return False

   def isSubIntf( self ):
      return True

   #----------------------------------------------------------------------------
   # Utility functions used by show functions.
   #----------------------------------------------------------------------------
   def addrStr( self ):
      return "address is %s" % self.addr()

   def addr( self ):
      return Ethernet.convertMacAddrCanonicalToDisplay( self.status().addr )

   def hardware( self ):
      return "subinterface"

   def bandwidth( self ):
      if Toggles.EbraToggleLib.toggleSubIntfSpeedEnabled():
         return self.status().speed * 1000
      if self.parentStatus() is None:
         return 0
      parentSpeed = self.parentStatus().speed
      if isinstance( parentSpeed, str ): # Parent is Ethernet
         return EthPhyIntf.bandwidthVals[ parentSpeed ]
      elif isinstance( parentSpeed, int ): # Parent is PortChannel
         return parentSpeed * 1000
      else:
         return 0

   def linkStatus( self ):
      ( _, ret ) = self.getStatus()
      return ret

   def autonegActive( self ):
      return False

   def getIntfDuplexStr( self ):
      if self.parentStatus() and hasattr( self.parentStatus(), 'duplex' ):
         return self.parentStatus().duplex
      else:
         return "duplexFull"

   def xcvrTypeStr( self ):
      return self.config().encapType + '-encapsulation'

   def isSubIntfBridgedOrPseudowire( self ):
      return ( self.intfStatus.forwardingModel == 'intfForwardingModelBridged' or
               self.status().intfId in subIntfPwStatus.subIntf )

   def ingressCountersSupported( self ):
      return ( ( bridgingHwEnabled.subintfIngressCountersEnabled and
                 not self.isSubIntfBridgedOrPseudowire() ) or
               ( bridgingHwEnabled.subintfL2IngressCountersEnabled and
                 self.isSubIntfBridgedOrPseudowire() ) )

   def egressCountersSupported( self ):
      return ( ( bridgingHwEnabled.subintfEgressCountersEnabled and
                 not self.isSubIntfBridgedOrPseudowire() ) or
               ( bridgingHwEnabled.subintfL2EgressCountersEnabled and
                 self.isSubIntfBridgedOrPseudowire() ) )

   def bumCountersSupported( self ):
      return bridgingHwEnabled.subintfBumCountersEnabled

   def umCountersSupported( self ):
      return bridgingHwEnabled.subintfUMCountersEnabled

   def l3CountersSupported( self ):
      return ( bridgingHwEnabled.subintfL3CountersEnabled and
               not self.isSubIntfBridgedOrPseudowire() )

   def l2CountersSupported( self ):
      return ( bridgingHwEnabled.subintfL2CountersEnabled and
               self.isSubIntfBridgedOrPseudowire() )

   def l3CountersRateSupported( self ):
      return ( bridgingHwEnabled.subintfL3CountersRateEnabled and
               not self.isSubIntfBridgedOrPseudowire() )

   def l2CountersRateSupported( self ):
      return ( bridgingHwEnabled.subintfL2CountersRateEnabled and
               self.isSubIntfBridgedOrPseudowire() )

   def countersSupported( self ):
      if self.lookupPhysical():
         return self.l2CountersSupported() or self.l3CountersSupported()
      else:
         return False

   def countersRateSupported( self ):
      if self.lookupPhysical():
         return self.l2CountersRateSupported() or self.l3CountersRateSupported()
      else:
         return False

   def countersDiscardSupported( self ):
      return False

   def getIntfCounterDir( self ):
      assert self.status()
      parent = self.status().parent
      if not parent:
         return None
      return subIntfCountersDir

   def updateInterfaceCountersModel( self, checkpoint=None, notFoundString=None,
                                     zeroOut=False ):
      intfCountersModel = InterfaceCounters( _name=self.status().intfId )
      intfCountersModel._ingressCounters = self.ingressCountersSupported()
      intfCountersModel._egressCounters = self.egressCountersSupported()
      intfCountersModel._umCounters = self.umCountersSupported()
      intfCountersModel._bumCounters = self.bumCountersSupported()
      intfCountersModel._l3Counters = self.l3CountersSupported()

      def stat( attr, supported=True ):
         if not supported:
            return 0
         currentValue = getattr( self.counter().statistics, attr, None )
         if currentValue is None:
            return notFoundString
         checkpointValue = 0
         if checkpoint:
            checkpointValue = getattr( checkpoint.statistics, attr, None )

         return currentValue - checkpointValue

      if not zeroOut:
         intfCountersModel.inOctets = stat( 'inOctets',
               self.ingressCountersSupported() )
         intfCountersModel.inUcastPkts = stat( 'inUcastPkts',
               self.ingressCountersSupported() )
         intfCountersModel.inMulticastPkts = stat( 'inMulticastPkts',
               self.ingressCountersSupported() and
               ( self.bumCountersSupported() or self.umCountersSupported() ) )
         intfCountersModel.inBroadcastPkts = stat( 'inBroadcastPkts',
               self.ingressCountersSupported() and self.bumCountersSupported() )

         intfCountersModel.outOctets = stat( 'outOctets',
               self.egressCountersSupported() )
         intfCountersModel.outUcastPkts = stat( 'outUcastPkts',
               self.egressCountersSupported() )
         intfCountersModel.outMulticastPkts = stat( 'outMulticastPkts',
               self.egressCountersSupported() and
               ( self.bumCountersSupported() or self.umCountersSupported() ) )
         intfCountersModel.outBroadcastPkts = stat( 'outBroadcastPkts',
               self.egressCountersSupported() and self.bumCountersSupported() )
      else:
         intfCountersModel.inOctets = 0
         intfCountersModel.inUcastPkts = 0
         intfCountersModel.inMulticastPkts = 0
         intfCountersModel.outOctets = 0
         intfCountersModel.outUcastPkts = 0
         intfCountersModel.outMulticastPkts = 0
         intfCountersModel.outBroadcastPkts = 0

      return intfCountersModel

   def ratePct( self, bps, pps ):
      """ Uses the link bandwidth to determinte the % of theoretical
      bitrate achieved including the 12 byte inter-frame gap and the
      8 byte preable.

      Returns None of bandwidth is not set"""

      maxBw = self.bandwidth() * 1000.0
      per = None
      if maxBw != 0:
         per = ( bps + 160 * pps ) * 100 / maxBw
      return per

   def rateValue( self, attr ):
      counter = self.counter()
      currentValue = getattr( counter, attr )
      return currentValue

   def updateInterfaceCountersRateModel( self ):
      if self.counter() is None: 
         return None
      intfCountersRateModel = InterfaceCountersRate( _name=self.name )
      intfCountersRateModel.description = self.description()
      intfCountersRateModel.interval = int( round(
            IntfCli.getActualLoadIntervalValue( self.config().loadInterval ) ) )
      intfCountersRateModel.inBpsRate = self.rateValue( "inBitsRate" )
      intfCountersRateModel.inPpsRate = self.rateValue( "inPktsRate" )
      intfCountersRateModel.inPktsRate = \
                     self.ratePct( intfCountersRateModel.inBpsRate,
                     intfCountersRateModel.inPpsRate )
      intfCountersRateModel.outBpsRate = self.rateValue( "outBitsRate" )
      intfCountersRateModel.outPpsRate = self.rateValue( "outPktsRate" )
      intfCountersRateModel.outPktsRate = \
                     self.ratePct( intfCountersRateModel.outBpsRate, 
                     intfCountersRateModel.outPpsRate )

      return intfCountersRateModel

   def getCounterCheckpoint( self, className=None, sessionOnly=False ):
      if not sessionOnly:
         # Counter agent manages the checkpoint
         x = subIntfCountersCheckpointDir
         if x is None:
            return None
         y = x.counterSnapshot.get( self.name )
         return y
      else:
         return IntfCli.Intf.getCounterCheckpoint( self,
               className='Interface::IntfCounter', sessionOnly=sessionOnly )

   def clearCounters( self, sessionOnly=False ):

      if not sessionOnly:
         # We have a hook to take care of global clear
         return
      else:
         IntfCli.Intf.clearCounters( self, sessionOnly=sessionOnly )

   # This method controls whether the "vlan ..." CLI commands are available in the
   # sub-interface if-mode.
   def vlanSubIntfSupported( self ):
      return True

def subIntfRuleValueFunc( mode, intf ):
   if theSubintfSupportedGuard( mode, intf ) is not None:
      raise CliParser.InvalidInputError()
   return SubIntf( intf, mode )

EthPhyAutoIntfType.cliSubIntfClazzIs( SubIntf )
subMatcher = PhysicalIntfRule.PhysicalIntfMatcher( 'Ethernet',
                                                helpdesc="Ethernet Sub Interface",
                                                subIntf=True,
                                                subIntfGuard=subintfSupportedGuard,
                                                value=subIntfRuleValueFunc )
IntfCli.Intf.addSubIntf( subMatcher, SubIntf, withIpSupport=True )

#-------------------------------------------------------------------------------
# Adds Subinterface-specific CLI commands to the "config-if" mode.
#-------------------------------------------------------------------------------
class SubIntfModelet( CliParser.Modelet ):
   def __init__( self, mode ):
      CliParser.Modelet.__init__( self )

   @staticmethod
   def shouldAddModeletRule( mode ):
      return mode.intf.isSubIntf()

   modeletParseTree = CliParser.ModeletParseTree()

#-------------------------------------------------------------------------------
# The "encapsulation dot1q vlan <vlan-id> [inner <vlan-id>]" command,
# in "config-if" mode.
#-------------------------------------------------------------------------------
def setVlanEncap( mode, args ):
   vlanId = args.get( 'VLAN' )
   innerVlanId = args.get( 'INNER_VLAN' )
   if innerVlanId:
      innerVlanId = Vlan( innerVlanId )
   dot1qEncapModeLegacy = Dot1qEncapMode.subIntfDot1qEncapConfigLegacy
   mode.intf.setVlanEncap( mode, dot1qEncapModeLegacy, vlanId, innerVlanId )

def noVlanEncap( mode, args ):
   dot1qEncapModeLegacy = Dot1qEncapMode.subIntfDot1qEncapConfigLegacy
   mode.intf.setVlanEncap( mode, dot1qEncapModeLegacy )

class EncapsulationDot1QVlanCmd( CliCommand.CliCommandClass ):
   syntax = "encapsulation dot1q vlan VLAN [ inner INNER_VLAN ]"
   noOrDefaultSyntax = "encapsulation dot1q vlan ..."
   data = {
      'encapsulation': 'configure encapsulation',
      'dot1q': '802.1q encapsulation',
      'vlan': 'VLAN tag',
      'VLAN': vlanIdMatcher,
      'inner': CliCommand.guardedKeyword( 'inner', helpdesc="Inner VLAN tag",
                                          guard=theQinQL3SubintfSupportedGuard ),
      'INNER_VLAN': CliMatcher.IntegerMatcher( 1, 4094, helpdesc="Inner VLAN ID" )
   }

   handler = setVlanEncap
   noOrDefaultHandler = noVlanEncap

SubIntfModelet.addCommandClass( EncapsulationDot1QVlanCmd )
#-------------------------------------------------------------------------------
# Adds SubIntf VLAN commands to "config-if" mode.
#-------------------------------------------------------------------------------
class SubIntfVlanModelet( CliParser.Modelet ):
   def __init__( self, mode ):
      CliParser.Modelet.__init__( self )

   @staticmethod
   def shouldAddModeletRule( mode ):
      return mode.intf.isSubIntf() and \
             mode.intf.vlanSubIntfSupported()

   modeletParseTree = CliParser.ModeletParseTree()

#--------------------------------------------------------------------------------
# [ no | default ] mac-address router MAC_ADDR
#--------------------------------------------------------------------------------
SubIntfVlanModelet.addCommandClass( IntfCli.MacAddressRouterAddrCmd )

#-------------------------------------------------------------------------------
# The "[ no | default ] mac address learning disabled" command,
# in "config-if" mode.
#
# The full syntax of this command is:
#
#   no mac address learning disabled
#   default mac address learning disabled
#   mac address learning disabled
#-------------------------------------------------------------------------------
matcherAddressKw = CliMatcher.KeywordMatcher(
   'address', helpdesc='Address keyword' )
matcherLearningKw = CliMatcher.KeywordMatcher(
   'learning', helpdesc='Learning keyword' )
matcherDisableKw = CliMatcher.KeywordMatcher(
   'disabled', helpdesc='Disable MAC address learning' )
nodeAddress = CliCommand.Node(
   matcher=matcherAddressKw, guard=theL2SubintfMacLearningDisableSupported )

class MacAddressLearningDisabledCmd( CliCommand.CliCommandClass ):
   syntax = 'mac address learning disabled'
   noOrDefaultSyntax = syntax
   data = {
      'mac' : CliToken.Mac.macMatcherForConfigIf,
      'address' : nodeAddress,
      'learning': matcherLearningKw,
      'disabled': matcherDisableKw
   }

   @staticmethod
   def handler( mode, args ):
      sic = getSwitchInputConfig( mode.intf.name )
      if not sic:
         return
      sic.macLearningEnabled = False

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      sic = getSwitchInputConfig( mode.intf.name )
      if not sic:
         return
      sic.macLearningEnabled = True

SubIntfVlanModelet.addCommandClass( MacAddressLearningDisabledCmd )

#-------------------------------------------------------------------------------
# The "vlan id <vlan-id>" command, in "config-if" mode.
# The "vlan name <vlan-name>" command, in "config-if" mode.
# The "no vlan id <trailing-garbage>" command, in "config-if" mode.
# The "no vlan id <trailing-garbage>" command, in "config-if" mode.
#-------------------------------------------------------------------------------
def noL2SubIntfVlan( sic ):
   sic.l2SubIntfVlan = Tac.Value( 'Bridging::VlanNameOrId', 'none', '', 0 )
   sic.enabled = False # Disable bridging

def setL2SubIntfVlan( mode, args ):
   no = CliCommand.isNoOrDefaultCmd( args )
   sic = getSwitchInputConfig( mode.intf.name )
   vname = args.get( 'VLAN_NAME' )
   vlanId = args.get( 'VLAN_ID' )

   if not sic:
      return

   if no :
      if ( args.get( 'name' ) and sic.l2SubIntfVlan.nameOrId == 'name' ) or \
         ( args.get( 'id' ) and sic.l2SubIntfVlan.nameOrId == 'id' ):
         noL2SubIntfVlan( sic )
      return
   elif vname:
      # Check if VLAN with given name exists. If not, show a warning but still
      # accept it.
      vlans = Vlan.getAll( mode )
      vlans = [ vlan for vlan in vlans if vlan.name( mode ) == vname ]
      if not vlans:
         mode.addWarning( 'VLAN with name %s does not exist' % vname )

      vlanNameOrId = Tac.Value( 'Bridging::VlanNameOrId', 'name', vname, 0 )
   elif vlanId:
      # Is this VLAN ID an internal VLAN? If so, don't accept in interactive
      # session only.
      vc = bridgingConfig.vlanConfig.get( vlanId.id )
      if vc and vc.internal and mode.session_.interactive_:
         mode.addError( vlanInUseErrStr( vlanId.id ) )
         return

      # Does this VLAN ID already exist? If not, create it now.
      if not vlanId.lookupConfig( mode ) and mode.session_.interactive_:
         mode.addWarning( 'VLAN does not exist. Creating vlan %d' %
                          vlanId.id )
         vlanId.create( mode )

      vlanNameOrId = Tac.Value( 'Bridging::VlanNameOrId', 'id', '', vlanId.id )
   else:
      assert 0, 'nameOrId error'

   sic.l2SubIntfVlan = vlanNameOrId
   sic.enabled = True # Enable bridging

class VlanIdOrNameCmd( CliCommand.CliCommandClass ):
   syntax = 'vlan ( id VLAN_ID ) | ( name VLAN_NAME )'
   noOrDefaultSyntax = 'vlan ( id | name ) ...'
   data = {
      'vlan': CliCommand.guardedKeyword( 'vlan',
                                         helpdesc='Configure VLAN attributes',
                                         guard=theL2SubintfSupportedGuard,
                                         canMerge=False ),
      'id': 'Identifier for a Virtual LAN',
      'VLAN_ID': vlanIdMatcher,
      'name': 'Name of the VLAN',
      'VLAN_NAME': vlanNameMatcher,
   }

   handler = setL2SubIntfVlan
   noOrDefaultHandler = handler

SubIntfVlanModelet.addCommandClass( VlanIdOrNameCmd )

#-------------------------------------------------------------------------------
# Associate the SubIntfModelet with the "config-if" mode.
#-------------------------------------------------------------------------------
IntfCli.IntfConfigMode.addModelet( SubIntfModelet )
IntfCli.IntfConfigMode.addModelet( SubIntfVlanModelet )

#-------------------------------------------------------------------------------
# Clear counters hook to clear subIntf counters
#-------------------------------------------------------------------------------
def clearCountersHook( mode, intfs, sessionOnly, allIntfs ):
   request = aleCliConfig.clearCountersRequest
   
   # We don't support per-session clear
   if sessionOnly:
      return

   if allIntfs:
      del request[ 'all' ]
      request[ 'all' ] = True
      return

   # Clear interface list
   for x in intfs:
      del request[ x.name ]
      request[ x.name ] = True

#-------------------------------------------------------------------------------
# A subinterface variant to the "show tech-support" cli
#-------------------------------------------------------------------------------
nodeSubIntf = CliCommand.guardedKeyword( 'subinterface',
      helpdesc='Show subinterface specific information',
      guard=theSubintfSupportedGuard )

def showTechSupportSubIntf( mode, args ):
   vrfs = allVrfConfig.vrf.keys() + [ 'default' ]
   arpCmds = [ 'show ip arp vrf ' + vrf for vrf in vrfs ]
   v6NeighborCmds = [ 'show ipv6 neighbors vrf ' + vrf for vrf in vrfs ]
   platform = getPlatformName( mode.sysname )
   platformCmds = []
   if platform == 'Sand':
      platformCmds = [
                       'show platform fap subinterface mapping',
                       'show platform fap subinterface vlan',
                     ]
   if platform == 'Strata':
      platformCmds = [
                       'show platform trident l2 shadow vlan-xlate',
                       'show platform trident l2 shadow egr-vlan-xlate',
                     ]
   cmds = [
            'show running-config sanitized',
            'show vrf',
            'show interfaces',
            'show ip interface brief',
            'show ipv6 interface brief',
          ] + arpCmds + v6NeighborCmds + platformCmds

   for cmd in cmds:
      print "\n------------- %s -------------\n"  % cmd
      sys.stdout.flush()
      try:
         mode.session_.runCmd( cmd, aaa=False )
      except CliParser.GuardError, e:
         print "(unavailable: %s)" % e.guardCode
      except CliParser.AlreadyHandledError:
         pass
      except KeyboardInterrupt:
         raise
      except:
         print "(unavailable)"
      TacSigint.check()

class showTechSupportSubIntfCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show tech-support subinterface'

   data = {
      'tech-support': TechSupportCli.techSupportKwMatcher,
      'subinterface': nodeSubIntf,
   }

   handler = showTechSupportSubIntf
   privileged = True

BasicCli.addShowCommandClass( showTechSupportSubIntfCmd )

def mountCounters():
   subIntfStatusDir.counterDir = subIntfCountersDir

#-------------------------------------------------------------------------------
# Have the Cli Agent mount all needed state from sysdb
#-------------------------------------------------------------------------------
def Plugin( entityManager ):
   global subIntfConfigDir, subIntfStatusDir, bridgingHwEnabled, bridgingInputConfig
   global bridgingConfig, bridgingHwCapabilities
   global allEthIntfConfigDir, allIntfStatusDir, intfCapability, allVrfConfig
   global aleCliConfig
   global subIntfCountersDir
   global subIntfCountersCheckpointDir
   global subIntfPwStatus

   subIntfConfigDir = ConfigMount.mount( entityManager, "interface/config/subintf",
                                         "Interface::SubIntfConfigDir", "w" )
   subIntfPwStatus = LazyMount.mount( entityManager, "pseudowire/agent/subintf",
                                      "Interface::SubIntfPwStatus", "r" )
   bridgingHwEnabled = LazyMount.mount( entityManager, "bridging/hwenabled",
                                        "Bridging::HwEnabled", "r" )
   bridgingHwCapabilities = LazyMount.mount( entityManager,
                                             "bridging/hwcapabilities",
                                             "Bridging::HwCapabilities", "r" )
   bridgingInputConfig = ConfigMount.mount( entityManager,
                                            "bridging/input/config/cli",
                                            "Bridging::Input::CliConfig", "w" )
   bridgingConfig = LazyMount.mount( entityManager, "bridging/config",
                                     "Bridging::Config", "r" )
   allEthIntfConfigDir = ConfigMount.mount( entityManager,
                                            "interface/config/eth/phy/slice",
                                            "Tac::Dir", "w" )
   allIntfStatusDir = LazyMount.mount( entityManager, "interface/status/all",
                                       "Interface::AllIntfStatusDir", "r" )
   intfCapability = LazyMount.mount( entityManager,
                                     "interface/hardware/capability",
                                     "Interface::Hardware::Capability", "r" )
   allVrfConfig = LazyMount.mount( entityManager, "ip/vrf/config",
                                   "Ip::AllVrfConfig", "r" )
   aleCliConfig = LazyMount.mount( entityManager, "hardware/ale/cliconfig",
                                   "Ale::HwCliConfig", "w" )
   
   shmemEm = SharedMem.entityManager( sysdbEm=entityManager )
   smi = Smash.mountInfo( 'reader' )
   subIntfCountersDir = shmemEm.doMount( 'interface/counter/subintf/current', 
         'Smash::Interface::AllIntfCounterDir', smi )
   subIntfCountersCheckpointDir = shmemEm.doMount(
      'interface/counter/subintf/snapshot',
      'Smash::Interface::AllIntfCounterSnapshotDir', smi )

   mg = entityManager.mountGroup()
   subIntfStatusDir = mg.mount( "interface/status/subintf", 
                                "Interface::SubIntfStatusDir", "r" )
   mg.close( mountCounters )
