#!/usr/bin/env python
# Copyright (c) 2007, 2008, 2009, 2010 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

#-------------------------------------------------------------------------------
# This module implements interface-specific Stp configuration.
#
# [no|default] spanning-tree link-type { point-to-point | shared }
# [no|default] spanning-tree [ mst <instance-id> ] cost <1-200,000,000>
# [no|default] spanning-tree [ mst <instance-id> ] port-priority <0-240>
# [no|default] spanning-tree portfast
# spanning-tree portfast {network, edge, normal, auto}
# no spanning-tree portfast {network, edge, auto}
# [no|default] spanning-tree bpduguard { enable | disable }
# [no|default] spanning-tree bpdufilter { enable | disable }
#-------------------------------------------------------------------------------
'''Interface configuration commands supported for Spanning Tree'''
import CliParser
import BasicCli
import LazyMount
import Tracing
from VlanCli import switchportEligible, vlanSetMatcher
import CliMatcher
from MultiRangeRule import MultiRangeMatcher
import CliCommand
import ConfigMount

from StpCliLib import stpPortConfig, loopGuardCliGuard
# Import Stp TAC object accessors, which create TAC objects if needed.
from StpCliUtil import (
      PortPriorityUnconfigured,
      pvstInstName,
      pvstVlanMstiPortConfig,
      stpConfigIs,
      stpInputConfig,
      stpInputConfigIs,
      stpMstiPortConfig,
      stpPortConfigByName,
      stpStpiPortConfig,
)

from StpConst import (
      PathCostMin,
      PathCostMax,
      PathCostUnknown,
      PortPriorityMin,
      PortPriorityMax,
      PortPriorityDefault,
      PortPriorityStepSize,
)

from StpCli import (
      matcherBpdufilter,
      matcherBpduguard,
)

from IntfCli import LoggingEventGlobalCmd, LoggingEventIntfCmd, IntfConfigMode
import TaggedPdu


__defaultTraceHandle__ = Tracing.Handle( 'StpCli' )

#-------------------------------------------------------------------------------
# Adds spanning-tree commands to the "config-if" mode.
#-------------------------------------------------------------------------------
class SpanningTreeModelet ( CliParser.Modelet ):
   def __init__( self, mode ):
      CliParser.Modelet.__init__( self )

   @staticmethod
   def shouldAddModeletRule( mode ):
      return switchportEligible( mode )

   modeletParseTree = CliParser.ModeletParseTree()

modelet = SpanningTreeModelet 

#-------------------------------------------------------------------------------
# Stp interface configuration token
#-------------------------------------------------------------------------------
matcherMstInstanceId = MultiRangeMatcher( rangeFn=lambda: ( 0, 4094 ),
      noSingletons=False,
      helpdesc='Specify a range of MST instances' )
matcherPathCost = CliMatcher.IntegerMatcher( PathCostMin, PathCostMax, 
         helpdesc='Change the interface\'s spanning tree path cost' )
matcherPriority = CliMatcher.IntegerMatcher( PortPriorityMin, PortPriorityMax,
         helpdesc='Port priority in increments of ' + str( PortPriorityStepSize ) )

#-------------------------------------------------------------------------------
# [no|default] spanning-tree link-type { point-to-point | shared }
#-------------------------------------------------------------------------------
class StpLinkTypeCmd( CliCommand.CliCommandClass ):
   syntax = 'spanning-tree link-type ( point-to-point | shared )'
   noOrDefaultSyntax = 'spanning-tree link-type ...'
   data = {
      'spanning-tree' : 'Spanning tree protocol',
      'link-type' : 'Specify a link type for spanning tree protocol use',
      'point-to-point' : 'Consider the interface as point-to-point',
      'shared' : 'Consider the interface as shared',
   }

   @staticmethod
   def handler ( mode, args ):
      portConfig = stpPortConfig( mode, True )
      linkType = 'forceTrue' if args.get( 'point-to-point' ) else 'forceFalse'
      portConfig.adminPointToPointMac = linkType 
   
   @staticmethod
   def noOrDefaultHandler ( mode, args ):
      portConfig = stpPortConfig( mode, False )
      if portConfig is not None:
         portConfig.adminPointToPointMac = portConfig.defaultAdminPointToPoint

modelet.addCommandClass( StpLinkTypeCmd )

#-------------------------------------------------------------------------------
# taggedPdu config 
# [no] l2-protocol encapsulation vlan tag <tag>
#-------------------------------------------------------------------------------
def setStpTaggedPdu( mode, vlanId ):
   portConfig = stpPortConfig( mode, True )
   portConfig.acceptTaggedPdu = True
   portConfig.allowedTag = vlanId.id

def clearStpTaggedPdu( portConfig ):
   if portConfig:
      portConfig.acceptTaggedPdu = False
      portConfig.allowedTag = 0

def noStpTaggedPdu( mode ):
   portConfig = stpPortConfig( mode, False )
   clearStpTaggedPdu( portConfig )

def delIntfStpTaggedPdu( intf ):
   portConfig = stpPortConfigByName( intf.name )
   clearStpTaggedPdu( portConfig )

TaggedPdu.taggedPduConfigHook.addExtension( setStpTaggedPdu )
TaggedPdu.noTaggedPduConfigHook.addExtension( noStpTaggedPdu )
TaggedPdu.taggedPduIntfDeletionHook.addExtension( delIntfStpTaggedPdu )
#-------------------------------------------------------------------------------
# [no|default] spanning-tree [ mst <instance-id> ] cost <cost>
#-------------------------------------------------------------------------------
class StpIntfCostCmd( CliCommand.CliCommandClass ):
   syntax = 'spanning-tree [ mst INSTANCE_IDS ] cost COST'
   noOrDefaultSyntax = 'spanning-tree [ mst INSTANCE_IDS ] cost ...'
   data = {
      'spanning-tree' : 'Spanning tree protocol',
      'mst' : 'Specify a range of MST Instances',
      'INSTANCE_IDS' : matcherMstInstanceId,
      'cost' : 'Change the interface\'s spanning tree path cost',
      'COST' : matcherPathCost,
   }

   @staticmethod
   def handler ( mode, args ):
      instIds = args.get( 'INSTANCE_IDS' )
      cost = args[ 'COST' ]
      if instIds is None:
         # Not specifying an instance means to configure the external path
         # cost. Configure the PortConfig value.
         Tracing.trace2( 'Configuring the PortConfig value', cost )
         portConfig = stpPortConfig( mode, True )
         portConfig.extPathCost = cost
      else:
         # MST specific, so it means to configure the internal path cost.
         # Configure the MstiPortConfig value
         Tracing.trace2( 'Configuring the MstiPortConfig value', cost )
         for instId in instIds.values():
            mstiPortConfig = stpMstiPortConfig( mode, instId, True )
            mstiPortConfig.intPathCost = cost
   
   @staticmethod
   def noOrDefaultHandler ( mode, args ):
      instIds = args.get( 'INSTANCE_IDS' )
      cost = PathCostUnknown
      if instIds is None:
         # Must be Rstp. Configure the PortConfig value
         portConfig = stpPortConfig( mode, False )
         if portConfig is not None:
            portConfig.extPathCost = cost
      else:
         # Must be MST. Configure the MstiPortConfig value
         for instId in instIds.values():
            mstiPortConfig = stpMstiPortConfig( mode, instId, False )
            if mstiPortConfig is not None:
               mstiPortConfig.intPathCost = cost

modelet.addCommandClass( StpIntfCostCmd )

#-------------------------------------------------------------------------------
# [no|default] spanning-tree vlan <vlan-list> cost <cost>
#-------------------------------------------------------------------------------
class StpVlanCostCmd( CliCommand.CliCommandClass ):
   syntax = 'spanning-tree vlan VLANS cost COST'
   noOrDefaultSyntax = 'spanning-tree vlan VLANS cost ...'
   data = {
      'spanning-tree' : 'Spanning tree protocol',
      'vlan' : 'Specify a range of VLANs',
      'VLANS' : vlanSetMatcher,
      'cost' : 'Change the interface\'s spanning tree path cost in VLAN(s)',
      'COST' : matcherPathCost,
   }

   @staticmethod
   def handler ( mode, args ):
      cost = args[ 'COST' ]
      Tracing.trace2( 'Configuring the MstiPortConfig value in VLAN', cost )
      vlanSet = args[ 'VLANS' ]
      for vlan in vlanSet.ids:
         stpiName = pvstInstName( vlan )
         stpiPortConfig = stpStpiPortConfig( mode, stpiName, True )
         stpiPortConfig.extPathCost = args[ 'COST' ] 

   @staticmethod
   def noOrDefaultHandler ( mode, args ):
      cost = PathCostUnknown
      Tracing.trace2( 'Configuring the MstiPortConfig value', cost )
      vlanSet = args[ 'VLANS' ]
      for vlan in vlanSet.ids:
         stpiName = pvstInstName( vlan )
         stpiPortConfig = stpStpiPortConfig( mode, stpiName, False )
         if stpiPortConfig is not None:
            stpiPortConfig.extPathCost = cost

modelet.addCommandClass( StpVlanCostCmd )

#-------------------------------------------------------------------------------
# [no|default] spanning-tree [ mst <instance-id> ] port-priority <0-240>
#-------------------------------------------------------------------------------d

# Helper function to make sure port priorty passed is valid
def validatePortPriority( mode, portPriority ):
   if ( portPriority % PortPriorityStepSize ) != 0:
      mode.addError( "Port Priority in increments of %d is required" % (
         PortPriorityStepSize ) )
      raise CliParser.AlreadyHandledError
   return True
            
class StpPortPriorityCmd( CliCommand.CliCommandClass ):
   syntax = 'spanning-tree [ mst INSTANCE_IDS ] port-priority PRIO'
   noOrDefaultSyntax = 'spanning-tree [ mst INSTANCE_IDS ] port-priority ...'
   data = {
      'spanning-tree' : 'Spanning tree protocol',
      'mst' : 'Specify a range of MST Instances',
      'INSTANCE_IDS' : matcherMstInstanceId,
      'port-priority' : 'Change the interface\'s spanning tree port priority',
      'PRIO' : matcherPriority,
   }

   @staticmethod
   def handler ( mode, args ):
      prio = args[ 'PRIO' ]
      validatePortPriority( mode, prio )

      instIds = args.get( 'INSTANCE_IDS' )
      if instIds is not None:
         for instId in instIds.values():
            mstiPortConfig = stpMstiPortConfig( mode, instId, True )
            mstiPortConfig.portPriority = prio 
      else:
         portConfig = stpPortConfig( mode, True )
         portConfig.portPriority = prio 

   @staticmethod
   def noOrDefaultHandler ( mode, args ):
      cost = PathCostUnknown
      Tracing.trace2( 'Configuring the MstiPortConfig value', cost )
      instIds = args.get( 'INSTANCE_IDS' )
      if instIds is not None:
         for instId in instIds.values():
            mstiPortConfig = stpMstiPortConfig( mode, instId, False )
            if mstiPortConfig is not None:
               mstiPortConfig.portPriority = PortPriorityUnconfigured
      else:
         portConfig = stpPortConfig( mode, False )
         if portConfig is not None:
            portConfig.portPriority = PortPriorityDefault

modelet.addCommandClass( StpPortPriorityCmd )

#-------------------------------------------------------------------------------
# [no|default] spanning-tree vlan <vlan-list> port-priority <0-240>
#-------------------------------------------------------------------------------d
class StpVlanPortPriorityCmd( CliCommand.CliCommandClass ):
   syntax = 'spanning-tree vlan VLANS port-priority PRIO'
   noOrDefaultSyntax = 'spanning-tree vlan VLANS port-priority ...'
   data = {
      'spanning-tree' : 'Spanning tree protocol',
      'vlan' : 'Specify a range of VLANs',
      'VLANS' : vlanSetMatcher,
      'port-priority' : 'Change the interface\'s spanning tree port priority',
      'PRIO' : matcherPriority,
   }

   @staticmethod
   def handler ( mode, args ):
      prio = args[ 'PRIO' ]
      validatePortPriority( mode, prio )

      vlanSet = args[ 'VLANS' ]
      for vlanId in vlanSet.ids:
         mstiPortConfig = pvstVlanMstiPortConfig( mode, vlanId, create=True )
         mstiPortConfig.portPriority = prio 

   @staticmethod
   def noOrDefaultHandler ( mode, args ):
      vlanSet = args[ 'VLANS' ]
      Tracing.trace2( 'Unconfiguring the MstiPortConfig priority value for vlans',
            vlanSet.ids )
      for vlanId in vlanSet.ids:
         mstiPortConfig = pvstVlanMstiPortConfig( mode, vlanId, False )
         if mstiPortConfig is not None:
            Tracing.trace2( 'Unconfiguring the MstiPortConfig priority value',
                  vlanId, mstiPortConfig.name )
            mstiPortConfig.portPriority = PortPriorityUnconfigured
         else:
            Tracing.trace2( 'Cannot unconfigure the MstiPortConfig',
                            'priority value (no MstiPortConfig)',
                            vlanId, mode.intf.name )

modelet.addCommandClass( StpVlanPortPriorityCmd )

#-------------------------------------------------------------------------------
# [no|default] spanning-tree portfast
#-------------------------------------------------------------------------------
class StpPortfastCmd( CliCommand.CliCommandClass ):
   syntax = 'spanning-tree portfast'
   noOrDefaultSyntax = syntax
   data = {
      'spanning-tree' : 'Spanning tree protocol',
      'portfast' : 'Enable an interface to move directly to forwarding on link up',
   }

   @staticmethod
   def handler ( mode, args ):
      Tracing.trace2( 'Configuring adminEdgePort' )

      portConfig = stpPortConfig( mode, True )
      portConfig.networkPort = False
      
      printWarning = True
      # Display the warning for only the first interface in the configured range
      intfRange = mode.intfRange()
      if intfRange is not None and intfRange.individualIntfModes_:
         if intfRange.individualIntfModes_[0] is not mode:
            printWarning = False

      if printWarning:
         mode.addWarning( 'portfast should only be enabled on ports '
                          'connected to a single host. Connecting hubs, '
                          'concentrators, switches, bridges, etc. to this '
                          'interface when portfast is enabled can cause '
                          'temporary bridging loops. Use with CAUTION.' )
      
      portConfig.adminEdgePort = True 
   
   @staticmethod
   def noOrDefaultHandler ( mode, args ):
      portConfig = stpPortConfig( mode, True )
      portConfig.autoEdgePort = True
      portConfig.adminEdgePort = portConfig.adminEdgePortDefault

modelet.addCommandClass( StpPortfastCmd )

#-------------------------------------------------------------------------------
# [no|default] spanning-tree portfast auto
#-------------------------------------------------------------------------------
class StpPortfastAutoCmd( CliCommand.CliCommandClass ):
   syntax = 'spanning-tree portfast auto'
   noOrDefaultSyntax = syntax
   data = {
      'spanning-tree' : 'Spanning tree protocol',
      'portfast' : 'Enable an interface to move directly to forwarding on link up',
      'auto' : 'Enable automatic detection of edge ports',
   }

   @staticmethod
   def handler ( mode, args ):
      portConfig = stpPortConfig( mode, True )
      portConfig.networkPort = False
      portConfig.autoEdgePort = True
   
   @staticmethod
   def noHandler ( mode, args ):
      portConfig = stpPortConfig( mode, True )
      portConfig.autoEdgePort = False

   defaultHandler = handler

modelet.addCommandClass( StpPortfastAutoCmd )

#-------------------------------------------------------------------------------
# [no|default] spanning-tree portfast network
#-------------------------------------------------------------------------------
class StpPortfastNetworkCmd( CliCommand.CliCommandClass ):
   syntax = 'spanning-tree portfast network'
   noOrDefaultSyntax = syntax
   data = {
      'spanning-tree' : 'Spanning tree protocol',
      'portfast' : 'Enable an interface to move directly to forwarding on link up',
      'network' : 'Configure the interface as an inter-switch link',
   }

   @staticmethod
   def handler ( mode, args ):
      portConfig = stpPortConfig( mode, True )
      portConfig.networkPort = True
      portConfig.adminEdgePort = False
      portConfig.autoEdgePort = False
   
   @staticmethod
   def noOrDefaultHandler ( mode, args ):
      portConfig = stpPortConfig( mode, True )
      portConfig.networkPort = False
      portConfig.adminEdgePort = False
      portConfig.autoEdgePort = True

modelet.addCommandClass( StpPortfastNetworkCmd )

#-------------------------------------------------------------------------------
# spanning-tree portfast normal
#-------------------------------------------------------------------------------
class StpPortfastNormalCmd( CliCommand.CliCommandClass ):
   syntax = 'spanning-tree portfast normal'
   data = {
      'spanning-tree' : 'Spanning tree protocol',
      'portfast' : 'Enable an interface to move directly to forwarding on link up',
      'normal' : 'Configure the interface as a normal spanning tree port',
   }

   @staticmethod
   def handler ( mode, args ):
      portConfig = stpPortConfig( mode, True )
      portConfig.networkPort = False
      portConfig.adminEdgePort = False
      portConfig.autoEdgePort = True

modelet.addCommandClass( StpPortfastNormalCmd )
      
#-------------------------------------------------------------------------------
# [no|default] spanning-tree portfast edge
#-------------------------------------------------------------------------------
class StpPortfastEdgeCmd( CliCommand.CliCommandClass ):
   syntax = 'spanning-tree portfast edge'
   noOrDefaultSyntax = syntax
   data = {
      'spanning-tree' : 'Spanning tree protocol',
      'portfast' : 'Enable an interface to move directly to forwarding on link up',
      'edge' : 'Configure the interface as an edge port (enable portfast)',
   }

   @staticmethod
   def handler ( mode, args ):
      portConfig = stpPortConfig( mode, True )
      portConfig.networkPort = False
      portConfig.adminEdgePort = True 

   @staticmethod
   def noHandler( mode, args ):
      portConfig = stpPortConfig( mode, True )
      portConfig.networkPort = False
      portConfig.adminEdgePort = False
      
   @staticmethod
   def defaultHandler( mode, args ):
      portConfig = stpPortConfig( mode, True )
      portConfig.networkPort = False
      portConfig.adminEdgePort = portConfig.adminEdgePortDefault

modelet.addCommandClass( StpPortfastEdgeCmd )

#-------------------------------------------------------------------------------
# [no|default] spanning-tree bpduguard { enable | disable }
#-------------------------------------------------------------------------------
def setBpduguard( mode, bpduguardSetting ):
   Tracing.trace2(  'Configuring bpduguard to %s', bpduguardSetting )
   portConfig = stpPortConfig( mode, True )
   if bpduguardSetting == 'enable':
      portConfig.bpduguard = 'bpduguardEnabled'
   else:
      portConfig.bpduguard = 'bpduguardDisabled'
   
def noBpduguard( mode, setting=None ):
   Tracing.trace2(  'Removing bpduguard setting' )
   portConfig = stpPortConfig( mode, True )
   portConfig.bpduguard = 'bpduguardDefault'

class StpBpduguardCmd( CliCommand.CliCommandClass ):
   syntax = 'spanning-tree bpduguard ( enable | disable )'
   # having the following be equal to syntax is wrong in terms of what the user
   # should enter, and adding the ellipses results in matching errors with the
   # ratelimit command below
   noOrDefaultSyntax = 'spanning-tree bpduguard [ enable | disable ]'
   data = {
      'spanning-tree' : 'Spanning tree protocol',
      'bpduguard' : matcherBpduguard,
      'enable' : 'Enable BPDU Guard',
      'disable' : 'Disable BPDU Guard',
   }

   @staticmethod
   def handler ( mode, args ):
      bpduguardSetting = args.get( 'enable', 'disable' )
      Tracing.trace2( 'Configuring bpduguard to', bpduguardSetting )
      portConfig = stpPortConfig( mode, True )
      if bpduguardSetting == 'enable':
         portConfig.bpduguard = 'bpduguardEnabled'
      else:
         portConfig.bpduguard = 'bpduguardDisabled'

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      Tracing.trace2( 'Removing bpduguard setting' )
      portConfig = stpPortConfig( mode, True )
      portConfig.bpduguard = 'bpduguardDefault'

modelet.addCommandClass( StpBpduguardCmd )
#-------------------------------------------------------------------------------
# [no|default] spanning-tree guard { loop | none | root }
#-------------------------------------------------------------------------------
class StpGuardCmd( CliCommand.CliCommandClass ):
   syntax = 'spanning-tree guard ( loop | none | root )'
   noOrDefaultSyntax = 'spanning-tree guard ...'
   data = {
      'spanning-tree' : 'Spanning tree protocol',
      'guard' : 'Change an interface\'s spanning tree guard mode',
      'loop' : CliCommand.Node(
         matcher=CliMatcher.KeywordMatcher( 'loop',
            helpdesc='Set guard mode to loop guard on interface' ),
         guard=loopGuardCliGuard ),
      'none' : 'Set guard mode to none',
      'root' : 'Set guard mode to root guard on interface',
   }

   @staticmethod
   def handler ( mode, args ):
      portConfig = stpPortConfig( mode, True )
      if 'loop' in args:
         portConfig.guard = 'loopguardEnabled'
      elif 'root' in args:
         portConfig.guard = 'rootguardEnabled'
      elif 'none' in args:
         portConfig.guard = 'guardDisabled'
      Tracing.trace2( 'Configuring guard to', portConfig.guard )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      Tracing.trace2( 'Removing loopguard setting (using global default)' )
      portConfig = stpPortConfig( mode, True )
      portConfig.guard = 'guardDefault'

modelet.addCommandClass( StpGuardCmd )

#-------------------------------------------------------------------------------
# [no|default] spanning-tree bpduguard rate-limit {enable | disable}
# [no|default] spanning-tree bpduguard rate-limit count <c> [ interval <i> ]
#-------------------------------------------------------------------------------
class StpBpduguardRateLimitCmd( CliCommand.CliCommandClass ):
   syntax = 'spanning-tree bpduguard rate-limit ( enable | disable )'
   # the following is different due to the ... rate-limit count command
   noOrDefaultSyntax = 'spanning-tree bpduguard rate-limit [ enable | disable ]'
   data = {
      'spanning-tree' : 'Spanning tree protocol',
      'bpduguard' : matcherBpduguard,
      'rate-limit' : 'BPDU Input Rate Limiter options',
      'enable' : 'Enable rate limiter on this port',
      'disable' : 'Disable rate limiter on this port',
   }

   @staticmethod
   def handler ( mode, args ):
      portConfig = stpPortConfig( mode, True )
      if 'enable' in args:
         portConfig.rateLimitEnabled = 'rateLimitOn'
      elif 'disable' in args:
         portConfig.rateLimitEnabled = 'rateLimitOff'

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      portConfig = stpPortConfig( mode, True )
      portConfig.rateLimitEnabled = 'rateLimitDefault'

modelet.addCommandClass( StpBpduguardRateLimitCmd )

class StpBpduguardRateLimitCountCmd( CliCommand.CliCommandClass ):
   syntax = 'spanning-tree bpduguard rate-limit count COUNT [ interval INTERVAL ]'
   noOrDefaultSyntax = 'spanning-tree bpduguard rate-limit count ...' 
   data = {
      'spanning-tree' : 'Spanning tree protocol',
      'bpduguard' : matcherBpduguard,
      'rate-limit' : 'BPDU Input Rate Limiter options',
      'count' : 'Set the max number of BPDUs per timer interval',
      'COUNT' : CliMatcher.IntegerMatcher( 1, 20000, 
         helpdesc='Max number of BPDUs per timer interval' ),
      'interval' : 'Set the timer interval',
      'INTERVAL' : CliMatcher.IntegerMatcher( 1, 15,
         helpdesc='Number of seconds in the BPDU Input Rate Limiter Timer' ),
   }

   @staticmethod
   def handler ( mode, args ):
      portConfig = stpPortConfig( mode, True )
      portConfig.rateLimitMaxCount = args[ 'COUNT' ]
      interval = args.get( 'INTERVAL' )
      if interval is not None: 
         portConfig.rateLimitInterval = interval

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      portConfig = stpPortConfig( mode, True )
      portConfig.rateLimitMaxCount = 0
      portConfig.rateLimitInterval = 0

modelet.addCommandClass( StpBpduguardRateLimitCountCmd )

#-------------------------------------------------------------------------------
# [no|default] spanning-tree bpdufilter { enable | disable }
#-------------------------------------------------------------------------------
class StpBpdufilterCmd( CliCommand.CliCommandClass ):
   syntax = 'spanning-tree bpdufilter ( enable | disable )'
   noOrDefaultSyntax = 'spanning-tree bpdufilter ...' 
   data = {
      'spanning-tree' : 'Spanning tree protocol',
      'bpdufilter' : matcherBpdufilter,
      'enable' : 'Enable BPDU Filter',
      'disable' : 'Disable BPDU Filter',
   }

   @staticmethod
   def handler ( mode, args ):
      bpdufilterSetting = args.get( 'enable', 'disable' )
      Tracing.trace2( 'Configuring bpdufilter to', bpdufilterSetting )
      portConfig = stpPortConfig( mode, True )
      if bpdufilterSetting == 'enable':
         portConfig.bpdufilter = 'bpdufilterEnabled' 
      else:
         portConfig.bpdufilter = 'bpdufilterDisabled' 

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      Tracing.trace2(  'Removing bpdufilter setting' )
      portConfig = stpPortConfig( mode, True )
      portConfig.bpdufilter = 'bpdufilterDefault' 

modelet.addCommandClass( StpBpdufilterCmd )

#-------------------------------------------------------------------------------
# The "[no] logging event spanning-tree" command.
# The "logging event spanning-tree use-global" command.
#-------------------------------------------------------------------------------
class LoggingEventStpIntf( LoggingEventIntfCmd ):
   syntax = "logging event spanning-tree"
   data = {
      "spanning-tree" : 'Spanning tree messages'
      }
   @classmethod
   def _enableHandler( cls, mode, args ):
      portConfig = stpPortConfig( mode, True )
      portConfig.spanTreeLogging = 'on'

   @classmethod
   def _disableHandler( cls, mode, args ):
      portConfig = stpPortConfig( mode, True )
      portConfig.spanTreeLogging = 'off'

   @classmethod
   def _useGlobalHandler( cls, mode, args ):
      portConfig = stpPortConfig( mode, True )
      portConfig.spanTreeLogging = 'useGlobal'

modelet.addCommandClass( LoggingEventStpIntf )

#-------------------------------------------------------------------------------
# The "[no|default] logging event spanning-tree global" command, in "config" mode.
#-------------------------------------------------------------------------------
class LoggingEventStpGlobal( LoggingEventGlobalCmd ):
   syntax = "logging event spanning-tree global"
   noOrDefaultSyntax = syntax
   data = {
      'spanning-tree' : 'Spanning tree messages',
      'global' : 'Global spanning tree message configuration'
   }
   @staticmethod
   def handler( mode, args ):
      config = stpInputConfig()
      config.spanTreeLogging = 'on'

   @staticmethod
   def noHandler( mode, args ):
      config = stpInputConfig()
      config.spanTreeLogging = 'off'

   # default is on
   defaultHandler = handler

BasicCli.GlobalConfigMode.addCommandClass( LoggingEventStpGlobal )

#-------------------------------------------------------------------------------
# Associate the SpanningTreeModelet with the "config-if" mode.
#-------------------------------------------------------------------------------
IntfConfigMode.addModelet( SpanningTreeModelet )

#-------------------------------------------------------------------------------
# Have the Cli Agent mount all needed state from sysdb
#-------------------------------------------------------------------------------
def Plugin( entityManager ):
   config = LazyMount.mount( entityManager, "stp/config", "Stp::Config", "r" )
   stpConfigIs( config )
   cliConfig = ConfigMount.mount( entityManager, "stp/input/config/cli",
                                  "Stp::Input::Config", "w" )
   stpInputConfigIs( cliConfig )
