#!/usr/bin/env python
# Copyright (c) 2007, 2008, 2009, 2010 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

#-------------------------------------------------------------------------------
# This module implements global Stp configuration.
#
#     [no] spanning-tree forward-time <4-30>
#     [no] spanning-tree max-age <6-40>
#     [no] spanning-tree max-hops <1-255>
#     [no] spanning-tree mode { rstp | mstp | none | backup | rapid-pvst }
#     [no] spanning-tree mst configuration
#     [no] spanning-tree [ mst <instance-id> ] root { primary | secondary }
#     [no] spanning-tree [ mst <instance-id> ] priority <0-61440>
#     [no] spanning-tree bpdu tx hold-count <1-10>
#     legacy: [no] spanning-tree transmit hold-count <1-10>
#     [no] spanning-tree edge-port bpduguard default
#     legacy: [no] spanning-tree portfast bpduguard default
#     [no] spanning-tree portchannel/etherchannel guard misconfig
#-------------------------------------------------------------------------------
'''Configuration commands supported for Spanning Tree'''
import BasicCli
import CliCommand
import CliMatcher
import CliParser
import ConfigMount
import ShowCommand
import LazyMount
import Tracing
import VlanCli
import Tac
from MultiRangeRule import MultiRangeMatcher

# Import Stp TAC object accessors, which create TAC objects if needed.
import StpCliLib
from StpCliLib import (
      MstInstCountMax,
      MstCistInstId,
      MstRevisionDefault
)
from StpCliLib import stpInputConfig
from CliMode.Stp import MstMode
from StpShowLib import getMstInfoCapi
from StpModels import SpanningTreeMstConfiguration
from TypeFuture import TacLazyType

from Toggles.StpToggleLib import toggleStpSuperRootEnabled

__defaultTraceHandle__ = Tracing.Handle( 'StpCli' )

TrapFeatureName = TacLazyType( 'Arnet::TrapFeatureName' )
# ConfigMount anchors
trapConfig = None

# LazyMount anchors
bridgingHwCapabilities = None
topoStatus = None

#-------------------------------------------------------------------------------
# Aliases for some useful token rules.
#-------------------------------------------------------------------------------
matcherMstInstanceId = MultiRangeMatcher( rangeFn=lambda: ( 0, 4094 ),
      noSingletons=False,
      helpdesc='Specify a range of MST instances' )
matcherVlanId = CliMatcher.KeywordMatcher( 'vlan-id',
      helpdesc='Specify VLANs to enable or disable' )
deprecatedNodeVlan = CliCommand.Node(
      matcher=CliMatcher.KeywordMatcher( 'vlan',
         helpdesc='Specify VLANs to enable or disable' ),
      deprecatedByCmd='spanning-tree vlan-id' )
matcherPriority = CliMatcher.IntegerMatcher( 0, 61440, 
            helpdesc='Bridge priority in increments of 4096' )
#--------------------------------------------------------------------------------
# [ no | default ] spanning-tree forward-time FORWARD_TIME
#--------------------------------------------------------------------------------
class ForwardTimeCmd( CliCommand.CliCommandClass ):
   syntax = 'spanning-tree forward-time FORWARD_TIME'
   noOrDefaultSyntax = 'spanning-tree forward-time ...'
   data = {
      'spanning-tree' : 'Spanning tree protocol',
      'forward-time' : 'Set the forward delay for the spanning tree',
      'FORWARD_TIME' : CliMatcher.IntegerMatcher( 4, 30,
         helpdesc='Number of seconds for the forward delay timer' ),
   }

   @staticmethod
   def handler( mode, args ):
      config = stpInputConfig()
      config.bridgeForwardDelay = args[ 'FORWARD_TIME' ]

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      config = stpInputConfig()
      config.bridgeForwardDelay = config.defaultBridgeForwardDelay

BasicCli.GlobalConfigMode.addCommandClass( ForwardTimeCmd )

#--------------------------------------------------------------------------------
# [ no | default ] spanning-tree hello-time HELLO_TIME
#--------------------------------------------------------------------------------
class HelloTimeCmd( CliCommand.CliCommandClass ):
   syntax = 'spanning-tree hello-time HELLO_TIME'
   noOrDefaultSyntax = 'spanning-tree hello-time ...'
   data = {
      'spanning-tree' : 'Spanning tree protocol',
      'hello-time' : 'Set the hello interval for the spanning tree',
      'HELLO_TIME' : CliMatcher.IntegerMatcher( 200, 10000,
         helpdesc='Number of milliseconds between generation of config BPDUs' ),
   }

   @staticmethod
   def handler( mode, args ):
      config = stpInputConfig()
      config.bridgeHelloTime = args[ 'HELLO_TIME' ]

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      config = stpInputConfig()
      config.bridgeHelloTime = config.defaultBridgeHelloTime

BasicCli.GlobalConfigMode.addCommandClass( HelloTimeCmd )

#--------------------------------------------------------------------------------
# [ no | default ] spanning-tree max-age MAX_AGE
#--------------------------------------------------------------------------------
class MaxAgeCmd( CliCommand.CliCommandClass ):
   syntax = 'spanning-tree max-age MAX_AGE'
   noOrDefaultSyntax = 'spanning-tree max-age ...'
   data = {
      'spanning-tree' : 'Spanning tree protocol',
      'max-age' : 'Set the max age interval for the spanning tree',
      'MAX_AGE' : CliMatcher.IntegerMatcher( 6, 40,
         helpdesc='Maximum number of seconds the information in a BPDU is valid' ),
   }

   @staticmethod
   def handler( mode, args ):
      stpInputConfig().maxAge = args[ 'MAX_AGE' ]

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      config = stpInputConfig()
      config.maxAge = config.defaultMaxAge

BasicCli.GlobalConfigMode.addCommandClass( MaxAgeCmd )

#--------------------------------------------------------------------------------
# [ no | default ] spanning-tree max-hops MAX_HOPS
#--------------------------------------------------------------------------------
class MaxHopsCmd( CliCommand.CliCommandClass ):
   syntax = 'spanning-tree max-hops MAX_HOPS'
   noOrDefaultSyntax = 'spanning-tree max-hops ...'
   data = {
      'spanning-tree' : 'Spanning tree protocol',
      'max-hops' : 'Set the max hops value for the spanning tree',
      'MAX_HOPS' : CliMatcher.IntegerMatcher( 1, 40,
         helpdesc='Maximum number of hops a BPDU is valid' ),
   }

   @staticmethod
   def handler( mode, args ):
      stpInputConfig().maxHops = args[ 'MAX_HOPS' ]

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      config = stpInputConfig()
      config.maxHops = config.defaultMaxHops

BasicCli.GlobalConfigMode.addCommandClass( MaxHopsCmd )

#--------------------------------------------------------------------------------
# [ no | default ] spanning-tree ( portchannel | etherchannel ) guard misconfig
#--------------------------------------------------------------------------------
class SpanningTreeGuardMisconfigCmd( CliCommand.CliCommandClass ):
   syntax = 'spanning-tree ( portchannel | etherchannel ) guard misconfig'
   noOrDefaultSyntax = syntax
   data = {
      'spanning-tree' : 'Spanning tree protocol',
      'portchannel' : 'STP configuration specific to port-channels',
      'etherchannel' : 'STP configuration specific to port-channels',
      'guard' : 'Guard port-channels',
      'misconfig' : 'Detect and disable port-channels connected to multiple devices',
   }

   @staticmethod
   def handler( mode, args ):
      stpInputConfig().portChannelGuardEnabled = True

   @staticmethod
   def noHandler( mode, args ):
      stpInputConfig().portChannelGuardEnabled = False

   @staticmethod
   def defaultHandler( mode, args ):
      config = stpInputConfig()
      config.portChannelGuardEnabled = config.portChannelGuardEnabledDefault

BasicCli.GlobalConfigMode.addCommandClass( SpanningTreeGuardMisconfigCmd )

#--------------------------------------------------------------------------------
# [ no | default ] spanning-tree bpduguard rate-limit count COUNT
#                                                               [ interval INTERVAL ]
#--------------------------------------------------------------------------------
matcherBpduguard = CliMatcher.KeywordMatcher( 'bpduguard',
      helpdesc='Configure BPDU Guard' )
matcherRateLimit = CliMatcher.KeywordMatcher( 'rate-limit',
      helpdesc='BPDU Input Rate Limit options' )
matcherInterval = CliMatcher.KeywordMatcher( 'interval',
      helpdesc='Set the timer interval' )

class RateLimitSettingsCmd( CliCommand.CliCommandClass ):
   syntax = 'spanning-tree bpduguard rate-limit count COUNT [ interval INTERVAL ]'
   noOrDefaultSyntax = 'spanning-tree bpduguard rate-limit count ...'
   data = {
      'spanning-tree' : 'Spanning tree protocol',
      'bpduguard' : matcherBpduguard,
      'rate-limit' : matcherRateLimit,
      'count' : 'Set the max number of BPDUs per timer interval',
      'COUNT' : CliMatcher.IntegerMatcher( 1, 20000,
         helpdesc='Max number of BPDUs per timer interval' ),
      'interval' : matcherInterval,
      'INTERVAL' : CliMatcher.IntegerMatcher( 1, 15,
         helpdesc='Number of seconds in the BPDU Input Rate Limit Timer' ),
   }

   @staticmethod
   def handler( mode, args ):
      config = stpInputConfig()
      config.rateLimitMaxCount = args[ 'COUNT' ]
      if 'interval' in args:
         config.rateLimitInterval = args[ 'INTERVAL' ]

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      config = stpInputConfig()
      config.rateLimitInterval = 0
      config.rateLimitMaxCount = 0

BasicCli.GlobalConfigMode.addCommandClass( RateLimitSettingsCmd )

#--------------------------------------------------------------------------------
# [ no | default ] spanning-tree bpduguard rate-limit default
#--------------------------------------------------------------------------------
class SpanningTreeBpduguardRateLimitDefaultCmd( CliCommand.CliCommandClass ):
   syntax = 'spanning-tree bpduguard rate-limit default'
   noOrDefaultSyntax = syntax
   data = {
      'spanning-tree' : 'Spanning tree protocol',
      'bpduguard' : matcherBpduguard,
      'rate-limit' : matcherRateLimit,
      'default' : 'Enable rate limit by default on all ports',
   }

   @staticmethod
   def handler( mode, args ):
      stpInputConfig().rateLimitEnabled = True

   @staticmethod
   def noHandler( mode, args ):
      stpInputConfig().rateLimitEnabled = False

   @staticmethod
   def defaultHandler( mode, args ):
      config = stpInputConfig()
      config.rateLimitEnabled = config.rateLimitEnabledDefault

BasicCli.GlobalConfigMode.addCommandClass( SpanningTreeBpduguardRateLimitDefaultCmd )

#--------------------------------------------------------------------------------
# [ no | default ] spanning-tree ( bridge assurance ) | ( transmit active )
#--------------------------------------------------------------------------------
class SpanningTreeBridgeAssuranceCmd( CliCommand.CliCommandClass ):
   syntax = 'spanning-tree ( bridge assurance ) | ( transmit active )'
   noOrDefaultSyntax = syntax
   data = {
      'spanning-tree' : 'Spanning tree protocol',
      'bridge' : CliCommand.Node(
         matcher=CliMatcher.KeywordMatcher( 'bridge',
            helpdesc='Spanning tree bridge options' ),
         deprecatedByCmd='spanning-tree transmit' ),
      'assurance' : 'Enable Bridge Assurance on all network ports',
      'transmit' : 'Set BPDU transmit parameters',
      'active' : 'Enable Bridge Assurance on all network ports',
   }

   @staticmethod
   def handler( mode, args ):
      stpInputConfig().bridgeAssurance = True

   @staticmethod
   def noHandler( mode, args ):
      stpInputConfig().bridgeAssurance = False

   @staticmethod
   def defaultHandler( mode, args ):
      config = stpInputConfig()
      config.bridgeAssurance = config.bridgeAssuranceDefault

BasicCli.GlobalConfigMode.addCommandClass( SpanningTreeBridgeAssuranceCmd )

#-------------------------------------------------------------------------------
# [no] spanning-tree mode { mstp | rstp | none | rapid-pvst }
#-------------------------------------------------------------------------------

# To disable Rstp spanning tree, configure 'spanning-tree mode none'
# To disable Mstp instance, go into mst config submode and do
# 'no instance <id>', to remove the instance.
def pvstSupportedGuard( mode, token ):
   if bridgingHwCapabilities.pvstSupported:
      return None
   return CliParser.guardNotThisPlatform

def backupInterfacesSupportedGuard( mode, token ):
   if bridgingHwCapabilities.backupInterfacesSupported:
      return None
   return CliParser.guardNotThisPlatform

class StpModeCmd( CliCommand.CliCommandClass ):
   syntax = 'spanning-tree mode ( none | mstp | rstp | rapid-pvst | backup )'
   noOrDefaultSyntax = 'spanning-tree mode ...'
   data = {
      'spanning-tree' : 'Spanning tree protocol',
      'mode' : 'Spanning tree operating mode',
      'none' : 'Disable spanning tree',
      'mstp' : 'Multiple spanning tree protocol',
      'rstp' : 'Rapid spanning tree protocol',
      'rapid-pvst' : CliCommand.Node(
         matcher=CliMatcher.KeywordMatcher( 'rapid-pvst',
            helpdesc='Per VLAN rapid spanning tree protocol' ), 
         guard=pvstSupportedGuard ),
      'backup' : CliCommand.Node(
         matcher=CliMatcher.KeywordMatcher( 'backup',
            helpdesc='Backup port mode' ), 
         guard=backupInterfacesSupportedGuard ),
   }
   
   @staticmethod
   def handler( mode, args ):
      stpMode = ( args.get( 'none' ) or
                  args.get( 'mstp' ) or
                  args.get( 'rstp' ) or
                  args.get( 'backup' ) or
                  'rapidPvstp' )
      config = stpInputConfig()
      config.forceProtocolVersion = stpMode
      if stpMode == 'none' or stpMode == 'backup':
         Tracing.trace2( 'Disabling spanning-tree' )
      else:
         Tracing.trace2( 'Enabling spanning-tree' )
         if stpMode == 'rapidPvstp':
            Tracing.trace2( 'request traping resources from platform' )
            trapConfig.features.add( TrapFeatureName.pvst )
         else:
            if TrapFeatureName.pvst in trapConfig.features:
               trapConfig.features.remove( TrapFeatureName.pvst )

      Tracing.trace2( "Protocol is ", config.forceProtocolVersion )
      
   @staticmethod
   def noOrDefaultHandler( mode, args ):
      config = stpInputConfig()
      config.forceProtocolVersion = config.defaultForceProtocolVersion
      Tracing.trace2( "Protocol is ", config.forceProtocolVersion )
      if config.forceProtocolVersion == 'rapidPvstp':
         trapConfig.features.add( TrapFeatureName.pvst )
      else:
         if TrapFeatureName.pvst in trapConfig.features:
            trapConfig.features.remove( TrapFeatureName.pvst )

BasicCli.GlobalConfigMode.addCommandClass( StpModeCmd )


#--------------------------------------------------------------------------------
# [ no | default ] spanning-tree ( vlan-id | vlan ) VLAN_SET
#--------------------------------------------------------------------------------
class StpVlansCmds( CliCommand.CliCommandClass ):
   syntax = 'spanning-tree ( vlan-id | vlan ) VLAN_SET'
   noOrDefaultSyntax = syntax
   data = {
      'spanning-tree' : 'Spanning tree protocol',
      'vlan' : deprecatedNodeVlan, 
      'vlan-id' : matcherVlanId,
      'VLAN_SET' : VlanCli.vlanSetMatcher,
   }

   @staticmethod
   def handler( mode, args ):
      vlanSet = args[ 'VLAN_SET' ]
      config = stpInputConfig()
      for vlanId in vlanSet.ids:
         del config.disabledVlan[ vlanId ]

   @staticmethod
   def noHandler( mode, args ):
      vlanSet = args[ 'VLAN_SET' ]
      config = stpInputConfig()
      for vlanId in vlanSet.ids:
         config.disabledVlan[ vlanId ] = True

   @staticmethod
   def defaultHandler( mode, args ):
      # Default form is the negative form. And negative form of the command gets
      # saved in the running-config. This is a bug because default should NOT alter
      # the running-config. Since the old syntax will be deprecated in Aug 2019, 
      # the behavior is not being fixed. Ideally, the default form for this command 
      # should be the positive form.
      if 'vlan' in args:
         StpVlansCmds.noHandler( mode, args )
      else:
         StpVlansCmds.handler( mode, args )

BasicCli.GlobalConfigMode.addCommandClass( StpVlansCmds )

#-------------------------------------------------------------------------------
# "[no] spanning-tree mst configuration" command.
#-------------------------------------------------------------------------------
class MstConfigMode( MstMode, BasicCli.ConfigModeBase ):

   #----------------------------------------------------------------------------
   # Attributes required of every Mode class.
   #----------------------------------------------------------------------------
   name = 'Spanning tree MSTP configuration'
   modeParseTree = CliParser.ModeParseTree()
   # We already have a previous implementation for 'show active'.
   showActiveCmdRegistered_ = True

   #----------------------------------------------------------------------------
   # Constructs a new MstConfigMode instance.  This is called every time the
   # user enters "config-mst" mode.
   #----------------------------------------------------------------------------
   def __init__( self, parent, session, mstConfig ):
      self.mstConfig = mstConfig
      MstMode.__init__( self, None )
      BasicCli.ConfigModeBase.__init__( self, parent, session )

   def onExit( self ):
      # This is called by the Cli Session when it is leaving our mode, whether
      # because of an exit command or a command from another mode being executed.
      Tracing.trace2( "MstConfigMode.onExit" )
      self.mstConfig.apply()
   
#--------------------------------------------------------------------------------
# [ no | default ] spanning-tree mst configuration
#--------------------------------------------------------------------------------
class SpanningTreeMstConfigurationCmd( CliCommand.CliCommandClass ):
   syntax = 'spanning-tree mst configuration'
   noOrDefaultSyntax = syntax
   data = {
      'spanning-tree' : 'Spanning tree protocol',
      'mst' : 'Specify a range of MST instances',
      'configuration' : 'Enter multiple spanning tree configuration mode',
   }

   @staticmethod
   def handler(  mode, args ):
      mstConfig = StpCliLib.MstConfig( stpInputConfig() )

      childMode = mode.childMode( MstConfigMode, mstConfig=mstConfig )
      # Tell the MstConfig to revert itself to the current configuration
      # to get started.
      childMode.mstConfig.revert()
      mode.session_.gotoChildMode( childMode )

   @staticmethod
   def noOrDefaultHandler(  mode, args ):
      mstConfig = StpCliLib.MstConfig( stpInputConfig() )
      mstConfig.default()
      mstConfig.apply()

BasicCli.GlobalConfigMode.addCommandClass( SpanningTreeMstConfigurationCmd )

#--------------------------------------------------------------------------------
# [ no | default ] spanning-tree mst pvst border
#--------------------------------------------------------------------------------
class SpanningTreeMstPvstBorderCmd( CliCommand.CliCommandClass ):
   syntax = 'spanning-tree mst pvst border'
   noOrDefaultSyntax = syntax
   data = {
      'spanning-tree' : 'Spanning tree protocol',
      'mst' : 'Specify a range of MST instances',
      'pvst' : 'Pvst border parameters',
      'border' : 'Enable pvst border',
   }

   @staticmethod
   def handler( mode, args ):
      stpInputConfig().mstPvstBorder = True

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      stpInputConfig().mstPvstBorder = False

BasicCli.GlobalConfigMode.addCommandClass( SpanningTreeMstPvstBorderCmd )

#--------------------------------------------------------------------------------
# [ no | default ] instance INST_ID vlan VLAN_SET
#--------------------------------------------------------------------------------
def setMstVlanMapping( mode, args ):
   instId = args[ 'INST_ID' ]
   vlanSet = args[ 'VLAN_SET' ]
   mstConfig = mode.mstConfig

   # We assume the cist is always present, so subtract it from the limit.
   mstInstCountMax = topoStatus.maximumMstpInstances - 1
   assert topoStatus.maximumMstpInstances <= ( MstInstCountMax + 1 )
   if ( instId != MstCistInstId ) and not mstConfig.inst( instId ):
      count = mstConfig.insts()
      if count >= mstInstCountMax:
         mode.addError( "Configuration ignored: only %d instances are "
                        "supported." % (mstInstCountMax + 1) )
         return
      
   mstConfig.vlanMapIs( vlanSet.ids, instId )

def deleteMstVlanMapping( mode, instId, vlanSet ):
   # Delete the mappings from the given vlans to the given instId.
   mstConfig = mode.mstConfig

   removeSet = set()
   vlanIds = vlanSet.ids
   for vlanId in vlanIds:
      if mstConfig.vlanMap( vlanId ) != instId:
         # We ignore requests to remove a mapping that doesn't exist.
         removeSet.add( vlanId )
   vlanIds.difference_update( removeSet )
   mstConfig.vlanMapIs( vlanIds, MstCistInstId )

def deleteMstInstAllVlanMappings( mode, instId ):
   # Delete all the vlan mappings to the given instId.  The MstConfig does
   # all the work for us.
   mstConfig = mode.mstConfig
   mstConfig.instDel( instId )

class MstVlanMappingCmd( CliCommand.CliCommandClass ):
   syntax = 'instance INST_ID vlan VLAN_SET'
   noOrDefaultSyntax = 'instance INST_ID [ vlan VLAN_SET ]'
   data = {
      'instance' : 'Map VLANs to an MST instance',
      'INST_ID' : CliMatcher.IntegerMatcher( 0 , 4094,
         helpdesc='MST instance ID' ),
      'vlan' : CliMatcher.KeywordMatcher( 'vlan',
         helpdesc='Range of VLANs to add to the instance mapping',
         alternates=[ 'vlans' ] ), # BUG454362
      'VLAN_SET' : VlanCli.vlanSetMatcher,
   }
   handler = setMstVlanMapping

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      if 'vlan' in args:
         deleteMstVlanMapping( mode, args[ 'INST_ID' ], args[ 'VLAN_SET' ] )
      else:
         deleteMstInstAllVlanMappings( mode, args[ 'INST_ID' ] )

MstConfigMode.addCommandClass( MstVlanMappingCmd )

#--------------------------------------------------------------------------------
# [ no | default ] name REGION_ID
#--------------------------------------------------------------------------------
class MstRegionIdCmd( CliCommand.CliCommandClass ):
   syntax = 'name REGION_ID'
   noOrDefaultSyntax = 'name ...'
   data = {
      'name' : 'Set configuration name',
      'REGION_ID' : CliMatcher.PatternMatcher( pattern='.+',
         helpdesc='Configuration Name', helpname='WORD' ),
   }

   @staticmethod
   def handler( mode, args ):
      regionId = args[ 'REGION_ID' ]
      # The region id is truncated to 32 chars with a warning if necessary.
      if len( regionId ) > 32:
         mode.addWarning( "Configuration name has been truncated to 32 characters!" )
         regionId = regionId[ : 32 ]

      mode.mstConfig.regionIdIs( regionId )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      mode.mstConfig.regionIdIs( '' )

MstConfigMode.addCommandClass( MstRegionIdCmd )

#--------------------------------------------------------------------------------
# [ no | default ] revision REV
#--------------------------------------------------------------------------------
class MstRevisionCmd( CliCommand.CliCommandClass ):
   syntax = 'revision REV'
   noOrDefaultSyntax = 'revision ...'
   data = {
      'revision' : 'Set configuration revision number',
      'REV' : CliMatcher.IntegerMatcher( 0, 65535,
         helpdesc='Configuration revision number' ),
   }

   @staticmethod
   def handler( mode, args ):
      mode.mstConfig.configRevisionIs( args[ 'REV' ] )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      mode.mstConfig.configRevisionIs( MstRevisionDefault )

MstConfigMode.addCommandClass( MstRevisionCmd )

#--------------------------------------------------------------------------------
# abort
#--------------------------------------------------------------------------------
class AbortCmd( CliCommand.CliCommandClass ):
   syntax = 'abort'
   data = {
      'abort' : 'Exit region configuration mode, aborting changes',
   }

   @staticmethod
   def handler( mode, args ):
      mode.mstConfig.revert()
      mode.session_.gotoParentMode()

MstConfigMode.addCommandClass( AbortCmd )

#--------------------------------------------------------------------------------
# show [ pending ]
# Strangely enough, the industry standard defaults just the show keyword
# to mean 'show pending'.
#--------------------------------------------------------------------------------
class PendingCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show [ pending ]'
   data = {
      'pending' : 'Display the new MST configuration to be applied',
   }
   cliModel = SpanningTreeMstConfiguration

   @staticmethod
   def handler( mode, args ):
      return getMstInfoCapi( mode.mstConfig, message='Pending MST configuration' )

MstConfigMode.addShowCommandClass( PendingCmd )

#--------------------------------------------------------------------------------
# show active | current
#--------------------------------------------------------------------------------
class ActiveCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show active | current'
   data = {
      'active' : 'Display the MST configuration in the current running-config',
      'current' : 'Display the MST configuration in the current running-config',
   }
   cliModel = SpanningTreeMstConfiguration

   @staticmethod
   def handler( mode, args ):
      # Make a new MstConfig object that we can revert to the current config
      # and extract the info from that.
      currentMstConfig = StpCliLib.MstConfig( stpInputConfig() )
      currentMstConfig.revert()
      return getMstInfoCapi( currentMstConfig, message='Active MST configuration' )

MstConfigMode.addShowCommandClass( ActiveCmd )

#-------------------------------------------------------------------------------
# [no] spanning-tree [ mst <instanceid> ] root { primary | secondary }
# [no] spanning-tree vlan-id <vlan-range> root { primary | secondary }
#
# legacy:
# [no] spanning-tree vlan <vlan-range> root { primary | secondary }
#-------------------------------------------------------------------------------

class StpMstRootPrimarySecondaryCmd( CliCommand.CliCommandClass ):
   syntax = ( 'spanning-tree [ mst INSTANCE_IDS ] root ( primary | secondary )' )
   noOrDefaultSyntax = 'spanning-tree [ mst INSTANCE_IDS ] root ...' 
   data = {
         'spanning-tree' : 'Spanning tree protocol',
         'mst' : 'Specify a range of MST Instances',
         'INSTANCE_IDS' : matcherMstInstanceId,
         'root' : 'Configure the switch as root',
         'primary' : 'Configure this switch as primary root for this spanning tree',
         'secondary'  : 'Configure this switch as secondary root'
   }

   @staticmethod
   def handler( mode, args ):
      instIds = args.get( 'INSTANCE_IDS' )
      if 'primary' in args:
         priority = StpCliLib.BridgePriorityPrimary
      else:
         priority = StpCliLib.BridgePrioritySecondary
      Tracing.trace9( 'InstId =', instIds, 'priority =', priority )
   
      if instIds is not None:
         for instId in instIds.values():
            mstiConfig = StpCliLib.stpMstiConfig( instId, True )
            mstiConfig.bridgePriority = priority
      else:
         mstiConfig = StpCliLib.stpMstiConfig( instIds, True )
         mstiConfig.bridgePriority = priority

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      instIds = args.get( 'INSTANCE_IDS' )
      priority = StpCliLib.BridgePriorityDefault
      Tracing.trace9( 'InstId =', instIds, 'priority =', priority )
   
      if instIds is not None:
         for instId in instIds.values():
            mstiConfig = StpCliLib.stpMstiConfig( instId, True )
            mstiConfig.bridgePriority = priority
      else:
         mstiConfig = StpCliLib.stpMstiConfig( instIds, True )
         mstiConfig.bridgePriority = priority

BasicCli.GlobalConfigMode.addCommandClass( StpMstRootPrimarySecondaryCmd )

def validateBridgePriority( mode, priority ):
   if ( priority % StpCliLib.BridgePriorityStepSize ) != 0:
      mode.addError( "Bridge Priority must be in increments of %d. "
                     "Allowed values are:\n"
                     "0     4096  8192  12288 16384 20480 24576 28672\n"
                     "32768 36864 40960 45056 49152 53248 57344 61440" %
                     StpCliLib.BridgePriorityStepSize )
      raise CliParser.AlreadyHandledError

class StpVlanRootPrimarySecondaryCmd( CliCommand.CliCommandClass ):
   syntax = 'spanning-tree ( vlan-id | vlan ) VLAN_SET root ( primary | secondary )'
   noOrDefaultSyntax = 'spanning-tree ( vlan-id | vlan ) VLAN_SET root ...'
   data = {
         'spanning-tree' : 'Spanning tree protocol',
         'vlan-id' : matcherVlanId, 
         'VLAN_SET' : VlanCli.vlanSetMatcher,
         'vlan' : deprecatedNodeVlan,
         'root' : 'Configure the switch as root',
         'primary' : 'Configure this switch as primary root for this spanning tree',
         'secondary'  : 'Configure this switch as secondary root'
   }

   @staticmethod
   def handler( mode, args ):
      vlanSet = args[ 'VLAN_SET' ]
      if 'primary' in args:
         priority = StpCliLib.BridgePriorityPrimary
      else:
         priority = StpCliLib.BridgePrioritySecondary
      Tracing.trace9( 'priority =', priority, 'vlans =', vlanSet.ids )

      for vlanId in vlanSet.ids:
         mstiConfig = StpCliLib.pvstVlanMstiConfig( vlanId, create=True )
         mstiConfig.bridgePriority = priority

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      vlanSet = args[ 'VLAN_SET' ]
      for vlanId in vlanSet.ids:
         mstiConfig = StpCliLib.pvstVlanMstiConfig( vlanId )
         if mstiConfig is not None:
            mstiConfig.bridgePriority = StpCliLib.BridgePriorityDefault
         else:
            Tracing.trace2( 'No spanning-tree PVST instance ', vlanId )
      
BasicCli.GlobalConfigMode.addCommandClass( StpVlanRootPrimarySecondaryCmd )

#-------------------------------------------------------------------------------
# [no] spanning-tree [ mst <instanceid> ] priority <priority>
# [no] spanning-tree vlan-id <vlan-range> priority <priority>
#
# legacy:
# [no] spanning-tree vlan <vlan-range> priority <priority>
#
# The industry standard cli has [diameter <dia> [ hello-time
# <hello-time> ] ] options hanging off the end of this command. However, it is
# not clear how to use these parameters with 802.1Q-2005 MSTP, where one BPDU
# carries configuration information for all the instances. For now, we don't
# implement these keywords.
#-------------------------------------------------------------------------------
class StpMstPriorityCmd( CliCommand.CliCommandClass ):
   syntax = 'spanning-tree [ mst INSTANCE_IDS ] priority PRIO'
   noOrDefaultSyntax = 'spanning-tree [ mst INSTANCE_IDS ] priority ...'
   data = {
         'spanning-tree' : 'Spanning tree protocol',
         'mst' : 'Specify a range of MST Instances',
         'INSTANCE_IDS' : matcherMstInstanceId,
         'priority' : 'Set bridge priority for spanning tree',
         'PRIO' : matcherPriority
   }

   @staticmethod
   def handler( mode, args ):
      instIds = args.get( 'INSTANCE_IDS' )
      prio = args[ 'PRIO' ]
      validateBridgePriority( mode, prio )
      Tracing.trace9( 'InstId =', instIds, 'priority =', prio )
   
      if instIds is not None:
         for instId in instIds.values():
            mstiConfig = StpCliLib.stpMstiConfig( instId, True )
            mstiConfig.bridgePriority = prio 
      else:
         mstiConfig = StpCliLib.stpMstiConfig( instIds, True )
         mstiConfig.bridgePriority = prio

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      instIds = args.get( 'INSTANCE_IDS' )
      prio = StpCliLib.BridgePriorityDefault
      Tracing.trace9( 'InstId =', instIds, 'priority =', prio )
   
      if instIds is not None:
         for instId in instIds.values():
            mstiConfig = StpCliLib.stpMstiConfig( instId, True )
            mstiConfig.bridgePriority = prio
      else:
         mstiConfig = StpCliLib.stpMstiConfig( instIds, True )
         mstiConfig.bridgePriority = prio
      
BasicCli.GlobalConfigMode.addCommandClass( StpMstPriorityCmd )

class StpVlanPriorityCmd( CliCommand.CliCommandClass ):
   syntax = 'spanning-tree ( vlan-id | vlan ) VLAN_SET priority PRIO'
   noOrDefaultSyntax = 'spanning-tree ( vlan-id | vlan ) VLAN_SET priority ...'
   data = {
         'spanning-tree' : 'Spanning tree protocol',
         'vlan-id' : matcherVlanId,
         'VLAN_SET' : VlanCli.vlanSetMatcher,
         'vlan' : deprecatedNodeVlan,
         'priority' : 'Set bridge priority for spanning tree',
         'PRIO' : matcherPriority
   }

   @staticmethod
   def handler( mode, args ):
      vlanSet = args[ 'VLAN_SET' ]
      prio = args[ 'PRIO' ]
      validateBridgePriority( mode, prio )
      Tracing.trace9( 'VlanId=', vlanSet, 'priority =', prio )

      for vlanId in vlanSet.ids:
         mstiConfig = StpCliLib.pvstVlanMstiConfig( vlanId, create=True )
         mstiConfig.bridgePriority = prio

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      vlanSet = args[ 'VLAN_SET' ]
      for vlanId in vlanSet.ids:
         mstiConfig = StpCliLib.pvstVlanMstiConfig( vlanId )
         if mstiConfig is not None:
            mstiConfig.bridgePriority = StpCliLib.BridgePriorityDefault
         else:
            Tracing.trace2( 'No spanning-tree PVST instance ', vlanId )
      

BasicCli.GlobalConfigMode.addCommandClass( StpVlanPriorityCmd )

#--------------------------------------------------------------------------------
# [ no | default ] spanning-tree ( bpdu tx hold-count ) | ( transmit hold-count ) 
#                                                                         HOLD_COUNT
#--------------------------------------------------------------------------------
class SpanningTreeBpduTxHoldCountHoldcountCmd( CliCommand.CliCommandClass ):
   syntax = ( 'spanning-tree ( bpdu tx hold-count ) | '
                            '( transmit HOLD_COUNT_DEP ) HOLD_COUNT' )
   noOrDefaultSyntax = ( 'spanning-tree ( bpdu tx hold-count ) | '
                                       '( transmit hold-count ) ...' )
   data = {
      'spanning-tree' : 'Spanning tree protocol',
      'bpdu' : 'Set BPDU parameters',
      'tx' : 'Set transmit parameters',
      'transmit': 'Set BPDU transmit parameters',
      'hold-count' : 'Set the hold count',
      'HOLD_COUNT_DEP' : CliCommand.Node(
         matcher=CliMatcher.KeywordMatcher( 'hold-count',
            helpdesc='Set the hold count' ),
         deprecatedByCmd='spanning-tree bpdu tx hold-count' ),
      'HOLD_COUNT' : CliMatcher.IntegerMatcher( 1, 10,
         helpdesc='Maximum number of BPDUs transmitted per second' ),
   }

   @staticmethod
   def handler( mode, args ):
      config = stpInputConfig()
      config.txHoldCount = args[ 'HOLD_COUNT' ]

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      config = stpInputConfig()
      config.txHoldCount = config.defaultTxHoldCount

BasicCli.GlobalConfigMode.addCommandClass( SpanningTreeBpduTxHoldCountHoldcountCmd )

#--------------------------------------------------------------------------------
# [ no | default ] spanning-tree ( portfast | edge-port ) bpdufilter default
#--------------------------------------------------------------------------------
matcherEdgePort = CliMatcher.KeywordMatcher( 'edge-port',
      helpdesc='Configure portfast' )
matcherPortfast = CliMatcher.KeywordMatcher( 'portfast',
      helpdesc='Configure portfast' )
matcherBpdufilter = CliMatcher.KeywordMatcher( 'bpdufilter',
      helpdesc='Configure BPDU Filter' )

class SpanningTreePortfastBpdufilterDefaultCmd( CliCommand.CliCommandClass ):
   syntax = 'spanning-tree ( portfast | edge-port ) bpdufilter default'
   noOrDefaultSyntax = syntax
   data = {
      'spanning-tree' : 'Spanning tree protocol',
      'edge-port' : matcherEdgePort,
      'portfast' : CliCommand.Node( matcher=matcherPortfast,
         deprecatedByCmd='spanning-tree edge-port <bpduguard | bpdufilter>' ),
      'bpdufilter' : matcherBpdufilter,
      'default' : 'Enable BPDU Filter on portfast interfaces',
   }

   @staticmethod
   def handler( mode, args ):
      stpInputConfig().portfastBpdufilter = True

   @staticmethod
   def noHandler( mode, args ):
      stpInputConfig().portfastBpdufilter = False

   @staticmethod
   def defaultHandler( mode, args ):
      config = stpInputConfig()
      config.portfastBpdufilter = config.portfastBpdufilterDefault

BasicCli.GlobalConfigMode.addCommandClass( SpanningTreePortfastBpdufilterDefaultCmd )

#--------------------------------------------------------------------------------
# [ no | default ] spanning-tree ( edge-port | portfast ) bpduguard default
#--------------------------------------------------------------------------------
class SpanningTreeEdgePortBpduguardDefaultCmd( CliCommand.CliCommandClass ):
   syntax = 'spanning-tree ( edge-port | portfast ) bpduguard default'
   noOrDefaultSyntax = syntax
   data = {
      'spanning-tree' : 'Spanning tree protocol',
      'portfast' : CliCommand.Node( matcher=matcherPortfast,
         deprecatedByCmd='spanning-tree edge-port <bpduguard | bpdufilter>' ),
      'edge-port' : matcherEdgePort,
      'bpduguard' : matcherBpduguard, 
      'default' : 'Enable BPDU Guard on portfast interfaces',
   }

   @staticmethod
   def handler( mode, args ):
      stpInputConfig().portfastBpduguard = True

   @staticmethod
   def noHandler( mode, args ):
      stpInputConfig().portfastBpduguard = False

   @staticmethod
   def defaultHandler( mode, args ):
      config = stpInputConfig()
      config.portfastBpduguard = config.portfastBpduguardDefault

BasicCli.GlobalConfigMode.addCommandClass( SpanningTreeEdgePortBpduguardDefaultCmd )

#-------------------------------------------------------------------------------
# [no] spanning-tree root super
#-------------------------------------------------------------------------------

class StpRootSuperCmd( CliCommand.CliCommandClass ):
   syntax = 'spanning-tree root super'
   noOrDefaultSyntax = syntax
   data = {
         'spanning-tree' : 'Spanning tree protocol',
         'root' : 'Configure the switch as root',
         'super' : 'Configure this switch as the network super root'
   }

   @staticmethod
   def handler( mode, args ):
      stpInputConfig().superRoot = True

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      stpInputConfig().superRoot = False

if toggleStpSuperRootEnabled():
   BasicCli.GlobalConfigMode.addCommandClass( StpRootSuperCmd )

#-------------------------------------------------------------------------------
# Have the Cli Agent mount all needed state from sysdb
#-------------------------------------------------------------------------------
def Plugin( entityManager ):
   cliConfig = ConfigMount.mount( entityManager, "stp/input/config/cli",
                                  "Stp::Input::Config", "w" )
   stpTxRxInputConfig = ConfigMount.mount( entityManager,
                                           "stp/txRx/input/config/cli",
                                           "StpTxRx::Input::Config", "w" )
   cliConfigReq = LazyMount.mount( entityManager, "stp/input/config/cliReq",
                                  "Stp::Input::ConfigReq", "w" )
   global bridgingHwCapabilities 
   bridgingHwCapabilities = LazyMount.mount( entityManager, 
      "bridging/hwcapabilities", "Bridging::HwCapabilities","r" )      
   global topoStatus
   topoStatus = LazyMount.mount( entityManager, "bridging/topology/status",
                                 "Bridging::Topology::Status", "r" )
   StpCliLib.stpInputConfigIs( cliConfig )
   StpCliLib.stpInputConfigReqIs( cliConfigReq )
   StpCliLib.stpTxRxInputConfigIs( stpTxRxInputConfig )

   global trapConfig 
   trapConfig = ConfigMount.mount( entityManager, 'hardware/trap/config/trapConfig',
                                   'Arnet::TrapConfig', 'w' )

