#!/usr/bin/env python
# Copyright (c) 2010-2014 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

# pylint: disable-msg=R0201

#-------------------------------------------------------------------------------
# This module implements bridging configuration.  In particular, it provides:
# -  the "[no] storm-control broadcast level <percent>" command
# -  the "[no] storm-control multicast level <percent>" command
#
# This module does not contain VLAN-specific commands.
#------------------------------------------------------------------------------

import Arnet
import BasicCli
import CliCommand
import CliMatcher
import CliParser
import CliPlugin.IntfCli as IntfCli
import ConfigMount
import Intf.IntfRange
import LazyMount
import ShowCommand
import Tac
from VlanCli import SwitchportModelet

BPS_FACTOR = { "bps": 1, "kbps": 1000, "mbps": 1000000, "gbps": 1000000000 }
BPS_RESOLUTION = { "bps": "bpsResolution", "kbps": "kbpsResolution",
                   "mbps": "mbpsResolution", "gbps": "gbpsResolution" }
BPS_UNITS = [ "bps", "Kbps", "Mbps", "Gbps" ]
BPS_EXCEED_MAX_ERROR = "Error: %s is greater than the maximum configurable value: %s"

# Module globals set up by the Plugin
config = None
statusDir = None
lagStatus = None

defaultThreshold = Tac.Value( "Bridging::StormControl::Threshold" )

def convertToHighestBps( level ):
   """Helper function to convert a bps rate into the largest unit possible.

   Parameters
   ----------
   level : int
      The bps rate from Threshold.

   Returns
   -------
   rate : int
      The rate.
   unit : str
      The unit for the rate above.
   """
   assert level >= 0
   # Multiply by 1k to simplify logic in the loop.
   rate = level * 1000
   unit = BPS_UNITS[ 0 ]
   for unit in BPS_UNITS:
      rate /= 1000
      if rate % 1000 != 0:
         break
   return ( rate, unit )

def sliceStatusDir():
   assert statusDir
   return statusDir[ 'slice' ]

def status():
   assert statusDir
   return statusDir[ 'all' ]

def stormControlSupported():
   for sliceStatus in sliceStatusDir().itervalues():
      if sliceStatus.stormControlSupported:
         return True
   return False

def stormControlGuard( mode, token ):
   if status().stormControlSupported:
      return None
   else:
      if stormControlSupported():
         return None
      else:
         return CliParser.guardNotThisPlatform

def stormControlDropCountGuard( mode, token ):
   if status().dropCountSupported:
      return None
   else:
      return CliParser.guardNotThisPlatform

def stormControlDefaultConfig( name ):
   return bool( config.intfConfig[ name ].uucastLevel == defaultThreshold and
                config.intfConfig[ name ].broadcastLevel == defaultThreshold and
                config.intfConfig[ name ].multicastLevel == defaultThreshold and
                config.intfConfig[ name ].allLevel == defaultThreshold and
                config.intfConfig[ name ].stormControlDropLoggingMode == 'useGlobal'
              )

def stormControlAllLevelGuard( mode, token ):
   if status().stormControlAllLevelSupported:
      return None
   else:
      return CliParser.guardNotThisPlatform

def stormControlUUcastLevelGuard( mode, token ):
   if status().dropCountSupported:
      if status().stormControlUUcastLevelSupported:
         return None
      else:
         return CliParser.guardNotThisPlatform
   elif stormControlSupported():
      for sliceStatus in sliceStatusDir().itervalues():
         if sliceStatus.stormControlUUcastLevelSupported:
            return None
      return CliParser.guardNotThisPlatform
   else:
      return CliParser.guardNotThisPlatform

def aggregateStormControlGuard( mode, token ):
   if status().aggregateStormControlSupported:
      return None
   else:
      return CliParser.guardNotThisPlatform

def stormControlPpsMeteringGuard( mode, token ):
   if status().ppsMeteringSupported:
      return None
   else:
      return CliParser.guardNotThisPlatform

def stormControlBpsMeteringGuard( mode, token ):
   if status().bpsMeteringSupported:
      return None
   else:
      return CliParser.guardNotThisPlatform

def stormLevelPercentRange( mode ):
   minlevel = 0.01
   if stormControlZeroRateSupported():
      minlevel = 0
   return ( minlevel, 99.99 )

def stormPpsPercentRange( mode ):
   minlevel = 1
   if stormControlZeroRateSupported():
      minlevel = 0
   return ( minlevel, defaultThreshold.maxLevelPps )

def stormBpsPercentRange( mode ):
   minlevel = 1
   if stormControlZeroRateSupported():
      minlevel = 0
   return ( minlevel, defaultThreshold.maxLevelBps )

def stormControlZeroRateSupported():
   agentRunning = False
   if status().zeroRateSupported:
      return True
   for sliceStatus in sliceStatusDir().itervalues():
      agentRunning = True
      if sliceStatus.zeroRateSupported:
         return True
   if agentRunning:
      return False
   else:
      return True

def stormLevelPpsRangeGuard( mode, token ):
   if stormControlZeroRateSupported():
      return None
   try:
      if int( token ) < 1:
         return CliParser.guardNotThisPlatform
   except ValueError:
      pass
   return None

def stormLevelBpsRangeGuard( mode, token ):
   if stormControlZeroRateSupported():
      return None
   if int( token ) < 1:
      return CliParser.guardNotThisPlatform
   return None

#-------------------------------------------------------------------------------
# General tokens
#-------------------------------------------------------------------------------
stormControlKw = CliCommand.guardedKeyword( 'storm-control',
      helpdesc='Configure storm-control',
      guard=stormControlGuard )
levelKwMatcher = CliMatcher.KeywordMatcher( 'level',
      helpdesc='Configure the maximum storm control level' )
levelMatcher = CliCommand.Node(
      matcher=CliMatcher.DynamicFloatMatcher(
         rangeFn=stormLevelPercentRange,
         helpdesc='Maximum bandwidth percentage allowed by storm control',
         helpname='LEVEL',
         precisionString = '%.5g' ) )
nodePps = CliCommand.guardedKeyword( 'pps',
      helpdesc='Configure the maximum storm control level in packets per second',
      guard=stormControlPpsMeteringGuard )
ppsMatcher = CliCommand.Node(
      matcher=CliMatcher.DynamicIntegerMatcher( rangeFn=stormPpsPercentRange,
         helpdesc='Maximum packet rate allowed by storm control',
         helpname='PPS' ),
      guard=stormLevelPpsRangeGuard )

nodeBpsRate = CliCommand.guardedKeyword( "rate",
      helpdesc="Configure the maximum storm control level in (g|m|k)bits per second",
      guard=stormControlBpsMeteringGuard )
bpsMatcher = CliCommand.Node(
      matcher=CliMatcher.DynamicIntegerMatcher( rangeFn=stormBpsPercentRange,
         helpdesc="Maximum bandwidth rate allowed by storm control",
         helpname="RATE" ),
      guard=stormLevelBpsRangeGuard )
bpsRateUnitMatcher = CliMatcher.EnumMatcher( {
   "bps": "Rate in bits per second",
   "kbps": "Rate in kilobits per second",
   "mbps": "Rate in megabits per second",
   "gbps": "Rate in gigabits per second",
} )

#--------------------------------------------------------------------------------
# [ no | default ] storm-control bum aggregate
#                                      traffic-class TRAFFIC_CLASS level pps PPS
# command in config mode
#--------------------------------------------------------------------------------
def setAggregateLevelPps( mode, args ):
   tc = args[ 'TRAFFIC_CLASS' ]
   level = args[ 'PPS' ]
   threshold = Tac.Value( 'Bridging::StormControl::Threshold', 'packetsPerSecond',
                           level )
   assert threshold != defaultThreshold
   if tc not in config.aggregateTcConfig:
      config.newAggregateTcConfig( tc )
   config.aggregateTcConfig[ tc ].level = threshold

def noAggregateLevelPps( mode, args ):
   tc = args[ 'TRAFFIC_CLASS' ]
   if tc in config.aggregateTcConfig:
      config.aggregateTcConfig[ tc ].level = defaultThreshold 
      del config.aggregateTcConfig[ tc ]

class AggregateLevelPpsCmd( CliCommand.CliCommandClass ):
   syntax = 'storm-control bum aggregate traffic-class TRAFFIC_CLASS level pps PPS'
   noOrDefaultSyntax = 'storm-control bum aggregate traffic-class TRAFFIC_CLASS ...'
   data = {
      'storm-control': stormControlKw,
      'bum': CliCommand.guardedKeyword( 'bum',
         helpdesc='Broadcast, Unknown-unicast or Multicast traffic',
         guard=aggregateStormControlGuard ),
      'aggregate': 'Aggregate storm-control',
      'traffic-class': 'Select packet Traffic-class',
      'TRAFFIC_CLASS': CliMatcher.IntegerMatcher( 0, 7,
         helpdesc='Traffic-class value' ),
      'level': levelKwMatcher,
      'pps': nodePps,
      'PPS': ppsMatcher,
   }
   handler = setAggregateLevelPps
   noOrDefaultHandler = noAggregateLevelPps

BasicCli.GlobalConfigMode.addCommandClass( AggregateLevelPpsCmd )

#--------------------------------------------------------------------------------
# [ no | default ] storm-control unknown-unicast level ( LEVEL | ( pps PPS ) )
# [ no | default ] storm-control unknown-unicast level rate RATE RATE_UNIT
#--------------------------------------------------------------------------------
maxLevelPercentage = \
    Tac.Value( 'Bridging::StormControl::Threshold' ).maxLevelPercentage

def setUUcastThreshold( mode, threshold ):
   name = mode.intf.name
   if threshold == defaultThreshold and name not in config.intfConfig :
      return
   if name in config.intfConfig:
      config.intfConfig[ name ].uucastLevel = threshold
      if stormControlDefaultConfig( name ):
         del config.intfConfig[ name ]
   else:
      config.newIntfConfig( name )
      config.intfConfig[ name ].uucastLevel = threshold

def setUUcastLevelPercentage( mode, level ):
   setUUcastThreshold(
      mode, Tac.Value( 'Bridging::StormControl::Threshold', 'percentage',
                       int( 0.5 + ( level * maxLevelPercentage ) / 100 ) ) )

def setUUcastLevelPps( mode, level ):
   setUUcastThreshold(
      mode, Tac.Value( 'Bridging::StormControl::Threshold', 'packetsPerSecond',
                       level ) )

def setUUcastLevelBps( mode, rate, rateUnit ):
   rateToSet = rate * BPS_FACTOR[ rateUnit ]
   if rateToSet > defaultThreshold.maxLevelBps:
      wantRate = '%s %s' % ( rate, rateUnit )
      maxRate = '%s %s' % convertToHighestBps( defaultThreshold.maxLevelBps ) 
      mode.addError( BPS_EXCEED_MAX_ERROR % ( wantRate.lower(), maxRate.lower() ) )
      return
   threshold = Tac.Value( 'Bridging::StormControl::Threshold', 'bitsPerSecond',
                          rateToSet )
   threshold.bitResolution = BPS_RESOLUTION[ rateUnit ]
   setUUcastThreshold( mode, threshold )

def noUUcastLevel( mode, args ):
   setUUcastLevelPercentage( mode, 100 )

class StormControlUnknownUnicastLevelLevelCmd( CliCommand.CliCommandClass ):
   syntax = ( 'storm-control unknown-unicast level ( LEVEL | ( pps PPS ) | '
              '( rate RATE RATE_UNIT ) )' )
   noOrDefaultSyntax = 'storm-control unknown-unicast ...'
   data = {
      'storm-control': stormControlKw,
      'unknown-unicast': CliCommand.guardedKeyword( 'unknown-unicast',
         helpdesc='Configure storm control for unknown unicast packets',
         guard=stormControlUUcastLevelGuard ),
      'level': levelKwMatcher,
      'LEVEL': levelMatcher,
      'pps': nodePps,
      'PPS': ppsMatcher,
      'rate': nodeBpsRate,
      'RATE': bpsMatcher,
      'RATE_UNIT': bpsRateUnitMatcher, 
   }
   @staticmethod
   def handler( mode, args ):
      if 'PPS' in args:
         setUUcastLevelPps( mode, args[ 'PPS' ] )
      elif 'RATE' in args:
         setUUcastLevelBps( mode, args[ 'RATE' ], args[ 'RATE_UNIT' ] )
      else:
         setUUcastLevelPercentage( mode, args[ 'LEVEL' ] )

   noOrDefaultHandler = noUUcastLevel

SwitchportModelet.addCommandClass( StormControlUnknownUnicastLevelLevelCmd )

#--------------------------------------------------------------------------------
# [ no | default ] storm-control broadcast level ( LEVEL | ( pps PPS ) )
# [ no | default ] storm-control broadcast level rate RATE RATE_UNIT
#--------------------------------------------------------------------------------
maxLevelPercentage = \
    Tac.Value( 'Bridging::StormControl::Threshold' ).maxLevelPercentage

def setBroadcastThreshold( mode, threshold ):
   name = mode.intf.name
   if( threshold == defaultThreshold and name not in config.intfConfig ):
      return
   if name in config.intfConfig:
      config.intfConfig[ name ].broadcastLevel = threshold
      if stormControlDefaultConfig( name ):
         del config.intfConfig[ name ]
   else:
      config.newIntfConfig( name )
      config.intfConfig[ name ].broadcastLevel = threshold

def setBroadcastLevelPercentage( mode, level ):
   setBroadcastThreshold(
      mode, Tac.Value( 'Bridging::StormControl::Threshold', 'percentage',
                       int( 0.5 + ( level * maxLevelPercentage ) / 100 ) ) )

def setBroadcastLevelPps( mode, level ):
   setBroadcastThreshold(
      mode, Tac.Value( 'Bridging::StormControl::Threshold', 'packetsPerSecond',
                       level ) )

def setBroadcastLevelBps( mode, rate, rateUnit ):
   rateToSet = rate * BPS_FACTOR[ rateUnit ]
   if rateToSet > defaultThreshold.maxLevelBps:
      wantRate = '%s %s' % ( rate, rateUnit )
      maxRate = '%s %s' % convertToHighestBps( defaultThreshold.maxLevelBps ) 
      mode.addError( BPS_EXCEED_MAX_ERROR % ( wantRate.lower(), maxRate.lower() ) )
      return
   threshold = Tac.Value( 'Bridging::StormControl::Threshold', 'bitsPerSecond',
                          rateToSet )
   threshold.bitResolution = BPS_RESOLUTION[ rateUnit ]
   setBroadcastThreshold( mode, threshold )

def noBroadcastLevel( mode, args ):
   setBroadcastLevelPercentage( mode, 100 )

class StormControlBroadcastLevelLevelCmd( CliCommand.CliCommandClass ):
   syntax = ( 'storm-control broadcast level ( LEVEL | ( pps PPS ) | '
              '( rate RATE RATE_UNIT ) )' )
   noOrDefaultSyntax = 'storm-control broadcast ...'
   data = {
      'storm-control': stormControlKw,
      'broadcast': 'Configure storm control for broadcast packets',
      'level': levelKwMatcher,
      'LEVEL': levelMatcher,
      'pps': nodePps,
      'PPS': ppsMatcher,
      'rate': nodeBpsRate,
      'RATE': bpsMatcher,
      'RATE_UNIT': bpsRateUnitMatcher, 
   }
   @staticmethod
   def handler( mode, args ):
      if 'PPS' in args:
         setBroadcastLevelPps( mode, args[ 'PPS' ] )
      elif 'RATE' in args:
         setBroadcastLevelBps( mode, args[ 'RATE' ], args[ 'RATE_UNIT' ] )
      else:
         setBroadcastLevelPercentage( mode, args[ 'LEVEL' ] )

   noOrDefaultHandler = noBroadcastLevel

SwitchportModelet.addCommandClass( StormControlBroadcastLevelLevelCmd )

#--------------------------------------------------------------------------------
# [ no | default ] storm-control multicast level ( LEVEL | ( pps PPS ) )
# [ no | default ] storm-control multicast level rate RATE RATE_UNIT
#--------------------------------------------------------------------------------
def setMulticastThreshold( mode, threshold ):
   name = mode.intf.name
   if( threshold == defaultThreshold and name not in config.intfConfig ):
      return
   if name in config.intfConfig:
      config.intfConfig[ name ].multicastLevel = threshold
      if stormControlDefaultConfig( name ):
         del config.intfConfig[ name ]
   else:
      config.newIntfConfig( name )
      config.intfConfig[ name ].multicastLevel = threshold

def setMulticastLevelPercentage( mode, level ):
   setMulticastThreshold(
      mode, Tac.Value( 'Bridging::StormControl::Threshold', 'percentage',
                       int( 0.5 + ( level * maxLevelPercentage ) / 100 ) ) )

def setMulticastLevelPps( mode, level ):
   setMulticastThreshold(
      mode, Tac.Value( 'Bridging::StormControl::Threshold', 'packetsPerSecond',
                       level ) )

def setMulticastLevelBps( mode, rate, rateUnit ):
   rateToSet = rate * BPS_FACTOR[ rateUnit ]
   if rateToSet > defaultThreshold.maxLevelBps:
      wantRate = '%s %s' % ( rate, rateUnit )
      maxRate = '%s %s' % convertToHighestBps( defaultThreshold.maxLevelBps ) 
      mode.addError( BPS_EXCEED_MAX_ERROR % ( wantRate.lower(), maxRate.lower() ) )
      return
   threshold = Tac.Value( 'Bridging::StormControl::Threshold', 'bitsPerSecond',
                          rateToSet )
   threshold.bitResolution = BPS_RESOLUTION[ rateUnit ]
   setMulticastThreshold( mode, threshold )

def noMulticastLevel( mode, args ):
   setMulticastLevelPercentage( mode, 100 )

class StormControlMulticastLevelLevelCmd( CliCommand.CliCommandClass ):
   syntax = ( 'storm-control multicast level ( LEVEL | ( pps PPS ) | '
              '( rate RATE RATE_UNIT ) )' )
   noOrDefaultSyntax = 'storm-control multicast ...'
   data = {
      'storm-control': stormControlKw,
      'multicast': 'Configure storm control for multicast packets',
      'level': levelKwMatcher,
      'LEVEL': levelMatcher,
      'pps': nodePps,
      'PPS': ppsMatcher,
      'rate': nodeBpsRate,
      'RATE': bpsMatcher,
      'RATE_UNIT': bpsRateUnitMatcher, 
   }
   @staticmethod
   def handler( mode, args ):
      if 'PPS' in args:
         setMulticastLevelPps( mode, args[ 'PPS' ] )
      elif 'RATE' in args:
         setMulticastLevelBps( mode, args[ 'RATE' ], args[ 'RATE_UNIT' ] )
      else:
         setMulticastLevelPercentage( mode, args[ 'LEVEL' ] )

   noOrDefaultHandler = noMulticastLevel

SwitchportModelet.addCommandClass( StormControlMulticastLevelLevelCmd )

#--------------------------------------------------------------------------------
# [ no | default ] storm-control all level ( LEVEL | ( pps PPS ) )
# [ no | default ] storm-control all level rate RATE RATE_UNIT
#--------------------------------------------------------------------------------
def setAllThreshold( mode, threshold ):
   name = mode.intf.name
   if( threshold == defaultThreshold and name not in config.intfConfig ):
      return
   if name in config.intfConfig:
      config.intfConfig[ name ].allLevel = threshold
      if stormControlDefaultConfig( name ):
         del config.intfConfig[ name ]
   else:
      config.newIntfConfig( name )
      config.intfConfig[ name ].allLevel = threshold

      if( config.intfConfig[ name ].uucastLevel != defaultThreshold or
         config.intfConfig[ name ].broadcastLevel != defaultThreshold or
          config.intfConfig[ name ].multicastLevel != defaultThreshold ):
         mode.addWarning( "'storm-control all' will override broadcast,"
                          " multicast and unknown-unicast limits" )

def setAllLevelPercentage( mode, level ):
   setAllThreshold( mode,
                    Tac.Value( 'Bridging::StormControl::Threshold',
                               'percentage',
                               int( 0.5 + ( level * maxLevelPercentage ) / 100 ) ) )

def setAllLevelPps( mode, level ):
   setAllThreshold( mode,
                    Tac.Value( 'Bridging::StormControl::Threshold',
                               'packetsPerSecond', level ) )

def setAllLevelBps( mode, rate, rateUnit ):
   rateToSet = rate * BPS_FACTOR[ rateUnit ]
   if rateToSet > defaultThreshold.maxLevelBps:
      wantRate = '%s %s' % ( rate, rateUnit )
      maxRate = '%s %s' % convertToHighestBps( defaultThreshold.maxLevelBps ) 
      mode.addError( BPS_EXCEED_MAX_ERROR % ( wantRate.lower(), maxRate.lower() ) )
      return
   threshold = Tac.Value( 'Bridging::StormControl::Threshold', 'bitsPerSecond',
                          rateToSet )
   threshold.bitResolution = BPS_RESOLUTION[ rateUnit ]
   setAllThreshold( mode, threshold )

def noAllLevel( mode, args ):
   setAllLevelPercentage( mode, 100 )

class StormControlAllLevelLevelCmd( CliCommand.CliCommandClass ):
   syntax = ( 'storm-control all level ( LEVEL | ( pps PPS ) | '
              '( rate RATE RATE_UNIT ) )' )
   noOrDefaultSyntax = 'storm-control all ...'
   data = {
      'storm-control': stormControlKw,
      'all': CliCommand.guardedKeyword( 'all',
         helpdesc='Configure storm control for all packets',
         guard=stormControlAllLevelGuard ),
      'level': levelKwMatcher,
      'LEVEL': levelMatcher,
      'pps': nodePps,
      'PPS': ppsMatcher,
      'rate': nodeBpsRate,
      'RATE': bpsMatcher,
      'RATE_UNIT': bpsRateUnitMatcher, 
   }
   @staticmethod
   def handler( mode, args ):
      if 'PPS' in args:
         setAllLevelPps( mode, args[ 'PPS' ] )
      elif 'RATE' in args:
         setAllLevelBps( mode, args[ 'RATE' ], args[ 'RATE_UNIT' ] )
      else:
         setAllLevelPercentage( mode, args[ 'LEVEL' ] )

   noOrDefaultHandler = noAllLevel

SwitchportModelet.addCommandClass( StormControlAllLevelLevelCmd )

#-------------------------------------------------------------------------------
# The "[no|default] logging event storm-control discards interval <val>" command, 
# in "config" mode.
# This command sets storm-control drop logging interval for all storm control 
# enabled interfaces
#-------------------------------------------------------------------------------
stormControlDropKw = CliCommand.guardedKeyword(
   'discards',
   'Discards due to storm control',
   stormControlDropCountGuard )

class LoggingEventStormControlInterval( IntfCli.LoggingEventGlobalCmd ):
   syntax = "logging event storm-control discards interval INTERVAL"
   noOrDefaultSyntax = "logging event storm-control discards interval ..."
   data = {
      'storm-control' : stormControlKw,
      'discards' : stormControlDropKw,
      'interval' : 'Logging interval',
      'INTERVAL' : CliMatcher.IntegerMatcher( 10, 65535,
                                            helpdesc='Logging interval in seconds' )
      }
   @staticmethod
   def handler( mode, args ):
      config.stormControlDropLogInterval = args.get( 'INTERVAL', 0 )

   noOrDefaultHandler = handler

BasicCli.GlobalConfigMode.addCommandClass( LoggingEventStormControlInterval )

#-------------------------------------------------------------------------------
# the global "[no|default] logging event storm-control discards global"
# command in config mode
# This command enables storm-control drop logging on all storm control enabled
# interfaces. This can be overriden by the interface specific command
#-------------------------------------------------------------------------------
class LoggingEventStormControlGlobal( IntfCli.LoggingEventGlobalCmd ):
   syntax = "logging event storm-control discards global"
   noOrDefaultSyntax = syntax
   data = {
      'storm-control' : stormControlKw,
      'discards' : stormControlDropKw,
      'global' : 'Configure global storm control discard logging'
      }
   @staticmethod
   def handler( mode, args ):
      config.stormControlDropLoggingMode = 'on'

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      config.stormControlDropLoggingMode = 'off'

BasicCli.GlobalConfigMode.addCommandClass( LoggingEventStormControlGlobal )

#------------------------------------------------------------------------------- 
# [no|default] logging event storm-control discards [ use-global ]
# interface specific enable disable command.
#-------------------------------------------------------------------------------
class LoggingEventStormControlIntf( IntfCli.LoggingEventIntfCmd ):
   syntax = "logging event stormcontrol discards"
   data = {
      'stormcontrol' : stormControlKw,
      'discards' : stormControlDropKw
      }
   @classmethod
   def _enableHandler( cls, mode, args ):
      config.newIntfConfig( mode.intf.name ).stormControlDropLoggingMode = 'on'

   @classmethod
   def _disableHandler( cls, mode, args ):
      config.newIntfConfig( mode.intf.name ).stormControlDropLoggingMode = 'off'

   @classmethod
   def _useGlobalHandler( cls, mode, args ):
      name = mode.intf.name
      intfConfig = config.intfConfig.get( name )
      if intfConfig:
         intfConfig.stormControlDropLoggingMode = 'useGlobal'
         if stormControlDefaultConfig( name ):
            del config.intfConfig[ name ]

SwitchportModelet.addCommandClass( LoggingEventStormControlIntf )

#-------------------------------------------------------------------------------
# the "show storm-control [<name>]" command, in enable mode
#-------------------------------------------------------------------------------
def showStormControlAggregate( mode, tcList ):
   aggregateTcHdrFormat = '%-10s %-18s %-12s %-5s'
   print 'Aggregate:'
   print aggregateTcHdrFormat % ( 'Type', 'Traffic Class', 'Level', 'Units' )
   for tc in tcList:
      level = config.aggregateTcConfig[ tc ].level
      print aggregateTcHdrFormat % ( 'BUM', tc, level.level, 'pps' )
   print

def showStormControl( mode, args ):
   intf = args.get( 'INTF' )
   intfs = []
   tcList = sorted( config.aggregateTcConfig.keys() )
   if status().aggregateStormControlSupported and intf is None \
         and tcList: 
      showStormControlAggregate( mode, tcList )
   
   if isinstance( intf, Intf.IntfRange.IntfList ):
      for i in intf.intfNames():
         intfs.append( i )
   elif intf:
      intfs = [ intf.name ]
   else:
      intfs = []
      if status().dropCountSupported:
         for i in status().allIntfStatus:
            intfs.append( i )
      elif stormControlSupported():
         for sliceStatus in sliceStatusDir().itervalues():
            for i in sliceStatus.intfStatus:
               intfs.append( i )
   # Port-channel interfaces can have status entries in more than one linecards
   # Remove duplicates possible with port-channels and sort
   intfs = Arnet.sortIntf( set( intfs ) )
   if not( intfs or tcList ):
      print "No storm control configured on any interface"
      return

   if not intfs:
      return
   # storm control count is only available on some platforms
   cntSupported = status().dropCountSupported

   hdrFormat = '%-10s %-16s %-8s %-5s %-10s %-8s %s %s'
   print hdrFormat % ( 'Port', 'Type', 'Level', 'Units', 'Rate(Mbps)', 'Status',
                       '     Drops' if cntSupported else '' , 'Reason' )

   mem2PortChannels = {}
   def getAllIntfsInPortChannels( intfList ):
      for poIntfName in lagStatus.intfStatus:
         for member in lagStatus.get( poIntfName ).member.keys():
            memShortName = IntfCli.Intf.getShortname( member )
            poShortName = IntfCli.Intf.getShortname( poIntfName )
            intfList[ memShortName ] = poShortName
      return intfList

   getAllIntfsInPortChannels( mem2PortChannels )

   for i in intfs:
      name = IntfCli.Intf.getShortname( i )
      statusInst = None
      dropCnt = { 'all': 0, 'multicast': 0, 'broadcast': 0, 'uucast':0 }
      totalRates = { 'all': 0, 'multicast': 0, 'broadcast': 0, 'uucast':0 }
      
      if status().dropCountSupported:
         if i not in status().allIntfStatus:
            continue

         intfStatus = None
         for intfStatus in status().allIntfStatus[ i ].intfStatus.keys():
            dropCnt[ 'all' ] += intfStatus.allDrop
            dropCnt[ 'broadcast' ] += intfStatus.broadcastDrop
            dropCnt[ 'multicast' ] += intfStatus.multicastDrop
            dropCnt[ 'uucast' ] += intfStatus.uucastDrop
            totalRates[ 'all' ] += intfStatus.allRate
            totalRates[ 'broadcast' ] += intfStatus.broadcastRate
            totalRates[ 'multicast' ] += intfStatus.multicastRate
            totalRates[ 'uucast' ] += intfStatus.uucastRate
      elif stormControlSupported():
         for sliceStatus in sliceStatusDir().itervalues():
            if i in sliceStatus.intfStatus:
               statusInst = sliceStatus
               totalRates[ 'all' ] += statusInst.intfStatus[ i ].allRate
               totalRates[ 'broadcast' ] += statusInst.intfStatus[ i ].broadcastRate
               totalRates[ 'multicast' ] += statusInst.intfStatus[ i ].multicastRate
               totalRates[ 'uucast' ] += statusInst.intfStatus[ i ].uucastRate
         if statusInst is None:
            continue
         intfStatus = statusInst.intfStatus[ i ]
      else:
         intfStatus = None
      displayLines = []
      stormControlAll = False
      
      if intfStatus.allLevel != defaultThreshold:
         if intfStatus.allLevel.type == 'percentage':
            displayLines.append(
               ("all", ('%u.%02d' % ( intfStatus.allLevel.level / 100,
                                      intfStatus.allLevel.level % 100 ) ),
                '%', totalRates[ 'all' ], False,
                ('%10d' %  dropCnt[ 'all' ] ) if cntSupported
                else '') )
         elif intfStatus.allLevel.type == 'packetsPerSecond':
            displayLines.append(
               ("all", ('%u' % intfStatus.allLevel.level ),
                'pps', '-', False,
                ('%10d' %  dropCnt[ 'all' ] ) if cntSupported
                else '') )           
         elif intfStatus.allLevel.type == 'bitsPerSecond':
            rate, unit = convertToHighestBps( intfStatus.allLevel.level )
            displayLines.append(
               ( 'all', ('%u' % rate ),
                 unit, '-', False,
                 ( '%10d' % ( dropCnt[ 'all' ] ) )
                 if cntSupported else '') )
         else:
            assert False, 'This level type is not defined'
         stormControlAll = True

      if intfStatus.uucastLevel != defaultThreshold:
         if intfStatus.uucastLevel.type == 'percentage':
            displayLines.append(
               ("unknown-unicast", ('%u.%02d' %
                              ( intfStatus.uucastLevel.level / 100,
                                intfStatus.uucastLevel.level % 100 ) ),
                '%', totalRates[ 'uucast' ], stormControlAll,
                ( '%10d' % ( dropCnt[ 'uucast' ] ) )
                if cntSupported else '') )
         elif intfStatus.uucastLevel.type == 'packetsPerSecond':
            displayLines.append(
               ("unknown-unicast", ('%u' % intfStatus.uucastLevel.level ),
                'pps', '-', stormControlAll,
                ( '%10d' % ( dropCnt[ 'uucast' ] ) )
                if cntSupported else '') )
         elif intfStatus.uucastLevel.type == 'bitsPerSecond':
            rate, unit = convertToHighestBps( intfStatus.uucastLevel.level )
            displayLines.append(
               ( 'unknown-unicast', ('%u' % rate ),
                 unit, '-', stormControlAll,
                 ( '%10d' % ( dropCnt[ 'uucast' ] ) )
                 if cntSupported else '') )
         else:
            assert False, 'This level type is not defined'

      if intfStatus.broadcastLevel != defaultThreshold:
         if intfStatus.broadcastLevel.type == 'percentage':
            displayLines.append(
               ("broadcast", ('%u.%02d' %
                              ( intfStatus.broadcastLevel.level / 100,
                                intfStatus.broadcastLevel.level % 100 ) ),
                '%', totalRates[ 'broadcast' ], stormControlAll,
                ( '%10d' % ( dropCnt[ 'broadcast' ] ) )
                if cntSupported else '') )
         elif intfStatus.broadcastLevel.type == 'packetsPerSecond':
            displayLines.append(
               ("broadcast", ('%u' % intfStatus.broadcastLevel.level ),
                'pps', '-', stormControlAll,
                ( '%10d' % ( dropCnt[ 'broadcast' ] ) )
                if cntSupported else '') )
         elif intfStatus.broadcastLevel.type == 'bitsPerSecond':
            rate, unit = convertToHighestBps( intfStatus.broadcastLevel.level )
            displayLines.append(
               ( 'broadcast', ('%u' % rate ),
                 unit, '-', stormControlAll,
                 ( '%10d' % ( dropCnt[ 'broadcast' ] ) )
                 if cntSupported else '') )
         else:
            assert False, 'This level type is not defined'

      if intfStatus.multicastLevel != defaultThreshold: 
         if intfStatus.multicastLevel.type == 'percentage':
            displayLines.append(
               ("multicast", ('%u.%02d' %
                              ( intfStatus.multicastLevel.level / 100,
                                intfStatus.multicastLevel.level % 100 ) ),
                '%', totalRates[ 'multicast' ], stormControlAll,
                ( '%10d' % ( dropCnt[ 'multicast' ] ) )
                if cntSupported else '') )
         elif intfStatus.multicastLevel.type == 'packetsPerSecond':
            displayLines.append(
               ("multicast", ('%u' % intfStatus.multicastLevel.level ),
                'pps', '-', stormControlAll,
                ( '%10d' % ( dropCnt[ 'multicast' ] ) )
                if cntSupported else '') )
         elif intfStatus.multicastLevel.type == 'bitsPerSecond':
            rate, unit = convertToHighestBps( intfStatus.multicastLevel.level )
            displayLines.append(
               ( 'multicast', ('%u' % rate ),
                 unit, '-', stormControlAll,
                 ( '%10d' % ( dropCnt[ 'multicast' ] ) )
                 if cntSupported else '') )
         else:
            assert False, 'This level type is not defined'
   
      firstTime = True
      for ( sctype, level, units, rate, dormant, drop ) in displayLines:
         intfName = name if firstTime else ''
         stormCtrStatus = 'active'
         reason = ''

         if name.startswith( 'Et' ) and name in mem2PortChannels.keys():
            stormCtrStatus = 'inactive'
            reason = 'member of %s' % mem2PortChannels[ name ]
         elif name.startswith( 'Po' ) and not lagStatus.get( i ).member:
            stormCtrStatus = 'inactive'            
            reason = 'no members'
         elif dormant:
            stormCtrStatus = 'inactive'
            reason = 'storm-control all active'
         print hdrFormat % ( intfName, sctype, str( level ), str( units ),
                             str( rate ),
                          stormCtrStatus, drop, reason )
         firstTime = False

class StormControlCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show storm-control [ INTF ]'
   data = {
      'storm-control': CliCommand.guardedKeyword( 'storm-control',
         helpdesc='Configure storm-control', guard=stormControlGuard ),
      'INTF': IntfCli.Intf.rangeMatcher,
   }
   handler = showStormControl
   privileged = True

BasicCli.addShowCommandClass( StormControlCmd )

defaultBurst = Tac.Value( "Bridging::StormControl::BurstInfo" )

def stormControlBurstGuard( mode, token ):
   if status().stormControlBurstSupported:
      return None
   else:
      for sliceStatus in sliceStatusDir().itervalues():
         if sliceStatus.stormControlBurstSupported:
            return None
         else:
            return CliParser.guardNotThisPlatform

def stormControlBurstDefaultConfig( name ):
   burstConfig = config.burstConfig[ name ]
   return burstConfig.burst == defaultBurst

def stormBurstTimeRange( mode ):
   minBurst = 10
   return( minBurst, defaultBurst.maxBurstTimeUs )

#-------------------------------------------------------------------------------
# General tokens
#-------------------------------------------------------------------------------
stormControlBurstKw = CliCommand.guardedKeyword( 'storm-control',
      helpdesc='Configure storm-control',
      guard=stormControlBurstGuard )
nodeBurst = CliMatcher.KeywordMatcher( 'burst',
      helpdesc='Configure the maximum storm control burst time' )
nodeTime = CliMatcher.KeywordMatcher( 'time',
      helpdesc='Configure the maximum storm control burst time' )
burstMatcher = CliCommand.Node(
      matcher=CliMatcher.DynamicIntegerMatcher( rangeFn=stormBurstTimeRange,
      helpdesc=
      'Maximum burst time in milliseconds or microseconds allowed by storm control',
      helpname='TIME' ) )

#--------------------------------------------------------------------------------
# [ no | default ] storm-control burst time TIME (milliseconds | microseconds)
#--------------------------------------------------------------------------------
def setBurst( mode, args ):
   time = args[ 'TIME' ]
   units = args[ 'UNITS' ]
   value = Tac.Value( 'Bridging::StormControl::BurstInfo', units, time )
   name = "burstConfig"
   if name in config.burstConfig:
      config.burstConfig[ name ].burst = value
      if stormControlBurstDefaultConfig( name ):
         del config.burstConfig[ name ]
   else:
      config.newBurstConfig( name )
      config.burstConfig[ name ].burst = value

def noBurst( mode, args ):
   name = "burstConfig"
   if name in config.burstConfig:
      del config.burstConfig[ name ]

class StormControlBurstCmd( CliCommand.CliCommandClass ):
   syntax = ( 'storm-control burst time TIME UNITS' )
   noOrDefaultSyntax = 'storm-control burst ...'
   data = {
      'storm-control': stormControlBurstKw,
      'burst' : nodeBurst,
      'time' : nodeTime,
      'TIME' : burstMatcher,
      'UNITS' : CliMatcher.EnumMatcher( {
         'microseconds':
         'Configure the maximum storm control burst in microseconds',
         'milliseconds':
         'Configure the maximum storm control burst in microseconds',
      } ),
   }
   handler = setBurst
   noOrDefaultHandler = noBurst

BasicCli.GlobalConfigMode.addCommandClass( StormControlBurstCmd )

#-------------------------------------------------------------------------------
# Destroy method of StormControlIntf class is called by Intf class when an interface
# object is deleted. Intf class will create a new instance of StormControlIntf
# and call destroy method on it.
# ------------------------------------------------------------------------------

def stormControlConfig():
   return config

class StormControlIntf( IntfCli.IntfDependentBase ):
   #----------------------------------------------------------------------------
   # Destroys the StormControlIntf IntfConfig object for this interface if it exists.
   #----------------------------------------------------------------------------
   def setDefault( self ):
      del stormControlConfig().intfConfig[ self.intf_.name ]

#-------------------------------------------------------------------------------
# Have the Cli Agent mount all needed state from sysdb
#-------------------------------------------------------------------------------
def Plugin( entityManager ):
   global config, statusDir, lagStatus
   config = ConfigMount.mount( entityManager, "bridging/stormcontrol/config",
                               "Bridging::StormControl::Config", "w" )
   statusDir = LazyMount.mount( entityManager, "bridging/stormcontrol/status",
                                "Tac::Dir", "ri" )
   lagStatus = LazyMount.mount( entityManager, "interface/status/eth/lag",
                                "Interface::EthLagIntfStatusDir", "r" )

   IntfCli.Intf.registerDependentClass( StormControlIntf )

