# Copyright (c) 2011 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import BasicCliUtil
import CliPlugin.FruCli as FruCli
import CliPlugin.FruModel as FruModel
import CliPlugin.ReloadCli as ReloadCli
import CliPlugin.TechSupportCli
import CliPlugin.StorageDevicesModels as StorageDevicesModels
import MmcFlashLib
import os
import SmartLib

# StorageDevShowInvCallback
#     fs --> hardware/entmib/fixedSystem
def StorageDevShowInvCallback ( entMib, model ):
   storageDevices = []
   if entMib.fixedSystem:
      storageDevices = entMib.fixedSystem.storageDevices.values()
   elif entMib.chassis:
      # Assume modular system
      for slot in sorted( entMib.chassis.cardSlot.values() ):
         card = slot.card
         if card:
            storageDevices += slot.card.storageDevices.values()

   mntFlashKey = None
   for storageDeviceMibEnt in storageDevices:
      if storageDeviceMibEnt.mount == '/mnt/flash':
         mntFlashKey = storageDeviceMibEnt.modelName.strip()
         break
   suffix = 2

   for storageDeviceMibEnt in storageDevices:
      modelName = storageDeviceMibEnt.modelName.strip()
      serialNum = storageDeviceMibEnt.serialNum.strip()
      firmwareRev = storageDeviceMibEnt.firmwareRev.strip()
      storageSize = storageDeviceMibEnt.sizeGB
      storageType = FruModel.storageTypes.get(
         storageDeviceMibEnt.storageType.upper(), 'unknown' )

      key = modelName
      if storageDeviceMibEnt.mount != '/mnt/flash':
         if key == mntFlashKey or key in model.emmcFlashDevices:
            key += '-%d' % suffix
            suffix += 1

      if storageType == 'eMMC':
         model.emmcFlashDevices[ key ] = FruModel.EmmcFlashDevice(
                   model=modelName,
                   mount=storageDeviceMibEnt.mount,
                   serialNum=serialNum,
                   firmwareRev=firmwareRev,
                   storageSize=storageSize )
      else:
         model.storageDevices[ key ] = FruModel.StorageDevice(
                   model=modelName,
                   mount=storageDeviceMibEnt.mount,
                   storageType=storageType,
                   serialNum=serialNum,
                   firmwareRev=firmwareRev,
                   storageSize=storageSize )

def showSystemHealthStorage( mode, args ):
   storageHealth = StorageDevicesModels.Devices()

   def getDevice( name ):
      device = storageHealth.devices.get( name )
      if device is None:
         device = StorageDevicesModels.StorageHealth()
         storageHealth.devices[ name ] = device
      return device

   smartHealths = SmartLib.smartHealthStatus()
   for name, info in smartHealths.items():
      device = getDevice( name )
      health = StorageDevicesModels.SmartHealth()
      health.reallocationsRemaining = info.get( SmartLib.REALLOCATIONS )
      health.wearLevel = info.get( SmartLib.WEAR_LIFE )
      health.health = info.get( SmartLib.HEALTH )

      device.smart = health
      storageHealth.devices[ name ] = device

   emmcLifetimes = MmcFlashLib.emmcLifetimes()
   for name, info in emmcLifetimes.items():
      device = getDevice( name )
      lifetime = StorageDevicesModels.EmmcLifetime()

      lifetime.slcWearLevel = info.get( MmcFlashLib.SLC )
      lifetime.mlcWearLevel = info.get( MmcFlashLib.MLC )
      lifetime.reservesLevel = info.get( MmcFlashLib.RESERVES )

      device.emmc = lifetime
      storageHealth.devices[ name ] = device

   return storageHealth

def resetSystemStorage( mode, args ):
   warningStr = ( "WARNING! This will destroy all\r\n"
                  "data and will NOT be recoverable.\r\n"
                  "Device will reboot into Aboot, and\r\n"
                  "execution may take up to one hour.\r\n"
                  "Would you like to proceed? [y/N] " )
   confirm = BasicCliUtil.confirm( mode, prompt=warningStr, answerForReturn=False )
   if not confirm:
      return

   def devExists( baseDir, devName ):
      return os.path.exists( os.path.join( baseDir, "%s.conf" % devName ) )

   devNames = [ 'crash', 'drive', 'flash' ]
   baseDir = ''
   # FILESYSTEM_ROOT is a temporary directory created during
   # breadth test Cli initialization
   baseDir = os.environ.get( 'FILESYSTEM_ROOT', '/mnt' )
   trigPath = os.path.join( baseDir, 'flash', '.trigResetSysStorage' )
   devStr = ''

   os.mknod( trigPath )

   devNames = [ dev + ':' for dev in devNames if devExists( baseDir, dev ) ]
   if len( devNames ) == 1:
      devStr = devNames[ 0 ]
   else:
      devStr += ', '.join( devNames[ :-1 ] )
      devStr += ' & %s' % devNames[ -1 ]

   mode.addWarning( "Rebooting to process %s" % devStr )
   # Don't try to reboot if running as btest
   if 'A4_CHROOT' not in os.environ:
      ReloadCli.doReload( mode, power=True, now=True )

#-------------------------------------------------------------------------------
# Register "show tech-support" commands
#-------------------------------------------------------------------------------
CliPlugin.TechSupportCli.registerShowTechSupportCmdCallback(
      '2018-10-15 16:56:01', lambda: [ 'show system health storage' ],
      summaryCmdCallback=lambda: [
         "show system health storage",
         ] )

#--------------------------------------------------
# Plugin method - Mount the objects we need from Sysdb
#--------------------------------------------------
def Plugin( entityManager ): #pylint: disable-msg=W0613
   # Register Show Inventory callback for Storage Devices.
   # Currently, there are no objects to be mounted.
   # This Plugin only registers a display function for 'show inventory'
   # command (Defined in FruCli.py). Show inventory takes care of mounting
   # appropriate objects.
   FruCli.registerShowInventoryCallback( StorageDevShowInvCallback )
   return
