# Copyright (c) 2014 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

from SslModel import Certificate
from SslModel import CertificateSigningRequest
from SslModel import Crl
from SslModel import Extension
from SslModel import PublicKey
from SslModel import DistinguishedName
from SslModel import DiffieHellmanParameters
from SslModel import RevokedCertificate
from SslModel import utc
from dateutil import parser
import datetime
import time
import Tac
import re

def _getDates( certFile ):
   output = Tac.run( [ 'openssl', 'x509', '-in', certFile, '-dates', '-noout' ], 
                    stdout=Tac.CAPTURE, stderr=Tac.CAPTURE,
                    ignoreReturnCode=True )
   matchObj = re.match( r'notBefore=(.*?)\s*notAfter=(.*)', output )
   return matchObj.group( 1 ), matchObj.group( 2 )
   
def getCertificateModel( certFile ):
   # ssl is a marked as memory hog module in Cli.
   # So, conditionally imported the ssl module only
   # when ssl related commands are executed.
   from M2Crypto import X509, m2
   import SslCertKey
   certModel = Certificate()
   certModel.subject = DistinguishedName()
   certModel.issuer = DistinguishedName()
   
   cert = X509.load_cert_string( SslCertKey.removeTrusted( certFile ) )
   
   certModel.version = cert.get_version()  + 1
   certModel.serialNumber = cert.get_serial_number()
   
   subject = cert.get_subject()
   certModel.subject.commonName = ""
   if subject.CN:
      certModel.subject.commonName = subject.CN
   if subject.Email:
      certModel.subject.email = subject.Email
   if subject.O:
      certModel.subject.organization = subject.O
   if subject.OU:
      certModel.subject.organizationUnit = subject.OU
   if subject.L:
      certModel.subject.locality = subject.L
   if subject.SP:
      certModel.subject.stateOrProvince = subject.SP
   if subject.C:
      certModel.subject.country = subject.C
   
   issuer = cert.get_issuer()
   certModel.issuer.commonName = ""
   if issuer.CN:
      certModel.issuer.commonName = issuer.CN
   if issuer.Email:
      certModel.issuer.email = issuer.Email
   if issuer.O:
      certModel.issuer.organization = issuer.O
   if issuer.OU:
      certModel.issuer.organizationUnit = issuer.OU
   if issuer.L:
      certModel.issuer.locality = issuer.L
   if issuer.SP:
      certModel.issuer.stateOrProvince = issuer.SP
   if issuer.C:
      certModel.issuer.country = issuer.C
      
   epochDt = datetime.datetime( 1970, 1, 1, tzinfo=utc )
   # The M2Crypto errors out for away dates
   # hence using openssl command to extract dates
   ( nb, na ) = _getDates( certFile )
   notBeforeDt = parser.parse( nb, tzinfos={ 'GMT' : utc } )
   notAfterDt = parser.parse( na, tzinfos={ 'GMT' : utc } )
   
   certModel.notBefore = int( ( notBeforeDt - epochDt ).total_seconds() )
   certModel.notAfter = int( ( notAfterDt - epochDt ).total_seconds() )
      
   pkey = cert.get_pubkey()
   
   publicKey =  PublicKey()
   with open( certFile, 'r' ) as fp:
      certData = fp.read()
      publicKey.encryptionAlgorithm = SslCertKey.getCertPublicKeyAlgo( certData )
      publicKey.size = SslCertKey.getCertPublicKeySize( certData )
      if publicKey.encryptionAlgorithm == 'RSA':
         rsakey = pkey.get_rsa()
         publicKey.modulus = int( pkey.get_modulus(), 16 )
         publicKey.publicExponent = int( m2.bn_to_hex( m2.mpi_to_bn( rsakey.e ) ),
                                         16 )
   certModel.publicKey = publicKey
   
   numExtension = cert.get_ext_count()
   for i in range( 0, numExtension ):
      ext = cert.get_ext_at( i )
      if not ext:
         continue
      
      try:
         extName = ext.get_name()
         extValue = ext.get_value()
         extCritical = ext.get_critical()
      except: # pylint: disable-msg=W0702
         continue
      
      if not extName or not extValue:
         continue
      
      certModel.extension[ extName ] = Extension( 
                                        value=extValue.strip(),
                                        critical=bool( extCritical ) )
   return certModel
   
   
def getPublicKeyModel( keyFile ):
   from M2Crypto import RSA, m2
   keyModel = PublicKey()
   rsa = RSA.load_key( keyFile )
   ( e, m ) = rsa.pub()
   keyModel.size = len( rsa )
   keyModel.publicExponent = int( m2.bn_to_hex( m2.mpi_to_bn( e ) ), 16 )
   keyModel.modulus = int( m2.bn_to_hex( m2.mpi_to_bn( m ) ), 16 )
   keyModel.encryptionAlgorithm = "RSA"
   
   return keyModel

def getCrlModel( crlFile ):
   crlModel = Crl()
   crlModel.issuer = DistinguishedName()
   output = Tac.run( [ 'openssl', 'crl', '-noout', '-crlnumber', '-issuer',
                       '-lastupdate', '-nextupdate', '-in', crlFile ],
                     stdout=Tac.CAPTURE, stderr=Tac.CAPTURE,
                     ignoreReturnCode=True )
   values = dict( k.split( '=', 1 ) for k in output.strip().split( '\n' ) )
   crlModel.crlNumber = int( values[ 'crlNumber' ], 16 )
   issuer = dict( k.split( '=', 1 ) for k in
                  values[ 'issuer' ].strip( '/' ).split( '/' ) )
   crlModel.issuer.commonName = issuer.get( 'CN' )
   crlModel.issuer.email = issuer.get( 'emailAddress' )
   crlModel.issuer.organization = issuer.get( 'O' )
   crlModel.issuer.organizationUnit = issuer.get( 'OU' )
   crlModel.issuer.stateOrProvince = issuer.get( 'ST' )
   crlModel.issuer.country = issuer.get( 'C' )
   lastUpdate = parser.parse( values[ 'lastUpdate' ], tzinfos={ 'GMT' : utc } )
   nextUpdate = parser.parse( values[ 'nextUpdate' ], tzinfos={ 'GMT' : utc } )
   
   epochDt = datetime.datetime( 1970, 1, 1, tzinfo=utc )
   crlModel.lastUpdate = int( ( lastUpdate - epochDt ).total_seconds() )
   crlModel.nextUpdate = int( ( nextUpdate - epochDt ).total_seconds() )

   #Get list of revoked certs in CRL
   txtOutput = Tac.run( [ 'openssl', 'crl', '-noout', '-text', '-in', crlFile ],
               stdout=Tac.CAPTURE, stderr=Tac.CAPTURE, ignoreReturnCode=True )
   pattern = r'\s+Serial Number: (?P<serial>\w+)\s+Revocation Date: (?P<date>.*)'
   for match in re.finditer( pattern, txtOutput, re.MULTILINE ):
      serialNumber, revocationDate = match.group( 'serial', 'date' )
      revokedCert = RevokedCertificate()
      revokedCert.serialNumber = serialNumber
      revokedCert.revocationDate = float( time.mktime( time.strptime(
          revocationDate, '%b %d %H:%M:%S %Y GMT')))
      crlModel.revokedList.append( revokedCert )

   return crlModel

def getDhparamsModel( dhparamsFile ):
   from M2Crypto import DH, m2
   dhModel = DiffieHellmanParameters()
   dh = DH.load_params( dhparamsFile )
   dhModel.size = len( dh ) << 3
   dhModel.prime = int( m2.bn_to_hex( m2.mpi_to_bn( dh.p ) ), 16 )
   dhModel.generator = int( m2.bn_to_hex( m2.mpi_to_bn( dh.g ) ), 16 )
   return dhModel

def getCertificateSigningRequestModel( csrFile ):
   # ssl is a marked as memory hog module in Cli.
   # So, conditionally imported the ssl module only
   # when ssl related commands are executed.
   from M2Crypto import X509, m2
   csrModel = CertificateSigningRequest()
   csrModel.subject = DistinguishedName()
   with open( csrFile ) as f:
      pemValue = f.read()

   csrModel.pemValue = pemValue
   csr = X509.load_request_string( pemValue )
   csrModel.version = csr.get_version()
   subject = csr.get_subject()
   csrModel.subject.commonName = subject.CN if subject.CN else ''
   csrModel.subject.email = subject.Email if subject.Email else None
   csrModel.subject.organization = subject.O if subject.O else None
   csrModel.subject.organizationUnit = subject.OU if subject.OU else None
   csrModel.subject.locality = subject.L if subject.L else None
   csrModel.subject.stateOrProvince = subject.SP if subject.SP else None
   csrModel.subject.country = subject.C if subject.C else None

   pkey = csr.get_pubkey()
   rsakey = pkey.get_rsa()

   publicKey = PublicKey()
   publicKey.size = len( rsakey )
   publicKey.modulus = int( pkey.get_modulus(), 16 )
   publicKey.publicExponent = int( m2.bn_to_hex( m2.mpi_to_bn( rsakey.e ) ), 16 )
   publicKey.encryptionAlgorithm = "RSA"
   csrModel.publicKey = publicKey
   return csrModel
