# Copyright (c) 2014 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import BasicCli
import Tac
import DefaultSslProfile
import CliCommand
import CliParser
import ConfigMgmtMode
import ConfigMount
import LazyMount
import CliPlugin.Security as Security
import CliPlugin.SslModel as SslModel
import CliMode.Ssl as SslProfileMode
import SslCliLib
import CliToken.Reset
import os
import os.path
import re
import SslCertKey
import ShowCommand
import CliMatcher

config = None
status = None
execRequest = None
CertKeyPair = Tac.Type( "Mgmt::Security::Ssl::CertKeyPair" )
Constants = Tac.Type( "Mgmt::Security::Ssl::Constants" )

def _listdir( path ):
   if not os.path.isdir( path ):
      return []
   return os.listdir( path )

sslMatcher = CliMatcher.KeywordMatcher(
   'ssl',
   helpdesc='Configure SSL related options' )
profileMatcher = CliMatcher.KeywordMatcher(
   'profile',
   helpdesc='Configure SSL profile' )
certificateKwMatcher = CliMatcher.KeywordMatcher(
   'certificate',
   helpdesc='Configure certificate for self authentication' )
verifyKwMatcher = CliMatcher.KeywordMatcher(
   'verify',
   helpdesc='Verify specified field in certificate' )
trustKwMatcher = CliMatcher.KeywordMatcher(
   'trust',
   helpdesc='Configure trusted certificate' )
chainKwMatcher = CliMatcher.KeywordMatcher(
   'chain',
   helpdesc='Configure chained certificate' )
requirementKwMatcher = CliMatcher.KeywordMatcher(
   'requirement',
   helpdesc='Add a check to the certificate for validity' )

# Show tokens
sslShowMatcher = CliMatcher.KeywordMatcher(
   'ssl',
   helpdesc='Show SSL status' )

profileNameMatcher = CliMatcher.DynamicNameMatcher(
                                             lambda mode: config.profileConfig,
                                             'SSL profile name' )
statusProfileNameMatcher = CliMatcher.DynamicNameMatcher(
                                             lambda mode: status.profileStatus,
                                             'SSL profile name' )
certificateNameMatcher = CliMatcher.DynamicNameMatcher(
                                 lambda mode: _listdir( Constants.certsDirPath() ),
                                 'Certificate name',
                                 pattern=r'[A-Za-z0-9_:{}\.\[\]-]+' )
keyNameMatcher = CliMatcher.DynamicNameMatcher(
                                 lambda mode: _listdir( Constants.keysDirPath() ),
                                 'Key name',
                                 pattern=r'[A-Za-z0-9_:{}\.\[\]-]+' )

def _getOutput( cmd, inp=None ):
   return Tac.run( cmd, asRoot=True, stdout=Tac.CAPTURE, stderr=Tac.CAPTURE,
         ignoreReturnCode=True, input=inp )

class ProfileConfigModeBase( BasicCli.ConfigModeBase ):

   def __init__( self, parent, session, profileName, profileType ):
      BasicCli.ConfigModeBase.__init__( self, parent, session )
      self.profileName = profileName

      if self.profileName not in config.profileConfig:
         self.profileConfig_ = config.newProfileConfig( self.profileName )
         self.profileConfig_.profileType = profileType
      else:
         self.profileConfig_ = config.profileConfig[ self.profileName ]
         if self.profileConfig_.profileType != profileType:
            return

   def enableTlsVersion( self, mask, add=False ):
      if add:
         mask |= self.profileConfig_.tlsVersion
      self.profileConfig_.tlsVersion = mask

   def disableTlsVersion( self, mask ):
      version = self.profileConfig_.tlsVersion & ~mask
      if not version:
         self.addError( "Cannot disable all TLS versions" )
         return
      self.profileConfig_.tlsVersion = version

   def enableFipsMode( self ):
      self.profileConfig_.fipsMode = True

   def disableFipsMode( self ):
      self.profileConfig_.fipsMode = False

   def enableCiphers( self, ciphers ):
      if 'Error' in _getOutput( [ 'openssl', 'ciphers', ciphers ] ):
         self.addError( 'Invalid cipher list.' )
         return
      self.profileConfig_.cipherSuite = ciphers

   def disableCiphers( self ):
      self.profileConfig_.cipherSuite = Constants.defaultCipherSuite()

   def setCertKey( self, certificate, key ):
      self.profileConfig_.certKeyPair = CertKeyPair( certificate, key )

   def noCertKey( self ):
      self.profileConfig_.certKeyPair = CertKeyPair( "", "" )

   def addTrustedCert( self, trustedCert ):
      if not trustedCert in self.profileConfig_.trustedCert:
         self.profileConfig_.trustedCert[ trustedCert ] = 1

   def noTrustedCert( self, trustedCert ):
      if trustedCert in self.profileConfig_.trustedCert:
         del self.profileConfig_.trustedCert[ trustedCert ]

   def addChainedCert( self, chainedCert ):
      if not chainedCert in self.profileConfig_.chainedCert:
         self.profileConfig_.chainedCert[ chainedCert ] = 1

   def noChainedCert( self, chainedCert ):
      if chainedCert in self.profileConfig_.chainedCert:
         del self.profileConfig_.chainedCert[ chainedCert ]

   def addCrl( self, crl ):
      if not crl in self.profileConfig_.crl:
         self.profileConfig_.crl[ crl ] = 1

   def noCrl( self, crl ):
      if crl in self.profileConfig_.crl:
         del self.profileConfig_.crl[ crl ]

   def enableExtendedParameters( self ):
      self.profileConfig_.verifyExtendedParameters = True

   def disableExtendedParameters( self ):
      self.profileConfig_.verifyExtendedParameters = False

   def enableVerifyBasicConstraintTrust( self ):
      self.profileConfig_.verifyBasicConstraintTrust = True

   def disableVerifyBasicConstraintTrust( self ):
      self.profileConfig_.verifyBasicConstraintTrust = False

   def enableVerifyBasicConstraintChain( self ):
      self.profileConfig_.verifyBasicConstraintChain = True

   def disableVerifyBasicConstraintChain( self ):
      self.profileConfig_.verifyBasicConstraintChain = False

   def enableVerifyExpiryDateEndCert( self ):
      self.profileConfig_.verifyExpiryDateEndCert = True

   def disableVerifyExpiryDateEndCert( self ):
      self.profileConfig_.verifyExpiryDateEndCert = False

   def setVerifyChainHasRootCA( self, setting ):
      self.profileConfig_.verifyChainHasRootCA = setting

   def setCommonNameRegex( self, regex ):
      regex = regex.pop()
      for exp in regex.split():
         try:
            assert re.compile( exp )
         except Exception:  # pylint: disable-msg=W0703
            self.addError( 'Invalid regular expression:' + exp )
            continue
         self.profileConfig_.commonNameRegex.add( exp )

   def noCommonNameRegex( self ):
      self.profileConfig_.commonNameRegex.clear()

   def enableVerifyHosenameMatch( self ):
      self.profileConfig_.verifyHostnameMatch = True

   def disableVerifyHosenameMatch( self ):
      self.profileConfig_.verifyHostnameMatch = False

class SslProfileConfigMode( SslProfileMode.SslProfileMode, 
                            ProfileConfigModeBase ):
   name = "SSL profile configuration"
   modeParseTree = CliParser.ModeParseTree()

   def __init__( self, parent, session, profileName ):
      SslProfileMode.SslProfileMode.__init__( self, profileName )
      ProfileConfigModeBase.__init__( self, parent, session, profileName,
                                      "profileTypeSsl" )

def _gotoSslProfileConfigMode( mode, args ):
   profileName = args[ 'PROFILE_NAME' ]
   childMode = mode.childMode( SslProfileConfigMode, profileName=profileName )
   if childMode.profileConfig_.profileType != "profileTypeSsl":
      mode.addError( "Not an SSL profile" )
      return
   mode.session_.gotoChildMode( childMode )

def _noSslProfile( mode, args ):
   profileName = args[ 'PROFILE_NAME' ]
   if profileName == DefaultSslProfile.ARISTA_PROFILE:
      mode.addError( "Cannot delete default profile" )
      return
   profileConfig = config.profileConfig.get( profileName )
   if profileConfig and profileConfig.profileType != "profileTypeSsl":
      mode.addError( "Not an SSL profile" )
      return
   del config.profileConfig[ profileName ]

def noSecurityConfig( mode ):
   for profileName in config.profileConfig:
      if profileName != DefaultSslProfile.ARISTA_PROFILE:
         del config.profileConfig[ profileName ]
   
class GotoSslProfileModeCmd( CliCommand.CliCommandClass ):
   syntax = 'ssl profile PROFILE_NAME'
   noOrDefaultSyntax = syntax
   data = {
            'ssl': sslMatcher,
            'profile': profileMatcher,
            'PROFILE_NAME': profileNameMatcher
          }

   handler = _gotoSslProfileConfigMode
   noOrDefaultHandler = _noSslProfile

Security.SecurityConfigMode.addCommandClass( GotoSslProfileModeCmd )

#--------------------------------------------------
# [no|default] certificate requirement hostname match
#--------------------------------------------------
class VerifyHostnameActionCmd( CliCommand.CliCommandClass ):
   syntax = 'certificate requirement hostname match'
   noOrDefaultSyntax = syntax
   data = {
            'certificate': certificateKwMatcher,
            'requirement': requirementKwMatcher,
            'hostname': 'Verify hostname for the certificate',
            'match': 'Hostname in certificate must match this device'
          }

   @staticmethod
   def handler( mode, args ):
      mode.enableVerifyHosenameMatch()

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      mode.disableVerifyHosenameMatch()

SslProfileConfigMode.addCommandClass( VerifyHostnameActionCmd )

#--------------------------------------------------
# [no|default] certificate requirement extended-key-usage
#--------------------------------------------------
class VerifyExtendedParametersCmd( CliCommand.CliCommandClass ):
   syntax = 'certificate requirement extended-key-usage'
   noOrDefaultSyntax = syntax
   data = {
            'certificate': certificateKwMatcher,
            'requirement': requirementKwMatcher,
            'extended-key-usage': 'Certificate extended key usage extension'
          }

   @staticmethod
   def handler( mode, args ):
      mode.enableExtendedParameters()

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      mode.disableExtendedParameters()

SslProfileConfigMode.addCommandClass( VerifyExtendedParametersCmd )

#--------------------------------------------------
# [no|default] [trust|chain] certificate requirement basic-constraints ca true
#--------------------------------------------------
class VerifyChainBasicConstraintTrustCmd( CliCommand.CliCommandClass ):
   syntax = '(trust|chain) certificate requirement basic-constraints ca true'
   noOrDefaultSyntax = syntax
   data = {
            'trust': trustKwMatcher,
            'chain': chainKwMatcher,
            'certificate': certificateKwMatcher,
            'requirement': requirementKwMatcher,
            'basic-constraints': 'Certificate basic constraint extension',
            'ca': 'Certificate authority attribute must satisfy the requirement',
            'true': 'Attribute must be set to true',
          }

   @staticmethod
   def handler( mode, args ):
      if 'trust' in args:
         mode.enableVerifyBasicConstraintTrust()
      else:
         mode.enableVerifyBasicConstraintChain()

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      if 'trust' in args:
         mode.disableVerifyBasicConstraintTrust()
      else:
         mode.disableVerifyBasicConstraintChain()

SslProfileConfigMode.addCommandClass( VerifyChainBasicConstraintTrustCmd )

#---------------------------------------------------
# [no|default] certificate policy expiry-date ignore
#---------------------------------------------------
class IgnoreExpiryDateCmd( CliCommand.CliCommandClass ):
   syntax = 'certificate policy expiry-date ignore'
   noOrDefaultSyntax = syntax
   data = {
            'certificate': certificateKwMatcher,
            'policy': 'Policy parameters',
            'expiry-date': 'Certificate expiry date',
            'ignore': 'Ignore a requirement',
          }

   @staticmethod
   def handler( mode, args ):
      mode.disableVerifyExpiryDateEndCert()

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      mode.enableVerifyExpiryDateEndCert()

SslProfileConfigMode.addCommandClass( IgnoreExpiryDateCmd )

#---------------------------------------------------
# [no|default] tls versions
# tls versions [add | remove] { 1.0 | 1.1 | 1.2 }...
#---------------------------------------------------
tlsVersionsMap = { '1.0': Constants.tlsv1,
                   '1.1': Constants.tlsv1_1,
                   '1.2': Constants.tlsv1_2 }

class TlsVersions( CliCommand.CliCommandClass ):
   syntax = 'tls versions [ add | remove ] { 1.0 | 1.1 | 1.2 }'
   noOrDefaultSyntax = 'tls versions ...'
   data = {
            'tls': 'Configure TLS settings',
            'versions': 'Configure TLS versions',
            'add': 'Add versions to the current list',
            'remove': 'Remove versions from the current list',
            '1.0': CliCommand.singleKeyword( '1.0', helpdesc='version 1.0' ),
            '1.1': CliCommand.singleKeyword( '1.1', helpdesc='version 1.1' ),
            '1.2': CliCommand.singleKeyword( '1.2', helpdesc='version 1.2' ),
          }

   @staticmethod
   def handler( mode, args ):
      mask = 0
      for k, v in tlsVersionsMap.iteritems():
         if k not in args:
            continue
         mask += v

      if 'remove' in args:
         mode.disableTlsVersion( mask )
      else:
         add = 'add' in args
         mode.enableTlsVersion( mask, add=add )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      mode.enableTlsVersion( Constants.allTlsVersion )

SslProfileConfigMode.addCommandClass( TlsVersions )

#--------------------------------------------------
# [no|default] fips restrictions
#--------------------------------------------------
class FipRestrictionsCmd( CliCommand.CliCommandClass ):
   syntax = 'fips restrictions'
   noOrDefaultSyntax = syntax
   data = {
            'fips': 'Configure FIPS settings',
            'restrictions': 'Configure FIPS restrictions',
          }

   @staticmethod
   def handler( mode, args ):
      mode.enableFipsMode()

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      mode.disableFipsMode()

SslProfileConfigMode.addCommandClass( FipRestrictionsCmd )

#--------------------------------------------------
# [no|default] cipher-list
#--------------------------------------------------
class CipherListCmd( CliCommand.CliCommandClass ):
   syntax = 'cipher-list CIPHERS'
   noOrDefaultSyntax = 'cipher-list ...'
   data = {
            'cipher-list': 'Configure a cipher list',
            'CIPHERS': CliMatcher.PatternMatcher( r'[\w:@!+-]+',
                                        helpname='WORD',
                                        helpdesc='Cipher list' )
          }

   @staticmethod
   def handler( mode, args ):
      mode.enableCiphers( args[ 'CIPHERS' ] )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      mode.disableCiphers()

SslProfileConfigMode.addCommandClass( CipherListCmd )

#--------------------------------------------------
# [no|default] certificate CERT_NAME key KEY_NAME
#--------------------------------------------------
class CertificateCmd( CliCommand.CliCommandClass ):
   syntax = 'certificate CERT_NAME key KEY_NAME'
   noOrDefaultSyntax = 'certificate ...'
   data = {
            'certificate': certificateKwMatcher,
            'CERT_NAME': certificateNameMatcher,
            'key': 'Configure key matching the certificate',
            'KEY_NAME': keyNameMatcher
          }

   @staticmethod
   def handler( mode, args ):
      mode.setCertKey( args[ 'CERT_NAME' ], args[ 'KEY_NAME' ] )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      mode.noCertKey()

SslProfileConfigMode.addCommandClass( CertificateCmd )

#--------------------------------------------------
# [no|default] trust certificate CERT_NAME
#--------------------------------------------------
class TrustCertificateCmd( CliCommand.CliCommandClass ):
   syntax = 'trust certificate CERT_NAME'
   data = {
            'trust': trustKwMatcher,
            'certificate': 'Certificate',
            'CERT_NAME': certificateNameMatcher
          }

   @staticmethod
   def handler( mode, args ):
      mode.addTrustedCert( args[ 'CERT_NAME' ] )

class NoTrustCertificateCmd( CliCommand.CliCommandClass ):
   noOrDefaultSyntax = 'trust certificate CERT_NAME'
   data = {
            'trust': trustKwMatcher,
            'certificate': 'Certificate',
            'CERT_NAME': CliMatcher.DynamicNameMatcher(
               lambda mode: config.profileConfig[ mode.profileName ].trustedCert,
               'Certificate name', 
               pattern=r'[A-Za-z0-9_:{}\.\[\]-]+' )
          }

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      mode.noTrustedCert( args[ 'CERT_NAME' ] )

SslProfileConfigMode.addCommandClass( TrustCertificateCmd )
SslProfileConfigMode.addCommandClass( NoTrustCertificateCmd )

#--------------------------------------------------
# [no|default] chain certificate CERT_NAME
#--------------------------------------------------
class ChainCertificateCmd( CliCommand.CliCommandClass ):
   syntax = 'chain certificate CERT_NAME'
   data = {
            'chain': chainKwMatcher,
            'certificate': 'Certificate',
            'CERT_NAME': certificateNameMatcher
          }

   @staticmethod
   def handler( mode, args ):
      mode.addChainedCert( args[ 'CERT_NAME' ] )

class NoChainCertificateCmd( CliCommand.CliCommandClass ):
   noOrDefaultSyntax = 'chain certificate CERT_NAME'
   data = {
            'chain': chainKwMatcher,
            'certificate': 'Certificate',
            'CERT_NAME': CliMatcher.DynamicNameMatcher(
               lambda mode: config.profileConfig[ mode.profileName ].chainedCert,
               'Certificate name', 
               pattern=r'[A-Za-z0-9_:{}\.\[\]-]+' )
          }

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      mode.noChainedCert( args[ 'CERT_NAME' ] )

SslProfileConfigMode.addCommandClass( ChainCertificateCmd )
SslProfileConfigMode.addCommandClass( NoChainCertificateCmd )

#--------------------------------------------------
# [no|default] chain certificate requirement include root-ca
#--------------------------------------------------
class ChainIncludeRootCACmd( CliCommand.CliCommandClass ):
   syntax = 'chain certificate requirement include root-ca'
   noOrDefaultSyntax = syntax
   data = {
            'chain': chainKwMatcher,
            'certificate': 'Certificate',
            'requirement': requirementKwMatcher,
            'include': 'Add a requirement to be included when validating',
            'root-ca': 'Root Certificate Authority'
          }

   @staticmethod
   def handler( mode, args ):
      mode.setVerifyChainHasRootCA( True )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      mode.setVerifyChainHasRootCA( False )

SslProfileConfigMode.addCommandClass( ChainIncludeRootCACmd )

#--------------------------------------------------
# [no|default] crl CERT_NAME
#--------------------------------------------------
class CrlCmd( CliCommand.CliCommandClass ):
   syntax = 'crl CERT_NAME'
   data = {
            'crl': 'Configure CRLs',
            'CERT_NAME': certificateNameMatcher,
          }

   @staticmethod
   def handler( mode, args ):
      mode.addCrl( args[ 'CERT_NAME' ] )

class NoCrlCmd( CliCommand.CliCommandClass ):
   noOrDefaultSyntax = 'crl CERT_NAME'
   data = {
            'crl': 'Configure CRLs',
            'CERT_NAME': CliMatcher.DynamicNameMatcher(
               lambda mode: config.profileConfig[ mode.profileName ].crl,
               'CRL name', 
               pattern=r'[A-Za-z0-9_:{}\.\[\]-]+' )
          }

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      mode.noCrl( args[ 'CERT_NAME' ] )

SslProfileConfigMode.addCommandClass( CrlCmd )
SslProfileConfigMode.addCommandClass( NoCrlCmd )

#------------------------------------------------------------
# [no|default] certificate common-name username regexp REG_EX
#------------------------------------------------------------
class CommonNameRegexCmd( CliCommand.CliCommandClass ):
   syntax = 'certificate common-name username regexp { <REGEX> }'
   noOrDefaultSyntax = 'certificate common-name username regexp ...'
   data = {
            'certificate': certificateKwMatcher,
            'common-name': 'Configure certificate Common Name settings',
            'username': 'Username settings',
            'regexp': 'Use regular expression to extract username',
            '<REGEX>': CliMatcher.StringMatcher( helpname='REGEX',
                                                    helpdesc='Regular expression' )
          }

   @staticmethod
   def handler( mode, args ):
      mode.setCommonNameRegex( args[ '<REGEX>' ] )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      mode.noCommonNameRegex()

SslProfileConfigMode.addCommandClass( CommonNameRegexCmd )

#-------------------------------------------------------------------------------
# The "show management security [ssl] certificate [<name>]" command
#-------------------------------------------------------------------------------
def _showSslCert( mode, args ):
   certName = args.get( 'CERT_NAME' )
   ret = SslModel.Certificates()
   if certName:
      certs = [ certName ]
   else:
      certs = _listdir( Constants.certsDirPath() )
   
   for cert in certs:
      certFile = Constants.certPath( cert )
      if ( os.path.isfile( certFile ) and
           SslCertKey.hasCertificate( open( certFile ).read() ) ):
         ret.certificates[ cert ] = SslCliLib.getCertificateModel( certFile )
   return ret

class ShowSecuritySslCert( ShowCommand.ShowCliCommandClass ):
   syntax = 'show management security ssl certificate [ CERT_NAME ]'
   data = {
            'management': ConfigMgmtMode.managementShowKwMatcher,
            'security': Security.securityShowMatcher,
            'ssl': sslShowMatcher,
            'certificate': 'Show certificate',
            'CERT_NAME': certificateNameMatcher
          }

   privileged = True
   cliModel = SslModel.Certificates
   handler = _showSslCert

BasicCli.addShowCommandClass( ShowSecuritySslCert )

#-------------------------------------------------------------------------------
# The "show management security ssl key [<name>]" command
#-------------------------------------------------------------------------------
def _showSslKey( mode, args ):
   keyName = args.get( 'KEY_NAME' )
   ret = SslModel.PublicKeys()
   if keyName:
      keys = [ keyName ]
   else:
      keys = _listdir( Constants.keysDirPath() )
   
   for key in keys:
      keyFile = Constants.keyPath( key )
      if os.path.isfile( keyFile ):
         ret.publicKeys[ key ] = SslCliLib.getPublicKeyModel( keyFile )
   return ret

class ShowSecuritySslKey( ShowCommand.ShowCliCommandClass ):
   syntax = 'show management security ssl key [ KEY_NAME ]'
   data = {
            'management': ConfigMgmtMode.managementShowKwMatcher,
            'security': Security.securityShowMatcher,
            'ssl': sslShowMatcher,
            'key': 'Show key',
            'KEY_NAME': keyNameMatcher

          }
   privileged = True
   cliModel = SslModel.PublicKeys
   handler = _showSslKey

BasicCli.addShowCommandClass( ShowSecuritySslKey )

#-------------------------------------------------------------------------------
# The "show management security ssl crl [<name>]" command
#-------------------------------------------------------------------------------
def _showSslCrl( mode, args ):
   crlName = args.get( 'CRL_NAME' )
   ret = SslModel.Crls()
   if crlName:
      crls = [ crlName ]
   else:
      crls = _listdir( Constants.certsDirPath() )

   for crl in crls:
      crlFile = Constants.certPath( crl )
      if ( os.path.isfile( crlFile ) and
           SslCertKey.hasCrl( open( crlFile ).read() ) ):
         ret.crls[ crl ] = SslCliLib.getCrlModel( crlFile )
   return ret

class ShowSecuritySslCrl( ShowCommand.ShowCliCommandClass ):
   syntax = 'show management security ssl crl [ CRL_NAME ]'
   data = {
            'management': ConfigMgmtMode.managementShowKwMatcher,
            'security': Security.securityShowMatcher,
            'ssl': sslShowMatcher,
            'crl': 'Show crl',
            'CRL_NAME': certificateNameMatcher

          }
   privileged = True
   cliModel = SslModel.Crls
   handler = _showSslCrl

BasicCli.addShowCommandClass( ShowSecuritySslCrl )

#-------------------------------------------------------------------------------
# The "show management security ssl diffie-hellman" command
#-------------------------------------------------------------------------------
def _dhparamsResetAttempted():
   return status.dhparamsResetProcessed or status.dhparamsLastResetFailed 

def _showSslDhparams( mode, args ):
   ret = SslModel.DiffieHellman()
   if status.dhparamsResetInProgress:
      ret.dhparamsResetInProgress = True
   else:
      if _dhparamsResetAttempted():
         ret.dhparamsLastResetFailed = status.dhparamsLastResetFailed
         if status.dhparamsResetProcessed:
            ret.dhparamsLastSuccessfulReset = int( status.dhparamsResetProcessed + 
                                                   Tac.utcNow() - 
                                                   Tac.now() )
      ret.dhparamsResetInProgress = False
      dhparamsFile = Constants.dhParamPath()
      if os.path.isfile( dhparamsFile ):
         ret.diffieHellmanParameters = SslCliLib.getDhparamsModel( dhparamsFile )
   return ret

class ShowSecuritySslDiffieHellman( ShowCommand.ShowCliCommandClass ):
   syntax = 'show management security ssl diffie-hellman'
   data = {
            'management': ConfigMgmtMode.managementShowKwMatcher,
            'security': Security.securityShowMatcher,
            'ssl': sslShowMatcher,
            'diffie-hellman': 'Show diffie-hellman parameters in use',

          }
   privileged = True
   cliModel = SslModel.DiffieHellman
   handler = _showSslDhparams

BasicCli.addShowCommandClass( ShowSecuritySslDiffieHellman )

#-------------------------------------------------------------------------------
# The "show management security ssl profile [ PROFILE_NAME ]" command
#-------------------------------------------------------------------------------
def _showSslProfileStatus( mode, args ):
   profileName = args.get( 'PROFILE_NAME' )
   ret = SslModel.SslStatus()
   if profileName:
      profiles = [ profileName ]
   else:
      profiles = status.profileStatus
   
   for name in profiles:
      profile = status.profileStatus.get( name )
      if profile is not None:
         profileStatusModel = SslModel.ProfileStatus()
         profileStatusModel.profileState = profile.state
         for err in profile.error.values():
            profErr = SslModel.ProfileError( 
                          errorAttr=err.errorAttr,
                          errorAttrValue=err.errorAttrValue, 
                          errorType=err.errorType )
            profileStatusModel.profileError.append( profErr )
            ret._hasError = True
         for warning in profile.warning.values():
            profWarning = SslModel.ProfileError(
                              errorAttr=warning.errorAttr,
                              errorAttrValue=warning.errorAttrValue,
                              errorType=warning.errorType )
            if profWarning not in profileStatusModel.profileError:
               profileStatusModel.profileError.append( profWarning )
               ret._hasError = True
         ret.profileStatus[ name ] =  profileStatusModel
   return ret
   
class ShowSecuritySslProfile( ShowCommand.ShowCliCommandClass ):
   syntax = 'show management security ssl profile [ PROFILE_NAME ]'
   data = {
            'management': ConfigMgmtMode.managementShowKwMatcher,
            'security': Security.securityShowMatcher,
            'ssl': sslShowMatcher,
            'profile': 'Show SSL profile status',
            'PROFILE_NAME': statusProfileNameMatcher

          }
   privileged = True
   cliModel = SslModel.SslStatus
   handler = _showSslProfileStatus

BasicCli.addShowCommandClass( ShowSecuritySslProfile )

#--------------------------------------------------------------------------------
# reset ssl diffie-hellman parameters
#--------------------------------------------------------------------------------
class ResetSslDiffieHellmanParametersCmd( CliCommand.CliCommandClass ):
   syntax = 'reset ssl diffie-hellman parameters'
   data = {
      'reset': CliToken.Reset.resetMatcher,
      'ssl': sslMatcher,
      'diffie-hellman': 'Diffie-hellman parameters',
      'parameters': 'Diffie-hellman parameters',
   }

   @staticmethod
   def handler( mode, args ):
      execRequest.dhparamsResetRequest = Tac.now()

BasicCli.EnableMode.addCommandClass( ResetSslDiffieHellmanParametersCmd )

def Plugin( entityManager ):
   global config, status, execRequest
   config = ConfigMount.mount( entityManager, "mgmt/security/ssl/config",
                               "Mgmt::Security::Ssl::Config", "w" )
   status = LazyMount.mount( entityManager, "mgmt/security/ssl/status",
                             "Mgmt::Security::Ssl::Status", "r" )
   execRequest = LazyMount.mount( entityManager, "mgmt/security/ssl/execRequest",
                                  "Mgmt::Security::Ssl::ExecRequest", "w" )
