# Copyright (c) 2010, 2011 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import itertools
import os
import sys
import time

import BasicCli, HostnameCli
import CliCommand
import CliMatcher
import CliToken.Reset
import CliToken
import ConfigMgmtMode
import CliParser
import ShowCommand
import LazyMount
import CommonGuards
import ConfigMount
import Tac
from eunuchs.in_h import IPPROTO_TCP
import AclCli
import AclCliLib
import AclCliModel
import DscpCliLib
from CliPlugin.VrfCli import VrfExprFactory, DEFAULT_VRF
from CliMode.VrfConfig import VrfConfigMode
from CliMode.SshTunnel import SshTunnelMode
from CliMode.SshTunnel import SshTunnelVrfMode
from CliMode.SshUser import SshUserMode
import Logging
import SysMgrLib
import SysMgrModels
import SshCertLib
import Cell
import FileCli

# ------------------------------
# Ssh config commands
#
# From config mode
#    management ssh    - enter ssh config mode
#    idle-timeout x - set the idle session timeout to x minutes
#    authentication mode [ password | keyboard-interactive ]
#          - make sshd enter specified mode
#    ciphers SSH_CIPHER1 ... SSH_CIPHERN - set SSH exclusive cipher list
#    [ no | default ] ciphers - clear filter and use all supported ciphers
#    [ no | default ] authentication - return sshd to keyboard-interactive mode
#    [ no | default ] shutdown - disables or enables sshd
#    [ no | default ] qos dscp <0-63> - set dscp value in IP header
#    vrf VRF - enter ssh vrf config mode
#
# From ssh vrf config mode
#    [ no | default ] shutdown - disables or enables sshd in vrf
#
#    For example:
#    management ssh
#       A
#       vrf x
#          B
#
#    A\B             B default             B shutdown            B no shutdown
#    A default       enabled\enabled       enabled\disabled      enabled\enabled
#    A shutdown      disabled\disabled     disabled\disabled     disabled\enabled
#    A no shutdown   enabled\enabled       enabled\disabled      enabled\enabled

CIPHERS = {
   "3des-cbc": "Triple DES (112 bit)",
   "aes128-cbc": "Advanced Encryption Standard (128 bit, CBC mode)",
   "aes192-cbc": "Advanced Encryption Standard (192 bit, CBC mode)",
   "aes256-cbc": "Advanced Encryption Standard (256 bit, CBC mode)",
   "aes128-ctr": "Advanced Encryption Standard (128 bit, counter mode)",
   "aes192-ctr": "Advanced Encryption Standard (192 bit, counter mode)",
   "aes256-ctr": "Advanced Encryption Standard (256 bit, counter mode)",
   "arcfour": "Arcfour stream cipher (RC4 like)",
   "arcfour128": "Arcfour stream cipher (RFC 4345, 128 bit)",
   "arcfour256": "Arcfour stream cipher (RFC 4345, 256 bit)",
   "blowfish-cbc": "Blowfish block cipher (128 bit, CBC mode)",
   "cast128-cbc": "CAST-128 (RFC 2144, 128 bit, CBC mode)",
   "aes128-gcm@openssh.com" :
      "Advanced Encryption Standard (128 bit, Galois/Counter mode )",
   "aes256-gcm@openssh.com" :
      "Advanced Encryption Standard (256 bit, Galois/Counter mode )",
   "chacha20-poly1305@openssh.com" :
      "ChaCha20 stream cipher and Poly1305 authenticator",
}

SUPPORTED_CIPHERS = [ 
   "3des-cbc", "aes128-cbc", "aes192-cbc", "aes256-cbc",
   "rijndael-cbc@lysator.liu.se", "aes128-ctr", "aes192-ctr",
   "aes256-ctr", "aes128-gcm@openssh.com", "aes256-gcm@openssh.com",
   "chacha20-poly1305@openssh.com" ]

FIPS_CIPHERS = {
   "aes128-ctr": "Advanced Encryption Standard (128 bit, counter mode)",
   "aes192-ctr": "Advanced Encryption Standard (192 bit, counter mode)",
   "aes256-ctr": "Advanced Encryption Standard (256 bit, counter mode)",
   "aes128-cbc": "Advanced Encryption Standard (128 bit, CBC mode)",
   "aes192-cbc": "Advanced Encryption Standard (192 bit, CBC mode)",
   "aes256-cbc": "Advanced Encryption Standard (256 bit, CBC mode)",
   "3des-cbc": "Triple DES (112 bit)",
   "aes128-gcm@openssh.com" :
      "Advanced Encryption Standard (128 bit, Galois/Counter mode )",
   "aes256-gcm@openssh.com" :
      "Advanced Encryption Standard (256 bit, Galois/Counter mode )",
}
KEXALGORITHMS = {
   "diffie-hellman-group-exchange-sha256": "Negotiated Group Exchange with SHA-256",
   "diffie-hellman-group-exchange-sha1": "Negotiated Group Exchange with SHA-1",
   "diffie-hellman-group14-sha1": "Oakley Group 14 with SHA-1",
   "diffie-hellman-group1-sha1" : "Oakley Group 1 with SHA-1",
   "curve25519-sha256@libssh.org" : "Elliptic Curve 255519 with SHA256",
   "ecdh-sha2-nistp521" : "Elliptic Curve NIST P521",
   "ecdh-sha2-nistp256" : "Elliptic Curve NIST P256",
   "ecdh-sha2-nistp384" : "Elliptic Curve NIST P384",
}
SUPPORTED_KEX_ALGORITHMS = [ 
   "diffie-hellman-group1-sha1", "diffie-hellman-group14-sha1",
   "diffie-hellman-group14-sha256", "diffie-hellman-group16-sha512",
   "diffie-hellman-group18-sha512",
   "diffie-hellman-group-exchange-sha1",
   "diffie-hellman-group-exchange-sha256",
   "ecdh-sha2-nistp256", "ecdh-sha2-nistp384", "ecdh-sha2-nistp521",
   "curve25519-sha256", "curve25519-sha256@libssh.org",
   "gss-gex-sha1-", "gss-group1-sha1-", "gss-group14-sha1-",
   "gss-group14-sha256-", "gss-group16-sha512-",
   "gss-nistp256-sha256-", "gss-curve25519-sha256-" ]
MACS = {
   "hmac-md5": "Hash Message Authentication Code MD5",
   "hmac-sha1": "Hash Message Authentication Code SHA-1",
   "hmac-ripemd160": "Hash Message Authentication Code RIPEMD-160",
   "hmac-sha1-96": "Hash Message Authentication Code SHA-1 for use in ESP and AH",
   "hmac-md5-96": "Hash Message Authentication Code MD5 for use in ESP and AH",
   "hmac-sha2-256": "Hash Message Authentication Code SHA2-256",
   "hmac-sha2-512": "Hash Message Authentication Code SHA2-512",
   "umac-128@openssh.com": "Hash Message Authentication Code UMAC-128",
   "hmac-sha1-etm@openssh.com" : "Encrypt-Then-Mac HMAC SHA1",
   "hmac-sha2-256-etm@openssh.com" : "Encrypt-Then-Mac HMAC SHA2-256",
   "hmac-sha2-512-etm@openssh.com" : "Encrypt-Then-Mac HMAC SHA2-512",
   "umac-128-etm@openssh.com" : "Encrypt-Then-Mac HMAC UMAC-128",
}
SUPPORTED_MACS = [ 
   "hmac-sha1", "hmac-sha1-96", "hmac-sha2-256", "hmac-sha2-512",
   "hmac-md5", "hmac-md5-96", "umac-64@openssh.com", "umac-128@openssh.com",
   "hmac-sha1-etm@openssh.com", "hmac-sha1-96-etm@openssh.com",
   "hmac-sha2-256-etm@openssh.com", "hmac-sha2-512-etm@openssh.com",
   "hmac-md5-etm@openssh.com", "hmac-md5-96-etm@openssh.com",
   "umac-64-etm@openssh.com", "umac-128-etm@openssh.com" ]
FIPS_MACS = {
   "hmac-sha1": "Hash Message Authentication Code SHA-1",
   "hmac-sha2-256": "Hash Message Authentication Code SHA2-256",
   "hmac-sha2-512": "Hash Message Authentication Code SHA2-512",
   "hmac-sha2-256-etm@openssh.com" : "Encrypt-Then-Mac HMAC SHA2-256",
   "hmac-sha2-512-etm@openssh.com" : "Encrypt-Then-Mac HMAC SHA2-512",
}
HOSTKEYALGORITHMS = {
      "dsa" : "Digital Signature Algorithm",
      "rsa" : "RSA Encryption Algorithm",
      "ed25519" : "Edwards-curve Digital Signature Algorithm 25519",
      "ecdsa-nistp256" : "Elliptic Curve Digital Signature Algorithm NIST-P256",
      "ecdsa-nistp521" : "Elliptic Curve Digital Signature Algorithm NIST-P521",
}
FIPS_HOSTKEYALGORITHMS = {
   "rsa" : "RSA Encryption Algorithm",
   "ecdsa-nistp256" : "Elliptic Curve Digital Signature Algorithm NIST-P256",
}
REKEYUNITS = {
      "kbytes" : "Kilobyte(s)",
      "mbytes" : "Megabyte(s)",
      "gbytes" : "Gigabyte(s)",
}

REKEYTIMEUNITS = {
   "seconds": "Second(s)",
   "minutes": "Minute(s)",
   "hours": "Hour(s)",
   "days": "Day(s)",
   "weeks": "Week(s)",
}

TCPFORWARDINGSETTINGS = {
   "all" : "Allow TCP forwarding for both local and remote",
   "local" : "Allow local forwarding only",
   "remote" : "Allow remote forwarding only",
}

authenticationMode = Tac.Type( "Mgmt::Ssh::AuthenticationMode" )
serverPort = Tac.Type( "Mgmt::Ssh::ServerPort" )
sshTunnel = Tac.newInstance( "Mgmt::Ssh::TunnelConfig", "arastra" )
SshConstants = Tac.Type( "Mgmt::Ssh::Constants" )
KeyType = Tac.Type( "Mgmt::Ssh::KeyType" )

Logging.logD( id="SECURITY_SSH_TUNNEL_CONFIGURED",
              severity=Logging.logInfo,
              format="SSH tunnel %s from local TCP port %s to "
              "%s:%s via %s@%s is fully configured.",
              explanation="An SSH tunnel was just configured and "
              "will try to connect.",
              recommendedAction=Logging.NO_ACTION_REQUIRED )

Logging.logD( id="SECURITY_SSH_TUNNEL_UNCONFIGURED",
              severity=Logging.logInfo,
              format="SSH tunnel %s from local TCP port %s to %s:%s via %s@%s"
              " has been unconfigured.",
              explanation="An SSH tunnel to a remote server just stopped running"
              " and will not be automatically restarted since it was terminated"
              " by a command.",
              recommendedAction=Logging.NO_ACTION_REQUIRED )

sshConfig = None
sshConfigReq = None
sshStatus = None
aclConfig = None
aclStatus = None
aclCheckpoint = None
aclCpConfig = None
entityMibStatus = None
dscpConfig = None

sshShowKwMatcher = CliMatcher.KeywordMatcher( 'ssh',
      helpdesc='Show SSH status and statistics' )
# Originally 'ecdsa' didn't specify the curve. Making legacy command
# to specify curve.
legacyEcdsaToken = 'ecdsa'
hostKeyMatcher = CliCommand.setCliExpression( HOSTKEYALGORITHMS,
      { 'ecdsa': 'Elliptic Curve Digital Signature Algorithm NIST-P521' },
      name='HOSTKEYALGOS' )
legacyEcdsaMatcher = CliMatcher.KeywordMatcher( 'ecdsa',
                    helpdesc='Elliptic Curve Digital Signature Algorithm NIST-P521' )
hostkeyKwMatcher = CliMatcher.KeywordMatcher( 'hostkey',
                                     helpdesc='Set SSH hostkey related options' )
showHostKeyKwMatcher = CliMatcher.KeywordMatcher( 'hostkey',
                                     helpdesc='Show sshd hostkey information' )
trustedCaMatcher = CliMatcher.DynamicNameMatcher(
                            lambda mode: _listdir( SshConstants.caKeysDirPath() ),
                            pattern=r'[A-Za-z0-9_:{}\.\[\]-]+',
                            helpdesc='CA public key filename',
                            helpname='WORD' )
hostCertMatcher = CliMatcher.DynamicNameMatcher(
                            lambda mode: _listdir( SshConstants.hostCertsDirPath() ),
                            pattern=r'[A-Za-z0-9_:{}\.\[\]-]+',
                            helpdesc='Switch\'s SSH hostkey cert filename',
                            helpname='WORD' )
revokeListMatcher = CliMatcher.DynamicNameMatcher(
                            lambda mode: _listdir(
                                             SshConstants.revokeListsDirPath() ),
                            pattern=r'[A-Za-z0-9_:{}\.\[\]-]+',
                            helpdesc='Revoked SSH user keys filename',
                            helpname='WORD' )
thousandIntMatcher = CliMatcher.IntegerMatcher( 1, 1000, helpdesc='Set Value' )
connectionKwMatcher = CliMatcher.KeywordMatcher( 'connection',
                                          helpdesc='Settings for SSH connections' )
authKwMatcher = CliMatcher.KeywordMatcher( 'authentication',
                                          helpdesc='Change authentication settings' )

def _listdir( path ):
   try:
      return os.listdir( path )
   except OSError:
      return []

class HostKeyAlgoExpression( CliCommand.CliExpression ):
   expression = 'KEYALGO | ecdsa'
   data = {  'KEYALGO': CliMatcher.EnumMatcher( HOSTKEYALGORITHMS ),
            'ecdsa': CliCommand.Node( matcher=legacyEcdsaMatcher, hidden=True,
                                      alias='KEYALGO' ) }

class SshTunnelConfigModelet( CliParser.Modelet ):
   modeletParseTree = CliParser.ModeletParseTree()

   def __init__( self, mode ):
      self.mode_ = mode
      self.tunnelName_ = mode.tunnelName_
      self.vrfConfigName_ = mode.vrfConfigName_
      # Ensure that the tunnel sysdb path exists
      self.config()
      CliParser.Modelet.__init__( self )

   def config( self ):
      holdingConfig = sshConfig
      if self.vrfConfigName_ and self.vrfConfigName_ != DEFAULT_VRF:
         holdingConfig = sshConfig.vrfConfig.newMember( self.vrfConfigName_ )
      tunnel = holdingConfig.tunnel.newMember( self.tunnelName_ )
      if not self.vrfConfigName_:
         tunnel.configuredInSshMode = True
      return tunnel

   def setLocalPort( self, args ):
      self.config().localPort = args[ 'PORT' ]

   def defaultLocalPort( self, args ):
      self.config().localPort = serverPort.invalid

   def setRemote( self, args ):
      self.config().remoteHost = args[ 'REMOTE_HOST' ]
      self.config().remotePort = args[ 'REMOTE_PORT' ]

   def defaultRemote( self, args ):
      self.config().remoteHost = ""
      self.config().remotePort = serverPort.invalid

   def setSshServer( self, args ):
      # Do a sanity check on the hostname or IP address that the user
      # entered. If it doesn't appear to be legal, print a warning,
      # but don't reject the entry. (The specified hostname may not
      # yet have been configured in DNS, for example.)
      sshServerAddress = args[ 'ADDR' ]
      sshServerUsername = args[ 'USER' ]
      sshServerPort = args.get( 'PORT' )
      HostnameCli.resolveHostname( self.mode_, sshServerAddress,
                                   doWarn=True )
      self.config().sshServerAddress = sshServerAddress
      self.config().sshServerUsername = sshServerUsername
      if sshServerPort:
         self.config().sshServerPort = sshServerPort
      else:
         self.config().sshServerPort = serverPort.defaultPort

   def defaultSshServer( self, args ):
      self.config().sshServerAddress = ""
      self.config().sshServerUsername = ""
      self.config().sshServerPort = serverPort.invalid

   def setServerAliveInterval( self, args ):
      self.config().serverAliveInterval = args[ 'SECONDS' ]

   def defaultServerAliveInterval( self, args ):
      self.config().serverAliveInterval = sshTunnel.serverAliveIntervalDefault

   def setServerAliveMaxLost( self, args ):
      self.config().serverAliveMaxLost = args[ 'SECONDS' ]

   def defaultServerAliveMaxLost( self, args ):
      self.config().serverAliveMaxLost = sshTunnel.serverAliveMaxLostDefault

   def setShutdown( self, args=None ):
      # pylint: disable-msg=E0602
      Logging.log( SECURITY_SSH_TUNNEL_UNCONFIGURED,
                   self.config().name,
                   str( self.config().localPort ),
                   self.config().remoteHost,
                   str( self.config().remotePort ),
                   self.config().sshServerUsername,
                   self.config().sshServerAddress )
      self.config().enable = False

   def disableShutdown( self, args=None ):
      # pylint: disable-msg=E0602
      Logging.log( SECURITY_SSH_TUNNEL_CONFIGURED,
                   self.config().name,
                   str( self.config().localPort ),
                   self.config().remoteHost,
                   str( self.config().remotePort ),
                   self.config().sshServerUsername,
                   self.config().sshServerAddress )
      self.config().enable = True

   def setUnlimitedRestarts( self, args ):
      self.config().unlimitedRestarts = True

   def defaultUnlimitedRestarts( self, args ):
      self.config().unlimitedRestarts = False

#-----------------------------------------------------------------------------------
# [ no | default ] local port PORT
#-----------------------------------------------------------------------------------
class LocalPort( CliCommand.CliCommandClass ):
   syntax = 'local port PORT'
   noOrDefaultSyntax = 'local port ...'
   data = {
            'local': 'Configure the local port for the tunnel',
            'port': 'Set the port to use',
            'PORT': CliMatcher.IntegerMatcher( serverPort.min, serverPort.max,
                                               helpdesc='Number of the port to '
                                                        'use' ),
          }
   handler = SshTunnelConfigModelet.setLocalPort
   noOrDefaultHandler = SshTunnelConfigModelet.defaultLocalPort

SshTunnelConfigModelet.addCommandClass( LocalPort )

#-----------------------------------------------------------------------------------
# [ no | default ] server-alive interval SECONDS
#-----------------------------------------------------------------------------------
class ServerAliveInterval( CliCommand.CliCommandClass ):
   syntax = 'server-alive interval SECONDS'
   noOrDefaultSyntax = 'server-alive interval ...'
   data = {
            'server-alive': 'Set SSH ServerAlive options',
            'interval': 'Time period ( in seconds ) to send SSH keep-alive packets',
            'SECONDS': thousandIntMatcher
          }
   handler = SshTunnelConfigModelet.setServerAliveInterval
   noOrDefaultHandler = SshTunnelConfigModelet.defaultServerAliveInterval

SshTunnelConfigModelet.addCommandClass( ServerAliveInterval )

#-----------------------------------------------------------------------------------
# [ no | default ] server-alive count-max SECONDS
#-----------------------------------------------------------------------------------
class ServerAliveCountMax( CliCommand.CliCommandClass ):
   syntax = 'server-alive count-max SECONDS'
   noOrDefaultSyntax = 'server-alive count-max ...'
   data = {
            'server-alive': 'Set SSH ServerAlive options',
            'count-max': ( 'Number of SSH keep-alive packets that can be lost '
                           'before the connection is assumed dead' ),
            'SECONDS': thousandIntMatcher
          }
   handler = SshTunnelConfigModelet.setServerAliveMaxLost
   noOrDefaultHandler = SshTunnelConfigModelet.defaultServerAliveMaxLost

SshTunnelConfigModelet.addCommandClass( ServerAliveCountMax )

#-----------------------------------------------------------------------------------
# [ no | default ] remote host REMOTE_HOST port REMOTE_PORT
#-----------------------------------------------------------------------------------
class TunnelRemoteHost( CliCommand.CliCommandClass ):
   syntax = 'remote host REMOTE_HOST port REMOTE_PORT'
   noOrDefaultSyntax = 'remote ...'
   data = {
            'remote': 'Configure the remote options for the tunnel',
            'host': 'Set the host to direct the tunnel to',
            'REMOTE_HOST': HostnameCli.IpAddrOrHostnameMatcher( ipv6=True ),
            'port': 'Set the port to use',
            'REMOTE_PORT': CliMatcher.IntegerMatcher( serverPort.min,
                                                      serverPort.max,
                                                      helpdesc='Number of the port '
                                                               'to use' ),
          }
   handler = SshTunnelConfigModelet.setRemote
   noOrDefaultHandler = SshTunnelConfigModelet.defaultRemote

SshTunnelConfigModelet.addCommandClass( TunnelRemoteHost )

#-----------------------------------------------------------------------------------
# [ no | default ] ssh-server ADDR user USER
#                  [ port PORT ]
#-----------------------------------------------------------------------------------
class SshServer( CliCommand.CliCommandClass ):
   syntax = 'ssh-server ADDR user USER [ port PORT ]'
   noOrDefaultSyntax = 'ssh-server ...'
   data = {
            'ssh-server': 'Configure the remote ssh-server to connect to',
            'ADDR': HostnameCli.IpAddrOrHostnameMatcher( ipv6=True ),
            'user': 'Username to login with',
            'USER': CliMatcher.PatternMatcher( '.+', helpname='WORD',
                                               helpdesc='Login name for ssh '
                                                        'server' ),
            'port': 'Port for SSH server (default 22)',
            'PORT': CliMatcher.IntegerMatcher( serverPort.min,
                                               serverPort.max,
                                               helpdesc='Number of the port to '
                                                        'use' ),
          }
   handler = SshTunnelConfigModelet.setSshServer
   noOrDefaultHandler = SshTunnelConfigModelet.defaultSshServer

SshTunnelConfigModelet.addCommandClass( SshServer )

#-----------------------------------------------------------------------------------
# [ no | default ] shutdown
#-----------------------------------------------------------------------------------
class TunnelShutdown( CliCommand.CliCommandClass ):
   syntax = 'shutdown'
   defaultSyntax = 'shutdown'
   noSyntax = 'shutdown'
   data = {
            'shutdown': 'Disable SSH Tunnel',
          }
   handler = SshTunnelConfigModelet.setShutdown
   defaultHandler = SshTunnelConfigModelet.setShutdown
   noHandler = SshTunnelConfigModelet.disableShutdown

SshTunnelConfigModelet.addCommandClass( TunnelShutdown )

#-----------------------------------------------------------------------------------
# [ no | default ] unlimited-restarts
#-----------------------------------------------------------------------------------
class TunnelUnlimitedRestarts( CliCommand.CliCommandClass ):
   syntax = 'unlimited-restarts'
   noOrDefaultSyntax = 'unlimited-restarts'
   data = {
            'unlimited-restarts': ( 'Ignore restart tokens and allow unlimited '
                                    'restarts' ),
          }
   handler = SshTunnelConfigModelet.setUnlimitedRestarts
   noOrDefaultHandler = SshTunnelConfigModelet.defaultUnlimitedRestarts
   hidden = True

SshTunnelConfigModelet.addCommandClass( TunnelUnlimitedRestarts )

class SshMainTunnelConfigMode( SshTunnelMode, BasicCli.ConfigModeBase ):

   name = "SSH Main Tunnel Configuration"
   modeParseTree = CliParser.ModeParseTree()

   def __init__( self, parent, session, tunnelName ):
      self.tunnelName_ = tunnelName
      self.vrfConfigName_ = None
      SshTunnelMode.__init__( self, tunnelName )
      BasicCli.ConfigModeBase.__init__( self, parent, session )

SshMainTunnelConfigMode.addModelet( SshTunnelConfigModelet )

class SshVrfTunnelConfigMode( SshTunnelVrfMode, BasicCli.ConfigModeBase ):

   name = "SSH VRF Tunnel Configuration"
   modeParseTree = CliParser.ModeParseTree()

   def __init__( self, parent, session, tunnelName, vrfConfig ):
      vrfConfigName = vrfConfig.name
      self.tunnelName_ = tunnelName
      self.vrfConfigName_ = vrfConfigName
      SshTunnelVrfMode.__init__( self, ( vrfConfigName, tunnelName ) )
      BasicCli.ConfigModeBase.__init__( self, parent, session )

SshVrfTunnelConfigMode.addModelet( SshTunnelConfigModelet )

class SshUserConfigModelet( CliParser.Modelet ):
   modeletParseTree = CliParser.ModeletParseTree()

   def __init__( self, mode ):
      self.mode_ = mode
      self.userName_ = mode.userName_
      # Ensure that the tunnel sysdb path exists
      self.config()
      CliParser.Modelet.__init__( self )

   def config( self ):
      return sshConfig.user.newMember( self.userName_ )
   
   def setSshUserTcpForwarding( self, args ):
      self.config().userTcpForwarding = args[ 'FORWARDING_SETTING' ]

   def noSshUserTcpForwarding( self, args ):
      self.config().userTcpForwarding = self.config().userTcpForwardingDefault

#-----------------------------------------------------------------------------------
# [ no | default ] tcp forwarding SETTING in User submode
#-----------------------------------------------------------------------------------
class UserTcpForwardingSetting( CliCommand.CliCommandClass ):
   syntax = 'tcp forwarding FORWARDING_SETTING'
   noOrDefaultSyntax = 'tcp forwarding'

   data = {
            'tcp': 'Configure SSH server TCP settings for user',
            'forwarding': 'TCP Forwarding option',
            'FORWARDING_SETTING': CliMatcher.EnumMatcher( TCPFORWARDINGSETTINGS )
          }
   handler = SshUserConfigModelet.setSshUserTcpForwarding
   noOrDefaultHandler = SshUserConfigModelet.noSshUserTcpForwarding

SshUserConfigModelet.addCommandClass( UserTcpForwardingSetting )

class SshUserConfigMode( SshUserMode, BasicCli.ConfigModeBase ):

   name = "SSH User Specification Configuration"
   modeParseTree = CliParser.ModeParseTree()

   def __init__( self, parent, session, userName ):
      self.userName_ = userName
      SshUserMode.__init__( self, userName )
      BasicCli.ConfigModeBase.__init__( self, parent, session )

SshUserConfigMode.addModelet( SshUserConfigModelet )

def generalSetKnownHost( config, host, keyAlgo, publicKey, cliMode ):
   mappedKeyAlgo = SysMgrLib.cliKeyTypeToTac[ keyAlgo ]
   configuredInSshMode = cliMode == SshConfigMode
   config.addKnownHost( Tac.Value( "Mgmt::Ssh::KnownHost",
                                    host,
                                    mappedKeyAlgo,
                                    publicKey,
                                    configuredInSshMode ) )

def generalNoKnownHost( config, host ):
   del config.knownHost[ host ]

#-----------------------------------------------------------------------------------
# show management ssh known-hosts [ vrf VRF ] [ HOST ]
#-----------------------------------------------------------------------------------
def knownHostString( knownHostEntry ):
   """
   Returns a string that represents a known-hosts entry for user consumption
   """
   # Translate key type to CLI format
   keyType = SysMgrLib.tacKeyTypeToCliKey[ knownHostEntry.type ]
   str1 = knownHostEntry.host
   str1 += " %s " % ( keyType )
   str1 += knownHostEntry.publicKey
   return str1

def showKnownHosts( mode, args ):
   vrfName = args.get( 'VRF', '' )
   hostFilter = args.get( 'HOST' )
   holdingConfig = sshConfig
   if vrfName and vrfName in sshConfig.vrfConfig:
      holdingConfig = sshConfig.vrfConfig[ vrfName ]
   for host in holdingConfig.knownHost:
      knownHostEntry = holdingConfig.knownHost[ host ]
      if hostFilter and host != hostFilter:
         continue
      print knownHostString( knownHostEntry )

vrfExprFactory = VrfExprFactory(
      helpdesc='Use a specific VRF' )

class ShowManagementSshKnownHosts( ShowCommand.ShowCliCommandClass ):
   syntax = ( 'show management ssh known-hosts [ VRF ] [ HOST ]' )
   data = {
            'management': ConfigMgmtMode.managementShowKwMatcher,
            'ssh': sshShowKwMatcher,
            'known-hosts': 'SSH Known Hosts public keys',
            'VRF': vrfExprFactory,
            'HOST': HostnameCli.IpAddrOrHostnameMatcher( ipv6=True ),
         }
   privileged = True
   handler = showKnownHosts

BasicCli.addShowCommandClass( ShowManagementSshKnownHosts )

#-------------------------------------------------------------------------------
# show management ssh hostkey KEYALGO public
#-------------------------------------------------------------------------------
def showPublicHostKey( mode, args ):
   keyAlgo = args[ 'KEYALGO' ]
   pubKeyPath = ""
   if keyAlgo == 'rsa':
      pubKeyPath = SysMgrLib.rsaKeyPath
   elif keyAlgo == 'dsa':
      pubKeyPath = SysMgrLib.dsaKeyPath
   elif keyAlgo == 'ed25519':
      pubKeyPath = SysMgrLib.ed25519KeyPath
   elif keyAlgo == 'ecdsa-nistp521':
      pubKeyPath = SysMgrLib.ecdsa521KeyPath
   elif keyAlgo == 'ecdsa-nistp256':
      pubKeyPath = SysMgrLib.ecdsa256KeyPath
   else:
      assert False, "Illegal key algorithm passed in"
   keyAlgoTac = SysMgrLib.cliKeyTypeToTac[ keyAlgo ]
   keyReqTime = sshConfigReq.sshKeyResetRequest.get( keyAlgoTac )
   keyProcTime = sshStatus.sshKeyResetProcessed.get( keyAlgoTac )
   if keyReqTime and keyProcTime < keyReqTime:
      mode.addError( "Key Generation currently in progress. Unable to display key." )
      return SysMgrModels.SshHostKey()
   pubKeyPath += '.pub'
   pubKey = ''
   try:
      with open( pubKeyPath, 'r' ) as f:
         pubKey = f.readline()
   except IOError:
      mode.addError( "Unable to find keyfile" )
      return SysMgrModels.SshHostKey()
   return SysMgrModels.SshHostKey( hostKey=pubKey.strip() )

class ShowManagementSshHostKey( ShowCommand.ShowCliCommandClass ):
   syntax = 'show management ssh hostkey KEYALGOEXPR public'
   data = {
            'management': ConfigMgmtMode.managementShowKwMatcher,
            'ssh': sshShowKwMatcher,
            'hostkey': showHostKeyKwMatcher,
            'KEYALGOEXPR': HostKeyAlgoExpression,
            'public': 'Show public key'
         }
   cliModel = SysMgrModels.SshHostKey
   handler = showPublicHostKey
BasicCli.addShowCommandClass( ShowManagementSshHostKey )

#-------------------------------------------------------------------------------
# show management ssh trusted-ca key public [ CAKEY ]
#-------------------------------------------------------------------------------
def readNonCommentedLines( mode, filename ):
   try:
      with open( filename, 'r' ) as fileObj:
         lines = fileObj.read().splitlines()
         return [ line for line in lines if line and not line.startswith( '#' ) ]
   except IOError:
      mode.addError( "Error displaying contents of file %s" % filename )
   return []

class ShowTrustedCaKey( ShowCommand.ShowCliCommandClass ):
   syntax = 'show management ssh trusted-ca key public [ CAKEY ]'
   data = { 'management': ConfigMgmtMode.managementShowKwMatcher,
            'ssh': sshShowKwMatcher,
            'trusted-ca': 'Show configured trusted CA information',
            'key': 'Show trusted CA keys',
            'public': 'Show trusted CA public keys',
            'CAKEY': trustedCaMatcher,
          }
   cliModel = SysMgrModels.TrustedCaKeys
   privileged = True

   @staticmethod
   def handler( mode, args ):
      caKeyFile = args.get( 'CAKEY' )
      filename = SshConstants.caKeyPath( SshConstants.allCaKeysFile )
      if caKeyFile:
         if caKeyFile in sshConfig.caKeyFiles:
            filename = SshConstants.caKeyPath( caKeyFile )
         else:
            return SysMgrModels.TrustedCaKeys()
      caKeys = readNonCommentedLines( mode, filename )
      return SysMgrModels.TrustedCaKeys( keys=caKeys )

BasicCli.addShowCommandClass( ShowTrustedCaKey )

#-------------------------------------------------------------------------------
# show management ssh hostkey server cert [ HOSTCERT ]
#-------------------------------------------------------------------------------
class ShowHostkeyCert( ShowCommand.ShowCliCommandClass ):
   syntax = 'show management ssh hostkey server cert [ HOSTCERT ]'
   data = { 'management': ConfigMgmtMode.managementShowKwMatcher,
            'ssh': sshShowKwMatcher,
            'hostkey': showHostKeyKwMatcher,
            'server': 'Show sshd hostkey information',
            'cert': 'Show sshd hostkey certificate information',
            'HOSTCERT': hostCertMatcher,
          }
   cliModel = SysMgrModels.HostCertificates

   @staticmethod
   def handler( mode, args ):
      hostCertFile = args.get( 'HOSTCERT' )
      hostCertFiles = sshConfig.hostCertFiles
      if hostCertFile:
         if hostCertFile in hostCertFiles:
            hostCertFiles = [ hostCertFile ]
         else:
            return SysMgrModels.HostCertificates()
      hostCerts = {}
      for hostCert in hostCertFiles:
         try:
            hostCertPath = SshConstants.hostCertPath( hostCert )
            # Validate the host cert file
            SshCertLib.validateHostCert( hostCertPath )
            hostCertList = readNonCommentedLines( mode, hostCertPath )
            if hostCertList:
               # Accept only the first host cert because that's the only one
               # that actually takes effect
               hostCerts[ hostCertPath ] = hostCertList[ 0 ]
            if len( hostCertList ) > 1:
               mode.addWarning( "Multiple host certificates in file %s."
                                " Only the first one is affecting the"
                                " configuration." % hostCertPath )
         except SshCertLib.SshHostCertError:
            mode.addError( "Invalid host certificate %s" % hostCertPath )
      return SysMgrModels.HostCertificates( certificates=hostCerts )

BasicCli.addShowCommandClass( ShowHostkeyCert )

#-------------------------------------------------------------------------------
# show management ssh user-keys revoke-list [ REVOKELIST ]
#-------------------------------------------------------------------------------
class ShowRevokeList( ShowCommand.ShowCliCommandClass ):
   syntax = 'show management ssh user-keys revoke-list [ REVOKELIST ]'
   data = { 'management': ConfigMgmtMode.managementShowKwMatcher,
            'ssh': sshShowKwMatcher,
            'user-keys': 'Show information about configured user keys',
            'revoke-list': 'Show revoked keys',
            'REVOKELIST': revokeListMatcher,
          }
   cliModel = SysMgrModels.RevokedKeys
   privileged = True

   @staticmethod
   def handler( mode, args ):
      revokeListFile = args.get( 'REVOKELIST' )
      filename = SshConstants.revokeListPath( SshConstants.allRevokeKeysFile )
      if revokeListFile:
         if revokeListFile in sshConfig.revokedUserKeysFiles:
            filename = SshConstants.revokeListPath( revokeListFile )
         else:
            return SysMgrModels.RevokedKeys()
      revokedKeys = readNonCommentedLines( mode, filename )
      return SysMgrModels.RevokedKeys( keys=revokedKeys )

BasicCli.addShowCommandClass( ShowRevokeList )

#-------------------------------------------------------------------------------
# show management ssh [ vrf VRF ]
#-------------------------------------------------------------------------------
def showSshStatus( mode, args ):
   vrfName = args.get( 'VRF' )
   holdingConfig = sshConfig
   holdingStatus = sshStatus
   vrfFound = False
   if vrfName and vrfName in sshConfig.vrfConfig and vrfName in sshStatus.vrfStatus:
      vrfFound = True
      holdingConfig = sshConfig.vrfConfig[ vrfName ]
      holdingStatus = sshStatus.vrfStatus[ vrfName ]
   elif vrfName and not vrfFound:
      mode.addError( "VRF %s not found under SSH management" % vrfName )
      return
   vrfDisplayStr = "Default VRF"
   if vrfName:
      vrfDisplayStr = "VRF %s" % vrfName
   sshdRunning = holdingStatus.enabled
   if sshdRunning:
      sshdStatus = "enabled"
   else:
      sshdStatus = "disabled"
   outputStr = "SSHD status for %s is %s\n" % ( vrfDisplayStr, sshdStatus )
   outputStr += "SSH connection limit is %s\n" % sshConfig.connLimit
   outputStr += "SSH per host connection limit is %s\n" % sshConfig.perHostConnLimit
   sshFipsStatus = "disabled"
   if sshConfig.fipsRestrictions:
      sshFipsStatus = "enabled"
   outputStr += "FIPS status: %s\n" % sshFipsStatus
   if holdingStatus.tunnel:
      outputStr += "SSH Tunnel Information:\n"
   subFmt = ' %s: %s\n'
   for tunnelConfig in holdingConfig.tunnel.itervalues():
      tunnelStatus = holdingStatus.tunnel.get(  tunnelConfig.name )
      if not tunnelStatus:
         # Tunnel Status should always be created by SuperServer.
         # In Cli --standalone however it will not exist
         continue
      tunnelRunning = ""
      if tunnelStatus.enabled:
         if tunnelStatus.active:
            if tunnelStatus.established:
               tunnelRunning = "established"
            else:
               tunnelRunning = "running"
         else:
            tunnelRunning = "not running"
      else:
         if tunnelConfig.enable and not tunnelStatus.fullyConfigured:
            tunnelRunning = "not fully configured"
         else:
            tunnelRunning = "shutdown"
      tunnelRemote = "%s@%s:%s" % ( tunnelConfig.sshServerUsername,
                                    tunnelConfig.sshServerAddress,
                                    tunnelConfig.sshServerPort )
      tunnelRemoteBinding = "%s:%s" % ( tunnelConfig.remoteHost,
                                        tunnelConfig.remotePort )
      tunnelMaxDrops = "%s packets / %s seconds" % ( tunnelConfig.serverAliveMaxLost,
                                                  tunnelConfig.serverAliveInterval )
      restartTime = "never"
      if tunnelStatus.lastRestartTime > 0:
         time.tzset()
         restartTime = time.ctime( tunnelStatus.lastRestartTime )
      outputStr += "SSH Tunnel %s\n" % tunnelConfig.name
      outputStr += subFmt % ( "Status", tunnelRunning )
      outputStr += subFmt % ( "Local Port", tunnelConfig.localPort )
      outputStr += subFmt % ( "SSH Server Port", tunnelRemote )
      outputStr += subFmt % ( "Remote Host Port", tunnelRemoteBinding )
      outputStr += subFmt % ( "Max Packet Drops", tunnelMaxDrops )
      outputStr += subFmt % ( "Restart Count", tunnelStatus.restartCount )
      outputStr += subFmt % ( "Last Restart Time", restartTime )
      outputStr += subFmt % ( "Last Restart Cause", tunnelStatus.lastRestartCause )

   if sshConfig.user:
      outputStr += "SSH User-specific Configuration:\n"
   for user in sorted( sshConfig.user ): 
      outputStr += "SSH User %s\n" % user
      outputStr += subFmt % ( "Allow TCP forwarding", 
                              sshConfig.user[ user ].userTcpForwarding )
   print outputStr

class ShowManagementSsh( ShowCommand.ShowCliCommandClass ):
   syntax = 'show management ssh [ VRF ]'
   data = {
            'management': ConfigMgmtMode.managementShowKwMatcher,
            'ssh': sshShowKwMatcher,
            'VRF': vrfExprFactory,
         }
   handler = showSshStatus

BasicCli.addShowCommandClass( ShowManagementSsh )

#-------------------------------------------------------------------------------
# show management ssh [ip|ipv6] access-list [ACL] [summary]
#-------------------------------------------------------------------------------
def showSshAcl( mode, args ):
   aclType = 'ip' if 'ip' in args else 'ipv6'
   name = args[ '<aclNameExpr>' ]
   return AclCli.showServiceAcl( mode,
                                 aclCpConfig,
                                 aclStatus,
                                 aclCheckpoint,
                                 aclType,
                                 name,
                                 supressVrf=False,
                                 serviceName='ssh' )

class ShowManagementSshAcl( ShowCommand.ShowCliCommandClass ):
   syntax = ( 'show management ssh'
              '('
              ' ( ip access-list [ ACL4 ] ) | '
              ' ( ipv6 access-list [ ACL6 ] ) '
              ')' )
   data = {
            'management': ConfigMgmtMode.managementShowKwMatcher,
            'ssh': sshShowKwMatcher,
            'ip': AclCli.ipKwForShowServiceAcl,
            'ipv6': AclCli.ipv6KwForShowServiceAcl,
            'access-list': AclCli.accessListKwMatcherForServiceAcl,
            'ACL4': AclCli.ipAclNameExpression,
            'ACL6': AclCli.ip6AclNameExpression
          }

   cliModel = AclCliModel.AllAclList
   privileged = True
   handler = showSshAcl

BasicCli.addShowCommandClass( ShowManagementSshAcl )

#----------------------------------------------------------------
# "clear management ssh counters ( ip | ipv6 ) access-list"
#----------------------------------------------------------------
class ClearIpAclCounters( CliCommand.CliCommandClass ):
   syntax = 'clear management ssh counters ( ip | ipv6 ) access-list'
   data = { 'clear': CliToken.Clear.clearKwNode,
            'management': ConfigMgmtMode.managementClearKwMatcher,
            'ssh': 'Clear SSH statistics',
            'counters': 'Connection Counters',
            'ip': AclCli.ipKwForClearServiceAclMatcher,
            'ipv6': AclCli.ipv6KwMatcherForClearServiceAcl,
            'access-list': AclCli.accessListKwMatcherForServiceAcl }

   @staticmethod
   def handler( mode, args ):
      if 'ip' in args:
         aclType = 'ip'
      elif 'ipv6' in args:
         aclType = 'ipv6'
      else:
         assert False
      AclCli.clearServiceAclCounters( mode, aclStatus, aclCheckpoint, aclType )

BasicCli.EnableMode.addCommandClass( ClearIpAclCounters )

#----------------------------------------------------------------
# SshConfigMode
#----------------------------------------------------------------
class SshConfigMode( ConfigMgmtMode.ConfigMgmtMode ):

   name = "Ssh configuration"
   modeParseTree = CliParser.ModeParseTree()

   def __init__( self, parent, session ):
      ConfigMgmtMode.ConfigMgmtMode.__init__( self, parent, session, "ssh" )
      self.aclConfig_ = aclConfig
      self.aclCpConfig_ = aclCpConfig
      self.dscpConfig_ = dscpConfig

   def config( self ):
      return sshConfig

   def _setIdleTimeout( self, timeout ):
      to = Tac.Value( "Mgmt::SessionTimeout::IdleSessionTimeout" )
      to.timeout = timeout
      self.config().idleTimeout = to

   def setIdleTimeout( self, args ):
      self._setIdleTimeout( args[ 'IDLETIME' ] * 60 )

   def noIdleTimeout( self, args=None ):
      self._setIdleTimeout( self.config().idleTimeout.defaultTimeout )

   def setAuthenticationMode( self, args ):
      if authenticationMode.password in args:
         self.config().authenticationMode = authenticationMode.password
      else:
         self.config().authenticationMode = authenticationMode.keyboardInteractive

   def noAuthenticationMode( self, args=None ):
      self.config().authenticationMode = authenticationMode.keyboardInteractive

   def checkAlgorithms( self, requested, fipsCompliant ):
      """
      Compares the requested algorithms against the list of FIPS compliant
      counterparts and prints a warning message for algorithms that are
      not compliant.

      Only a warning message is printed and no further actions are taken.
      """
      if self.config().fipsRestrictions:
         for algo in requested:
            if algo not in fipsCompliant:
               self.addWarning(
                     "%s is disabled when using FIPS algorithms." % ( algo, ) )

   def checkSupported( self, requested, supported ):
      for algo in requested:
         if algo not in supported:
            self.addWarning(
                  "%s is not supported." % ( algo, ) )
   def setCiphers( self, args ):
      """
      Sets the cipher filter to allow only those in the iterable ciphers.
      """
      ciphers = args[ 'CIPHERS' ]
      self.checkAlgorithms( ciphers, FIPS_CIPHERS )
      self.checkSupported( ciphers, SUPPORTED_CIPHERS )
      self.config().cipher = ' '.join( ciphers )

   def noCiphers( self, args ):
      """
      Clear the SSH cipher filter list. Accept all ciphers again.
      """
      self.config().cipher = ''

   def defaultCiphers( self, args=None ):
      """
      Set cipher list to default safe ciphers.
      """
      self.config().cipher = self.config().cipherDefault

   def setKeyExchange( self, args ):
      """
      Sets the kex filter to allow only those in the iterable kex.
      """
      kexs = args[ 'KEX' ]
      self.checkSupported( kexs, SUPPORTED_KEX_ALGORITHMS )
      self.config().kex = ' '.join( args[ 'KEX' ] )

   def noKeyExchange( self, args ):
      """
      Clear the SSH kex filter list. Accept all kex methods again.
      """
      self.config().kex = ''

   def defaultKeyExchange( self, args=None ):
      """
      Set the kex filter list to safe defaults.
      """
      self.config().kex = self.config().kexDefault

   def setMac( self, args ):
      """
      Sets the mac filter to allow only those in the iterable mac.
      """
      macs = args[ 'MAC' ]
      self.checkAlgorithms( macs, FIPS_MACS )
      self.checkSupported( macs, SUPPORTED_MACS )
      self.config().mac = ' '.join( macs )

   def noMac( self, args ):
      """
      Clear the SSH mac filter list. Accept all mac algorithms again.
      """
      self.config().mac = ''

   def defaultMac( self, args=None ):
      """
      Set the mac list to the default safe MACs
      """
      self.config().mac = self.config().macDefault

   def setKnownHost( self, args ):
      host = args[ 'HOST' ]
      keyAlgo = args[ 'KEYALGO' ]
      publicKey = args[ 'PUB_KEY' ]
      generalSetKnownHost( self.config(), host, keyAlgo, publicKey,
                           cliMode=SshConfigMode )

   def noKnownHost( self, args ):
      host = args[ 'HOST' ]
      knownHosts = self.config().knownHost
      if not host in knownHosts or not knownHosts[ host ].configuredInSshMode:
         return
      generalNoKnownHost( self.config(), host )

   def _setServerPort( self, port ):
      self.config().serverPort = port
      for aclType in ( 'ip', 'ipv6' ):
         for serviceAclVrfConfig in \
               self.aclCpConfig_.cpConfig[ aclType ].serviceAcl.itervalues():
            serviceConfig = serviceAclVrfConfig.service.get( 'ssh' )
            if serviceConfig:
               serviceConfig.ports = str( port )
      self.updateDscpRules()

   def setServerPort( self, args ):
      self._setServerPort( args[ 'PORT' ] )

   def noServerPort ( self, args=None ):
      self._setServerPort( serverPort.defaultPort )

   def fipsRestrictions( self, args ):
      self.config().fipsRestrictions = True
      self.checkAlgorithms( self.config().cipher.split(), FIPS_CIPHERS )
      self.checkAlgorithms( self.config().mac.split(), FIPS_MACS )
      self.checkAlgorithms( self.config().hostkey.split(), FIPS_HOSTKEYALGORITHMS )

   def noFipsRestrictions( self, args=None ):
      self.config().fipsRestrictions = False

   def setRekeyData( self, args ):
      """
      Set the rekey data limit.
      """
      amount = args[ 'AMOUNT' ]
      rekeyUnit = args[ 'REKEYUNIT' ]

      if not SysMgrLib.checkRekeyDataLimit( amount, rekeyUnit ):
         self.addError( "Rekey data frequency must be less than 4GB." )
         return
      self.config().rekeyDataAmount = amount
      self.config().rekeyDataUnit = rekeyUnit

   def defaultRekeyData( self, args=None ):
      self.config().rekeyDataAmount = self.config().rekeyDataAmountDefault
      self.config().rekeyDataUnit = self.config().rekeyDataUnitDefault

   def setRekeyTime( self, args ):
      """
      Set the rekey time limit.
      """
      self.config().rekeyTimeLimit = args[ 'TIMELIMIT' ]
      self.config().rekeyTimeUnit = args[ 'REKEYTIMEUNIT' ]

   def defaultRekeyTime( self, args=None ):
      """
      By default have a time limit of 0 (i.e., no rekeying based on time).
      """
      self.config().rekeyTimeLimit = self.config().rekeyTimeLimitDefault
      self.config().rekeyTimeUnit = self.config().rekeyTimeUnitDefault

   def setHostKeyAlgorithms( self, args ):
      """
      Set the HostKey filter to allow the given HostKey Algorithm
      """
      # Filter for the legacy ecdsa hostkey
      # Order doesn't matter for hostkeys, so we can use a set
      hostKeySet = set( args[ 'HOSTKEYALGOS' ] )
      if legacyEcdsaToken in hostKeySet:
         hostKeySet.remove( legacyEcdsaToken )
         hostKeySet.add( "ecdsa-nistp521" )

      self.config().hostkey = ' '.join( hostKeySet )

   def defaultHostKeyAlgorithms( self, args=None ):
      """
      Set the HostKey filters to allow all HostKey Algorithms
      """
      self.config().hostkey = self.config().hostkeyDefault

   def _setLoginGraceTime( self, loginGraceTime ):
      """
      Set the time period allowed for completing a login.
      """
      loginTimeout = Tac.Value( "Mgmt::SessionTimeout::SuccessfulLoginTimeout" )
      loginTimeout.timeout = loginGraceTime
      self.config().successfulLoginTimeout = loginTimeout

   def setLoginGraceTime( self, args ):
      """
      Set the time period allowed for completing a login.
      """
      self._setLoginGraceTime( args[ 'TIMEOUT' ] )

   def noLoginGraceTime( self, args ):
      """
      Sets the time period allowed for completing a login to
      infinite.
      """
      self._setLoginGraceTime( 0 )

   def defaultLoginGraceTime( self, args ):
      """
      Set the time period allowed for completing a login back
      to the default.
      """
      self._setLoginGraceTime( self.config().successfulLoginTimeout.defaultTimeout )

   def setLogLevel( self, args ):
      """Set the log level for SSHD"""
      self.config().logLevel = args[ 'LOGLEVEL' ]

   def noLogLevel( self, args=None ):
      """Restore the log level to default for SSHD"""
      self.config().logLevel = self.config().logLevelDefault

   def enableLoggingTarget( self, args=None ):
      """Enable SSH/SSHD system logging"""
      self.config().loggingTargetEnabled = True
   
   def disableLoggingTarget( self, args=None ):
      """Disable SSH/SSHD system logging """
      self.config().loggingTargetEnabled = False

   def checkHostKeys( self, args ):
      """
      Turn on strictly checking host keys.
      """
      self.config().enforceCheckHostKeys = True

   def noCheckHostKeys( self, args=None ):
      """
      Turn off strictly checking host keys.
      """
      self.config().enforceCheckHostKeys = False

   def setPermitEmptyPasswords( self, args ):
      if 'auto' in args:
         val = 'automatic'
      elif 'permit' in args:
         val = 'permit'
      elif 'deny' in args:
         val = 'deny'
      self.config().permitEmptyPasswords = val

   def setPermitEmptyPasswordsDefault( self, args=None ):
      self.config().permitEmptyPasswords = \
          self.config().permitEmptyPasswordsDefault

   def setClientAliveInterval( self, args ):
      self.config().clientAliveInterval = args[ 'INTERVAL' ]

   def setClientAliveIntervalDefault( self, args=None ):
      self.config().clientAliveInterval = \
         self.config().clientAliveIntervalDefault

   def setClientAliveCountMax( self, args ):
      self.config().clientAliveCountMax = args[ 'SECONDS' ]

   def setClientAliveCountMaxDefault( self, args=None ):
      self.config().clientAliveCountMax = \
         self.config().clientAliveCountMaxDefault

   def shutdown( self, args ):
      self.config().serverState = "disabled"

   def noShutdown( self, args=None ):
      self.config().serverState = "enabled"

   def setIpAcl( self, args ):
      aclName = args[ 'ACL_NAME' ]
      vrfName = args.get( 'VRF' )
      AclCliLib.setServiceAcl( self, 'ssh', IPPROTO_TCP,
                               self.aclConfig_, self.aclCpConfig_, aclName,
                               'ip', vrfName, port=[ self.config().serverPort ] )

   def noIpAcl( self, args=None ):
      if args:
         aclName = args.get( 'ACL_NAME' )
         vrfName = args.get( 'VRF' )
      else:
         aclName = None
         vrfName = None
      AclCliLib.noServiceAcl( self, 'ssh', self.aclConfig_, self.aclCpConfig_,
                              aclName, 'ip', vrfName )

   def setIp6Acl( self, args ):
      aclName = args[ 'ACL_NAME' ]
      vrfName = args.get( 'VRF' )
      AclCliLib.setServiceAcl( self, 'ssh', IPPROTO_TCP,
                               self.aclConfig_, self.aclCpConfig_, aclName,
                               'ipv6', vrfName, port=[ self.config().serverPort ] )

   def noIp6Acl( self, args=None ):
      if args:
         aclName = args.get( 'ACL_NAME' )
         vrfName = args.get( 'VRF' )
      else:
         aclName = None
         vrfName = None
      AclCliLib.noServiceAcl( self, 'ssh', self.aclConfig_, self.aclCpConfig_,
                              aclName, 'ipv6', vrfName )

   def gotoSshMainTunnelConfigMode( self, args ):
      tunnelName = args[ 'TUNNELNAME' ]
      childMode = self.childMode( SshMainTunnelConfigMode, tunnelName=tunnelName )
      self.session_.gotoChildMode( childMode )

   def noMainSshTunnel( self, args ):
      tunnelName = args[ 'TUNNELNAME' ]
      if not tunnelName in sshConfig.tunnel or \
         not sshConfig.tunnel[ tunnelName ].configuredInSshMode:
         return
      childMode = self.childMode( SshMainTunnelConfigMode, tunnelName=tunnelName )
      childMode.modeletMap[ SshTunnelConfigModelet ].setShutdown()
      del sshConfig.tunnel[ tunnelName ]

   def updateDscpRules( self ):
      dscpValue = self.config().dscpValue

      if not dscpValue:
         del self.dscpConfig_.protoConfig[ 'ssh' ]
         return

      protoConfig = self.dscpConfig_.newProtoConfig( 'ssh' )
      ruleColl = protoConfig.rule
      ruleColl.clear()

      for vrf in itertools.chain( [ DEFAULT_VRF ], self.config().vrfConfig ):
         # Traffic to external ssh server.
         DscpCliLib.addDscpRule( ruleColl, '0.0.0.0', 22, False, vrf, 'tcp',
                                 dscpValue )
         DscpCliLib.addDscpRule( ruleColl, '::', 22, False, vrf, 'tcp', dscpValue,
                                 v6=True )

         # Traffic from internal ssh server.
         DscpCliLib.addDscpRule( ruleColl, '0.0.0.0', self.config().serverPort, True,
                                 vrf, 'tcp', dscpValue )
         DscpCliLib.addDscpRule( ruleColl, '::', self.config().serverPort, True, vrf,
                                 'tcp', dscpValue, v6=True )

   def setDscp( self, args ):
      self.config().dscpValue = args[ 'DSCP' ]
      self.updateDscpRules()

   def noDscp( self, args=None ):
      self.config().dscpValue = self.config().dscpValueDefault
      self.updateDscpRules()

   def setConnectionLimit( self, args ):
      config = self.config()
      config.connLimit = args.get( 'LIMIT', config.connLimitDefault )

   def setPerHostConnectionLimit( self, args ):
      config = self.config()
      config.perHostConnLimit = args.get( 'LIMIT', config.perHostConnLimitDefault )

   def setCaKeyFile( self, args ):
      self.config().caKeyFiles.clear()
      for fileName in args[ 'FILENAME' ]:
         self.config().caKeyFiles.add( fileName )

   def noCaKeyFile( self, args ):
      files = args.get( 'FILENAME' )
      if files:
         for fileName in files:
            self.config().caKeyFiles.remove( fileName )
      else:
         self.config().caKeyFiles.clear()

   def setHostCertFile( self, args ):
      self.config().hostCertFiles.clear()
      certFiles = args[ 'FILENAME' ]
      hostCertKeyTypes = SshCertLib.hostCertsByKeyTypes( certFiles )
      for keyType, certs in hostCertKeyTypes.items():
         # If the keyType is invalid, it's probably because cert doesn't exist,
         # since we validate the host certs when they are copied to the fs. 
         # Don't need to complain now; user can copy the file later.
         if keyType == KeyType.invalid:
            continue
         if len( certs ) > 1:
            warnMsg = ( "Host certificates %s have the same key type: %s."
                        " Only one of these certificates will have an effect on"
                        " the SSH config."
                        " Please remove the conflicting certificates." )
            self.addWarning( warnMsg % ( ", ".join( certs ), keyType ) )
      for fileName in certFiles:
         self.config().hostCertFiles.add( fileName )

   def noHostCertFile( self, args ):
      files = args.get( 'FILENAME' )
      if files:
         for fileName in files:
            self.config().hostCertFiles.remove( fileName )
      else:
         self.config().hostCertFiles.clear()

   def setRevokedUserKeysFile( self, args ):
      self.config().revokedUserKeysFiles.clear()
      for fileName in args[ 'FILENAME' ]:
         self.config().revokedUserKeysFiles.add( fileName )

   def noRevokedUserKeysFile( self, args ):
      files = args.get( 'FILENAME' )
      if files:
         for fileName in files:
            self.config().revokedUserKeysFiles.remove( fileName )
      else:
         self.config().revokedUserKeysFiles.clear()

   def noSshCertConf (self, args=None ):
      self.config().caKeyFiles.clear()
      self.config().hostCertFiles.clear()
      self.config().revokedUserKeysFiles.clear()

   def gotoSshUserConfigMode( self, args ):
      userName = args[ 'USER' ]
      childMode = self.childMode( SshUserConfigMode, userName=userName )
      self.session_.gotoChildMode( childMode )

   def noSshUser( self, args ):
      del self.config().user[ args[ 'USER' ] ]

def gotoSshConfigMode( mode, args ):
   childMode = mode.childMode( SshConfigMode )
   mode.session_.gotoChildMode( childMode )

def defaultSshConfig( mode, args ):
   childMode = mode.childMode( SshConfigMode )
   childMode.noIdleTimeout()
   childMode.noAuthenticationMode()
   childMode.noServerPort()
   childMode.defaultCiphers()
   childMode.noFipsRestrictions()
   childMode.defaultMac()
   childMode.defaultKeyExchange()
   childMode.defaultHostKeyAlgorithms()
   childMode.noCheckHostKeys()
   childMode.setPermitEmptyPasswordsDefault()
   childMode.setClientAliveIntervalDefault()
   childMode.setClientAliveCountMaxDefault()
   for tunnelName in sshConfig.tunnel:
      childMode.noMainSshTunnel( { 'TUNNELNAME': tunnelName } )
   childMode.noIpAcl()
   childMode.noIp6Acl()
   childMode.noShutdown()
   childMode.noDscp()
   childMode.setConnectionLimit( {} )
   childMode.setPerHostConnectionLimit( {} )
   childMode.removeComment()
   childMode.noSshCertConf()
   childMode.noLogLevel()
   childMode.disableLoggingTarget()
   for userName in sshConfig.user:
      childMode.noSshUser( { 'USER': userName } )

def doResetHostKey( mode, args ):
   keyAlgo = args[ 'KEYALGO' ]
   if sshConfig.fipsRestrictions and \
      not ( keyAlgo == "rsa" or keyAlgo == "ecdsa-nistp256" ):
      mode.addError( "Can only regenerate FIPS keys when FIPS mode is on" )
   else:
      mappedKeyAlgo = SysMgrLib.cliKeyTypeToTac[ keyAlgo ]
      sshConfigReq.sshKeyResetRequest[ mappedKeyAlgo ] = Tac.now()

#-----------------------------------------------------------------------------------
# [ no | default ] management ssh
#-----------------------------------------------------------------------------------
class ResetSshHostKey( CliCommand.CliCommandClass ):
   syntax = 'reset ssh hostkey KEYALGOEXPR'
   data = {
            'reset': CliToken.Reset.resetKwApi,
            'ssh': 'Configure ssh',
            'hostkey': CliCommand.Node(
                        matcher=CliMatcher.KeywordMatcher( 'hostkey',
                                             helpdesc='Regenerate sshd hostkeys' ),
                        guard=CommonGuards.ssoStandbyGuard ),
            'KEYALGOEXPR': HostKeyAlgoExpression
          }
   handler = doResetHostKey

BasicCli.EnableMode.addCommandClass( ResetSshHostKey )

#-----------------------------------------------------------------------------------
# [ no | default ] management ssh
#-----------------------------------------------------------------------------------
class ManagmentSsh( CliCommand.CliCommandClass ):
   syntax = 'management ssh'
   noOrDefaultSyntax = 'management ssh'
   data = {
            'management': ConfigMgmtMode.managementKwMatcher,
            'ssh': 'Configure ssh'
          }
   handler = gotoSshConfigMode
   noOrDefaultHandler = defaultSshConfig

BasicCli.GlobalConfigMode.addCommandClass( ManagmentSsh )

#-----------------------------------------------------------------------------------
# (config-mgmt-ssh) [ no | default ] connection limit LIMIT
#-----------------------------------------------------------------------------------
class SshSessionLimit( CliCommand.CliCommandClass ):
   syntax = 'connection limit LIMIT'
   noOrDefaultSyntax = syntax.replace( 'LIMIT', '...' )
   data = {
            'connection': connectionKwMatcher,
            'limit': 'Set maximum number of SSH sessions',
            'LIMIT': CliMatcher.IntegerMatcher( 0, 100,
                                                  helpdesc='Maximum number of '
                                                           'SSH connections' ),
          }
   handler = SshConfigMode.setConnectionLimit
   noOrDefaultHandler = SshConfigMode.setConnectionLimit

SshConfigMode.addCommandClass( SshSessionLimit )

#-----------------------------------------------------------------------------------
# (config-mgmt-ssh) [ no | default ] connection per-host LIMIT
#-----------------------------------------------------------------------------------
class SshPerHostLimit( CliCommand.CliCommandClass ):
   syntax = 'connection per-host LIMIT'
   noOrDefaultSyntax = 'connection per-host ...'
   data = {
            'connection': connectionKwMatcher,
            'per-host': 'Set maximum number of SSH sessions from a single host',
            'LIMIT': CliMatcher.IntegerMatcher( 1, 20,
                                                  helpdesc='Maximum number of '
                                                           'SSH connections' ),
          }
   handler = SshConfigMode.setPerHostConnectionLimit
   noOrDefaultHandler = SshConfigMode.setPerHostConnectionLimit

SshConfigMode.addCommandClass( SshPerHostLimit )

#-----------------------------------------------------------------------------------
# [ no | default ] known-hosts HOST KEYALGOEXPR PUB_KEY
#-----------------------------------------------------------------------------------
knownHostsConfigKwMatcher = CliMatcher.KeywordMatcher( 'known-hosts',
                                          alternates=[ 'known-host' ],
                                          helpdesc='SSH Known Hosts public keys' )

class KnownHosts( CliCommand.CliCommandClass ):
   syntax = 'known-hosts HOST KEYALGOEXPR PUB_KEY'
   noOrDefaultSyntax = 'known-hosts HOST ...'
   data = {
            'known-hosts': knownHostsConfigKwMatcher,
            'HOST': HostnameCli.IpAddrOrHostnameMatcher( ipv6=True ),
            'KEYALGOEXPR': HostKeyAlgoExpression,
            'PUB_KEY': CliMatcher.PatternMatcher( pattern='[A-Za-z0-9+/=]+',
                                                  helpdesc='Base 64 string',
                                                  helpname='STRING' ),
          }
   handler = SshConfigMode.setKnownHost
   noOrDefaultHandler = SshConfigMode.noKnownHost
   hidden = True

SshConfigMode.addCommandClass( KnownHosts )

#-----------------------------------------------------------------------------------
# RFE40658 : Allow precise configuration of SSH allowed ciphers
# [ no | default ] cipher CIPHERS
#-----------------------------------------------------------------------------------
CiphersExpression = CliCommand.setCliExpression( CIPHERS, name='CIPHERS' )
class Cipher( CliCommand.CliCommandClass ):
   syntax = 'cipher CIPHEREXPR'
   noSyntax = 'cipher ...'
   defaultSyntax = 'cipher ...'
   data = {
            'cipher': 'Exclusive list of cryptographic ciphers for sshd to speak',
            'CIPHEREXPR': CiphersExpression
          }
   handler = SshConfigMode.setCiphers
   noHandler = SshConfigMode.noCiphers
   defaultHandler = SshConfigMode.defaultCiphers

SshConfigMode.addCommandClass( Cipher )

#-----------------------------------------------------------------------------------
# Hidden command to deal with legacy upgreades that use 'ciphers'
# [ no | default ] ciphers CIPHERS
#-----------------------------------------------------------------------------------
class Ciphers( CliCommand.CliCommandClass ):
   syntax = 'ciphers CIPHEREXPR'
   data = {
            'ciphers': 'Exclusive list of cryptographic ciphers for sshd to speak',
            'CIPHEREXPR': CiphersExpression
          }
   handler = SshConfigMode.setCiphers
   hidden = True

SshConfigMode.addCommandClass( Ciphers )

#-----------------------------------------------------------------------------------
# [ no | default ] key-exchange KEXEXPR
#-----------------------------------------------------------------------------------
class KeyExchange( CliCommand.CliCommandClass ):
   syntax = 'key-exchange KEXEXPR'
   noSyntax = 'key-exchange ...'
   defaultSyntax = 'key-exchange ...'
   data = {
            'key-exchange': ( 'Exclusive list of key-exchange methods for sshd '
                              'to speak' ),
            'KEXEXPR': CliCommand.setCliExpression( KEXALGORITHMS, name='KEX' )
          }
   handler = SshConfigMode.setKeyExchange
   noHandler = SshConfigMode.noKeyExchange
   defaultHandler = SshConfigMode.defaultKeyExchange

SshConfigMode.addCommandClass( KeyExchange )

#-----------------------------------------------------------------------------------
# [ no | default ] mac MAC
#-----------------------------------------------------------------------------------
class MacCmd( CliCommand.CliCommandClass ):
   syntax = 'mac MACEXPR'
   noSyntax = 'mac ...'
   defaultSyntax = 'mac ...'
   data = {
            'mac': 'Exclusive list of MAC algorithms for sshd to speak',
            'MACEXPR': CliCommand.setCliExpression( MACS, name='MAC' )
          }
   handler = SshConfigMode.setMac
   noHandler = SshConfigMode.noMac
   defaultHandler = SshConfigMode.defaultMac

SshConfigMode.addCommandClass( MacCmd )

#-----------------------------------------------------------------------------------
# [ no | default ] idle-timeout IDLETIME
#-----------------------------------------------------------------------------------
class IdleTimeout( CliCommand.CliCommandClass ):
   syntax = 'idle-timeout IDLETIME'
   noOrDefaultSyntax = 'idle-timeout ...'
   data = {
            'idle-timeout': 'Set idle session timeout(minutes)',
            'IDLETIME': CliMatcher.IntegerMatcher( 0, 86400,
                                                   helpdesc='Idle session timeout '
                                                            'in minutes' )
          }
   handler = SshConfigMode.setIdleTimeout
   noOrDefaultHandler = SshConfigMode.noIdleTimeout

SshConfigMode.addCommandClass( IdleTimeout )
#-----------------------------------------------------------------------------------
# [ no | default ] authentication mode ( password | keyboard-interactive )
#-----------------------------------------------------------------------------------
class AuthenticationModeCmd( CliCommand.CliCommandClass ):
   syntax = 'authentication mode ( password | keyboard-interactive )'
   noOrDefaultSyntax = 'authentication mode ...'
   data = {
            'authentication': authKwMatcher,
            'mode': 'Change authentication mode',
            'password': 'Makes sshd enter password mode',
            'keyboard-interactive': 'Makes sshd enter keyboard-interactive mode',
          }
   handler = SshConfigMode.setAuthenticationMode
   noOrDefaultHandler = SshConfigMode.noAuthenticationMode

SshConfigMode.addCommandClass( AuthenticationModeCmd )
#-----------------------------------------------------------------------------------
# [ no | default ] authentication empty-passwords ( auto | permit | deny )
#-----------------------------------------------------------------------------------
class AuthenticationEmptyPasswords( CliCommand.CliCommandClass ):
   syntax = 'authentication empty-passwords ( auto | permit | deny )'
   noOrDefaultSyntax = 'authentication empty-passwords ...'
   data = {
            'authentication': authKwMatcher,
            'empty-passwords': 'Whether to allow empty passwords',
            'auto': 'Auto-determine whether to allow empty passwords (default)',
            'permit': 'Allow empty passwords',
            'deny': 'Do not allow empty passwords',
          }
   handler = SshConfigMode.setPermitEmptyPasswords
   noOrDefaultHandler = SshConfigMode.setPermitEmptyPasswordsDefault

SshConfigMode.addCommandClass( AuthenticationEmptyPasswords )
#-----------------------------------------------------------------------------------
# [ no | default ] client-alive interval INTERVAL
#-----------------------------------------------------------------------------------
class ClientAliveInterval( CliCommand.CliCommandClass ):
   syntax = 'client-alive interval INTERVAL'
   noOrDefaultSyntax = 'client-alive interval ...'
   data = {
            'client-alive': 'Set SSH ClientAlive options',
            'interval': 'Time period ( in seconds ) to send SSH keep-alive packets',
            'INTERVAL': thousandIntMatcher
          }
   handler = SshConfigMode.setClientAliveInterval
   noOrDefaultHandler = SshConfigMode.setClientAliveIntervalDefault

SshConfigMode.addCommandClass( ClientAliveInterval )
#-----------------------------------------------------------------------------------
# [ no | default ] client-alive count-max SECONDS
#-----------------------------------------------------------------------------------
class ClientAliveCountMax( CliCommand.CliCommandClass ):
   syntax = 'client-alive count-max SECONDS'
   noOrDefaultSyntax = 'client-alive count-max ...'
   data = {
            'client-alive': 'Set SSH ClientAlive options',
            'count-max': ( 'Number of keep-alive packets that can be sent without '
                           'a response before the connection is assumed dead' ),
            'SECONDS': thousandIntMatcher
          }
   handler = SshConfigMode.setClientAliveCountMax
   noOrDefaultHandler = SshConfigMode.setClientAliveCountMaxDefault

SshConfigMode.addCommandClass( ClientAliveCountMax )
#-----------------------------------------------------------------------------------
# [ no | default ] rekey frequency AMOUNT REKEYUNIT
#-----------------------------------------------------------------------------------
class RekeyFrequency( CliCommand.CliCommandClass ):
   syntax = 'rekey frequency AMOUNT REKEYUNIT'
   noOrDefaultSyntax = 'rekey frequency ...'
   data = {
            'rekey': 'When to rekey ssh connection',
            'frequency': 'rekey upon meeting criteria',
            'AMOUNT': CliMatcher.IntegerMatcher( 1, sys.maxint,
                                       helpdesc='Amount of data before rekeying' ),
            'REKEYUNIT': CliMatcher.EnumMatcher( REKEYUNITS )
          }
   handler = SshConfigMode.setRekeyData
   noOrDefaultHandler = SshConfigMode.defaultRekeyData

SshConfigMode.addCommandClass( RekeyFrequency )
#-----------------------------------------------------------------------------------
# [ no | default ] rekey interval TIMELIMIT REKEYTIMEUNIT
#-----------------------------------------------------------------------------------
class RekeyInterval( CliCommand.CliCommandClass ):
   syntax = 'rekey interval TIMELIMIT REKEYTIMEUNIT'
   noOrDefaultSyntax = 'rekey interval ...'
   data = {
            'rekey': 'When to rekey ssh connection',
            'interval': 'rekey after alloted time',
            'TIMELIMIT': CliMatcher.IntegerMatcher( 0, 99999,
                                       helpdesc='Amount of time before rekeying' ),
            'REKEYTIMEUNIT': CliMatcher.EnumMatcher( REKEYTIMEUNITS )
          }
   handler = SshConfigMode.setRekeyTime
   noOrDefaultHandler = SshConfigMode.defaultRekeyTime

SshConfigMode.addCommandClass( RekeyInterval )
#-----------------------------------------------------------------------------------
# [ no | default ] hostkey client strict-checking
#-----------------------------------------------------------------------------------
class HostkeyClient( CliCommand.CliCommandClass ):
   syntax = 'hostkey client strict-checking'
   noOrDefaultSyntax = 'hostkey client strict-checking ...'
   data = {
            'hostkey': hostkeyKwMatcher,
            'client': 'hostkey settings for ssh connections from the switch',
            'strict-checking': 'Enforce strict host key checking',
          }
   handler = SshConfigMode.checkHostKeys
   noOrDefaultHandler = SshConfigMode.noCheckHostKeys

SshConfigMode.addCommandClass( HostkeyClient )
#-----------------------------------------------------------------------------------
# [ no | default ] hostkey server HOSTKEYALGOS
#-----------------------------------------------------------------------------------
hostkeyServerKwMatcher = CliMatcher.KeywordMatcher( 'server',
                                     helpdesc='Switch\'s SSH hostkey settings' )

class HostkeyServer( CliCommand.CliCommandClass ):
   syntax = 'hostkey server HOSTKEYALGOS'
   noOrDefaultSyntax = 'hostkey server ...'
   data = {
            'hostkey': hostkeyKwMatcher,
            'server' : hostkeyServerKwMatcher,
            'HOSTKEYALGOS': hostKeyMatcher
          }
   handler = SshConfigMode.setHostKeyAlgorithms
   noOrDefaultHandler = SshConfigMode.defaultHostKeyAlgorithms

SshConfigMode.addCommandClass( HostkeyServer )
#-----------------------------------------------------------------------------------
# [ no | default ] server-port PORT
#-----------------------------------------------------------------------------------
class ServerPort( CliCommand.CliCommandClass ):
   syntax = 'server-port PORT'
   noOrDefaultSyntax = 'server-port ...'
   data = {
            'server-port': 'Change server port',
            'PORT': CliMatcher.IntegerMatcher( serverPort.min, serverPort.max,
                                               helpdesc='Number of the port to '
                                                        'use' ),
          }
   handler = SshConfigMode.setServerPort
   noOrDefaultHandler = SshConfigMode.noServerPort

SshConfigMode.addCommandClass( ServerPort )
#-----------------------------------------------------------------------------------
# [ no | default ] fips restrictions
#-----------------------------------------------------------------------------------
class FipsRestrictions( CliCommand.CliCommandClass ):
   syntax = 'fips restrictions'
   noOrDefaultSyntax = 'fips restrictions ...'
   data = {
            'fips': 'FIPS settings',
            'restrictions': 'Use FIPS compliant algorithms',
          }
   handler = SshConfigMode.fipsRestrictions
   noOrDefaultHandler = SshConfigMode.noFipsRestrictions

SshConfigMode.addCommandClass( FipsRestrictions )
#-----------------------------------------------------------------------------------
# [ no | default ] shutdown
#-----------------------------------------------------------------------------------
class Shutdown( CliCommand.CliCommandClass ):
   syntax = 'shutdown'
   noOrDefaultSyntax = 'shutdown ...'
   data = {
            'shutdown': 'Disable sshd'
          }
   handler = SshConfigMode.shutdown
   noOrDefaultHandler = SshConfigMode.noShutdown

SshConfigMode.addCommandClass( Shutdown )
#-----------------------------------------------------------------------------------
# [ no | default ] login timeout TIMEOUT
#-----------------------------------------------------------------------------------
class LoginTimeout( CliCommand.CliCommandClass ):
   syntax = 'login timeout TIMEOUT'
   noSyntax = 'login timeout'
   defaultSyntax = 'login timeout'
   data = {
            'login': 'Set options related to logging in',
            'timeout': 'Set the time allowed for a user to log in via ssh',
            'TIMEOUT': CliMatcher.IntegerMatcher( 1, 600,
                                            helpdesc='time allowed ( in seconds ).' )
          }
   handler = SshConfigMode.setLoginGraceTime
   noHandler = SshConfigMode.noLoginGraceTime
   defaultHandler = SshConfigMode.defaultLoginGraceTime

SshConfigMode.addCommandClass( LoginTimeout )
#-----------------------------------------------------------------------------------
# [ no | default ] log-level LOGLEVEL
#-----------------------------------------------------------------------------------
LOG_LEVELS = ( 'quiet', 'fatal', 'error', 'info', 'verbose',
               'debug', 'debug1', 'debug2', 'debug3' )
LOG_LEVELS_HELPDESC = { i: i.upper() + ' log level' for i in LOG_LEVELS }

class LogLevel( CliCommand.CliCommandClass ):
   syntax = 'log-level LOGLEVEL'
   noOrDefaultSyntax = 'log-level ...'
   data = {
            'log-level': 'Configure SSH daemon logging level',
            'LOGLEVEL': CliMatcher.EnumMatcher( LOG_LEVELS_HELPDESC )
          }
   handler = SshConfigMode.setLogLevel
   noOrDefaultHandler = SshConfigMode.noLogLevel

SshConfigMode.addCommandClass( LogLevel )
#-----------------------------------------------------------------------------------
# [ no | default ] logging target system
#-----------------------------------------------------------------------------------
class LoggingTarget( CliCommand.CliCommandClass ):
   syntax = 'logging target system'
   noOrDefaultSyntax = syntax
   data = {
            'logging': 'Configure SSH system logging',
            'target': 'Specify SSH/SSHD service syslog target',
            'system': 'Set SSH log messages destination to system log buffer'
          }
   handler = SshConfigMode.enableLoggingTarget
   noOrDefaultHandler = SshConfigMode.disableLoggingTarget

SshConfigMode.addCommandClass( LoggingTarget )
#-----------------------------------------------------------------------------------
# [ no | default ] ip access-group NAME [ vrf VRF ] in
#-----------------------------------------------------------------------------------
vrfWithNameExprFactory = VrfExprFactory(
      helpdesc='Configure the VRF in which to apply the access control list',
      inclDefaultVrf=True )
class IpAccessGroup( CliCommand.CliCommandClass ):
   syntax = 'ip access-group ACL_NAME [ VRF ] in'
   noOrDefaultSyntax = 'ip access-group [ ACL_NAME ] [ VRF ] in'
   data = {
            'ip': 'SSH IP configuration',
            'access-group': 'Configure access control list',
            'ACL_NAME': AclCli.standardIpAclNameMatcher,
            'VRF': vrfWithNameExprFactory,
            'in': 'Inbound packets'

          }
   handler = SshConfigMode.setIpAcl
   noOrDefaultHandler = SshConfigMode.noIpAcl

SshConfigMode.addCommandClass( IpAccessGroup )
#-----------------------------------------------------------------------------------
# [ no | default ] ipv6 access-group NAME [ vrf VRF ] in
#-----------------------------------------------------------------------------------
class Ipv6AccessGroup( CliCommand.CliCommandClass ):
   syntax = 'ipv6 access-group ACL_NAME [ VRF ] in'
   noOrDefaultSyntax = 'ipv6 access-group [ ACL_NAME ] [ VRF ] in'
   data = {
            'ipv6': 'SSH IPv6 configuration',
            'access-group': 'Configure access control list',
            'ACL_NAME': AclCli.standardIp6AclNameMatcher,
            'VRF': vrfWithNameExprFactory ,
            'in': 'Inbound packets'

          }
   handler = SshConfigMode.setIp6Acl
   noOrDefaultHandler = SshConfigMode.noIp6Acl

SshConfigMode.addCommandClass( Ipv6AccessGroup )
#-----------------------------------------------------------------------------------
# [ no | default ] tunnel TUNNELNAME
#-----------------------------------------------------------------------------------
class Tunnel( CliCommand.CliCommandClass ):
   syntax = 'tunnel TUNNELNAME'
   noOrDefaultSyntax = 'tunnel TUNNELNAME ...'
   data = {
            'tunnel': 'manage named SSH tunnel',
            'TUNNELNAME': CliMatcher.PatternMatcher( pattern='[A-Za-z0-9_-]+',
                                        helpdesc='SSH tunnel name', helpname='WORD' )
          }
   handler = SshConfigMode.gotoSshMainTunnelConfigMode
   noOrDefaultHandler = SshConfigMode.noMainSshTunnel
   hidden = True

SshConfigMode.addCommandClass( Tunnel )

#-----------------------------------------------------------------------------------
# (config-mgmt-ssh) [no | default] trusted-ca key public FILE1 [FILE2 ...]
#-----------------------------------------------------------------------------------
class SshCaPublicKeyFile( CliCommand.CliCommandClass ):
   syntax = "trusted-ca key public { FILENAME }"
   noOrDefaultSyntax = "trusted-ca key public [ { FILENAME } ]"
   data = {
            'trusted-ca': 'Configure trusted CA',
            'key'       : 'Configure CA public key',
            'public'    : 'Configure CA public key file',
            'FILENAME': trustedCaMatcher,
          }
   handler = SshConfigMode.setCaKeyFile
   noOrDefaultHandler = SshConfigMode.noCaKeyFile

SshConfigMode.addCommandClass( SshCaPublicKeyFile )

#-----------------------------------------------------------------------------------
# (config-mgmt-ssh) [no | default] hostkey server cert FILE1 [FILE2 ...]
#-----------------------------------------------------------------------------------
class SshHostCertFile( CliCommand.CliCommandClass ):
   syntax = "hostkey server cert { FILENAME }"
   noOrDefaultSyntax = "hostkey server cert [ { FILENAME } ]"
   data = {
            'hostkey'   : hostkeyKwMatcher,
            'server'    : hostkeyServerKwMatcher,
            'cert'      : 'Configure switch\'s hostkey cert file',
            'FILENAME': hostCertMatcher,
          }
   handler = SshConfigMode.setHostCertFile
   noOrDefaultHandler = SshConfigMode.noHostCertFile

SshConfigMode.addCommandClass( SshHostCertFile )

#-----------------------------------------------------------------------
# (config-mgmt-ssh) [no|default] user-keys revoke-list FILE
#-----------------------------------------------------------------------
class SshRevokedKeysFile( CliCommand.CliCommandClass ):
   syntax = "user-keys revoke-list { FILENAME }"
   noOrDefaultSyntax = "user-keys revoke-list [ { FILENAME } ]"
   data = {
            'user-keys'   : 'SSH user keys\' settings',
            'revoke-list' : 'Configure revoked SSH user keys file',
            'FILENAME'  : revokeListMatcher,
          }
   handler = SshConfigMode.setRevokedUserKeysFile
   noOrDefaultHandler = SshConfigMode.noRevokedUserKeysFile

SshConfigMode.addCommandClass( SshRevokedKeysFile )

#-----------------------------------------------------------------------------------
# [ no | default ] username user
#-----------------------------------------------------------------------------------
class UserConfig( CliCommand.CliCommandClass ):
   syntax = 'username USER'
   noOrDefaultSyntax = 'username USER ...'
   data = {
            'username': 'Enter SSH user specific configuration submode',
            'USER': CliMatcher.PatternMatcher( pattern='[A-Za-z0-9_-]+',
                                    helpdesc='SSH user name', helpname='WORD' )
          }
   handler = SshConfigMode.gotoSshUserConfigMode
   noOrDefaultHandler = SshConfigMode.noSshUser

SshConfigMode.addCommandClass( UserConfig )

#-----------------------------------------------------------------------------------
# [ no | default ] qos dscp DSCP
#-----------------------------------------------------------------------------------
DscpCliLib.addQosDscpCommandClass( SshConfigMode, SshConfigMode.setDscp,
                                   SshConfigMode.noDscp )

#-------------------------------------------------------------------------------
# "vrf VRF" config mode
#-------------------------------------------------------------------------------
class SshVrfConfigMode( VrfConfigMode, BasicCli.ConfigModeBase ):
   name = "SSH VRF Configuration"
   modeParseTree = CliParser.ModeParseTree()

   def __init__( self, parent, session, vrfName ):
      VrfConfigMode.__init__( self, ( vrfName, "ssh", sshConfig ) )
      BasicCli.ConfigModeBase.__init__( self, parent, session )

   def gotoSshVrfTunnelConfigMode( self, args ):
      tunnelName = args[ 'TUNNELNAME' ]
      childMode = self.childMode( SshVrfTunnelConfigMode,
                                  tunnelName=tunnelName,
                                  vrfConfig=self.config() )
      self.session_.gotoChildMode( childMode )

   def noSshVrfTunnel( self, args ):
      self._noSshVrfTunnel( args[ 'TUNNELNAME' ] )

   def _noSshVrfTunnel( self, tunnelName ):
      conf = sshConfig if self.vrfName_ == DEFAULT_VRF else self.config()
      if tunnelName not in conf.tunnel or \
         conf.tunnel[ tunnelName ].configuredInSshMode:
         return
      childMode = self.childMode( SshVrfTunnelConfigMode,
                                  tunnelName=tunnelName,
                                  vrfConfig=self.config() )
      childMode.modeletMap[ SshTunnelConfigModelet ].setShutdown()
      del conf.tunnel[ tunnelName ]

   def setKnownHost( self, args ):
      host = args[ 'HOST' ]
      keyAlgo = args[ 'KEYALGO' ]
      publicKey = args[ 'PUB_KEY' ]
      conf = sshConfig if self.vrfName_ == DEFAULT_VRF else self.config()
      generalSetKnownHost( conf, host, keyAlgo, publicKey,
                           cliMode=SshVrfConfigMode )

   def noKnownHost( self, args ):
      host = args[ 'HOST' ]
      conf = sshConfig if self.vrfName_ == DEFAULT_VRF else self.config()
      if not host in conf.knownHost or conf.knownHost[ host ].configuredInSshMode:
         return
      generalNoKnownHost( conf, host )

def gotoSshVrfConfigMode( mode, args ):
   vrfName = args[ 'VRF' ]
   childMode = mode.childMode( SshVrfConfigMode, vrfName=vrfName )
   mode.session_.gotoChildMode( childMode )
   mode.updateDscpRules()

def noSshVrfConfigMode( mode, args ):
   vrfName = args[ 'VRF' ]
   childMode = mode.childMode( SshVrfConfigMode, vrfName=vrfName )
   for tunnelName in childMode.config().tunnel:
      # pylint: disable-msg=protected-access
      childMode._noSshVrfTunnel( tunnelName )
      # pylint: enable-msg=protected-access
   del sshConfig.vrfConfig[ vrfName ]
   mode.updateDscpRules()

#-----------------------------------------------------------------------------------
# [ no | default ] shutdown
#-----------------------------------------------------------------------------------
class VrfConfigShutdown( CliCommand.CliCommandClass ):
   syntax = 'shutdown'
   noSyntax = 'shutdown ...'
   defaultSyntax = 'shutdown ...'
   data = {
            'shutdown': 'Disable sshd',
          }
   @staticmethod
   def handler( mode, args ):
      SshVrfConfigMode.shutdown( mode )

   @staticmethod
   def noHandler( mode, args ):
      SshVrfConfigMode.noShutdown( mode )

   @staticmethod
   def defaultHandler( mode, args ):
      SshVrfConfigMode.defaultShutdown( mode )

SshVrfConfigMode.addCommandClass( VrfConfigShutdown )

#-----------------------------------------------------------------------------------
# [ no | default ] tunnel TUNNELNAME
#-----------------------------------------------------------------------------------
class VrfConfigTunnel( CliCommand.CliCommandClass ):
   syntax = 'tunnel TUNNELNAME'
   noOrDefaultSyntax = 'tunnel TUNNELNAME ...'
   data = {
            'tunnel': 'manage named SSH tunnel',
            'TUNNELNAME': CliMatcher.PatternMatcher( pattern='[A-Za-z0-9_-]+',
                                       helpdesc='SSH tunnel name', helpname='WORD' ),
          }
   handler = SshVrfConfigMode.gotoSshVrfTunnelConfigMode
   noOrDefaultHandler = SshVrfConfigMode.noSshVrfTunnel

SshVrfConfigMode.addCommandClass( VrfConfigTunnel )

#-----------------------------------------------------------------------------------
# [ no | default ] known-hosts HOST KEYALGOEXPR PUB_KEY
#-----------------------------------------------------------------------------------
class VrfKnownHosts( CliCommand.CliCommandClass ):
   syntax = 'known-hosts HOST KEYALGOEXPR PUB_KEY'
   noOrDefaultSyntax = 'known-hosts HOST ...'
   data = {
            'known-hosts': 'SSH Known Hosts public keys',
            'HOST': HostnameCli.IpAddrOrHostnameMatcher( ipv6=True ),
            'KEYALGOEXPR': HostKeyAlgoExpression,
            'PUB_KEY': CliMatcher.PatternMatcher( pattern='[A-Za-z0-9+/=]+',
                                                      helpdesc='Base 64 string',
                                                      helpname='STRING' ),
          }
   handler = SshVrfConfigMode.setKnownHost
   noOrDefaultHandler = SshVrfConfigMode.noKnownHost

SshVrfConfigMode.addCommandClass( VrfKnownHosts )

#-----------------------------------------------------------------------------------
# [ no | default ] vrf VRF
#-----------------------------------------------------------------------------------
class VrfMode( CliCommand.CliCommandClass ):
   syntax = 'VRF'
   noOrDefaultSyntax = syntax
   data = {
            'VRF': VrfExprFactory( helpdesc='Enter VRF sub-mode',
                                   inclDefaultVrf=True ),
          }
   handler = gotoSshVrfConfigMode
   noOrDefaultHandler = noSshVrfConfigMode

SshConfigMode.addCommandClass( VrfMode )

def verifyHook( mode, hashName ):
   if hashName == 'md5' and sshConfig.fipsRestrictions:
      mode.addError( "MD5 is not allowed when using FIPS algorithms." )
      return False
   return True

def Plugin( entityManager ):
   global sshConfig, sshConfigReq, sshStatus
   global aclConfig, aclCpConfig,  entityMibStatus
   global aclCheckpoint
   global aclStatus
   global dscpConfig
   sshConfig = ConfigMount.mount( entityManager, "mgmt/ssh/config",
                                  "Mgmt::Ssh::Config", "w" )
   sshConfigReq = LazyMount.mount( entityManager, "mgmt/ssh/configReq",
                                  "Mgmt::Ssh::ConfigReq", "w" )
   sshStatus = LazyMount.mount( entityManager, Cell.path( "mgmt/ssh/status" ),
                                "Mgmt::Ssh::Status", "r" )
   aclConfig = ConfigMount.mount( entityManager, "acl/config/cli",
                                  "Acl::Input::Config", "w" )
   aclCpConfig = ConfigMount.mount( entityManager, "acl/cpconfig/cli",
                                  "Acl::Input::CpConfig", "w" )
   entityMibStatus = LazyMount.mount( entityManager,  "hardware/entmib",
                                      "EntityMib::Status", "r" )
   dscpConfig = ConfigMount.mount( entityManager,  "mgmt/dscp/config",
                                   "Mgmt::Dscp::Config", "w" )
   aclStatus = LazyMount.mount( entityManager, "acl/status/all",
                                "Acl::Status", "r" )
   aclCheckpoint = LazyMount.mount( entityManager, "acl/checkpoint",
                                   "Acl::CheckpointStatus", "w" )
   FileCli.verifyHook.addExtension( verifyHook )
