#!/usr/bin/env python
# Copyright (c) 2017 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

from ArnetModel import IpGenericAddress
from CliModel import Model
from CliModel import ( Bool,
                       Dict,
                       Enum,
                       GeneratorList,
                       Int,
                       List,
                       Str,
                       Submodel )
from TableOutput import createTable, Format
from HumanReadable import formatTimeInterval
from Tac import utcNow
from SrTePolicyCommonLib import tacEnlp
from SrTePolicyLibModel import SrTeSegmentListVia

SL_STATE_VALID = 'Valid'
SL_STATE_INVALID = 'Invalid'

def protocolStrMap( protocol ):
   enumStrMap = { 'bgp' : 'BGP',
                  'static' : 'Static' }
   return enumStrMap[ protocol ]

def formatCountersStr( prefix, _packets, _bytes ):
   if _packets is not None and _bytes is not None:
      return ", %s: %d packets, %d bytes" % ( prefix, _packets, _bytes )
   else:
      return ", %s: not available" % ( prefix )

def timeLapsedString( timestamp ):
   '''
   Convert the timestamp in seconds from epoch to string representing
   the time elapsed from now
   @timestamp : Integer representation of monotonic time since epoch
   '''
   if timestamp is None:
      return None
   deltaAge = utcNow() - timestamp
   if deltaAge < 0:
      deltaAge = 0
   return formatTimeInterval( deltaAge )

# Common models shared by all SrTePolicy show commands
class SrTeSegment( Model ):
   mplsLabelSid = Int( help="Segment Identifier in the form of an MPLS label",
                       optional=True )

# Models for showing policies
class SrTePolicyPathGroupSegmentList( Model ):
   segmentListId = Int( help="Internal Id of the segment list "
                             "for valid candidate paths",
                        optional=True )
   segmentListValid = Bool( help="Validity of the segment list "
                                 "for valid candidate paths",
                            optional=True )
   sbfdState = Enum( values=[ "adminDown", "down", "init", "up" ],
                     help="State of SBFD session", optional=True )
   segments = List( valueType=SrTeSegment,
                    help="Segments that make up the segment list" )
   weight = Int( "Weight of the segment list" )
   txPackets = Int( help="Total number of transmitted packets", optional=True )
   txBytes = Int( help="Total number of transmitted bytes", optional=True )
   counterState = Enum( values=( 'active', 'notAvailable' ), optional=True,
                        help="Counter state of the segment list" )
   vias = List( valueType=SrTeSegmentListVia,
                optional=True,
                help="List of next hops for the primary path of the segment list, "
                "if the segment list is valid" )
   backupVias = List( valueType=SrTeSegmentListVia,
                      optional=True,
                      help="List of next hops for the backup path "
                      "of the segment list, if the segment list is valid" )
   invalidReason = Enum( values=( 'unresolvedTopLabel', 'segmentListMissing',
                                  'noResolvedLabels',
                                  'resolvedLabelsExceedPlatformMsd',
                                  'sbfdDown' ),
                         optional=True,
                         help="Reason why this segment list is invalid" )

   def stateStr( self ):
      if self.segmentListValid is None:
         return ""
      if self.segmentListValid is True:
         return " State: " + SL_STATE_VALID
      else:
         return " State: " + SL_STATE_INVALID

   def sbfdStateStr( self ):
      if self.segmentListValid is None or self.sbfdState is None:
         return ""
      return ", SBFD State: " + self.sbfdState.capitalize()

   def idStr( self ):
      if self.segmentListId is not None:
         return ", ID: %u" % self.segmentListId
      return ""

   def invalidReasonStr( self ):
      invalidReasonMap = {
         'unresolvedTopLabel': 'Top label is not resolved',
         'segmentListMissing': 'Segment list is not present',
         'noResolvedLabels': 'Resolved segment list has no labels',
         'resolvedLabelsExceedPlatformMsd':'Resolved segment list exceeds Platform' +
         ' Limit',
         'sbfdDown':'SBFD state is Down',
      }
      return invalidReasonMap[ self.invalidReason ]

   def render( self ):
      countersStr = formatCountersStr( "Counters", self.txPackets, self.txBytes )
      print "\t\tSegment List:%s%s%s%s" % ( self.stateStr(), self.sbfdStateStr(),
                                            self.idStr(), countersStr )
      # For now all segments are mpls labels
      lblStkStr = "\t\t\tLabel Stack: ["
      lblStkStr += ' '.join( "%u" % segment.mplsLabelSid
                             for segment in self.segments )
      lblStkStr += "], Weight: %u" % self.weight
      print lblStkStr
      if self.invalidReason:
         print "\t\t\tInvalid Reason: %s" % ( self.invalidReasonStr() )
      if self.segmentListValid:
         for via in self.vias:
            viaStr = "\t\t\tResolved Label Stack: ["
            viaStr += ' '.join( '%u' % label for label in via.mplsLabels )
            viaStr += "], Next hop: %s, Interface: %s" % ( via.nexthop,
                                                         via.interface.stringValue )
            print viaStr
         if self.backupVias is not None:
            for backupVia in self.backupVias:
               backupViaStr = "\t\t\tBackup Resolved Label Stack: ["
               backupViaStr += \
                  ' '.join( '%u' % label for label in backupVia.mplsLabels )
               backupViaStr += \
                  "], Next hop: %s, Interface: %s" % ( backupVia.nexthop,
                                                   backupVia.interface.stringValue )
               print backupViaStr


class SrTePolicyBindingSid( Model ):
   bindingSidType = Enum( values=( "mpls", "ipv6" ),
                          default="mpls",
                          help="Binding Segment Identifier Type" )
   mplsLabelSid = Int( help="Binding Segment Identifier in the form of an MPLS "
                       "label",
                       optional=True )
class SrTePolicyOriginator( Model ):
   nodeAddress = IpGenericAddress( help="Node address of the originator" )
   asn = Str( help="Autonomous System Number" )

class SrTePolicyPathGroup( Model ):
   state = Enum( values=( "active", "valid", "invalid" ),
                 help="State of the candidate path" )
   protocol = Enum( values=( "bgp", "static" ),
                  help="Protocol source of the candidate path" )
   originator = Submodel( valueType=SrTePolicyOriginator,
                      help="Originator of the candidate path" )
   discriminator = Int( help="Protocol specific discriminator of the "
                             "candidate path",
                        optional=True )
   preference = Int( help="Preference of the candidate path" )
   bindingSid = Submodel( valueType=SrTePolicyBindingSid,
                          help="Binding Segment Identifier of the candidate path",
                          optional=True )
   enlp = Enum( values=( "ipv4", "ipv6", "ipv4AndIpv6", "none", "localPolicy" ),
                help="Explicit null label policy of the candidate path" )
   segmentLists = List( valueType=SrTePolicyPathGroupSegmentList,
                        help="Segment lists that make up the candidate path" )
   lastChangeTime = Int( optional=True,
                         help="UTC seconds since epoch when the candidate path "
                              "was last updated" )
   activeTime = Int( optional=True,
                     help="UTC seconds since epoch when the candidate "
                          "path was selected as active" )
   notActiveReason = Enum( values=( 'higherPreferenceAvailable',
                                    'betterProtocolAvailable',
                                    'lowerOriginAvailable',
                                    'higherDiscriminatorAvailable',
                                    'samePriorityActive',
                                    'segmentListsUnresolved',
                                    'segmentListsMissing',
                                    'bindingSidConflict',
                                    'bindingSidOutOfSrlbRange',
                                    'bindingSidNotConfigured',
                                    'bindingSidInUseByOtherApplication' ),
                           optional=True,
                           help="Reason why this candidate path is not active" )

   def protocolStr( self ):
      enumStrMap = { 'bgp' : 'BGP',
                     'static' : 'Static' }
      return enumStrMap[ self.protocol ]

   def reasonStr( self ):
      reasonStrMap = {
         'higherPreferenceAvailable': 'Higher preference path available',
         'betterProtocolAvailable': 'Better protocol path available',
         'lowerOriginAvailable': 'Lower originator path available',
         'higherDiscriminatorAvailable': 'Higher discriminator path available',
         'samePriorityActive': 'Same priority path active',
         'segmentListsUnresolved': 'No resolved segment lists',
         'segmentListsMissing': 'No configured segment lists',
         'bindingSidConflict': 'Binding SID conflicts with another policy',
         'bindingSidOutOfSrlbRange': 'Binding SID out of SRLB range',
         'bindingSidNotConfigured': 'No Binding SID is configured',
         'bindingSidInUseByOtherApplication':
                                    'Binding SID in use by other application'
      }
      return reasonStrMap[ self.notActiveReason ]

   def render( self ):
      timeLapsedStr = timeLapsedString( self.lastChangeTime )
      ageStr = "" if timeLapsedStr is None else ", modified: %s ago" % timeLapsedStr
      activeAgeStr = ""
      if self.state == 'active':
         timeActiveStr = timeLapsedString( self.activeTime )
         if timeActiveStr is not None:
            activeAgeStr = " (for %s)" % timeActiveStr
      print "\tPath group: State: %s%s%s" \
                  % ( self.state, activeAgeStr, ageStr )
      if self.state != 'active' and self.notActiveReason is not None:
         reasonHdr = "Inactive" if self.state == 'valid' else "Invalid"
         print "\t\t%s reason: %s" % ( reasonHdr, self.reasonStr() )
      print "\t\tProtocol: %s" % self.protocolStr()
      if self.originator is not None:
         print "\t\tOriginator: %s(AS%s)" % ( self.originator.nodeAddress,
                                              self.originator.asn )
      if self.discriminator is not None:
         print "\t\tDiscriminator: %u" % self.discriminator
      print "\t\tPreference: %u" % self.preference
      if self.bindingSid is not None:
         print "\t\tBinding SID: %u" % self.bindingSid.mplsLabelSid
      enlpStr = tacEnlp( self.enlp ).toShowString()
      print "\t\tExplicit null label policy: %s" % enlpStr
      for sl in self.segmentLists:
         sl.render()

class SrTePolicyBase( Model ):
   endpoint = IpGenericAddress( help="Segment Routing Traffic Engineering Policy "
                                "End point" )
   color = Int( help="Segment Routing Traffic Engineering Policy Color" )
   name = Str( help="Policy Name", optional=True )
   description = Str( help="Policy Description", optional=True )
   txPackets = Int( help="Aggregate number of transmitted packets for the policy",
                    optional=True )
   txBytes = Int( help="Aggregate number of transmitted bytes for the policy",
                  optional=True )

class SrTePolicyActive( SrTePolicyBase ):
   # For active policies, show only the path group that became active. Don't show
   # the others
   pathGroup = Submodel( valueType=SrTePolicyPathGroup,
                         help="Selected candidate path for the Segment Routing "
                         "Traffic Engineering policy" )
   def render( self ):
      countersStr = formatCountersStr( "Counters", self.txPackets, self.txBytes )
      nameStr = '' if self.name is None else ", Name: {}".format( self.name )
      print "Endpoint {} Color {}{}{}".format(
            self.endpoint, self.color, nameStr, countersStr )
      if self.description is not None:
         print "Description: %s" % ( self.description )
      self.pathGroup.render()

class SrTePolicyCandidates( SrTePolicyBase ):
   pathGroups = GeneratorList( valueType=SrTePolicyPathGroup,
                               help="Candidate paths for the Segment Routing "
                               "Traffic Engineering policy" )
   def render( self ):
      printEC = True
      # The path groups are sorted by state
      for pathGroup in self.pathGroups:
         # The pathGroup entries could be added or deleted in between the time the
         # policy list was considered for rendering and the time it is actually
         # rendered. Hence render the endpoint and color just before rendering the
         # pathGroup
         if printEC == True:
            countersStr = formatCountersStr( "Counters", self.txPackets,
                                             self.txBytes )
            nameStr = '' if self.name is None else ", Name: {}".format( self.name )
            print "Endpoint {} Color {}{}{}".format(
                  self.endpoint, self.color, nameStr, countersStr )
            if self.description is not None:
               print "Description: %s" % ( self.description )
            printEC = False
         pathGroup.render()

class SrTePolicyCandidatesModel( Model ):
   policies = GeneratorList( valueType=SrTePolicyCandidates,
                             help="List of Segment Routing Traffic "
                                  "Engineering policies and its candidates" )
   def render( self ):
      # The policies are sorted based on: Endpoint ( with v4 first ) followed
      # by the color
      for policy in self.policies:
         policy.render()

class SrTePolicyActiveModel( Model ):
   policies = GeneratorList( valueType=SrTePolicyActive,
                             help="List of Segment Routing Traffic "
                                  "Engineering policies installed in "
                                  "the forwarding plane" )
   def render( self ):
      for policy in self.policies:
         policy.render()

class SrTePolicyCandidatesVrfModel( Model ):
   vrfs = Dict( keyType=str, valueType=SrTePolicyCandidatesModel,
                help="Per VRF Segment Routing Traffic Engineering Policies" )
   def render( self ):
      for vrfPolicies in self.vrfs.values():
         # Don't print VRF name, the model has it for future use
         vrfPolicies.render()

class SrTePolicyActiveVrfModel( Model ):
   vrfs = Dict( keyType=str, valueType=SrTePolicyActiveModel,
                help="Per VRF Active Segment Routing Traffic Engineering Policies" )
   def render( self ):
      for vrfPolicies in self.vrfs.values():
         # Don't print VRF name, the model has it for future use
         vrfPolicies.render()

# Models for showing segment lists

class SrTePolicySegmentList( Model ):
   segments = List( valueType=SrTeSegment,
                    help="List of segments composing the segment list" )
   valid = Bool( default=True,
                 help="Validity of the segment list" )
   vias = List( valueType=SrTeSegmentListVia, optional=True,
                help="List of next hops for the primary path of the segment list" )
   backupVias = List( valueType=SrTeSegmentListVia, optional=True,
         help="List of nexthops for the backup path of the segment list" )

   def renderSegmentList( self, table, slId, validityFilter ):
      if validityFilter is not None:
         assert self.valid == validityFilter

      primaryRow = ( str( slId ),
                     '[%s]' % ' '.join( str( s.mplsLabelSid )
                                        for s in self.segments ) )
      if validityFilter is None:
         primaryRow += ( SL_STATE_VALID if self.valid else SL_STATE_INVALID, )

      def _viaColumns( via ):
         return ( '[%s]' % ' '.join( map( str, via.mplsLabels ) ),
                  via.nexthop,
                  via.interface.stringValue,
                  )

      def _backupViaColumns( via ):
         return ( 'backup ' + '[%s]' % ' '.join( map( str, via.mplsLabels ) ),
                  via.nexthop,
                  via.interface.stringValue,
                  )


      if self.vias:
         primaryRow += _viaColumns( self.vias[ 0 ] )

      table.newRow( *primaryRow )

      # Number of empty columns we need to fill for the via-only rows
      # The via columns are labels, nexthop, interface
      numNonViaCols = len( primaryRow ) - 3
      emptyCols = ( '', ) * numNonViaCols
      if self.vias:
         for via in self.vias[ 1: ]:
            table.newRow( *( emptyCols + _viaColumns( via ) ) )
      if self.backupVias:
         table.newRow(
            *( emptyCols + _backupViaColumns( self.backupVias[ 0 ] ) ) )

class SrTePolicySegmentLists( Model ):
   segmentLists = Dict( keyType=int, valueType=SrTePolicySegmentList,
                        help="A mapping from segment list ID to the segment list" )

class SrTePolicySegmentListCounters( Model ):
   txPackets = Int( help="Total number of transmitted packets", optional=True )
   txBytes = Int( help="Total number of transmitted bytes", optional=True )
   vias = List( valueType=SrTeSegmentListVia, optional=True,
                help="List of next hops for the segment list" )

   def renderSegmentList( self, table, slId ):

      def _viaColumns( via ):
         return ( '[%s]' % ' '.join( map( str, via.mplsLabels ) ),
                  via.nexthop,
                  via.interface.stringValue,
                  )

      primaryRow = ( str( slId ), )

      if self.vias:
         primaryRow += _viaColumns( self.vias[ 0 ] )

      primaryRow += ( self.txPackets if self.txPackets is not None else '-',
                      self.txBytes if self.txBytes is not None else '-' )

      table.newRow( *primaryRow )

      # Number of empty columns we need to fill for the via-only rows
      emptyCols = ( '', )
      if self.vias:
         for via in self.vias[ 1: ]:
            table.newRow( *( emptyCols + _viaColumns( via ) ) )

class SrTePolicySegmentListsCounters( Model ):
   segmentLists = Dict( keyType=int, valueType=SrTePolicySegmentListCounters,
                 help="A mapping from segment list ID to the segment list counters" )

class VrfSrTePolicySegmentListsCounters( Model ):
   vrfs = Dict( keyType=str, valueType=SrTePolicySegmentListsCounters,
                help="A mapping from VRF to the segment lists it contains" )

   def render( self ):
      headings = ( 'SL ID', 'Resolved Labels', 'Next hop', 'Interface',
                   'Packets', 'Bytes' )

      table = createTable( headings, tableWidth=120 )
      fmt = Format( justify='left' )
      table.formatColumns( *( ( fmt, ) * len( headings ) ) )

      # The only vrf should be default right now
      for vrfSegmentLists in self.vrfs.itervalues():
         for slId in sorted( vrfSegmentLists.segmentLists.keys() ):
            sl = vrfSegmentLists.segmentLists[ slId ]
            sl.renderSegmentList( table, slId )

      print table.output()

class VrfSrTePolicySegmentLists( Model ):
   vrfs = Dict( keyType=str, valueType=SrTePolicySegmentLists,
                help="A mapping from VRF to the segment lists it contains" )
   _validityFilter = Bool( optional=True,
                           help="Hidden property to indicate to render() to "
                                "display a table filtered by validity" )

   def render( self ):
      headings = ( 'SL ID', 'Policy SL' )
      if self._validityFilter is None:
         headings += ( 'State', )

      if self._validityFilter is not False:
         headings += ( 'Resolved Labels', 'Next hop', 'Interface' )

      table = createTable( headings, tableWidth=120 )
      fmt = Format( justify='left' )
      table.formatColumns( *( ( fmt, ) * len( headings ) ) )

      # The only vrf should be default right now
      for vrfSegmentLists in self.vrfs.itervalues():
         for slId in sorted( vrfSegmentLists.segmentLists.keys() ):
            sl = vrfSegmentLists.segmentLists[ slId ]
            sl.renderSegmentList( table, slId, self._validityFilter )

      print table.output()

# Models for showing policy FEC table

class SrTePolicyFecVia( Model ):
   weight = Int( help='Segment list weight' )
   segmentListId = Int( help='Segment list identifier' )

class SrTePolicyFecInfo( Model ):
   policy = Submodel( valueType=SrTePolicyBase,
                      optional=True,
                      help="Segment routing traffic engineering"
                           " policy information" )
   vias = List( valueType=SrTePolicyFecVia, optional=True,
                help='List storing the policy FEC vias' )

   def renderPolicyFecInfo( self, table, fecId ):
      if self.vias:
         firstVia = self.vias[ 0 ]
         table.newRow( '0x%x' % fecId,
                       str( self.policy.endpoint ) if self.policy else "-",
                       str( self.policy.color ) if self.policy else "-",
                       firstVia.segmentListId,
                       str( firstVia.weight ) )
         for via in self.vias[ 1 : ]:
            table.newRow( '', '', '',
                          via.segmentListId,
                          str( via.weight ) )

class SrTePolicyFecTable( Model ):
   fecs = Dict( keyType=long, valueType=SrTePolicyFecInfo, optional=True,
                help='A mapping from policy FEC ID to policy information' )

   def render( self ):
      headings = ( "FEC ID", "Endpoint", "Color", "Segment List ID", "Weight" )
      table = createTable( headings, tableWidth=120 )
      fmt = Format( justify='left' )
      table.formatColumns( *( ( fmt, ) * len( headings ) ) )
      for fecId, fecInfo in sorted( self.fecs.items() ):
         fecInfo.renderPolicyFecInfo( table, fecId )
      print table.output()

summaryHeader = ( 'Source', 'Active', 'Valid', 'Invalid', 'Total' )
totalHeader = ( 'Active', 'Valid', 'Invalid', 'Total' )
# Model for 'show traffic-engineering segment-routing policy summary'
# For path group statistics
class SrTePolicyPathGroupStatisticsEntryModel( Model ):
   activePathGroups = Int( help='Number of candidate paths in active state',
                           default=0 )
   validPathGroups = Int( help='Number of candidate paths that are valid but ' \
                             'not active', default=0 )
   invalidPathGroups = Int( help='Number of candidate paths in invalid state',
                            default=0 )
   totalPathGroups = Int( help='All candidate paths in any state',
                            default=0 )

class SrTePolicyPathGroupStatisticsModel( Model ):
   sources = Dict( keyType=str, valueType=SrTePolicyPathGroupStatisticsEntryModel,
                   help='Per source Segment Routing Traffic Engineering Policy path '
                   'group statistics' )
   total = Submodel( valueType=SrTePolicyPathGroupStatisticsEntryModel,
                     help='Total path group statistics for all Segment Routing '
                          'Traffic Engineering Policy sources' )
   def render( self ):
      print "Path group statistics:\n"
      headings = summaryHeader
      table = createTable( headings, tableWidth=120 )
      fmt = Format( justify='left' )
      table.formatColumns( *( ( fmt, ) * len( headings ) ) )
      for src in self.sources:
         primaryRow = ( protocolStrMap( src ), )
         primaryRow += ( str( self.sources[ src ].activePathGroups ), )
         primaryRow += ( str( self.sources[ src ].validPathGroups ), )
         primaryRow += ( str( self.sources[ src ].invalidPathGroups ), )
         primaryRow += ( str( self.sources[ src ].totalPathGroups ), )
         table.newRow( *primaryRow )
      lastRow = ( "Total", )
      lastRow += ( str( self.total.activePathGroups ), )
      lastRow += ( str( self.total.validPathGroups ), )
      lastRow += ( str( self.total.invalidPathGroups ), )
      lastRow += ( str( self.total.totalPathGroups ), )
      table.newRow( *lastRow )

      print table.output()

# For policy statistics
class SrTePolicyStatisticsEntryModel( Model ):
   activePolicies = Int( help='Number of policies in active state',
                         default=0 )
   validPolicies = Int( help='Number of policies that are valid but not active',
                           default=0 )
   invalidPolicies = Int( help='Number of policies in invalid state',
                          default=0 )
   totalPolicies = Int( help='All policies in any state',
                          default=0 )

class SrTePolicyStatisticsModel( Model ):
   sources = Dict( keyType=str, valueType=SrTePolicyStatisticsEntryModel,
                   help='Per source Segment Routing Traffic Engineering Policy '
                   'statistics' )
   total = Submodel( valueType=SrTePolicyStatisticsEntryModel,
                     help='Total Segment Routing Traffic Engineering policies ' \
                     'with unique Endpoint, Color values irresepctive of source' )
   def render( self ):
      print "Per source policy statistics:\n"
      headings = summaryHeader
      table = createTable( headings, tableWidth=120 )
      fmt = Format( justify='left' )
      table.formatColumns( *( ( fmt, ) * len( headings ) ) )
      for src in self.sources:
         table.newRow( protocolStrMap( src ),
                       str( self.sources[ src ].activePolicies ),
                       str( self.sources[ src ].validPolicies ),
                       str( self.sources[ src ].invalidPolicies ),
                       str( self.sources[ src ].totalPolicies ) )
      table.newRow( 'Total',
                    str( self.total.activePolicies ),
                    str( self.total.validPolicies ),
                    str( self.total.invalidPolicies ),
                    str( self.total.totalPolicies ) )
      print table.output()

# For segment list statistics
class SrTePolicySegmentListStatisticsModel( Model ):
   validSegmentLists = Int( help='Number of segment lists in valid state',
                             default=0 )
   invalidSegmentLists = Int( help='Number of segment lists in invalid state',
                              default=0 )
   validSegmentListsWithBackup = Int(
         help='Number of valid segment lists with backup', default=0 )
   def render( self ):
      print "Segment list state:\n"
      print '{:25} {:>10}'.format( "Valid segment lists:", self.validSegmentLists )
      print '{:25} {:>10}'.format( "Invalid segment lists:",
                                   self.invalidSegmentLists )
      print '{:32} {:>10}'.format( "Valid segment lists with backup:",
                                   self.validSegmentListsWithBackup )

class SrTePolicySummaryModel( Model ):
   perEndpointColorStatistics = Submodel( valueType=SrTePolicyStatisticsEntryModel,
                                          help='Per endpoint and color Segment '
                                          'Routing Traffic Engineering Policy '
                                          'statistics' )
   policyStatistics = Submodel( valueType=SrTePolicyStatisticsModel,
                                help='Per endpoint and color and source Segment '
                                'Routing Traffic Engineering Policy statistics' )
   pathgroupStatistics = Submodel( valueType=SrTePolicyPathGroupStatisticsModel,
                                   help='Segment Routing Traffic Engineering Policy '
                                   'path group related Statistics' )
   segmentListStatistics = Submodel( valueType=SrTePolicySegmentListStatisticsModel,
                                     help='Segment Routing Traffic Engineering '
                                     'Policy segment list related statistics' )
   fecOptimizationsSupported = Bool( default=False,
                                     help='Indicates whether this device supports '
                                     'FEC optimizations or not' )
   bindingSidConflictInvalidPathGroupsCount = Int( default=0,
                                                   help='Number of invalid '
                                                   'candidate paths due to Binding '
                                                   'SID conflict with other '
                                                   'applications' )
   bindingSidConflictInactivePoliciesCount = Int( default=0,
                                                  help='Number of inactive policies '
                                                  'due to Binding SID conflict with '
                                                  'other policies' )
   def render( self ):
      # render perEndpointColorStatistics right here
      # and delegate the rendering of other models to specific
      # render functions.
      print "\nPer endpoint, color statistics:\n"
      headings = totalHeader
      table = createTable( headings, tableWidth=120 )
      fmt = Format( justify='left' )
      table.formatColumns( *( ( fmt, ) * len( headings ) ) )
      table.newRow( str( self.perEndpointColorStatistics.activePolicies ),
                    str( self.perEndpointColorStatistics.validPolicies ),
                    str( self.perEndpointColorStatistics.invalidPolicies ),
                    str( self.perEndpointColorStatistics.totalPolicies ) )
      print table.output()

      self.perEndpointColorStatistics.render()
      self.policyStatistics.render()
      self.pathgroupStatistics.render()
      self.segmentListStatistics.render()
      print "\nPath groups invalid due to BSID conflict with other applications:", \
              self.bindingSidConflictInvalidPathGroupsCount
      print "Policies not active due to BSID conflict with other policies:", \
              self.bindingSidConflictInactivePoliciesCount
      boolStr = 'does not support'
      if self.fecOptimizationsSupported:
         boolStr = 'supports'
      print "\nPlatform %s FEC optimizations" % ( boolStr )

class SrTePolicySummaryVrfModel( Model ):
   __revision__ = 2
   vrfs = Dict( keyType=str, valueType=SrTePolicySummaryModel,
                help='Per VRF Segment Routing Traffic Engineering Policy summary' )
   def render( self ):
      for vrfSummary in self.vrfs.values():
         vrfSummary.render()

class SrTePolicyCbfFecOverride( Model ):
   ipVersion = Enum( values=[ 'ipv4', 'ipv6' ], help="IP version" )
   dscp = Int( help="DSCP value" )
   defaultFecId = Int( help="Default FEC ID" )
   overrideFecId = Int( help="Overriding SR-TE Policy FEC ID" )

class SrTePolicyCbfFecModel( Model ):
   overrides = List( valueType=SrTePolicyCbfFecOverride,
                     help="Policy FEC override entry" )

   def cmpKey( self, item ):
      return ( item.ipVersion, item.defaultFecId, item.dscp )

   def render( self ):
      headings = ( "IP Version", "Default FEC ID", "DSCP", "Overriding FEC ID" )
      table = createTable( headings )
      for overrideEntry in sorted( self.overrides, key=self.cmpKey ):
         table.newRow( overrideEntry.ipVersion, "0x%x" % overrideEntry.defaultFecId,
                       overrideEntry.dscp, "0x%x" % overrideEntry.overrideFecId )
      print table.output()

class SrTePolicyCbfFecVrfModel( Model ):
   vrfs = Dict( keyType=str, valueType=SrTePolicyCbfFecModel,
                help='Per VRF Segment Routing Traffic Engineering CBF override'
                'entries' )

   def render( self ):
      for vrfSummary in self.vrfs.values():
         vrfSummary.render()
