# Copyright (c) 2017 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

from operator import attrgetter

import AclCliLib
import AgentDirectory
import AgentCommandRequest
import BasicCli
import CliCommand
import CliExtensions
import CliMatcher
import CliParser
import CliToken
import ConfigMount
import LazyMount
import MultiRangeRule
import ShowCommand
import SharedMem
import Smash
import Tac
import TeCli
import TechSupportCli

from Arnet import IpAddress, IpGenAddr
from ArnetLib import bgpFormatAsn
from CliMode.SrTeColorMapping import SrTeColorDscpModeBase
from CliMode.SrTePolicy import ( SrTeModeBase,
                                 SrTePolicyModeBase,
                                 SrTePolicyPathGroupModeBase
                               )
from CliPlugin.IpGenAddrMatcher import IpGenAddrMatcher
from CliPlugin.TechSupportCli import techSupportKwMatcher, extendedKwMatcher
from CliPlugin.BfdCli import matcherInterval
from CliPlugin.BfdGlobalConfigMode import ( matcherTxRxIntervalMs,
                                            matcherMultiplierVal,
                                            SbfdReflectorDiscExpr,
                                            reflectorAdapter )
from IpLibConsts import DEFAULT_VRF
import SrTePolicyCommonLib
from SrTePolicyCommonLib import tacColor, tacEnlpEnum, srTePolicyStatusPath
from SrTePolicyLib import (
   policySrcToString,
   tacProtocol,
   tacPreference,
   tacWeight,
   tacSegmentListId,
   tacCandidateId,
   tacPolicyId,
   tacPolicyInvalidReason,
   MplsLabel,
   MplsLabelOperation,
   Preference,
   Weight,
)
from SrTePolicyLibCli import getSegmentListVias
from SrTePolicy import (
   tacIndex,
   tacPriority,
   tacStaticSegList,
   Index,
)
from SrTePolicyModel import (
   SrTePolicyBase,
   SrTeSegment,
   SrTePolicyBindingSid,
   SrTePolicyOriginator,
   SrTePolicyPathGroupSegmentList,
   SrTePolicyPathGroup,
   SrTePolicyCandidates,
   SrTePolicyCandidatesModel,
   SrTePolicyCandidatesVrfModel,
   SrTePolicyActive,
   SrTePolicyActiveModel,
   SrTePolicyActiveVrfModel,
   SrTePolicySegmentList,
   SrTePolicySegmentLists,
   SrTePolicyFecTable,
   SrTePolicyFecInfo,
   SrTePolicyFecVia,
   VrfSrTePolicySegmentLists,
   SrTePolicySegmentListCounters,
   SrTePolicySegmentListsCounters,
   VrfSrTePolicySegmentListsCounters,
   SrTePolicySummaryModel,
   SrTePolicySummaryVrfModel,
   SrTePolicyPathGroupStatisticsEntryModel,
   SrTePolicyStatisticsEntryModel,
   SrTePolicyStatisticsModel,
   SrTePolicyPathGroupStatisticsModel,
   SrTePolicySegmentListStatisticsModel,
   SrTePolicyCbfFecVrfModel,
   SrTePolicyCbfFecModel,
   SrTePolicyCbfFecOverride
)
from TunnelCli import (
   TunnelTableIdentifier,
   readMountTunnelTable,
)
from TypeFuture import TacLazyType
from Toggles.SrTePolicyToggleLib import (
   toggleSrTePolicyMetricEnabled,
)

TunnelId = TacLazyType( "Tunnel::TunnelTable::TunnelId" )
DynTunnelIntfId = TacLazyType( "Arnet::DynamicTunnelIntfId" )
TacDscp = TacLazyType( 'Arnet::DscpValue' )
FecIdIntfId = TacLazyType( 'Arnet::FecIdIntfId' )
FecId = TacLazyType( 'Smash::Fib::FecId' )
ConfigSessionType = TacLazyType( "Bfd::SessionType" )
BfdOperState = TacLazyType( "Bfd::OperState" )
SegmentListSharing = TacLazyType( "SrTePolicy::SegmentListSharing" )

srConfig = None
colorDscpMap = None
mplsConfig = None
mplsHwCapability = None
routingHwStatus = None
policyStatus = None
policyValid = None
policyKeyToIdMap = None
allPolicy = None
allSl = None
slStatus = None
slTunnelTable = None
srteForwardingStatus = None
bgpPolicyInput = None
staticPolicyInput = None
policyCounterClear = None
policyCounterTable = None
bfdStatusPeer = None
fecOverrideTable = None
policyNameToKeyMap = None


#--------------------------------------------------------
# Shared configuration tokens
#--------------------------------------------------------
matcherColorValue = CliMatcher.IntegerMatcher( tacColor.min, tacColor.max,
                                               helpdesc='Color value' )
matcherSegmentRouting = CliMatcher.KeywordMatcher( 'segment-routing',
      helpdesc='Segment Routing related information' )
matcherPolicy = CliMatcher.KeywordMatcher( 'policy',
      helpdesc='Show Segment Routing Traffic Engineering policies' )
matcherTrafficEngineering = CliMatcher.KeywordMatcher( 'traffic-engineering',
      helpdesc='Traffic Engineering related information' )
matcherCbf = CliMatcher.KeywordMatcher( 'cbf', helpdesc='Class based forwarding' )
matcherBgp = CliMatcher.KeywordMatcher( 'bgp',
      helpdesc='Show policies learnt via BGP' )
matcherColor = CliMatcher.KeywordMatcher( 'color',
      helpdesc='Color' )
matcherEndpoint = CliMatcher.KeywordMatcher( 'endpoint',
      helpdesc='Show policies for a given endpoint' )
matcherId = CliMatcher.KeywordMatcher( 'id',
      helpdesc='Segment list ID' )
matcherSegmentList = CliMatcher.KeywordMatcher( 'segment-list',
      helpdesc='Show Segment Routing Traffic Engineering policy segment lists' )
matcherSource = CliMatcher.KeywordMatcher( 'source',
      helpdesc='Show policies for a given source' )
matcherStatic = CliMatcher.KeywordMatcher( 'static',
      helpdesc='Show policies configured statically' )
#--------------------------------------------------------
# SrTePolicy Agent running ?
#--------------------------------------------------------
def isSrTePolicyAgentRunning( mode ):
   return AgentDirectory.agentIsRunning( mode.entityManager.sysname(), 'SrTePolicy' )

#--------------------------------------------------------
# Segment-routing currently only support mpls dataplane
#--------------------------------------------------------
def mplsSupportedGuard( mode, token ):
   if mplsHwCapability.mplsSupported:
      return None
   else:
      return CliParser.guardNotThisPlatform

#---------------------------------
# [ no | default ] segment-routing
#---------------------------------
def delSegmentRouting():
   """Remove all policy configuration on 'no segment-routing'
      or 'no traffic-engineering'
   """
   srConfig.policy.clear()
   colorDscpMap.colorToDscp.clear()
   srConfig.enabled = False

class CfgSegmentRoutingCmd( CliCommand.CliCommandClass ):
   syntax = 'segment-routing'
   noOrDefaultSyntax = syntax
   data = {
      'segment-routing': CliCommand.guardedKeyword( 'segment-routing',
                                                    'Segment Routing configuration',
                                                    guard=mplsSupportedGuard )
   }

   @staticmethod
   def handler( mode, args ):
      childMode = mode.childMode( SrTeMode )
      mode.session_.gotoChildMode( childMode )
      srConfig.enabled = True

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      delSegmentRouting()

TeCli.RouterGlobalTeMode.addCommandClass( CfgSegmentRoutingCmd )

#---------------------------------------------------------------
# Remove all segment-routing configs when the parent is removed
# i.e., "no router traffic-engineering" in config mode.
#---------------------------------------------------------------
class TeSubMode( TeCli.TeDependentBase ):
   def setDefault( self ):
      delSegmentRouting()

#-------------------------------------------------------------------------------
# [ no | default ] color-dscp-mappings
#-------------------------------------------------------------------------------
class RouterSrTeColorDscpMappingsMode( SrTeColorDscpModeBase,
                                       BasicCli.ConfigModeBase ):
   name = "Color / DSCP mapping configuration"
   modeParseTree = CliParser.ModeParseTree()

   def __init__( self, parent, session ):
      SrTeColorDscpModeBase.__init__( self, None )
      BasicCli.ConfigModeBase.__init__( self, parent, session )

def cbfToggleEnabledGuard( mode, token ):
   if routingHwStatus.srTeCbfSupported:
      return None
   return CliParser.guardNotThisPlatform

class CfgColorDscpMappingsCmd( CliCommand.CliCommandClass ):
   syntax = 'color-dscp-mappings'
   noOrDefaultSyntax = syntax
   data = {
      'color-dscp-mappings': CliCommand.guardedKeyword(
         'color-dscp-mappings',
         RouterSrTeColorDscpMappingsMode.name,
         guard=cbfToggleEnabledGuard ),
   }

   @staticmethod
   def handler( mode, args ):
      childMode = mode.childMode( RouterSrTeColorDscpMappingsMode )
      mode.session_.gotoChildMode( childMode )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      colorDscpMap.colorToDscp.clear()

#-------------------------------------------------------------------------------
# [ no | default ] color COLOR ( dscp DSCP | default )
# in te-sr color-dscp-mappings mode
#-------------------------------------------------------------------------------
def dscpAclNames( mode ):
   return { k: v[ 1 ] for k, v in AclCliLib.dscpAclNames.items() }

def dscpRange():
   return TacDscp.min, TacDscp.max

class CfgColorDscpMappingEntryCmd( CliCommand.CliCommandClass ):
   syntax = 'color COLOR ( dscp ( { DSCP | DSCPACL } | DSCPS ) ) | default'
   noOrDefaultSyntax = ( 'color COLOR [ ( dscp ( { DSCP | DSCPACL } | DSCPS ) ) '
                         ' | default ]' )
   data = {
      'color': "Configure the mapping mapping for a color",
      'COLOR': matcherColorValue,
      'default': 'Associate the specified color with any unmapped DSCP values',
      'dscp': "Map the specified color to specific DSCP values",
      'DSCP': CliMatcher.IntegerMatcher( *dscpRange(),
                                         helpdesc='DSCP value' ),
      'DSCPACL': CliMatcher.DynamicKeywordMatcher( dscpAclNames ),
      'DSCPS': CliCommand.Node(
                  matcher=MultiRangeRule.MultiRangeMatcher(
                               rangeFn=dscpRange,
                               noSingletons=False,
                               helpdesc='Differentiated Services Code Point (DSCP) '
                                        'numeric values or ranges',
                  priority=CliParser.PRIO_HIGH ) ),
   }

   @staticmethod
   def handler( mode, args ):
      if 'default' in args:
         color = args[ 'COLOR' ]
         if color in colorDscpMap.colorToDscp:
            # Clean up any state for specific DSCPs
            if not colorDscpMap.colorToDscp[ color ].defaultDscps:
               colorDscpMap.colorToDscp[ color ].dscp.clear()
               colorDscpMap.colorToDscp[ color ].cliSaveUseSymbolic_i.clear()
         colorToDscp = colorDscpMap.newColorToDscp( color )
         colorToDscp.defaultDscps = True
         return
      dscpValues = args.get( 'DSCP', [] )
      dscpValues += args.get( 'DSCPACL', [] )
      dscpValues += args[ 'DSCPS' ].values() if 'DSCPS' in args else []
      colorToDscp = colorDscpMap.newColorToDscp( args[ 'COLOR' ] )
      # Clean up any state for default DSCPs.
      if colorToDscp.defaultDscps:
         colorToDscp.defaultDscps = False
      for value in dscpValues:
         dscp, dscpUseName = AclCliLib.dscpValueFromCli( mode, value )
         colorToDscp.dscp[ dscp ] = True
         colorToDscp.cliSaveUseSymbolic_i[ dscp ] = dscpUseName

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      color = args[ 'COLOR' ]
      if color not in colorDscpMap.colorToDscp:
         return
      if 'default' in args:
         # Removing default DSCPs setting.  If set delete the color, otherwise
         # do nothing.
         if colorDscpMap.colorToDscp[ color ].defaultDscps:
            del colorDscpMap.colorToDscp[ color ]
         return
      colorToDscp = colorDscpMap.colorToDscp[ color ]
      dscpValues = args.get( 'DSCP', [] )
      dscpValues += args.get( 'DSCPACL', [] )
      dscpValues += args[ 'DSCPS' ].values() if 'DSCPS' in args else []
      if dscpValues:
         for value in dscpValues:
            dscp, _ = AclCliLib.dscpValueFromCli( mode, value )
            del colorToDscp.dscp[ dscp ]
            del colorToDscp.cliSaveUseSymbolic_i[ dscp ]
         if not colorToDscp.dscp and \
            not colorDscpMap.colorToDscp[ color ].defaultDscps:
            del colorDscpMap.colorToDscp[ color ]
      else:
         del colorDscpMap.colorToDscp[ color ]

RouterSrTeColorDscpMappingsMode.addCommandClass( CfgColorDscpMappingEntryCmd )

#-------------------------------------------------------------------------------
# [  no | default ] policy endpoint ADDR color COLOR
#-------------------------------------------------------------------------------
class CfgPolicyCmd( CliCommand.CliCommandClass ):
   syntax = 'policy endpoint ADDR color COLOR'
   noOrDefaultSyntax = syntax
   data = {
         'policy': 'Segment Routing policy configuration',
         'endpoint': 'Endpoint',
         'ADDR': IpGenAddrMatcher( 'IPv4 or IPv6 address' ),
         'color': matcherColor,
         'COLOR': matcherColorValue,
   }

   @staticmethod
   def handler( mode, args ):
      endPoint = args[ 'ADDR' ]
      color = args[ 'COLOR' ]
      key = SrTePolicyCommonLib.PolicyKey( endPoint, color )
      policy = srConfig.policy.get( key )
      if policy is None:
         policy = srConfig.policy.newMember( key )

      childMode = mode.childMode( SrTePolicyMode, policy=policy,
                                  endPoint=endPoint, color=color )
      mode.session_.gotoChildMode( childMode )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      endPoint = args[ 'ADDR' ]
      color = args[ 'COLOR' ]
      key = SrTePolicyCommonLib.PolicyKey( endPoint, color )
      # Remove the mapping for the policyName as this policy is going
      # away.
      if key in srConfig.policy:
         policyCfg = srConfig.policy[ key ]
         policyName = policyCfg.policyName
         if policyName in policyNameToKeyMap.policy:
            del policyNameToKeyMap.policy[ policyName ]
      # Remove the policy config.
      del srConfig.policy[ key ]

class SrTeMode( SrTeModeBase, BasicCli.ConfigModeBase ):
   name = "Segment Routing Configuration"
   modeParseTree = CliParser.ModeParseTree()

   def __init__( self, parent, session ):
      SrTeModeBase.__init__( self, None )
      BasicCli.ConfigModeBase.__init__( self, parent, session )

SrTeMode.addCommandClass( CfgPolicyCmd )
SrTeMode.addCommandClass( CfgColorDscpMappingsCmd )

class CfgColoredTunnelRibCmd( CliCommand.CliCommandClass ):
   syntax = 'rib system-colored-tunnel-rib'
   noOrDefaultSyntax = syntax
   data = {
      'rib': 'Include policies in a resolution RIB',
      'system-colored-tunnel-rib': 'The system colored tunnel RIB',
   }

   @staticmethod
   def handler( mode, args ):
      srConfig.populateColoredTunnelRib = True

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      srConfig.populateColoredTunnelRib = False

matcherCostValue = CliMatcher.IntegerMatcher( 0, 255,
      helpdesc='The IGP cost value' )
matcherIgpCost = CliMatcher.KeywordMatcher( 'igp-cost',
      helpdesc='IGP metric and preference values for next'
               'hops resolving via the policy' )
matcherIgpPreference = CliMatcher.KeywordMatcher( 'preference',
      helpdesc='IGP preference for the policy' )

class CfgIgpCostPreference( CliCommand.CliCommandClass ):
   syntax = 'igp-cost preference COST'
   noOrDefaultSyntax = 'igp-cost preference ...'
   data = {
      'igp-cost': matcherIgpCost,
      'preference': matcherIgpPreference,
      'COST': matcherCostValue,
   }

   @staticmethod
   def handler( mode, args ):
      srConfig.igpPrefCost = args.get( 'COST', srConfig.defaultIgpPrefCost )
   noOrDefaultHandler = handler

SrTeMode.addCommandClass( CfgColoredTunnelRibCmd )
SrTeMode.addCommandClass( CfgIgpCostPreference )

class SrTePolicyMode( SrTePolicyModeBase, BasicCli.ConfigModeBase ):
   name = "Segment Routing Policy Configuration"
   modeParseTree = CliParser.ModeParseTree()

   def __init__( self, parent, session, policy, endPoint, color ):
      SrTePolicyModeBase.__init__( self, ( endPoint, color ) )
      BasicCli.ConfigModeBase.__init__( self, parent, session )
      self.policy = policy
      self.endPoint = endPoint
      self.color = color


#-------------------------------------------------------
# [  no | default ] binding-sid SEGMENT_ID
#-------------------------------------------------------
class CfgBindingSidCmd( CliCommand.CliCommandClass ):
   syntax = 'binding-sid SEGMENT_ID'
   noOrDefaultSyntax = 'binding-sid [ SEGMENT_ID ]'
   data = {
         'binding-sid': 'Binding segment identifier',
         'SEGMENT_ID': CliMatcher.IntegerMatcher(
                          MplsLabel.unassignedMin,
                          MplsLabel.max,
                          helpdesc='Value of the MPLS label' ),
   }

   @staticmethod
   def handler( mode, args ):
      bsidVal = args[ 'SEGMENT_ID' ]
      if mode.policy.bindingSid == bsidVal:
         return
      # Warn the user if the label is not in SRLB range.
      srlbRange = mplsConfig.labelRange[ 'srlb' ]
      if not srlbRange.base <= bsidVal < ( srlbRange.base + srlbRange.size ):
         mode.addWarning( 'Binding SID is out of SRLB range' )
      # Set the Binding SID to the policy.
      mode.policy.bindingSid = MplsLabel( bsidVal )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      if mode.policy.bindingSid == MplsLabel():
         return
      # Initialize the binding SID in the policy to default.
      mode.policy.bindingSid = MplsLabel()

SrTePolicyMode.addCommandClass( CfgBindingSidCmd )

policyNamePattern = r'[^\s]*'
policyNameMatcher = CliMatcher.PatternMatcher(
      pattern=policyNamePattern,
      helpdesc='Policy Name',
      helpname='WORD' )

#------------------------------------------------------
# [ no | default ] name POLICY_NAME
#------------------------------------------------------
class PolicyNameCmd( CliCommand.CliCommandClass ):
   syntax = 'name POLICY_NAME'
   noOrDefaultSyntax = 'name'
   data = {
      'name' : 'Policy Name',
      'POLICY_NAME' : policyNameMatcher,
   }

   @staticmethod
   def handler( mode, args ):
      policyName = args[ 'POLICY_NAME' ]
      policyKey = mode.policy.key

      if mode.policy.policyName == policyName:
         return

      if mode.policy.policyName != '':
         # The policy name is being modified. Remove the mapping for the
         # existing name.
         tacPolicyName = Tac.newInstance( "SrTePolicy::PolicyName",
                                          mode.policy.policyName )
         del policyNameToKeyMap.policy[ tacPolicyName ]

      tacPolicyName = Tac.newInstance( "SrTePolicy::PolicyName", policyName )
      if tacPolicyName in policyNameToKeyMap.policy:
         policyKey = policyNameToKeyMap.policy[ tacPolicyName ]
         mode.addError( "The policy name: {} has already been configured for "
                        "the policy with endpoint: {} and color: {}".format(
                           policyName, policyKey.endpoint.stringValue,
                           str( policyKey.color ) ) )
         return
      policyNameToKeyMap.policy[ tacPolicyName ] = policyKey
      mode.policy.policyName = tacPolicyName

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      if mode.policy.policyName == '':
         return
      else:
         tacPolicyName = Tac.newInstance( "SrTePolicy::PolicyName",
                                          mode.policy.policyName )
         del policyNameToKeyMap.policy[ tacPolicyName ]
      # There is no policyName configured by default
      mode.policy.policyName = Tac.newInstance( 'SrTePolicy::PolicyName', '' )

SrTePolicyMode.addCommandClass( PolicyNameCmd )

#-------------------------------------------------------
# [ no | default ] description <description>
#-------------------------------------------------------

class PolicyDescriptionCmd( CliCommand.CliCommandClass ):
   syntax = 'description DESC'
   noOrDefaultSyntax = 'description ...'
   data = {
      'description' : 'Policy Description',
      'DESC' : CliMatcher.StringMatcher( helpname='LINE',
               helpdesc='Description for the policy' )
   }

   @staticmethod
   def handler( mode, args ):
      mode.policy.policyDescription = args[ 'DESC' ]

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      mode.policy.policyDescription = ''

SrTePolicyMode.addCommandClass( PolicyDescriptionCmd )

#-------------------------------------------------------
# [ no | default ] path-group preference PREF
#-------------------------------------------------------
class CfgPathGroupCmd( CliCommand.CliCommandClass ):
   syntax = 'path-group preference PREF'
   noOrDefaultSyntax = syntax
   data = {
         'path-group': 'Candidate path group',
         'preference': 'Path group preference',
         'PREF': CliMatcher.IntegerMatcher( tacPreference.min,
                                            tacPreference.max,
                                            helpdesc='Preference value' ),
   }

   @staticmethod
   def handler( mode, args ):
      pvalue = args[ 'PREF' ]
      pref = Preference( pvalue )
      path = mode.policy.candidatePath.get( pref )
      if path is None:
         path = mode.policy.candidatePath.newMember( pref )

      childMode = mode.childMode( SrTePolicyPathGroupMode,
                                  endPoint=mode.endPoint, color=mode.color,
                                  preference=pvalue, path=path )
      mode.session_.gotoChildMode( childMode )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      pref = Preference( args[ 'PREF' ] )
      del mode.policy.candidatePath[ pref ]

SrTePolicyMode.addCommandClass( CfgPathGroupCmd )

matcherMetricValue = CliMatcher.IntegerMatcher( 0, 255,
      helpdesc='IGP metric value' )
matcherIgpMetric = CliMatcher.KeywordMatcher( 'metric',
      helpdesc='IGP metric for the policy' )

#---------------------------------------------------------
# [ no | default ] igp-cost preference COST metric METRIC
#---------------------------------------------------------
class CfgIgpCostMetricCmd( CliCommand.CliCommandClass ):
   syntax = 'igp-cost { ( preference COST ) | ( metric METRIC ) }'
   noOrDefaultSyntax = 'igp-cost ...'
   data = {
         'igp-cost': matcherIgpCost,
         'preference': CliCommand.Node( matcher=matcherIgpPreference, maxMatches=1 ),
         'COST': CliCommand.Node( matcherCostValue, maxMatches=1 ),
         'metric': CliCommand.Node( matcher=matcherIgpMetric, maxMatches=1 ),
         'METRIC': CliCommand.Node( matcher=matcherMetricValue, maxMatches=1 ),
        }

   @staticmethod
   def handler( mode, args ):
      metric = args.get( 'METRIC', mode.policy.costMetric.defaultIgpMetric )

      cost = args.get( 'COST' )
      if cost is None:
         cost = 0
         costSet = False
      else:
         costSet = True

      mode.policy.costMetric = Tac.Value( "SrTePolicy::PolicyCostMetric", cost=cost,
                                           metric=metric, costSet=costSet )

   noOrDefaultHandler = handler

if toggleSrTePolicyMetricEnabled():
   SrTePolicyMode.addCommandClass( CfgIgpCostMetricCmd )

#-------------------------------------------------------------------
# [no|default] sbfd remote-discriminator DISC
#                   [ interval <INTERVAL> multiplier <MULTIPLIER> ]
#-------------------------------------------------------------------
def PolicySbfdConfig( disc, interval, multiplier, disIsU32 ):
   return Tac.Value( 'SrTePolicy::Sbfd::PolicySbfdConfig', disc, interval, 
                                                      multiplier, disIsU32 )

def sbfdHandler( mode, args ):
   multDef = Tac.Type( 'Bfd::BfdMultiplier' ).defval
   interval = args.get( 'INT', 0 )
   multiplier = args.get( 'MULT', multDef )
   disc, isU32 = args[ 'DISC' ]
   mode.policy.sbfdConfig = PolicySbfdConfig( disc, interval, multiplier, isU32 )

class CfgSbfdCmd( CliCommand.CliCommandClass ):
   syntax = 'sbfd remote-discriminator DISC [ interval INT multiplier MULT ]'
   noOrDefaultSyntax = 'sbfd remote-discriminator ...'
   data = {
         'sbfd': 'Seamless BFD',
         'remote-discriminator': 'The remote discriminator',
         'DISC': SbfdReflectorDiscExpr,
         'interval': matcherInterval,
         'INT': matcherTxRxIntervalMs,
         'multiplier': 'Sets the SBFD multiplier',
         'MULT': matcherMultiplierVal,
         }

   adapter = reflectorAdapter

   handler = sbfdHandler
   noOrDefaultHandler = handler

SrTePolicyMode.addCommandClass( CfgSbfdCmd )

class SrTePolicyPathGroupMode( SrTePolicyPathGroupModeBase,
                               BasicCli.ConfigModeBase ):
   name = "Segment Routing Policy Path Group Configuration"
   modeParseTree = CliParser.ModeParseTree()

   def __init__( self, parent, session, endPoint, color, preference, path ):
      SrTePolicyPathGroupModeBase.__init__( self, ( endPoint, color, preference ) )
      BasicCli.ConfigModeBase.__init__( self, parent, session )
      self.path = path

#-------------------------------------------------------------------
# [ no | default ] explicit-null ( ipv4 | ipv6 | ipv4 ipv6 | none )
#-------------------------------------------------------------------
class CfgEnlpCmd( CliCommand.CliCommandClass ):
   syntax = '''explicit-null [ ipv4 | ipv6 | ( ipv4 ipv6 ) | none ]'''
   noOrDefaultSyntax = syntax
   data = {
         'explicit-null': 'Explicit null label policy',
         'ipv4': 'Push IPv4 explicit null label for IP packet',
         'ipv6': 'Push IPv6 explicit null label for IPv6 packet',
         'none': 'No explicit null label be pushed'
   }

   @staticmethod
   def handler( mode, args ):
      if 'ipv4' in args:
         if 'ipv6' in args:
            mode.path.enlp = tacEnlpEnum.ipv4AndIpv6
         else:
            mode.path.enlp = tacEnlpEnum.ipv4
      elif 'ipv6' in args:
         mode.path.enlp = tacEnlpEnum.ipv6
      elif 'none' in args:
         mode.path.enlp = tacEnlpEnum.none

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      mode.path.enlp = tacEnlpEnum.localPolicy

SrTePolicyPathGroupMode.addCommandClass( CfgEnlpCmd )

def maxLabelStackSize():
   """
   From the CLI allow as many labels as Arnet::MplsLabelOperation can hold.
   We don't check platform limits in the CLI because what matters is the resolved
   segment list as that is what gets programmed in hardware. The label stack
   can expand or shrink based on the resolution.
   Arnet::MplsLabelOperation can hold one more than the deepest label stack we can
   impose on any of our platforms as it's possible for the label stack to shrink if
   the top label was popped as result of resolution.
   """
   MplsStackEntryIndex = Tac.Type( 'Arnet::MplsStackEntryIndex' )
   return MplsStackEntryIndex.max + 1

#------------------------------------------------------------------------------------
# [ no | default ] segment-list label-stack STACK [ weight WEIGHT ] [ index INDEX ]
#------------------------------------------------------------------------------------
class CfgSegmentListCmd( CliCommand.CliCommandClass ):
   syntax = '''segment-list label-stack { STACK }
               [ ( weight WEIGHT [ index INDEX   ] )
               | ( index INDEX   [ weight WEIGHT ] ) ]'''
   noOrDefaultSyntax = syntax
   data = {
         'segment-list': 'Segment list for path group',
         'label-stack': 'Segment list as MPLS Label Stack',
         'weight': 'Weight',
         'index': 'Index',
         'WEIGHT': CliMatcher.IntegerMatcher( tacWeight.min, tacWeight.max,
                                              helpdesc='Weight value' ),
         'INDEX': CliMatcher.IntegerMatcher( tacIndex.min + 1, tacIndex.max,
                                             helpdesc='Index value' ),
         'STACK': CliCommand.Node(
                     matcher=CliMatcher.IntegerMatcher( MplsLabel.min,
                                                        MplsLabel.max,
                                                        helpdesc='MPLS label' ),
                     maxMatches=maxLabelStackSize() ),
         }

   @staticmethod
   def _segListStrToLabelStack( segments ):
      newSegList = MplsLabelOperation()
      index = 0
      for index, segment in enumerate( segments ):
         newSegList.labelStackIs( index, MplsLabel( int( segment ) ) )
      newSegList.stackSize = index + 1
      return newSegList

   @staticmethod
   def handler( mode, args ):
      weight = args.get( 'WEIGHT' )
      index = args.get( 'INDEX' )
      segList = CfgSegmentListCmd._segListStrToLabelStack( args[ 'STACK' ] )
      for sl, wIdx in mode.path.staticSegmentList.items():
         if sl != segList:
            if index == wIdx.index:
               mode.addError( 'Index %d is already assigned to another segment-list'
                              % index )
               return
         elif wIdx.index and index != wIdx.index:
            mode.addError( 'Index mismatch, expected index %d' % wIdx.index )
            return
      slWeightIndex = tacStaticSegList( Weight( weight ), Index( index ) )
      mode.path.staticSegmentList[ segList ] = slWeightIndex

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      segList = CfgSegmentListCmd._segListStrToLabelStack( args[ 'STACK' ] )
      del mode.path.staticSegmentList[ segList ]

SrTePolicyPathGroupMode.addCommandClass( CfgSegmentListCmd )

def isSlValidAndGetTunEntry( slId, sl ):
   '''
   Return a tuple ( True, tunnelEntry ) if the @sl with id @slId is valid; otherwise
   return ( False, None )
   '''
   tunnelEntry = None
   slResolved = None
   valid = False

   if sl is None or sl.size == 0:
      return ( valid, tunnelEntry )
   else:
      # The segment list is automatically invalid if it has no segments
      slResolved = slStatus.status.get( slId )

   if slResolved == 'success':
      # Be permissive of the tunnel entry not being present yet (race condition).
      # We'll treat the SL as invalid if we can't find the tunnel yet.
      tunnelId = tacSegmentListId( slId ).getTunnelId()
      tunnelEntry = slTunnelTable.entry.get( tunnelId, None )

   valid = tunnelEntry is not None
   return ( valid, tunnelEntry )

srTeTunnelCountersHook = CliExtensions.CliHook()

def getCountersForTunnelEntry( tunnelEntry ):
   '''
   Given a tunnelEntry return packet & byte counters if available
   '''
   result = ( None, None )
   # Try to use CLI hook if available
   for hook in srTeTunnelCountersHook.extensions():
      result = hook( tunnelEntry.tunnelId )
      break

   return result

def getCountersForPolicy( policyKey ):
   '''Given a policy key, return packet & byte counters if available'''
   result = ( None, None )
   policyCounter = policyCounterTable.policyCounter.get( policyKey )
   policySnapshot = policyCounterTable.policySnapshot.get( policyKey )
   if policyCounter and policySnapshot:
      pkts = policyCounter.pkts - policySnapshot.pkts
      octets = policyCounter.octets - policySnapshot.octets
      result = ( pkts, octets )
   elif policyCounter:
      result = ( policyCounter.pkts, policyCounter.octets )
   return result

# ----------------------------------------------------------------
# show traffic-engineering segment-routing policy active ...
# ----------------------------------------------------------------
def getSourceFilter( args ):
   if 'static' in args:
      return [ tacProtocol.staticProtocol ]
   if 'bgp' in args:
      return [ tacProtocol.bgpProtocol ]
   else:
      return [ tacProtocol.staticProtocol, tacProtocol.bgpProtocol ]

class ShowSrTePolicyActiveCmd( ShowCommand.ShowCliCommandClass ):
   syntax = '''show traffic-engineering segment-routing policy
               active [ source ( bgp | static ) ] [ endpoint ADDR color COLOR ]'''
   data = {
      'traffic-engineering': matcherTrafficEngineering,
      'segment-routing': matcherSegmentRouting,
      'policy': matcherPolicy,
      'active': 'Show active Segment Routing Traffic Engineering policies',
      'source': matcherSource,
      'bgp': matcherBgp,
      'static': matcherStatic,
      'endpoint': matcherEndpoint,
      'ADDR': IpGenAddrMatcher( 'IPv4 or IPv6 address' ),
      'color': matcherColor,
      'COLOR': matcherColorValue,
   }
   cliModel = SrTePolicyActiveVrfModel

   @staticmethod
   def handler( mode, args ):
      ep = args.get( 'ADDR' )
      color = args.get( 'COLOR' )
      sourceFilter = getSourceFilter( args )
      vrfModel = SrTePolicyActiveVrfModel()
      active = SrTePolicyActiveModel()
      active.policies = populatePolicies( ep, color, [ 'active' ], sourceFilter )
      vrfModel.vrfs[ DEFAULT_VRF ] = active
      return vrfModel

BasicCli.addShowCommandClass( ShowSrTePolicyActiveCmd )

# -----------------------------------------------------------------------
# show traffic-engineering segment-routing policy [ summary ]
# -----------------------------------------------------------------------
def populatePolicyStatistics():
   sourceDict = { tacProtocol.staticProtocol : 0,
                  tacProtocol.bgpProtocol : 0 }

   # Terminology:
   # We are obtaining counters for:
   # Each ( endpoint, color ) -> Per endpoint, color statistics
   # each ( endpoint, color, source ) -> Per endpoint, color, source statistics
   # and each candidate path -> Candidate Path Statistics
   # Note that for 'Per endpoint, color statistics' we don't care about the source
   # etc. we are only counting how many (E, C) are in a given state. Similarly,
   # for 'Per Source' Statistics, we are only counting per state counters of
   # policies within a given source and don't care about preferences etc.

   # PathGroup counters.
   activePathGroupCnt = { tacProtocol.staticProtocol : 0,
                          tacProtocol.bgpProtocol : 0 }
   validPathGroupCnt = { tacProtocol.staticProtocol : 0,
                            tacProtocol.bgpProtocol : 0 }
   invalidPathGroupCnt = { tacProtocol.staticProtocol : 0,
                           tacProtocol.bgpProtocol : 0 }

   # Per-Source Policy counters
   activePolicyCnt = { tacProtocol.staticProtocol : 0,
                       tacProtocol.bgpProtocol : 0 }
   validPolicyCnt = { tacProtocol.staticProtocol : 0,
                         tacProtocol.bgpProtocol : 0 }
   invalidPolicyCnt = { tacProtocol.staticProtocol : 0,
                        tacProtocol.bgpProtocol : 0 }

   # Policy ( E,C ) counters
   totalValidPolicies = 0
   totalInValidPolicies = 0
   totalActivePolicies = 0

   # Walk through each policy (endpoint, color) in policyKeyToIdMap
   for key, policy in policyKeyToIdMap.policy.iteritems():
      # While we iterate through the policies, for each endpoint, color (E,C)
      # Mark the ones seen as valid or invalid using the following dicts
      sourceInvalid = { }
      sourceValid = { tacProtocol.staticProtocol : False,
                      tacProtocol.bgpProtocol : False }
      active = policyStatus.status.get( key )
      validCandidates = policyValid.policy.get( key )
      if validCandidates is None:
         totalInValidPolicies += 1
      elif not active:
         totalValidPolicies += 1

      # For each Endpoint, Color, now walk through the composite candidate paths.
      for candidateId in policy.candidate:
         # For any protocol present in list of candidates for this policy,
         # we start with assuming it is an invalid policy unless proven wrong
         # by finding a match in validCandidates
         if not sourceInvalid.get( candidateId.protocol ):
            # We start with an assumption that this candidate path is invalid
            # and wait for it to be proven False later on.
            sourceInvalid.update( { candidateId.protocol : True } )

         pie = allPolicy.policyIdEntry.get( tacPolicyId( key, candidateId ) )
         if validCandidates and pie:
            priority = tacPriority( pie.preference, pie.id.candidateId.protocol,
                              pie.originator, pie.id.candidateId.discriminator )
            if priority in validCandidates.candidate:
               # This is a valid candidate path for this policy
               sourceInvalid[ pie.id.candidateId.protocol ] = False
               if not active or \
                  active.candidateId.protocol != pie.id.candidateId.protocol:
                  # Either this policy does not have an active candidate path Or,
                  # the active candidate path is from a different source compared to
                  # the source of the candidate path we have at hand now.
                  sourceValid[ pie.id.candidateId.protocol ] = True
               if active and active.candidateId == pie.id.candidateId:
                  # This candidate path is the active one.
                  activePathGroupCnt[ active.candidateId.protocol ] += 1
                  activePolicyCnt[ active.candidateId.protocol ] += 1
                  totalActivePolicies += 1
               else:
                  validPathGroupCnt[ pie.id.candidateId.protocol ] += 1
               continue

         # This is an invalid candidate path
         invalidPathGroupCnt[ candidateId.protocol ] += 1

      for source in sourceInvalid:
         if sourceInvalid[ source ]:
            invalidPolicyCnt[ source ] += 1
      for source in sourceValid:
         if sourceValid[ source ]:
            validPolicyCnt[ source ] += 1

   pathGroupNumbers = SrTePolicyPathGroupStatisticsModel()
   pgTotalEntry = SrTePolicyPathGroupStatisticsEntryModel()
   policyNumbers = SrTePolicyStatisticsModel()
   globalPolicyEntry = SrTePolicyStatisticsEntryModel()
   totalPolicyEntry = SrTePolicyStatisticsEntryModel()
   for source in sourceDict:
      pgEntry = SrTePolicyPathGroupStatisticsEntryModel()
      policyEntry = SrTePolicyStatisticsEntryModel()

      # ACTIVE
      pgEntry.activePathGroups = activePathGroupCnt[ source ]
      pgTotalEntry.activePathGroups += pgEntry.activePathGroups
      policyEntry.activePolicies = activePolicyCnt[ source ]

      # VALID
      pgEntry.validPathGroups = validPathGroupCnt[ source ]
      pgTotalEntry.validPathGroups += pgEntry.validPathGroups
      policyEntry.validPolicies = validPolicyCnt[ source ]

      # INVALID
      pgEntry.invalidPathGroups = invalidPathGroupCnt[ source ]
      pgTotalEntry.invalidPathGroups += pgEntry.invalidPathGroups
      policyEntry.invalidPolicies = invalidPolicyCnt[ source ]

      # TOTAL per Source
      pgEntry.totalPathGroups = pgEntry.activePathGroups + \
                                pgEntry.validPathGroups + \
                                pgEntry.invalidPathGroups
      policyEntry.totalPolicies = policyEntry.activePolicies + \
                                  policyEntry.validPolicies + \
                                  policyEntry.invalidPolicies
      totalPolicyEntry.activePolicies += policyEntry.activePolicies
      totalPolicyEntry.validPolicies += policyEntry.validPolicies
      totalPolicyEntry.invalidPolicies += policyEntry.invalidPolicies
      totalPolicyEntry.totalPolicies += policyEntry.totalPolicies

      pgTotalEntry.totalPathGroups += pgEntry.totalPathGroups

      pathGroupNumbers.sources[ policySrcToString( source ) ] = pgEntry
      policyNumbers.sources[ policySrcToString( source ) ] = policyEntry

   globalPolicyEntry.activePolicies = totalActivePolicies
   globalPolicyEntry.validPolicies = totalValidPolicies
   globalPolicyEntry.invalidPolicies = totalInValidPolicies
   globalPolicyEntry.totalPolicies = totalActivePolicies + \
                                    totalValidPolicies + totalInValidPolicies
   policyNumbers.total = totalPolicyEntry
   pathGroupNumbers.total = pgTotalEntry

   return ( pathGroupNumbers, policyNumbers, globalPolicyEntry )

def getSbfdPeerStatus( slId, sbfdConfig ):
   peerStatus = None
   tunnelId = tacSegmentListId( slId ).getTunnelId()
   if sbfdConfig and sbfdConfig.sbfdRemoteDiscriminator:
      remoteDisc = sbfdConfig.sbfdRemoteDiscriminator
      peerAddr = IpGenAddr( IpAddress( remoteDisc ).stringValue )
      sbfdPeer = Tac.Value( 'Bfd::Peer', peerAddr, DEFAULT_VRF )
      sbfdPeer.tunnelId = tunnelId
      sbfdPeer.type = ConfigSessionType.sbfdInitiator
      peerStatus = bfdStatusPeer.peerStatus.get( sbfdPeer )
   return peerStatus

def populateSegmentListStatistics():
   slStats = SrTePolicySegmentListStatisticsModel()
   mayValidSegmentLists = len( slStatus.status )
   for slId in slStatus.status:
      key = None
      if slId in allSl.segmentList:
         key = allSl.segmentList[ slId ].policyId.policyKey
      policy = srConfig.policy.get( key ) if key else None
      sbfdConfig = policy.sbfdConfig if policy else None
      peerStatus = getSbfdPeerStatus( slId, sbfdConfig )
      if ( peerStatus and
           peerStatus.clientStatus.status == BfdOperState.down ):
         mayValidSegmentLists = mayValidSegmentLists - 1
   slStats.validSegmentLists = mayValidSegmentLists
   slStats.invalidSegmentLists = len( allSl.segmentList ) - slStats.validSegmentLists
   slIds = allSl.segmentList.iterkeys()
   slStats.validSegmentListsWithBackup = 0
   for slId in slIds:
      sl = allSl.segmentList.get( slId )
      if sl is None:
         # The requested SL doesn't exist, so skip it.
         continue
      valid, tunnelEntry = isSlValidAndGetTunEntry( slId, sl )
      backupVias = []
      if valid:
         _, backupVias = getSegmentListVias( tunnelEntry )
      if backupVias:
         slStats.validSegmentListsWithBackup += 1
   return slStats

def countBindingSidConflictInvalidPathGroups():
   totalConflictedPathGroups = 0
   for allCandidates in policyKeyToIdMap.policy.values():
      for cInfo in allCandidates.candidate.values():
         bsidReason = cInfo.reason
         if bsidReason == 'policyBsidInConflictWithOtherApp':
            totalConflictedPathGroups += 1
   return totalConflictedPathGroups

def countBindingSidConflictInactivePolicies():
   totalInactivePolicies = 0
   for key in policyValid.policy:
      active = policyStatus.status.get( key )
      if not active:
         totalInactivePolicies += 1
   return totalInactivePolicies

class ShowSrTePolicySummaryCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show traffic-engineering segment-routing policy summary'
   data = {
      'traffic-engineering': matcherTrafficEngineering,
      'segment-routing': matcherSegmentRouting,
      'policy': matcherPolicy,
      'summary': 'Show summary of Segment Routing Traffic Engineering policies',
   }
   cliModel = SrTePolicySummaryVrfModel

   @staticmethod
   def handler( mode, args ):
      if not isSrTePolicyAgentRunning( mode ):
         mode.addError( 'SR TE Policy Agent is not running' )
         return None

      if srConfig.segmentListSharing == SegmentListSharing.withAll:
         mode.addError( 'The statistics for segment lists is counted based on '
                        'config SegmentListSharing.withinPolicy while current '
                        'config is SegmentListSharing.withAll' )

      vrfModel = SrTePolicySummaryVrfModel()
      summary = SrTePolicySummaryModel()
      pathGroupStats, policyStats, globalPolicyStats = populatePolicyStatistics()
      summary.perEndpointColorStatistics = globalPolicyStats
      summary.policyStatistics = policyStats
      summary.pathgroupStatistics = pathGroupStats
      summary.segmentListStatistics = populateSegmentListStatistics()
      summary.bindingSidConflictInvalidPathGroupsCount = \
                                    countBindingSidConflictInvalidPathGroups()
      summary.bindingSidConflictInactivePoliciesCount = \
                                    countBindingSidConflictInactivePolicies()
      summary.fecOptimizationsSupported = routingHwStatus.mplsExternalFecSupported
      vrfModel.vrfs[ DEFAULT_VRF ] = summary
      return vrfModel

BasicCli.addShowCommandClass( ShowSrTePolicySummaryCmd )

# -----------------------------------------------------------------------
# show traffic-engineering segment-routing policy [ valid | invalid ] ...
# -----------------------------------------------------------------------

policyNamesCompletion = [ CliParser.Completion( 'WORD', 'PolicyName',
   literal=False ) ]

def policyNames( mode ):
   return { PolicyName: "" for PolicyName in policyNameToKeyMap.policy }

class ShowSrTePolicyCmd( ShowCommand.ShowCliCommandClass ):
   syntax = '''show traffic-engineering segment-routing policy
              [ valid | invalid ]
              [ source ( bgp | static ) ]
              [ endpoint ADDR color COLOR ]
              [ name NAME ]
            '''
   data = {
      'traffic-engineering': matcherTrafficEngineering,
      'segment-routing': matcherSegmentRouting,
      'policy': matcherPolicy,
      'valid': 'Show valid Segment Routing Traffic Engineering policies',
      'invalid': 'Show invalid Segment Routing Traffic Engineering policies',
      'source': matcherSource,
      'bgp': matcherBgp,
      'static': matcherStatic,
      'name': CliCommand.Node(
                  matcher=CliMatcher.KeywordMatcher(
                     'name',
                     helpdesc='Show policy for a given name' ) ),
      'NAME': CliMatcher.DynamicKeywordMatcher( policyNames,
         emptyTokenCompletion=policyNamesCompletion ),
      'endpoint': matcherEndpoint,
      'ADDR': IpGenAddrMatcher( 'IPv4 or IPv6 address' ),
      'color': matcherColor,
      'COLOR': matcherColorValue,
   }
   cliModel = SrTePolicyCandidatesVrfModel

   @staticmethod
   def handler( mode, args ):
      name = args.get( 'NAME' )
      ep = args.get( 'ADDR' )
      color = args.get( 'COLOR' )
      if 'valid' in args:
         stateFilter = [ 'valid' ]
      elif 'invalid' in args:
         stateFilter = [ 'invalid' ]
      else:
         stateFilter = [ 'active', 'valid', 'invalid' ]
      sourceFilter = getSourceFilter( args )
      vrfModel = SrTePolicyCandidatesVrfModel()
      candidates = SrTePolicyCandidatesModel()
      candidates.policies = populatePolicies(
            ep, color, stateFilter, sourceFilter, name )
      vrfModel.vrfs[ DEFAULT_VRF ] = candidates
      return vrfModel

BasicCli.addShowCommandClass( ShowSrTePolicyCmd )

class ShowSrTePolicyFecCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show traffic-engineering segment-routing policy fec [ FECID ]'
   data = {
      'traffic-engineering': matcherTrafficEngineering,
      'segment-routing': matcherSegmentRouting,
      'policy': matcherPolicy,
      'fec': 'Show Segment Routing Traffic Engineering FEC',
      'FECID': CliMatcher.IntegerMatcher( 1, 0xFFFFFFFFFFFFFFFF,
                                          helpdesc='FEC identifier' ),
   }
   cliModel = SrTePolicyFecTable

   @staticmethod
   def handler( mode, args ):
      return populatePolicyFec( args.get( 'FECID' ) )

BasicCli.addShowCommandClass( ShowSrTePolicyFecCmd )

def populatePolicyFec( fecId ):
   fecTable = SrTePolicyFecTable()
   keys = []
   if fecId is not None:
      keys.append( long( fecId ) )
   else:
      keys = [ long( key ) for key in srteForwardingStatus.fec ]

   for key in sorted( keys ):
      fec = srteForwardingStatus.fec.get( key )
      if fec is None:
         continue
      fecInfo = SrTePolicyFecInfo()
      fecIdToPolicyKeyInfo = policyStatus.fecIdToPolicyKey.get( key )
      if fecIdToPolicyKeyInfo is not None:
         policy = SrTePolicyBase()
         policy.endpoint = fecIdToPolicyKeyInfo.policyKey.endpoint
         policy.color = fecIdToPolicyKeyInfo.policyKey.color
         fecInfo.policy = policy
      for via in fec.via.itervalues():
         fecVia = SrTePolicyFecVia()
         if not DynTunnelIntfId.isDynamicTunnelIntfId( via.intfId ):
            continue
         fecVia.segmentListId = \
            TunnelId( DynTunnelIntfId.tunnelId( via.intfId ) ).tunnelIndex()
         fecVia.weight = via.weight
         fecInfo.vias.append( fecVia )
      fecInfo.vias = sorted( fecInfo.vias, key=attrgetter( 'segmentListId' ) )
      fecTable.fecs[ key ] = fecInfo
   return fecTable

def populatePathGroupSegmentList( wsl, allSlArg, policyId, updateReason=False,
                                  sbfdConfig=None ):
   sl = SrTePolicyPathGroupSegmentList()
   sl.segmentListId = wsl.segmentListId
   sl.weight = wsl.weight
   sl.vias = None
   sl.backupVias = None
   sl.counterState = 'notAvailable'
   slId = sl.segmentListId

   segList = allSlArg.segmentList.get( slId )
   if segList is not None:
      for i in xrange( segList.size ):
         sl.segments.append( SrTeSegment(
            mplsLabelSid=segList.segment[ i ].mplsLabel ) )

   tunnelId = tacSegmentListId( slId ).getTunnelId()
   peerStatus = getSbfdPeerStatus( slId, sbfdConfig )
   if peerStatus:
      sl.sbfdState = peerStatus.clientStatus.status

   if segList is None or segList.size == 0:
      sl.segmentListValid = False
      if updateReason:
         sl.invalidReason = "segmentListMissing"
      return sl

   if slStatus.statusWithSbfd:
      # if Sbfd toggle is turned on, read segListStatus from statusWithSbfd
      # otherwise read from status
      sbfdKey = Tac.Value( "SrTePolicy::Sbfd::SbfdKey", slId, policyId )
      segListStatus = slStatus.statusWithSbfd.get( sbfdKey, None )
   else:
      segListStatus = slStatus.status.get( slId, None )
   if segListStatus != 'success':
      sl.segmentListValid = False
      if updateReason:
         if segListStatus is None:
            sl.invalidReason = "unresolvedTopLabel"
         elif segListStatus == 'zeroLabels':
            sl.invalidReason = 'noResolvedLabels'
         elif segListStatus == 'platformLabelStackDepthExceeded':
            sl.invalidReason = 'resolvedLabelsExceedPlatformMsd'
         elif segListStatus == 'sbfdDown':
            sl.invalidReason = 'sbfdDown'
      return sl

   tunnelEntry = slTunnelTable.entry.get( tunnelId, None )
   if tunnelEntry is None:
      sl.segmentListValid = False
      if updateReason:
         sl.invalidReason = "unresolvedTopLabel"
      return sl

   if not slStatus.statusWithSbfd:
      if sl.sbfdState and sl.sbfdState == BfdOperState.down:
         sl.segmentListValid = False
         if updateReason:
            sl.invalidReason = 'sbfdDown'
         return sl

   sl.segmentListValid = True
   vias, backupVias = getSegmentListVias( tunnelEntry )
   sl.vias = vias
   sl.backupVias = backupVias
   sl.txPackets, sl.txBytes = getCountersForTunnelEntry( tunnelEntry )
   if sl.txPackets is not None and sl.txBytes is not None:
      sl.counterState = 'active'

   return sl

def populatePolicyPathGroup( pie, state, lastModified,
                             activeModified=None, updateReason=False,
                             sbfdConfig=None ):
   pathGroup = SrTePolicyPathGroup()
   pathGroup.state = state
   pathGroup.protocol = policySrcToString( pie.id.candidateId.protocol )
   pathGroup.preference = pie.preference
   pathGroup.enlp = pie.enlp
   pathGroup.lastChangeTime = int( Tac.utcNow() - ( Tac.now() - lastModified ) )
   if state == 'active':
      pathGroup.activeTime = int( Tac.utcNow() - ( Tac.now() - activeModified ) )
   if pathGroup.protocol != "static":
      pathGroup.discriminator = pie.id.candidateId.discriminator
   pathGroup.originator = SrTePolicyOriginator(
      nodeAddress=pie.originator.nodeAddress,
      asn=bgpFormatAsn( pie.originator.asn ) )
   if pie.bindingSid != MplsLabel.null:
      pathGroup.bindingSid = SrTePolicyBindingSid( mplsLabelSid=pie.bindingSid )
   for index in sorted( pie.wSegmentList ):
      wsl = pie.wSegmentList.get( index )
      if wsl is not None:
         pathGroup.segmentLists.append(
            populatePathGroupSegmentList( wsl, allSl, pie.id, updateReason,
                                          sbfdConfig=sbfdConfig ) )
   return pathGroup

def pathGroupExists( key, stateFilter, sourceFilter ):
   '''
   Returns True for the given PolicyKey when there is a PathGroup
   that can be displayed for the given constraints of stateFilter
   and sourceFilter
   '''
   ap = policyStatus.status.get( key )
   if 'active' in stateFilter and ap is not None:
      if ap.candidateId.protocol in sourceFilter:
         return True
   valid = policyValid.policy.get( key )
   if 'valid' in stateFilter and valid is not None:
      for priority in valid.candidate:
         if priority.protocol in sourceFilter:
            candId = tacCandidateId( priority.protocol, priority.discriminator )
            if ap is None or ap.candidateId != candId:
               return True
   if 'invalid' in stateFilter:
      allCandidates = policyKeyToIdMap.policy.get( key )
      if allCandidates is None:
         return False
      if valid is None:
         return bool( allCandidates.candidate )
      else:
         validCnt = 0
         for priority in valid.candidate:
            if priority.protocol in sourceFilter:
               validCnt += 1
         allCnt = 0
         for candidateId in allCandidates.candidate:
            if candidateId.protocol in sourceFilter:
               allCnt += 1
         return True if allCnt > validCnt else False
   return False

def updatePathGroupInvalidReason( pathGroup, bsidReason=None ):
   # Update the reason based on the binding-sid
   if bsidReason is not None:
      if bsidReason == tacPolicyInvalidReason.policyBsidOutOfSrlb:
         pathGroup.notActiveReason = "bindingSidOutOfSrlbRange"
      elif bsidReason == tacPolicyInvalidReason.policyBsidNotConfigured:
         pathGroup.notActiveReason = "bindingSidNotConfigured"
      elif bsidReason == tacPolicyInvalidReason.policyBsidInConflictWithOtherApp:
         pathGroup.notActiveReason = "bindingSidInUseByOtherApplication"
      elif bsidReason == tacPolicyInvalidReason.policyNoSegmentLists:
         pathGroup.notActiveReason = "segmentListsMissing"
   else:
      # Update the reason based on the segment-list status
      resolvedSl = False
      for segmentList in pathGroup.segmentLists:
         if segmentList.segmentListValid:
            resolvedSl = True
            break
      if not resolvedSl:
         pathGroup.notActiveReason = "segmentListsUnresolved"

def getPathGroupInactiveReason( pathPriority, activePriority ):

   if pathPriority == activePriority:
      # If this is the best path and not in status, it must be a BSID conflict
      return 'bindingSidConflict'

   compareResult = activePriority.compare( pathPriority )

   if compareResult == 'isHigherPreference':
      return 'higherPreferenceAvailable'
   elif compareResult == 'isBetterProtocol':
      return 'betterProtocolAvailable'
   elif compareResult == 'isLowerOrigin':
      return 'lowerOriginAvailable'
   if srConfig.useDiscriminatorInTieBreaking and \
      compareResult == 'isHigherDiscriminator':
      return 'higherDiscriminatorAvailable'
   if activePriority.sameExceptDiscriminator( pathPriority ):
      return 'samePriorityActive'
   return None

def policyInputTableForProtocol( protocol ):
   if protocol == tacProtocol.staticProtocol:
      return staticPolicyInput
   if protocol == tacProtocol.bgpProtocol:
      return bgpPolicyInput
   return None

def populatePolicyInvalidPathGroup( pId, lastModified ):
   table = policyInputTableForProtocol( pId.candidateId.protocol )
   if table is None:
      return None
   policy = table.policyInputEntry.get( pId )
   if policy is None:
      return None
   pathGroup = SrTePolicyPathGroup()
   pathGroup.state = 'invalid'
   pathGroup.protocol = policySrcToString( pId.candidateId.protocol )
   pathGroup.preference = policy.preference
   pathGroup.enlp = policy.enlp
   pathGroup.lastChangeTime = int( Tac.utcNow() - ( Tac.now() - lastModified ) )
   if pathGroup.protocol != "static":
      pathGroup.discriminator = pId.candidateId.discriminator
   pathGroup.originator = SrTePolicyOriginator(
                                       nodeAddress=policy.originator.nodeAddress,
                                       asn=bgpFormatAsn( policy.originator.asn ) )
   if policy.bindingSid != MplsLabel.null:
      pathGroup.bindingSid = SrTePolicyBindingSid( mplsLabelSid=policy.bindingSid )
   for index in sorted( policy.segmentList ):
      sl = populateInvalidPathGroupSegmentList( policy, index )
      pathGroup.segmentLists.append( sl )
   return pathGroup

def populateInvalidPathGroupSegmentList( policy, index ):
   sl = SrTePolicyPathGroupSegmentList()
   sl.vias = None
   sl.backupVias = None
   policySl = policy.segmentList[ index ].segList
   sl.weight = policy.segmentList[ index ].weight
   for i in range( policySl.stackSize ):
      sl.segments.append( SrTeSegment( mplsLabelSid=policySl.labelStack( i ) ) )
   return sl

def populatePathGroups( key, stateFilter, sourceFilter, updateReason ):
   allCandidates = policyKeyToIdMap.policy.get( key )
   if allCandidates is None:
      return
   validCandidates = policyValid.policy.get( key )
   active = policyStatus.status.get( key )
   policy = srConfig.policy.get( key )
   sbfdConfig = policy.sbfdConfig if policy else None
   # Populate the active path first
   if 'active' in stateFilter and active is not None:
      pie = allPolicy.policyIdEntry.get( tacPolicyId( key, active.candidateId ) )
      cInfo = allCandidates.candidate.get( active.candidateId )
      if pie is not None and cInfo is not None and \
            pie.id.candidateId.protocol in sourceFilter:
         pathGroup = populatePolicyPathGroup( pie, "active", cInfo.lastModified,
                                              active.lastModified, updateReason,
                                              sbfdConfig=sbfdConfig )
         yield pathGroup
   # Populate the valid paths next
   if 'valid' in stateFilter and validCandidates is not None:
      for path in validCandidates.candidate:
         if path.protocol not in sourceFilter:
            continue
         pathId = tacCandidateId( path.protocol, path.discriminator )
         cInfo = allCandidates.candidate.get( pathId )
         if cInfo is None:
            continue
         if active is not None and active.candidateId == pathId:
            continue
         pie = allPolicy.policyIdEntry.get( tacPolicyId( key, pathId ) )
         if pie is not None:
            pathGroup = populatePolicyPathGroup( pie, "valid",
                                                 cInfo.lastModified,
                                                 updateReason=updateReason,
                                                 sbfdConfig=sbfdConfig )
            if updateReason:
               pathGroup.notActiveReason = getPathGroupInactiveReason(
                  path, validCandidates.active.priority )
            yield pathGroup
   # Populate the invalid paths
   if 'invalid' in stateFilter:
      for candidateId, cInfo in allCandidates.candidate.items():
         if candidateId.protocol not in sourceFilter:
            continue
         pId = tacPolicyId( key, candidateId )
         pie = allPolicy.policyIdEntry.get( pId )
         if pie is None:
            pathGroup = populatePolicyInvalidPathGroup( pId, cInfo.lastModified )
            if pathGroup is not None:
               if updateReason:
                  updatePathGroupInvalidReason( pathGroup, bsidReason=cInfo.reason )
               yield pathGroup
            continue
         if validCandidates is not None:
            priority = tacPriority(
               pie.preference, pie.id.candidateId.protocol,
               pie.originator, pie.id.candidateId.discriminator )
            if priority in validCandidates.candidate:
               continue
         pathGroup = populatePolicyPathGroup( pie, "invalid", cInfo.lastModified,
                                              updateReason=updateReason,
                                              sbfdConfig=sbfdConfig )
         if updateReason:
            updatePathGroupInvalidReason( pathGroup )
         yield pathGroup

def populatePolicies( ep, color, stateFilter, sourceFilter, name=None ):
   keys = []
   if name is not None:
      tacPolicyName = Tac.newInstance( "SrTePolicy::PolicyName",
                                       str( name ) )
      if tacPolicyName in policyNameToKeyMap.policy:
         policyKey = policyNameToKeyMap.policy[ tacPolicyName ]
         keys.append( SrTePolicyCommonLib.PolicyKey(
            policyKey.endpoint, policyKey.color ) )
         updateReason = True
   elif ep is not None and color is not None:
      keys.append( SrTePolicyCommonLib.PolicyKey( ep, color ) )
      updateReason = True
   else:
      keys = [ key for key in policyKeyToIdMap.policy ]
      updateReason = False

   for key in sorted( keys ):
      # policyEntry.pathGroups uses a GeneratorList. When there is no
      # pathgroup exist matching the criteria for filter, the policyEntry
      # generation should also be skipped. Hence check for the pathGroup
      # existence first.
      if not pathGroupExists( key, stateFilter, sourceFilter ):
         continue
      if [ 'active' ] == stateFilter:
         pathGroup = next( populatePathGroups( key, stateFilter, sourceFilter,
                                               updateReason=updateReason ) )
         if pathGroup is None:
            continue
         policyEntry = SrTePolicyActive()
         policyEntry.pathGroup = pathGroup
      else:
         policyEntry = SrTePolicyCandidates()
         policyEntry.pathGroups = populatePathGroups( key, stateFilter,
                                                      sourceFilter,
                                                      updateReason=updateReason )
      policyEntry.endpoint = key.endpoint
      policyEntry.color = key.color
      policyEntry.name = None
      policyEntry.description = None
      policy = srConfig.policy.get( key )
      if policy is not None and policy.policyName:
         policyEntry.name = policy.policyName
      if policy is not None and policy.policyDescription:
         policyEntry.description = policy.policyDescription
      ( pkts, octets ) = getCountersForPolicy( key )
      policyEntry.txPackets = pkts
      policyEntry.txBytes = octets
      yield policyEntry

#-----------------------------------------------------------------
# clear traffic-engineering segment-routing policy ... counters
#-----------------------------------------------------------------
class ClearPolicyCountersCmd( CliCommand.CliCommandClass ):
   syntax = '''clear traffic-engineering segment-routing policy
               [ endpoint ADDR color COLOR ] counters
            '''
   data = {
      'clear': CliToken.Clear.clearKwNode,
      'traffic-engineering': matcherTrafficEngineering,
      'segment-routing': matcherSegmentRouting,
      'policy': matcherPolicy,
      'endpoint': matcherEndpoint,
      'ADDR': IpGenAddrMatcher( 'IPv4 or IPv6 address' ),
      'color': matcherColor,
      'COLOR': matcherColorValue,
      'counters': 'Show packet & byte egress counters for segment lists',
   }

   @staticmethod
   def handler( mode, args ):
      policyEntriesFound = 0
      ep = args.get( 'ADDR' )
      color = args.get( 'COLOR' )

      if ep is None and color is None:
         # clear all policies by toggling the clearAllPolicyCountersRequest flag
         policyEntriesFound = len( policyCounterTable.policyCounter )
         policyCounterClear.clearAllPolicyCountersRequest = False
         policyCounterClear.clearAllPolicyCountersRequest = True
      else:
         # clear a single policy
         policyKey = SrTePolicyCommonLib.PolicyKey( ep, color )
         if policyKey in policyCounterTable.policyCounter:
            policyEntriesFound += 1
            policyCounterClear.clearPolicyCountersRequest.clear()
            policyCounterClear.clearPolicyCountersRequest[ policyKey ] = True

      if policyEntriesFound:
         print "%d policy counter entr%s cleared successfully" % (
            policyEntriesFound, 'y' if policyEntriesFound == 1 else 'ies' )
      else:
         mode.addWarning( "No policy counter entries found" )

BasicCli.EnableMode.addCommandClass( ClearPolicyCountersCmd )

# ----------------------------------------------------------------
# show traffic-engineering segment-routing policy segment-list ...
# ----------------------------------------------------------------
slIdValueMatcher = CliMatcher.IntegerMatcher( tacSegmentListId.min,
                                              tacSegmentListId.max,
                                              helpdesc='Segment list ID value' )

def buildSegmentListsModel( validityFilter, slIdFilter=None, counters=False ):
   slModelDict = {}
   if slIdFilter is None:
      slIds = allSl.segmentList.iterkeys()
   else:
      slIds = [ slIdFilter ]

   for slId in slIds:
      sl = allSl.segmentList.get( slId )
      if sl is None:
         # The requested SL doesn't exist, so skip it.
         continue

      valid, tunnelEntry = isSlValidAndGetTunEntry( slId, sl )

      if validityFilter is not None and validityFilter != valid:
         # We're filtering this entry out, so don't bother constructing it.
         continue

      vias = None
      backupVias = None
      if valid:
         vias, backupVias = getSegmentListVias( tunnelEntry )

      if counters:
         txPackets, txBytes = getCountersForTunnelEntry( tunnelEntry )
         slModel = SrTePolicySegmentListCounters(
            vias=vias,
            txPackets=txPackets,
            txBytes=txBytes,
         )
      else:
         slModel = SrTePolicySegmentList(
            segments=[ SrTeSegment( mplsLabelSid=sl.segment[ i ].mplsLabel )
                       for i in xrange( sl.size ) ],
            valid=valid,
            vias=vias,
            backupVias=backupVias,
         )

      slModelDict[ slId ] = slModel

   if counters:
      return SrTePolicySegmentListsCounters( segmentLists=slModelDict )
   else:
      return SrTePolicySegmentLists( segmentLists=slModelDict )

class ShowSrTePolicySegmentListCmd( ShowCommand.ShowCliCommandClass ):
   syntax = '''show traffic-engineering segment-routing policy segment-list
               [ valid | invalid | ( id ID ) ]
            '''
   data = {
      'traffic-engineering': matcherTrafficEngineering,
      'segment-routing': matcherSegmentRouting,
      'policy': matcherPolicy,
      'segment-list': matcherSegmentList,
      'valid': 'Show valid Segment Routing Traffic Engineering policy segment lists',
      'invalid': 'Show invalid Segment Routing Traffic Engineering policy '
                 'segment lists',
      'id': matcherId,
      'ID': slIdValueMatcher,
   }
   cliModel = VrfSrTePolicySegmentLists

   @staticmethod
   def handler( mode, args ):
      validityFilter = None
      if 'valid' in args:
         validityFilter = True
      elif 'invalid' in args:
         validityFilter = False

      slId = args.get( 'ID' )
      segmentLists = buildSegmentListsModel( validityFilter, slIdFilter=slId )
      vrfSls = VrfSrTePolicySegmentLists(
         vrfs={ DEFAULT_VRF: segmentLists },
         _validityFilter=validityFilter )
      return vrfSls

BasicCli.addShowCommandClass( ShowSrTePolicySegmentListCmd )

class ShowSrTePolicySegmentListCountersCmd( ShowCommand.ShowCliCommandClass ):
   syntax = '''show traffic-engineering segment-routing policy segment-list counters
               [ id ID ]
            '''
   data = {
      'traffic-engineering': matcherTrafficEngineering,
      'segment-routing': matcherSegmentRouting,
      'policy': matcherPolicy,
      'segment-list': matcherSegmentList,
      'counters': 'Show packet & byte egress counters for segment lists',
      'id': matcherId,
      'ID': slIdValueMatcher,
   }
   cliModel = VrfSrTePolicySegmentListsCounters

   @staticmethod
   def handler( mode, args ):
      slId = args.get( 'ID' )
      vrfSls = VrfSrTePolicySegmentListsCounters( vrfs={
         DEFAULT_VRF: buildSegmentListsModel( True, slIdFilter=slId,
                                              counters=True ) } )
      return vrfSls

BasicCli.addShowCommandClass( ShowSrTePolicySegmentListCountersCmd )

#--------------------------------------------------------------------------------
# SrTePolicy commands in 'show tech-support'
#--------------------------------------------------------------------------------
def _srTeShowTechCmds():
   return [
      'show traffic-engineering segment-routing cbf fec',
      'show traffic-engineering segment-routing policy',
      'show traffic-engineering segment-routing policy segment-list',
      'show traffic-engineering segment-routing policy fec',
   ]
TechSupportCli.registerShowTechSupportCmdCallback( '2017-09-18 11:35:37',
      _srTeShowTechCmds,
      summaryCmdCallback=lambda: [
         'show traffic-engineering segment-routing policy summary' ] )

#--------------------------------------------------------------------------------
# show tech-support extended srtepolicy
#-------------------------------------------------------------------------------
class ShowSrTeTechSupportCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show tech-support extended srtepolicy'
   data = {
         'tech-support': techSupportKwMatcher,
         'extended': extendedKwMatcher,
         'srtepolicy': 'Show detailed state of SrTePolicy Agent',
   }
   privileged = True

   @staticmethod
   def handler( mode, args ):
      AgentCommandRequest.runSocketCommand( mode.entityManager,
                                             'SrTePolicy',
                                             'SrTePolicyDump',
                                             'DUMP_SRTEPOLICYSTATE' )

BasicCli.addShowCommandClass( ShowSrTeTechSupportCmd )
#--------------------------------------------------------------------------------
# show traffic-engineering segment-routing cbf fec [ FECID ] ]
#-------------------------------------------------------------------------------

class ShowSrTeCbfFecCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show traffic-engineering segment-routing cbf fec [ FECID ]'
   data = {
      'traffic-engineering': matcherTrafficEngineering,
      'segment-routing': matcherSegmentRouting,
      'cbf': matcherCbf, 
      'fec': 'Show Class based forwarding override FEC',
      'FECID': CliMatcher.IntegerMatcher( 1, 0xFFFFFFFFFFFFFFFF,
                                          helpdesc='FEC identifier' )
   }

   cliModel = SrTePolicyCbfFecVrfModel

   @staticmethod
   def handler( mode, args ):
      vrfModel = SrTePolicyCbfFecVrfModel()
      srTePolicyCbfFecModel = SrTePolicyCbfFecModel()

      fecId = args.get( 'FECID' )
      for overrideKey in fecOverrideTable.override:
         if fecId and fecId != overrideKey.fecId:
            continue
         overrideEntry = fecOverrideTable.override.get( overrideKey )
         if not overrideEntry:
            continue
         fecOverrideEntry = SrTePolicyCbfFecOverride()
         fecOverrideEntry.defaultFecId = overrideEntry.key.fecId
         fecOverrideEntry.overrideFecId = overrideEntry.fecId
         fecOverrideEntry.ipVersion = overrideEntry.key.af
         fecOverrideEntry.dscp = overrideEntry.key.dscp
         srTePolicyCbfFecModel.overrides.append( fecOverrideEntry )

      vrfModel.vrfs[ DEFAULT_VRF ] = srTePolicyCbfFecModel
      return vrfModel

BasicCli.addShowCommandClass( ShowSrTeCbfFecCmd )


def Plugin( entityManager ):
   global srConfig
   global colorDscpMap
   global mplsConfig
   global mplsHwCapability
   global routingHwStatus
   global policyStatus
   global policyValid
   global policyKeyToIdMap
   global allPolicy
   global allSl
   global slStatus
   global slTunnelTable
   global srteForwardingStatus
   global bgpPolicyInput
   global staticPolicyInput
   global policyCounterClear
   global policyCounterTable
   global bfdStatusPeer
   global fecOverrideTable
   global policyNameToKeyMap

   srConfig = ConfigMount.mount( entityManager,
                                 "te/segmentrouting/srtepolicy/staticconfig",
                                 "SrTePolicy::Config", "w" )
   colorDscpMap = ConfigMount.mount( entityManager,
                                     "te/segmentrouting/color-dscp/config",
                                     "SrTePolicy::ColorDscpMap", "w" )
   policyCounterClear = ConfigMount.mount(
      entityManager,
      "te/segmentrouting/srtepolicy/policycounterclear",
      "SrTePolicy::Counters::PolicyCliClearConfig",
      "w" )
   policyNameToKeyMap = ConfigMount.mount(
      entityManager,
      "te/segmentrouting/srtepolicy/policynametokeymap",
      "SrTePolicy::PolicyNameToKeyMap", "w" )
   mplsConfig = LazyMount.mount( entityManager,
                                 "routing/mpls/config",
                                 "Mpls::Config", "r" )
   mplsHwCapability = LazyMount.mount( entityManager,
                                       "routing/hardware/mpls/capability",
                                       "Mpls::Hardware::Capability",
                                       "r" )
   routingHwStatus = LazyMount.mount( entityManager,
                                      "routing/hardware/status",
                                      "Routing::Hardware::Status", "r" )
   policyValid = LazyMount.mount( entityManager,
                                  "te/segmentrouting/srtepolicy/policyvalid",
                                  "SrTePolicy::EcPolicy", "r" )
   policyKeyToIdMap = LazyMount.mount( entityManager,
                                      "te/segmentrouting/srtepolicy/policykeyidmap",
                                      "SrTePolicy::PolicyKeyToIdMap", "r" )
   allPolicy = LazyMount.mount( entityManager,
                                "te/segmentrouting/srtepolicy/allpolicy",
                                "SrTePolicy::PolicySelectionInput", "r" )
   allSl = LazyMount.mount( entityManager,
                            "te/segmentrouting/srtepolicy/allsegmentlist",
                            "SrTePolicy::Export::SegmentListCollection",
                            "r" )
   slStatus = LazyMount.mount( entityManager,
                               "te/segmentrouting/srtepolicy/segmentliststatus",
                               "SrTePolicy::SegmentListStatus",
                               "r" )
   slTunnelTable = readMountTunnelTable(
      TunnelTableIdentifier.srTeSegmentListTunnelTable, entityManager )
   TeCli.TeModeDependents.registerDependentClass( TeSubMode, priority=20 )

   smashEm = SharedMem.entityManager( sysdbEm=entityManager )
   readerInfo = Smash.mountInfo( 'reader' )
   srteForwardingStatus = smashEm.doMount( "forwarding/srte/status",
                                           "Smash::Fib::ForwardingStatus",
                                           readerInfo )
   bgpPolicyInput = smashEm.doMount( "srTePolicy/Bgp",
                                     "SrTePolicy::PolicyInput",
                                     readerInfo )
   staticPolicyInput = smashEm.doMount( "srTePolicy/Static",
                                        "SrTePolicy::PolicyInput",
                                        readerInfo )
   policyCounterTable = smashEm.doMount( "srTePolicy/policyCounterTable",
                                         "SrTePolicy::Counters::PolicyCounterTable",
                                         readerInfo )
   policyStatus = smashEm.doMount( srTePolicyStatusPath(),
                                   "SrTePolicy::PolicyStatus", readerInfo )
   bfdStatusPeer = LazyMount.mount( entityManager, 'bfd/status/peer',
                                    'Bfd::StatusPeer', 'r' )
   fecOverrideTable = smashEm.doMount( "te/segmentrouting/fecoverride/config",
                                       "Qos::Smash::FecOverrideConfig",
                                       readerInfo )
