# Copyright (c) 2018 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import Arnet
import Tac
from Intf.IntfRange import intfListToCanonical
from IntfModel import Interface
from CliModel import Model, Enum, List, Dict
from EthIntfLib import linkModeCapabilitiesToStr

_serdesSpeedValues = [
      speed[ len( 'compatibility' ): ]
      for speed in Tac.Type( 'Interface::SpeedCompatSetting' ).attributes
      if speed != 'compatibilityUnknown' ]

TRIM = len( 'compatibility' )

class SerdesSpeed( Model ):
   speed = Enum( values=_serdesSpeedValues, help='SerDes speed' )

class SpeedGroupStatus( Model ):
   interfaces = List( valueType=Interface, help='Group member interfaces' )
   masterInterface = Interface( help='Master interface of the group' )
   supportedSerdesSpeeds = List( valueType=SerdesSpeed,
                                 help='Supported serdes speeds' )
   defaultSerdesSpeeds = List( valueType=SerdesSpeed, help='Default serdes speeds' )
   configuredSerdesSpeeds = List( valueType=SerdesSpeed,
                                  help='Configured serdes speeds' )
   activeSerdesSpeeds = List( valueType=SerdesSpeed, help='Active serdes speeds' )

   def toModel( self, status, config ):
      self.interfaces = status.memberIntf.keys()
      self.masterInterface = status.masterIntf
      self.supportedSerdesSpeeds = self._enumsToSerdesSpeeds(
            status.settingPriority.values() )
      self.defaultSerdesSpeeds = self._enumsToSerdesSpeeds(
            status.settingPriority.values()[ :status.settingCount ] )
      self.configuredSerdesSpeeds = self._enumsToSerdesSpeeds(
            config.setting.keys() if config else [] )
      self.activeSerdesSpeeds = self._enumsToSerdesSpeeds(
            status.setting.keys() )
      return self

   def _enumsToSerdesSpeeds( self, enums ):
      return [ SerdesSpeed( speed=e[ TRIM: ] ) for e in enums ]

   def _serdesSpeedsString( self, serdesSpeeds ):
      return ','.join( sorted( [ s.speed for s in serdesSpeeds ] ) )

   def supportedSerdesSpeedsString( self ):
      return self._serdesSpeedsString( self.supportedSerdesSpeeds )

   def defaultSerdesSpeedsString( self ):
      if self.masterInterface:
         return "Per %s" % self.masterInterface.shortName
      else:
         return self._serdesSpeedsString( self.defaultSerdesSpeeds )

   def configuredSerdesSpeedsString( self ):
      if self.configuredSerdesSpeeds:
         return self._serdesSpeedsString( self.configuredSerdesSpeeds )
      else:
         return 'default'

   def activeSerdesSpeedsString( self ):
      if self.activeSerdesSpeeds:
         return self._serdesSpeedsString( self.activeSerdesSpeeds )
      elif self.masterInterface:
         return "Per %s" % self.masterInterface.shortName
      else:
         return 'unknown'

def firstIntfKey( item ):
   '''Key function for sorting by first interface.
   '''
   _, value = item
   return Arnet.intfNameKey( value.interfaces[ 0 ] )

class SpeedGroupStatuses( Model ):
   groups = Dict( keyType=str, valueType=SpeedGroupStatus,
         help="Mapping between a speed group's name and its status" )

   def render( self ):
      if not self.groups:
         return

      colWidths = ( 5, 17, 11, 11, 11, 11 )
      hdrFmt = ' '.join( '%%-%ds' % w for w in colWidths )
      print hdrFmt % ( 'Group', 'Interfaces', 'Supported',
                       'Default', 'Configured', 'Active' )
      print hdrFmt % tuple( '-' * w for w in colWidths )

      for group, data in sorted( self.groups.items(), key=firstIntfKey ):
         print hdrFmt % ( group,
                          ','.join( intfListToCanonical( data.interfaces,
                                                         noHoleRange=True ) ),
                          data.supportedSerdesSpeedsString(),
                          data.defaultSerdesSpeedsString(),
                          data.configuredSerdesSpeedsString(),
                          data.activeSerdesSpeedsString() )

class SpeedGroupCap( Model ):
   interfaces = List( valueType=Interface, help='Group member interfaces' )
   availableSpeedConfigurations = Dict( keyType=str, valueType=str,
         help='Mapping between a SerDes speed and available speed configurations' )

   def showLanesInStr( self, status ):
      for setting in status.settingPriority.itervalues():
         caps = status.supportedModes[ setting ]
         if( caps.mode50GbpsFull1Lane or caps.mode100GbpsFull2Lane or
             caps.mode200GbpsFull4Lane or caps.mode200GbpsFull8Lane or
             caps.mode400GbpsFull8Lane ):
            return True
      return False

   def toModel( self, status ):
      self.interfaces = status.memberIntf.keys()
      showLanesInStr = self.showLanesInStr( status )
      for setting in status.settingPriority.itervalues():
         caps = status.supportedModes[ setting ]
         lmCapStr = linkModeCapabilitiesToStr( caps, showLanes=showLanesInStr )
         self.availableSpeedConfigurations[ setting[ TRIM: ] ] = lmCapStr
      return self

class SpeedGroupCaps( Model ):
   groups = Dict( keyType=str, valueType=SpeedGroupCap,
         help="Mapping between a speed group's name and its capabilities" )

   def render( self ):
      if not self.groups:
         return

      colWidths = ( 5, 17, 12, 46 )
      hdrFmt = ' '.join( '%%-%ds' % w for w in colWidths )
      print hdrFmt % ( 'Group', 'Interfaces', 'Select Value',
                       'Available Speed Configurations' )
      print hdrFmt % tuple( '-' * w for w in colWidths )
      for group, data in sorted( self.groups.items(), key=firstIntfKey ):
         availSpeedConfigsDict = data.availableSpeedConfigurations
         for setting, string in sorted( availSpeedConfigsDict.iteritems() ):
            print hdrFmt % ( group,
                             ','.join( intfListToCanonical( data.interfaces,
                                                            noHoleRange=True ) ),
                             setting,
                             string )
