# Copyright (c) 2018 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

from __future__ import absolute_import, division, print_function

import BasicCli, CliToken.Hardware, CliParser, Tac
import BasicCliUtil, ConfigMount, LazyMount
import Intf.IntfRange as IntfRange
import CliPlugin.SpeedGroupModel as SpeedGroupModel
import CliPlugin.FruCli as FruCli
import CliPlugin.SpeedGroupRule as SpeedGroupRule
from IntfRangePlugin.SpeedGroup import SpeedGroupType
import ProductAttributes
import CliCommand
import CliMatcher
import ShowCommand

speedGroupConfigSliceDir = None
speedGroupStatusSliceDir = None

def getSpeedGroupConfigDir( name ):
   if ProductAttributes.productAttributes().productAttributes.hasSwitchcard:
      sliceName = "Switchcard1"
   elif '/' in name:
      # Modular system
      sliceName = "Linecard" + name.split( '/' )[ 0 ]
   else:
      sliceName = "FixedSystem"
   return speedGroupConfigSliceDir[ sliceName ]

def checkSpeedGroupChangeCancel( mode, group, compatibility ):
   promptTemplate = "Changing the speed group setting of {0} may cause member " \
                    "interfaces in speed group to flap."
   warningPrompt = ""
   if mode.session.commandConfirmation():
      changedGroup = []
      for name in IntfRange.intfListFromCanonical( [ group ] ):
         nameWithNoType = IntfRange.intfTypeFromName( name )[ 1 ]
         speedGroupConfigDir = getSpeedGroupConfigDir( nameWithNoType )
         speedGroupConfig = speedGroupConfigDir.get( nameWithNoType )
         if speedGroupConfig:
            curConfig = set( speedGroupConfig.setting.keys() )
         else:
            curConfig = set()
         if compatibility:
            newConfig = set( compatibility )
         else:
            newConfig = set()
         if newConfig != curConfig:
            changedGroup.append( nameWithNoType )
      if changedGroup:
         warningPrompt = promptTemplate.format( group )

   if warningPrompt:
      mode.addWarning( warningPrompt )
      promptText = "Do you wish to proceed with this command? [y/N]"
      ans = BasicCliUtil.confirm( mode, promptText, answerForReturn=False )

      # See if user cancelled the command.
      if not ans:
         abortMsg = "Command aborted by user for %s" % group
         mode.addMessage( abortMsg )
         return True

   return False

def configSpeedGroup( mode, group, compatibility=None ):
   if checkSpeedGroupChangeCancel( mode, group, compatibility ):
      return

   for name in IntfRange.intfListFromCanonical( [ group ] ):
      nameWithNoType = IntfRange.intfTypeFromName( name )[ 1 ]
      if compatibility:
         speedGroupConfigDir = getSpeedGroupConfigDir( nameWithNoType )
         speedGroupConfig = speedGroupConfigDir.get( nameWithNoType )
         if speedGroupConfig is None:
            speedGroupConfig = speedGroupConfigDir.newGroup( nameWithNoType )
         curConfig = set( speedGroupConfig.setting.keys() )
         curGenId = speedGroupConfig.settingGeneration.id
         # Delete unconfigured compatibility settings
         for setting in speedGroupConfig.setting:
            if setting not in compatibility:
               del speedGroupConfig.setting[ setting ]
         # Add configured compatibility settings
         for attr in compatibility:
            speedGroupConfig.setting[ attr ] = True
         newConfig = set( speedGroupConfig.setting.keys() )
         if newConfig != curConfig:
            speedGroupConfig.settingGeneration = Tac.Value( "Ark::Generation",
                                                            curGenId + 1, True )
      else:
         # no/default command delete the entity
         speedGroupConfigDir = getSpeedGroupConfigDir( nameWithNoType )
         del speedGroupConfigDir.group[ nameWithNoType ]

# Add a hidden tokenRule to parse a speed-group when the group in
# SpeedGroupStatusDir has not been created.
speedGroupNameNode = CliCommand.Node(
   matcher=SpeedGroupRule.SpeedGroupMatcher( priority=CliParser.PRIO_LOW ),
   storeSharedResult=True,
   hidden=True )

speedGroupNameRangeNode = CliCommand.Node(
   matcher=IntfRange.IntfRangeMatcher(
         explicitIntfTypes=( SpeedGroupType.type, ),
         dollarCompHelpText='Group list end' ),
   storeSharedResult=True )

def ethLinkModeSetToHelpDesc( elms ):
   compatibilityList = []
   if elms.mode1GbpsFull:
      compatibilityList.append( '1' )
   if elms.mode10GbpsFull:
      compatibilityList.append( '10' )
   if elms.mode25GbpsFull:
      compatibilityList.append( '25' )
   if elms.mode40GbpsFull:
      compatibilityList.append( '40' )
   if elms.mode50GbpsFull:
      compatibilityList.append( '50-2' )
   if elms.mode50GbpsFull1Lane:
      compatibilityList.append( '50-1' )
   if elms.mode100GbpsFull:
      compatibilityList.append( '100-4' )
   if elms.mode100GbpsFull2Lane:
      compatibilityList.append( '100-2' )
   if elms.mode200GbpsFull4Lane:
      compatibilityList.append( '200-4' )
   if elms.mode400GbpsFull8Lane:
      compatibilityList.append( '400-8' )

   return '%sGbE' % '/'.join( compatibilityList )

def getKeyword( context, speed ):
   intfRange = ( context.sharedResult.get( 'SPEED-GROUP' ) or
                 context.sharedResult[ 'INTF' ] )
   speedGroups = []
   for name in IntfRange.intfListFromCanonical( [ intfRange ] ):
      speedGroups.append( IntfRange.intfTypeFromName( name )[ 1 ] )
   helpDescList = []
   statuses = getSpeedGroupStatusesAll( speedGroups )
   if statuses:
      for status in statuses.itervalues():
         compatibility = status.supportedModes.get( tokenToCompatEnum[ speed ] )
         if compatibility:
            helpDesc = ethLinkModeSetToHelpDesc( compatibility )
            if helpDesc not in helpDescList:
               helpDescList.append( helpDesc )
      if not helpDescList:
         # return an empty dictionary to guard this token
         return {}
      helpDescList.sort( key=len )
   else:
      helpDesc = { '10g' : '1/10/40GbE',
                   '25g' : '25/50-2/100-4GbE',
                   '50g' : '50-1/100-2/200-4/400-8GbE' } [ speed ]
      helpDescList.append( helpDesc )
   return { speed : 'Allow ' + ' or '.join( helpDescList ) }

def speedGroup10g( mode, context ):
   return getKeyword( context, '10g' )

def speedGroup25g( mode, context ):
   return getKeyword( context, '25g' )

def speedGroup50g( mode, context ):
   return getKeyword( context, '50g' )

matcher10g = CliMatcher.DynamicKeywordMatcher( speedGroup10g, passContext=True )
matcher25g = CliMatcher.DynamicKeywordMatcher( speedGroup25g, passContext=True )
matcher50g = CliMatcher.DynamicKeywordMatcher( speedGroup50g, passContext=True )

speedSet = {
   '10g': CliCommand.Node( matcher=matcher10g ),
   '25g': CliCommand.Node( matcher=matcher25g ),
   '50g': CliCommand.Node( matcher=matcher50g ),
   }

compatEnumToToken = { 'compatibility10g' : '10g',
                      'compatibility25g' : '25g',
                      'compatibility50g' : '50g' }
tokenToCompatEnum = { y : x for x, y in compatEnumToToken.iteritems() }

class HardwareSpeedGroupCommand( CliCommand.CliCommandClass ):
   syntax = "hardware SPEED-GROUP | INTF serdes COMPAT-SPEEDS"
   noOrDefaultSyntax = "hardware SPEED-GROUP | INTF serdes ..."
   data = {
      "hardware": CliToken.Hardware.hardwareForConfigMatcher,
      "SPEED-GROUP": speedGroupNameNode,
      "INTF": speedGroupNameRangeNode,
      'serdes': 'serdes speed compatibility',
      'COMPAT-SPEEDS': CliCommand.setCliExpression( speedSet, optional=False,
                                                    name='compatibility' )
      }
   @staticmethod
   def handler( mode, args ):
      group = args.get( 'SPEED-GROUP' ) or args[ 'INTF' ]
      compatibility = [ tokenToCompatEnum[ x ] for x in args[ 'compatibility' ] ]
      configSpeedGroup( mode, group, compatibility=compatibility )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      group = args.get( 'SPEED-GROUP' ) or args[ 'INTF' ]
      configSpeedGroup( mode, group, compatibility=None )

BasicCli.GlobalConfigMode.addCommandClass( HardwareSpeedGroupCommand )

def getSpeedGroupStatusesAll( speedGroup ):
   # Returns a map of speedGroup name to speedGroupsStatus. When speedGroup is
   # an empty list, every group in every speedGroupStatusDir is added to the map.
   speedGroupStatuses = {}
   for linecard in speedGroupStatusSliceDir.itervalues():
      for speedGroupStatusDir in linecard.itervalues():
         if not speedGroup:
            speedGroupStatuses.update( speedGroupStatusDir.group )
         else:
            for group in speedGroup:
               status = speedGroupStatusDir.group.get( group )
               if status is not None:
                  speedGroupStatuses[ group ] = status

   return speedGroupStatuses

def getSpeedGroupStatuses( speedGroup ):
   statuses = getSpeedGroupStatusesAll( speedGroup )
   groupStatuses = SpeedGroupModel.SpeedGroupStatuses()
   for name, status in statuses.iteritems():
      groupStatus = SpeedGroupModel.SpeedGroupStatus()
      config = getSpeedGroupConfigDir( name ).get( name )
      groupStatuses.groups[ name ] = groupStatus.toModel( status, config )

   return groupStatuses

class ShowHardwareSpeedGroupStatusCommand( ShowCommand.ShowCliCommandClass ):
   syntax = "show hardware speed-group | SPEED-GROUP status"
   data = {
      "hardware": CliToken.Hardware.hardwareForShowMatcher,
      "speed-group": "Hardware speed group",
      "SPEED-GROUP": IntfRange.IntfRangeMatcher(
         explicitIntfTypes=( SpeedGroupType.type, ),
         dollarCompHelpText='Group list end' ),
      "status": "Show speed group status"
      }
   cliModel = SpeedGroupModel.SpeedGroupStatuses
   @staticmethod
   def handler( mode, args ):
      speedGroup = []
      intfRange = args.get( 'SPEED-GROUP' )
      if intfRange:
         for name in IntfRange.intfListFromCanonical( [ intfRange ] ):
            speedGroup.append( IntfRange.intfTypeFromName( name )[ 1 ] )

      return getSpeedGroupStatuses( speedGroup )

BasicCli.addShowCommandClass( ShowHardwareSpeedGroupStatusCommand )

def getSpeedGroupCaps( speedGroup ):
   statuses = getSpeedGroupStatusesAll( speedGroup )
   groupCaps = SpeedGroupModel.SpeedGroupCaps()
   for name, status in statuses.iteritems():
      groupCap = SpeedGroupModel.SpeedGroupCap()
      groupCaps.groups[ name ] = groupCap.toModel( status )

   return groupCaps

class ShowHardwareSpeedGroupConfigCommand( ShowCommand.ShowCliCommandClass ):
   syntax = "show hardware speed-group | SPEED-GROUP configurations"
   data = {
      "hardware": CliToken.Hardware.hardwareForShowMatcher,
      "speed-group": "Hardware speed group",
      "SPEED-GROUP": IntfRange.IntfRangeMatcher(
         explicitIntfTypes=( SpeedGroupType.type, ),
         dollarCompHelpText='Group list end' ),
      "configurations": "Show available speed configurations"
      }
   cliModel = SpeedGroupModel.SpeedGroupCaps

   @staticmethod
   def handler( mode, args ):
      speedGroup = []
      intfRange = args.get( 'SPEED-GROUP' )
      if intfRange:
         for name in IntfRange.intfListFromCanonical( [ intfRange ] ):
            speedGroup.append( IntfRange.intfTypeFromName( name )[ 1 ] )

      return getSpeedGroupCaps( speedGroup )

BasicCli.addShowCommandClass( ShowHardwareSpeedGroupConfigCommand )

def _updateSpeedGroupHelpDescForModular():
   """ Updates the hardware speed group type to have the correct help description
   for slot/group numbers. """
   SpeedGroupType.type.helpDesc = [ 'Slot number', 'Group number' ]

FruCli.registerModularSystemCallback( _updateSpeedGroupHelpDescForModular )

def Plugin( em ):
   global speedGroupConfigSliceDir
   global speedGroupStatusSliceDir
   speedGroupConfigSliceDir = ConfigMount.mount( em,
         "interface/archer/config/eth/phy/speedgroup/slice", "Tac::Dir", "wi" )
   speedGroupStatusSliceDir = LazyMount.mount( em,
         "interface/archer/status/eth/phy/speedgroup/slice", "Tac::Dir", "ri" )
