#!/usr/bin/env python
# Copyright (c) 2019 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

from __future__ import absolute_import, division, print_function

import BasicCli
import CliCommand
import CliMatcher
from CliToken.Ip import ipv4MatcherForShow
from CliToken.Ipv6 import ipv6MatcherForShow
import CliPlugin.AclCli as AclCli
import CliPlugin.SnmpCli as SnmpCli
from CliPlugin.SnmpModel import ShowSnmpHosts, SnmpInterface, SnmpChassis, \
      ShowSnmpUsers, Contact, SnmpLocation, SnmpBuffer, SnmpCommunityModel, \
      SnmpEngineIdModel, SnmpNotificationModel, SnmpViewModel, SnmpGroupModel, \
      SnmpTotalModel, SnmpMibTranslateModel, SnmpRegisteredOidsModel
from CliPlugin.AclCliModel import AllAclList
from CliPlugin import ShowTaskSchedulerModel
import ShowCommand
import Toggles.SnmpToggleLib

matcherGroup = CliMatcher.KeywordMatcher( 'group',
      helpdesc='Details of groups in SNMPv3' )
matcherHost = CliMatcher.KeywordMatcher( 'host',
      helpdesc='Show SNMP notification hosts' )
matcherMib = CliMatcher.KeywordMatcher( 'mib',
      helpdesc='Show SNMP MIB contents' )
matcherNotification = CliMatcher.KeywordMatcher( 'notification',
      helpdesc='Show SNMP traps' )
matcherScheduler = CliMatcher.KeywordMatcher( 'scheduler',
      helpdesc='SNMP scheduler statistics' )
matcherUser = CliMatcher.KeywordMatcher( 'user',
      helpdesc='Details of users in SNMPv3' )
matcherV2Mib = CliMatcher.KeywordMatcher( 'v2-mib',
      helpdesc='Show SNMPv2-MIB information' )
matcherView = CliMatcher.KeywordMatcher( 'view', helpdesc='Show SNMP views' )
groupNameMatcher = CliMatcher.PatternMatcher( SnmpCli.groupNameRe, helpname='WORD',
      helpdesc='Group name' )
oidMatcher = CliMatcher.PatternMatcher( pattern=SnmpCli.oidPattern,
         helpdesc='An object-ID (e.g., SNMPv2-MIB::sysDescr.0)', helpname='OID' )

#--------------------------------------------------------------------------------
# show snmp chassis
#--------------------------------------------------------------------------------
class SnmpChassisCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show snmp chassis'
   data = {
      'snmp': SnmpCli.snmpKwMatcher,
      'chassis': CliCommand.Node( 
         matcher=CliMatcher.KeywordMatcher( 'chassis',
            helpdesc='Show SNMP chassis' ),
         deprecatedByCmd='show snmp v2-mib chassis' )
   }
   handler = SnmpCli.showSnmpChassis
   cliModel = SnmpChassis

BasicCli.addShowCommandClass( SnmpChassisCmd )

#--------------------------------------------------------------------------------
# show snmp v2-mib chassis
#--------------------------------------------------------------------------------
class SnmpV2MibChassisCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show snmp v2-mib chassis'
   data = {
      'snmp': SnmpCli.snmpKwMatcher,
      'v2-mib': matcherV2Mib,
      'chassis': 'Show SNMPv2-MIB chassis',
   }
   handler = SnmpCli.showSnmpChassis
   cliModel = SnmpChassis

BasicCli.addShowCommandClass( SnmpV2MibChassisCmd )

#--------------------------------------------------------------------------------
# show snmp contact
#--------------------------------------------------------------------------------
class SnmpContactCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show snmp contact'
   data = {
      'snmp': SnmpCli.snmpKwMatcher,
      'contact': CliCommand.Node( 
         matcher=CliMatcher.KeywordMatcher( 'contact',
            helpdesc='Show SNMP contact' ),
         deprecatedByCmd='show snmp v2-mib contact' )
   }
   handler = SnmpCli.showSnmpContact
   cliModel = Contact

BasicCli.addShowCommandClass( SnmpContactCmd )

#--------------------------------------------------------------------------------
# show snmp v2-mib contact
#--------------------------------------------------------------------------------
class SnmpV2MibContactCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show snmp v2-mib contact'
   data = {
      'snmp': SnmpCli.snmpKwMatcher,
      'v2-mib': matcherV2Mib,
      'contact': 'Show SNMPv2-MIB contact',
   }
   handler = SnmpCli.showSnmpContact
   cliModel = Contact

BasicCli.addShowCommandClass( SnmpV2MibContactCmd )

#--------------------------------------------------------------------------------
# show snmp
#--------------------------------------------------------------------------------
class SnmpCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show snmp'
   data = {
      'snmp': SnmpCli.snmpKwMatcher,
   }
   handler = SnmpCli.showSnmpBasic
   cliModel = SnmpTotalModel

BasicCli.addShowCommandClass( SnmpCmd )

#--------------------------------------------------------------------------------
# show snmp community
#--------------------------------------------------------------------------------
class SnmpCommunityCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show snmp community'
   data = {
      'snmp': SnmpCli.snmpKwMatcher,
      'community': 'Show SNMP communities',
   }
   cliModel = SnmpCommunityModel
   handler = SnmpCli.showSnmpCommunity

BasicCli.addShowCommandClass( SnmpCommunityCmd )

#--------------------------------------------------------------------------------
# show snmp engineID
#--------------------------------------------------------------------------------
class SnmpEngineidCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show snmp engineID'
   data = {
      'snmp': SnmpCli.snmpKwMatcher,
      'engineID': 'Show SNMP engineID',
   }
   handler = SnmpCli.showSnmpEngineId
   cliModel = SnmpEngineIdModel

BasicCli.addShowCommandClass( SnmpEngineidCmd )

#--------------------------------------------------------------------------------
# show snmp group [ GROUP ]
#--------------------------------------------------------------------------------
class SnmpGroupCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show snmp group [ GROUP ]'
   data = {
      'snmp': SnmpCli.snmpKwMatcher,
      'group': matcherGroup,
      'GROUP': groupNameMatcher
   }
   handler = SnmpCli.showSnmpGroup
   cliModel = SnmpGroupModel

BasicCli.addShowCommandClass( SnmpGroupCmd )

#--------------------------------------------------------------------------------
# show snmp host
#--------------------------------------------------------------------------------
class SnmpHostCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show snmp host'
   data = {
      'snmp': SnmpCli.snmpKwMatcher,
      'host': CliCommand.Node( matcher=matcherHost, 
                               deprecatedByCmd='show snmp notification host' )
   }
   handler = SnmpCli.showSnmpHost
   cliModel = ShowSnmpHosts

BasicCli.addShowCommandClass( SnmpHostCmd )

#--------------------------------------------------------------------------------
# show snmp notification host
#--------------------------------------------------------------------------------
class SnmpNotificationHostCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show snmp notification host'
   data = {
      'snmp': SnmpCli.snmpKwMatcher,
      'notification': matcherNotification,
      'host': matcherHost,
   }
   handler = SnmpCli.showSnmpHost
   cliModel = ShowSnmpHosts

BasicCli.addShowCommandClass( SnmpNotificationHostCmd )

#--------------------------------------------------------------------------------
# show snmp source-interface
#--------------------------------------------------------------------------------
class SnmpSourceInterfaceCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show snmp source-interface'
   data = {
      'snmp': SnmpCli.snmpKwMatcher,
      'source-interface': CliCommand.Node(
         matcher=CliMatcher.KeywordMatcher( 'source-interface',
            helpdesc='Show SNMP source interface' ),
         deprecatedByCmd='show snmp local-interface' )
   }
   handler = SnmpCli.showSnmpSourceInterface
   cliModel = SnmpInterface

BasicCli.addShowCommandClass( SnmpSourceInterfaceCmd )

#--------------------------------------------------------------------------------
# show snmp local-interface
#--------------------------------------------------------------------------------
class SnmpLocalInterfaceCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show snmp local-interface'
   data = {
      'snmp': SnmpCli.snmpKwMatcher,
      'local-interface': 'Show SNMP local interface',
   }
   handler = SnmpCli.showSnmpSourceInterface
   cliModel = SnmpInterface

BasicCli.addShowCommandClass( SnmpLocalInterfaceCmd )

#--------------------------------------------------------------------------------
# show snmp location
#--------------------------------------------------------------------------------
class SnmpLocationCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show snmp location'
   data = {
      'snmp': SnmpCli.snmpKwMatcher,
      'location': CliCommand.Node(
         matcher=CliMatcher.KeywordMatcher( 'location',
            helpdesc='Show SNMP location' ),
         deprecatedByCmd='show snmp v2-mib location' )
   }
   handler = SnmpCli.showSnmpLocation
   cliModel = SnmpLocation

BasicCli.addShowCommandClass( SnmpLocationCmd )

#--------------------------------------------------------------------------------
# show snmp v2-mib location
#--------------------------------------------------------------------------------
class SnmpV2MibLocationCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show snmp v2-mib location'
   data = {
      'snmp': SnmpCli.snmpKwMatcher,
      'v2-mib': matcherV2Mib,
      'location': 'Show SNMPv2-MIB location',
   }
   handler = SnmpCli.showSnmpLocation
   cliModel = SnmpLocation

BasicCli.addShowCommandClass( SnmpV2MibLocationCmd )

#--------------------------------------------------------------------------------
# show snmp mib
#--------------------------------------------------------------------------------
class SnmpMibCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show snmp mib'
   data = {
      'snmp': SnmpCli.snmpKwMatcher,
      'mib': matcherMib,
   }
   cliModel = SnmpRegisteredOidsModel
   handler = SnmpCli.showSnmpMibRegistry

BasicCli.addShowCommandClass( SnmpMibCmd )

#--------------------------------------------------------------------------------
# show snmp mib translate TRANSLATE
#--------------------------------------------------------------------------------
class SnmpMibTranslateCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show snmp mib translate OID'
   data = {
      'snmp': SnmpCli.snmpKwMatcher,
      'mib': matcherMib,
      'translate': 'Translate between OID <-> name',
      'OID': oidMatcher,
   }
   cliModel = SnmpMibTranslateModel
   handler = SnmpCli.showSnmpMibTranslate

BasicCli.addShowCommandClass( SnmpMibTranslateCmd )

#--------------------------------------------------------------------------------
# show snmp net-snmp buffer
#--------------------------------------------------------------------------------
class SnmpNetSnmpBufferCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show snmp net-snmp buffer'
   data = {
      'snmp': SnmpCli.snmpKwMatcher,
      'net-snmp': 'Show NET-SNMP',
      'buffer': 'Show NET-SNMP buffer information',
   }
   handler = SnmpCli.showSnmpBuffer
   cliModel = SnmpBuffer
   hidden = True

BasicCli.addShowCommandClass( SnmpNetSnmpBufferCmd )

#--------------------------------------------------------------------------------
# show snmp ( notification | trap )
#--------------------------------------------------------------------------------
class SnmpNotificationCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show snmp ( notification | trap )'
   data = {
      'snmp': SnmpCli.snmpKwMatcher,
      'notification': matcherNotification,
      'trap': CliCommand.Node(
         matcher=CliMatcher.KeywordMatcher( 'trap', helpdesc='Show SNMP traps' ),
         deprecatedByCmd='show snmp notification' )
   }
   cliModel = SnmpNotificationModel
   handler = SnmpCli.showSnmpNotification

BasicCli.addShowCommandClass( SnmpNotificationCmd )

#--------------------------------------------------------------------------------
# Deprecate 'show snmp scheduler reset'.  It was a lazy implementation
# and doesn't fit the new one.
#--------------------------------------------------------------------------------
class SnmpSchedulerResetCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show snmp scheduler reset'
   data = {
      'snmp': SnmpCli.snmpKwMatcher,
      'scheduler': matcherScheduler,
      'reset': CliCommand.singleNode( CliMatcher.KeywordMatcher( 'reset',
         helpdesc='Reset scheduler statistics' ),
         deprecatedByCmd='clear agent Snmp task scheduler' ),
      }

   @staticmethod
   def handler( mode, args ):
      # Since it's deprecated, this should never get called, but
      # if it is, let's make sure that the user knows that it didn't
      # work.
      mode.addError( 'Use "clear agent Snmp task scheduler"' )

BasicCli.addShowCommandClass( SnmpSchedulerResetCmd )

#--------------------------------------------------------------------------------
# show snmp scheduler [ { ( verbose | internal | history ) } ]
#--------------------------------------------------------------------------------
class SnmpSchedulerCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show snmp scheduler [ { ( verbose | internal | history ) } ]'
   data = {
      'snmp': SnmpCli.snmpKwMatcher,
      'scheduler': matcherScheduler,
      'verbose': CliCommand.singleNode( CliMatcher.KeywordMatcher( 'verbose',
         helpdesc='Include more information' ) ),
      'internal': CliCommand.Node(
         matcher=CliMatcher.KeywordMatcher( 'internal',
            helpdesc='Display memory addresses of tasks' ),
         maxMatches=1, hidden=True ),
      'history': CliCommand.Node(
         matcher=CliMatcher.KeywordMatcher( 'history',
            helpdesc='Display scheduler event history' ),
         maxMatches=1, hidden=True ),
   }
   handler = SnmpCli.doShowSnmpScheduler
   privileged = True
   cliModel = ShowTaskSchedulerModel.Overall

BasicCli.addShowCommandClass( SnmpSchedulerCmd )

#--------------------------------------------------------------------------------
# show snmp user [ USER ]
#--------------------------------------------------------------------------------
class SnmpUserCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show snmp user [ USER ]'
   data = {
      'snmp': SnmpCli.snmpKwMatcher,
      'user': matcherUser,
      'USER': CliMatcher.PatternMatcher( pattern=SnmpCli.userNameRe, 
         helpdesc='SNMP user name', helpname='WORD' ),
   }
   handler = SnmpCli.showSnmpUser
   cliModel = ShowSnmpUsers

BasicCli.addShowCommandClass( SnmpUserCmd )

#--------------------------------------------------------------------------------
# show snmp view [ VIEW ]
#--------------------------------------------------------------------------------
class SnmpViewCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show snmp view [ VIEW ]'
   data = {
      'snmp': SnmpCli.snmpKwMatcher,
      'view': matcherView,
      'VIEW': CliMatcher.PatternMatcher( pattern=SnmpCli.viewNameRe,
         helpdesc='SNMP view name', helpname='WORD' ),
   }
   handler = SnmpCli.showSnmpView
   cliModel = SnmpViewModel

BasicCli.addShowCommandClass( SnmpViewCmd )

#--------------------------------------------------------------------------------
# show snmp mib { ( context CONTEXT ) | ( numeric ( index | all ) ) }
#               ( get | get-next { OID } ) | ( ( walk | table ) WALK_OID )
#--------------------------------------------------------------------------------
class ShowSnmpMibCmd( ShowCommand.ShowCliCommandClass ):
   syntax = ( 'show snmp mib '
               '[ { ( context CONTEXT ) | ( numeric ( index | all ) ) } ]'
               '( ( get | get-next ) { OID } ) | ( ( walk | table ) WALK_OID )' )
   data = {
      'snmp': SnmpCli.snmpKwMatcher,
      'mib': 'Show SNMP MIB contents',
      'context': 'Specify an SNMP context',
      'CONTEXT': CliCommand.singleNode(
         CliMatcher.PatternMatcher( SnmpCli.contextPattern,
            helpname='CONTEXT', helpdesc='An SNMP context, e.g., VRF name' ) ),
      'numeric': CliCommand.singleNode(
         CliCommand.CliMatcher.KeywordMatcher( 'numeric',
            helpdesc='Display OIDs numerically' ) ),
      'index': 'Display table indexes numerically',
      'all': 'Display complete object identifiers numerically',
      'get': 'Get one object',
      'get-next': 'Get the next object',
      'OID': CliMatcher.PatternMatcher( SnmpCli.oidPattern, helpname='OID',
         helpdesc='An object-ID (e.g., SNMPv2-MIB::sysDescr.0)' ),
      'walk': 'Walk a subtree',
      'table': 'Get the contents of a table',
      'WALK_OID': CliMatcher.PatternMatcher( SnmpCli.oidPattern, helpname='OID',
         helpdesc='An object-ID (e.g., IP-MIB::ipAddrTable)' ),
      }
   handler = SnmpCli.showSnmpMib
BasicCli.addShowCommandClass( ShowSnmpMibCmd )

#---------------------------------------------------------------------------
# show snmp  ( ipv4 access-list [IP4ACL] ) | ( ipv6 access-list [ IP6ACL ] )
#---------------------------------------------------------------------------
class ShowSnmpAccessCmd( ShowCommand.ShowCliCommandClass ):
   syntax = ( 'show snmp ( ( ipv4 access-list [ IP4ACL ] ) | '
              '( ipv6 access-list [ IP6ACL ] ) )' )
   data = {
      'snmp': SnmpCli.snmpKwMatcher,
      'ipv4': ipv4MatcherForShow,
      'ipv6': ipv6MatcherForShow,
      'access-list': AclCli.accessListKwMatcherForServiceAcl,
      'IP4ACL': AclCli.ipAclNameExpression,
      'IP6ACL': AclCli.ip6AclNameExpression,
      }
   handler = SnmpCli.showSnmpAccessLists
   cliModel = AllAclList

srcAddrACLsEnabled = Toggles.SnmpToggleLib.toggleSnmpSrcAddressACLsEnabled()
if srcAddrACLsEnabled:
   BasicCli.addShowCommandClass( ShowSnmpAccessCmd )
