#!/usr/bin/env python
# Copyright (c) 2019 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

from __future__ import absolute_import, division, print_function

import BasicCliModes
import CliCommand
import CliMatcher
import CliPlugin.AclCli as AclCli
from CliPlugin.IntfCli import Intf
import CliPlugin.VrfCli as VrfCli
import CliPlugin.SnmpCli as SnmpCli
import DscpCliLib
import HostnameCli
import Url
import Toggles.SnmpToggleLib

matcherEnable = CliMatcher.KeywordMatcher( 'enable',
      helpdesc='Enable traps to all configured recipients' )
matcherLocalInterface = CliMatcher.KeywordMatcher( 'local-interface',
      helpdesc='Configure the source interface for SNMP notifications' )
matcherTraps = CliMatcher.KeywordMatcher( 'traps',
      helpdesc='Enable traps to all configured recipients' )
engineIdMatcher = CliMatcher.PatternMatcher( pattern=SnmpCli.engineIdRe,
      helpname='WORD',
      helpdesc=( 'engine ID octet string (up to %s hexadecimal characters)' % 
         SnmpCli.engineIdLengthMax ) )
engineIdStrictMatcher = CliMatcher.PatternMatcher( pattern=SnmpCli.engineIdReStrict,
      helpname='WORD',
      helpdesc=SnmpCli.engineIdStrictHelpDesc )
groupNameMatcher = CliMatcher.PatternMatcher( pattern=SnmpCli.groupNameRe,
         helpdesc='SNMP group for the user', helpname='WORD' )

#--------------------------------------------------------------------------------
# [ no | default ] snmp-server chassis-id CHASSIS_ID
#--------------------------------------------------------------------------------
class SnmpServerChassisIdCmd( CliCommand.CliCommandClass ):
   syntax = 'snmp-server chassis-id CHASSIS_ID'
   noOrDefaultSyntax = 'snmp-server chassis-id ...'
   data = {
      'snmp-server': SnmpCli.snmpServerKwMatcher,
      'chassis-id': 'SNMP chassis identifier',
      'CHASSIS_ID': CliMatcher.StringMatcher(
         helpdesc='The chassis identifier string' ),
   }
   handler = SnmpCli.setChassis
   noOrDefaultHandler = SnmpCli.noChassis

BasicCliModes.GlobalConfigMode.addCommandClass( SnmpServerChassisIdCmd )

#--------------------------------------------------------------------------------
# ( no | default ) snmp-server
#--------------------------------------------------------------------------------
class SnmpServerCmd( CliCommand.CliCommandClass ):
   noOrDefaultSyntax = 'snmp-server'
   data = {
      'snmp-server': SnmpCli.snmpServerKwMatcher,
   }
   noOrDefaultHandler = SnmpCli.noSnmpServer

BasicCliModes.GlobalConfigMode.addCommandClass( SnmpServerCmd )

#--------------------------------------------------------------------------------
# [ no | default ] snmp-server community COMMUNITY [ view VIEW ] [ ro | rw ] 
#                                                  [ ipv6 IP6_ACL ] [ IP_ACL ]
#--------------------------------------------------------------------------------
class SnmpServerCommunityCmd( CliCommand.CliCommandClass ):
   syntax = ( 'snmp-server community COMMUNITY [ view VIEW ] [ ro | rw ] '
                                              '[ ipv6 IP6_ACL ] [ IP_ACL ]' )
   noOrDefaultSyntax = 'snmp-server community COMMUNITY ...'
   data = {
      'snmp-server': SnmpCli.snmpServerKwMatcher,
      'community': 'Community name configuration',
      'COMMUNITY': CliMatcher.PatternMatcher( pattern=SnmpCli.communityStringRe,
         helpdesc='Community name', helpname='WORD' ),
      'view': 'Limits community access to a view',
      'VIEW': CliMatcher.PatternMatcher( pattern=SnmpCli.viewNameRe,
         helpdesc='MIB view name', helpname='WORD' ),
      'ro': 'Only reads are permitted',
      'rw': 'Read-write access with this community string',
      'ipv6': 'Specify an IPv6 access list',
      'IP6_ACL': AclCli.standardIp6AclNameMatcher,
      'IP_ACL': AclCli.standardIpAclNameMatcher,
   }
   handler = SnmpCli.setCommunity
   noOrDefaultHandler = SnmpCli.noCommunity

BasicCliModes.GlobalConfigMode.addCommandClass( SnmpServerCommunityCmd )

#--------------------------------------------------------------------------------
# [ no | default ] snmp-server contact CONTACT
#--------------------------------------------------------------------------------
class SnmpServerContactCmd( CliCommand.CliCommandClass ):
   syntax = 'snmp-server contact CONTACT'
   noOrDefaultSyntax = 'snmp-server contact ...'
   data = {
      'snmp-server': SnmpCli.snmpServerKwMatcher,
      'contact': 'The sysContact string',
      'CONTACT': CliMatcher.StringMatcher(
                                 helpdesc='Person to contact about this system' ),
   }
   handler = SnmpCli.setContact
   noOrDefaultHandler = SnmpCli.noContact

BasicCliModes.GlobalConfigMode.addCommandClass( SnmpServerContactCmd )

#--------------------------------------------------------------------------------
# [ no | default ] snmp-server enable traps [ TRAP_TYPE [ { SUBTYPE | TRAP } ] ]
#--------------------------------------------------------------------------------
class SnmpServerEnableTrapsCmd( CliCommand.CliCommandClass ):
   syntax = 'snmp-server enable traps [ TRAP_TYPE [ { SUBTYPE | TRAP } ] ]'
   noOrDefaultSyntax = syntax
   data = {
      'snmp-server': SnmpCli.snmpServerKwMatcher,
      'enable': 'Enable traps to all configured recipients',
      'traps': 'Enable traps to all configured recipients',
      'TRAP_TYPE': CliCommand.Node(
         CliMatcher.DynamicKeywordMatcher( SnmpCli.getNotificationTypes ),
         storeSharedResult=True ),
      'SUBTYPE': CliCommand.Node(
         CliMatcher.DynamicKeywordMatcher( SnmpCli.getNotificationSubtypes,
            passContext=True ),
         storeSharedResult=True ),
      'TRAP': CliCommand.Node(
         CliMatcher.DynamicKeywordMatcher( SnmpCli.getSpecificNotifications,
            passContext=True ),
         storeSharedResult=True )
   }
   handler = SnmpCli.enableNotifications
   noHandler = SnmpCli.disableNotifications
   defaultHandler = SnmpCli.defaultNotifications

BasicCliModes.GlobalConfigMode.addCommandClass( SnmpServerEnableTrapsCmd )

#--------------------------------------------------------------------------------
# [ no | default ] snmp-server engineID
#                       ( local | ( remote REMOTE [ udp-port UDP_PORT ] ) ) ENGINE_ID
#--------------------------------------------------------------------------------
class SnmpServerEngineidCmd( CliCommand.CliCommandClass ):
   syntax = ( 'snmp-server engineID '
              '( local | ( remote REMOTE [ udp-port UDP_PORT ] ) ) ENGINE_ID' )
   noOrDefaultSyntax = ( 'snmp-server engineID '
                         '( local | ( remote REMOTE [ udp-port UDP_PORT ] ) ) ...' )
   data = {
      'snmp-server': SnmpCli.snmpServerKwMatcher,
      'engineID': 'SNMPv3 engine ID configuration',
      'local': 'Local SNMP agent',
      'remote': 'Remote SNMP agent',
      'REMOTE': HostnameCli.IpAddrOrHostnameMatcher( ipv6=True,
         helpdesc='Hostname or IP address of remote SNMP notification host' ),
      'udp-port': "The remote SNMP notification host's UDP port number",
      'UDP_PORT': CliMatcher.IntegerMatcher( 0, 65535,
         helpdesc="The remote SNMP notification host's UDP port number" ),
      'ENGINE_ID': engineIdMatcher
   }
   handler = SnmpCli.setEngineId
   noOrDefaultHandler = SnmpCli.noEngineId

BasicCliModes.GlobalConfigMode.addCommandClass( SnmpServerEngineidCmd )

#--------------------------------------------------------------------------------
# [ no | default ] snmp-server extension OID HANDLER [ one-shot ]
#--------------------------------------------------------------------------------
class SnmpServerExtensionCmd( CliCommand.CliCommandClass ):
   syntax = 'snmp-server extension OID HANDLER [ one-shot ]'
   noOrDefaultSyntax = 'snmp-server extension OID ...'
   data = {
      'snmp-server': SnmpCli.snmpServerKwMatcher,
      'extension': 'Configure extension script to serve an OID range',
      'OID': CliMatcher.PatternMatcher( pattern='[0-9\\.]+',
         helpdesc='Extension root oid (fully numeric)', helpname='WORD' ),
      'HANDLER': Url.UrlMatcher( lambda fs: hasattr( fs, 'location_' ),
         'script location', acceptSimpleFile=False ),
      'one-shot': 'Use inefficient one-shot interface',
   }
   handler = SnmpCli.setExtension
   noOrDefaultHandler = SnmpCli.noExtension

BasicCliModes.GlobalConfigMode.addCommandClass( SnmpServerExtensionCmd )

#--------------------------------------------------------------------------------
# snmp-server group GROUP_NAME ( v1 | v2c | ( v3 AUTH_LEVEL ) ) [ context CONTEXT ] 
#         [ { ( read READ_VIEW ) | ( write WRITE_VIEW ) | ( notify NOTIFY_VIEW ) } ]
#--------------------------------------------------------------------------------
authLevels = { 
   'auth': 'Group using authentication but not privacy',
   'noauth': 'Group using neither authentication nor privacy',
   'priv': 'Group using both authentication and privacy',
   }

class SnmpServerGroupCmd( CliCommand.CliCommandClass ):
   syntax = ( 'snmp-server group GROUP_NAME ( v1 | v2c | ( v3 AUTH_LEVEL ) ) '
               '[ context CONTEXT ] '
               '[ { ( read READ_VIEW ) | ( write WRITE_VIEW ) | '
                  '( notify NOTIFY_VIEW ) } ]' )
   noOrDefaultSyntax = ( 'snmp-server group GROUP_NAME '
                         '( v1 | v2c | ( v3 AUTH_LEVEL ) ) ...' )
   data = {
      'snmp-server': SnmpCli.snmpServerKwMatcher,
      'group': 'SNMP USM group',
      'GROUP_NAME': groupNameMatcher,
      'v1': 'SNMP v1 security group',
      'v2c': 'SNMP v2c security group',
      'v3': 'SNMP USM group',
      'AUTH_LEVEL': CliMatcher.EnumMatcher( authLevels ),
      'context': 'Specify a context to associate with the group',
      'CONTEXT': CliMatcher.PatternMatcher( pattern=SnmpCli.contextNameRe,
         helpdesc='SNMP group context identifier', helpname='WORD' ),
      'read': CliCommand.singleNode( CliMatcher.KeywordMatcher( 'read',
         helpdesc='View to restrict read access' ) ),
      'READ_VIEW': CliMatcher.PatternMatcher( pattern=SnmpCli.viewNameRe,
         helpdesc='SNMP view name for reading', helpname='WORD' ),
      'write': CliCommand.singleNode( CliMatcher.KeywordMatcher( 'write',
         helpdesc='View to restrict write access' ) ),
      'WRITE_VIEW': CliMatcher.PatternMatcher( pattern=SnmpCli.viewNameRe,
         helpdesc='SNMP view name for writing', helpname='WORD' ),
      'notify': CliCommand.singleNode( CliMatcher.KeywordMatcher( 'notify',
         helpdesc='View to restrict notifications' ) ),
      'NOTIFY_VIEW': CliMatcher.PatternMatcher( pattern=SnmpCli.viewNameRe,
         helpdesc='SNMP view name for notifications', helpname='WORD' ),
   }
   handler = SnmpCli.setGroup
   noOrDefaultHandler = SnmpCli.noGroup

BasicCliModes.GlobalConfigMode.addCommandClass( SnmpServerGroupCmd )

#--------------------------------------------------------------------------------
# [ no | default ] snmp-server host HOST [ VRF ]
#                                        [ ( traps | informs ) ] 
#                                        [ version ( 1 | 2c | ( 3 AUTH_LEVEL ) ) ] 
#                                        SECURITY_NAME
#                                        [ udp-port UDP_PORT ]
#--------------------------------------------------------------------------------
v3AuthPrivLevels = { 
   'auth': 'SNMPv3 security level - authNoPriv',
   'noauth': 'SNMPv3 security level - noAuthNoPriv',
   'priv': 'Use the SNMPv3 authAndPriv security level',
   }

class SnmpServerHostCmd( CliCommand.CliCommandClass ):
   syntax = ( 'snmp-server host HOST [ VRF ] [ ( traps | informs ) ] '
              '[ version ( 1 | 2c | ( 3 AUTH_LEVEL ) ) ] '
              'SECURITY_NAME [ udp-port UDP_PORT ]' )
   noOrDefaultSyntax = syntax
   data = {
      'snmp-server': SnmpCli.snmpServerKwMatcher,
      'host': 'Notification destinations',
      'HOST': HostnameCli.IpAddrOrHostnameMatcher( ipv6=True,
         helpdesc='Hostname or IP address of SNMP notification host' ),
      'VRF': VrfCli.VrfExprFactory(
                helpdesc='Specify the VRF in which the host is configured' ),
      'traps': 'Use SNMP trap messages',
      'informs': 'Use SNMP inform messages',
      'version': 'Notification message SNMP version',
      '1': 'Use SNMPv1',
      '2c': 'Use SNMPv2c',
      '3': 'Use SNMPv3',
      'AUTH_LEVEL': CliMatcher.EnumMatcher( v3AuthPrivLevels ),
      'SECURITY_NAME': CliMatcher.PatternMatcher( pattern=SnmpCli.communityStringRe,
         helpdesc='Community or user name',
         helpname='WORD' ),
      'udp-port': 'UDP destination port for notification messages',
      'UDP_PORT': CliMatcher.IntegerMatcher( 0, 65535,
         helpdesc='UDP destination port for notification messages' ),
   }
   handler = SnmpCli.setHost
   noOrDefaultHandler = SnmpCli.noHost

BasicCliModes.GlobalConfigMode.addCommandClass( SnmpServerHostCmd )

#--------------------------------------------------------------------------------
# [ no | default ] snmp-server [ VRF ] ( local-interface | source-interface )
# IP_INTF
#--------------------------------------------------------------------------------
class SnmpServerLocalInterfaceIpintfCmd( CliCommand.CliCommandClass ):
   syntax = 'snmp-server [ VRF ] ( local-interface | source-interface ) IP_INTF'
   noOrDefaultSyntax = syntax.replace( 'IP_INTF', '...' )
   data = {
      'snmp-server': SnmpCli.snmpServerKwMatcher,
      'VRF': VrfCli.VrfExprFactory(
                helpdesc='Specify the VRF in which the source address is used' ),
      'local-interface': matcherLocalInterface,
      'source-interface': CliCommand.Node(
         matcher=CliMatcher.KeywordMatcher( 'source-interface',
            helpdesc='Configure the source interface for SNMP notifications' ),
         deprecatedByCmd='snmp-server local-interface' ),
      'IP_INTF': Intf.matcherWithIpSupport,
   }
   handler = SnmpCli.setSrcIntf
   noOrDefaultHandler = SnmpCli.noSrcIntf

BasicCliModes.GlobalConfigMode.addCommandClass( SnmpServerLocalInterfaceIpintfCmd )

#--------------------------------------------------------------------------------
# [ no | default ] snmp-server location LOCATION
#--------------------------------------------------------------------------------
class SnmpServerLocationCmd( CliCommand.CliCommandClass ):
   syntax = 'snmp-server location LOCATION'
   noOrDefaultSyntax = 'snmp-server location ...'
   data = {
      'snmp-server': SnmpCli.snmpServerKwMatcher,
      'location': 'The sysLocation string',
      'LOCATION': CliMatcher.StringMatcher(
         helpdesc='The physical location of this system' ),
   }
   handler = SnmpCli.setLocation
   noOrDefaultHandler = SnmpCli.noLocation

BasicCliModes.GlobalConfigMode.addCommandClass( SnmpServerLocationCmd )

#--------------------------------------------------------------------------------
# [ no | default ] snmp-server net-snmp buffer ( client | server )
#                                              ( send | receive ) RECEIVE
#--------------------------------------------------------------------------------
class SnmpServerNetSnmpBufferCmd( CliCommand.CliCommandClass ):
   syntax = ( 'snmp-server net-snmp buffer ( client | server ) ( send | receive ) '
               'SIZE' )
   noOrDefaultSyntax = ( 'snmp-server net-snmp buffer ( client | server ) '
                                                     '( send | receive ) ...' )
   data = {
      'snmp-server': SnmpCli.snmpServerKwMatcher,
      'server': 'Server buffer',
      'net-snmp': 'NET-SNMP configuration',
      'buffer': 'Modify the buffer size',
      'client': 'Client buffer',
      'send': 'Send buffer',
      'receive': 'Receive buffer',
      'SIZE': CliMatcher.StringMatcher( helpdesc='Enter the size of the buffer',
         helpname='INTEGER' ),
   }
   hidden = True
   handler = SnmpCli.setBuffer
   noOrDefaultHandler = SnmpCli.noBuffer

BasicCliModes.GlobalConfigMode.addCommandClass( SnmpServerNetSnmpBufferCmd )

#--------------------------------------------------------------------------------
# [ no | default ] snmp-server notification log entry limit LIMIT
#--------------------------------------------------------------------------------
class SnmpServerNotificationLogEntryLimitCmd( CliCommand.CliCommandClass ):
   syntax = 'snmp-server notification log entry limit LIMIT'
   noOrDefaultSyntax = 'snmp-server notification log entry limit ...'
   data = {
      'snmp-server': SnmpCli.snmpServerKwMatcher,
      'notification': 'Configure SNMP notifications',
      'log': 'Log of notifications sent by the local agent',
      'entry': 'Notifications held in the log',
      'limit': ( 'Maximum number of notifications in the log '
                 '(nlmConfigGlobalEntryLimit)' ),
      'LIMIT': CliMatcher.IntegerMatcher( 1,
         SnmpCli.SnmpConfig.notificationLogEntryLimitMax,
         helpdesc='Number of notifications' ),
   }
   handler = SnmpCli.setNotificationLogEntryLimit
   noOrDefaultHandler = SnmpCli.noNotificationLogEntryLimit

BasicCliModes.GlobalConfigMode.addCommandClass(
      SnmpServerNotificationLogEntryLimitCmd )

#--------------------------------------------------------------------------------
# [ no | default ] snmp-server objects OBJECTS disable
#--------------------------------------------------------------------------------
class SnmpServerObjectsDisableCmd( CliCommand.CliCommandClass ):
   syntax = 'snmp-server objects OBJECTS disable'
   noOrDefaultSyntax = 'snmp-server objects OBJECTS disable ...'
   data = {
      'snmp-server': SnmpCli.snmpServerKwMatcher,
      'objects': 'Disable implementation of a group of objects',
      # pylint: disable-msg=protected-access
      'OBJECTS': CliMatcher.DynamicKeywordMatcher( SnmpCli._getObjectsTypes ),
      'disable': 'Disable implementation of a group of objects',
   }
   handler = SnmpCli.disableObjects
   noOrDefaultHandler = SnmpCli.noDisableObjects

BasicCliModes.GlobalConfigMode.addCommandClass( SnmpServerObjectsDisableCmd )

#--------------------------------------------------------------------------------
# [ no | default ] snmp-server qos dscp DSCP
#--------------------------------------------------------------------------------
DscpCliLib.addQosDscpCommandClass( BasicCliModes.GlobalConfigMode,
                                   SnmpCli.setDscp, SnmpCli.noDscp,
                                   tokenProto=SnmpCli.snmpServerKwMatcher )

#--------------------------------------------------------------------------------
# [ no | default ] snmp-server transmit max-size MAXSIZE
#--------------------------------------------------------------------------------
class SnmpServerTransmitMaxSizeCmd( CliCommand.CliCommandClass ):
   syntax = 'snmp-server transmit max-size SIZE'
   noOrDefaultSyntax = 'snmp-server transmit max-size ...'
   data = {
      'snmp-server': SnmpCli.snmpServerKwMatcher,
      'transmit': 'Configuration related to transmission of SNMP messages',
      'max-size': 'Configure the maximum size of transmitted SNMP messages',
      'SIZE': CliMatcher.IntegerMatcher( SnmpCli.SnmpConfig.transmitMsgSizeMin,
         SnmpCli.SnmpConfig.transmitMsgSizeMax,
         helpdesc='Maximum number of bytes in SNMP message (UDP/TCP payload)' ),
   }
   handler = SnmpCli.setTransmitMsgSize
   noOrDefaultHandler = SnmpCli.noDefaultTransmitMsgSize

BasicCliModes.GlobalConfigMode.addCommandClass( SnmpServerTransmitMaxSizeCmd )

#--------------------------------------------------------------------------------
# [ no | default ] snmp-server transport tcp
#--------------------------------------------------------------------------------
class SnmpServerTransportTcpCmd( CliCommand.CliCommandClass ):
   syntax = 'snmp-server transport tcp'
   noOrDefaultSyntax = 'snmp-server transport tcp ...'
   data = {
      'snmp-server': SnmpCli.snmpServerKwMatcher,
      'transport': 'Enable snmpd transport layer protocol',
      'tcp': 'TCP',
   }
   handler = SnmpCli.setTcpTransport
   noHandler = SnmpCli.noTcpTransport
   defaultHandler = SnmpCli.setTcpTransport

BasicCliModes.GlobalConfigMode.addCommandClass( SnmpServerTransportTcpCmd )

#--------------------------------------------------------------------------------
# [ no | default ] snmp-server user USER GROUP_NAME 
#                            [ remote REMOTE [ udp-port UDP_PORT ] ]
#                            ( v1 | v2c |
#                            ( v3 [ localized ENGINE_ID ] [ AUTH_AND_PRIV_EXPR ] ) )
#--------------------------------------------------------------------------------
userAuthDict = { 
      'md5': 'User authenticated using HMAC MD5',
      'sha': 'Use HMAC SHA-1 algorithm for authentication',
      'sha224': 'Use HMAC SHA-224 algorithm for authentication',
      'sha256': 'Use HMAC SHA-256 algorithm for authentication',
      'sha384': 'Use HMAC SHA-384 algorithm for authentication',
      'sha512': 'Use HMAC SHA-512 algorithm for authentication',
   }

privacyAlgoDict = {
      'des': 'Use DES algorithm for encryption',
      'aes': 'Use AES-128 algorithm for encryption',
      # BUG222340: no 3des yet
      #'3des': 'Use 3DES algorithm for encryption',
      'aes192': 'Use AES-192 algorithm for encryption',
      'aes256': 'Use AES-256 algorithm for encryption',
   }

class AuthAndPrivExpr( CliCommand.CliExpression ):
   expression = 'auth AUTH_TYPE AUTH_PHRASE [ priv PRIV_TYPE PRIV_PHRASE ]'
   data = { 
      'auth': 'User authentication settings',
      'AUTH_TYPE': CliMatcher.EnumMatcher( userAuthDict ),
      'AUTH_PHRASE': CliCommand.Node(
         matcher=CliMatcher.PatternMatcher( pattern=SnmpCli.authPassPhraseRe,
            helpdesc='authentication passphrase for user', helpname='HEX_STRING' ),
         sensitive=True ),
      'priv': 'User privacy settings',
      'PRIV_TYPE': CliMatcher.EnumMatcher( privacyAlgoDict ),
      'PRIV_PHRASE': CliCommand.Node(
         matcher=CliMatcher.PatternMatcher( pattern=SnmpCli.privPassPhraseRe,
            helpdesc='privacy passphrase for user', helpname='HEX_STRING' ),
         sensitive=True ),
   }

class SnmpServerUserCmd( CliCommand.CliCommandClass ):
   syntax = ( 'snmp-server user USER GROUP_NAME '
                     '[ remote REMOTE [ udp-port UDP_PORT ] ]'
                     '( v1 | v2c | ( v3 [ localized ENGINE_ID ] '
                                       '[ AUTH_AND_PRIV_EXPR ] ) )' )
   noOrDefaultSyntax = ( 'snmp-server user USER GROUP_NAME '
                           '[ remote REMOTE [ udp-port UDP_PORT ] ] '
                           '( v1 | v2c | v3 ) ...' )
   data = {
      'snmp-server': SnmpCli.snmpServerKwMatcher,
      'user': 'SNMP user configuration',
      'USER': CliMatcher.PatternMatcher( pattern=SnmpCli.userNameRe,
         helpdesc='SNMP user name', helpname='WORD' ),
      'GROUP_NAME': groupNameMatcher,
      'remote': 'System where an SNMPv3 user is hosted',
      'REMOTE': HostnameCli.IpAddrOrHostnameMatcher( ipv6=True,
         helpdesc='Hostname or IP address of remote SNMP entity' ),
      'udp-port': 'UDP port used by the remote SNMP system',
      'UDP_PORT': CliMatcher.IntegerMatcher( 0, 65535,
         helpdesc="UDP port used by the remote SNMP system" ),
      'v1': 'SNMPv1 security',
      'v2c': 'SNMPv2c security',
      'v3': 'SNMPv3 security',
      'localized': 'Specify localized auth and privacy passphrases',
      'ENGINE_ID': engineIdStrictMatcher,
      'AUTH_AND_PRIV_EXPR': AuthAndPrivExpr
   }
   handler = SnmpCli.setUser
   noOrDefaultHandler = SnmpCli.noUser

BasicCliModes.GlobalConfigMode.addCommandClass( SnmpServerUserCmd )

#--------------------------------------------------------------------------------
# [ no | default ] snmp-server view VIEW OID_TREE ( included | excluded )
#--------------------------------------------------------------------------------
class SnmpServerViewCmd( CliCommand.CliCommandClass ):
   syntax = 'snmp-server view VIEW OID_TREE ( included | excluded )'
   noOrDefaultSyntax = 'snmp-server view VIEW [ OID_TREE ] ...'
   data = {
      'snmp-server': SnmpCli.snmpServerKwMatcher,
      'view': 'SNMPv2 MIB view configuration',
      'VIEW': CliMatcher.PatternMatcher( pattern=SnmpCli.viewNameRe,
         helpdesc='SNMP view name', helpname='WORD' ),
      'OID_TREE': CliMatcher.PatternMatcher( pattern=SnmpCli.oidTreeRe,
         helpdesc='MIB family name', helpname='WORD' ),
      'included': 'Include the named MIB family in the view',
      'excluded': 'Exclude the named MIB family from the view',
   }
   handler = SnmpCli.setView
   noOrDefaultHandler = SnmpCli.noView

BasicCliModes.GlobalConfigMode.addCommandClass( SnmpServerViewCmd )

#--------------------------------------------------------------------------------
# [ no | default ] snmp-server VRF
#--------------------------------------------------------------------------------
class SnmpServerVrfCmd( CliCommand.CliCommandClass ):
   syntax = 'snmp-server VRF'
   noOrDefaultSyntax = 'snmp-server VRF'
   data = {
      'snmp-server': SnmpCli.snmpServerKwMatcher,
      'VRF': VrfCli.VrfExprFactory(
                    helpdesc='Configure in which VRF SNMP is enabled' ),
   }
   handler = SnmpCli.setVrf
   noOrDefaultHandler = SnmpCli.noVrf

BasicCliModes.GlobalConfigMode.addCommandClass( SnmpServerVrfCmd )

#-----------------------------------------------------------------------------
# [ no | default ] snmp-server ( ( ipv4 access-list IP_ACL ) |
#                                ( ipv6 access-list IP6_ACL ) ) [ VRF ]
#-----------------------------------------------------------------------------
class SnmpServerIpAccessListCmd( CliCommand.CliCommandClass ):
   syntax = ( 'snmp-server ( ( ipv4 access-list IP_ACL ) |'
                           ' ( ipv6 access-list IP6_ACL ) )'
                           ' [ VRF ]' )
   noOrDefaultSyntax = syntax
   data = {
      'snmp-server': SnmpCli.snmpServerKwMatcher,
      'ipv4': AclCli.ipv4KwForServiceAclMatcher,
      'ipv6': AclCli.ipv6KwMatcherForServiceAcl,
      'access-list': AclCli.accessListKwMatcherForServiceAcl,
      'IP_ACL': AclCli.ipAclNameMatcher,
      'IP6_ACL': AclCli.ip6AclNameMatcher,
      'VRF': VrfCli.VrfExprFactory(
                helpdesc='VRF in which to apply the access control list' ),
   }
   handler = SnmpCli.setIpAcl
   noOrDefaultHandler = SnmpCli.noIpAcl

srcAddrACLsEnabled = Toggles.SnmpToggleLib.toggleSnmpSrcAddressACLsEnabled()
if srcAddrACLsEnabled:
   BasicCliModes.GlobalConfigMode.addCommandClass( SnmpServerIpAccessListCmd )
