# Copyright (c) 2017 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

from CliModel import Model
from CliModel import Submodel
from CliModel import Int
from CliModel import Float
from CliModel import Str
from CliModel import List
from CliModel import OrderedDict
import datetime
import LinuxCmdConstants

def parseTimeDelta( timeDelta ):
   upTime = datetime.timedelta( seconds=timeDelta )
   upTimeMin = upTime.seconds / 60
   upTimeHr = upTimeMin / 60
   upTimeMin = upTimeMin % 60
   upTimeSec = upTime.seconds % 60
   return upTime.days, upTimeHr, upTimeMin, upTimeSec

class SystemTimeInfo( Model ):
   currentTime = Float( help='the current timestamp' )
   upTime = Float( help='the length of time the system has been running' )
   users = Int( help='the number of users that are currently logged on' )
   loadAvg = List( valueType=float, 
                   help='the average number of processes in the system '
                        'that are in a runnable or uninterruptable state '
                        'for the past 1, 5, and 15 minutes.' )

   def convertUptime( self ):
      # convert UTC time to local time
      timeInfo = datetime.datetime.fromtimestamp(
                                    self.currentTime ).strftime( ' %H:%M:%S up ' )
      # calculate the total system running time
      ( upTimeDay, upTimeHr, upTimeMin, _ ) = parseTimeDelta( self.upTime )
      if upTimeDay > 0:
         timeInfo += '%d day%s, ' % ( upTimeDay, '' if upTimeDay == 1 else 's' )
      if upTimeHr > 0:
         timeInfo += '{:2d}:{:02d}, '.format( upTimeHr, upTimeMin )
      else:
         timeInfo += '%d min, ' % upTimeMin
      # get the logging on users
      timeInfo += '%2d user%s, ' % ( self.users, '' if self.users == 1 else 's' )
      # get the load info
      timeInfo += ' load average: {:.2f}, {:.2f}, {:.2f}'.format(
                        self.loadAvg[ 0 ], self.loadAvg[ 1 ], self.loadAvg[ 2 ] )
      return timeInfo

   def render( self ):
      print self.convertUptime()

class ProcessInfo( Model ):
   cmd = Str( help='the name of the command that launched the process' )
   cpuPct = Float( help='the percentage of CPU used by the process' )
   memPct = Float( help='the percentage of memory used by the process' )
   ttyName = Str( help='the terminal associated with the process' )
   state = Str( help='the process state code' )
   startTime = Float( help='the starting time of the process' )
   totalActiveTime = Int( help='the length of time the process has been active' )

class SystemProcesses( Model ):
   timeInfo = Submodel( SystemTimeInfo, help='uptime and load averages info' )
   processes = OrderedDict( keyType=int, valueType=ProcessInfo,
                            help='Mappping of a process pid to '
                                 'the detailed process information' )
   
   def _convertToStartTime( self, timestamp ):
      timeDelta = self.timeInfo.currentTime - timestamp
      upTime = datetime.timedelta( seconds=timeDelta )
      if upTime.days > 0:
         return datetime.datetime.fromtimestamp( timestamp ).strftime( '  %b %d' )
      else:
         return datetime.datetime.fromtimestamp( timestamp ).strftime( '%H:%M:%S' )

   def _convertToTime( self, timeDelta ):
      timeStr = ''
      ( upDay, upHr, upMin, upSec ) = parseTimeDelta( timeDelta )
      if upDay > 0:
         timeStr += '%d-' % upDay
      timeStr += '{:02}:{:02}:{:02}'.format( upHr, upMin, upSec )
      return timeStr

   def _convertCpuPct( self, cpuPct ):
      return int( cpuPct ) if cpuPct > 99.9 else cpuPct
   
   def _getProcOutputFormat( self, pid, procInfo, fmTemplate ):
      ''' Adjust the process output format, the length of left-padded columns 
          may change due to the expansion of previous columns.
          For example, if the pid length is larger than format length 5, 
          the length of tty will be reduced to keep the rest output format
               PID %CPU %MEM TT       STAT  STARTED     TIME CMD
             17111 13.8  0.1 pts/4    Ssl+ 19:55:05 00:00:19 Cli [interac
             171111 12.5  0.1 pts/3   S+   17:33:05 00:00:10 Fake process '''
      # amount of rightward displacement because a column was not wide enough
      rightWard = 0
      # amount of room the TT, STAT column can provide to correct 
      # the rightward push from previous columns
      ttLeftWard = 0
      statLeftWard = 0
      ttLength = LinuxCmdConstants.TT_LENGTH
      statLength = LinuxCmdConstants.STAT_LENGTH

      pidLen = len( str( pid ) )
      cpuPctLen = len( str( self._convertCpuPct( procInfo.cpuPct ) ) )
      memPctLen = len( str( procInfo.memPct ) )
      ttyLen = len( procInfo.ttyName )
      statLen = len( procInfo.state )

      # get the total expansion of column PID, CPU, MEM, TT, STAT
      # and the possible shrinkage from left-padded columns TT and STAT
      if pidLen > LinuxCmdConstants.PID_LENGTH:
         rightWard += pidLen - LinuxCmdConstants.PID_LENGTH
      if cpuPctLen > LinuxCmdConstants.PCPU_LENGTH:
         rightWard += cpuPctLen - LinuxCmdConstants.PCPU_LENGTH
      if memPctLen > LinuxCmdConstants.PMEM_LENGTH:
         rightWard += memPctLen - LinuxCmdConstants.PMEM_LENGTH
      if ttyLen  > LinuxCmdConstants.TT_LENGTH:
         rightWard += ttyLen - LinuxCmdConstants.TT_LENGTH
      else:
         ttLeftWard = LinuxCmdConstants.TT_LENGTH - ttyLen 
      if statLen > LinuxCmdConstants.STAT_LENGTH:
         rightWard += statLen - LinuxCmdConstants.STAT_LENGTH
      else:
         statLeftWard = LinuxCmdConstants.STAT_LENGTH - statLen 

      # if the expansion is smaller than the TT shrinkage, 
      # only need to reset the first TT column length
      if rightWard <= ttLeftWard:
         ttLength = LinuxCmdConstants.TT_LENGTH - rightWard
      # if the expansion is smaller than the TT and STAT shrinkage,
      # reset both columns length
      elif rightWard <= ( ttLeftWard + statLeftWard ):
         ttLength = ttyLen
         rest = rightWard - ( LinuxCmdConstants.TT_LENGTH - ttLength )
         statLength = LinuxCmdConstants.STAT_LENGTH - rest
      # if the expansion is larger than the total shrinkage,
      # reset TT and STAT columns length to the actual content length
      else:
         ttLength = ttyLen
         statLength = statLen

      return fmTemplate % ( ttLength, statLength )

   def render( self ):
      self.timeInfo.render()
      fmTemplate = '{:>%d} {:>%d} {:>%d} %s %s {:>%d} {:>%d} {}' % (
                   LinuxCmdConstants.PID_LENGTH, LinuxCmdConstants.PCPU_LENGTH,
                   LinuxCmdConstants.PMEM_LENGTH, '{:%d}', '{:%d}',
                   LinuxCmdConstants.STARTED_LENGTH, LinuxCmdConstants.TIME_LENGTH )
      
      headerFm = fmTemplate % ( LinuxCmdConstants.TT_LENGTH,
                                LinuxCmdConstants.STAT_LENGTH )
      print headerFm.format( "PID", "%CPU", "%MEM", "TT",
                             "STAT", "STARTED", "TIME", "CMD" )
      
      # pylint: disable-msg=no-member
      for ( key, processInfo ) in self.processes.items():
         procFm = self._getProcOutputFormat( key, processInfo, fmTemplate )
         print procFm.format( key, self._convertCpuPct( processInfo.cpuPct ), 
                              processInfo.memPct, processInfo.ttyName, 
                              processInfo.state, 
                              self._convertToStartTime( processInfo.startTime ), 
                              self._convertToTime( processInfo.totalActiveTime ),
                              processInfo.cmd )

class ThreadsInfo( Model ):
   total = Int( help='the number of total tasks or threads' )
   run = Int( help='the total number of running tasks' )
   sleep = Int( help='the total number of sleeping tasks' )
   stop = Int( help='the total number of stopped tasks' )
   zombie = Int( help='the total number of zombie tasks' )

class CpuInfo( Model ):
   user = Float( help='the percentage of time running un-niced user processes' )
   system = Float( help='the percentage of time running kernel processes' )
   nice = Float( help='the percentage of time running niced user processes' )
   idle = Float( help='the percentage of idle time' )
   ioWait = Float( help='the percentage of time waiting for I/O completion' )
   hwIrq = Float( help='the percentage of time spent servicing hardware interrupts' )
   swIrq = Float( help='the percentage of time spent servicing software interrupts' )
   stolen = Float( help='the percentage of time stolen from '
                        'this vm by the hypervisor' )

class MemInfo( Model ):
   memTotal = Int( help='the amount of total memory in Kbytes' )
   memUsed = Int( help='the amount of memory currently being used in Kbytes' )
   memFree = Int( help='the amount of unused memory in Kbytes' )
   memBuffer = Int( help='the amount of memory has been used in Kbytes' )

class SystemMemSum( Model ):
   physicalMem = Submodel( MemInfo, help='system physical memory info' )
   swapMem = Submodel( MemInfo, help='system virtual memory info' )

class ProcessActivityInfo( Model ):
   cmd = Str( help='the command that launched the process' )
   userName = Str( help='the effective user name of the owner of the processes' )
   priority = Str( help='the process priority' )
   niceValue = Int( help='the nice value of the process corresponding to '
                         'a user-space concept' )
   virtMem = Str( help='the amount of virtual memory used by the process' )
   residentMem = Str( help='the resident memory size' )
   sharedMem = Str( help='the shared memory used by the process' )
   status = Str( help='the process status' )
   cpuPctType = Str( help='CPU percentage display format' )
   cpuPct = Float( help='the percentage of CPU used by the process' )
   memPct = Float( help='the percentage of memory used by the process' )
   activeTime = Int( help='the length of time the process has been active '
                          'in Centiseconds' )

class SystemTopInfo( Model ):
   timeInfo = Submodel( SystemTimeInfo, help='uptime and load averages info' )
   threadsStateInfo = Submodel( ThreadsInfo, 
                                help='the number of threads for each state' )
   cpuInfo = OrderedDict( keyType=str, valueType=CpuInfo, 
                          help='Mapping of each CPU to its state percentages '
                               'based on the interval since the last refresh' )
   memInfo = Submodel( SystemMemSum, 
                       help='System physical and virtual memory usage' )
   processes = OrderedDict( keyType=int, valueType=ProcessActivityInfo,
                            help='Mapping of a process pid to'
                                 'the detailed process activity' )

   def _convertToTimePlus( self, timeDelta ):
      if timeDelta < 0:
         return '?'

      timeStrList = []
      sec = timeDelta / 100
      m = sec / 60
      h = m / 60
      # generate all timeStr format
      timeStrList.append( '{}:{:02}.{:02}'.format( m, sec % 60, timeDelta % 100 ) )
      timeStrList.append( '{}:{:02}'.format( m, sec % 60 ) )
      timeStrList.append( '{},{:02}'.format( h, m % 60 ) )
      timeStrList.append( '{}h'.format( h ) )
      timeStrList.append( '{}d'.format( h / 24 ) )
      timeStrList.append( '{}w'.format( h / 24 / 7 ) )

      for s in timeStrList:
         if LinuxCmdConstants.TOP_TIME_LENGTH >= len( s ):
            return s
      return '?'

   def _getUnitType( self, total ):
      if total > 99999999 * 1024:
         return 'GiB'
      elif total > 99999999:
         return 'MiB'
      else:
         return 'KiB'

   def render( self ):
      print 'top -%s' % self.timeInfo.convertUptime()
      print ( 'Tasks: {:3d} total, {:3d} running, {:3d} sleeping, '
              '{:3d} stopped, {:3d} zombie'.format( self.threadsStateInfo.total, 
                 self.threadsStateInfo.run, self.threadsStateInfo.sleep, 
                 self.threadsStateInfo.stop, self.threadsStateInfo.zombie ) )
      # pylint: disable-msg=no-member
      for ( cpuName, cpuInfo ) in self.cpuInfo.items():
         print ( '{}:{:5.1f} us,{:5.1f} sy,{:5.1f} ni,{:5.1f} id,{:5.1f} wa,'
                 '{:5.1f} hi,{:5.1f} si,{:5.1f} st'.format( cpuName, cpuInfo.user,
                    cpuInfo.system, cpuInfo.nice, cpuInfo.idle, cpuInfo.ioWait,
                    cpuInfo.hwIrq, cpuInfo.swIrq, cpuInfo.stolen ) )
      
      unitType = self._getUnitType( self.memInfo.physicalMem.memTotal )
      scale = LinuxCmdConstants.UNIT_TABLE[ unitType ]
      print '{} Mem:  {:8d} total, {:8d} used, {:8d} free, {:8d} buffers'.format( 
            unitType, self.memInfo.physicalMem.memTotal / scale,
            self.memInfo.physicalMem.memUsed / scale,
            self.memInfo.physicalMem.memFree / scale,
            self.memInfo.physicalMem.memBuffer / scale )
      print '{} Swap: {:8d} total, {:8d} used, {:8d} free, {:8d} cached\n'.format(
            unitType, self.memInfo.swapMem.memTotal / scale,
            self.memInfo.swapMem.memUsed / scale,
            self.memInfo.swapMem.memFree / scale,
            self.memInfo.swapMem.memBuffer / scale )
      
      fm = ( "{:>%d} {:%d} {:>%d} {:>%d} {:>%d} {:>%d} {:>%d} {:>%d} "
             "{:>%d} {:>%d} {:>%d} {}" % ( LinuxCmdConstants.PID_LENGTH,
             LinuxCmdConstants.TOP_USER_LENGTH, LinuxCmdConstants.TOP_PR_LENGTH,
             LinuxCmdConstants.TOP_NI_LENGTH, LinuxCmdConstants.TOP_VIRT_LENGTH,
             LinuxCmdConstants.TOP_RES_LENGTH, LinuxCmdConstants.TOP_SHR_LENGTH,
             LinuxCmdConstants.TOP_S_LENGTH, LinuxCmdConstants.TOP_PCPU_LENGTH,
             LinuxCmdConstants.PMEM_LENGTH, LinuxCmdConstants.TOP_TIME_LENGTH ) )
      print fm.format( "PID", "USER", "PR", "NI", "VIRT", "RES", 
                       "SHR", "S", "%CPU", "%MEM", "TIME+ ", "COMMAND" )

      # pylint: disable-msg=no-member
      for ( pid, activityInfo ) in self.processes.items(): 
         print fm.format( pid, activityInfo.userName, activityInfo.priority, 
                          activityInfo.niceValue, activityInfo.virtMem, 
                          activityInfo.residentMem, activityInfo.sharedMem,
                          activityInfo.status, 
                          activityInfo.cpuPctType.format( activityInfo.cpuPct ), 
                          '{:.1f}'.format( activityInfo.memPct ), 
                          self._convertToTimePlus( activityInfo.activeTime ), 
                          activityInfo.cmd )
