# Copyright (c) 2019 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

from __future__ import absolute_import, division, print_function

import re

import BasicCli
import CliCommand
import CliMatcher
import CliModel
import CliToken.Cli
import ShowCommand
import Tac
import Tracing

__defaultTraceHandle__ = Tracing.Handle( 'ShowCliCommands' )
t3 = Tracing.trace3

ruleKeyCompiledRe = re.compile( '(\\W|no|default)*' )

class Clis( CliModel.Model ):
   clis = CliModel.List( valueType=str, help='CLI commands in the current mode' )

   def render( self ):
      for cli in self.clis:
         print( cli )

class AllClis( CliModel.Model ):
   allClis = CliModel.Dict(
               keyType=str, valueType=Clis,
               help='CLI commands in all modes, keyed by mode name' )

   def render( self ):
      for mode, clis in sorted( self.allClis.items() ):
         for cli in clis.clis:
            print( '%s: %s' % ( mode, cli ) )

verbatimOrTruncate = object()

class ShowCliCommandsOptionsExpr( CliCommand.CliExpression ):
   '''Expression is: options in any order,
   but 'verbatim' and 'truncate' are mutually exclusive.
   '''
   expression = '{ hidden | functions | ( truncate LENGTH ) | verbatim }'
   data = {
         'verbatim': CliCommand.singleKeyword( 'verbatim',
                                                'Show syntaxes as-is',
                                                sharedMatchObj=verbatimOrTruncate ),
         'hidden': CliCommand.singleKeyword( 'hidden',
                                              'Show hidden commands as well',
                                              hidden=True ),
         'functions': CliCommand.singleKeyword( 'functions',
                                                 'Show callback functions as well',
                                                 hidden=True ),
         'truncate': CliCommand.singleKeyword( 'truncate',
                                                'Truncate commands to N characters '
                                                '(200 by default, 0 means no limit)',
                                                sharedMatchObj=verbatimOrTruncate ),
         'LENGTH': CliMatcher.IntegerMatcher( 0, 2**32 - 1,
                                               helpdesc='Command length limit' ),
   }

   @staticmethod
   def adapter( mode, args, argsList ):
      args.pop( 'truncate', None )
      options = {
            'verbatim': bool( args.pop( 'verbatim', False ) ),
            'hidden': bool( args.pop( 'hidden', False ) ),
            'functions': bool( args.pop( 'functions', False ) ),
            'truncate': args.pop( 'LENGTH', 200 ),
      }
      args[ 'OPTIONS' ] = options

#-----------------------------------------------------------------------------------
# show cli commands [ OPTIONS ]
#-----------------------------------------------------------------------------------
def ruleKey( s ):
   # start with the first word character after any combination of no|default
   m = ruleKeyCompiledRe.match( s )
   return s[ m.end() : ]

def getModeSyntaxes( mode, args ):
   clis = list( mode.getSyntaxes( **args[ 'OPTIONS' ] ) )
   for modeletClass in getattr( mode, 'modeletClasses', [] ):
      clis.extend( modeletClass.getSyntaxes( **args[ 'OPTIONS' ] ) )
   t3( 'found', len( clis ), 'rules for', mode )
   return Clis( clis=sorted( clis, key=ruleKey ) )

class ShowCliCommands( ShowCommand.ShowCliCommandClass ):
   syntax = 'show cli commands [ OPTIONS ]'
   data = {
         'cli': CliToken.Cli.cliForShowMatcher,
         'commands': 'Show CLI commands',
         'OPTIONS': ShowCliCommandsOptionsExpr,
   }
   cliModel = Clis
   privileged = True

   @staticmethod
   def handler( mode, args ):
      """Show rules from the current mode."""
      lastMode = mode.session_.modeOfLastPrompt() or mode
      return getModeSyntaxes( lastMode, args )

BasicCli.addShowCommandClass( ShowCliCommands )

#-----------------------------------------------------------------------------------
# show cli commands all-modes [ OPTIONS ]
#-----------------------------------------------------------------------------------
class ShowCliCommandsAllModes( ShowCommand.ShowCliCommandClass ):
   syntax = 'show cli commands all-modes [ OPTIONS ]'
   data = {
         'cli': CliToken.Cli.cliForShowMatcher,
         'commands': 'Show CLI commands',
         'all-modes': 'Show commands from all modes',
         'OPTIONS': ShowCliCommandsOptionsExpr,
   }
   cliModel = AllClis
   privileged = True

   @staticmethod
   def handler( mode, args ):
      """Show rules from all modes in the system."""
      submodes = [ BasicCli.UnprivMode, BasicCli.EnableMode,
            BasicCli.GlobalConfigMode ]
      submodes.extend( BasicCli.ConfigModeBase.__subclasses__() )
      visited = set()
      model = AllClis()
      while submodes:
         submode = submodes.pop( 0 )
         if submode in visited:
            continue

         visited.add( submode )
         model.allClis[ submode.__name__ ] = getModeSyntaxes( submode, args )
         submodes = submode.__subclasses__() + submodes
      return model

BasicCli.addShowCommandClass( ShowCliCommandsAllModes )
