# Copyright (c) 2008-2010, 2011 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

from __future__ import absolute_import, division, print_function

import os
import sys

import BasicCli
import BasicCliUtil
import CmdExtension
import CliMatcher
import CliCommand
import Tac
import TerminalUtil

#--------------------------------------------------
# Field Engineering Bash Command (with timeout)
#--------------------------------------------------

def verifyTimeout( *args, **kwargs ):
   return '<seconds>' in kwargs[ 'args' ]

def bashGetMode( *args, **kwargs ):
   assert args, "Unable to find the CLI Mode"
   return args[ 0 ]

bashArgMatcher = CliMatcher.StringMatcher( helpname="ARG",
                                 helpdesc="Executable arguments and/or switches" )
timeoutMatcher = CliMatcher.IntegerMatcher( 1, 32000, helpdesc='seconds' )

class BashCmd( CliCommand.CliCommandClass ):
   syntax = """bash [ timeout <seconds> ] [ <arg> ]"""
   data = { 'bash': 'field engineer bash command',
            'timeout': 'Set the maximum number of seconds a command will run',
            '<seconds>': timeoutMatcher,
            '<arg>': bashArgMatcher }

   # pylint: disable-msg=C0322
   @staticmethod
   @BasicCliUtil.EapiIncompatible( overrideFn=verifyTimeout, getModeFn=bashGetMode,
         additionalMsg="To run a bash command over the Command API, "
                       " use the 'timeout  <seconds>' option." )
   def handler( mode, args ):
      timeout = args.get( "<seconds>", None )
      cmdArgs = args.get( "<arg>", None )

      cliCmdExt = CmdExtension.getCmdExtender()
      # Set terminal width to automatic and restore after exiting bash
      with TerminalUtil.NoTerminalSettings( mode.session_.terminalCtx_ ):
         if cmdArgs:
            try:
               if timeout:
                  # CliServer doesn't allow SIGALRM, so use timeout
                  # under bash mode.
                  newargs = ( "timeout -k3 -s9 %d " % timeout ) + cmdArgs
               else:
                  newargs = cmdArgs

               cmd = [ "/bin/bash", "-c", newargs ]
               cliCmdExt.runCmd( cmd,
                                 mode.session,
                                 stdin=sys.stdin,
                                 stderr=sys.stderr, stdout=sys.stdout,
                                 env=os.environ )
            except Tac.SystemCommandError, e:
               if timeout and e.error == -9:
                  # timeout returned -9, meaning original command timed out
                  mode.addError( "Command '%s' timed out" % cmdArgs )
               else:
                  mode.addError( "'%s' returned error code: %s" % ( cmdArgs,
                              str( e.error ) ) )
                  if e.output:
                     print( e.output )
            except EnvironmentError as e:
               mode.addError( e.strerror )
         else:
            # Enable Ctrl-Z in the bash command as it has been disabled in Cli.
            with TerminalUtil.CtrlZ( True ):
               if not os.environ.get( "A4_CHROOT" ):
                  cmd = [ '/bin/bash', '-l' ]
               else: # in our workspaces, when running btests
                  # make sure a funky PS1 definition does not mess up the prompt
                  cmd = [ '/bin/bash', '--noprofile', '-l' ]
               env = os.environ.copy()
               env[ 'SHELL' ] = '/bin/bash'
               env[ 'EOSCLIRUNNING' ] = '1'
               try:
                  p = cliCmdExt.subprocessPopen( cmd, mode.session, env=env,
                                                 stdin=sys.stdin,
                                                 close_fds=True )
                  p.wait()
               except EnvironmentError as e:
                  mode.addError( e.strerror )

#
# Register the bash command by adding a new class
#
BasicCli.EnableMode.addCommandClass( BashCmd )
