#!/usr/bin/env python
# Copyright (c) 2018 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

from datetime import datetime

from CliModel import Int, List, Model, Str, Dict
from MgmtSecurityLib import Lifetime
from ReversibleSecretCli import encodeKey

def displayDate( date ):
   fmt = "%B %d %Y, %H:%M UTC"
   return datetime.utcfromtimestamp( date ).strftime( fmt )

def displayExpiration( lifetimeEnd ):
   if lifetimeEnd == 0:
      return "Does not expire"
   return displayDate( lifetimeEnd )

def displayDuration( start, end ):
   """
   The start / end times are in UTC seconds. Convert them back into datetime
   string representations or 'infinite'.
   """
   if Lifetime( start, end ).isInfinite():
      return 'infinite'
   return "%s to %s" % ( displayDate( start ), displayDate( end ) )

class SecretModel( Model ):
   secretID = Int( help='The id of the secret' )
   secret = Str( help='The secret used for authentication' )
   rxLifetimeStart = Int( help='The time (UTC) when the secret is valid for '
                          'receiving data' )
   rxLifetimeEnd = Int( help='The time (UTC) when the secret is no longer valid for '
                        'receiving data' )
   txLifetimeStart = Int( help='The time (UTC) when the secret is valid for '
                          'transmitting data' )
   txLifetimeEnd = Int( help='The time (UTC) when the secret is no longer valid for '
                        'transmitting data' )

   def render( self ):
      print "   ID %s" % self.secretID
      print "      Secret: %s" % encodeKey( self.secret,
                                            algorithm='MD5' )
      print "      Receive lifetime: %s" % displayDuration( self.rxLifetimeStart,
                                                            self.rxLifetimeEnd )
      print "      Transmit lifetime: %s" % displayDuration( self.txLifetimeStart,
                                                             self.txLifetimeEnd )

class SharedSecretProfile( Model ):
   profileName = Str( help='The name associated with the shared secret profile',
                      optional=True )
   currentRxId = Int( help='The id of the current secret being used for '
                          'receival', optional=True )
   currentTxId = Int( help='The id of the current secret being used for '
                          'transmition', optional=True )
   currentRxExpiration = Int( help='The date and time when the current receive '
                              'secret expires', optional=True )
   currentTxExpiration = Int( help='The date and time when the current transmit '
                              'secret expires', optional=True )
   rxRotation = List( help='The order in which receive secrets will be used',
                      valueType=str )
   txRotation = List( help='The order in which transmit secrets will be used',
                      valueType=str )
   secrets = List( help='The secrets currently configured in the profile',
                   valueType=SecretModel )

   def render( self ):
      if self.profileName is None:
         return

      print 'Profile: %s\n' % self.profileName
      if self.currentRxId is not None:
         print 'Current receive secret: ID: %s, Expires: %s' % (
           self.currentRxId, displayExpiration( self.currentRxExpiration ) )
      else:
         print 'No current receive secret'
      if self.currentTxId is not None:
         print 'Current transmit secret: ID: %s, Expires: %s\n' % (
            self.currentTxId, displayExpiration( self.currentTxExpiration ) )
      else:
         print 'No current transmit secret\n'

      print 'Receive secret rotation order: %s' % (
         ", ".join( self.rxRotation ) )
      print 'Transmit secret rotation order: %s\n' % (
         ", ".join( self.txRotation ) )

      print 'Secrets:'
      for secret in self.secrets:
         secret.render()

class SharedSecretProfiles( Model ):
   profiles = Dict( valueType=SharedSecretProfile,
                    help="Shared secret profiles indexed by name" )

   def render( self ):
      for profile in self.profiles.values():
         profile.render()
         print '\n'
