# Copyright (c) 2018 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

from Ark import timestampToStr
from ArnetModel import (
   IpGenericAddrAndPort,
   IpGenericAddress,
   MacAddress,
)
from CliModel import (
   Bool,
   Dict,
   Enum,
   Float,
   Int,
   List,
   Model,
   Str,
   Submodel,
)
from Ethernet import convertMacAddrToDisplay
from FlowTrackerCliUtil import(
   FtrTypeEnum,
   addressStr,
   ftrTypeShowStr,
   protocolStr,
   timeStr,
)
from IntfModels import Interface
from SftCliLib import hoFlowStateShowStr
from SftCliUtil import HoFlowStateEnum
from SftLib import valueOrUnknown
import TableOutput
import Tac
import TacSigint
from Toggles.FlowTrackerToggleLib import toggleHwOffloadIpv4Enabled
from TypeFuture import TacLazyType

IpProtoType = TacLazyType( 'Arnet::IpProtocolNumber' )
FtConsts = TacLazyType( 'FlowTracking::Constants' )

def pindent( indent, *args ):
   print " " * ( 2 * indent ) + " ".join( map( str, args ) )

def macAddrStrOrNone( macAddr ):
   if macAddr is None:
      return ''
   else:
      return convertMacAddrToDisplay( macAddr.stringValue )

def timeStrOrNone( timestamp ):
   if timestamp is None:
      return ''
   else:
      return timeStr( timestamp )

def utcTimestampToStr( timestamp ):
   return timestampToStr( timestamp, now=Tac.utcNow() )

def lastSentStr( timestamp ):
   if timestamp is None:
      return ''
   else:
      return ', last sent ' + utcTimestampToStr( timestamp )

def getLastClearedStr( timestamp ):
   if timestamp is None:
      return ''
   else:
      return ' (Last cleared {})'.format( utcTimestampToStr( timestamp ) )

def collectorAddr( addrAndPort ):
   if addrAndPort.ip.af == 'ipv6':
      fmtStr = '[{}]:{}'
   else:
      fmtStr = '{}:{}'
   return fmtStr.format( addrAndPort.ip, addrAndPort.port )

def templateIdNum( staticTemplateIdEnum ):
   return FtConsts.reservedTemplateId( staticTemplateIdEnum )

# We will move to Stash when it is ready. For now, redender python
# Json model as a temporary solution

class FlowKeyModel( Model ):
   vrfName = Str( help="VRF name" )
   vlanId = Int( help="Flow VLAN ID" )
   srcAddr = IpGenericAddress( help="Source IP address" )
   dstAddr = IpGenericAddress( help="Destination IP address" )
   ipProtocol = Enum( help="IP protocol", values=IpProtoType.attributes,
                      optional=True )
   ipProtocolNumber = Int( help="IP protocol number" )
   srcPort = Int( help="Source port" )
   dstPort = Int( help="Destination port" )

class FlowDetailModel( Model ):
   srcEthAddr = MacAddress( help="Flow source MAC address" )
   dstEthAddr = MacAddress( help="Flow destination MAC address" )
   if toggleHwOffloadIpv4Enabled():
      sampledBytesReceived = Int( help="Number of sampled bytes received",
                                  optional=True )
      sampledPktsReceived = Int( help="Number of sampled packets received",
                                 optional=True )
      hwBytesReceived = Int( help="Number of bytes received in hardware",
                             optional=True )
      hwPktsReceived = Int( help="Number of packets received in hardware",
                            optional=True )
   flowLabel = Int( help="IPv6 flow label", optional=True )
   tos = Int( help="TOS in IP header" )
   tcpFlags = Str( help="Flow TCP flags" )
   lastPktTime = Float( help="Last packet received time" )
   ingressIntf = Interface( help="Flow ingress interface" )
   egressVlanId = Int( help="Flow egress VLAN ID" )
   # egressIntf may not be an IntfId, i.e. multicast/discard
   egressIntf = Str( help="Flow egress interface" )
   srcAs = Int( help="BGP source AS" )
   dstAs = Int( help="BGP destination AS" )
   nextHopIp = IpGenericAddress( help="Next hop IP address" )
   bgpNextHopIp = IpGenericAddress( help="BGP next hop IP address" )
   srcPrefixLen = Int( help="Source prefix length" )
   dstPrefixLen = Int( help="Destination prefix length" )
   vni = Str( help="VXLAN Network Identifier" )
   tunnelState = Str( help="State of the VXLAN tunnel" )

class FlowModel( Model ):
   key = Submodel( valueType=FlowKeyModel, help="Flow key" )
   bytesReceived = Int( help="Number of bytes received" )
   pktsReceived = Int( help="Number of packets received" )
   startTime = Float( help="Flow start time" )
   flowDetail = Submodel( valueType=FlowDetailModel, optional=True,
                          help="Flow detailed information" )

class GroupModel( Model ):
   flows = List( valueType=FlowModel, help="List of flows" )

class TrackerModel( Model ):
   groups = Dict( keyType=str, valueType=GroupModel,
                  help="A mapping bewteen group name and group" )
   numFlows = Int( help="Total number of flows" )

# Sample Tracking model Json output
# {
#     "running": true,
#     "trackers": {
#         "ftTest1": {
#             "groups": {
#                 "IPv4": {
#                     "flows": [
#                         {
#                             "bytesReceived": 0,
#                             "key": {
#                                 "vrfName": "default",
#                                 "vlanId": 0,
#                                 "srcAddr": "1.1.1.1",
#                                 "dstPort": 1,
#                                 "ipProtocol": "ipProtoTcp",
#                                 "dstAddr": "2.2.2.1",
#                                 "srcPort": 1
#                             },
#                             "startTime": 1543015649.679666,
#                             "pktsReceived": 0,
#                             "flowDetail": {
#                                 "lastPktTime": 0.0,
#                                 "bgpNextHopIp": "0.0.0.0",
#                                 "srcAs": 0,
#                                 "srcEthAddr": "00:00:00:00:00:00",
#                                 "dstPrefixLen": 0,
#                                 "tos": 0,
#                                 "ingressIntf": "",
#                                 "tcpFlags": "none",
#                                 "dstAs": 0,
#                                 "egressVlanId": 0,
#                                 "egressIntf": "",
#                                 "dstEthAddr": "00:00:00:00:00:00",
#                                 "nextHopIp": "0.0.0.0",
#                                 "vni": "unknown",
#                                 "tunnelState": "unknown",
#                                 "sampledBytesReceived": 0,
#                                 "sampledPktsReceived": 0,
#                                 "hwBytesReceived": 0,
#                                 "hwPktsReceived": "0,
#                             }
#                         }
#                     ]
#                 },
#                 "IPv6": {
#                     "flows": [
#                         {
#                             "bytesReceived": 0,
#                             "key": {
#                                 "vrfName": "default",
#                                 "vlanId": 0,
#                                 "srcAddr": "2001:db8:1:1::1",
#                                 "dstPort": 1,
#                                 "ipProtocol": "ipProtoTcp",
#                                 "dstAddr": "2001:db8:2:2::1",
#                                 "srcPort": 1
#                             },
#                             "startTime": 1543015649.681361,
#                             "pktsReceived": 0,
#                             "flowDetail": {
#                                 "lastPktTime": 0.0,
#                                 "bgpNextHopIp": "::",
#                                 "srcAs": 0,
#                                 "srcEthAddr": "00:00:00:00:00:00",
#                                 "dstPrefixLen": 0,
#                                 "tos": 0,
#                                 "ingressIntf": "",
#                                 "tcpFlags": "none",
#                                 "srcPrefixLen": 0,
#                                 "dstAs": 0,
#                                 "egressVlanId": 0,
#                                 "egressIntf": "",
#                                 "dstEthAddr": "00:00:00:00:00:00",
#                                 "nextHopIp": "::",
#                                 "vni": "unknown",
#                                 "tunnelState": "unknown"
#                                 "flowLabel": 0,
#                             }
#                         }
#                     ]
#                 }
#             },
#             "numFlows": 2
#         }
#     }
# }
class TrackingModel( Model ):
   trackers = Dict( keyType=str, valueType=TrackerModel,
                    help="A mapping between tracker name and tracker" )
   running = Bool( help="Sampled flow tracking is active" )
   ftrType = Enum( help="FlowTracker Type", values=FtrTypeEnum.attributes )

   def vlanIdOrRouted( self, vlanId, ingress=True, intf=None ):
      if ingress is True or intf:
         # For ingress vlan 0 means routed,
         # For egress, if egressIntf is known vlan 0 means routed
         #   else, vlan 0 mean unknown
         return "routed" if vlanId == 0 else vlanId
      else:
         return vlanId if vlanId else "unknown"

   def renderFlowDetail( self, flow, key ):
      detail = flow.flowDetail
      tab = " " * 4
      print "%sFlow: %s %s - %s, VRF: %s, VLAN: %s" % \
            ( tab,
                  protocolStr( key.ipProtocol, key.ipProtocolNumber ),
                  addressStr( key.srcAddr, key.srcPort ),
                  addressStr( key.dstAddr, key.dstPort ),
                  key.vrfName, self.vlanIdOrRouted( key.vlanId ) )
      tab = " " * 6
      print "%sStart time: %s, Last packet time: %s" % \
            ( tab, timeStr( flow.startTime ), timeStr( detail.lastPktTime ) )
      flowLabel = ", Flow label: " + \
                  str( detail.flowLabel ) if detail.flowLabel is not None else ''
      print "%sPackets: %d, Bytes: %d, TOS: %d, TCP flags: %s%s"\
            % ( tab, flow.pktsReceived, flow.bytesReceived,
            detail.tos, detail.tcpFlags, flowLabel )
      if toggleHwOffloadIpv4Enabled() and detail.sampledPktsReceived is not None:
         print "%sSampled packets: %d, Sampled bytes: %d, Hardware packets: %d,"\
               " Hardware bytes: %d" % ( tab, detail.sampledPktsReceived,
                     detail.sampledBytesReceived, detail.hwPktsReceived,
                     detail.hwBytesReceived )
      print "%sSource MAC: %s, Destination MAC: %s" % \
            ( tab, convertMacAddrToDisplay( detail.srcEthAddr.stringValue ),
                  convertMacAddrToDisplay( detail.dstEthAddr.stringValue ) )
      print "%sIngress Interface: %s, Egress VLAN: %s, Egress Interface: %s" % \
            ( tab, valueOrUnknown( detail.ingressIntf ),
                  self.vlanIdOrRouted( detail.egressVlanId, False,
                     detail.egressIntf ), detail.egressIntf )
      print "%sNext hop: %s, BGP next hop: %s (AS %s), Source AS: %s" % \
            ( tab, valueOrUnknown( detail.nextHopIp ),
            valueOrUnknown( detail.bgpNextHopIp ),
            valueOrUnknown( detail.dstAs ), valueOrUnknown( detail.srcAs ) )
      print "%sSource prefix length: %s, Destination prefix length: %s" % \
            ( tab, valueOrUnknown( detail.srcPrefixLen ),
            valueOrUnknown( detail.dstPrefixLen ) )
      print "%sVNI: %s, tunnelState: %s\n" % \
            ( tab, detail.vni, detail.tunnelState )

   def render( self ):
      if not self.running:
         print "%s flow tracking is not active" % ftrTypeShowStr[ self.ftrType ]
         return

      headings = ( "VRF", "VLAN", "Source", "Destination", "Protocol",
                   "Start Time", "Pkts", "Bytes" )
      formatLeft = TableOutput.Format( justify="left" )
      formatLeft.noPadLeftIs( True )
      formatCenter = TableOutput.Format( justify="center" )
      formatRight = TableOutput.Format( justify="right" )

      for trackerName, tracker in sorted( self.trackers.items() ):
         print "Tracker: %s, Flows: %s\n" % ( trackerName, tracker.numFlows )
         for groupName, group in sorted( tracker.groups.items() ):
            if not group.flows:
               continue

            # check the first flow to see if we need to display detailed format
            if group.flows[ 0 ].flowDetail:
               print "  Group: %s, Flows: %s" % ( groupName, len( group.flows ) )
            else:
               table = TableOutput.createTable( headings )
               table.formatColumns( formatLeft, formatLeft, formatLeft, formatLeft,
                                    formatCenter, formatRight, formatRight,
                                    formatRight )
               print "Group: %s, Flows: %s\n" % ( groupName, len( group.flows ) )

            for flow in group.flows:
               key = flow.key
               if flow.flowDetail:
                  self.renderFlowDetail( flow, key )
               else:
                  table.newRow( key.vrfName, self.vlanIdOrRouted( key.vlanId ),
                                addressStr( key.srcAddr, key.srcPort ),
                                addressStr( key.dstAddr, key.dstPort ),
                                protocolStr( key.ipProtocol, key.ipProtocolNumber ),
                                timeStr( flow.startTime ),
                                flow.pktsReceived,
                                flow.bytesReceived )
               TacSigint.check()

            if not group.flows[ 0 ].flowDetail:
               print table.output()

# hardware offload flow-table counters
class HoFlowCounterEntryModel( Model ):
   key = Submodel( valueType=FlowKeyModel,
                   help="Hardware offload Flow counter key" )
   ingressIntf = Interface( help="Flow ingress interface" )
   pktsReceived = Int( help="Number of packets received" )
   bytesReceived = Int( help="Number of bytes received" )
   createTime = Float( help="Hardware offload Flow counter entry create time" )
   updateTime = Float( help="Hardware offload Flow counter entry update time" )

class HoGroupCounterModel( Model ):
   flows = List( valueType=HoFlowCounterEntryModel, help="List of flows" )

class HoTrackerCounterModel( Model ):
   groups = Dict( keyType=str, valueType=HoGroupCounterModel,
                  help="A mapping bewteen group name and group" )
   numFlows = Int( help="Total number of flows" )

# Sample Hardware Offload FlowTable Counters model Json output
# {
#     "running": true,
#     "ftrType": sampled,
#     "trackers": {
#         "ftr1": {
#             "groups": {
#                 "IPv4": {
#                     "flows": [
#                         {
#                             "key": {
#                                 "srcAddr": "1.1.1.1",
#                                 "ipProtocolNumber": 6,
#                                 "dstAddr": "2.2.2.1",
#                                 "vrfName": "default",
#                                 "ipProtocol": "ipProtoTcp",
#                                 "dstPort": 1,
#                                 "srcPort": 1
#                                 "vlanId": 0,
#                             },
#                             "ingressIntf": "Ethernet1",
#                             "pktsReceived": 10,
#                             "bytesReceived": 1280,
#                             "createTime": 1587357689.51372,
#                             "updateTime": 1587357689.51372,
#                         }
#                     ]
#                 },
#             },
#             "numFlows": 1
#         }
#     }
# }

class HoFlowCountersModel( Model ):
   trackers = Dict( keyType=str, valueType=HoTrackerCounterModel,
                    help="A mapping between tracker name and tracker" )
   running = Bool( help="Sampled flow tracking is active" )
   ftrType = Enum( help="FlowTracker Type", values=FtrTypeEnum.attributes )

   def vlanIdOrRouted( self, vlanId ):
      return vlanId or 'routed'

   def render( self ):
      if not self.running:
         print "%s flow tracking is not active" % ftrTypeShowStr[ self.ftrType ]
         return

      headings = ( "VRF", "VLAN", "Source", "Destination", "Protocol", "Interface",
                   "Packets", "Bytes", "Create Time", "Update Time" )
      formatLeft = TableOutput.Format( justify="left" )
      formatLeft.noPadLeftIs( True )
      formatCenter = TableOutput.Format( justify="center" )
      formatRight = TableOutput.Format( justify="right" )

      for trackerName in sorted( self.trackers, key=str.lower ):
         tracker = self.trackers[ trackerName ]
         print "Tracker: %s, Flows: %s\n" % ( trackerName, tracker.numFlows )
         for groupName, group in sorted( tracker.groups.items() ):
            if not group.flows:
               continue

            table = TableOutput.createTable( headings )
            table.formatColumns( formatLeft, formatLeft, formatLeft, formatLeft,
                                 formatCenter, formatRight, formatRight,
                                 formatRight )
            print "Group: %s, Flows: %s\n" % ( groupName, len( group.flows ) )

            for flow in group.flows:
               key = flow.key
               table.newRow( key.vrfName, self.vlanIdOrRouted( key.vlanId ),
                             addressStr( key.srcAddr, key.srcPort ),
                             addressStr( key.dstAddr, key.dstPort ),
                             protocolStr( key.ipProtocol, key.ipProtocolNumber ),
                             flow.ingressIntf.stringValue, flow.pktsReceived,
                             flow.bytesReceived, timeStr( flow.createTime ),
                             timeStr( flow.updateTime ) )
               TacSigint.check()

            print table.output()

# hardware offload flow-table
class HoFlowDetailModel( Model ):
   srcEthAddr = MacAddress( help="Flow source MAC address", optional=True )
   dstEthAddr = MacAddress( help="Flow destination MAC address", optional=True )
   swCreateTime = Float( help="Hardware offload Flow config entry create time",
                         optional=True )
   hwUpdateTime = Float( help="Hardware offload Flow status entry update time" )

class HoFlowEntryModel( Model ):
   key = Submodel( valueType=FlowKeyModel,
                   help="Hardware offload Flow Status key" )
   ingressIntf = Interface( help="Flow ingress interface" )
   state = Enum( values=HoFlowStateEnum.attributes,
                 help="Hardware offload Flow state" )
   hwCreateTime = Float( help="Hardware offload Flow status entry create time" )
   flowDetail = Submodel( valueType=HoFlowDetailModel, optional=True,
                          help="Hardware offload Flow detailed information" )

class HoGroupModel( Model ):
   flows = List( valueType=HoFlowEntryModel, help="List of flows" )

class HoTrackerModel( Model ):
   groups = Dict( keyType=str, valueType=HoGroupModel,
                  help="A mapping bewteen group name and group" )
   numFlows = Int( help="Total number of flows" )

# Sample Hardware Offload FlowTable model Json output
# {
#     "running": true,
#     "ftrType": sampled,
#     "trackers": {
#         "ftr1": {
#             "groups": {
#                 "IPv4": {
#                     "flows": [
#                         {
#                             "hwCreateTime": 1587357689.51372,
#                             "state": "active",
#                             "key": {
#                                 "srcAddr": "1.1.1.1",
#                                 "ipProtocolNumber": 6,
#                                 "dstAddr": "2.2.2.1",
#                                 "vrfName": "default",
#                                 "ipProtocol": "ipProtoTcp",
#                                 "dstPort": 1,
#                                 "srcPort": 1
#                                 "vlanId": 0,
#                             },
#                             "flowDetail": {
#                                 "swCreateTime": 1587357689.51372,
#                                 "hwUpdateTime": 1587357689.51372,
#                                 "srcEthAddr": "00:01:02:03:04:05",
#                                 "dstEthAddr": "00:00:01:01:01:01",
#                             }
#                             "ingressIntf": "Ethernet1",
#                         }
#                     ]
#                 },
#             },
#             "numFlows": 1
#         }
#     }
# }

class HoFlowTableModel( Model ):
   trackers = Dict( keyType=str, valueType=HoTrackerModel,
                    help="A mapping between tracker name and tracker" )
   running = Bool( help="Sampled flow tracking is active" )
   ftrType = Enum( help="FlowTracker Type", values=FtrTypeEnum.attributes )

   def vlanIdOrRouted( self, vlanId ):
      return vlanId or 'routed'

   def renderHoFlowDetail( self, flow, key ):
      detail = flow.flowDetail
      tab = " " * 4
      print "%sFlow: %s %s - %s, VRF: %s, VLAN: %s, HW State: %s" % \
            ( tab,
                  protocolStr( key.ipProtocol, key.ipProtocolNumber ),
                  addressStr( key.srcAddr, key.srcPort ),
                  addressStr( key.dstAddr, key.dstPort ),
                  key.vrfName, self.vlanIdOrRouted( key.vlanId ),
                  hoFlowStateShowStr[ flow.state ] )
      tab = " " * 6
      print "%sIngress Interface: %s, SW Create time: %s"\
            % ( tab, flow.ingressIntf, timeStrOrNone( detail.swCreateTime ) )
      print "%sHW Create time: %s, HW Last Update time: %s"\
            % ( tab, timeStr( flow.hwCreateTime ), timeStr( detail.hwUpdateTime ) )
      print "%sSource MAC: %s, Destination MAC: %s"\
            % ( tab, macAddrStrOrNone( detail.srcEthAddr ),
                macAddrStrOrNone( detail.dstEthAddr ) )

   def render( self ):
      '''
      Sample detail output:

      Tracker: ftr1, Flows: 1
        Group: IPv4, Flows: 1
          Flow: UDP 1.1.1.1:0 - 1.2.1.2:0, VRF: red, VLAN: 2, HW State: active

               Ingress Interface: 'Ethernet1', SW Create time: 2020-04-19 21:41:29.0
               HW Create time: 2020-04-19 21:41:29.513720, HW Last Update time:
2020-04-19 21:41:29.513720
               Source MAC: 0001.0203.0405, Destination MAC: 0000.0101.0101

      '''

      if not self.running:
         print "%s flow tracking is not active" % ftrTypeShowStr[ self.ftrType ]
         return

      headings = ( "VRF", "VLAN", "Source", "Destination", "Protocol", "Interface",
                   "State", "Create Time" )
      formatLeft = TableOutput.Format( justify="left" )
      formatLeft.noPadLeftIs( True )
      formatCenter = TableOutput.Format( justify="center" )
      formatRight = TableOutput.Format( justify="right" )

      for trackerName in sorted( self.trackers, key=str.lower ):
         tracker = self.trackers[ trackerName ]
         print "Tracker: %s, Flows: %s\n" % ( trackerName, tracker.numFlows )
         for groupName, group in sorted( tracker.groups.items() ):
            if not group.flows:
               continue

            # check the first flow to see if we need to display detailed format
            if group.flows[ 0 ].flowDetail:
               print "  Group: %s, Flows: %s" % ( groupName, len( group.flows ) )
            else:
               table = TableOutput.createTable( headings )
               table.formatColumns( formatLeft, formatLeft, formatLeft, formatLeft,
                                    formatCenter, formatRight, formatRight,
                                    formatRight )
               print "Group: %s, Flows: %s\n" % ( groupName, len( group.flows ) )

            for flow in group.flows:
               key = flow.key
               if flow.flowDetail:
                  self.renderHoFlowDetail( flow, key )
               else:
                  table.newRow( key.vrfName, self.vlanIdOrRouted( key.vlanId ),
                                addressStr( key.srcAddr, key.srcPort ),
                                addressStr( key.dstAddr, key.dstPort ),
                                protocolStr( key.ipProtocol, key.ipProtocolNumber ),
                                flow.ingressIntf.stringValue,
                                hoFlowStateShowStr[ flow.state ],
                                timeStr( flow.hwCreateTime ) )
               TacSigint.check()

            if not group.flows[ 0 ].flowDetail:
               print table.output()

# flow tracking counters
class CollectorSetCounters( Model ):
   flowRecords = Int(
      help="Number of flow records sent to the collector" )

class CollectorTemplateCounters( Model ):
   templateType = Enum(
      help="IPFIX template type",
      values=( "template", "optionsTemplate", ) )
   templates = Int(
      help="Number of templates sent to the collector" )

class CollectorTimestamp( Model ):
   message = Float(
      help="UTC time of last exported message",
      optional=True )
   template = Float(
      help="UTC time of last exported template",
      optional=True )
   dataRecord = Float(
      help="UTC time of last exported data record",
      optional=True )
   optionsData = Float(
      help="UTC time of last exported options data record",
      optional=True )

class CollectorCounters( Model ):
   addrAndPort = Submodel(
      help="Collector IP address and port number",
      valueType=IpGenericAddrAndPort )
   exportedMessageTotalCount = Int(
      help="The total number of messages sent to the collector" )
   exportedFlowRecordTotalCount = Int(
      help="The total number of flow records sent to the collector" )
   exportedOctetTotalCount = Int(
      help="The total number of bytes sent to the collector" )
   lastUpdates = Submodel(
      help="UTC time of last events",
      valueType=CollectorTimestamp )
   sets = Dict(
      help="A mapping of IPFIX template ID (set ID) to per-set-id counters",
      keyType=int,
      valueType=CollectorSetCounters )
   templates = List(
      help="A list of per-template counters",
      valueType=CollectorTemplateCounters )

class ExporterCounters( Model ):
   exporterType = Enum(
      help="Exporter type",
      values=( "ipfix", ) )
   clearTime = Float(
      help="UTC time when counters were last cleared",
      optional=True )
   collectors = List(
      help="A list of per-collector counters",
      valueType=CollectorCounters )

def exporterTypeStr( exporterType ):
   mapping = {
      'ipfix' : 'IPFIX',
   }
   return mapping[ exporterType ]

if toggleHwOffloadIpv4Enabled():
   class HardwareOffloadCounters( Model ):
      __public__ = False
      flowsActive = Int( help="Number of active flows in hardware",
            optional=True )
      flowsCreated = Int( help="Number of flows created in hardware",
            optional=True )
      flowsDeleted = Int( help="Number of flows deleted from hardware",
            optional=True )
      flowsPending = Int(
            help="Number of flows pending to be created in hardware",
            optional=True )
      pktsReceived = Int(
            help="Number of packets received for hardware flows",
            optional=True )
      pktsDiscarded = Int(
            help="Number of packets discarded in CPU for hardware flows",
            optional=True )
      pktsHardwareMiss = Int(
            help="Number of packets that didn't hit hardware flows",
            optional=True )
      pktsHardwareFailed = Int(
            help="Number of packets that failed to create hardware flow",
            optional=True )
      clearTime = Float(
            help="UTC time when counters were last cleared",
            optional=True )

class FlowGroupCounters( Model ):
   activeFlows = Int(
      help="Number of active flows" )
   expiredFlows = Int(
      help="Cumulative expired flow count" )
   flows = Int(
      help="Cumulative flow count" )
   packets = Int(
      help="Cumulative count of sampled packets" )
   if toggleHwOffloadIpv4Enabled():
      hardwareOffload = Submodel(
         valueType=HardwareOffloadCounters,
         help="Hardware flow group counters",
         optional=True )

class TrackerCounters( Model ):
   activeFlows = Int(
      help="Number of active flows" )
   expiredFlows = Int(
      help="Cumulative expired flow count" )
   flows = Int(
      help="Cumulative flow count" )
   packets = Int(
      help="Cumulative count of sampled packets" )
   clearTime = Float(
      help="UTC time when counters were last cleared",
      optional=True )
   flowGroups = Dict(
      help="A mapping of flow group name to counters",
      keyType=str,
      valueType=FlowGroupCounters )
   exporters = Dict(
      help="A mapping of exporter name to counters",
      valueType=ExporterCounters )

class FtrCounters( Model ):
   running = Bool( help="Flow tracking agent is running" )
   trackers = Dict(
      help="A mapping of tracker name to counters",
      keyType=str,
      valueType=TrackerCounters )
   flows = Int( help="Cumulative flow count", optional=True )
   activeFlows = Int( help="Cumulative active flow count", optional=True )
   expiredFlows = Int( help="Cumulative expired flow count", optional=True )
   packets = Int( help="Cumulative count of sampled packets", optional=True )
   clearTime = Float(
         help="UTC time when total counters were last cleared",
         optional=True )
   ftrType = Enum( help="FlowTracker Type", values=FtrTypeEnum.attributes )
   if toggleHwOffloadIpv4Enabled():
      hardwareOffload = Submodel(
         valueType=HardwareOffloadCounters,
         help="Hardware flow group counters",
         optional=True )

   def renderFlowGroup( self, indent, groupInfo ):
      pindent( indent, '{} flows, {} RX packets'.format( groupInfo.activeFlows,
                                                         groupInfo.packets ) )
      if toggleHwOffloadIpv4Enabled():
         self.renderHardwareOffload( indent, groupInfo.hardwareOffload )

   def renderCollector( self, indent, collector ):
      pindent( indent, '{} messages{}'.format( collector.exportedMessageTotalCount,
         lastSentStr( collector.lastUpdates.message ) ) )
      flowRecords = 0
      optionsDataRecords = 0
      for setId, setCounter in collector.sets.items():
         if setId in xrange( templateIdNum( FtConsts.dataTemplateMin ),
               templateIdNum( FtConsts.dataTemplateMax ) + 1 ):
            flowRecords += setCounter.flowRecords
         else:
            optionsDataRecords += setCounter.flowRecords
      pindent( indent, '{} flow records{}'.format( flowRecords,
         lastSentStr( collector.lastUpdates.dataRecord ) ) )
      pindent( indent, '{} options data records{}'.format( optionsDataRecords,
         lastSentStr( collector.lastUpdates.optionsData ) ) )
      templates = sum( [ x.templates for x in collector.templates ] )
      pindent( indent, '{} templates{}'.format( templates,
         lastSentStr( collector.lastUpdates.template ) ) )

   def renderExporter( self, indent, expInfo ):
      for collector in expInfo.collectors:
         pindent( indent, 'Collector:', collector.addrAndPort.ip, "port",
                  collector.addrAndPort.port )
         self.renderCollector( indent + 1, collector )

   def renderTrackerDetails( self, indent, tracker ):
      lastCleared = getLastClearedStr( tracker.clearTime )
      pindent( indent, '{} flows, {} RX packets{}'.format( tracker.activeFlows,
                                                           tracker.packets,
                                                           lastCleared ) )
      pindent( indent,
               'Flows created: {}, expired: {}'.format( tracker.flows,
                                                        tracker.expiredFlows ) )
      for fgName in sorted( tracker.flowGroups ):
         groupInfo = tracker.flowGroups[ fgName ]
         pindent( indent, 'Group:', fgName )
         self.renderFlowGroup( indent + 1, groupInfo )

      for expName in sorted( tracker.exporters ):
         expInfo = tracker.exporters[ expName ]
         expType = exporterTypeStr( expInfo.exporterType )
         lastCleared = getLastClearedStr( expInfo.clearTime )
         pindent( indent, 'Exporter:', expName, '({}){}'.format( expType,
                                                                 lastCleared ) )
         self.renderExporter( indent + 1, expInfo )

   def renderTrackers( self, indent ):
      for trName in sorted( self.trackers ):
         tracker = self.trackers[ trName ]
         pindent( indent, 'Tracker:', trName )
         self.renderTrackerDetails( indent + 1, tracker )

   if toggleHwOffloadIpv4Enabled():
      def renderHardwareOffload( self, indent, counter ):
         if counter:
            pindent( indent, 'Hardware offload:' )
            flowsStr = 'Flows: '
            if counter.flowsActive is not None:
               flowsStr += '{} active, '.format( counter.flowsActive )
            if counter.flowsPending is not None:
               flowsStr += '{} pending, '.format( counter.flowsPending )
            flowsStr += '{} created, {} deleted'.format(
                  counter.flowsCreated, counter.flowsDeleted )
            pindent( indent + 1, flowsStr )
            pindent( indent + 1,
                  'Packets: {} received, {} discards, {} offload miss, '
                  '{} offload failed'.format(
                     counter.pktsReceived, counter.pktsDiscarded,
                     counter.pktsHardwareMiss, counter.pktsHardwareFailed ) )

   def render( self ):
      '''
      Render FtrCounters and all submodels.

      Sample output:
        Total active flows: 1050, RX packets: 30130
         (Last cleared 4 days, 19:55:12 ago)
        Total flows created: 4000, expired: 3050
        Hardware offload:
          Flows: 340 active, 30 pending, 430 created, 80 deleted
          Packets: 1000 received, 50 discards, 30 offload miss, 100 offload failed
        Tracker: ftr1
          500 flows, 25040 RX packets (Last cleared 4 days, 19:55:12 ago)
          Flows created: 2000, expired: 1500
          Group: IPv4
            200 flows, 5040 RX packets
            Hardware offload:
              Flows: 200 created, 40 deleted
              Packets: 500 received, 0 discards, 15 offload miss, 60 offload failed
          Group: IPv6
            300 flows, 20000 RX packets
          Exporter: exp1 (IPFIX) (Last cleared 3 days, 18:55:12 ago)
            Collector: 10.0.1.6 port 2055
              1500 messages, last sent 0:03:30 ago
              200 flow records, last sent 0:03:30 ago
              1000 options data records, last sent 0:02:30 ago
              300 templates, last sent 0:05:00 ago
        Tracker: ftr2
          550 flows, 5090 RX packets (Last cleared 3 days, 17:55:12 ago)
          Flows created: 2000, expired: 1550
          Group: IPv4
            220 flows, 5040 RX packets
            Hardware offload:
              Flows: 230 created, 40 deleted
              Packets: 500 received, 0 discards, 15 offload miss, 40 offload failed
          Group: IPv6
            330 flows, 50 RX packets
          Exporter: exp1 (IPFIX) (Last cleared 2 days, 16:55:12 ago)
            Collector: 10.0.5.6 port 2055
              1500 messages, last sent 0:03:30 ago
              200 flow records, last sent 0:03:30 ago
              100 options data records, last sent 0:02:30 ago
              400 templates, last sent 0:05:00 ago
            Collector: b::6 port 1345
              1500 messages, last sent 0:03:30 ago
              2 flow records, last sent 0:03:30 ago
              100 options data records, last sent 0:02:30 ago
              5 templates, last sent 0:05:00 ago
            Collector: b::6 port 5454
              1500 messages, last sent 0:05:00 ago
              0 flow records
              0 options data records
              30 templates, last sent 0:05:00 ago
      '''
      if not self.running:
         print "%s flow tracking is not active" % ftrTypeShowStr[ self.ftrType ]
         return

      indent = 0
      lastCleared = getLastClearedStr( self.clearTime )
      pindent( indent,
            'Total active flows: {}, RX packets: {}{}'.format( self.activeFlows,
            self.packets,
            lastCleared ) )
      pindent( indent, 'Total flows created: {}, expired: {}'.format( self.flows,
                                                           self.expiredFlows ) )
      if toggleHwOffloadIpv4Enabled():
         self.renderHardwareOffload( indent, self.hardwareOffload )
      self.renderTrackers( indent )
